% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ggcoefstats.R
\name{ggcoefstats}
\alias{ggcoefstats}
\title{Dot-and-whisker plots for regression analyses}
\usage{
ggcoefstats(
  x,
  output = "plot",
  statistic = NULL,
  bf.message = TRUE,
  p.adjust.method = "none",
  coefficient.type = c("beta", "location", "coefficient"),
  effsize = "eta",
  partial = TRUE,
  nboot = 500,
  meta.analytic.effect = FALSE,
  meta.type = "parametric",
  conf.int = TRUE,
  conf.level = 0.95,
  k = 2,
  exclude.intercept = TRUE,
  exponentiate = FALSE,
  sort = "none",
  xlab = "regression coefficient",
  ylab = "term",
  title = NULL,
  subtitle = NULL,
  only.significant = FALSE,
  caption = NULL,
  caption.summary = TRUE,
  point.args = list(size = 3, color = "blue"),
  errorbar.args = list(height = 0),
  vline = TRUE,
  vline.args = list(size = 1, linetype = "dashed"),
  stats.labels = TRUE,
  stats.label.color = NULL,
  stats.label.args = list(size = 3, direction = "y"),
  package = "RColorBrewer",
  palette = "Dark2",
  direction = 1,
  ggtheme = ggplot2::theme_bw(),
  ggstatsplot.layer = TRUE,
  messages = FALSE,
  ...
)
}
\arguments{
\item{x}{A model object to be tidied, or a tidy data frame containing
results. If a data frame is to be plotted, it \emph{must} contain columns named
\code{term} (names of predictors), or \code{estimate} (corresponding estimates of
coefficients or other quantities of interest). Other optional columns are
\code{conf.low} and \code{conf.high} (for confidence intervals); \code{p.value}. It is
important that all \code{term} names should be unique. Function internally uses
\code{broom::tidy} or \code{parameters::model_parameters} to get a tidy dataframe.}

\item{output}{Character describing the expected output from this function:
\code{"plot"} (visualization of regression coefficients) or \code{"tidy"} (tidy
dataframe of results from \code{broom::tidy}) or \code{"glance"} (object from
\code{broom::glance}) or \code{"augment"} (object from \code{broom::augment}).}

\item{statistic}{Which statistic is to be displayed (either \code{"t"} or \code{"f"}or
\code{"z"}) in the label. This is especially important if the \code{x} argument in
\code{ggcoefstats} is a dataframe in which case the function wouldn't know what
kind of model it is dealing with.}

\item{bf.message}{Logical that decides whether results from running a
Bayesian meta-analysis assuming that the effect size \emph{d} varies across
studies with standard deviation \emph{t} (i.e., a random-effects analysis)
should be displayed in caption. Defaults to \code{TRUE}.}

\item{p.adjust.method}{Adjustment method for \emph{p}-values for multiple
comparisons. Possible methods are: \code{"holm"}, \code{"hochberg"}, \code{"hommel"},
\code{"bonferroni"}, \code{"BH"}, \code{"BY"}, \code{"fdr"}, \code{"none"}. Default is no correction
(\code{"none"}). This argument is relevant for multiplicity correction for
multiway ANOVA designs (see,
\href{https://link.springer.com/article/10.3758/s13423-015-0913-5}{Cramer
et al., 2015}).}

\item{coefficient.type}{Relevant only for ordinal regression models (\code{clm} ,
\code{clmm}, \code{"svyolr"}, and \code{polr}), this argument decides which parameters are
display in the plot. Available parameters are: parameter that measures the
\strong{intercept}, i.e. the log-odds distance between response values
(\code{"alpha"}); effects on the \strong{location} (\code{"beta"}); or effects on the
\strong{scale} (\code{"zeta"}). For \code{clm} and \code{clmm} models, by default, only
\code{"beta"} (a vector of regression parameters) parameters will be show. Other
options are \code{"alpha"} (a vector of threshold parameters) or \code{"both"}. For
\code{polr} models, by default, only \code{"coefficient"} will be shown. Other option
is to show \code{"zeta"} parameters. Note that, from \verb{broom 0.7.0} onward,
coefficients will be renamed and \code{"intercept"} type coefficients will
correspond to \code{"alpha"} parameters, \code{"location"} type coefficients will
correspond to \code{"beta"} parameters, and \code{"scale"} type coefficients will
correspond to \code{"zeta"} parameters.}

\item{effsize}{Character describing the effect size to be displayed: \code{"eta"}
(default) or \code{"omega"}. This argument is relevant
only for models objects of class \code{aov}, \code{anova}, and \code{aovlist}.}

\item{partial}{Logical that decides if partial eta-squared or omega-squared
are returned (Default: \code{TRUE}). If \code{FALSE}, eta-squared or omega-squared
will be returned. Valid only for objects of class \code{aov}, \code{anova}, or
\code{aovlist}.}

\item{nboot}{Number of bootstrap samples for confidence intervals for partial
eta-squared and omega-squared (Default: \code{500}). This argument is relevant
only for models objects of class \code{aov}, \code{anova}, and \code{aovlist}.}

\item{meta.analytic.effect}{Logical that decides whether subtitle for
meta-analysis via linear (mixed-effects) models (default: \code{FALSE}). If
\code{TRUE}, input to argument \code{subtitle} will be ignored. This will be mostly
relevant if a data frame with estimates and their standard errors is
entered as input to \code{x} argument.}

\item{meta.type}{Type of statistics used to carry out random-effects
meta-analysis. If \code{"parametric"} (default), \code{metafor::rma} function will be
used. If \code{"robust"}, \code{metaplus::metaplus} function will be used. If
\code{"bayes"}, \code{metaBMA::meta_random} function will be used.}

\item{conf.int}{Logical. Decides whether to display confidence intervals as
error bars (Default: \code{TRUE}).}

\item{conf.level}{Numeric deciding level of confidence intervals (Default:
\code{0.95}). For \code{MCMC} model objects (\code{Stan}, \code{JAGS}, etc.), this will be
probability level for CI.}

\item{k}{Number of decimal places expected for results displayed in labels
(Default : \code{k = 2}).}

\item{exclude.intercept}{Logical that decides whether the intercept should be
excluded from the plot (Default: \code{TRUE}).}

\item{exponentiate}{If \code{TRUE}, the \code{x}-axis will be logarithmic (Default:
\code{FALSE}). Note that exponents for the coefficient estimates and associated
standard errors plus confidence intervals are computed by the underlying
tidying packages (\code{broom}/\code{parameters}) and not done by \code{ggcoefstats}. So
this might not work if the underlying packages don't support
exponentiation.}

\item{sort}{If \code{"none"} (default) do not sort, \code{"ascending"} sort by
increasing coefficient value, or \code{"descending"} sort by decreasing
coefficient value.}

\item{xlab, ylab}{Labels for \code{x}- and \code{y}- axis variables, respectively
(Defaults: \code{"regression coefficient"} and \code{"term"}).}

\item{title}{The text for the plot title.}

\item{subtitle}{The text for the plot subtitle. The input to this argument
will be ignored if \code{meta.analytic.effect} is set to \code{TRUE}.}

\item{only.significant}{If \code{TRUE}, only stats labels for significant effects
is shown (Default: \code{FALSE}). This can be helpful when a large number of
regression coefficients are to be displayed in a single plot. Relevant only
when the \code{output} is a plot.}

\item{caption}{Text to display as caption. This argument is relevant only
when \code{output = "caption"}.}

\item{caption.summary}{Logical. Decides whether the model summary should be
displayed as a cation to the plot (Default: \code{TRUE}). Color of the line
segment. Defaults to the same color as the text.}

\item{point.args}{Additional arguments that will be passed to
\code{ggplot2::geom_point} geom. Please see documentation for that function to
know more about these arguments.}

\item{errorbar.args}{Additional arguments that will be passed to
\code{ggplot2::geom_errorbarh} geom. Please see documentation for that function
to know more about these arguments.}

\item{vline}{Decides whether to display a vertical line (Default: \code{"TRUE"}).}

\item{vline.args}{Additional arguments that will be passed to
\code{ggplot2::geom_vline} geom. Please see documentation for that function to
know more about these arguments.}

\item{stats.labels}{Logical. Decides whether the statistic and \emph{p}-values for
each coefficient are to be attached to each dot as a text label using
\code{ggrepel} (Default: \code{TRUE}).}

\item{stats.label.color}{Color for the labels. If \code{stats.label.color} is
\code{NULL}, colors will be chosen from the specified \code{package} (Default:
\code{"RColorBrewer"}) and \code{palette} (Default: \code{"Dark2"}).}

\item{stats.label.args}{Additional arguments that will be passed to
\code{ggrepel::geom_label_repel} geom. Please see documentation for that
function to know more about these arguments.}

\item{package}{Name of package from which the palette is desired as string
or symbol.}

\item{palette}{Name of palette as string or symbol.}

\item{direction}{Either \code{1} or \code{-1}. If \code{-1} the palette will be reversed.}

\item{ggtheme}{A function, \code{ggplot2} theme name. Default value is
\code{ggplot2::theme_bw()}. Any of the \code{ggplot2} themes, or themes from
extension packages are allowed (e.g., \code{ggthemes::theme_fivethirtyeight()},
\code{hrbrthemes::theme_ipsum_ps()}, etc.).}

\item{ggstatsplot.layer}{Logical that decides whether \code{theme_ggstatsplot}
theme elements are to be displayed along with the selected \code{ggtheme}
(Default: \code{TRUE}). \code{theme_ggstatsplot} is an opinionated theme layer that
override some aspects of the selected \code{ggtheme}.}

\item{messages}{Deprecated. Retained only for backward compatibility.}

\item{...}{Additional arguments to tidying method.}
}
\value{
Plot with the regression coefficients' point estimates as dots with
confidence interval whiskers and other statistical details included as
labels.
}
\description{
Dot-and-whisker plots for regression analyses
}
\examples{
\donttest{
# for reproducibility
set.seed(123)

# -------------- with model object --------------------------------------

# model object
mod <- lm(formula = mpg ~ cyl * am, data = mtcars)

# to get a plot
ggstatsplot::ggcoefstats(x = mod, output = "plot")

# to get a tidy dataframe
ggstatsplot::ggcoefstats(x = mod, output = "tidy")

# to get a glance summary
ggstatsplot::ggcoefstats(x = mod, output = "glance")

# to get augmented dataframe
ggstatsplot::ggcoefstats(x = mod, output = "augment")

# -------------- with custom dataframe -----------------------------------

# creating a dataframe
df <-
  structure(
    list(
      term = structure(
        c(3L, 4L, 1L, 2L, 5L),
        .Label = c(
          "Africa",
          "Americas", "Asia", "Europe", "Oceania"
        ),
        class = "factor"
      ),
      estimate = c(
        0.382047603321706,
        0.780783111514665,
        0.425607573765058,
        0.558365541235078,
        0.956473848429961
      ),
      std.error = c(
        0.0465576338644502,
        0.0330218199731529,
        0.0362834986178494,
        0.0480571500648261,
        0.062215818388157
      ),
      statistic = c(
        8.20590677855356,
        23.6444603038067,
        11.7300588415607,
        11.6187818146078,
        15.3734833553524
      ),
      conf.low = c(
        0.290515146096969,
        0.715841986960399,
        0.354354575031406,
        0.46379116008131,
        0.827446138277154
      ),
      conf.high = c(
        0.473580060546444,
        0.845724236068931,
        0.496860572498711,
        0.652939922388847,
        1.08550155858277
      ),
      p.value = c(
        3.28679518728519e-15,
        4.04778497135963e-75,
        7.59757330804449e-29,
        5.45155840151592e-26,
        2.99171217913312e-13
      ),
      df.residual = c(
        394L, 358L, 622L,
        298L, 22L
      )
    ),
    row.names = c(NA, -5L),
    class = c(
      "tbl_df",
      "tbl", "data.frame"
    )
  )

# plotting the dataframe
ggstatsplot::ggcoefstats(
  x = df,
  statistic = "t",
  meta.analytic.effect = TRUE,
  k = 3
)

# -------------- getting model summary ------------------------------

# model
library(lme4)
lmm1 <- lme4::lmer(
  formula = Reaction ~ Days + (Days | Subject),
  data = sleepstudy
)

# dataframe with model summary
ggstatsplot::ggcoefstats(x = lmm1, output = "glance")

# -------------- getting augmented dataframe ------------------------------

# setup
set.seed(123)
library(survival)

# fit
cfit <-
  survival::coxph(formula = Surv(time, status) ~ age + sex, data = lung)

# augmented dataframe
ggstatsplot::ggcoefstats(
  x = cfit,
  data = lung,
  output = "augment",
  type.predict = "risk"
)
}
}
\references{
\url{https://indrajeetpatil.github.io/ggstatsplot/articles/web_only/ggcoefstats.html}
}
