% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gkwreg-datasets.R
\docType{data}
\name{ImpreciseTask}
\alias{ImpreciseTask}
\title{Imprecise Probabilities for Sunday Weather and Boeing Stock Task}
\format{
A data frame with 242 observations on 3 variables:
\describe{
\item{task}{factor with levels \verb{Boeing stock} and \verb{Sunday weather}.
Indicates which task the participant performed.}
\item{location}{numeric. Average of the lower estimate for the event not to
occur and the upper estimate for the event to occur (proportion).}
\item{difference}{numeric. Difference between upper and lower probability
estimates, measuring imprecision or uncertainty.}
}
}
\source{
Taken from Smithson et al. (2011) supplements.
}
\usage{
ImpreciseTask
}
\description{
Data from a cognitive psychology experiment where participants estimated upper
and lower probabilities for events to occur and not to occur. The study examines
judgment under uncertainty with imprecise probability assessments.
}
\details{
All participants in the study were either first- or second-year undergraduate
students in psychology at Australian universities, none of whom had a strong
background in probability theory or were familiar with imprecise probability
theories.

For the Sunday weather task, participants were asked to estimate the probability
that the temperature at Canberra airport on Sunday would be higher than a
specified value.

For the Boeing stock task, participants were asked to estimate the probability
that Boeing's stock would rise more than those in a list of 30 companies.

For each task, participants were asked to provide lower and upper estimates for
the event to occur and not to occur.
}
\examples{
\donttest{
require(gkwreg)
require(gkwdist)

data(ImpreciseTask)

# Example 1: Basic model with task effects
# Probability location varies by task type and uncertainty level
fit_kw <- gkwreg(location ~ task * difference,
  data = ImpreciseTask,
  family = "kw"
)
summary(fit_kw)

# Interpretation:
# - Alpha: Task type and uncertainty (difference) interact to affect
#   probability estimates
# - Different tasks may have different baseline probability assessments

# Example 2: Heteroscedastic model
# Precision of estimates may vary by task and uncertainty
fit_kw_hetero <- gkwreg(
  location ~ task * difference |
    task + difference,
  data = ImpreciseTask,
  family = "kw"
)
summary(fit_kw_hetero)

# Interpretation:
# - Beta: Variability in estimates differs between tasks
#   Higher uncertainty (difference) may lead to less precise estimates

# Example 3: McDonald distribution for extreme uncertainty
# Some participants may show very extreme probability assessments
fit_mc <- gkwreg(
  location ~ task * difference | # gamma: full interaction
    task * difference | # delta: full interaction
    task, # lambda: task affects extremity
  data = ImpreciseTask,
  family = "mc",
  control = gkw_control(
    method = "BFGS",
    maxit = 1500
  )
)
summary(fit_mc)

# Interpretation:
# - Lambda varies by task: Weather vs. stock may produce
#   different patterns of extreme probability assessments
}
}
\references{
Smithson, M., Merkle, E.C., and Verkuilen, J. (2011). Beta Regression Finite
Mixture Models of Polarization and Priming. \emph{Journal of Educational and
Behavioral Statistics}, \strong{36}(6), 804--831.
\doi{10.3102/1076998610396893}

Smithson, M., and Segale, C. (2009). Partition Priming in Judgments of
Imprecise Probabilities. \emph{Journal of Statistical Theory and Practice},
\strong{3}(1), 169--181.
}
\keyword{datasets}
