% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gkwreg-control.R
\name{gkw_control}
\alias{gkw_control}
\alias{print.gkw_control}
\title{Control Parameters for Generalized Kumaraswamy Regression}
\usage{
gkw_control(
  method = c("nlminb", "BFGS", "Nelder-Mead", "CG", "SANN", "L-BFGS-B"),
  start = NULL,
  fixed = NULL,
  hessian = TRUE,
  maxit = 500,
  reltol = sqrt(.Machine$double.eps),
  abstol = 0,
  trace = 0,
  silent = TRUE,
  eval.max = 500,
  iter.max = 300,
  step.min = 1e-08,
  step.max = 1,
  x.tol = 1.5e-08,
  rel.tol = sqrt(.Machine$double.eps),
  alpha = 1,
  beta = 0.5,
  gamma = 2,
  warn.1d.NelderMead = TRUE,
  type = 1,
  temp = 10,
  tmax = 10,
  lmm = 5,
  factr = 1e+07,
  pgtol = 0,
  REPORT = NULL,
  fnscale = 1,
  parscale = NULL,
  ndeps = NULL,
  ...
)

\method{print}{gkw_control}(x, ...)
}
\arguments{
\item{method}{Character string specifying the optimization algorithm.
Options: \code{"nlminb"} (default), \code{"BFGS"}, \code{"Nelder-Mead"},
\code{"CG"}, \code{"SANN"}, \code{"L-BFGS-B"}. If \code{"nlminb"}, uses
\code{\link[stats]{nlminb}}; otherwise uses \code{\link[stats]{optim}}
with the specified method.}

\item{start}{Optional named list of starting values for regression coefficients.
Names should match parameter names (alpha, beta, gamma, delta, lambda).
If \code{NULL} (default), starting values are determined automatically.}

\item{fixed}{Optional named list of parameters to hold fixed at specific
values during estimation. Currently experimental. Default \code{NULL}.}

\item{hessian}{Logical. If \code{TRUE} (default), compute the Hessian matrix
via \code{\link[TMB]{sdreport}} to obtain standard errors and variance-covariance
matrix. Set to \code{FALSE} for faster fitting when standard errors are not needed.}

\item{maxit}{Integer. Maximum number of iterations for the optimizer.
Default 500 for derivative-based methods, 10000 for SANN.
Increase for difficult optimization problems.}

\item{reltol}{Numeric. Relative convergence tolerance for the optimizer.
Default \code{sqrt(.Machine$double.eps)} approx. 1.5e-8. Smaller values require
tighter convergence but may increase computation time. Used by Nelder-Mead,
BFGS, and CG methods.}

\item{abstol}{Numeric. Absolute convergence tolerance. Default 0.
Used by some optimization methods as an additional stopping criterion.}

\item{trace}{Integer. Controls verbosity of the optimizer.
\itemize{
\item 0: Silent (default)
\item 1: Print iteration progress
\item 2+: Print detailed diagnostic information (up to 6 for L-BFGS-B)
}
Ignored if \code{silent = TRUE}.}

\item{silent}{Logical. If \code{TRUE} (default), suppress all progress messages
from TMB compilation and optimization. Set to \code{FALSE} for debugging or
to monitor long-running fits.}

\item{eval.max}{Integer. Maximum number of function evaluations (nlminb only).
Default 500. Increase for difficult optimization problems.}

\item{iter.max}{Integer. Maximum number of iterations (nlminb only).
Default 300. Usually less than \code{eval.max}.}

\item{step.min}{Numeric. Minimum step length (nlminb only). Default 1e-8.
Controls how small steps can become before stopping.}

\item{step.max}{Numeric. Maximum step length (nlminb only). Default 1.
Useful for preventing overshooting in difficult optimization problems.}

\item{x.tol}{Numeric. Tolerance for parameter convergence (nlminb only).
Default 1.5e-8. Optimizer stops if parameter changes are smaller than this.}

\item{rel.tol}{Numeric. Relative tolerance for function value (nlminb only).
Default \code{sqrt(.Machine$double.eps)}. Alternative specification of
relative tolerance.}

\item{alpha}{Numeric. Reflection factor for Nelder-Mead method. Default 1.0.
Only used when \code{method = "Nelder-Mead"}.}

\item{beta}{Numeric. Contraction factor for Nelder-Mead method. Default 0.5.
Only used when \code{method = "Nelder-Mead"}.}

\item{gamma}{Numeric. Expansion factor for Nelder-Mead method. Default 2.0.
Only used when \code{method = "Nelder-Mead"}.}

\item{warn.1d.NelderMead}{Logical. Whether to warn when Nelder-Mead is used
for one-dimensional optimization. Default \code{TRUE}.}

\item{type}{Integer. Update formula for CG method. Options:
\itemize{
\item 1: Fletcher-Reeves update
\item 2: Polak-Ribiere update
\item 3: Beale-Sorenson update
}
Default 1. Only used when \code{method = "CG"}.}

\item{temp}{Numeric. Starting temperature for SANN method. Default 10.
Only used when \code{method = "SANN"}.}

\item{tmax}{Integer. Number of function evaluations at each temperature
for SANN method. Default 10. Only used when \code{method = "SANN"}.}

\item{lmm}{Integer. Number of BFGS updates retained in L-BFGS-B method.
Default 5. Only used when \code{method = "L-BFGS-B"}.}

\item{factr}{Numeric. Convergence tolerance factor for L-BFGS-B method.
Convergence occurs when the reduction in the objective is within this
factor of the machine tolerance. Default 1e7 (tolerance ~1e-8).
Only used when \code{method = "L-BFGS-B"}.}

\item{pgtol}{Numeric. Tolerance on the projected gradient for L-BFGS-B method.
Default 0 (check suppressed). Only used when \code{method = "L-BFGS-B"}.}

\item{REPORT}{Integer. Frequency of progress reports for BFGS, L-BFGS-B and SANN
methods when \code{trace > 0}. Default 10 for BFGS/L-BFGS-B, 100 for SANN.}

\item{fnscale}{Numeric. Overall scaling to be applied to the function value
and gradient during optimization. Default 1. If negative, turns the problem
into a maximization problem.}

\item{parscale}{Numeric vector. Scaling values for parameters. Optimization
is performed on par/parscale. Default \code{rep(1, n_params)}.}

\item{ndeps}{Numeric vector. Step sizes for finite-difference approximation
to the gradient. Default 1e-3.}

\item{...}{Additional arguments passed to the optimizer. Allows fine-grained
control without formally adding parameters. Advanced users only.}

\item{x}{An object of class \code{"gkw_control"}.}
}
\value{
An object of class \code{"gkw_control"}, which is a list containing
all control parameters with validated and default-filled values. This object
is passed to \code{gkwreg()} via the \code{control} argument.
}
\description{
Auxiliary function for controlling \code{gkwreg()} fitting process.
This function consolidates all technical/advanced fitting options in one place,
keeping the main \code{gkwreg()} interface clean and user-friendly. Follows
the same design pattern as \code{\link[stats]{glm.control}},
\code{\link[betareg]{betareg.control}}, and similar control functions in R.
}
\details{
This function provides a centralized way to set all technical parameters
for model fitting. It serves several purposes:
\itemize{
\item \strong{Clean interface}: \code{gkwreg()} has fewer arguments
\item \strong{Organized documentation}: All technical options documented here
\item \strong{Input validation}: Parameters validated before fitting
\item \strong{Extensibility}: New options can be added without changing \code{gkwreg()}
\item \strong{Backward compatibility}: Old code continues working
}

\strong{Method-specific parameters:}

Each optimization method accepts different control parameters:
\itemize{
\item \strong{Nelder-Mead}: \code{alpha}, \code{beta}, \code{gamma}, \code{maxit},
\code{reltol}, \code{abstol}, \code{trace}, \code{REPORT}, \code{warn.1d.NelderMead}
\item \strong{BFGS}: \code{maxit}, \code{reltol}, \code{abstol}, \code{trace}, \code{REPORT}
\item \strong{CG}: \code{type}, \code{maxit}, \code{reltol}, \code{abstol}, \code{trace}
\item \strong{SANN}: \code{temp}, \code{tmax}, \code{maxit}, \code{trace}, \code{REPORT}
\item \strong{L-BFGS-B}: \code{lmm}, \code{factr}, \code{pgtol}, \code{trace}, \code{REPORT}
}

\strong{When to use gkw_control():}

Most users never need to adjust these settings. Use \code{gkw_control()} when:
\itemize{
\item Optimization fails to converge (increase \code{maxit}, adjust tolerances)
\item Debugging fit problems (set \code{silent = FALSE}, \code{trace = 1})
\item Comparing optimizers (try \code{method = "BFGS"} vs \code{"nlminb"})
\item Fine-tuning performance (disable \code{hessian} if SEs not needed)
\item Using custom starting values (\code{start = list(...)})
}

\strong{Recommended practices:}
\itemize{
\item Start with defaults, only adjust if needed
\item Increase \code{maxit} before adjusting tolerances
\item Use \code{trace = 1} to diagnose convergence issues
\item Disable \code{hessian} for speed if only point estimates needed
\item Try different \code{method}s if one fails (BFGS often more robust)
\item For L-BFGS-B with bounds, adjust \code{factr} and \code{pgtol} if needed
}
}
\examples{
\donttest{
# Default control (used automatically if not specified)
ctrl <- gkw_control()
print(ctrl)

# Increase iterations for difficult problem
ctrl_robust <- gkw_control(maxit = 1000, trace = 1)

# Try alternative optimizer
ctrl_bfgs <- gkw_control(method = "BFGS")

# Fast fitting without standard errors
ctrl_fast <- gkw_control(hessian = FALSE)

# Verbose debugging
ctrl_debug <- gkw_control(silent = FALSE, trace = 2)

# Custom starting values
ctrl_start <- gkw_control(
  start = list(
    alpha = c(0.5, 0.2),
    beta = c(1.0, -0.3)
  )
)

# Configure Nelder-Mead with custom reflection/contraction
ctrl_nm <- gkw_control(
  method = "Nelder-Mead",
  alpha = 1.5,
  beta = 0.75
)

# Configure L-BFGS-B for bounded optimization
ctrl_lbfgsb <- gkw_control(
  method = "L-BFGS-B",
  factr = 1e6,
  lmm = 10
)

# Configure SANN for rough surfaces
ctrl_sann <- gkw_control(
  method = "SANN",
  temp = 20,
  tmax = 20,
  maxit = 20000
)
}

}
\references{
Nocedal, J., & Wright, S. J. (2006). \emph{Numerical Optimization} (2nd ed.).
Springer.

Belisle, C. J. P. (1992). Convergence theorems for a class of simulated
annealing algorithms on R^d. \emph{Journal of Applied Probability}, 29, 885-895.

Byrd, R. H., Lu, P., Nocedal, J. and Zhu, C. (1995). A limited memory
algorithm for bound constrained optimization.
\emph{SIAM Journal on Scientific Computing}, 16, 1190-1208.
}
\seealso{
\code{\link{gkwreg}} for the main fitting function,
\code{\link[stats]{nlminb}}, \code{\link[stats]{optim}} for optimizer details,
\code{\link[betareg]{betareg.control}} for similar design pattern.
}
\author{
Lopes, J. E.
}
