% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/R6Model.R
\name{Model}
\alias{Model}
\title{A GLMM Model}
\description{
A GLMM Model

A GLMM Model
}
\details{
An R6 class representing a GLMM model


For the generalised linear mixed model 

\deqn{Y \sim F(\mu,\sigma)}
\deqn{\mu = h^-1(X\beta + Z\gamma)}
\deqn{\gamma \sim MVN(0,D)}

where h is the link function. A Model in comprised of a \link[glmmrBase]{MeanFunction} object, which defines the family F, 
link function h, and fixed effects design matrix X, and a \link[glmmrBase]{Covariance} object, which defines Z and D. The class provides
methods for analysis and simulation with these models.

This class provides methods for generating the matrices described above and data simulation, and serves as a base for extended functionality 
in related packages.

The class by default calculates the covariance matrix of the observations as:

\deqn{\Sigma = W^{-1} + ZDZ^T}

where _W_ is a diagonal matrix with the WLS iterated weights for each observation equal
to, for individual _i_ \eqn{\phi a_i v(\mu_i)[h'(\mu_i)]^2} (see Table 2.1 in McCullagh 
and Nelder (1989) <ISBN:9780412317606>). For very large designs, this can be disabled as
the memory requirements can be prohibitive. 

See \href{https://github.com/samuel-watson/glmmrBase/blob/master/README.md}{glmmrBase} for a 
detailed guide on model specification.


Calls the respective print methods of the linked covariance and mean function objects.


The matrices X and Z both have n rows, where n is the number of observations in the model/design.


**Number of clusters**
Returns a data frame describing the number of independent clusters or groups at each level in the design. For example,
if there were cluster-periods nested in clusters, then the top level would be clusters, and the second level would be 
cluster periods.


Using `update_parameters()` is the preferred way of updating the parameters of the 
mean or covariance objects as opposed to direct assignment, e.g. `self$covariance$parameters <- c(...)`. 
The function calls check functions to automatically update linked matrices with the new parameters.
If using direct assignment, call `self$check()` afterwards.
}
\examples{

## ------------------------------------------------
## Method `Model$new`
## ------------------------------------------------

#create a data frame describing a cross-sectional parallel cluster
#randomised trial
df <- nelder(~(cl(10)*t(5)) > ind(10))
df$int <- 0
df[df$cl > 5, 'int'] <- 1

mf1 <- MeanFunction$new(
  formula = ~ factor(t) + int - 1,
  data=df
)
cov1 <- Covariance$new(
  data = df,
  formula = ~ (1|gr(cl)) + (1|gr(cl*t))
)
des <- Model$new(
  covariance = cov1,
  mean = mf1,
  family = stats::gaussian(),
  var_par = 1
)

#alternatively we can pass the data directly to Model
#here we will specify a cohort study and provide parameter values
df <- nelder(~ind(20) * t(6))
df$int <- 0
df[df$t > 3, 'int'] <- 1

des <- Model$new(
  covariance = list(
    formula = ~ (1|gr(ind)),
    parameters = c(0.25)),
  mean = list(
    formula = ~ int,
    parameters = c(1,0.5)),
  data = df,
  family = stats::poisson())
                  
#an example of a spatial grid with two time points
df <- nelder(~ (x(10)*y(10))*t(2))
spt_design <- Model$new(covariance = list( formula = ~(1|fexp(x,y)*ar1(t))),
                         mean = list(formula = ~ 1),
                         data = df,
                         family = stats::gaussian()) 

## ------------------------------------------------
## Method `Model$n_cluster`
## ------------------------------------------------

df <- nelder(~(cl(10)*t(5)) > ind(10))
df$int <- 0
df[df$cl > 5, 'int'] <- 1

des <- Model$new(
  covariance = list(formula = ~ (1|gr(cl)) + (1|gr(cl*t))),
  mean = list(formula = ~ factor(t) + int - 1),
  data = df, 
  family = stats::gaussian(),
  var_par = 1
)
des$n_cluster() ## returns two levels of 10 and 50

## ------------------------------------------------
## Method `Model$subset_rows`
## ------------------------------------------------

#generate a stepped wedge design and remove the first sequence
des <- stepped_wedge(8,10,icc=0.05)
ids_to_keep <- which(des$mean_function$data$J!=1)
des$subset_rows(ids_to_keep)

## ------------------------------------------------
## Method `Model$subset_cols`
## ------------------------------------------------

#generate a stepped wedge design and remove first and last time periods
des <- stepped_wedge(8,10,icc=0.05)
des$subset_cols(c(2:8,10))

## ------------------------------------------------
## Method `Model$sim_data`
## ------------------------------------------------

df <- nelder(~(cl(10)*t(5)) > ind(10))
df$int <- 0
df[df$cl > 5, 'int'] <- 1
des <- Model$new(
  covariance = list(
    formula = ~ (1|gr(cl)*ar1(t)),
    parameters = c(0.25,0.8)),
  mean = list(
    formula = ~ factor(t) + int - 1,
    parameters = c(rep(0,5),0.6)),
  data = df,
  family = stats::binomial()
)
ysim <- des$sim_data()

## ------------------------------------------------
## Method `Model$check`
## ------------------------------------------------

df <- nelder(~(cl(10)*t(5)) > ind(10))
df$int <- 0
df[df$cl > 5, 'int'] <- 1
des <- Model$new(
  covariance = list(
    formula = ~ (1|gr(cl)*ar1(t)),
    parameters = c(0.25,0.8)),
  mean = list(
    formula = ~ factor(t) + int - 1,
    parameters = c(rep(0,5),0.6)),
  data = df, 
  family = stats::binomial()
)
des$check() #does nothing
des$covariance$parameters <- c(0.1,0.9)
des$check() #updates 
des$mean_function$parameters <- c(rnorm(5),0.1)
des$check() #updates 

## ------------------------------------------------
## Method `Model$update_parameters`
## ------------------------------------------------

df <- nelder(~(cl(10)*t(5)) > ind(10))
df$int <- 0
df[df$cl > 5, 'int'] <- 1
des <- Model$new(
  covariance = list(
    formula = ~ (1|gr(cl)*ar1(t))),
  mean = list(
    formula = ~ factor(t) + int - 1),
  data = df, 
  family = stats::binomial()
)
des$update_parameters(cov.pars = c(0.1,0.9))

## ------------------------------------------------
## Method `Model$power`
## ------------------------------------------------

df <- nelder(~(cl(10)*t(5)) > ind(10))
df$int <- 0
df[df$cl > 5, 'int'] <- 1
des <- Model$new(
  covariance = list(
    formula = ~ (1|gr(cl)) + (1|gr(cl*t)),
    parameters = c(0.25,0.1)),
  mean = list(
    formula = ~ factor(t) + int - 1,
    parameters = c(rep(0,5),0.6)),
  data = df, 
  family = stats::gaussian(),
  var_par = 1
)
des$power() #power of 0.90 for the int parameter
}
\seealso{
\link[glmmrBase]{nelder}, \link[glmmrBase]{MeanFunction}, \link[glmmrBase]{Covariance}
}
\section{Public fields}{
\if{html}{\out{<div class="r6-fields">}}
\describe{
\item{\code{covariance}}{A \link[glmmrBase]{Covariance} object defining the random effects covariance.}

\item{\code{mean_function}}{A \link[glmmrBase]{MeanFunction} object, defining the mean function for the model, including the data and covariate design matrix X.}

\item{\code{exp_condition}}{A vector indicting the unique experimental conditions for each observation, see Details.}

\item{\code{Sigma}}{The overall covariance matrix for the observations. Calculated and updated automatically as \eqn{W^{-1} + ZDZ^T} where W is an n x n 
diagonal matrix with elements on the diagonal equal to the GLM iterated weights. See Details.}

\item{\code{var_par}}{Scale parameter required for some distributions (Gaussian, Gamma, Beta).}
}
\if{html}{\out{</div>}}
}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-Model-fitted}{\code{Model$fitted()}}
\item \href{#method-Model-new}{\code{Model$new()}}
\item \href{#method-Model-print}{\code{Model$print()}}
\item \href{#method-Model-n}{\code{Model$n()}}
\item \href{#method-Model-n_cluster}{\code{Model$n_cluster()}}
\item \href{#method-Model-subset_rows}{\code{Model$subset_rows()}}
\item \href{#method-Model-subset_cols}{\code{Model$subset_cols()}}
\item \href{#method-Model-sim_data}{\code{Model$sim_data()}}
\item \href{#method-Model-check}{\code{Model$check()}}
\item \href{#method-Model-update_parameters}{\code{Model$update_parameters()}}
\item \href{#method-Model-information_matrix}{\code{Model$information_matrix()}}
\item \href{#method-Model-power}{\code{Model$power()}}
\item \href{#method-Model-clone}{\code{Model$clone()}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Model-fitted"></a>}}
\if{latex}{\out{\hypertarget{method-Model-fitted}{}}}
\subsection{Method \code{fitted()}}{
Return predicted values based on the currently stored parameter values in `mean_function`
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Model$fitted(type = "link")}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{type}}{One of either "`link`" for values on the scale of the link function, or "`response`" 
for values on the scale of the response}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A \link[Matrix]{Matrix} class object containing the predicted values
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Model-new"></a>}}
\if{latex}{\out{\hypertarget{method-Model-new}{}}}
\subsection{Method \code{new()}}{
Create a new Model object
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Model$new(
  covariance,
  mean,
  data = NULL,
  family = NULL,
  var_par = NULL,
  verbose = TRUE,
  skip.sigma = FALSE
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{covariance}}{Either a \link[glmmrBase]{Covariance} object, or an equivalent list of arguments
that can be passed to `Covariance` to create a new object. At a minimum the list must specify a formula.
If parameters are not included then they are initialised to 0.5.}

\item{\code{mean}}{Either a \link[glmmrBase]{MeanFunction} object, or an equivalent list of arguments
that can be passed to `MeanFunction` to create a new object. At a minimum the list must specify a formula.
If parameters are not included then they are initialised to 0.}

\item{\code{data}}{A data frame with the data required for the mean function and covariance objects. This argument
can be ignored if data are provided to the covariance or mean arguments either via `Covariance` and `MeanFunction`
object, or as a member of the list of arguments to both `covariance` and `mean`.}

\item{\code{family}}{A family object expressing the distribution and link function of the model, see \link[stats]{family}. This 
argument is optional if the family is provided either via a `MeanFunction` or `MeanFunction`
objects, or as members of the list of arguments to `mean`.}

\item{\code{var_par}}{Scale parameter required for some distributions, including Gaussian. Default is NULL.}

\item{\code{verbose}}{Logical indicating whether to provide detailed output}

\item{\code{skip.sigma}}{Logical indicating whether to skip the creating of the covariance matrix Sigma. For 
very large designs with thousands of observations or more, the covariance matrix will be too big to 
fit in memory, so this option will prevent sigma being created.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A new Model class object
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{#create a data frame describing a cross-sectional parallel cluster
#randomised trial
df <- nelder(~(cl(10)*t(5)) > ind(10))
df$int <- 0
df[df$cl > 5, 'int'] <- 1

mf1 <- MeanFunction$new(
  formula = ~ factor(t) + int - 1,
  data=df
)
cov1 <- Covariance$new(
  data = df,
  formula = ~ (1|gr(cl)) + (1|gr(cl*t))
)
des <- Model$new(
  covariance = cov1,
  mean = mf1,
  family = stats::gaussian(),
  var_par = 1
)

#alternatively we can pass the data directly to Model
#here we will specify a cohort study and provide parameter values
df <- nelder(~ind(20) * t(6))
df$int <- 0
df[df$t > 3, 'int'] <- 1

des <- Model$new(
  covariance = list(
    formula = ~ (1|gr(ind)),
    parameters = c(0.25)),
  mean = list(
    formula = ~ int,
    parameters = c(1,0.5)),
  data = df,
  family = stats::poisson())
                  
#an example of a spatial grid with two time points
df <- nelder(~ (x(10)*y(10))*t(2))
spt_design <- Model$new(covariance = list( formula = ~(1|fexp(x,y)*ar1(t))),
                         mean = list(formula = ~ 1),
                         data = df,
                         family = stats::gaussian()) 
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Model-print"></a>}}
\if{latex}{\out{\hypertarget{method-Model-print}{}}}
\subsection{Method \code{print()}}{
Print method for `Model` class
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Model$print()}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{...}}{ignored}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Model-n"></a>}}
\if{latex}{\out{\hypertarget{method-Model-n}{}}}
\subsection{Method \code{n()}}{
Returns the number of observations in the model
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Model$n(...)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{...}}{ignored}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Model-n_cluster"></a>}}
\if{latex}{\out{\hypertarget{method-Model-n_cluster}{}}}
\subsection{Method \code{n_cluster()}}{
Returns the number of clusters at each level
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Model$n_cluster()}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{...}}{ignored}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A data frame with the level, number of clusters, and variables describing each level.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{df <- nelder(~(cl(10)*t(5)) > ind(10))
df$int <- 0
df[df$cl > 5, 'int'] <- 1

des <- Model$new(
  covariance = list(formula = ~ (1|gr(cl)) + (1|gr(cl*t))),
  mean = list(formula = ~ factor(t) + int - 1),
  data = df, 
  family = stats::gaussian(),
  var_par = 1
)
des$n_cluster() ## returns two levels of 10 and 50
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Model-subset_rows"></a>}}
\if{latex}{\out{\hypertarget{method-Model-subset_rows}{}}}
\subsection{Method \code{subset_rows()}}{
Subsets the design keeping specified observations only

Given a vector of row indices, the corresponding rows will be kept and the 
other rows will be removed from the mean function and covariance
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Model$subset_rows(index)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{index}}{Integer or vector integers listing the rows to keep}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
The function updates the object and nothing is returned
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{#generate a stepped wedge design and remove the first sequence
des <- stepped_wedge(8,10,icc=0.05)
ids_to_keep <- which(des$mean_function$data$J!=1)
des$subset_rows(ids_to_keep)
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Model-subset_cols"></a>}}
\if{latex}{\out{\hypertarget{method-Model-subset_cols}{}}}
\subsection{Method \code{subset_cols()}}{
Subsets the columns of the design 

Removes the specified columns from the linked mean function object's X matrix.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Model$subset_cols(index)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{index}}{Integer or vector of integers specifying the indexes of the columns to keep}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
The function updates the object and nothing is returned
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{#generate a stepped wedge design and remove first and last time periods
des <- stepped_wedge(8,10,icc=0.05)
des$subset_cols(c(2:8,10))
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Model-sim_data"></a>}}
\if{latex}{\out{\hypertarget{method-Model-sim_data}{}}}
\subsection{Method \code{sim_data()}}{
Generates a realisation of the design

Generates a single vector of outcome data based upon the 
specified GLMM design
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Model$sim_data(type = "y")}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{type}}{Either 'y' to return just the outcome data, or 'data'
to return a data frame with the simulated outcome data alongside the model data}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
Either a vector or a data frame
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{df <- nelder(~(cl(10)*t(5)) > ind(10))
df$int <- 0
df[df$cl > 5, 'int'] <- 1
des <- Model$new(
  covariance = list(
    formula = ~ (1|gr(cl)*ar1(t)),
    parameters = c(0.25,0.8)),
  mean = list(
    formula = ~ factor(t) + int - 1,
    parameters = c(rep(0,5),0.6)),
  data = df,
  family = stats::binomial()
)
ysim <- des$sim_data()
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Model-check"></a>}}
\if{latex}{\out{\hypertarget{method-Model-check}{}}}
\subsection{Method \code{check()}}{
Checks for any changes in linked objects and updates.

Checks for any changes in any object and updates all linked objects if
any are detected. Generally called automatically.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Model$check(verbose = TRUE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{verbose}}{Logical indicating whether to report if any updates are made, defaults to TRUE}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
Linked objects are updated by nothing is returned
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{df <- nelder(~(cl(10)*t(5)) > ind(10))
df$int <- 0
df[df$cl > 5, 'int'] <- 1
des <- Model$new(
  covariance = list(
    formula = ~ (1|gr(cl)*ar1(t)),
    parameters = c(0.25,0.8)),
  mean = list(
    formula = ~ factor(t) + int - 1,
    parameters = c(rep(0,5),0.6)),
  data = df, 
  family = stats::binomial()
)
des$check() #does nothing
des$covariance$parameters <- c(0.1,0.9)
des$check() #updates 
des$mean_function$parameters <- c(rnorm(5),0.1)
des$check() #updates 
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Model-update_parameters"></a>}}
\if{latex}{\out{\hypertarget{method-Model-update_parameters}{}}}
\subsection{Method \code{update_parameters()}}{
Updates the parameters of the mean function and/or the covariance function
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Model$update_parameters(mean.pars = NULL, cov.pars = NULL, verbose = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{mean.pars}}{(Optional) Vector of new mean function parameters}

\item{\code{cov.pars}}{(Optional) Vector of new covariance function(s) parameters}

\item{\code{verbose}}{Logical indicating whether to provide more detailed feedback}
}
\if{html}{\out{</div>}}
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{df <- nelder(~(cl(10)*t(5)) > ind(10))
df$int <- 0
df[df$cl > 5, 'int'] <- 1
des <- Model$new(
  covariance = list(
    formula = ~ (1|gr(cl)*ar1(t))),
  mean = list(
    formula = ~ factor(t) + int - 1),
  data = df, 
  family = stats::binomial()
)
des$update_parameters(cov.pars = c(0.1,0.9))
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Model-information_matrix"></a>}}
\if{latex}{\out{\hypertarget{method-Model-information_matrix}{}}}
\subsection{Method \code{information_matrix()}}{
Generates the information matrix
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Model$information_matrix()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
A PxP matrix
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Model-power"></a>}}
\if{latex}{\out{\hypertarget{method-Model-power}{}}}
\subsection{Method \code{power()}}{
Estimates the power of the design described by the model using the square root
of the relevant element of the GLS variance matrix:

 \deqn{(X^T\Sigma^{-1}X)^{-1}}
 
Note that this is equivalent to using the "design effect" for many
models.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Model$power(alpha = 0.05)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{alpha}}{Numeric between zero and one indicating the type I error rate. 
Default of 0.05.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A data frame describing the parameters, their values, expected standard
errors and estimated power.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{df <- nelder(~(cl(10)*t(5)) > ind(10))
df$int <- 0
df[df$cl > 5, 'int'] <- 1
des <- Model$new(
  covariance = list(
    formula = ~ (1|gr(cl)) + (1|gr(cl*t)),
    parameters = c(0.25,0.1)),
  mean = list(
    formula = ~ factor(t) + int - 1,
    parameters = c(rep(0,5),0.6)),
  data = df, 
  family = stats::gaussian(),
  var_par = 1
)
des$power() #power of 0.90 for the int parameter
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Model-clone"></a>}}
\if{latex}{\out{\hypertarget{method-Model-clone}{}}}
\subsection{Method \code{clone()}}{
The objects of this class are cloneable with this method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Model$clone(deep = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{deep}}{Whether to make a deep clone.}
}
\if{html}{\out{</div>}}
}
}
}
