#' @import graphics
#'
#' @title plot method for class 'gmvarpred' objects
#'
#' @description \code{plot.gmvarpred} is plot method for gmvarpred objects.
#'
#' @inheritParams predict.gmvar
#' @param x object of class \code{'gmvarpred'} generated by \code{predict.gmvar}.
#' @param add_grid should grid be added to the plots?
#' @param ... arguments passed to \code{grid} which plots grid to the figure.
#' @details This method is used plot forecasts of GMVAR processes
#' @inherit in_paramspace_int references
#' @export

plot.gmvarpred <- function(x, ..., nt, mix_weights=TRUE, add_grid=TRUE) {
  old_par <- par(no.readonly=TRUE)
  on.exit(par(old_par))
  gmvarpred <- x
  data <- as.ts(gmvarpred$gmvar$data)
  n_obs <- nrow(data)
  d <- ncol(data)
  q <- gmvarpred$q
  M <- gmvarpred$gmvar$model$M
  mixing_weights <- gmvarpred$gmvar$mixing_weights
  n_mix <- nrow(mixing_weights)
  mix_weights <- mix_weights & M > 1 # Don't plot mixing weights if M == 1
  if(missing(nt)) {
    nt <- round(nrow(data)*0.15)
  } else {
    stopifnot(nt > 0 & nt %% 1 == 0)
    if(nt > nrow(data)) {
      warning("nt > nrow(data), using nt = nrow(data)")
      nt <- nrow(data)
    }
  }
  if(mix_weights) {
    par(mfrow=c(d + 1, 1), mar=c(2.5, 2.5, 2.1, 1))
  } else {
    par(mfrow=c(d, 1), mar=c(2.5, 2.5, 2.1, 1))
  }
  make_ts <- function(x, mix=FALSE) { # Make ts that has the first value the same as the last value of the observed series/estim. m.weights.
    if(mix) {
      last_obs <- mixing_weights[n_mix,]
    } else {
      last_obs <- data[n_obs,]
    }
    ts(rbind(last_obs, x), start=time(data)[n_obs], frequency=frequency(data))
  }
  ts_pred <- make_ts(gmvarpred$pred)
  ts_mix_pred <- make_ts(gmvarpred$mix_pred, mix=TRUE)
  ts_dat <- ts(data[(n_obs - nt + 1):n_obs,], start=time(data)[n_obs - nt + 1], frequency=frequency(data))
  ts_mix <- ts(mixing_weights[(n_mix - nt + 1):n_mix,], start=time(data)[n_obs - nt + 1], frequency=frequency(data))
  t0 <- time(ts_pred)
  ts_names <- attributes(data)$dimnames[[2]]
  reg_names <- attributes(gmvarpred$mix_pred)$dimnames[[2]]
  if(gmvarpred$pi_type != "none") {
    pred_ints <- aperm(gmvarpred$pred_ints, perm=c(1, 3, 2)) # [step, series, quantiles]
    mix_pred_ints <- aperm(gmvarpred$mix_pred_ints, perm=c(1, 3, 2)) # [step, series, quantiles]
  }

  # All values to indicate ylims
  if(gmvarpred$pi_type == "none") {
    all_val <- lapply(1:d, function(j) c(ts_dat[,j], ts_pred[,j]))
  } else {
    all_val <- lapply(1:d, function(j) c(ts_dat[,j], ts_pred[,j], simplify2array(pred_ints, higher=TRUE)[, j, ]))
  }

  # Prediction intervals, we lapply through quantiles [, , q]
  ts_fun_fact <- function(inds) function(pred_ints, mix=FALSE) lapply(inds, function(i1) make_ts(pred_ints[, , i1], mix))
  if(gmvarpred$pi_type == "two-sided") {
    ts1_lapply <- ts_fun_fact(1:(length(q)/2)) # Lower bounds
    ts2_lapply <- ts_fun_fact((length(q)/2 + 1):length(q)) # Upper bounds

    ints1 <- pred_ints
    ints1_mix <- mix_pred_ints

  } else { # Lower or upper
    ts1_lapply <- function(pred_ints, mix=FALSE) lapply(1:length(q), function(i1) make_ts(pred_ints, mix)[-1,]) # Lower or upper bound (dummy)
    ts2_lapply <- ts_fun_fact(seq_along(q))

    myfun <- ifelse(gmvarpred$pi_type == "upper", min, max)
    myfun2 <- ifelse(gmvarpred$pi_type == "upper", function(x) x - 3, function(x) x + 3)
    ints1 <- vapply(1:d, function(j) rep(myfun2(round(myfun(all_val[[j]]))), times=nrow(ts_pred)), numeric(nrow(ts_pred)))
    ints1_mix <- matrix(ifelse(gmvarpred$pi_type == "upper", 0, 1), nrow=nrow(ts_pred), ncol=M)
  }

  ts1 <- ts1_lapply(ints1)
  ts2 <- ts2_lapply(pred_ints)

  if(mix_weights) {
    ts1_mix <- ts1_lapply(ints1_mix, mix=TRUE)
    ts2_mix <- ts2_lapply(mix_pred_ints, mix=TRUE)
  }

  # Plot forecasts for the series
  draw_poly <- function(ts1_or_ts2, pred_ts, col) polygon(x=c(t0, rev(t0)), y=c(ts1_or_ts2, rev(pred_ts)), col=col, border=NA)
  col_pred <- grDevices::rgb(0, 0, 1, 0.2)
    for(i1 in 1:d) {
    ts.plot(ts_dat[,i1], ts_pred[,i1], gpars=list(col=c("black", "blue"), lty=1:2, ylim=c(floor(min(all_val[[i1]])), ceiling(max(all_val[[i1]]))), main=ts_names[i1]))
    if(add_grid) grid(...)
    if(gmvarpred$pi_type %in% c("two-sided", "upper", "lower")) {
      for(i2 in 1:length(gmvarpred$pi)) {
        draw_poly(ts1[[i2]][,i1], ts_pred[,i1], col=col_pred)
        draw_poly(ts2[[i2]][,i1], ts_pred[,i1], col=col_pred)
      }
    }
  }

  # Plot forecasts for the mixing weights
  if(mix_weights) {

    # Point forecasts
    colpal_mw <- grDevices::colorRampPalette(c("blue", "turquoise1", "green", "red"))(M)
    colpal_mw2 <- grDevices::adjustcolor(colpal_mw, alpha.f=0.5)
    ts.plot(ts_mix, ts_mix_pred, gpars=list(col=c(colpal_mw2, colpal_mw), ylim=c(0, 1), lty=c(rep(1, M), rep(2, M)), main="Mixing weights"))
    legend("topleft", legend=paste0("regime ", 1:M), bty="n", col=colpal_mw, lty=1, lwd=2,
           text.font=2, cex=0.9, x.intersp=0.5, y.intersp=1)
    if(add_grid) grid(...)

    # Individual prediction intervals as for the mixing weights
    colpal_mw3 <- grDevices::adjustcolor(colpal_mw, alpha.f=0.2)
    if(gmvarpred$pi_type %in% c("two-sided", "upper", "lower")) {
      for(m in 1:M) { # Go through regimes
        for(i2 in 1:length(gmvarpred$pi)) { # Go through the prediction intervals
          draw_poly(ts1_mix[[i2]][,m], ts_mix_pred[,m], col=colpal_mw3[m])
          draw_poly(ts2_mix[[i2]][,m], ts_mix_pred[,m], col=colpal_mw3[m])
        }
      }
    }

  }

  invisible(gmvarpred)
}


#' @describeIn GIRF plot method
#' @inheritParams print.girf
#' @param add_grid should grid be added to the plots?
#' @param ... arguments passed to \code{grid} which plots grid to the figure.
#' @inherit in_paramspace_int references
#' @export

plot.girf <- function(x, add_grid=FALSE, ...) {
  old_par <- par(no.readonly=TRUE)
  on.exit(par(old_par))

  girf <- x
  girf_res <- girf$girf_res
  n_resp <- ncol(girf_res[[1]]$point_est)
  resp_names <- colnames(girf_res[[1]]$point_est)
  n_girf <- length(girf_res)
  par(las=1)
  layoutmat <- matrix(seq_len(n_resp*n_girf), ncol=n_girf, byrow=FALSE)
  layout(layoutmat)

  # Function to plot the GIRF for each response separately
  plot_girf <- function(resp_ind, main="", xaxt="n", ylab="") {

    # Plot point estimate
    point_est <- girf_i1$point_est[, resp_ind]
    conf_ints <- girf_i1$conf_ints[, , resp_ind]
    plot(x=0:(length(point_est) - 1), y=point_est, type="l", ylim=c(min(0, min(conf_ints)), max(0, max(conf_ints))),
         main=main, ylab="", xlab="", xaxt=xaxt, lwd=2, col="blue")
    title(ylab=ylab, line=4, cex.lab=1, font.lab=2)
    if(add_grid) grid(...)

    # Plot confidence intervals
    inds <- 0:girf$N
    draw_poly <- function(up_or_low) polygon(x=c(inds, rev(inds)), y=c(up_or_low, rev(point_est)), col=grDevices::rgb(0, 0, 1, 0.2), border=NA)

    for(i1 in 1:length(girf$ci)) {
      draw_poly(conf_ints[, i1]) # lower
      draw_poly(conf_ints[, ncol(conf_ints) + 1 - i1]) # upper
    }

    abline(h=0, lty=3, col="red")
  }

  # Loop through the variables
  for(i1 in 1:n_girf) {
    girf_i1 <- girf_res[[i1]]

    # Plot the GIRF of variable i1
    par(mar=c(0, 5, 3, 2))
    plot_girf(resp_ind=1, main=paste(names(girf_res)[i1], "shock"), ylab=resp_names[1])
    par(mar=c(0, 5, 0, 2))
    for(i2 in 2:(n_resp - 1)) {
      plot_girf(resp_ind=i2, ylab=resp_names[i2])
    }
    par(mar=c(2.6, 5, 0, 2))
    plot_girf(resp_ind=n_resp, xaxt="s", ylab=resp_names[n_resp])
  }
}


#' @import graphics
#' @describeIn GMVAR plot method for class 'gmvar'
#' @param x object of class \code{'gmvar'} generated by \code{fitGMVAR} or \code{GMVAR}.
#' @param ... currectly not used.
#' @details The first plot displays the time series together with estimated mixing weights.
#'   The second plot displays (Gaussian) kernel density estimates of the individual series
#'   together with the marginal stationary density implied by the model. The colored regimewise
#'   stationary densities are multiplied with the mixing weight parameter estimates.
#' @export

plot.gmvar <- function(x, ...) {
  gmvar <- x
  check_null_data(gmvar)
  data <- as.ts(gmvar$data)
  n_obs <- nrow(data)
  p <- gmvar$model$p
  M <- gmvar$model$M
  d <- ncol(data)
  ts_mw <- ts(rbind(matrix(NA, nrow=p, ncol=M), gmvar$mixing_weights),
              start=start(data), frequency=frequency(data)) # First p observations are starting values

  # Time series and mixing weights
  old_par <- par(no.readonly=TRUE)
  on.exit(par(old_par))
  graphics::par(mfrow=c(2, 1), mar=c(2.5, 2.5, 2.1, 1))
  colpal_ts <- grDevices::colorRampPalette(c("darkgreen", "darkblue", "darkmagenta", "red3"))(d)
  colpal_mw <- grDevices::colorRampPalette(c("blue", "turquoise1", "green", "red"))(M)
  names_ts <- colnames(data)
  names_mw <- paste0("mix.comp.", 1:M)
  draw_legend <- function(nams, cols) {
    legend("topleft", legend=nams, bty="n", col=cols, lty=1, lwd=2, text.font=2, cex=0.6, x.intersp=0.5, y.intersp=1)
  }

  ts.plot(data, gpars=list(main="Time series", col=colpal_ts, lty=1:d))
  draw_legend(names_ts, cols=colpal_ts)
  ts.plot(ts_mw, gpars=list(main="Mixing weights", ylim=c(0, 1), col=colpal_mw, lty=2))
  draw_legend(names_mw, cols=colpal_mw)

  # Marginal stationary distributions
  grDevices::devAskNewPage(TRUE)
  nrows <- max(ceiling(log2(d) - 1), 1)
  ncols <- ceiling(d/nrows)
  graphics::par(mfrow=c(nrows, ncols), mar=c(2.5, 2.5, 2.1, 1))

  alphas <- pick_alphas(p=p, M=M, d=d, params=gmvar$params)
  means <- get_regime_means(gmvar) # Regs in cols, d in rows
  vars <- vapply(1:M, function(m) diag(get_regime_autocovs(gmvar)[, , 1, m]), numeric(d)) # Regs in cols, d in rows
  reg_dens <- function(d1, m, xx) alphas[m]*dnorm(xx, mean=means[d1, m], sd=sqrt(vars[d1, m])) # Marginal stat dens of d1:th series and m:th regime multiplied with mix weight param
  d1_dens_f <- function(d1, xx) rowSums(vapply(1:M, function(m) reg_dens(d1, m, xx), numeric(length(xx)))) # Marginal stat dens of d1:th series

  for(d1 in 1:d) {
    data_dens <- density(data[,d1], kernel="gaussian")
    mod_mean <- gmvar$uncond_moments$uncond_mean[d1]
    mod_sd <- sqrt(gmvar$uncond_moments$autocovs[d1, d1, 1])
    x0 <- min(mod_mean - 3*mod_sd, min(data_dens$x))
    x1 <- max(mod_mean + 3*mod_sd, max(data_dens$x))
    xpp <- seq(from=x0, to=x1, length.out=500)
    mod_dens <- d1_dens_f(d1=d1, xx=xpp)
    y0 <- 0
    y1 <- max(c(data_dens$y, mod_dens))

    # Plot the densities
    plot(x=data_dens$x, y=data_dens$y, xlim=c(x0, x1), ylim=c(y0, y1), main=paste("Density:", names_ts[d1]),
         ylab="", xlab="", cex.axis=0.8, font.axis=2, type="l")
    lines(x=xpp, y=mod_dens, type="l", lty=2, lwd=2, col="darkgrey")
    for(m in 1:M) {
      lines(x=xpp, y=reg_dens(d1=d1, m=m, xx=xpp), type="l", lty=3, col=colpal_mw[m])
    }
    if(d1 == 1) draw_legend(names_mw, cols=colpal_mw)
  }
}



#' @import graphics
#' @describeIn quantile_residual_tests Plot p-values of the autocorrelation and conditional
#'  heteroskedasticity tests.
#' @inheritParams print.qrtest
#' @export

plot.qrtest <- function(x, ...) {
  old_par <- par(no.readonly=TRUE)
  on.exit(par(old_par))
  qrtest <- x
  par(mfrow=c(1, 2), mar=c(5.1, 3.1, 3.1, 1.1))

  plot_pvalues <- function(which_ones) { # ac_res, ch_res
    res <- qrtest[[which(names(qrtest) == which_ones)]]
    pvals <- res$test_results$p_val
    seq_pvals <- seq_along(pvals)
    plot(pvals, ylim=c(0, 1), xlim=c(min(seq_pvals) - 0.2, max(seq_pvals) + 0.2), ylab="", xlab="lags",
         main=ifelse(which_ones == "ac_res", "Autocorrelation", "Cond. h.skedasticity"),
         xaxt="n", yaxt="n", pch=16, col="blue")
    axis(side=1, at=seq_pvals, labels=res$test_results$lags)
    levels <- c(0.01, 0.05, 0.10, seq(from=0.20, to=1.00, by=0.20))
    axis(side=2, at=levels, las=1, cex.axis=0.8)
    abline(h=0, lwd=2)
    abline(h=c(0.01, 0.05, 0.10, 1.00), lty=2, col=c(rep("red", 3), "green4"))
    segments(x0=seq_pvals, y0=0, y1=pvals, x1=seq_pvals, ...)
    points(pvals)
  }

  plot_pvalues("ac_res")
  plot_pvalues("ch_res")
}

