% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simulateGMVAR.R
\name{simulateGMVAR}
\alias{simulateGMVAR}
\title{Simulate from GMVAR process}
\usage{
simulateGMVAR(
  gmvar,
  nsimu,
  init_values = NULL,
  ntimes = 1,
  drop = TRUE,
  seed = NULL,
  girf_pars = NULL
)
}
\arguments{
\item{gmvar}{an object of class \code{'gmvar'} created with \code{fitGMVAR} or \code{GMVAR}.}

\item{nsimu}{number of observations to be simulated.}

\item{init_values}{a size \eqn{(pxd)} matrix specifying the initial values to be used in the simulation, where
d is the number of time series in the system.
The \strong{last} row will be used as initial values for the first lag, the second last row for second lag etc. If not
specified, initial values will be drawn from the stationary distribution of the process.}

\item{ntimes}{how many sets of simulations should be performed?}

\item{drop}{if \code{TRUE} (default) then the components of the returned list are coerced to lower dimension if \code{ntimes==1}, i.e.,
\code{$sample} and \code{$mixing_weights} will be matrices, and \code{$component} will be vector.}

\item{seed}{set seed for the random number generator?}

\item{girf_pars}{This argument is used internally in the estimation of generalized impulse response functions (see \code{?GIRF}). You
should ignore it.}
}
\value{
If \code{drop==TRUE} and \code{ntimes==1} (default): \code{$sample}, \code{$component}, and \code{$mixing_weights} are matrices.
  Otherwise, returns a list with...
  \describe{
    \item{\code{$sample}}{a size (\code{nsimu}\eqn{ x d x }\code{ntimes}) array containing the samples: the dimension \code{[t, , ]} is
     the time index, the dimension \code{[, d, ]} indicates the marginal time series, and the dimension \code{[, , i]} indicates
     the i:th set of simulations.}
    \item{\code{$component}}{a size (\code{nsimu}\eqn{ x }\code{ntimes}) matrix containing the information from which mixture component each
     value was generated from.}
    \item{\code{$mixing_weights}}{a size (\code{nsimu}\eqn{ x M x }\code{ntimes}) array containing the mixing weights corresponding to the
     sample: the dimension \code{[t, , ]} is the time index, the dimension \code{[, m, ]} indicates the regime, and the dimension
     \code{[, , i]} indicates the i:th set of simulations.}
  }
}
\description{
\code{simulateGMVAR} simulates observations from a GMVAR process.
}
\details{
The argument \code{ntimes} is intended for forecasting: a GMVAR process can be forecasted by simulating its possible future values.
 One can easily perform a large number simulations and calculate the sample quantiles from the simulated values to obtain prediction
 intervals (see the forecasting example).
}
\examples{
 # GMVAR(1,2), d=2 process, initial values from the stationary
 # distribution
 params12 <- c(0.55, 0.112, 0.344, 0.055, -0.009, 0.718, 0.319, 0.005,
  0.03, 0.619, 0.173, 0.255, 0.017, -0.136, 0.858, 1.185, -0.012, 0.136,
  0.674)
 mod12 <- GMVAR(p=1, M=2, d=2, params=params12)
 set.seed(1)
 sim12 <- simulateGMVAR(mod12, nsimu=500)
 plot.ts(sim12$sample)
 ts.plot(sim12$mixing_weights, col=c("blue", "red"), lty=2)
 plot(sim12$component, type="l")

 # Structural GMVAR(2, 2), d=2 model identified with sign-constraints:
 params22s <- c(0.36, 0.121, 0.484, 0.072, 0.223, 0.059, -0.151, 0.395,
  0.406, -0.005, 0.083, 0.299, 0.218, 0.02, -0.119, 0.722, 0.093, 0.032,
   0.044, 0.191, 0.057, 0.172, -0.46, 0.016, 3.518, 5.154, 0.58)
 W_22 <- matrix(c(1, 1, -1, 1), nrow=2, byrow=FALSE)
 mod22s <- GMVAR(gdpdef, p=2, M=2, params=params22s,
  structural_pars=list(W=W_22))
 sim22s <- simulateGMVAR(mod22s, nsimu=100)
 plot.ts(sim22s$sample)

 ## FORECASTING EXAMPLE ##
 # Forecast 5-steps-ahead, 500 sets of simulations with initial
 # values from the data:
 # GMVAR(2,2), d=2 model
 params22 <- c(0.36, 0.121, 0.223, 0.059, -0.151, 0.395, 0.406, -0.005,
  0.083, 0.299, 0.215, 0.002, 0.03, 0.484, 0.072, 0.218, 0.02, -0.119,
   0.722, 0.093, 0.032, 0.044, 0.191, 1.101, -0.004, 0.105, 0.58)
 mod22 <- GMVAR(gdpdef, p=2, M=2, params=params22)
 sim22 <- simulateGMVAR(mod22, nsimu=5, ntimes=500)

 # Point forecast + 95\% prediction intervals:
 apply(sim22$sample, MARGIN=1:2, FUN=quantile, probs=c(0.025, 0.5, 0.972))

 # Similar forecast for the mixing weights:
 apply(sim22$mixing_weights, MARGIN=1:2, FUN=quantile,
       probs=c(0.025, 0.5, 0.972))
}
\references{
\itemize{
   \item Kalliovirta L., Meitz M. and Saikkonen P. 2016. Gaussian mixture vector autoregression.
         \emph{Journal of Econometrics}, \strong{192}, 485-498.
   \item Lütkepohl H. 2005. New Introduction to Multiple Time Series Analysis,
         \emph{Springer}.
   \item McElroy T. 2017. Computation of vector ARMA autocovariances.
         \emph{Statistics and Probability Letters}, \strong{124}, 92-96.
   \item Virolainen S. 2020. Structural Gaussian mixture vector autoregressive model. Unpublished working
         paper, available as arXiv:2007.04713.
 }
}
\seealso{
\code{\link{fitGMVAR}}, \code{\link{GMVAR}}, \code{\link{diagnostic_plot}}, \code{\link{predict.gmvar}},
 \code{\link{profile_logliks}}, \code{\link{quantile_residual_tests}}, \code{\link{GIRF}}, \code{\link{GFEVD}}
}
