% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gp_draw.R, R/gp_pred.R
\name{gp_draw}
\alias{gp_draw}
\alias{pred}
\alias{gp_pred}
\title{Make predictions with a GP model}
\usage{
gp_draw(
  gp,
  xnew,
  draws = NULL,
  transform = TRUE,
  target = FALSE,
  marginal = FALSE,
  cfind = NULL,
  jitter = NULL,
  seed = NULL,
  ...
)

gp_pred(
  gp,
  xnew,
  var = FALSE,
  quantiles = NULL,
  transform = FALSE,
  cfind = NULL,
  jitter = NULL,
  quad_order = 15,
  ...
)
}
\arguments{
\item{gp}{A GP model object.}

\item{xnew}{N-by-d matrix of input values (N is the number of test points and d
the input dimension).
Can also be a vector of length N if the model has only a single input.}

\item{draws}{Number of draws to generate from the predictive distribution for the
latent values.}

\item{transform}{Whether to transform the draws of latent values to the same scale
as the target y, that is, through the response (or inverse-link) function.}

\item{target}{If TRUE, draws values for the target variable \code{y} instead of the latent
function values.}

\item{marginal}{If TRUE, then draws for each test point are only marginally correct, but the
covariance structure between test points is not retained. However, this will make the sampling
considerably faster in some cases, and can be useful if one is interested only in looking
at the marginal predictive distributions for a large number of test locations
(for example, in posterior predictive checking).}

\item{cfind}{Indices of covariance functions to be used in the prediction. By default uses
all covariance functions.}

\item{jitter}{Magnitude of diagonal jitter for covariance matrices for numerical stability.
Default is 1e-6.}

\item{seed}{Random seed for draws.}

\item{...}{Additional parameters that might be needed. For example \code{offset} or 
keyword \code{trials} for binomial and beta-binomial likelihoods.}

\item{var}{Whether to compute the predictive variances along with predictive mean.}

\item{quantiles}{Vector of probabilities between 0 and 1 indicating which quantiles are to
be predicted.}

\item{quad_order}{Quadrature order in order to compute the mean and variance on
the transformed scale.}
}
\value{
\code{gp_pred} returns a list with fields giving the predictive mean, variance and
quantiles (the last two are computed only if requested). \code{gp_draw} returns an N-by-draws
matrix of random draws from the predictive distribution, where N is the number of test points.
}
\description{
Function \code{gp_pred} can be used to make analytic predictions for the latent function
values at test points, whereas \code{gp_draw}
can be used to draw from the predictive distribution (or from the prior if the GP has
not been fitted yet.)
}
\section{References}{


Rasmussen, C. E. and Williams, C. K. I. (2006). Gaussian processes for machine learning. MIT Press.
}

\examples{

# Generate some toy data
set.seed(1242)
n <- 50
x <- matrix(rnorm(n * 3), nrow = n)
f <- sin(x[, 1]) + 0.5 * x[, 2]^2 + x[, 3]
y <- f + 0.5 * rnorm(n)
x <- data.frame(x1 = x[, 1], x2 = x[, 2], x3 = x[, 3])

# More than one covariance function; one for x1 and x2, and another one for x3
cf1 <- cf_nn(c("x1", "x2"), prior_sigma0 = prior_half_t(df = 4, scale = 2))
cf2 <- cf_sexp("x3")
cfs <- list(cf1, cf2)
lik <- lik_gaussian()
gp <- gp_init(cfs, lik)
gp <- gp_optim(gp, x, y, maxiter = 500)

# plot the predictions with respect to x1, when x2 = x3 = 0
xt <- cbind(x1 = seq(-3, 3, len = 50), x2 = 0, x3 = 0)
pred <- gp_pred(gp, xt)
plot(xt[, "x1"], pred$mean, type = "l")

# draw from the predictive distribution
xt <- cbind(x1 = seq(-3, 3, len = 50), x2 = 0, x3 = 0)
draws <- gp_draw(gp, xt, draws = 100)
plot(xt[, "x1"], draws[, 1], type = "l")
for (i in 2:50) {
  lines(xt[, "x1"], draws[, i])
}

# plot effect with respect to x3 only
xt <- cbind("x3" = seq(-3, 3, len = 50))
pred <- gp_pred(gp, xt, cfind = 2)
plot(xt, pred$mean, type = "l")


}
