% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/all_growthmodels.R
\name{all_growthmodels}
\alias{all_growthmodels}
\alias{all_growthmodels.formula}
\alias{all_growthmodels.function}
\title{Fit Nonlinear Growth Models to Data Frame}
\usage{
all_growthmodels(...)

\method{all_growthmodels}{formula}(formula, data, p, lower = -Inf,
  upper = Inf, which = names(p), FUN = NULL, method = "Marq",
  transform = c("none", "log"), ..., subset = NULL,
  ncores = detectCores(logical = FALSE))

\method{all_growthmodels}{function}(FUN, p, data, grouping = NULL,
  time = "time", y = "value", lower = -Inf, upper = Inf,
  which = names(p), method = "Marq", transform = c("none", "log"), ...,
  ncores = detectCores(logical = FALSE))
}
\arguments{
\item{\dots}{generic parameters, including parameters passed to the optimizer.}

\item{formula}{model formula specifying dependent, independent and grouping
variables in the form:
\code{dependent ~ independent | group1 + group2 + \dots}.}

\item{data}{data frame of observational data.}

\item{p}{named vector of start parameters and initial values of the growth model.}

\item{lower}{lower bound of the parameter vector.}

\item{upper}{upper bound of the parameter vector.}

\item{which}{vector of parameter names that are to be fitted.}

\item{FUN}{function of growth model to be fitted.}

\item{method}{character vector specifying the optimization algorithm.}

\item{transform}{fit model to non-transformed or log-transformed data.}

\item{subset}{a specification of the rows to be used: defaults to all rows.}

\item{ncores}{number of CPU cores used for parallel computation. The number
of real cores is detected automatically by default,
but fort debugging purposes it could be wise to set \code{ncores = 1}.
Usage of logical (hyperthreading) cores does not speed up computation.}

\item{grouping}{vector of grouping variables defining subsets in the data frame.}

\item{time}{character vector with name of independent variable.}

\item{y}{character vector with name of dependent variable.}
}
\value{
object containing the parameters of all fits.
}
\description{
Determine maximum growth rates by nonlinear fits for
a series of experiments.
}
\examples{

data(bactgrowth)
splitted.data <- multisplit(value ~ time | strain + conc + replicate,
                 data = bactgrowth)

## show which experiments are in splitted.data
names(splitted.data)

## get table from single experiment
dat <- splitted.data[["D:0:1"]]

fit0 <- fit_spline(dat$time, dat$value)

fit1 <- all_splines(value ~ time | strain + conc + replicate,
                 data = bactgrowth, spar = 0.5)

\donttest{
## these examples require some CPU power and may take a bit longer

## initial parameters
p <- c(coef(fit0), K = max(dat$value))

## avoid negative parameters
lower = c(y0 = 0, mumax = 0, K = 0)

## fit all models
fit2 <- all_growthmodels(value ~ time | strain + conc + replicate,
          data = bactgrowth, FUN=grow_logistic,
          p = p, lower = lower, ncores = 2)

results1 <- results(fit1)
results2 <- results(fit2)
plot(results1$mumax, results2$mumax, xlab="smooth splines", ylab="logistic")

## experimental: nonlinear model as part of the formula

fit3 <- all_growthmodels(
          value ~ grow_logistic(time, parms) | strain + conc + replicate,
          data = bactgrowth, p = p, lower = lower, ncores = 2)

## this allows also to fit to the 'global' data set or any subsets
fit4 <- all_growthmodels(
          value ~ grow_logistic(time, parms),
          data = bactgrowth, p = p, lower = lower, ncores = 1)
plot(fit4)

fit5 <- all_growthmodels(
          value ~ grow_logistic(time, parms) | strain + conc,
          data = bactgrowth, p = p, lower = lower, ncores = 2)
plot(fit5)
}

}
\seealso{
Other fitting functions: \code{\link{all_easylinear}},
  \code{\link{all_splines}}, \code{\link{fit_easylinear}},
  \code{\link{fit_growthmodel}}, \code{\link{fit_spline}}
}
