% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/block_average.R
\name{block_average}
\alias{block_average}
\title{Calculate yearly means for event metrics.}
\usage{
block_average(data, x = t, y = temp, report = "full")
}
\arguments{
\item{data}{Accepts the data returned by the \code{\link{detect_event}} function.}

\item{x}{This column is expected to contain a vector of dates as per the
specification of \code{make_whole}. If a column headed \code{t} is present in
the dataframe, this argument may be ommitted; otherwise, specify the name of
the column with dates here.}

\item{y}{This is a column containing the measurement variable. If the column
name differs from the default (i.e. \code{temp}), specify the name here.}

\item{report}{Specify either \code{full} or \code{partial}. Selecting \code{full} causes
the report to contain NAs for any years in which no events were detected
(except for \code{count}, which will be zero in those years), while \code{partial}
reports only the years wherein events were detected. The default is \code{full}.}
}
\value{
The function will return a data frame of the averaged (or aggregate)
metrics. It includes the following:
  \item{year}{The year over which the metrics were averaged.}
  \item{temp_mean}{Seawater temperature for the specified year [deg. C].}
  \item{temp_min}{The minimum temperature for the specified year [deg. C].}
  \item{temp_max}{The maximum temperature for the specified year [deg. C].}
  \item{count}{The number of events per year.}
  \item{duration}{The average duration of events per year [days].}
  \item{intensity_mean}{The average event "mean intensity" in each year [deg. C].}
  \item{intensity_max}{The average event "maximum (peak) intensity" in each year
  [deg. C].}
  #'   \item{intensity_max_max}{The maximum event "maximum (peak) intensity" in
  each year [deg. C].}
  \item{intensity_var}{The average event "intensity variability" in each year
  [deg. C].}
  \item{intensity_cumulative}{The average event "cumulative intensity" in each year
  [deg. C x days].}
  \item{rate_onset}{Average event onset rate in each year [deg. C / days].}
  \item{rate_decline}{Average event decline rate in each year [deg. C / days].}
  \item{total_days}{Total number of events days in each year [days].}
  \item{total_icum}{Total cumulative intensity over all events in each year [deg. C x days].}

\code{intensity_max_relThresh}, \code{intensity_mean_relThresh},
\code{intensity_var_relThresh}, and \code{intensity_cumulative_relThresh}
are as above except relative to the threshold (e.g., 90th percentile) rather
than the seasonal climatology.

\code{intensity_max_abs}, \code{intensity_mean_abs}, \code{intensity_var_abs}, and
\code{intensity_cumulative_abs} are as above except as absolute magnitudes
rather than relative to the seasonal climatology or threshold.
}
\description{
Calculate yearly means for event metrics.
}
\details{
This function needs to be provided with the full output from the \code{detect}
function. Note that the yearly averages are calculted only for complete years
(i.e. years that start/end part-way through the year at the beginning or end
of the original time series are removed from the calculations).

This function differs from the python implementation of the function of the
same name (i.e., \code{blockAverage}, see \url{https://github.com/ecjoliver/marineHeatWaves})
in that we only provide the ability to calculate the average (or aggregate)
event metrics in 'blocks' of one year, while the python version allows
arbitrary (integer) block sizes.
}
\examples{
ts <- ts2clm(sst_WA, climatologyPeriod = c("1983-01-01", "2012-12-31"))
res <- detect_event(ts)
out <- block_average(res)
summary(glm(count ~ year, out, family = "poisson"))

library(ggplot2)

ggplot(data = out, aes(x = year, y = count)) +
  geom_point(colour = "salmon") +
  geom_line() +
  labs(x = NULL, y = "Number of events")

}
\references{
Hobday, A.J. et al. (2016), A hierarchical approach to defining
marine heatwaves, Progress in Oceanography, 141, pp. 227-238,
doi: 10.1016/j.pocean.2015.12.014
}
\author{
Albertus J. Smit, Eric C. J. Oliver
}
