% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/realizedMeasures.R
\name{rSemiCov}
\alias{rSemiCov}
\title{Realized semicovariance}
\usage{
rSemiCov(
  rData,
  cor = FALSE,
  alignBy = NULL,
  alignPeriod = NULL,
  makeReturns = FALSE
)
}
\arguments{
\item{rData}{an \code{xts} or \code{data.table} object containing returns or prices, possibly for multiple assets over multiple days.}

\item{cor}{boolean, in case it is \code{TRUE}, and the input data is multivariate, the correlation is returned instead of the covariance matrix. \code{FALSE} by default.}

\item{alignBy}{character, indicating the time scale in which \code{alignPeriod} is expressed. Possible values are: \code{"secs"}, \code{"seconds"}, \code{"mins"}, \code{"minutes"}, \code{"hours"}}

\item{alignPeriod}{positive numeric, indicating the number of periods to aggregate over. For example, to aggregate
based on a 5-minute frequency, set \code{alignPeriod = 5} and \code{alignBy = "minutes"}.}

\item{makeReturns}{boolean, should be \code{TRUE} when \code{rData} contains prices instead of returns. \code{FALSE} by default.}
}
\value{
In case the data consists of one day a list of five \eqn{N \times N} matrices are returned. These matrices are named \code{mixed}, \code{positive}, \code{negative}, \code{concordant}, and \code{rCov}.
The latter matrix corresponds to the realized covariance estimator and is thus named like the function \code{\link{rCov}}.
In case the data spans more than one day, the list for each day will be put into another list named according to the date of the estimates.
}
\description{
Calculate the Realized Semicovariances (rSemiCov).
Let \eqn{ r_{t,i} } be an intraday \eqn{M x N} return matrix and \eqn{i=1,...,M}
the number of intraday returns. Then, let \eqn{r_{t,i}^{+} = max(r_{t,i},0)} and \eqn{r_{t,i}^{-} = min(r_{t,i},0)}.

Then, the realized semicovariance is given by the following three matrices:

\deqn{
 \mbox{pos}_t =\sum_{i=1}^{M} r^{+}_{t,i} r^{+'}_{t,i}
}
\deqn{
 \mbox{neg}_t =\sum_{i=1}^{M} r^{-}_{t,i} r^{-'}_{t,i}
}
\deqn{
 \mbox{mixed}_t =\sum_{i=1}^{M} (r^{+}_{t,i} r^{-'}_{t,i} + r^{-}_{t,i} r^{+'}_{t,i})
}

The mixed covariance matrix will have 0 on the diagonal.
From these three matrices, the realized covariance can be constructed as \eqn{pos + neg + mixed}.
The concordant semicovariance matrix is \eqn{pos + neg}.
The off-diagonals of the concordant matrix is always positive, while for the mixed matrix, it is always negative.
}
\details{
In the case that cor is \code{TRUE}, the mixed matrix will be an \eqn{N \times N} matrix filled with NA as mapping the mixed covariance matrix into correlation space is impossible due to the 0-diagonal.
}
\examples{
# Realized semi-variance/semi-covariance for prices aligned
# at 5 minutes.

# Univariate:
rSV = rSemiCov(rData = sampleTData[, list(DT, PRICE)], alignBy = "minutes",
                   alignPeriod = 5, makeReturns = TRUE)
rSV
\dontrun{
library("xts")
# Multivariate multi day:
rSC <- rSemiCov(sampleOneMinuteData, makeReturns = TRUE) # rSC is a list of lists
# We extract the covariance between stock 1 and stock 2 for all three covariances.
mixed <- sapply(rSC, function(x) x[["mixed"]][1,2])
neg <- sapply(rSC, function(x) x[["negative"]][1,2])
pos <- sapply(rSC, function(x) x[["positive"]][1,2])
covariances <- xts(cbind(mixed, neg, pos), as.Date(names(rSC)))
colnames(covariances) <- c("mixed", "neg", "pos")
# We make a quick plot of the different covariances
plot(covariances)
addLegend(lty = 1) # Add legend so we can distinguish the series.
}
}
\references{
Bollerslev, T., Li, J., Patton, A. J., and Quaedvlieg, R. (2020). Realized semicovariances. \emph{Econometrica}, 88, 1515-1551.
}
\author{
Emil Sjoerup.
}
\keyword{volatility}
