#' @title Plot tau
#' @export
#' @family plot
#' @description Plot the rep-specific tau parameters
#'   of a fitted hierarchical model.
#' @return A `ggplot` object
#' @param mcmc Data frame of posterior samples generated by
#'   [hbl_mcmc_hierarchical()].
#' @examples
#' if (!identical(Sys.getenv("HBL_TEST", unset = ""), "")) {
#' set.seed(0)
#' data <- hbl_sim_independent(n_continuous = 2)$data
#' tmp <- utils::capture.output(
#'   suppressWarnings(
#'     mcmc <- hbl_mcmc_hierarchical(
#'       data,
#'       chains = 1,
#'       warmup = 10,
#'       iter = 20,
#'       seed = 0
#'     )
#'   )
#' )
#' hbl_plot_tau(mcmc)
#' }
hbl_plot_tau <- function(mcmc) {
  true(is.data.frame(mcmc))
  tau <- dplyr::select(mcmc, starts_with("tau"))
  true(
    ncol(tau) > 1,
    message = "tau not found. hbl_plot_tau() requires the hierarchical model."
  )
  tau$sample <- seq_len(nrow(tau))
  tau <- tidyr::pivot_longer(
    tau,
    -any_of("sample"),
    names_to = "Parameter",
    values_to = "Value"
  )
  ggplot2::ggplot(tau) +
    ggplot2::geom_histogram(
      ggplot2::aes(x = Value, y = ..density..),
      bins = 30
    ) +
    ggplot2::ylab("Density") +
    ggplot2::facet_wrap(~Parameter) +
    ggplot2::theme_gray(20)
}
