#This file is used by roxygen2 to generate man files (documentation) for data
#sets included in the package.


#'Reference tissue masses and flows from tables in McNally et al. 2014.
#'
#'Reference tissue masses, flows, and residual variance distributions from
#'Tables 1, 4, and 5 of McNally et al. 2014.
#'
#'@format A data.table with variables: \describe{\item{\code{tissue}}{Body
#'  tissue} \item{\code{gender}}{Gender: Male or Female}
#'  \item{\code{mass_ref}}{Reference mass in kg, from Reference Man}
#'  \item{\code{mass_cv}}{Coefficient of variation for mass}
#'  \item{\code{mass_dist}}{Distribution for mass: Normal or Log-normal}
#'  \item{\code{flow_ref}}{Reference flow in L/h, from Reference Man}
#'  \item{\code{flow_cv}}{Coefficient of variation for flow (all normally
#'  distributed)} \item{\code{height_ref}}{Reference heights (by gender)}
#'  \item{\code{CO_ref}}{Reference cardiac output by gender}
#'  \item{\code{flow_frac}}{Fraction of CO flowing to each tissue:
#'  \code{flow_ref}/\code{CO_ref}}}
#'@source McNally K, Cotton R, Hogg A, Loizou G. "PopGen: A virtual human
#'  population generator." Toxicology 315, 70-85, 2004.
#'@keywords data
#'@keywords httk-pop
#'
#'@author Caroline Ring
#'
#' @references 
#' \insertRef{ring2017identifying}{httk}
"mcnally_dt"

#' A timestamp of table creation
#'
#' The Tables.RData file is separately created as part of building a new
#' release of HTTK. This time stamp indicates the script used to build the file
#' and when it was run.
#'
#' @author John Wambaugh
"Tables.Rdata.stamp"

#' Reference for EPA Physico-Chemical Data
#'
#' The physico-chemical data in the chem.phys_and_invitro.data table are
#' obtained from EPA's Comptox Chemicals dashboard. This variable indicates
#' the date the Dashboard was accessed.
#' @source \url{https://comptox.epa.gov/dashboard}
#'
#' @author John Wambaugh
"EPA.ref"

#'Pre-processed NHANES data.
#'
#'NHANES data on demographics, anthropometrics, and some laboratory measures,
#'cleaned and combined into a single data set.
#'
#'@format A data.table with 23620 rows and 12
#'  variables.  \describe{ \item{seqn}{NHANES unique identifier for individual
#'  respondents.} \item{sddsrvyr}{NHANES two-year cycle: one of "NHANES
#'  2013-2014", "NHANES 2015-2016", "NHANES 2017-2018".} \item{riagendr}{Gender:
#'  "Male" or "Female"} \item{ridreth1}{Race/ethnicity category: one of "Mexican
#'  American", "Non-Hispanic White", "Non-Hispanic Black", "Other", "Other
#'  Hispanic".} \item{ridexagm}{Age in months at the time of examination (if not
#'  recorded by NHANES, it was imputed based on age at the time of screening)}
#'  \item{ridexagy}{Age in years at the time of examination (if not recorded by
#'  NHANES, it was imputed based on age at the time of screening)}
#'  \item{bmxwt}{Weight in kg} \item{lbxscr}{Serum creatinine, mg/dL}
#'  \item{lbxhct}{Hematocrit, percent by volume of blood composed of red blood
#'  cells} \item{\code{wtmec6yr}}{6-year sample weights for combining 3 cycles,
#'  computed by dividing 2-year sample weights by 3.}
#'  \item{\code{bmxhtlenavg}}{Average of height and recumbent length if both
#'  were measured; if only one was measured, takes value of the one that was
#'  measured.} \item{\code{weight_class}}{One of Underweight, Normal,
#'  Overweight, or Obese. Assigned using methods in
#'  \code{\link{get_weight_class}}.} }
#'
#'@source \url{https://wwwn.cdc.gov/nchs/nhanes/Default.aspx}
#'
#'@keywords data
#'@keywords httk-pop
#'
#'@author Caroline Ring
#'
#' @references 
#' \insertRef{ring2017identifying}{httk}
"mecdt"

#'CDC BMI-for-age charts
#'
#'Charts giving the BMI-for-age percentiles for boys and girls ages 2-18
#'
#'For children ages 2 to 18, weight class depends on the BMI-for-age percentile.
#'\describe{
#'\item{Underweight}{<5th percentile}
#'\item{Normal weight}{5th-85th percentile}
#'\item{Overweight}{85th-95th percentile}
#'\item{Obese}{>=95th percentile}
#'}
#'
#' @format A data.table with 434 rows and 5 variables: \describe{
#'   \item{Sex}{Female or Male} \item{Agemos}{Age in months} \item{P5}{The 5th
#'   percentile BMI for the corresponding sex and age} \item{P85}{The 85th
#'   percentile BMI for the corresponding sex and age} \item{P95}{The 95th
#'   percentile BMI for the corresponding sex and age} }
#' @source \url{https://www.cdc.gov/growthcharts/data/zscore/bmiagerev.csv}
#'
#'@keywords data
#'@keywords httk-pop
#'
#'@author Caroline Ring
#'
#' @references 
#' \insertRef{ring2017identifying}{httk}
"bmiage"

#'WHO weight-for-length charts
#'
#'Charts giving weight-for-length percentiles for boys and girls under age 2.
#'
#'For infants under age 2, weight class depends on weight for length percentile.
#'#'\describe{ \item{Underweight}{<2.3rd percentile} \item{Normal
#'weight}{2.3rd-97.7th percentile} \item{Obese}{>=97.7th percentile} }
#'
#'@format a data.table with 262 rows and 4 variables:
#'
#'  \describe{ \item{Sex}{"Male" or "Female"} \item{Length}{Recumbent length in
#'  cm} \item{P2.3}{The 2.3rd percentile weight in kg for the corresponding sex
#'  and recumbent length} \item{P97.7}{The 97.7th percentile weight in kg for
#'  the corresponding sex and recumbent length}}
#'
#'@source
#'\url{https://www.who.int/tools/child-growth-standards/standards/weight-for-length-height}
"wfl"
#'@keywords data
#'
#'@author Caroline Ring
#'@keywords httk-pop
#'
#' @references 
#' \insertRef{ring2017identifying}{httk}
"wfl"

#'KDE bandwidth for residual variability in height/weight
#'
#'Bandwidths used for a two-dimensional kernel density estimation of the joint
#'distribution of residual errors around smoothing spline fits of height vs. age
#'and weight vs. age for NHANES respondents in each of ten combinations of sex
#'and race/ethnicity categories.
#'
#'Each matrix is a variance-covariance matrix for a two-dimensional normal
#'distribution: this is the bandwidth to be used for a two-dimensional kernel
#'density estimation (KDE) (using a two-dimensional normal kernel) of the joint
#'distribution of residual errors around smoothing spline fits of height vs. age
#'and weight vs. age for NHANES respondents in the specified sex and
#'race/ethnicity category. Optimal bandwidths were pre-calculated by doing the
#'smoothing spline fits, getting the residuals, then calling
#'\code{\link[ks]{kde}} on the residuals (which calls \code{\link[ks]{Hpi}} to
#'compute the plug-in bandwidth).
#'
#'Used by HTTK-Pop only in "virtual individuals" mode (i.e.
#'\code{\link{httkpop_generate}} with \code{method = "v"}), in
#'\code{\link{gen_height_weight}}.
#'
#'@format A named list with 10 elements, each a matrix with 2 rows and 2
#'  columns. Each list element corresponds to, and is named for, one combination
#'  of NHANES sex categories (Male and Female) and NHANES race/ethnicity
#'  categories (Mexican American, Other Hispanic, Non-Hispanic White,
#'  Non-Hispanic Black, and Other).
#'
#'@keywords data
#'
#'@author Caroline Ring
#'@keywords httk-pop
#' @references 
#' \insertRef{ring2017identifying}{httk}
"hw_H" 

#'KDE bandwidths for residual variability in hematocrit
#'
#'Bandwidths used for a one-dimensional kernel density estimation of the
#'distribution of residual errors around smoothing spline fits of hematocrit vs.
#'age  for NHANES respondents in each of ten combinations of sex and
#'race/ethnicity categories.
#'
#'Each matrix is the standard deviation for a normal distribution: this is the
#'bandwidth to be used for a kernel density estimation (KDE) (using a normal
#'kernel) of the distribution of residual errors around smoothing spline fits of
#'hematocrit vs. age for NHANES respondents in the specified sex and
#'race/ethnicity category. Optimal bandwidths were pre-calculated by doing the
#'smoothing spline fits, getting the residuals, then calling
#'\code{\link[ks]{kde}} on the residuals (which calls \code{\link[ks]{hpi}} to
#'compute the plug-in bandwidth).
#'
#'Used by HTTK-Pop only in "virtual individuals" mode (i.e.
#'\code{\link{httkpop_generate}} with \code{method = "v"}), in
#'\code{\link{estimate_hematocrit}}.
#'
#'@format A named list with 10 elements, each a numeric value. Each list element
#'  corresponds to, and is named for, one combination of NHANES sex categories
#'  (Male and Female) and NHANES race/ethnicity categories (Mexican American,
#'  Other Hispanic, Non-Hispanic White, Non-Hispanic Black, and Other).
#'
#'@keywords data
#'
#'@author Caroline Ring
#'@keywords httk-pop
#' @references 
#' \insertRef{ring2017identifying}{httk}
"hct_h" 

#'KDE bandwidths for residual variability in serum creatinine
#'
#'Bandwidths used for a one-dimensional kernel density estimation of the
#'distribution of residual errors around smoothing spline fits of serum
#'creatinine vs. age  for NHANES respondents in each of ten combinations of sex
#'and race/ethnicity categories.
#'
#'Each matrix is the standard deviation for a normal distribution: this is the
#'bandwidth to be used for a kernel density estimation (KDE) (using a normal
#'kernel) of the distribution of residual errors around smoothing spline fits of
#'serum creatinine vs. age for NHANES respondents in the specified sex and
#'race/ethnicity category. Optimal bandwidths were pre-calculated by doing the
#'smoothing spline fits, getting the residuals, then calling
#'\code{\link[ks]{kde}} on the residuals (which calls \code{\link[ks]{hpi}} to
#'compute the plug-in bandwidth).
#'
#'Used by HTTK-Pop only in "virtual individuals" mode (i.e.
#'\code{\link{httkpop_generate}} with \code{method = "v"}), in
#'\code{\link{gen_serum_creatinine}}.
#'
#'@format A named list with 10 elements, each a numeric value. Each list element
#'  corresponds to, and is named for, one combination of NHANES sex categories
#'  (Male and Female) and NHANES race/ethnicity categories (Mexican American,
#'  Other Hispanic, Non-Hispanic White, Non-Hispanic Black, and Other).
#'
#'@keywords data
#'
#'@author Caroline Ring
#'@keywords httk-pop
#' @references 
#' \insertRef{ring2017identifying}{httk}
"scr_h" 

#' Microtiter Plate Well Descriptions for Armitage et al. (2014) Model
#'
#' Microtiter Plate Well Descriptions for Armitage et al. (2014) model from
#' Honda et al. (2019)
#'
#' @format A data frame / data table with 11 rows and 8 variables:
#' \describe{
#'   \item{sysID}{Identifier for each multi-well plate system}
#'   \item{well_desc}{Well description}
#'   \item{well_number}{Number of wells on plate}
#'   \item{area_bottom}{Area of well bottom in mm^2}
#'   \item{cell_yield}{Number of cells}
#'   \item{diam}{Diameter of well in mm}
#'   \item{v_total}{Total volume of well in uL)}
#'   \item{v_working}{Working volume of well in uL}
#' }
#'
#' @keywords data
#'
#' @author Greg Honda
#'
#' @references 
#' \insertRef{armitage2014application}{httk} 
#'
#' \insertRef{honda2019using}{httk}
"well_param"

#' DRUGS|NORMAN: Pharmaceutical List with EU, Swiss, US Consumption Data
#'
#' SWISSPHARMA is a list of pharmaceuticals with consumption data from
#' Switzerland, France, Germany and the USA, used for a suspect
#' screening/exposure modelling approach described in
#' Singer et al 2016, DOI: 10.1021/acs.est.5b03332. The original data is
#' available on the NORMAN Suspect List Exchange.
#'
#'@source \url{https://comptox.epa.gov/dashboard/chemical_lists/swisspharma}
#'@keywords data
#'
#' @references 
#' \insertRef{wambaugh2019assessing}{httk}
"pharma"

#' in vitro Toxicokinetic Data from Wambaugh et al. (2019)
#'
#' These data are the new HTTK in vitro data for chemicals reported in Wambaugh
#' et al. (2019) They
#' are the processed values used to make the figures in that manuscript.
#' These data summarize the results of Bayesian analysis of the in vitro
#' toxicokinetic experiments conducted by Cyprotex to characterize fraction
#' unbound in the presence of pooled human plasma protein and the intrnsic
#' hepatic clearance of the chemical by pooled human hepatocytes.
#'
#' @format A data frame with 496 rows and 17 variables:
#' \describe{
#'   \item{Compound}{The name of the chemical}
#'   \item{CAS}{The Chemical Abstracts Service Registry Number}
#'   \item{Human.Clint}{Median of Bayesian credible interval for intrinsic
#' hepatic clearance (uL/min/million hepatocytes)]}
#'   \item{Human.Clint.pValue}{Probability that there is no clearance}
#'   \item{Human.Funbound.plasma}{Median of Bayesian credibl interval for
#' fraction of chemical free in the presence of plasma}
#'   \item{pKa_Accept}{pH(s) at which hydrogen acceptor sites (if any) are at
#' equilibrium}
#'   \item{pKa_Donor}{pH(s) at which hydrogne donor sites (if any) are at
#' equilibrium}
#'   \item{DSSTox_Substance_Id}{Identifier for CompTox Chemical Dashboard}
#'   \item{SMILES}{Simplified Molecular-Input Line-Entry System structure
#' description}
#'   \item{Human.Clint.Low95}{Lower 95th percentile of Bayesian credible
#' interval for intrinsic hepatic clearance (uL/min/million hepatocytes)}
#'   \item{Human.Clint.High95}{Uppper 95th percentile of Bayesian credible
#' interval for intrinsic hepatic clearance (uL/min/million hepatocytes)}
#'   \item{Human.Clint.Point}{Point estimate of intrinsic hepatic clearance
#' (uL/min/million hepatocytes)}
#'   \item{Human.Funbound.plasma.Low95}{Lower 95th percentile of Bayesian credible
#' interval for fraction of chemical free in the presence of plasma}
#'   \item{Human.Funbound.plasma.High95}{Upper 95th percentile of Bayesian credible
#' interval for fraction of chemical free in the presence of plasma}
#'   \item{Human.Funbound.plasma.Point}{Point estimate of the fraction of
#' chemical free in the presence of plasma}
#'   \item{MW}{Molecular weight (Daltons)}
#'   \item{logP}{log base ten of octanol:water partiion coefficient}
#' }
#' @source Wambaugh et al. (2019)
#'
#'@keywords data
#'
#'@author John Wambaugh
#'
#' @references 
#' \insertRef{wambaugh2019assessing}{httk}
"wambaugh2019"

#' Summary of published toxicokinetic time course experiments
#'
#' This data set summarizes the time course data in the chem.invivo.PK.data
#' table. Maximum concentration (Cmax), time integrated plasma concentration
#' for the duration of treatment (AUC.treatment) and extrapolated to zero
#' concentration (AUC.infinity) as well as half-life are calculated. Summary
#' values are given for each study and dosage. These data can be used to
#' evaluate toxicokinetic model predictions.
#'
#'
#' @docType data
#' @format A data.frame containing 100 rows and 25 columns.
#' @author John Wambaugh
#' @references Aanderud L, Bakke OM (1983). Pharmacokinetics of antipyrine,
#' paracetamol, and morphine in rat at 71 ATA. Undersea Biomed Res.
#' 10(3):193-201. PMID: 6636344
#'
#' Aasmoe L, Mathiesen M, Sager G (1999). Elimination of methoxyacetic acid and
#' ethoxyacetic acid in rat. Xenobiotica. 29(4):417-24. PMID: 10375010
#'
#' Ako RA. Pharmacokinetics/pharmacodynamics (PK/PD) of oral diethylstilbestrol
#' (DES) in recurrent prostate cancer patients and of oral dissolving film
#' (ODF)-DES in rats. PhD dissertation, College of Pharmacy, University of
#' Houston, USA, 2011.
#'
#' Anadon A, Martinez-Larranaga MR, Fernandez-Cruz ML, Diaz MJ, Fernandez MC,
#' Martinez MA (1996). Toxicokinetics of deltamethrin and its 4'-HO-metabolite
#' in the rat. Toxicol Appl Pharmacol. 141(1):8-16. PMID: 8917670
#'
#' Binkerd PE, Rowland JM, Nau H, Hendrickx AG (1988). Evaluation of valproic
#' acid (VPA) developmental toxicity and pharmacokinetics in Sprague-Dawley
#' rats. Fundam Appl Toxicol. 11(3):485-93. PMID: 3146521
#'
#' Boralli VB, Coelho EB, Cerqueira PM, Lanchote VL (2005). Stereoselective
#' analysis of metoprolol and its metabolites in rat plasma with application to
#' oxidative metabolism. J Chromatogr B Analyt Technol Biomed Life Sci.
#' 823(2):195-202. PMID: 16029965
#'
#' Chan MP, Morisawa S, Nakayama A, Kawamoto Y, Sugimoto M, Yoneda M (2005).
#' Toxicokinetics of 14C-endosulfan in male Sprague-Dawley rats following oral
#' administration of single or repeated doses. Environ Toxicol. 20(5):533-41.
#' PMID: 16161119
#'
#' Cruz L, Castaneda-Hernandez G, Flores-Murrieta FJ, Garcia-Lopez P,
#' Guizar-Sahagun G (2002). Alteration of phenacetin pharmacokinetics after
#' experimental spinal cord injury. Proc West Pharmacol Soc. 45:4-5. PMID:
#' 12434508
#'
#' Della Paschoa OE, Mandema JW, Voskuyl RA, Danhof M (1998).
#' Pharmacokinetic-pharmacodynamic modeling of the anticonvulsant and
#' electroencephalogram effects of phenytoin in rats. J Pharmacol Exp Ther.
#' 284(2):460-6. PMID: 9454785
#'
#' Du B, Li X, Yu Q, A Y, Chen C (2010). Pharmacokinetic comparison of orally
#' disintegrating, beta-cyclodextrin inclusion complex and conventional tablets
#' of nicardipine in rats. Life Sci J. 7(2):80-4.
#'
#' Farris FF, Dedrick RL, Allen PV, Smith JC (1993). Physiological model for
#' the pharmacokinetics of methyl mercury in the growing rat. Toxicol Appl
#' Pharmacol. 119(1):74-90. PMID: 8470126
#'
#' Hays SM, Elswick BA, Blumenthal GM, Welsch F, Conolly RB, Gargas ML (2000).
#' Development of a physiologically based pharmacokinetic model of
#' 2-methoxyethanol and 2-methoxyacetic acid disposition in pregnant rats.
#' Toxicol Appl Pharmacol. 163(1):67-74. PMID: 10662606
#'
#' Igari Y, Sugiyama Y, Awazu S, Hanano M (1982). Comparative physiologically
#' based pharmacokinetics of hexobarbital, phenobarbital and thiopental in the
#' rat. J Pharmacokinet Biopharm. 10(1):53-75. PMID: 7069578
#'
#' Ito K, Houston JB (2004). Comparison of the use of liver models for
#' predicting drug clearance using in vitro kinetic data from hepatic
#' microsomes and isolated hepatocytes. Pharm Res. 21(5):785-92. PMID: 15180335
#'
#' Jia L, Wong H, Wang Y, Garza M, Weitman SD (2003). Carbendazim: disposition,
#' cellular permeability, metabolite identification, and pharmacokinetic
#' comparison with its nanoparticle. J Pharm Sci. 92(1):161-72. PMID: 12486692
#'
#' Kawai R, Mathew D, Tanaka C, Rowland M (1998). Physiologically based
#' pharmacokinetics of cyclosporine A: extension to tissue distribution
#' kinetics in rats and scale-up to human. J Pharmacol Exp Ther. 287(2):457-68.
#' PMID: 9808668
#'
#' Kim YC, Kang HE, Lee MG (2008). Pharmacokinetics of phenytoin and its
#' metabolite, 4'-HPPH, after intravenous and oral administration of phenytoin
#' to diabetic rats induced by alloxan or streptozotocin. Biopharm Drug Dispos.
#' 29(1):51-61. PMID: 18022993
#'
#' Kobayashi S, Takai K, Iga T, Hanano M (1991). Pharmacokinetic analysis of
#' the disposition of valproate in pregnant rats. Drug Metab Dispos.
#' 19(5):972-6. PMID: 1686245
#'
#' Kotegawa T, Laurijssens BE, Von Moltke LL, Cotreau MM, Perloff MD,
#' Venkatakrishnan K, Warrington JS, Granda BW, Harmatz JS, Greenblatt DJ
#' (2002). In vitro, pharmacokinetic, and pharmacodynamic interactions of
#' ketoconazole and midazolam in the rat. J Pharmacol Exp Ther. 302(3):1228-37.
#' PMID: 12183684
#'
#' Krug AK, Kolde R, Gaspar JA, Rempel E, Balmer NV, Meganathan K, Vojnits K,
#' Baquie M, Waldmann T, Ensenat-Waser R, Jagtap S, Evans RM, Julien S,
#' Peterson H, Zagoura D, Kadereit S, Gerhard D, Sotiriadou I, Heke M,
#' Natarajan K, Henry M, Winkler J, Marchan R, Stoppini L, Bosgra S, Westerhout
#' J, Verwei M, Vilo J, Kortenkamp A, Hescheler J, Hothorn L, Bremer S, van
#' Thriel C, Krause KH, Hengstler JG, Rahnenfuhrer J, Leist M, Sachinidis A
#' (2013). Human embryonic stem cell-derived test systems for developmental
#' neurotoxicity: a transcriptomics approach. Arch Toxicol. 87(1):123-43. PMID:
#' 23179753
#'
#' Leon-Reyes MR, Castaneda-Hernandez G, Ortiz MI (2009). Pharmacokinetic of
#' diclofenac in the presence and absence of glibenclamide in the rat. J Pharm
#' Pharm Sci. 12(3):280-7. PMID: 20067705
#'
#' Nagata M, Hidaka M, Sekiya H, Kawano Y, Yamasaki K, Okumura M, Arimori K
#' (2007). Effects of pomegranate juice on human cytochrome P450 2C9 and
#' tolbutamide pharmacokinetics in rats. Drug Metab Dispos. 35(2):302-5. PMID:
#' 17132763
#'
#' Okiyama M, Ueno K, Ohmori S, Igarashi T, Kitagawa H (1988). Drug
#' interactions between imipramine and benzodiazepines in rats. J Pharm Sci.
#' 77(1):56-63. PMID: 2894451
#'
#' Pelissier-Alicot AL, Schreiber-Deturmeny E, Simon N, Gantenbein M,
#' Bruguerolle B (2002). Time-of-day dependent pharmacodynamic and
#' pharmacokinetic profiles of caffeine in rats. Naunyn Schmiedebergs Arch
#' Pharmacol. 365(4):318-25. PMID: 11919657
#'
#' Piersma AH, Bosgra S, van Duursen MB, Hermsen SA, Jonker LR, Kroese ED, van
#' der Linden SC, Man H, Roelofs MJ, Schulpen SH, Schwarz M, Uibel F, van
#' Vugt-Lussenburg BM, Westerhout J, Wolterbeek AP, van der Burg B (2013).
#' Evaluation of an alternative in vitro test battery for detecting
#' reproductive toxicants. Reprod Toxicol. 38:53-64. PMID: 23511061
#'
#' Pollack GM, Li RC, Ermer JC, Shen DD (1985). Effects of route of
#' administration and repetitive dosing on the disposition kinetics of
#' di(2-ethylhexyl) phthalate and its mono-de-esterified metabolite in rats.
#' Toxicol Appl Pharmacol. Jun 30;79(2):246-56. PMID: 4002226
#'
#' Saadeddin A, Torres-Molina F, Carcel-Trullols J, Araico A, Peris JE (2004).
#' Pharmacokinetics of the time-dependent elimination of all-trans-retinoic
#' acid in rats. AAPS J. 6(1):1-9. PMID: 18465253
#'
#' Satterwhite JH, Boudinot FD (1991). Effects of age and dose on the
#' pharmacokinetics of ibuprofen in the rat. Drug Metab Dispos. 19(1):61-7.
#' PMID: 1673423
#'
#' Szymura-Oleksiak J, Panas M, Chrusciel W (1983). Pharmacokinetics of
#' imipramine after single and multiple intravenous administration in rats. Pol
#' J Pharmacol Pharm. 35(2):151-7. PMID: 6622297
#'
#' Tanaka C, Kawai R, Rowland M (2000). Dose-dependent pharmacokinetics of
#' cyclosporin A in rats: events in tissues. Drug Metab Dispos. 28(5):582-9.
#' PMID: 10772639
#'
#' Timchalk C, Nolan RJ, Mendrala AL, Dittenber DA, Brzak KA, Mattsson JL
#' (2002). A Physiologically based pharmacokinetic and pharmacodynamic
#' (PBPK/PD) model for the organophosphate insecticide chlorpyrifos in rats and
#' humans. Toxicol Sci. Mar;66(1):34-53. PMID: 11861971
#'
#' Tokuma Y, Sekiguchi M, Niwa T, Noguchi H (1988). Pharmacokinetics of
#' nilvadipine, a new dihydropyridine calcium antagonist, in mice, rats,
#' rabbits and dogs. Xenobiotica 18(1):21-8. PMID: 3354229
#'
#' Treiber A, Schneiter R, Delahaye S, Clozel M (2004). Inhibition of organic
#' anion transporting polypeptide-mediated hepatic uptake is the major
#' determinant in the pharmacokinetic interaction between bosentan and
#' cyclosporin A in the rat. J Pharmacol Exp Ther. 308(3):1121-9. PMID:
#' 14617681
#'
#' Tsui BC, Feng JD, Buckley SJ, Yeung PK (1994). Pharmacokinetics and
#' metabolism of diltiazem in rats following a single intra-arterial or single
#' oral dose. Eur J Drug Metab Pharmacokinet. 19(4):369-73. PMID: 7737239
#'
#' Wambaugh, John F., et al. "Toxicokinetic triage for environmental
#' chemicals." Toxicological Sciences (2015): 228-237.
#'
#' Wang Y, Roy A, Sun L, Lau CE (1999). A double-peak phenomenon in the
#' pharmacokinetics of alprazolam after oral administration. Drug Metab Dispos.
#' 27(8):855-9. PMID: 10421610
#'
#' Wang X, Lee WY, Or PM, Yeung JH (2010). Pharmacokinetic interaction studies
#' of tanshinones with tolbutamide, a model CYP2C11 probe substrate, using
#' liver microsomes, primary hepatocytes and in vivo in the rat. Phytomedicine.
#' 17(3-4):203-11. PMID: 19679455
#'
#' Yang SH, Lee MG (2008). Dose-independent pharmacokinetics of ondansetron in
#' rats: contribution of hepatic and intestinal first-pass effects to low
#' bioavailability. Biopharm Drug Dispos. 29(7):414-26. PMID: 18697186
#'
#' Yeung PK, Alcos A, Tang J (2009). Pharmacokinetics and Hemodynamic Effects
#' of Diltiazem in Rats Following Single vs Multiple Doses In Vivo. Open Drug
#' Metab J. 3:56-62.
#' @source Wambaugh et al. 2018 Toxicological Sciences, in press
#' @keywords data
"chem.invivo.PK.summary.data"



#' Parameter Estimates from Wambaugh et al. (2018)
#'
#' This table includes 1 and 2 compartment fits of plasma concentration vs time
#' data aggregated from chem.invivo.PK.data, performed in Wambaugh et al. 2018.
#' Data includes volume of distribution (Vdist, L/kg), elimination rate (kelim,
#' 1/h), gut absorption rate (kgutabs, 1/h), fraction absorbed (Fabsgut), and
#' steady state concentration (Css, mg/L).
#'
#'
#' @docType data
#' @format data.frame
#' @author John Wambaugh
#' @source Wambaugh et al. 2018 
#' @references
#' \insertRef{wambaugh2018evaluating}{httk}
#' @keywords data
"chem.invivo.PK.aggregate.data"


#' Raw Bayesian in vitro Toxicokinetic Data Analysis from Wambaugh et al. (2019)
#'
#' These data are the new HTTK in vitro data for chemicals reported in Wambaugh
#' et al. (2019) They
#' are the output of different Bayesian models evaluated to compare using a
#' single protein concentration vs. the new three concentration titration
#' protocol. These data summarize the results of Bayesian analysis of the in vitro
#' toxicokinetic experiments conducted by Cyprotex to characterize fraction
#' unbound in the presence of pooled human plasma protein and the intrnsic
#' hepatic clearance of the chemical by pooled human hepatocytes.
#' This file includes replicates (diferent CompoundName id's but same chemical')
#'
#' @format A data frame with 530 rows and 28 variables:
#' \describe{
#'   \item{DTXSID}{Identifier for CompTox Chemical Dashboard}
#'   \item{Name}{The name of the chemical}
#'   \item{CAS}{The Chemical Abstracts Service Registry Number}
#'   \item{CompoundName}{Sample name provided by EPA to Cyprotex}
#'   \item{Fup.point}{Point estimate of the fraction of
#' chemical free in the presence of plasma}
#'   \item{Base.Fup.Med}{Median of Bayesian credible interval for
#' fraction of chemical free in the presence of plasma for analysis of 100%
#' physiological plasma protein data only (base model)}
#'   \item{Base.Fup.Low}{Lower 95th percentile of Bayesian credible
#' interval for fraction of chemical free in the presence of plasma for analysis of 100%
#' physiological plasma protein data only (base model)}
#'   \item{Base.Fup.High}{Upper 95th percentile of Bayesian credible
#' interval for fraction of chemical free in the presence of plasma for analysis of 100%
#' physiological plasma protein data only (base model)}
#'   \item{Affinity.Fup.Med}{Median of Bayesian credible interval for
#' fraction of chemical free in the presence of plasma for analysis of protein
#' titration protocol data (affinity model)}
#'   \item{Affinity.Fup.Low}{Lower 95th percentile of Bayesian credible
#' interval for fraction of chemical free in the presence of plasma for analysis of protein
#' titration protocol data (affinity model)}
#'   \item{Affinity.Fup.High}{Upper 95th percentile of Bayesian credible
#' interval for fraction of chemical free in the presence of plasma for analysis of protein
#' titration protocol data (affinity model)}
#'   \item{Affinity.Kd.Med}{Median of Bayesian credible interval for
#' protein binding affinity from analysis of protein
#' titration protocol data (affinity model)}
#'   \item{Affinity.Kd.Low}{Lower 95th percentile of Bayesian credible
#' interval for protein binding affinity from analysis of protein
#' titration protocol data (affinity model)}
#'   \item{Affinity.Kd.High}{Upper 95th percentile of Bayesian credible
#' interval for protein binding affinity from analysis of protein
#' titration protocol data (affinity model)}
#'   \item{Decreases.Prob}{Probability that the chemical concentration decreased
#' systematiclally during hepatic clearance assay.}
#'   \item{Saturates.Prob}{Probability that the rate of chemical concentration
#' decrease varied between the 1 and 10 uM hepatic clearance experiments.}
#'   \item{Slope.1uM.Median}{Estimated slope for chemcial concentration decrease
#' in the 1 uM hepatic clearance assay.}
#'   \item{Slope.10uM.Median}{Estimated slope for chemcial concentration decrease
#' in the 10 uM hepatic clearance assay.}
#'   \item{CLint.1uM.Median}{Median of Bayesian credible interval for intrinsic
#' hepatic clearance at 1 uM initital chemical concentration (uL/min/million hepatocytes)]}
#'   \item{CLint.1uM.Low95th}{Lower 95th percentile of Bayesian credible
#' interval for intrinsic hepatic clearance at 1 uM initital chemical
#' concentration (uL/min/million hepatocytes)}
#'   \item{CLint.1uM.High95th}{Uppper 95th percentile of Bayesian credible
#' interval for intrinsic hepatic clearance at 1 uM initital chemical
#' concentration(uL/min/million hepatocytes)}
#'   \item{CLint.10uM.Median}{Median of Bayesian credible interval for intrinsic
#' hepatic clearance at 10 uM initital chemical concentration (uL/min/million hepatocytes)]}
#'   \item{CLint.10uM.Low95th}{Lower 95th percentile of Bayesian credible
#' interval for intrinsic hepatic clearance at 10 uM initital chemical
#' concentration (uL/min/million hepatocytes)}
#'   \item{CLint.10uM.High95th}{Uppper 95th percentile of Bayesian credible
#' interval for intrinsic hepatic clearance at 10 uM initital chemical
#' concentration(uL/min/million hepatocytes)}
#'   \item{CLint.1uM.Point}{Point estimate of intrinsic hepatic clearance
#' (uL/min/million hepatocytes) for 1 uM initial chemical concentration}
#'   \item{CLint.10uM.Point}{Point estimate of intrinsic hepatic clearance
#' (uL/min/million hepatocytes) for 10 uM initial chemical concentration}
#'   \item{Fit}{Classification of clearance observed}
#'   \item{SMILES}{Simplified Molecular-Input Line-Entry System structure
#' description}
#' }
#' @source Wambaugh et al. (2019)
#'
#'@keywords data
#'
#'@author John Wambaugh
#'
#' @references 
#' \insertRef{wambaugh2019assessing}{httk}
"wambaugh2019.raw"

#' NHANES Chemical Intake Rates for chemicals in Wambaugh et al. (2019)
#'
#' These data are a subset of the Bayesian inferrences reported by Ring et al.
#' (2017) from the U.S. Centers for Disease Control and Prevention (CDC)
#' National Health and Nutrition Examination Survey (NHANES). They reflect the
#' populaton median intake rate (mg/kg body weight/day), with uncertainty.
#'
#' @format A data frame with 20 rows and 4 variables:
#' \describe{
#'   \item{lP}{The median of the Bayesian credible interval for median population
#' intake rate (mg/kg bodyweight/day)}
#'   \item{lP.min}{The lower 95th percentile of the Bayesian credible interval for median population
#' intake rate (mg/kg bodyweight/day)}
#'   \item{lP.max}{The upper 95th percentile of the Bayesian credible interval for median population
#' intake rate (mg/kg bodyweight/day)}
#'   \item{CASRN}{The Chemical Abstracts Service Registry Number}
#' }
#' @source Wambaugh et al. (2019)
#'
#'@keywords data
#'
#'@author John Wambaugh
#'
#' @references 
#' \insertRef{ring2017identifying}{httk}
#' 
#' \insertRef{wambaugh2019assessing}{httk}
"wambaugh2019.nhanes"

#' ExpoCast SEEM3 Consensus Exposure Model Predictions for Chemical Intake Rates
#'
#' These data are a subset of the Bayesian inferrences reported by Ring et al.
#' (2019) for a consensus model of twelve exposue predictors. The predictors
#' were calibrated based upon their ability to predict intake rates inferred
# 'from the U.S. Centers for Disease Control and Prevention (CDC)
#' National Health and Nutrition Examination Survey (NHANES). They reflect the
#' populaton median intake rate (mg/kg body weight/day), with uncertainty.
#'
#' @format A data frame with 385 rows and 38 variables:
#' @source Wambaugh et al. (2019)
#'
#'@keywords data
#'
#'@author John Wambaugh
#'
#' @references 
#' \insertRef{Ring2017SEEM}{httk}
#'
#' \insertRef{wambaugh2019assessing}{httk}
"wambaugh2019.seem3"


#' Physico-chemical properties and in vitro measurements for toxicokinetics
#'
#' This data set contains the necessary information to make basic,
#' high-throughput toxicokinetic (HTTK) predictions for compounds, including
#' Funbound.plasma, molecular weight (g/mol), logP, logMA (membrane affinity),
#' intrinsic clearance(uL/min/10^6 cells), and pKa. These data have been
#' compiled from multiple sources, and can be used to parameterize a variety of
#' toxicokinetic models. See variable EPA.ref for information on the reference EPA.
#'
#' In some cases the rapid equilbrium dailysis method (Waters et al., 2008)
#' fails to yield detectable concentrations for the free fraction of chemical. 
#' In those cases we assume the compound is highly bound (that is, Fup approaches
#' zero). For some calculations (for example, steady-state plasma concentration)
#' there is precendent (Rotroff et al., 2010) for using half the average limit 
#' of detection, that is 0.005. We do not recomend using other models where 
#' quantities like partition coefficients must be predicted using Fup. We also
#' do not recomend including the value 0.005 in training sets for Fup predictive
#' models. 
#'
#' \strong{Note} that in some cases the \strong{Funbound.plasma} and the 
#' \strong{intrinsic clearance} are
#' \emph{provided as a series of numbers separated by commas}. These values are the 
#' result of Bayesian analysis and characterize a distribution: the first value
#' is the median of the distribution, while the second and third values are the 
#' lower and upper 95th percentile (that is qunatile 2.5 and 97.5) respectively.
#' For intrinsic clearance a fourth value indicating a p-value for a decrease is
#' provided. Typically 4000 samples were used for the Bayesian analusis, such
#' that a p-value of "0" is equivale to "<0.00025". See Wambaugh et al. (2019)
#' for more details.
#'
#' Any one chemical compound \emph{may have multiple ionization equilibria} 
#' (see Strope et al., 2018) may both for donating or accepting a proton (and
#' therefore changing charge state). If there are multiple equlibria of the same
#' type (donor/accept])the are concatonated by commas.
#'
#' All species-specific information is initially from experimental measurements.
#' The functions \code{\link{load_sipes2017}}, \code{\link{load_pradeep2020}}, 
#' and \code{\link{load_dawson2021}} may be used to add in silico, structure-based
#' predictions for many thousands of additional compounds to this table.
#'
#' @docType data
#' @format A data.frame containing 9411 rows and 54 columns.
#' \tabular{lll}{
#' \strong{Column Name} \tab \strong{Description} \tab \strong{Units} \cr
#'  Compound \tab The preferred name of the chemical compound \tab none \cr                      
#'  CAS\tab The preferred Chemical Abstracts Service Registry Number \tab none \cr                     
#'  CAS.Checksum \tab A logical indicating whether the CAS number is valid \tab none \cr                   
#'  DTXSID \tab DSSTox Structure ID 
#' (\url{https://comptox.epa.gov/dashboard}) \tab none \cr                  
#'  Formula \tab The proportions of atoms within the chemical compound  \tab none \cr                   
#'  All.Compound.Names \tab All names of the chemical as they occured in the
#' data \tab none \cr                                           
#'  logHenry \tab The log10 Henry's law constant (Conc_air = 10^logH * Conc_liquid)\tab 
#' log10(atmosphers*m^3/mole) \cr                 
#'  logHenry.Reference \tab Reference for Henry's law constant \tab \cr           
#'  logP \tab The log10 octanol:water partition coefficient (PC)\tab log10 unitless ratio \cr                 
#'  logP.Reference \tab Reference for logPow \tab \cr               
#'  logPwa \tab The log10 water:air PC \tab log10 unitless ratio \cr                 
#'  logPwa.Reference \tab Reference for logPwa \tab \cr             
#'  logMA \tab The log10 phospholipid:water PC or
#' "Membrane affinity" \tab unitless ratio \cr                
#'  logMA.Reference \tab Reference for membrane affinity \tab \cr   
#'  logWSol \tab The log10 water solubility \tab log10(mole/L) \cr                  
#'  logWSol.Reference \tab Reference for logWsol \tab \cr              
#'  MP \tab The chemical compound melting point \tab degrees Celsius \cr                  
#'  MP.Reference \tab Reference for melting point \tab \cr                   
#'  MW \tab The chemical compound molecular weight \tab g/mol \cr                
#'  MW.Reference \tab Reference for molecular weight \tab \cr                 
#'  pKa_Accept \tab The hydrogen acceptor equilibria concentrations 
#'  \tab logarithm \cr              
#'  pKa_Accept.Reference \tab Reference for pKa_Accept \tab \cr           
#'  pKa_Donor \tab The hydrogen acceptor equilibria concentrations 
#' \tab logarithm \cr               
#'  pKa_Donor.Reference \tab Reference for pKa_Donor \tab \cr             
#'  All.Species \tab All species for which data were available \tab none \cr                
#'  DTXSID.Reference \tab Reference for DTXSID \tab \cr               
#'  Formula.Reference \tab Reference for chemical formulat \tab \cr             
#'  [SPECIES].Clint \tab (Primary hepatocyte suspension) 
#' intrinsic hepatic clearance. \emph{Entries with comma separated values are Bayesian estimates of
#' the Clint distribution - displayed as the median, 95th credible interval
#' (that is quantile 2.5 and 97.5, respectively), and p-value.} \tab uL/min/10^6 hepatocytes \cr                   
#'  [SPECIES].Clint.pValue \tab Probability that there is no clearance observed.
#'  Values close to 1 indicate clearance is not statistically significant. \tab none \cr           
#'  [SPECIES].Clint.pValue.Ref \tab Reference for Clint pValue \tab  \cr   
#'  [SPECIES].Clint.Reference \tab Reference for Clint \tab  \cr         
#'  [SPECIES].Caco2.Pab \tab Caco-2 Apical-to-Basal Membrane Permeability \tab 10^-6 cm/s \cr           
#'  [SPECIES].Caco2.Pab.Reference \tab Reference for Caco-2 Membrane Permeability \tab \cr
#'  [SPECIES].Fabs \tab In vivo measured fraction of an oral dose of chemical 
#' absorbed from the gut lumen into the gut \tab unitless fraction \cr           
#'  [SPECIES].Fabs.Reference \tab Reference for Fabs \tab \cr        
#'  [SPECIES].Fgut \tab In vivo measured fraction of an oral dose of chemical 
#' that passes gut metabolism and clearance \tab unitless fraction \cr           
#'  [SPECIES].Fgut.Reference \tab Reference for Fgut \tab \cr        
#'  [SPECIES].Foral \tab In vivo measued fractional systemic bioavailability of 
#' an oral dose, modeled as he product of Fabs * Fgut * Fhep (where Fhep is 
#' first pass hepatic metabolism). \tab unitless fraction \cr           
#'  [SPECIES].Foral.Reference \tab Reference for Foral \tab \cr        
#'  [SPECIES].Funbound.plasma \tab Chemical fraction unbound in presence of 
#' plasma proteins (fup). \emph{Entries with comma separated values are Bayesian estimates of
#' the fup distribution - displayed as the median and 95th credible interval
#' (that is quantile 2.5 and 97.5, respectively).} \tab unitless fraction \cr         
#'  [SPECIES].Funbound.plasma.Ref\tab Reference for Funbound.plasma \tab \cr 
#'  [SPECIES].Rblood2plasma \tab Chemical concentration blood to plasma ratio \tab unitless ratio \cr         
#'  [SPECIES].Rblood2plasma.Ref \tab Reference for Rblood2plasma \tab  \cr        
#'  Chemical.Class \tab All classes to which the chemical has been assigned \tab \cr
#' }
#'
#' @seealso \code{\link{get_physchem_param}} 
#' @seealso \code{\link{get_invitroPK_param}} 
#' @seealso \code{\link{add_chemtable}} 
#'
#' @author John Wambaugh
#'
#' @references CompTox Chemicals Dashboard (\url{https://comptox.epa.gov/dashboard})
#'
#' EPI Suite, https://www.epa.gov/opptintr/exposure/pubs/episuite.htm
#'
#' Brown, Hayley S., Michael Griffin, and J. Brian Houston. "Evaluation of 
#' cryopreserved human hepatocytes as an alternative in vitro system to 
#' microsomes for the prediction of metabolic clearance." Drug metabolism and 
#' disposition 35.2 (2007): 293-301.
#'
#' Gulden, Michael, et al. "Impact of protein binding on the availability and 
#' cytotoxic potency of organochlorine pesticides and chlorophenols in vitro." 
#' Toxicology 175.1-3 (2002): 201-213.
#'
#' Hilal, S., Karickhoff, S. and Carreira, L. (1995). A rigorous test for
#' SPARC's chemical reactivity models: Estimation of more than 4300 ionization
#' pKas. Quantitative Structure-Activity Relationships 14(4), 348-355.
#' 
#' \insertRef{honda2019using}{httk}
#'
#' \insertRef{ito2004comparison}{httk}
#'
#' Jones, O. A., Voulvoulis, N. and Lester, J. N. (2002). Aquatic environmental
#' assessment of the top 25 English prescription pharmaceuticals. Water
#' research 36(20), 5013-22.
#' 
#' Jones, Barry C., et al. "An investigation into the prediction of in vivo 
#' clearance for a range of flavin-containing monooxygenase substrates." 
#' Drug metabolism and disposition 45.10 (2017): 1060-1067.
#'
#' Lau, Y. Y., Sapidou, E., Cui, X., White, R. E. and Cheng, K. C. (2002).
#' Development of a novel in vitro model to predict hepatic clearance using
#' fresh, cryopreserved, and sandwich-cultured hepatocytes. Drug Metabolism and
#' Disposition 30(12), 1446-54.
#'
#' \insertRef{linakis2020development}{httk}
#' 
#' Lombardo, F., Berellini, G., & Obach, R. S. (2018). Trend analysis of a 
#' database of intravenous pharmacokinetic parameters in humans for 1352 drug 
#' compounds. Drug Metabolism and Disposition, 46(11), 1466-1477.
#'
#' McGinnity, D. F., Soars, M. G., Urbanowicz, R. A. and Riley, R. J. (2004).
#' Evaluation of fresh and cryopreserved hepatocytes as in vitro drug
#' metabolism tools for the prediction of metabolic clearance. Drug Metabolism
#' and Disposition 32(11), 1247-53, 10.1124/dmd.104.000026.
#'
#' Naritomi, Y., Terashita, S., Kagayama, A. and Sugiyama, Y. (2003). Utility
#' of Hepatocytes in Predicting Drug Metabolism: Comparison of Hepatic
#' Intrinsic Clearance in Rats and Humans in Vivo and in Vitro. Drug Metabolism
#' and Disposition 31(5), 580-588, 10.1124/dmd.31.5.580.
#'
#' Obach, R. S. (1999). Prediction of human clearance of twenty-nine drugs from
#' hepatic microsomal intrinsic clearance data: An examination of in vitro
#' half-life approach and nonspecific binding to microsomes. Drug Metabolism
#' and Disposition 27(11), 1350-9.
#'
#' Paini, Alicia; Cole, Thomas; Meinero, Maria; Carpi, Donatella; Deceuninck, 
#' Pierre; Macko, Peter; Palosaari, Taina; Sund, Jukka; Worth, Andrew; Whelan, 
#' Maurice (2020):  EURL ECVAM in vitro hepatocyte clearance and blood plasma 
#' protein binding dataset for 77 chemicals. European Commission, Joint Research 
#' Centre (JRC) [Dataset] PID: https://data.europa.eu/89h/a2ff867f-db80-4acf-8e5c-e45502713bee
#' 
#' Paixao, P., Gouveia, L. F., & Morais, J. A. (2012). Prediction of the human
#' oral bioavailability by using in vitro and in silico drug related parameters
#' in a physiologically based absorption model. International journal of
#' pharmaceutics, 429(1), 84-98.
#'
#' Pirovano, Alessandra, et al. "QSARs for estimating intrinsic hepatic
#' clearance of organic chemicals in humans." Environmental toxicology and
#' pharmacology 42 (2016): 190-197.
#'
#' Riley, Robert J., Dermot F. McGinnity, and Rupert P. Austin. "A unified 
#' model for predicting human hepatic, metabolic clearance from in vitro 
#' intrinsic clearance data in hepatocytes and microsomes." Drug Metabolism and 
#' Disposition 33.9 (2005): 1304-1311.
#'
#' \insertRef{schmitt2008general}{httk}
#'
#' Shibata, Y., Takahashi, H., Chiba, M. and Ishii, Y. (2002). Prediction of
#' Hepatic Clearance and Availability by Cryopreserved Human Hepatocytes: An
#' Application of Serum Incubation Method. Drug Metabolism and Disposition
#' 30(8), 892-896, 10.1124/dmd.30.8.892.
#'
#' Sohlenius-Sternbeck, Anna-Karin, et al. "Practical use of the regression 
#' offset approach for the prediction of in vivo intrinsic clearance from 
#' hepatocytes." Xenobiotica 42.9 (2012): 841-853.
#' 
#' Tonnelier, A., Coecke, S. and Zaldivar, J.-M. (2012). Screening of chemicals
#' for human bioaccumulative potential with a physiologically based
#' toxicokinetic model. Archives of Toxicology 86(3), 393-403,
#' 10.1007/s00204-011-0768-0.
#'
#' Uchimura, Takahide, et al. "Prediction of human blood-to-plasma drug
#' concentration ratio." Biopharmaceutics & drug disposition 31.5-6 (2010):
#' 286-297.
#'
#' \insertRef{wambaugh2019assessing}{httk}
#' 
#' \insertRef{wetmore2012integration}{httk}
#'
#' \insertRef{wetmore2013relative}{httk}
#'
#' \insertRef{wetmore2015incorporating}{httk}
#'
#' F. L. Wood, J. B. Houston and D. Hallifax
#' 'Drug Metabolism and Disposition November 1, 2017, 45 (11) 1178-1188; 
#' DOI: https://doi.org/10.1124/dmd.117.077040
#' @source Wambaugh, John F., et al. "Toxicokinetic triage for environmental
#' chemicals." Toxicological Sciences (2015): 228-237.
#' @keywords data
"chem.physical_and_invitro.data"


#' Tox21 2015 Active Hit Calls (EPA)
#'
#' The ToxCast and Tox21 research programs employ batteries of high-throughput
#' assays to assess chemical bioactivity in vitro. Not every chemical is tested
#' through every assay. Most assays are conducted in concentration response,
#' and each corresponding assay endpoint is analyzed statistically to determine
#' if there is a concentration-dependent response or "hit" using the ToxCast
#' Pipeline.  Most assay endpoint-chemical combinations are non-responsive.
#' Here, only the hits are treated as potential indicators of bioactivity. This
#' bioactivity does not have a direct toxicological interpretation. The October
#' 2015 release (invitrodb_v2) of the ToxCast and Tox21 data were used for this
#' analysis. This object contains just the chemicals in Wambaugh et al. (2019)
#' and only the quantiles across all assays for the ACC.
#'
#' @name wambaugh2019.tox21
#' @docType data
#' @format A data.table with 401 rows and 6 columns
#' @author John Wambaugh
#' @references Kavlock, Robert, et al. "Update on EPA's ToxCast program:
#' providing high-throughput decision support tools for chemical risk
#' management." Chemical research in toxicology 25.7 (2012): 1287-1302.
#'
#' Tice, Raymond R., et al. "Improving the human hazard characterization of
#' chemicals: a Tox21 update." Environmental health perspectives 121.7 (2013):
#' 756-765.
#'
#' Richard, Ann M., et al. "ToxCast chemical landscape: paving the road to 21st
#' century toxicology." Chemical research in toxicology 29.8 (2016): 1225-1251.
#'
#' Filer, Dayne L., et al. "tcpl: the ToxCast pipeline for high-throughput
#' screening data." Bioinformatics 33.4 (2016): 618-620.
#' 
#' Wambaugh, John F., et al. "Assessing Toxicokinetic Uncertainty and 
#' Variability in Risk Prioritization." Toxicological Sciences 172.2 (2019): 
#' 235-251.
#'
#' @keywords data
"wambaugh2019.tox21"



#' Howgate 2006
#'
#' This data set is only used in Vignette 5.
#'
#' @docType data
#' @format A data.table containing 24 rows and 11 columns.
#' @keywords data
#' @author Caroline Ring
#' @references
#' Howgate, E. M., et al. "Prediction of in vivo drug clearance from in vitro
#' data. I: impact of inter-individual variability." Xenobiotica 36.6 (2006):
#' 473-497.
"howgate"


#' Johnson 2006
#'
#' This data set is only used in Vignette 5.
#'
#' @docType data
#'
#' @format A data.table containing 60 rows and 11 columns.
#'
#' @author Caroline Ring
#' @references
#' Johnson, Trevor N., Amin Rostami-Hodjegan, and Geoffrey T. Tucker.
#' "Prediction of the clearance of eleven drugs and associated variability in
#' neonates, infants and children." Clinical pharmacokinetics 45.9 (2006):
#' 931-956.
#'
#' @keywords data
"johnson"


#' Published Pharmacokinetic Parameters from Obach et al. 2008
#'
#' This data set is used in Vignette 4 for steady state concentration.
#'
#'
#' @docType data
#' @format A data.frame containing 670 rows and 8 columns.
#' @references Obach, R. Scott, Franco Lombardo, and Nigel J. Waters. "Trend
#' analysis of a database of intravenous pharmacokinetic parameters in humans
#' for 670 drug compounds." Drug Metabolism and Disposition 36.7 (2008):
#' 1385-1405.
#' @keywords data
"Obach2008"


#' NHANES Exposure Data
#'
#' This data set is only used in Vignette 6.
#'
#' @docType data
#'
#' @format A data.table containing 1060 rows and 5 columns.
#'
#' @author Caroline Ring
#'
#' @references
#' Wambaugh, John F., et al. "High throughput heuristics for prioritizing human
#' exposure to environmental chemicals." Environmental science & technology
#' 48.21 (2014): 12760-12767.
#'
#' @keywords data
"onlyp"



#' Partition Coefficient Data
#'
#' Measured rat in vivo partition coefficients and data for predicting them.
#'
#'
#' @docType data
#' @format A data.frame.
#' @author Jimena Davis and Robert Pearce
#' @references
#' \insertRef{schmitt2008general}{httk}
#'
#' \insertRef{schmitt2008corrigendum}{httk}
#'
#' Poulin, P. and F.P. Theil, A priori prediction of tissue: plasma partition
#' coefficients of drugs to facilitate the use of physiologically based
#' pharmacokinetic models in drug discovery. Journal of pharmaceutical
#' sciences, 2000. 89(1): p. 16-35.
#'
#' Rodgers, T. and M. Rowland, Physiologically based pharmacokinetic modelling
#' 2: predicting the tissue distribution of acids, very weak bases, neutrals
#' and zwitterions. Journal of pharmaceutical sciences, 2006. 95(6): p.
#' 1238-1257.
#'
#' Rodgers, T., D. Leahy, and M. Rowland, Physiologically based pharmacokinetic
#' modeling 1: predicting the tissue distribution of moderate-to-strong bases.
#' Journal of pharmaceutical sciences, 2005. 94(6): p. 1259-1276.
#'
#' Rodgers, T., D. Leahy, and M. Rowland, Tissue distribution of basic drugs:
#' Accounting for enantiomeric, compound and regional differences amongst
#' beta-blocking drugs in rat. Journal of pharmaceutical sciences, 2005. 94(6):
#' p. 1237-1248.
#'
#' Gueorguieva, I., et al., Development of a whole body physiologically based
#' model to characterise the pharmacokinetics of benzodiazepines. 1: Estimation
#' of rat tissue-plasma partition ratios. Journal of pharmacokinetics and
#' pharmacodynamics, 2004. 31(4): p. 269-298.
#'
#' Poulin, P., K. Schoenlein, and F.P. Theil, Prediction of adipose tissue:
#' plasma partition coefficients for structurally unrelated drugs. Journal of
#' pharmaceutical sciences, 2001. 90(4): p. 436-447.
#'
#' Bjorkman, S., Prediction of the volume of distribution of a drug: which
#' tissue-plasma partition coefficients are needed? Journal of pharmacy and
#' pharmacology, 2002. 54(9): p. 1237-1245.
#'
#' \insertRef{yun2013correlation}{httk}
#'
#' Uchimura, T., et al., Prediction of human blood-to-plasma drug concentration
#' ratio. Biopharmaceutics & drug disposition, 2010. 31(5-6): p. 286-297.
#' @keywords data
"pc.data"



#' Species-specific physiology parameters
#'
#' This data set contains values from Davies and Morris (1993) necessary to
#' paramaterize a toxicokinetic model for human, mouse, rat, dog, or rabbit.
#' The temperature for each species are taken from Reece (2015),
#' Jordon (1995), and Stammers (1926). Mean residence time for the small 
#' intestine is from Grandoni et al. (2019). Human small intestine radius is
#' from Yu et al. (1999). Rat small intestine radius is from Griffin and
#' O'Driscoll (2008).
#'
#' @docType data
#' 
#' @format A data.frame containing 18 rows and 7 columns.
#' 
#' @author John Wambaugh and Nisha Sipes
#' 
#' @references 
#' \insertRef{davies1993physiological}{httk}
#' 
#' \insertRef{brown1997physiological}{httk}
#'
#' \insertRef{birnbaum1994physiological}{httk}
#'
#' \insertRef{reece201514}{httk}
#'
#' \insertRef{stammers1926blood}{httk}
#' 
#' \insertRef{jordan1995temperature}{httk}
#'
#' \insertRef{grandoni2019building}{httk}
#'
#' \insertRef{griffin2008models}{httk} 
#'
#' @examples
#' # We can add a new species (for example, wolverines) by adding new information
#' # to the physiology.data and tissue.data tables. It can be convenient to start by
#' # by replicating the data from another species and adjusting as appropriate:
#'
#' # Copy physiology data from rabbit:
#' new.species <- physiology.data[,"Rabbit"]
#' names(new.species) <- physiology.data[,"Parameter"]
#' rabbit.BW <- new.species["Average BW"] 
#' # Rausch and Pearson (1972) https://doi.org/10.2307/3799057 :
#' new.species["Average BW"] <- 31.2 
#' # Thiel et al. (2019) https://doi.org/10.1186/s12983-019-0319-8 :
#' new.species["Average Body Temperature"] <- 38.5 
#' 
#' # Add new physiology data column to physiology.data table"
#' physiology.data <- cbind(physiology.data, new.species)
#' colnames(physiology.data)[length(colnames(physiology.data))] <- "Wolverine"
#' 
#' # Copy tissue data from rabbit:
#' new.tissue.data <- subset(tissue.data,Species=="Rabbit")
#' new.tissue.data$Species <- "Wolverine"
#' 
#' # Add new tissue data rows to tissue.data table:
#' tissue.data <- rbind(tissue.data, new.tissue.data)
#' 
#' # Species is now available for calculations:
#' calc_mc_css(chem.cas="80-05-7",
#'             species="wolverine",
#'             parameterize.args.list =list(default.to.human=TRUE),
#'             suppress.messages=TRUE,
#'             samples = 100)
#'
#' @keywords data
"physiology.data"



#' Tissue composition and species-specific physiology parameters
#'
#' This data set contains values from Schmitt (2008) and Ruark et al. (2014)
#' describing the composition of specific tissues and from Birnbaum et al.
#' (1994) describing volumes of and blood flows to those tissues, allowing
#' parameterization of toxicokinetic models for human, mouse, rat, dog, or
#' rabbit. Tissue volumes were calculated by converting the fractional mass of
#' each tissue with its density (both from ICRP), lumping the remaining tissues
#' into the rest-of-body, excluding the mass of the gastrointestinal contents.
#'
#' Many of the parameters were compiled initially in Table 2 of Schmitt (2009).
#' The full list of tissue variables described is:
#' \tabular{lll}{
#' \strong{Variable} \tab \strong{Description} \tab \strong{Units} \cr
#' Fcell \tab Cellular fraction of total tissue volume \tab fraction \cr
#' Fint \tab Interstitial fraction of total tissue volume \tab fraction \cr
#' FWc \tab Fraction of cell volume that is water \tab fraction \cr
#' FLc \tab Fraction of cell volume that is lipid \tab fraction \cr
#' FPc \tab Fraction of cell volume that is protein \tab fraction \cr
#' Fn_Lc \tab Fraction of cellular lipid tht is neutral lipid \tab fraction \cr
#' Fn_PLc \tab Fraction of cellular lipid tht is neutral phospholipid \tab fraction \cr
#' Fa_PLc \tab Fraction of cellular lipid tht is acidic phospholipid \tab fraction \cr
#' pH \tab Negative logarithm of H+ ion concentration \tab unitless\cr
#' Density \tab Tissue density \tab g/cm^3 \cr
#' Vol \tab Tissue volume \tab L/kg \cr
#' Flow \tab Blood flow to tissue \tab mL/min/kg^(3/4) \cr
#' }
#'
#' New tissues can be added to this table to generate
#' their partition coefficients.
#'
#' @seealso \code{\link{predict_partitioning_schmitt}}
#' 
#' @docType data
#' 
#' @format A data.frame containing 406 rows and 5 columns.
#' \tabular{ll}{
#' \strong{Column} \tab \strong{Description} \cr
#' Tissue \tab The tissue being described \cr                      
#' Species \tab The species being described \cr                      
#' Reference \tab The reference for the value reported \cr                      
#' variable \tab The aspect of the tissue being characterized \cr                      
#' value \tab The value for the variable for the given tissue and species \cr                      
#' }
#' 
#' @author John Wambaugh, Robert Pearce, and Nisha Sipes
#' 
#' @references
#' \insertRef{birnbaum1994physiological}{httk}
#'
#' \insertRef{ruark2014predicting}{httk}
#'
#' \insertRef{schmitt2008general}{httk}
#'
#' \insertRef{snyder1974report}{httk}
#'
#' \insertRef{wambaugh2015toxicokinetic}{httk}
#'
#' @examples
#' # We can add thyroid to the tissue data by making a row containing
#' # its data, subtracting the volumes and flows from the rest-of-body, 
#' # and binding the row to tissue.data. Here we assume it contains the same 
#' # partition coefficient data as the spleen and a tenth of the volume and  
#' # blood flow:
#' new.tissue <- subset(tissue.data,Tissue == "spleen")
#' new.tissue[, "Tissue"] <- "thyroid"
#' new.tissue[new.tissue$variable %in% c("Vol (L/kg)",
#' "Flow (mL/min/kg^(3/4))"),"value"] <- new.tissue[new.tissue$variable
#' %in% c("Vol (L/kg)","Flow (mL/min/kg^(3/4))"),"value"] / 10
#' tissue.data[tissue.data$Tissue == "rest", "value"] <-
#' tissue.data[tissue.data$Tissue == "rest", "value"] -
#' new.tissue[new.tissue$variable %in% c("Vol (L/kg)",
#' "Flow (mL/min/kg^(3/4))"),"value"]
#' tissue.data <- rbind(tissue.data, new.tissue)
#'
#' # We can add a new species (for example, wolverines) by adding new information
#' # to the physiology.data and tissue.data tables. It can be convenient to start by
#' # by replicating the data from another species and adjusting as appropriate:
#'
#' # Copy physiology data from rabbit:
#' new.species <- physiology.data[,"Rabbit"]
#' names(new.species) <- physiology.data[,"Parameter"]
#' rabbit.BW <- new.species["Average BW"] 
#' # Rausch and Pearson (1972) https://doi.org/10.2307/3799057 :
#' new.species["Average BW"] <- 31.2 
#' # Thiel et al. (2019) https://doi.org/10.1186/s12983-019-0319-8 :
#' new.species["Average Body Temperature"] <- 38.5 
#' 
#' # Add new physiology data column to physiology.data table"
#' physiology.data <- cbind(physiology.data, new.species)
#' colnames(physiology.data)[length(colnames(physiology.data))] <- "Wolverine"
#' 
#' # Copy tissue data from rabbit:
#' new.tissue.data <- subset(tissue.data,Species=="Rabbit")
#' new.tissue.data$Species <- "Wolverine"
#' 
#' # Add new tissue data rows to tissue.data table:
#' tissue.data <- rbind(tissue.data, new.tissue.data)
#' 
#' # Species is now available for calculations:
#' calc_mc_css(chem.cas="80-05-7",
#'             species="wolverine",
#'             parameterize.args.list =list(default.to.human=TRUE),
#'             suppress.messages=TRUE,
#'             samples = 100)
#'
#' @keywords data
"tissue.data"

#' Published toxicokinetic predictions based on in vitro data from Wetmore et
#' al. 2012.
#'
#' This data set overlaps with Wetmore.data and is used only in Vignette 4 for
#' steady state concentration.
#'
#'
#' @docType data
#' @format A data.frame containing 13 rows and 15 columns.
#' @references 
#' \insertRef{wetmore2012integration}{httk}
#' @keywords data
"Wetmore2012"

#' Metabolism data involved in Linakis 2020 vignette analysis.
#'
#'
#' @docType data
#' @format A data.frame containing x rows and y columns.
#' @author Matt Linakis
#'
#' @source Matt Linakis
#' @references
#' \insertRef{linakis2020development}{httk}
#' @keywords data
"metabolism_data_Linakis2020"

#' Concentration data involved in Linakis 2020 vignette analysis.
#'
#' These rat and human TK concentration vs. time (CvT) data are drawn from
#' the CvTdb (Sayre et el., 2020). Concentrations have all been converted to
#' the units of uM. All data are from inhalation studies.
#'
#' Abbreviations used for sampling matrix:
#' BL : blood
#' EEB : end-exhaled breath
#' MEB : mixed exhaled breath
#' VBL : venous blood
#' ABL : arterial blood
#' EB : unspecified exhaled breath sample (assumed to be EEB)
#' PL: plasma
#' +W with work/exercise
#'
#' \tabular{ll}{
#' \strong{Column Name} \tab \strong{Description} \cr
#'  PREFERRED_NAME        \tab Substance preferred name \cr                      
#'  DTXSID     \tab Identifier for CompTox Chemical Dashboard \cr
#'  CASRN  \tab Chemical abstracts service registration number\cr
#'  AVERAGE_MASS  \tab Substance molecular weight g/mol \cr
#'   DOSE DOSE_U   \tab Inhalation exposure concentration in parts per million \cr  
#'  EXP_LENGTH   \tab Duration of inhalation exposur \cr
#'   TIME  \tab  Measurment time \cr          
#'  TIME_U   \tab  Time units for all times reported \cr 
#'  CONC_SPECIES  \tab Species for study \cr 
#'  SAMPLING_MATRIX   \tab Matrix analyzed \cr 
#'  SOURCE_CVT   \tab Data source identifier within CvTdb \cr 
#'  ORIG_CONC_U   \tab Original reported units for concentration \cr 
#'  CONCENTRATION   \tab Analyte concentration in uM units\cr 
#' }
#'
#' @docType data
#' @format A data.frame containing 2142 rows and 16 columns.
#' @author Matt Linakis
#' @references 
#' \insertRef{linakis2020development}{httk}
#' \insertRef{sayre2020database}{httk}
#' @source Matt Linakis
#' @keywords data
"concentration_data_Linakis2020"

#' Supplementary output from Linakis 2020 vignette analysis.
#'
#'
#' @docType data
#' @format A data.frame containing x rows and y columns.
#' @author Matt Linakis
#' @references 
#' \insertRef{linakis2020development}{httk}
#' @source Matt Linakis
#' @keywords data
"supptab1_Linakis2020"

#' More supplementary output from Linakis 2020 vignette analysis.
#'
#'
#' @docType data
#' @format A data.frame containing x rows and y columns.
#' @author Matt Linakis
#' @references 
#' \insertRef{linakis2020development}{httk}
#' @source Matt Linakis
#' @keywords data
"supptab2_Linakis2020"

#' Literature In Vivo Data on Doses Causing Neurological Effects
#'
#' Studies were selected from Table 1 in Mundy et al., 2015, as
#' the studies in that publication were cited as examples of
#' compounds with evidence for developmental neurotoxicity. There
#' were sufficient in vitro toxicokinetic data available for this
#' package for only 6 of the 42 chemicals.
#'
#' @docType data
#'
#' @format A data.frame containing 14 rows and 16 columns.
#'
#' @author Timothy J. Shafer
#'
#' @references
#' Frank, Christopher L., et al. "Defining toxicological tipping points
#' in neuronal network development." Toxicology and Applied
#' Pharmacology 354 (2018): 81-93.
#'
#' Mundy, William R., et al. "Expanding the test set: Chemicals with
#' potential to disrupt mammalian brain development." Neurotoxicology
#' and Teratology 52 (2015): 25-35.
#'
#' @keywords data
"Frank2018invivo"

#' Pearce et al. 2017 data
#'
#' This table includes the adjusted and unadjusted regression parameter
#' estimates for the chemical-specifc plasma
#' protein unbound fraction (fup) in 12 different tissue types.
#'
#' Predictions were made with regression models, 
#' as reported in Pearce et al. (2017). 
#'
#' @name pearce2017regression
#' @aliases Pearce2017Regression
#' @docType data
#' @format data.frame
#' @author Robert G. Pearce
#' @references
#' \insertRef{pearce2017evaluation}{httk}
#' @seealso \code{\link{predict_partitioning_schmitt}} 
#' @source Pearce et al. 2017 Regression Models
#' @keywords data
"pearce2017regression"

#' Dawson et al. 2021 data
#'
#' This table includes QSAR (Random Forest) model predicted values for unbound
#' fraction plasma protein (fup) and intrinsic hepatic clearance (clint) for a
#' subset of chemicals in the Tox21 library
#' (see \url{https://www.epa.gov/chemical-research/toxicology-testing-21st-century-tox21}).
#'
#' Predictions were made with a set of Random Forest QSAR models,
#' as reported in Dawson et al. (2021). 
#'
#' @name dawson2021
#' @aliases Dawson2021
#' @docType data
#' @format data.frame
#' @author Daniel E. Dawson
#' @seealso \code{\link{load_dawson2021}} 
#' @references
#' \insertRef{dawson2021qsar}{httk}
#' @keywords data
"dawson2021"

#' Kapraun et al. 2019 data
#' 
#' A list object containing time-varying parameters for the human maternal-fetal
#' HTTK model. List elements contain scalar coefficients for the polynomial,
#' logistic, Gompertz, and other functions of time describing blood flow rates,
#' tissue volumes, hematocrits, and other anatomical/physiological quantities
#' that change in the human mother and her fetus during pregnancy and gestation.
#' 
#' @name kapraun2019
#' @aliases Kapraun2019
#' @docType data
#' @format list
#' @author Dustin F. Kapraun
#' @references
#' \insertRef{kapraun2019empirical}{httk}
#' @source
#' \insertRef{kapraun2022fetalmodel}{httk}
#' @keywords data
"kapraun2019"

#' Pradeep et al. 2020
#' 
#' This table includes Support Vector Machine and Random Forest model predicted
#' values for unbound fraction plasma protein (fup) and intrinsic hepatic
#' clearance (clint) values for a subset of chemicals in the Tox21 library
#' (see \url{https://www.epa.gov/chemical-research/toxicology-testing-21st-century-tox21}).
#'
#' Prediction were made with Support Vector Machine and Random Forest models,
#' as reported in Pradeep et al. (2020).
#' 
#' @name pradeep2020
#' @aliases Pradeep2020
#' @docType data
#' @format data.frame
#' @seealso \code{\link{load_pradeep2020}} 
#' @references 
#' \insertRef{pradeep2020chemstr}{httk}
#' @keywords data
"pradeep2020"

#' Aylward et al. 2014
#' 
#' Aylward et al. (2014) compiled measurements of the ratio of maternal to fetal 
#' cord blood chemical concentrations at birth for a range of chemicals with 
#' environmental routes of exposure, including bromodiphenyl ethers, fluorinated 
#' compounds, organochlorine pesticides, polyaromatic hydrocarbons, tobacco smoke 
#' components, and vitamins.
#' 
#' @name aylward2014
#' @aliases Aylward2014
#' @docType data
#' @format data.frame
#' @references
#' \insertRef{Aylward2014matfet}{httk}
#' @source
#' \insertRef{kapraun2022fetalmodel}{httk}
#' @keywords data
"aylward2014"

#' AUCs for Pregnant and Non-Pregnant Women
#' 
#' Dallmann et al. (2018) includes compiled literature descriptions of 
#' toxicokinetic summary statistics, including time-integrated plasma 
#' concentrations (area under the curve or AUC) for drugs administered to a 
#' sample of subjects including both pregnant and non-pregnant women. The 
#' circumstances of the dosing varied slightly between drugs and are summarized 
#' in the table.
#' 
#' @name pregnonpregaucs
#' @aliases pregnonpregaucs
#' @docType data
#' @format data.frame
#' @references
#' \insertRef{dallmann2018pregpbtk}{httk}
#' @source
#' \insertRef{kapraun2022fetalmodel}{httk}
#' @keywords data
"pregnonpregaucs"  

#' Partition Coefficients from PK-Sim
#' 
#' Dallmann et al. (2018) made use of PK-Sim to predict chemical- and tissue-
#' specific partition coefficients. The methods include both the default
#' PK-Sim approach and PK-Sim Standard and Rodgers & Rowland (2006).
#' 
#' @name pksim.pcs
#' @docType data
#' @format data.frame
#' @references
#' \insertRef{dallmann2018pregpbtk}{httk}
#' @source
#' \insertRef{kapraun2022fetalmodel}{httk}
#' @keywords data
"pksim.pcs"  

#' Fetal Partition Coefficients
#' 
#' Partition coefficients were measured for tissues, including placenta, in 
#' vitro by Csanady et al. (2002) for Bisphenol A and Diadzen. Curley et al. 
#' (1969) measured the concentration of a variety of pesticides in the cord 
#' blood of newborns and in the tissues of infants that were stillborn. 
#' 
#' Three of the chemicals studied by Curley et al. (1969) were modeled by 
#' Weijs et al. (2013) using the same partition coefficients for mother and 
#' fetus. The values used represented "prior knowledge" summarizing the 
#' available literature. 
#' 
#' @name fetalpcs
#' @aliases fetalPCs
#' @docType data
#' @format data.frame
#' @references
#' \insertRef{Csanady2002fetalpc}{httk}
#' \insertRef{Curley1969fetalpc}{httk}
#' \insertRef{Weijs2013fetalpc}{httk}
#' @source
#' \insertRef{kapraun2022fetalmodel}{httk}
#' @keywords data
"fetalpcs"

#' Wang et al. 2018
#' Wang et al. (2018) screened the blood of 75 pregnant women for the presence 
#' of environmental organic acids (EOAs) and identified mass spectral features 
#' corresponding to 453 chemical formulae of which 48 could be mapped to likely 
#' structures. Of the 48 with tentative structures the identity of six were 
#' confirmed with available chemical standards. 
#' @name wang2018
#' @aliases Wang2018
#' @docType data
#' @format data.frame
#' @references
#' \insertRef{Wang2018matbloodnta}{httk}
#' @source
#' \insertRef{kapraun2022fetalmodel}{httk}
#' @keywords data
"wang2018"

#' ToxCast Example Data
#' The main page for the ToxCast data is here:
#' https://www.epa.gov/comptox-tools/exploring-toxcast-data
#' Most useful to us is a single file containing all the hits across all chemcials
#' and assays:
#' https://clowder.edap-cluster.com/datasets/6364026ee4b04f6bb1409eda?space=62bb560ee4b07abf29f88fef
#' 
#' As of November, 2022 the most recent version was 3.5 and was available as an
#' .Rdata file (invitrodb_3_5_mc5.Rdata)
#' 
#'  Unfortunately for this vignette there are too many ToxCast data to fit into a
#' 5mb R package. So we will subset to just the shemicals for the 
#' "Intro to IVIVE" vignette and distribute
#' only those data. In addition, out of 78 columns in the data, we will keep only 
#' eight.
#' @name example.toxcast
#' @docType data
#' @format data.frame
#' @keywords data
"example.toxcast"

#' SEEM Example Data
#' We can grab SEEM daily intake rate predictions already in RData format from
#' https://github.com/HumanExposure/SEEM3RPackage/tree/main/SEEM3/data
#' Download the file Ring2018Preds.RData
#' 
#' We do not have the space to distribute all the SEEM predictions within
#' this R package, but we can give you our "Intro to IVIVE" example chemicals
#' @name example.seem
#' @docType data
#' @format data.frame
#' @keywords data
#' @references
#' \insertRef{Ring2017SEEM}{httk}
"example.seem"

#' Measured Caco-2 Apical-Basal Permeability Data
#'
#' In vitro Caco-2 membrane permeabilities characterize how readily absobed/transported
#' a chemical is. These measurements are all for the apical-to-basal Caco-2
#' orientation.
#' These data were either measured by EPA or collected by other others, as
#' indicated by the column 'Data Origin'.
#' Anywhere that the values is reported by three numbers separated by a comma 
#' (this also happens for plasma protein binding) the three values are: median, 
#' lower 95 percent confidence intervals, upper 95 percent confidence interval. Unless you are 
#' doing monte carlo work it makes sense to ignore the second and third values.
#'
#' \tabular{lll}{
#' \strong{Column Name} \tab \strong{Description} \tab \strong{Units} \cr
#' DTXSID \tab EPA's DSSTox Structure ID (\url{https://comptox.epa.gov/dashboard}) \tab \cr
#' Pab \tab Apical-to-basal Caco-2 permeability \tab 10^-6 cm/s \cr
#' Data Origin \tab The reference which collected/generated the measurement \tab \cr
#' Test \tab Whether (1) or not (0) the data was withheld from model building to be used in the QSPR test set \tab \cr
#' CAS \tab Chemical Abstracts Service Registry Number \tab \cr
#' }
#' @references
#' \insertRef{obringer2016suitability}{httk}
#' 
#' \insertRef{lanevskij2019physicochemical}{httk}
#' 
#' \insertRef{gaulton2012chembl}{httk}
#' 
#' \insertRef{honda2025impact}{httk}
#' @name honda2023.data
"honda2023.data"

#' Predicted Caco-2 Apical-Basal Permeabilities
#'
#' Honda et al. (2023) describes the construction of a machine-learning 
#' quantitative structure-property relationship (QSPR )model for in vitro Caco-2 
#' membrane permeabilites. That model was used to make chemical-specific 
#' predictions provided in this table.
#'
#' \tabular{lll}{
#' \strong{Column Name} \tab \strong{Description} \tab \strong{Units} \cr
#' DTXSID \tab EPA's DSSTox Structure ID (\url{https://comptox.epa.gov/dashboard}) \tab \cr
#' Pab.Class.Pred \tab Predicted Pab rate of slow (1), moderate (2), or fast (3) \tab \cr
#' Pab.Pred.AD \tab Whether (1) or not (0) the chemical is anticipated to be withing the QSPR domain of applicability \tab \cr
#' CAS \tab Chemical Abstracts Service Registry Number \tab \cr
#' Pab.Quant.Pred \tab Median and 95-percent interval for values within the predicted class's training data  moderate (2), or fast (3) \tab 10^-6 cm/s \cr
#' }
#' @references
#' \insertRef{honda2025impact}{httk}
#' @seealso \code{\link{load_honda2023}} 
#' @name honda2023.qspr
"honda2023.qspr"

#' Historical Performance of R Package httk
#'
#' This table records the historical performance and other metrics 
#' of the R package "httk" as
#' profiled with the function \code{\link{benchmark_httk}}. There is a row for
#' each version and a column for each benchmark or metric. This table is used
#' to generate graphs comparing the current version to past performance in order
#' to help identify unintended degradtion of package capabilities.
#'
#' \tabular{ll}{
#' \strong{Column Name} \tab \strong{Description} \cr
#'  Version \tab The release of httk (major.minor.patch) \cr                      
#'  N.steadystate \tab The number of chemicals for which Css can be predicted for the steady-state model \cr
#'  calc_analytic.units \tab The ratio of the output of \code{\link{calc_analytic_css}} in mg/L to uM multiplied by 1000/MW (should be 1)\cr
#'  calc_mc.units  \tab The ratio of the output of \code{\link{calc_mc_css}} in mg/L to uM multiplied by 1000/MW (should be 1) \cr
#'  solve_pbtk.units  \tab The ratio of a Cplasma value from \code{\link{solve_pbtk}} in mg/L to uM multiplied by 1000/MW (should be 1) \cr  
#'  RMSLE.Wetmore \tab Root mean squared log10 error between Css predictions from httk and published values from Wetmore papers (both 95th percentile) \cr
#'  N.Wetmore  \tab  Number of chemicals used in RMSLE evaluation \cr          
#'  RMSLE.noMC  \tab  RMSLE between 95th percentile Css prediction and median prediction \cr 
#'  N.noMC  \tab Number of chemicals used in RMSLE evaluation \cr 
#'  RMSLE.InVivoCss  \tab RMSLE for predictions of in vivo measured Css \cr 
#'  N.InVivoCss  \tab Number of chemicals used in RMSLE evaluation  \cr 
#'  RMSLE.InVivoAUC   \tab RMSLE for predictions of in vivo measured AUCs \cr 
#'  N.InVivoAUC  \tab Number of chemicals used in RMSLE evaluation \cr 
#'  RMSLE.InVivoCmax \tab RMSLE for predictions of in vivo measured Cmax \cr 
#'  N.InVivoCmax  \tab Number of chemicals used in RMSLE evaluation \cr 
#'  RMSLE.TissuePC \tab RMSLE for predicted tissue:plasma partition coefficients \cr  
#'  N.TissuePC  \tab Number of chemicals used in RMSLE evaluation \cr                
#'  Notes  \tab Why benchmarks/metrics may have changed \cr       
#' }
#'
#' @seealso \code{\link{benchmark_httk}} 
#' @references
#' \insertRef{davidson2025enabling}{httk}
#' @name httk.performance
"httk.performance"

#' Sipes et al. 2017 data
#'
#' This table includes in silico predicted chemical-specifc plasma protein 
#' unbound fraction (fup) and intrinsic hepatic clearance values for the entire
#' Tox21 library 
#' (see \url{https://www.epa.gov/chemical-research/toxicology-testing-21st-century-tox21}). 
#' Predictions were made with Simulations Plus ADMET predictor,
#' as reported in Sipes et al. (2017). 
#'
#' @name sipes2017
#' @aliases Sipes2017
#' @docType data
#' @format data.frame
#' @author Nisha Sipes
#' @references
#' \insertRef{sipes2017intuitive}{httk}
#' @source ADMET, Simulations Plus
#' @seealso \code{\link{load_sipes2017}} 
#' @keywords data
"sipes2017"

#' ToxCast thyroid-related bioactivity data 
#'
#' Truong et al. 2025 uses ToxCast data for 4 thyroid-related assay endpoints
#' concerning inhibition of deiodinases ("DIO1", "DIO2", "DIO3", and "IYD") and 
#' identified 120 priority chemicals with activity for at least one deiodinase. 
#' These 120 chemicals were curated after assessment for target selectivity and 
#' assay interference. 
#'
#' The AC50s (in uM) for each of the 120 chemicals were retrieved from ToxCast 
#' invitrodb v3.5 and are used in the "Full Human Gestational IVIVE" vignette.
#' 
#' @name thyroid.ac50s
#' @docType data
#' @format data.table and data.frame
#' @keywords data
#' @references 
#' \insertRef{truong2025fullpregnancy}{httk}
"thyroid.ac50s"

#' SEEM3 Example Data for Truong et al. 2025 
#' 
#' We can grab SEEM daily intake rate predictions already in RData format from
#' https://github.com/HumanExposure/SEEM3RPackage/blob/main/scripts/
#' Download the file chem.preds-2018-11-28.RData
#' 
#' We do not have the space to distribute all the SEEM predictions within
#' this R package, but we can give you our "Full Human Gestational IVIVE" example chemicals.
#' @name truong25.seem3
#' @docType data
#' @format data.table and data.frame
#' @keywords data
#' @references 
#' \insertRef{truong2025fullpregnancy}{httk}
#'
#' \insertRef{Ring2017SEEM}{httk}
"truong25.seem3"

#' Literature Measurements of In Vitro Cellular and Nominal Concentration 
#' 
#' @author Meredith Scherer
#' @name Scherer2025.IVD
#' @docType data
#' @format data.table and data.frame
'Scherer2025.IVD' 

#' Dimitrijevic et al. (2022)In Vitro Cellular and Nominal Concentration 
#' 
#' @author Jon Arnot
#' @name Dimitrijevic.IVD
#' @docType data
#' @format data.table and data.frame
#' @references
#' \insertRef{dimitrijevic2022toward}{httk}
'Dimitrijevic.IVD'

#' ToxCast In Vitro Assay Descriptors
#' 
#' @author Madison Feshuk
#' @name invitro.assay.params
#' @docType data
#' @format data.table and data.frame
'invitro.assay.params'

#' Machine Learning PFAS Half-Life Predictions from Dawson et al. 2023 
#' 
#' Dawson et al. (2023) Supplemental Information S3 includes half-life predictions 
#' for 6603 PFAS, of which 3890 are estimated to be within the applicability
#' domain (AD) for humans. This machine learning (ML) model predicts PFAS 
#' half-life as one of four categories. The ML model was trained to a dataset 
#' of 91 in vivo measured TK half-lives across 11 PFAS, 4 species, and two 
#' sexes. Predictions were a function of compound-specific physico-chemical 
#' descriptors, species-specific physiological descriptors, and an indicator 
#' variable for sex. The kinetics of PFAS are thought to be complicated by 
#' active transport, both through either proximal tubular resorption (into the 
#' blood) (Andersen et al. 2006) or secretion (into the urine) 
#' (Kudo et al. 2002). The ML model uses several species- and structure-derived
#' surrogates for estimating the likelihood of active PFAS transport. Geometry
#' of the proximal tubule was a surrogate for transporter expression: since
#' secretion/resorption transporters line the surface of the proximal tubule,
#' the amount of surface area provides an upper limit on the amount of
#' transporter expression. PFAS similarity to three distinct endogenous ligands
#' was considered as a surrogate for transporter affinity. 
#' 
#' The Dawson et al. (2023) half-life categories are:
#' \tabular{ll}{
#' \strong{Category} \tab \strong{Range of Half-Lives} \cr
#'  1 \tab < 12 hours \cr                      
#'  2 \tab < 1 week \cr                      
#'  3 \tab < 2 months \cr                      
#'  4 \tab > 2 months \cr                      
#'  }
#'
#' The data.frame contains the following columns:
#' \tabular{ll}{
#' \strong{Column Name} \tab \strong{Description} \cr
#'  DTXSID \tab CompTox Chemicals Dashboard substance identifier \cr                      
#'  Species \tab Species for which the prediction was made \cr
#'  Sex \tab Sex for which the prediction was made \cr
#'  DosingAdj \tab Route of dose administration -- intravenous, oral, or other \cr
#'  ClassPredFull \tab The predicted half-life class (category) \cr  
#'  ClassModDomain \tab AD estimated from chemical classes of training set \cr
#'  AMAD \tab AD including AD predicted for each model used for descriptors \cr
#' }
#' 
#' @name dawson2023
#' @docType data
#' @format data.frame
#' @keywords data
#' @references 
#' \insertRef{dawson2023machine}{httk}
#'
#' \insertRef{andersen2006pharmacokinetic}{httk}
#'
#' \insertRef{kudo2002sex}{httk}
"dawson2023"

#' Interspecies In vivo Clearance Data for PFAS
#' 
#' If the chemical- and species-specific TK half-life (\eqn{t_{1/2}}) and volume of 
#' distribution (\eqn{V_{d}}) are known, a whole-body plasma clearance rate can be 
#' calculated as: \deqn{Cl_{tot} = ln(1/2) / t_{1/2} * V_{d}}
#' The first term involving the natural logarithm of 1/2 converts half-life (h) 
#' into an elimination rate (1/h) so that \eqn{Cl_{tot}} has units of 
#' L/kg bodyweight/h. 
#' Dawson et al. (2023) reported a table of in vivo PFAS \eqn{t_{1/2}} data for multiple 
#' species across eleven PFAS. Most of the measured data are for rodents. These
#'  data result from a series of reviews of the literature by 
#' Lau et al. (most recently Fenton et al. 2021) that were further revised for
#' Dawson et al. (2023).
#' Dawson et al. (2023) Supplemental Information S2.5 compiled \eqn{V_{d}}
#' values that 
#' were used here for calculating total clearance. A dataset of 
#' literature-derived values of \eqn{V_{d}} was compiled starting from 
#' Pizzurro et al. (2019) Table 2, which compiled 38 observations spanning 
#' five PFAS chemicals, four species, and both sexes from various literature 
#' sources. To these we added 24 calculated \eqn{V_{d}} observations in rat for three 
#' chemicals across a range of doses and routes from Huang et al. (2019). 
#' Further values for \eqn{V_{d}} were collected from the peer-review literature 
#' (Dzierlenga et al. 2020; Lau et al. 2020; Lou et al. 2009; Tatum-Gibbs et al. 2011).
#'  The total data set includes 128 values for \eqn{V_{d}} from 8 PFAS chemicals 
#' across 4 species. 
#' A \eqn{Cl_{tot}} was calculated using the above equation for every chemical- and 
#' species-specific half-life reported in the
#'  Dawson et al. (2023) supplemental materials. For chemicals without
#'  species- and compound-specific measurements for \eqn{V_{d}} we used the median
#'  in vivo measured PFAS Vd of 0.205 L/kg BW.
#' 
#' The data.frame contains the following columns:
#' \tabular{ll}{
#' \strong{Column Name} \tab \strong{Description} \cr
#'  DTXSID \tab CompTox Chemicals Dashboard substance identifier \cr                      
#'  Species \tab Species for which the clearance was calculated \cr
#'  Sex \tab Sex for which the clearance was calculated \cr
#'  HalfLifeHours \tab Half-life in hours \cr
#'  HlReference \tab Reference(s) for half-life \cr  
#'  VdLpkgbw  \tab Volume of distribution in L/kg body weight \cr
#'  VdReference \tab Reference for volume of distribution \cr
#'  Kelimph  \tab Elimination rate in 1/hour \cr
#'  ClLphpkgbw \tab Total clearance in L/h/kg body weight \cr
#' }
#' 
#' @name pfas.clearance
#' @docType data
#' @format data.frame
#' @keywords data
#' @references 
#' \insertRef{dawson2023machine}{httk}
#'
#' \insertRef{fenton2021per}{httk}
#'
#' \insertRef{pizzurro2019interspecies}{httk}
#'
#' \insertRef{huang2019toxicokinetics}{httk}
#'
#' \insertRef{dzierlenga2020toxicokinetics}{httk}
#'
#' \insertRef{lau2020pharmacokinetic}{httk}
#'
#' \insertRef{lou2009modeling}{httk}
#'
#' \insertRef{tatum2011comparative}{httk}
"pfas.clearance"
