#' Fit an epi curve
#'
#' @param dat An [incidence2::incidence] object.
#' @param model The regression model to fit (can be "poisson" or "negbin").
#' @param alpha Value of alpha used to calculate confidence intervals; defaults
#'   to 0.05 which corresponds to a 95% confidence interval.
#' @param ... Additional arguments to pass to [stats::glm()] for
#'   `model = "poisson"` or [MASS::glm.nb()] for `model = "negbin"`.
#'
#' @return An object of class `incidence2_fit`.
#'
#' @export
fit_curve <- function(dat, model, ...) {
  UseMethod("fit_curve")
}

#' @rdname fit_curve
#' @aliases fit_curve.default
#' @export
fit_curve.default <- function(dat, model, ...) {
  stop(sprintf("Not implemented for class %s",
               paste(class(dat), collapse = ", ")))
}


#' @rdname fit_curve
#' @aliases fit_curve.incidence2 incidence2_fit
#' @export
fit_curve.incidence2 <- function(dat,
                                 model = c("poisson", "negbin"),
                                 alpha = 0.05,
                                 ...) {
  model <- match.arg(model)
  groups <- incidence2::get_group_names(dat)
  dates <- incidence2::get_dates_name(dat)
  date_group <- incidence2::get_date_group_names(dat)
  count <- incidence2::get_counts_name(dat)
  
  fmla <- stats::as.formula(paste(count, "~", dates))
  trending_model <- switch(
    model,
    negbin = trending::glm_nb_model(fmla, ...),
    poisson = trending::glm_model(fmla, family = "poisson", ...),
    stop('Invalid model. Please use one of "negbin" or "poisson".')
  )

  if (!is.null(groups)) {
    out <- dplyr::nest_by(grouped_df(dat, groups))
    fiterr <- lapply(
      out$data,
      function(x) safely(trending::fit)(trending_model, x)
    )
    fiterr <- base_transpose(fiterr)
    prederr <- lapply(
      fiterr[[1]],
      function(x) safely(stats::predict)(x)
    )
    prederr <- base_transpose(prederr)
    model <- lapply(
      fiterr[[1]],
      function(x) safely(trending::get_model)(x)
    )
    model <- base_transpose(model)
    out$model <- model[[1]]
    out$estimates <-prederr[[1]] 
    out$fitting_warning <- fiterr[[2]]
    out$fitting_error <- fiterr[[3]]
    out$prediction_warning <-prederr[[3]]
    out$prediction_error <-prederr[[3]]
    out$data <- NULL
    warning_vars <- c("fitting_warning", "prediction_warning")
    error_vars <- c("fitting_error", "prediction_error")

  } else {
    fitted_model <- trending::fit(trending_model, data = dat)
    model <- trending::get_model(fitted_model)
    estimates <- stats::predict(fitted_model)
    out <- tibble::tibble(
      model = list(model),
      estimates = list(estimates)
    )
    warning_vars <- NULL
    error_vars <- NULL

  }

  # create subclass of tibble
  out <- tibble::new_tibble(out,
                            groups = groups,
                            date = dates,
                            date_group = date_group,
                            count = count,
                            interval = incidence2::get_interval(dat),
                            cumulative = attr(dat, "cumulative"),
                            model = "model",
                            fitted = "estimates",
                            warning_vars = warning_vars,
                            error_vars = error_vars,
                            nrow = nrow(out),
                            class = "incidence2_fit")

  tibble::validate_tibble(out)
}


