% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sat_vapor_pressure.R
\encoding{UTF-8}
\name{sat_vapor_pressure}
\alias{sat_vapor_pressure}
\title{Saturation Vapor Pressure for Water}
\usage{
sat_vapor_pressure(
  T,
  units = c("SI", "Eng", "Absolute"),
  formula = c("Huang", "Buck", "IAPWS")
)
}
\arguments{
\item{T}{numeric vector that contains the temperature (degrees Celsius,
degrees Fahrenheit, or Kelvin)}

\item{units}{character vector that contains the system of units (options are
\code{SI} for International System of Units, \code{Eng} for English units
(United States Customary System in the United States and Imperial Units in
the United Kingdom), or \code{Absolute} for Absolute Units)}

\item{formula}{character vector that contains the source of the formula used
to compute the saturation vapor pressure (options are Huang, Buck, IAPWS)}
}
\value{
the saturation vapor pressure for water as a numeric vector. The unit
  for \code{SI} and \code{Absolute} is Pascal (Pa), but the unit for \code{Eng}
  is pounds per square inch (psi). The units are not returned.
}
\description{
This function solves for the saturation vapor pressure of water using only
the temperature of the water in either units of degrees Celsius, degrees
Fahrenheit, or Kelvin.
}
\details{
The simplified equation is expressed as

\deqn{P_s = \exp{\frac{34.494 - \frac{4924.99}{t + 237.1}}{(t + 105)^1.57}}}

for (t > 0 C)

\describe{
  \item{\emph{P_s}}{the saturation vapor pressure (Pa or psi)}
  \item{\emph{t}}{the water temperature, degrees Celsius}
}
}
\note{
Note: Please refer to the references for the formulas (Huang = Reference 1,
IAPWS = Reference 2, and Buck = Reference 3)

Note: Please refer to the iemisc: Comparing Saturated Vapor Pressure
Formulas to the Reference vignette for the comparisons to the reference
saturated vapor pressure
}
\examples{

# Example 1 - Example from the hydraulics package

library("iemisc")

vps <- hydraulics::svp(T = 10, units = "SI"); vps

vps2 <- sat_vapor_pressure(T = 10, units = "SI", formula = "Huang"); vps2





# Example 2 - from the Reference

library("iemisc")

sat_vapor_pressure(T = c(0.01, seq(from = 20, to = 100, by = 20)), units = "SI", formula = "Huang")





# Example 3 - compare with saturation_pressure_H2O from aiRthermo

install.load::load_package("iemisc", "units")

T <- 40

# create a numeric vector with the units of degrees Celsius
T_C <- set_units(T, "degree_C")
T_C

# create a numeric vector to convert from degrees Celsius to Kelvin
T_K <- T_C
T_K

# create a numeric vector with the units of Kelvin
units(T_K) <- make_units(K)

pre <- aiRthermo::saturation_pressure_H2O(drop_units(T_K))
pre

sat_vapor_pressure(T = drop_units(T_K), units = "Absolute", formula = "Huang")






}
\references{
\enumerate{
   \item Huang, J. (2018). "A Simple Accurate Formula for Calculating Saturation Vapor Pressure of Water and Ice", \emph{Journal of Applied Meteorology and Climatology}, 57(6), 1265-1272. Retrieved Nov 4, 2021, \url{https://journals.ametsoc.org/view/journals/apme/57/6/jamc-d-17-0334.1.xml}
   \item The International Association for the Properties of Water and Steam. IAPWS SR1-86(1992). "Revised Supplementary Release on Saturation Properties of Ordinary Water Substance", September 1992, \url{http://www.iapws.org/relguide/Supp-sat.pdf}
   \item Holger Vömel, National Center for Atmospheric Research Earth Observing Laboratory, "Saturation vapor pressure formulations", \url{https://web.archive.org/web/20170623040102/http://cires1.colorado.edu/~voemel/vp.html}. Retrieved thanks to the Internet Archive: Wayback Machine
}
}
\author{
Irucka Embry
}
