% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/irboost.R
\name{irboost}
\alias{irboost}
\title{fit a robust predictive model with iteratively reweighted boosting algorithm}
\usage{
irboost(
  x,
  y,
  weights,
  cfun = "ccave",
  s = 1,
  delta = 0.1,
  dfun = "reg:squarederror",
  iter = 10,
  nrounds = 100,
  del = 1e-10,
  trace = FALSE,
  ...
)
}
\arguments{
\item{x}{input matrix, of dimension nobs x nvars; each row is an observation vector. Can accept \code{dgCMatrix}}

\item{y}{response variable. Quantitative for \code{dfun="greg:squarederror"}, \code{dfun="count:poisson"} (non-negative counts) or \code{dfun="reg:gamma"} (positive). For \code{dfun="binary:logitraw" or "binary:hinge"}, \code{y} should be a factor with two levels}

\item{weights}{vector of nobs with non-negative weights}

\item{cfun}{concave component of CC-family, can be \code{"hacve", "acave", "bcave", "ccave"}, 
\code{"dcave", "ecave", "gcave", "hcave"}. See Table 2 in https://arxiv.org/pdf/2010.02848.pdf}

\item{s}{tuning parameter of \code{cfun}. \code{s > 0} and can be equal to 0 for \code{cfun="tcave"}. If \code{s} is too close to 0 for    \code{cfun="acave", "bcave", "ccave"}, the calculated weights can become 0 for all observations, thus crash the program}

\item{delta}{a small positive number provided by user only if \code{cfun="gcave"} and \code{0 < s <1}}

\item{dfun}{type of convex component in the CC-family, the second C, or convex down, that's where the name \code{dfun} comes from. It is the same as \code{objective} in the \code{xgboost} package.
  \itemize{
    \item \code{reg:squarederror} Regression with squared loss.
    \item \code{binary:logitraw} logistic regression for binary classification, predict linear predictor, not probabilies.
    \item \code{binary:hinge} hinge loss for binary classification. This makes predictions of -1 or 1, rather than   producing probabilities.
    \item \code{multi:softprob} softmax loss function for multiclass problems. The result contains predicted probabilities of each data point in each class, say p_k, k=0, ..., nclass-1. Note, \code{label} is coded as in [0, ..., nclass-1]. The loss function cross-entropy for the i-th observation is computed as -log(p_k) with k=lable_i, i=1, ..., n.
    \item \code{count:poisson}: Poisson regression for count data, predict mean of poisson distribution.
    \item \code{reg:gamma}: gamma regression with log-link, predict mean of gamma distribution. The implementation in \code{xgboost} takes a parameterization in the exponential family:\cr
xgboost/src/src/metric/elementwise_metric.cu.\cr
In particularly, there is only one parameter psi and set to 1. The implementation of the IRCO algorithm follows this parameterization. See Table 2.1, McCullagh and Nelder, Generalized linear models, Chapman & Hall, 1989, second edition.
    \item \code{reg:tweedie}: Tweedie regression with log-link. See also \code{tweedie_variance_power} in range: (1,2). A value close to 2 is like a gamma distribution. A value close to 1 is like a Poisson distribution.
}}

\item{iter}{number of iteration in the IRCO algorithm}

\item{nrounds}{boosting iterations within each IRCO iteration}

\item{del}{convergency criteria in the IRCO algorithm, no relation to \code{delta}}

\item{trace}{if \code{TRUE}, fitting progress is reported}

\item{...}{other arguments passing to \code{xgboost}}
}
\value{
An object with S3 class \code{xgboost} with the additional elments:
\itemize{
  \item \code{weight_update_log} a matrix of \code{nobs} row by {iter}      column of observation weights in each iteration of the IRCO algorithm
  \item \code{weight_update} a vector of observation weights in the last IRCO iteration that produces the final model fit
  \item\code{loss_log}{sum of loss value of the composite function \code{cfun(dfun)} in each IRCO iteration. Note, \code{cfun} requires \code{dfun} non-negative in some cases. Thus some \code{dfun} needs attentions. For instance, with \code{dfun="reg:gamma"}, the loss value is defined gamma-nloglik - (1+log(min(y))). The second term is introduced such that the loss value is non-negative. In fact, gamma-nloglik=y/ypre + log(ypre) in the \code{xgboost}, where ypre is the mean prediction value, can be negative. It can be derived that for fixed \code{y}, the minimum value of gamma-nloglik is achived at ypre=y, or 1+log(y). Thus, among all \code{y} values, the minimum of gamma-nloglik is 1+log(min(y)).}
}
}
\description{
Fit a predictive model with the iteratively reweighted convex optimization (IRCO) that minimizes the robust loss functions in the CC-family (concave-convex). The convex optimization is conducted by functional descent boosting algorithm in the R package \pkg{xgboost}. The iteratively reweighted boosting (IRBoost) algorithm reduces the weight of the observation that leads to a large loss; it also provides weights to help identify outliers. Applications include the robust generalized
linear models and extensions, where the mean is related to the predictors by boosting, and robust accelerated failure time models.
}
\examples{
\donttest{
# regression, logistic regression, hinge regression, Poisson regression
x <- matrix(rnorm(100*2),100,2)
g2 <- sample(c(0,1),100,replace=TRUE)
fit1 <- irboost(x, g2, cfun="acave",s=0.5, dfun="reg:squarederror", trace=TRUE, 
                verbose=0, max.depth=1, nrounds=50)
fit2 <- irboost(x, g2, cfun="acave",s=0.5, dfun="binary:logitraw", trace=TRUE,  
                verbose=0, max.depth=1, nrounds=50)
fit3 <- irboost(x, g2, cfun="acave",s=0.5, dfun="binary:hinge", trace=TRUE,  
                verbose=0, max.depth=1, nrounds=50)
fit4 <- irboost(x, g2, cfun="acave",s=0.5, dfun="count:poisson", trace=TRUE,      
                verbose=0, max.depth=1, nrounds=50)

# Gamma regression
x <- matrix(rnorm(100*2),100,2)
g2 <- sample(rgamma(100, 1))
library("xgboost")
fit5 <- xgboost(x, g2, objective="reg:gamma", max.depth=1, nrounds=50)
fit6 <- irboost(x, g2, cfun="acave",s=5, dfun="reg:gamma", trace=TRUE, 
                verbose=0, max.depth=1, nrounds=50)
plot(predict(fit5, x), predict(fit6, x))
hist(fit6$weight_update)
plot(fit6$loss_log)
summary(fit6$weight_update)

# Tweedie regression 
fit6t <- irboost(x, g2, cfun="acave",s=5, dfun="reg:tweedie", trace=TRUE, 
                verbose=0, max.depth=1, nrounds=50)
# Gamma vs Tweedie regression
hist(fit6$weight_update)
hist(fit6t$weight_update)
plot(predict(fit6, x), predict(fit6t, x))

# multiclass classification in iris dataset:
lb <- as.numeric(iris$Species)-1
num_class <- 3
set.seed(11)

# xgboost
bst <- xgboost(data=as.matrix(iris[, -5]), label=lb,
max_depth=4, eta=0.5, nthread=2, nrounds=10, subsample=0.5,
objective="multi:softprob", num_class=num_class)
# predict for softmax returns num_class probability numbers per case:
pred <- predict(bst, as.matrix(iris[, -5]))
# reshape it to a num_class-columns matrix
pred <- matrix(pred, ncol=num_class, byrow=TRUE)
# convert the probabilities to softmax labels
pred_labels <- max.col(pred)-1
# classification error
sum(pred_labels!=lb)/length(lb)

# irboost
fit7 <- irboost(x=as.matrix(iris[, -5]), y=lb, cfun="acave", s=50,
                dfun="multi:softprob", trace=TRUE, verbose=0, 
                max.depth=4, eta=0.5, nthread=2, nrounds=10, 
                subsample=0.5, num_class=num_class)
pred7 <- predict(fit7, as.matrix(iris[, -5]))
pred7 <- matrix(pred7, ncol=num_class, byrow=TRUE)
# convert the probabilities to softmax labels
pred7_labels <- max.col(pred7) - 1
# classification error: 0!
sum(pred7_labels != lb)/length(lb)
table(pred_labels, pred7_labels)
hist(fit6$weight_update)
}
}
\references{
Wang, Zhu (2021), \emph{Unified Robust Boosting}, arXiv eprint, \url{https://arxiv.org/abs/2101.07718}
}
\author{
Zhu Wang\cr Maintainer: Zhu Wang \email{zhuwang@gmail.com}
}
\keyword{classification}
\keyword{regression}
