% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/reval_mst.R
\name{reval_mst}
\alias{reval_mst}
\title{Recursion-based MST evaluation method}
\usage{
reval_mst(
  x,
  D = 1,
  route_map,
  module,
  cut_score,
  theta = seq(-5, 5, 1),
  intpol = TRUE,
  range.tcc = c(-7, 7),
  tol = 1e-04
)
}
\arguments{
\item{x}{A data frame containing the metadata for the item bank, which includes
important information for each item such as the number of score categories and the
IRT model applied. This metadata is essential for evaluating the MST panel, with
items selected based on the specifications in the \code{module} argument. To construct
this item metadata efficiently, the \code{\link{shape_df}} function is recommended.
Further details on utilizing item bank metadata along with \code{module} for
MST panel evaluation are provided below.}

\item{D}{A scaling factor in IRT models to make the logistic function as
close as possible to the normal ogive function (if set to 1.7). Default is 1.}

\item{route_map}{A binary square matrix that defines the MST structure, illustrating
transitions between modules and stages. This concept and structure are inspired by
the \code{transMatrix} argument in the \code{randomMST} function
from the \pkg{mstR} package (Magis et al., 2017), which provides a framework for representing MST pathways.
For comprehensive understanding and examples of constructing \code{route_map},
refer to the \pkg{mstR} package (Magis et al., 2017) documentation. Also see below for details.}

\item{module}{A binary matrix that maps items from the item bank specified in \code{x}
to modules within the MST framework. This parameter's structure is analogous to the
\code{modules} argument in the \code{randomMST}  function of the \pkg{mstR}
package, enabling precise item-to-module assignments for MST configurations.
For detailed instructions on creating and utilizing the \code{module} matrix
effectively, consult the documentation of the \pkg{mstR} package (Magis et al., 2017).
Also see below for details.}

\item{cut_score}{A list defining cut scores for routing test takers through MST stages.
Each list element is a vector of cut scores for advancing participants to subsequent
stage modules. In a 1-3-3 MST configuration, for example, \code{cut_score} might be
defined as \code{cut_score = list(c(-0.5, 0.5), c(-0.6, 0.6))}, where c(-0.5, 0.5)
are thresholds for routing from the first to the second stage, and c(-0.6, 0.6)
for routing from the second to the third stage. This setup facilitates tailored
test progression based on performance. Further examples and explanations are
available below.}

\item{theta}{A vector of ability levels (theta) at which the MST panel's performance
is assessed. This allows for the evaluation of measurement precision and bias
across a continuum of ability levels. The default range is \code{theta = seq(-5, 5, 0.1)}.}

\item{intpol}{A logical value to enable linear interpolation in the inverse
test characteristic curve (TCC) scoring, facilitating ability estimate approximation
for observed sum scores not directly obtainable from the TCC, such as those below
the sum of item guessing parameters. Default is TRUE, applying interpolation
to bridge gaps in the TCC. Refer to \code{\link{est_score}} for more details and
consult Lim et al. (2020) for insights into the interpolation technique
within inverse TCC scoring.}

\item{range.tcc}{A vector to define the range of ability estimates for
inverse TCC scoring, expressed as the two numeric values for lower and upper
bounds. Default is to c(-7, 7).}

\item{tol}{A numeric value of the convergent tolerance for the  inverse TCC
scoring. For the inverse TCC scoring, the bisection method is used for
optimization. Default is 1e-4.}
}
\value{
This function returns a list of seven internal objects. The four objects are:

\item{panel.info}{A list of several sub-objects containing detailed information
about the MST panel configuration, including:
    \describe{
      \item{config}{A nested list indicating the arrangement of modules across
      stages, showing which modules are included in each stage. For example,
      the first stage includes module 1, the second stage includes
      modules 2 to 4, and so forth.}
      \item{pathway}{A matrix detailing all possible pathways through the MST panel.
      Each row represents a unique path a test taker might take, based on their
      performance and the cut scores defined.}
      \item{n.module}{A named vector indicating the number of modules available
      at each stage.}
      \item{n.stage}{A single numeric value representing the total number of
      stages in the MST panel.}
   }
}

\item{item.by.mod}{A list where each entry represents a module
in the MST panel, detailing the item metadata within that module. Each module's
metadata includes item IDs, the number of categories, the IRT model
used (model), and the item parameters (e.g., par.1, par.2, par.3).}

\item{item.by.path}{A list containing item metadata arranged
according to the paths through the MST structure. This detailed
breakdown allows for an analysis of item characteristics along specific
MST paths. Each list entry corresponds to a testing stage and path,
providing item metadata. This structure facilitates the examination of
how items function within the context of each unique path through the MST.}

\item{eq.theta}{
Estimated ability levels (\eqn{\theta}) corresponding to the observed
scores, derived from the inverse TCC scoring method. This provides the
estimated \eqn{\theta} values for each potential pathway through the
MST stages. For each stage, \eqn{\theta} values are calculated for each
path, indicating the range of ability levels across the test takers.
For instance, in a three-stage MST, the \code{eq.theta} list may contain
\eqn{\theta} estimates for multiple paths within each stage, reflecting
the progression of ability estimates as participants move through the test.
The example below illustrates the structure of \code{eq.theta} output for a
1-3-3 MST panel with varying paths:
     \describe{
       \item{stage.1}{\code{path.1} shows \eqn{\theta} estimates ranging from -7 to +7,
       demonstrating the initial spread of abilities.}
       \item{stage.2}{Multiple paths (\code{path.1}, \code{path.2}, ...) each with
       their own \eqn{\theta} estimates, indicating divergence in ability levels
       based on test performance.}
       \item{stage.3}{Further refinement of \eqn{\theta} estimates across paths,
       illustrating the final estimation of abilities after the last stage.}
  }
}

\item{cdist.by.mod}{A list where each entry contains the conditional distributions
of the observed scores for each module given the ability levels.}

\item{jdist.by.path}{Joint distributions of observed scores for different paths
at each stage in a MST panel. The example below outlines the organization of
\code{jdist.by.path} data in a hypothetical 1-3-3 MST panel:
     \describe{
       \item{stage.1}{Represents the distribution at the initial stage, indicating
       the broad spread of test-taker abilities at the outset.}
       \item{stage.2}{Represents the conditional joint distributions of the observed
       scores as test-takers move through different paths at the stage 2, based on
       their performance in earlier stages.}
       \item{stage.3}{Represents a further refinement of joint distribution of
       observed scores as test-takers move through different paths at the final
       stage 3, based on their performance in earlier stages.}
 }
}

\item{eval.tb}{
A table summarizing the measurement precision of the MST panel. It contains the true
ability levels (\code{theta}) with the mean ability estimates (\code{mu}), variance
(\code{sigma2}), bias, and conditional standard error of measurement (CSEM) given
the true ability levels. This table highlights the MST panel's accuracy and
precision across different ability levels, providing insights into its effectiveness
in estimating test-taker abilities.}
}
\description{
This function evaluates the measurement precision and bias in
Multistage-adaptive Test (MST) panels using a recursion-based evaluation
method introduced by Lim et al. (2020). This function computes conditional
biases and standard errors of measurement (CSEMs) across a range of IRT
ability levels, facilitating efficient and accurate MST panel
assessments without extensive simulations.
}
\details{
The \code{\link{reval_mst}} function evaluates an MST panel by
implementing a recursion-based method to assess measurement precision
across IRT ability levels. This approach, detailed in Lim et al. (2020),
enables the computation of conditional biases and CSEMs efficiently,
bypassing the need for extensive simulations traditionally required
for MST evaluation.

The \code{module} argument, used in conjunction with the item bank metadata
\code{x}, systematically organizes items into modules for MST panel evaluation.
Each row of \code{x} corresponds to an item, detailing its characteristics
like score categories and IRT model. The \code{module} matrix, structured
with the same number of rows as \code{x} and columns representing modules,
indicates item assignments with 1s. This precise mapping enables
the \code{\link{reval_mst}} function to evaluate the MST panel's
performance by analyzing how items within each module contribute
to measurement precision and bias, reflecting the tailored progression
logic inherent in MST designs.

The \code{route_map} argument is essential for defining the MST's structure
by indicating possible transitions between modules. Similar to the \code{transMatrix}
in the \pkg{mstR} package (Magis et al., 2017), \code{route_map} is a binary matrix that outlines
which module transitions are possible within an MST design. Each "1" in the matrix
represents a feasible transition from one module (row) to another (column),
effectively mapping the flow of test takers through the MST based on their
performance. For instance, a "1" at the intersection of row \emph{i} and
column \emph{j} indicates the possibility for test takers to progress from
the module corresponding to row \emph{i} directly to the module denoted
by column \emph{j}. This structure allows \code{\link{reval_mst}} to simulate and evaluate
the dynamic routing of test takers through various stages and modules of the MST panel.

To further detail the \code{cut_score} argument with an illustration:
In a 1-3-3 MST configuration, the list \code{cut_score = list(c(-0.5, 0.5), c(-0.6, 0.6))}
operates as a decision guide at each stage. Initially, all test takers start
in the first module. Upon completion, their scores determine their next stage module:
scores below -0.5 route to the first module of the next stage, between -0.5
and 0.5 to the second, and above 0.5 to the third. This pattern allows for
dynamic adaptation, tailoring the test path to individual performance levels.
}
\examples{
\donttest{
## ------------------------------------------------------------------------------
# Evaluation of a 1-3-3 MST panel using simMST data.
# This simulation dataset was utilized in Lim et al.'s (2020) simulation study.
# Details:
#    (a) Panel configuration: 1-3-3 MST panel
#    (b) Test length: 24 items (each module contains 8 items across all stages)
#    (c) IRT model: 3-parameter logistic model (3PLM)
## ------------------------------------------------------------------------------
# Load the necessary library
library(dplyr)
library(tidyr)
library(ggplot2)

# Import item bank metadata
x <- simMST$item_bank

# Import module information
module <- simMST$module

# Import routing map
route_map <- simMST$route_map

# Import cut scores for routing to subsequent modules
cut_score <- simMST$cut_score

# Import ability levels (theta) for evaluating measurement precision
theta <- simMST$theta

# Evaluate MST panel using the reval_mst() function
eval <-
  reval_mst(x,
    D = 1.702, route_map = route_map, module = module,
    cut_score = cut_score, theta = theta, range.tcc = c(-5, 5)
  )

# Review evaluation results
# The evaluation result table below includes conditional biases and
# standard errors of measurement (CSEMs) across ability levels
print(eval$eval.tb)

# Generate plots for biases and CSEMs
p_eval <-
  eval$eval.tb \%>\%
  dplyr::select(theta, bias, csem) \%>\%
  tidyr::pivot_longer(
    cols = c(bias, csem),
    names_to = "criterion", values_to = "value"
  ) \%>\%
  ggplot2::ggplot(mapping = ggplot2::aes(x = theta, y = value)) +
  ggplot2::geom_point(mapping = ggplot2::aes(shape = criterion), size = 3) +
  ggplot2::geom_line(
    mapping = ggplot2::aes(
      color = criterion,
      linetype = criterion
    ),
    linewidth = 1.5
  ) +
  ggplot2::labs(x = expression(theta), y = NULL) +
  ggplot2::theme_classic() +
  ggplot2::theme_bw() +
  ggplot2::theme(legend.key.width = unit(1.5, "cm"))
print(p_eval)
}

}
\references{
Magis, D., Yan, D., & Von Davier, A. A. (2017). \emph{Computerized adaptive
and multistage testing with R: Using packages catR and mstR}. Springer.

Lim, H., Davey, T., & Wells, C. S. (2020). A recursion-based analytical
approach to evaluate the performance of MST. \emph{Journal of Educational
Measurement, 58}(2), 154-178.
}
\seealso{
\code{\link{shape_df}}, \code{\link{est_score}}
}
\author{
Hwanggyu Lim \email{hglim83@gmail.com}
}
