% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/relation.R
\name{relation-locate}
\alias{relation-locate}
\alias{iv_locate_overlaps}
\alias{iv_locate_precedes}
\alias{iv_locate_follows}
\title{Locate relationships between two ivs}
\usage{
iv_locate_overlaps(
  needles,
  haystack,
  ...,
  type = "any",
  missing = "equals",
  no_match = NA_integer_,
  remaining = "drop",
  multiple = "all"
)

iv_locate_precedes(
  needles,
  haystack,
  ...,
  closest = FALSE,
  missing = "equals",
  no_match = NA_integer_,
  remaining = "drop",
  multiple = "all"
)

iv_locate_follows(
  needles,
  haystack,
  ...,
  closest = FALSE,
  missing = "equals",
  no_match = NA_integer_,
  remaining = "drop",
  multiple = "all"
)
}
\arguments{
\item{needles, haystack}{\verb{[iv]}

Interval vectors used for relation matching.
\itemize{
\item Each element of \code{needles} represents the interval to search for.
\item \code{haystack} represents the intervals to search in.
}

Prior to comparison, \code{needles} and \code{haystack} are coerced to the same type.}

\item{...}{These dots are for future extensions and must be empty.}

\item{type}{\verb{[character(1)]}

The type of relationship to find. One of:
\itemize{
\item \code{"any"}: Finds any overlap whatsoever between an interval in \code{needles}
and an interval in \code{haystack}.
\item \code{"within"}: Finds when an interval in \code{needles} is completely within
(or equal to) an interval in \code{haystack}.
\item \code{"contains"}: Finds when an interval in \code{needles} completely contains
(or equals) an interval in \code{haystack}.
\item \code{"equals"}: Finds when an interval in \code{needles} is exactly equal to
an interval in \code{haystack}.
\item \code{"starts"}: Finds when the start of an interval in \code{needles} matches the
start of an interval in \code{haystack}.
\item \code{"ends"}: Finds when the end of an interval in \code{needles} matches the end
of an interval in \code{haystack}.
}}

\item{missing}{\verb{[integer(1) / "equals" / "drop" / "error"]}

Handling of missing intervals in \code{needles}.
\itemize{
\item \code{"equals"} considers missing intervals in \code{needles} as exactly equal
to missing intervals in \code{haystack} when determining if there is a
matching relationship between them.
\item \code{"drop"} drops missing intervals in \code{needles} from the result.
\item \code{"error"} throws an error if any intervals in \code{needles} are missing.
\item If a single integer is provided, this represents the value returned in
the \code{haystack} column for intervals in \code{needles} that are missing.
}}

\item{no_match}{Handling of \code{needles} without a match.
\itemize{
\item \code{"drop"} drops \code{needles} with zero matches from the result.
\item \code{"error"} throws an error if any \code{needles} have zero matches.
\item If a single integer is provided, this represents the value returned in
the \code{haystack} column for observations of \code{needles} that have zero
matches. The default represents an unmatched needle with \code{NA}.
}}

\item{remaining}{Handling of \code{haystack} values that \code{needles} never matched.
\itemize{
\item \code{"drop"} drops remaining \code{haystack} values from the result.
Typically, this is the desired behavior if you only care when \code{needles}
has a match.
\item \code{"error"} throws an error if there are any remaining \code{haystack}
values.
\item If a single integer is provided (often \code{NA}), this represents the value
returned in the \code{needles} column for the remaining \code{haystack} values
that \code{needles} never matched. Remaining \code{haystack} values are always
returned at the end of the result.
}}

\item{multiple}{Handling of \code{needles} with multiple matches. For each needle:
\itemize{
\item \code{"all"} returns all matches detected in \code{haystack}.
\item \code{"any"} returns any match detected in \code{haystack} with no guarantees on
which match will be returned. It is often faster than \code{"first"} and
\code{"last"} if you just need to detect if there is at least one match.
\item \code{"first"} returns the first match detected in \code{haystack}.
\item \code{"last"} returns the last match detected in \code{haystack}.
\item \code{"warning"} throws a warning if multiple matches are detected, but
otherwise falls back to \code{"all"}.
\item \code{"error"} throws an error if multiple matches are detected.
}}

\item{closest}{\verb{[TRUE / FALSE]}

Should only the closest relationship be returned?

If \code{TRUE}, will only return the closest interval(s) in \code{haystack} that
the current value of \code{needles} either precedes or follows. Note that
multiple intervals can still be returned if there are ties, which can
be resolved using \code{multiple}.}
}
\value{
A data frame containing two integer columns named \code{needles} and \code{haystack}.
}
\description{
This family of functions locates different types of relationships between
two ivs. It works similar to \code{\link[base:match]{base::match()}}, where \code{needles[i]} checks for
a relationship in all of \code{haystack}. Unlike \code{match()}, \emph{all} matching
relationships are returned, rather than just the first.
\itemize{
\item \code{iv_locate_overlaps()} locates a specific \code{type} of overlap between the two
ivs.
\item \code{iv_locate_precedes()} locates where \code{needles[i]} precedes (i.e. comes
before) any interval in \code{haystack}.
\item \code{iv_locate_follows()} locates where \code{needles[i]} follows (i.e. comes
after) any interval in \code{haystack}.
}

These functions return a two column data frame. The \code{needles} column is an
integer vector pointing to locations in \code{needles}. The \code{haystack} column is
an integer vector pointing to locations in \code{haystack} with a matching
relationship.
}
\examples{
x <- iv_pairs(
  as.Date(c("2019-01-05", "2019-01-10")),
  as.Date(c("2019-01-07", "2019-01-15")),
  as.Date(c("2019-01-20", "2019-01-31"))
)

y <- iv_pairs(
  as.Date(c("2019-01-01", "2019-01-03")),
  as.Date(c("2019-01-04", "2019-01-08")),
  as.Date(c("2019-01-07", "2019-01-09")),
  as.Date(c("2019-01-10", "2019-01-20")),
  as.Date(c("2019-01-15", "2019-01-20"))
)

x
y

# Find any overlap between `x` and `y`
loc <- iv_locate_overlaps(x, y)
loc

iv_align(x, y, locations = loc)

# Find where `x` contains `y` and drop results when there isn't a match
loc <- iv_locate_overlaps(x, y, type = "contains", no_match = "drop")
loc

iv_align(x, y, locations = loc)

# Find where `x` precedes `y`
loc <- iv_locate_precedes(x, y)
loc

iv_align(x, y, locations = loc)

# Filter down to find only the closest interval in `y` of all the intervals
# where `x` preceded it
loc <- iv_locate_precedes(x, y, closest = TRUE)

iv_align(x, y, locations = loc)

# Note that `closest` can result in duplicates if there is a tie.
# `2019-01-20` appears as an end date twice in `haystack`.
loc <- iv_locate_follows(x, y, closest = TRUE)
loc

iv_align(x, y, locations = loc)

# Force just one of the ties to be returned by using `multiple`.
# Here we just request any of the ties, with no guarantee on which one.
loc <- iv_locate_follows(x, y, closest = TRUE, multiple = "any")
loc

iv_align(x, y, locations = loc)

# ---------------------------------------------------------------------------

a <- iv(NA, NA)
b <- iv(c(NA, NA), c(NA, NA))

# By default, missing intervals in `needles` are seen as exactly equal to
# missing intervals in `haystack`, which means that they overlap
iv_locate_overlaps(a, b)

# If you'd like missing intervals in `needles` to always be considered
# unmatched, set `missing = NA`
iv_locate_overlaps(a, b, missing = NA)
}
\seealso{
\link[=relation-detect]{Detecting relationships}

\link[=relation-detect-pairwise]{Detecting relationships pairwise}

\link[=allen-relation-locate]{Locating relations from Allen's Interval Algebra}
}
