% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/jamba-heatmap-order.R
\name{heatmap_column_order}
\alias{heatmap_column_order}
\title{Return Heatmap column order from ComplexHeatmap heatmap object}
\usage{
heatmap_column_order(hm, which_heatmap = NULL)
}
\arguments{
\item{hm}{\code{Heatmap} or \code{HeatmapList} object as defined by the
Bioconductor package via \code{ComplexHeatmap::Heatmap()}.}

\item{which_heatmap}{used to specify a specific heatmap with \code{hm} is
provided as a \code{HeatmapList}. When \code{NULL} (default) the first heatmap
in \code{hm@ht_list} is used. When one value is supplied, only that heatmap
is used. When multiple values are supplied, a \code{list} is returned.
Input can be either:
\itemize{
\item \code{numeric} - indicating the heatmap number in \code{hm@ht_list}
\item \code{character} - indicating the heatmap name seen in \code{names(hm@ht_list)}
}}
}
\value{
output depends upon the heatmap:
\itemize{
\item When heatmap columns are grouped using \code{column_split}, and
when the data matrix contains colnames,
returns a \code{character} vector of colnames in the order they
appear in the heatmap. When there are no colnames, \code{integer}
column index values are returned. If the heatmap has column labels,
they are returned as vector names.
\item When columns are grouped using \code{column_split}, it returns a
\code{list} of vectors as described above. The \code{list}
is named using the \code{column_title} labels only when there is
an equal number of column labels.
}
}
\description{
Return Heatmap column order from ComplexHeatmap heatmap object
}
\details{
This function is a helpful utility to return the fully
qualified list of colnames in a \code{ComplexHeatmap::Heatmap}
object.

The core intention is for the output to be usable with the
original data matrix used in the heatmap. Therefore, the
vector values are \code{colnames()} when present, or \code{integer}
column index values when there are no \code{colnames()}. If heatmap
\code{column_labels} are defined, they are returned as \code{names()}.

Note that \code{names()} are assigned inside \code{try()} to allow the
case where \code{column_labels}, or \code{column_title} labels cannot be
coerced to \code{character} values, for example using \code{gridtext}
for markdown formatting.
}
\examples{
if (check_pkg_installed("ComplexHeatmap")) {
   set.seed(123);

   mat <- matrix(stats::rnorm(18 * 24),
      ncol=24);
   rownames(mat) <- paste0("row", seq_len(18))
   colnames(mat) <- paste0("column", seq_len(24))

   # obtaining row order first causes a warning message
   hm1 <- ComplexHeatmap::Heatmap(mat);

   # best practice is to draw() and store output in an object
   # to ensure the row orders are absolutely fixed
   hm1_drawn <- ComplexHeatmap::draw(hm1);
   print(heatmap_row_order(hm1_drawn))
   print(heatmap_column_order(hm1_drawn))

   # row and column split
   hm1_split <- ComplexHeatmap::Heatmap(mat,
      column_split=3, row_split=3, border=TRUE);
   hm1_split_drawn <- ComplexHeatmap::draw(hm1_split);
   print(heatmap_row_order(hm1_split_drawn))
   print(heatmap_column_order(hm1_split_drawn))

   # display two heatmaps side-by-side
   mat2 <- mat + stats::rnorm(18*24);
   hm2 <- ComplexHeatmap::Heatmap(mat2, border=TRUE, row_split=4);

   hm1hm2_drawn <- ComplexHeatmap::draw(hm1_split + hm2,
      ht_gap=grid::unit(1, "cm"));
   print(heatmap_row_order(hm1hm2_drawn))
   print(heatmap_row_order(hm1hm2_drawn, which_heatmap=2))
   # by default the order uses the first heatmap
   print(heatmap_column_order(hm1hm2_drawn))
   # the second heatmap can be returned
   print(heatmap_column_order(hm1hm2_drawn, which_heatmap=2))
   # or a list of heatmap orders can be returned
   print(heatmap_column_order(hm1hm2_drawn, which_heatmap=1:2))

   # stacked vertical heatmaps
   hm1hm2_drawn_tall <- ComplexHeatmap::draw(
      ComplexHeatmap::`\%v\%`(hm1_split, hm2),
      ht_gap=grid::unit(1, "cm"));
   print(heatmap_row_order(hm1hm2_drawn))
   print(heatmap_row_order(hm1hm2_drawn, which_heatmap=2))
   print(heatmap_row_order(hm1hm2_drawn, which_heatmap=1:2))
   print(heatmap_row_order(hm1hm2_drawn,
      which_heatmap=names(hm1hm2_drawn@ht_list)))

   # annotation heatmap
   ha <- ComplexHeatmap::rowAnnotation(left=rownames(mat))
   ha_drawn <- ComplexHeatmap::draw(ha + hm1)
   print(sdim(ha_drawn@ht_list))
   print(heatmap_row_order(ha_drawn))
   print(heatmap_column_order(ha_drawn))

   # stacked vertical heatmaps with top annotation
   ta <- ComplexHeatmap::HeatmapAnnotation(top=colnames(mat))
   hm1_ha <- ComplexHeatmap::Heatmap(mat,
      left_annotation=ha,
      column_split=3, row_split=3, border=TRUE);
   hm1hm2_drawn_tall <- ComplexHeatmap::draw(
      ComplexHeatmap::`\%v\%`(ta,
         ComplexHeatmap::`\%v\%`(hm1_ha, hm2)),
      ht_gap=grid::unit(1, "cm"));
   print(sdim(hm1hm2_drawn_tall@ht_list))
   print(heatmap_row_order(hm1hm2_drawn_tall))
   print(heatmap_row_order(hm1hm2_drawn_tall, 2))
}

}
\seealso{
Other jam heatmap functions: 
\code{\link{cell_fun_label}()},
\code{\link{heatmap_row_order}()}
}
\concept{jam heatmap functions}
