% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/kde1d.R
\name{kde1d}
\alias{kde1d}
\title{Univariate local-polynomial likelihood kernel density estimation}
\usage{
kde1d(x, xmin = NaN, xmax = NaN, mult = 1, bw = NA, deg = 2)
}
\arguments{
\item{x}{vector (or one-column matrix/data frame) of observations; can be
\code{numeric} or \code{ordered}.}

\item{xmin}{lower bound for the support of the density (only for continuous
data); \code{NaN} means no boundary.}

\item{xmax}{upper bound for the support of the density (only for continuous
data); \code{NaN} means no boundary.}

\item{mult}{positive bandwidth multiplier; the actual bandwidth used is
\eqn{bw*mult}.}

\item{bw}{bandwidth parameter; has to be a positive number or \code{NA}; the
latter uses the direct plug-in methodology of Sheather and Jones (1991).}

\item{deg}{degree of the polynomial; either \code{0}, \code{1}, or \code{2} for log-constant,
log-linear, and log-quadratic fitting, respectively.}
}
\value{
An object of class \code{kde1d}.
}
\description{
The estimator can handle for bounded, unbounded, and discrete support, see
\emph{Details}.
}
\details{
A gaussian kernel is used in all cases. If \code{xmin} or \code{xmax} are
finite, the density estimate will be 0 outside of \eqn{[xmin, xmax]}. A
log-transform is used if there is only one boundary (see, Geenens and Wang,
2018); a probit transform is used if there are two (see, Geenens, 2014).
Discrete variables are handled via jittering (see, Nagler, 2018a, 2018b).
}
\examples{

## unbounded data
x <- rnorm(500)                    # simulate data
fit <- kde1d(x)                    # estimate density
dkde1d(0, fit)                     # evaluate density estimate
summary(fit)                       # information about the estimate
plot(fit)                          # plot the density estimate
curve(dnorm(x), add = TRUE,        # add true density
      col = "red")

## bounded data, log-linear
x <- rgamma(500, shape = 1)        # simulate data
fit <- kde1d(x, xmin = 0, deg = 1) # estimate density
dkde1d(seq(0, 5, by = 1), fit)     # evaluate density estimate
summary(fit)                       # information about the estimate
plot(fit)                          # plot the density estimate
curve(dgamma(x, shape = 1),        # add true density
      add = TRUE, col = "red",
      from = 1e-3)

## discrete data
x <- rbinom(500, size = 5, prob = 0.5)  # simulate data
x <- ordered(x, levels = 0:5)           # declare as ordered
fit <- kde1d(x)                         # estimate density
dkde1d(sort(unique(x)), fit)            # evaluate density estimate
summary(fit)                            # information about the estimate
plot(fit)                               # plot the density estimate
points(ordered(0:5, 0:5),               # add true density
       dbinom(0:5, 5, 0.5), col = "red")

}
\references{
Geenens, G. (2014). \emph{Probit transformation for kernel density estimation on
the unit interval}. Journal of the American Statistical Association,
109:505, 346-358, \href{https://arxiv.org/abs/1303.4121}{arXiv:1303.4121}

Geenens, G., Wang, C. (2018). \emph{Local-likelihood transformation kernel
density estimation for positive random variables.} Journal of Computational
and Graphical Statistics, to appear,
\href{https://arxiv.org/abs/1602.04862}{arXiv:1602.04862}

Nagler, T. (2018a). \emph{A generic approach to nonparametric function
estimation with mixed data.} Statistics & Probability Letters, 137:326–330,
\href{https://arxiv.org/abs/1704.07457}{arXiv:1704.07457}

Nagler, T. (2018b). \emph{Asymptotic analysis of the jittering kernel density
estimator.} Mathematical Methods of Statistics, in press,
\href{https://arxiv.org/abs/1705.05431}{arXiv:1705.05431}

Sheather, S. J. and Jones, M. C. (1991). A reliable data-based bandwidth
selection method for kernel density estimation. Journal of the Royal
Statistical Society, Series B, 53, 683–690.
}
\seealso{
\code{\link[=dkde1d]{dkde1d()}}, \code{\link[=pkde1d]{pkde1d()}}, \code{\link[=qkde1d]{qkde1d()}}, \code{\link[=rkde1d]{rkde1d()}},
\code{\link[=plot.kde1d]{plot.kde1d()}}, \code{\link[=lines.kde1d]{lines.kde1d()}}
}
