% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/CovFormulas.R
\name{parseCovFormula}
\alias{parseCovFormula}
\title{Parse a Formula or Expression Describing a Composite
Covariance Kernel}
\usage{
parseCovFormula(formula, where = .GlobalEnv, trace = 0)
}
\arguments{
\item{formula}{A formula or expression describing a covariance
kernel. See \bold{Examples}.}

\item{where}{An environment where kernel objects and top
parameters are searched for.}

\item{trace}{Integer level of verbosity.}
}
\value{
A list with the results of parsing. Although the results
content is easy to understand, the function is not intended to be
used by the final user, and the results may change in future
versions.
}
\description{
Parse a formula (or expression) describing a composite covariance
kernel.
}
\details{
The formula involves existing covariance kernel objects and must
define a valid kernel composition rule. For instance the sum and
the product of kernels, the convex combination of kernels are
classically used. The kernels objects are used in the formula with
parentheses as is they where functions calls with no formal
arguments e.g. \code{obj( )}. Non-kernel objects used in the
formula must be numeric scalar parameters and are called \emph{top}
parameters. The covariance objects must exist in the environment
defined by \code{where} because their slots will be used to
identify the inputs and the parameters of the composite kernel
defined by the formula.
}
\note{
The parsing separates covariance objects from top
parameters.  It retrieves information about the kernel inputs and
parameters from the slots. Obviously, any change in the
covariances objects after the parsing (e.g. change in the
parameters names or values) will not be reported in the results of
the parsing, so kernel any needed customization must be done prior
to the parsing.
}
\section{Caution}{
 Only relatively simple formulas are correctly
parsed. So use only formulas having a structure similar to one of
those given in the examples. In case of problems, error messages
are likely to be difficult to understand.
}

\examples{
## =========================================================================
## build some kernels (with their inputNames) in the global environment
## =========================================================================

myCovExp3 <- kMatern(d = 3, nu = "1/2")
inputNames(myCovExp3) <- c("x", "y", "z")

myCovGauss2 <- kGauss(d = 2)
inputNames(myCovGauss2) <- c("temp1", "temp2")

k <- kMatern(d = 1)
inputNames(k) <- "x"

ell <- kMatern(d = 1)
inputNames(ell) <- "y"

## =========================================================================
## Parse a formula. This formula is stupid because 'myCovGauss2'
## and 'myCovExp3' should be CORRELATION kernels and not
## covariance kernels to produce an identifiable model.
## =========================================================================

cov <- ~ tau2 * myCovGauss2() * myCovExp3() + sigma2 * k()
pf <- parseCovFormula(cov, trace = 1)

## =========================================================================
## Parse a formula with ANOVA composition
## =========================================================================

cov1 <- ~ tau2 * myCovGauss2() * myCovExp3() + sigma2 * (1 + k()) * (1 + ell())
pf1 <- parseCovFormula(cov1, trace = 1) 

}
\author{
Yves Deville
}
