### iid2.R --- 
#----------------------------------------------------------------------
## author: Brice Ozenne
## created: okt 12 2017 (13:16) 
## Version: 
## last-updated: mar 13 2018 (12:48) 
##           By: Brice Ozenne
##     Update #: 427
#----------------------------------------------------------------------
## 
### Commentary: 
## 
### Change Log:
#----------------------------------------------------------------------
## 
### Code:

## * Documentation - iid2
#' @title  Extract corrected i.i.d. decomposition
#' @description  Extract corrected i.i.d. decomposition from a gaussian linear model.
#' @name iid2
#'
#' @param object a linear model or a latent variable model
#' @param param [optional] the fitted parameters.
#' @param data [optional] the data set.
#' @param cluster [integer vector] the grouping variable relative to which the observations are iid.
#' Only required for \code{gls} models without correlation structure.
#' @param bias.correct [logical] should the standard errors of the coefficients be corrected for small sample bias? Only relevant if the \code{sCorrect} function has not yet be applied to the object.
#' @param robust [logical] if \code{FALSE}, the i.i.d. decomposition is rescaled such its the squared sum equals the model-based standard error (instead of the robust standard error).
#' @param ... arguments to be passed to \code{sCorrect}.
#'
#' @details If argument \code{p} or \code{data} is not null, then the small sample size correction is recomputed to correct the influence function.
#'
#' @seealso \code{\link{sCorrect}} to obtain \code{lm2}, \code{gls2}, \code{lme2}, or \code{lvmfit2} objects.
#'
#' @return A matrix containing the 1st order influence function relative to each sample (in rows)
#' and each model coefficient (in columns).
#' 
#' @examples
#' n <- 5e1
#' p <- 3
#' X.name <- paste0("X",1:p)
#' link.lvm <- paste0("Y~",X.name)
#' formula.lvm <- as.formula(paste0("Y~",paste0(X.name,collapse="+")))
#'
#' m <- lvm(formula.lvm)
#' distribution(m,~Id) <- sequence.lvm(0)
#' set.seed(10)
#' d <- sim(m,n)
#'
#' ## linear model
#' e.lm <- lm(formula.lvm,data=d)
#' iid.tempo <- iid2(e.lm, bias.correct = FALSE)
#' range(iid.tempo[,1:4]-iid(e.lm))
#' 
#' 
#' ## latent variable model
#' e.lvm <- estimate(lvm(formula.lvm),data=d)
#' iid.tempo <- iid2(e.lvm, bias.correct = FALSE)
#' range(iid.tempo-iid(e.lvm))
#' ## difference due to the use of the observed info matrix vs. the expected one.
#'
#' ## rescale i.i.d using model-based standard error
#' iid.tempo <- iid2(e.lvm, robust = FALSE, bias.correct = FALSE)
#' diag(crossprod(iid.tempo))-diag(vcov(e.lvm))
#'
#' @concept small sample inference
#' @concept iid decomposition
#' @export
`iid2` <-
  function(object, ...) UseMethod("iid2")

## * iid2.lm
#' @rdname iid2
#' @export
iid2.lm <- function(object, param = NULL, data = NULL, bias.correct = TRUE, ...){

    sCorrect(object, param = param, data = data,
             score = TRUE, df = FALSE) <- bias.correct

    ### ** export
    return(iid2(object, ...))
}

## * iid2.gls
#' @rdname iid2
#' @export
iid2.gls <- function(object, cluster,
                     param = NULL, data = NULL, bias.correct = TRUE, ...){

    sCorrect(object, cluster = cluster, param = param, data = data,
             score = TRUE, df = FALSE) <- bias.correct

    ### ** export
    return(iid2(object, ...))
}

## * iid2.lme
#' @rdname iid2
#' @export
iid2.lme <- iid2.lm

## * iid2.lvmfit
#' @rdname iid2
#' @export
iid2.lvmfit <- iid2.lm


## * iid2.lm2
#' @rdname iid2
#' @export
iid2.lm2 <- function(object, param = NULL, data = NULL, robust = TRUE, ...){

### ** compute the score
    if(!is.null(param) || !is.null(data)){
        args <- object$sCorrect$args
        args$df <- FALSE
        args$score <- TRUE
        object$sCorrect <- do.call(sCorrect,
                                   args = c(list(object, param = param, data = data),
                                            args))
    }else if(is.null(object$sCorrect$score)){
        stop("set argument \'score\' to TRUE when calling sCorrect \n")
    }

### ** compute iid
    res <- score2(object) %*% object$sCorrect$vcov.param
    if(robust == FALSE){
        vec.sigma <- sqrt(diag(object$sCorrect$vcov.param))
        vec.sigma.robust <- sqrt(apply(res^2,2,sum))
        res <- sweep(res, MARGIN = 2, FUN = "*", STATS = vec.sigma/vec.sigma.robust)
    }
    
### ** export
    return(res)

}

## * iid2.gls
#' @rdname iid2
#' @export
iid2.gls2 <- iid2.lm2

## * iid2.lme
#' @rdname iid2
#' @export
iid2.lme2 <- iid2.lm2

## * iid2.lvmfit
#' @rdname iid2
#' @export
iid2.lvmfit2 <- iid2.lm2



##----------------------------------------------------------------------
### iid2.R ends here
