% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/segment_trees.R
\name{segment_trees}
\alias{segment_trees}
\title{Individual tree segmentation}
\usage{
segment_trees(las, algorithm, attribute = "treeID", uniqueness = "incremental")
}
\arguments{
\item{las}{An object of class \link[lidR:LAS-class]{LAS} or \link[lidR:LAScatalog-class]{LAScatalog}.}

\item{algorithm}{function. An algorithm of individual tree segmentation. \code{lidR} has:
\link{dalponte2016}, \link{watershed}, \link{li2012} and \link{silva2016}.
More experimental algorithms may be found in the package \href{https://github.com/Jean-Romain/lidRplugins}{lidRplugins}.}

\item{attribute}{character. The returned LAS object as a new extra byte attribute (in a new column).
This parameter controls the name of the new attribute. Default is \code{"treeID"}.}

\item{uniqueness}{character. A method to compute a unique ID. Can be 'incremental', 'gpstime' or
'bitmerge'. See section 'Uniqueness'. This feature must be considered as 'experimental'.}
}
\value{
If the input is a \code{LAS} object, return a \code{LAS} object. If the input is a
\code{LAScatalog}, returns a \code{LAScatalog}.
}
\description{
Individual tree segmentation with several possible algorithms. The returned point cloud has a new
extra byte attribute named after the parameter \code{attribute} independently of the algorithm used.
}
\section{Uniqueness}{


By default the tree IDs are numbered from 1 to n, n being the number of trees found. The problem
with such incremental numbering is that, while it ensures a unique ID is assigned for each tree in
a given point-cloud, it also guarantees duplication of tree IDs in different tiles or chunks when
processing a \code{LAScatalog}. This is because each file is processed independently of the others and potentially
in parallel on different computers. Thus, the index always restarts at 1 on each file or chunk. Worse,
in a tree segmentation process, a tree that is located exactly between 2 files will have two different
IDs for its two halves.

This is why we introduced some uniqueness strategies that are all imperfect and that should be seen
as experimental. Please report any troubleshooting. Using a uniqueness-safe strategy ensures that
trees from different files will not share the same IDs. Moreover, it also means that two halves of a tree
on the edge of a processing chunk will be assigned the same ID.

\describe{
\item{incremental}{Number from 0 to n. This method \strong{does not} ensure uniqueness of the IDs. This
is the legacy method.}
\item{gpstime}{This method uses the gpstime of the highest point of a tree (apex) to create a
unique ID. This ID is not an integer but a 64-bit decimal number which is suboptimal but at
least it is exepected to be unique \strong{if the gpstime attribute is consistent across files}.
If inconsistencies with gpstime are reported (for example gpstime records the week time and was
reset to 0 in a coverage that takes more than a week to complete), there is a (low) probability to get
ID attribution errors.}
\item{bitmerge}{This method uses the XY coordinates of the highest point (apex) of a tree to
create a single number with a bitwise operation. First, XY coordinates are converted to integers using the
scales and offsets of the point-cloud. Then the ID is computed with X * 2^32 + Y to combine twice the 32-bits
of information into a 64-bit number. For example, if the apex is at (10.32, 25.64)
with a scale factor of 0.01 and an offset of 0, the integer coordinates are X = 1032 and Y = 2564
and the ID is 4432406252036. Such methods return a 64-bit integer but because 64-bit integers do
not exist in R it is converted to a 64-bit decimal number that is guaranteed to be unique
\strong{if all files have the same offsets and scale factors}.}
}

All the proposed options are suboptimal because they either do not guarantee uniqueness in all cases
(inconsistencies in the collection of files), or they imply that IDs are based on non-integers or
meaningless numbers. But at the very least we expect this to work for simple cases.
}

\section{Working with a \code{LAScatalog}}{

This section appears in each function that supports a LAScatalog as input.\cr

In \code{lidR} when the input of a function is a \link[lidR:LAScatalog-class]{LAScatalog} the
function uses the LAScatalog processing engine. The user can modify the engine options using
the \link[lidR:catalog_options_tools]{available options}. A careful reading of the
\link[lidR:LAScatalog-class]{engine documentation} is recommended before processing \code{LAScatalogs}. Each
\code{lidR} function should come with a section that documents the supported engine options.\cr

The \code{LAScatalog} engine supports \code{.lax} files that \emph{significantly} improve the computation
speed of spatial queries using a spatial index. Users should really take advantage a \code{.lax} files,
but this is not mandatory.\cr
}

\section{Supported processing options}{

Supported processing options for a \code{LAScatalog} (in bold). For more details see the
\link[lidR:LAScatalog-class]{LAScatalog engine documentation}:
\itemize{
\item \strong{chunk size}: How much data is loaded at once.
\item \strong{chunk buffer*}: Mandatory to get a continuous output without edge effects. The buffer
is always removed once processed and will never be returned either in R or in files.
\item \strong{chunk alignment}: Align the processed chunks.
\item \strong{progress}: Displays a progression estimation.
\item \strong{output files*}: Mandatory because the output is likely to be too big to be returned
in R and needs to be written in las/laz files. Supported templates are \code{\{XLEFT\}}, \code{\{XRIGHT\}},
\code{\{YBOTTOM\}}, \code{\{YTOP\}}, \code{\{XCENTER\}}, \code{\{YCENTER\}} \code{\{ID\}} and, if
chunk size is equal to 0 (processing by file), \code{\{ORIGINALFILENAME\}}.
\item select: The function will write files equivalent to the original ones. Thus \code{select = "*"}
and cannot be changed.
\item \strong{filter}: Read only points of interest.
}
}

\examples{
LASfile <- system.file("extdata", "MixedConifer.laz", package="lidR")
las <- readLAS(LASfile, select = "xyz", filter = "-drop_z_below 0")

# Using Li et al. (2012)
las <- segment_trees(las, li2012(R = 3, speed_up = 5))
plot(las, color = "treeID")
}
