% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lp_lin_iv.R
\name{lp_lin_iv}
\alias{lp_lin_iv}
\title{Compute linear impulse responses with identified shock (instrument variable approach)}
\usage{
lp_lin_iv(endog_data, instr = NULL, lags_endog_lin = NULL,
  exog_data = NULL, lags_exog = NULL, contemp_data = NULL,
  lags_criterion = NaN, max_lags = NaN, trend = NULL,
  confint = NULL, hor = NULL, num_cores = NULL)
}
\arguments{
\item{endog_data}{A \link{data.frame}, containing the dependent variables.}

\item{instr}{One column \link{data.frame} including the values of the instrument to shock with.
The row length has to be the same as \emph{endog_data}.}

\item{lags_endog_lin}{NaN or integer. NaN if lags are chosen by lag length criterion. Integer for number of lags for \emph{endog_data}.}

\item{exog_data}{A \link{data.frame}, containing exogenous variables for the VAR. The row length has to be the same as \emph{endog_data}.
Lag lengths for exogenous variables have to be given and will no be determined via a lag length criterion.}

\item{lags_exog}{NULL or Integer. Integer for the number of lags for the exogenous data.}

\item{contemp_data}{A \link{data.frame}, containing exogenous data with contemporaneous impact.
The row length has to be the same as \emph{endog_data}.}

\item{lags_criterion}{NaN or character. NaN means that the number of lags
will be given at \emph{lags_endog_lin}. The character refers to the corresponding lag length criterion ('AICc', 'AIC' or 'BIC').}

\item{max_lags}{NaN or integer. Maximum number of lags if \emph{lags_criterion} is character with lag length criterion. NaN otherwise.}

\item{trend}{Integer. No trend =  0 , include trend = 1, include trend and quadratic trend = 2.}

\item{confint}{Double. Width of confidence bands. 68\% = 1, 90\% = 1.65, 95\% = 1.96.}

\item{hor}{Integer. Number of horizons for impulse responses.}

\item{num_cores}{NULL or Integer. The number of cores to use for the estimation. If NULL, the function will
use the maximum number of cores minus one.}
}
\value{
A list containing:



\item{irf_lin_mean}{A \link{matrix} containing the impulse responses.
                   The row in each matrix denotes the response of the \emph{ith}
                   variable to the (instrument) shock. The columns are the horizons.}

\item{irf_lin_low}{A \link{matrix} containing all lower confidence bands of
                   the impulse responses, based on robust standard errors by Newey and West (1987).
                   Properties are equal to \emph{irf_lin_mean}.}

\item{irf_lin_up}{A \link{matrix} containing all upper confidence bands of
                   the impulse responses, based on robust standard errors by Newey and West (1987).
                   Properties are equal to \emph{irf_lin_mean}.}

\item{specs}{A list with properties of \emph{endog_data} for the plot function. It also contains
            lagged data (y_lin and x_lin) used for the estimations of the irfs.}
}
\description{
Compute linear impulse responses with local projections and identified shock, i.e.
instrument variable approach (see e.g. Jordà et al., 2015; and Ramey and Zubairy, 2018).
}
\examples{
\donttest{

# This example replicates a result from the Supplementary Appendix
# by Ramey and Zubairy (2018) (RZ-18).

# Load data
 ag_data       <- ag_data
 sample_start  <- 7
 sample_end    <- dim(ag_data)[1]

# Endogenous data
 endog_data    <- ag_data[sample_start:sample_end,3:5]

# Shock ('Instrument')
 shock         <- ag_data[sample_start:sample_end, 3]

# Estimate linear model
 results_lin_iv <- lp_lin_iv(endog_data,
                               lags_endog_lin = 4,
                               instr          = shock,
                               exog_data      = NULL,
                               lags_exog      = NULL,
                               contemp_data   = NULL,
                               lags_criterion = NaN,
                               max_lags       = NaN,
                               trend          = 0,
                               confint        = 1.96,
                               hor            = 20,
                               num_cores      = NULL)


# Make and save plots
 iv_lin_plots    <- plot_lin(results_lin_iv)

# * The first element of 'iv_lin_plots' shows the response of the first
#   variable (Gov) to the chosen (instrument-)shock (here Gov).
# * The second element of 'iv_lin_plots' shows the response of the second
#   variable (Tax) to the chosen (instrument-)shock (Gov).
# * ...

# This plot replicates the left plot in the mid-panel of Figure 12 in the
# Supplementary Appendix by RZ-18.
 iv_lin_plots[[1]]

# Show all impulse responses by using 'ggpubr' and 'gridExtra'
# The package does not depend on those packages so they have to be installed
 library(ggpubr)
 library(gridExtra)

 lin_plots_all <- sapply(iv_lin_plots, ggplotGrob)
 marrangeGrob(lin_plots_all, nrow = ncol(endog_data), ncol = 1, top = NULL)

}
}
\references{
Akaike, H. (1974). "A new look at the statistical model identification", \emph{IEEE Transactions on Automatic Control}, 19 (6): 716–723.

Auerbach, A. J., and  Gorodnichenko Y. (2012). "Measuring the Output Responses to Fiscal Policy."
\emph{American Economic Journal: Economic Policy}, 4 (2): 1-27.

Hurvich, C. M., and  Tsai, C.-L. (1989), "Regression and time series model selection in small samples",
\emph{Biometrika}, 76(2): 297–307

Jordà, Ò. (2005). "Estimation and Inference of Impulse Responses by Local Projections."
\emph{American Economic Review}, 95 (1): 161-182.

Jordà, Ò, Schularick, M., Taylor, A.M. (2015), "Betting the house", \emph{Journal of International Economics},
96, S2-S18.

Newey, W.K., and West, K.D. (1987). “A Simple, Positive-Definite, Heteroskedasticity and
Autocorrelation Consistent Covariance Matrix.” \emph{Econometrica}, 55: 703–708.

Ramey, V.A., and Zubairy, S. (2018). "Government Spending Multipliers in Good Times
and in Bad: Evidence from US Historical Data." \emph{Journal of Political Economy},
126(2): 850 - 901.

Schwarz, Gideon E. (1978). "Estimating the dimension of a model", \emph{Annals of Statistics}, 6 (2): 461–464.
}
\seealso{
\url{https://adaemmerp.github.io/lpirfs/README_docs.html}
}
\author{
Philipp Adämmer
}
