% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lp_nl_iv.R
\name{lp_nl_iv}
\alias{lp_nl_iv}
\title{Compute nonlinear impulse responses with identified shock}
\usage{
lp_nl_iv(
  endog_data,
  lags_endog_nl = NULL,
  shock = NULL,
  cumul_mult = FALSE,
  instr = NULL,
  exog_data = NULL,
  lags_exog = NULL,
  contemp_data = NULL,
  lags_criterion = NaN,
  max_lags = NaN,
  trend = NULL,
  confint = NULL,
  use_nw = TRUE,
  nw_lag = NULL,
  nw_prewhite = FALSE,
  adjust_se = FALSE,
  hor = NULL,
  switching = NULL,
  lag_switching = TRUE,
  use_logistic = TRUE,
  use_hp = NULL,
  lambda = NULL,
  gamma = NULL,
  num_cores = 1
)
}
\arguments{
\item{endog_data}{A \link{data.frame}, containing all endogenous variables for the VAR.}

\item{lags_endog_nl}{NaN or integer. NaN if lags are chosen by a lag length criterion. Integer for number of lags for \emph{endog_data}.}

\item{shock}{One column \link{data.frame}, including the instrument to shock with.
The row length has to be the same as \emph{endog_data}.}

\item{cumul_mult}{Boolean. Estimate cumulative multipliers? TRUE or FALSE (default). If TRUE, cumulative responses
are estimated via: \deqn{y_{(t+h)} - y_{(t-1)},} where h = 0,..., H-1.
This option is only available for \emph{lags_criterion = NaN}.}

\item{instr}{Deprecated input name. Use \emph{shock} instead. See \emph{shock} for details.}

\item{exog_data}{A \link{data.frame}, containing exogenous variables. The row length has to be the same as \emph{endog_data}.
Lag lengths for exogenous variables have to be given and will no be determined via a lag length criterion.}

\item{lags_exog}{NULL or Integer. Integer for the number of lags for the exogenous data.}

\item{contemp_data}{A \link{data.frame}, containing exogenous data with contemporaneous impact. This data will not be lagged.
The row length has to be the same as \emph{endog_data}.}

\item{lags_criterion}{NaN or character. NaN means that the number of lags
will be given at \emph{lags_endog_nl}. Possible lag length criteria are 'AICc', 'AIC' or 'BIC'.}

\item{max_lags}{NaN or integer. Maximum number of lags (if \emph{lags_criterion} = 'AICc', 'AIC', 'BIC'). NaN otherwise.}

\item{trend}{Integer. Include no trend =  0 , include trend = 1, include trend and quadratic trend = 2.}

\item{confint}{Double. Width of confidence bands. 68\% = 1; 90\% = 1.65; 95\% = 1.96.}

\item{use_nw}{Boolean. Use Newey-West (1987) standard errors for impulse responses? TRUE (default) or FALSE.}

\item{nw_lag}{Integer. Specifies the maximum lag with positive weight for the Newey-West estimator. If set to NULL (default), the lag increases with
with the number of horizon.}

\item{nw_prewhite}{Boolean. Should the estimators be pre-whitened? TRUE of FALSE (default).}

\item{adjust_se}{Boolen. Should a finite sample adjsutment be made to the covariance matrix estimators? TRUE or FALSE (default).}

\item{hor}{Integer. Number of horizons for impulse responses.}

\item{switching}{Numeric vector. A column vector with the same length as \emph{endog_data}. This series can either
be decomposed via the Hodrick-Prescott filter (see Auerbach and Gorodnichenko, 2013) or
directly plugged into the following smooth transition function:
\deqn{ F_{z_t} = \frac{exp(-\gamma z_t)}{1 + exp(-\gamma z_t)}. }
Warning: \eqn{F_{z_t}} will be lagged by one and then multiplied with the data.
If the variable shall not be lagged, the vector has to be given with a lead of one.
The data for the two regimes are: \cr
Regime 1 = (1-\eqn{F(z_{t-1})})*y_{(t-p)}, \cr
Regime 2 = \eqn{F(z_{t-1})}*y_{(t-p)}.}

\item{lag_switching}{Boolean. Use the first lag of the values of the transition function? TRUE (default) or FALSE.}

\item{use_logistic}{Boolean. Use logistic function to separate states? TRUE (default) or FALSE. If FALSE, the values of the switching variable
have to be binary (0/1).}

\item{use_hp}{Boolean. Use HP-filter? TRUE or FALSE.}

\item{lambda}{Double. Value of \eqn{\lambda} for the Hodrick-Prescott filter (if use_hp = TRUE).}

\item{gamma}{Double. Positive number which is used in the transition function.}

\item{num_cores}{Integer. The number of cores to use for the estimation. If NULL, the function will
use the maximum number of cores minus one.}
}
\value{
A list containing:

\item{irf_s1_mean}{A \link{matrix}, containing the impulse responses of the first regime.
                   The row in each matrix denotes the responses of the \emph{ith}
                   variable to the shock. The columns are the horizons.}

\item{irf_s1_low}{A \link{matrix}, containing all lower confidence bands of
                   the impulse responses, based on robust standard errors by Newey and West (1987).
                   Properties are equal to \emph{irf_s1_mean}.}

\item{irf_s1_up}{A \link{matrix}, containing all upper confidence bands of the
                   impulse responses, based on robust standard errors by Newey and West (1987).
                   Properties are equal to \emph{irf_s1_mean}.}

\item{irf_s2_mean}{A \link{matrix}, containing all impulse responses for the second regime.
                   The row in each matrix denotes the responses of the \emph{ith} variable to the shock.
                   The columns denote the horizon.}

\item{irf_s2_low}{A \link{matrix}, containing all lower confidence bands of the responses,
                   based on robust standard errors by Newey and West (1987). Properties are equal to \emph{irf_s2_mean}.}

\item{irf_s2_up}{A \link{matrix}, containing all upper confidence bands of the responses, based on
                   robust standard errors by Newey and West (1987). Properties are equal to \emph{irf_s2_mean}.}

\item{specs}{A list with properties of \emph{endog_data} for the plot function. It also contains
            lagged data (y_nl and x_nl) used for the estimations of the impulse responses, and the selected lag lengths when an information criterion has been used.}

\item{fz}{A vector, containing the values of the transition function F(z_{t-1}).}
}
\description{
Compute nonlinear impulse responses with local projections and identified shock.
The data can be separated into two states by a smooth transition function as applied in Auerbach and Gorodnichenko (2012),
or by a simple dummy approach.
}
\examples{
\donttest{

# This example replicates results from the Supplementary Appendix
# by Ramey and Zubairy (2018) (RZ-18).

# Load and prepare data
 ag_data           <- ag_data
 sample_start      <- 7
 sample_end        <- dim(ag_data)[1]
 endog_data        <- ag_data[sample_start:sample_end, 3:5]

# The shock is estimated by RZ-18
 shock             <- ag_data[sample_start:sample_end, 7]

# Include four lags of the 7-quarter moving average growth rate of GDP
# as exogenous variables (see RZ-18)
 exog_data         <- ag_data[sample_start:sample_end, 6]

# Use the 7-quarter moving average growth rate of GDP as switching variable
# and adjust it to have suffiently long recession periods.
 switching_variable <- ag_data$GDP_MA[sample_start:sample_end] - 0.8

# Estimate local projections
 results_nl_iv <- lp_nl_iv(endog_data,
                           lags_endog_nl     = 3,
                           shock             = shock,
                           exog_data         = exog_data,
                           lags_exog         = 4,
                           trend             = 0,
                           confint           = 1.96,
                           hor               = 20,
                           switching         = switching_variable,
                           use_hp            = FALSE,
                           gamma             = 3)

# Show all impulse responses
plot(results_nl_iv)

# Make and save individual plots
 plots_nl_iv <- plot_nl(results_nl_iv)

# Show single impulse responses
# Compare with red line of left plot (lower panel) in Figure 12 in Supplementary Appendix of RZ-18.
 plot(plots_nl_iv$gg_s1[[1]])
# Compare with blue line of left plot (lower panel) in Figure 12 in Supplementary Appendix of RZ-18.
 plot(plots_nl_iv$gg_s2[[1]])

# Show diagnostics. The first element shows the reaction of the first endogenous variable.
summary(results_nl_iv)

}
}
\references{
Akaike, H. (1974). "A new look at the statistical model identification", \emph{IEEE Transactions on Automatic Control}, 19 (6): 716–723.

Auerbach, A. J., and  Gorodnichenko Y. (2012). "Measuring the Output Responses to Fiscal Policy."
\emph{American Economic Journal: Economic Policy}, 4 (2): 1-27.

Auerbach, A. J., and Gorodnichenko Y. (2013). "Fiscal Multipliers in Recession and Expansion."
\emph{NBER Working Paper Series}. Nr 17447.

Blanchard, O., and Perotti, R. (2002). “An Empirical Characterization of the
Dynamic Effects of Changes in Government Spending and Taxes on Output.” \emph{Quarterly
Journal of Economics}, 117(4): 1329–1368.

Hurvich, C. M., and Tsai, C.-L. (1989), "Regression and time series model selection in small samples",
\emph{Biometrika}, 76(2): 297–307

Jordà, Ò. (2005) "Estimation and Inference of Impulse Responses by Local Projections."
\emph{American Economic Review}, 95 (1): 161-182.

Jordà, Ò, Schularick, M., Taylor, A.M. (2015), "Betting the house", \emph{Journal of International Economics},
96, S2-S18.

Newey, W.K., and West, K.D. (1987). “A Simple, Positive-Definite, Heteroskedasticity and
Autocorrelation Consistent Covariance Matrix.” \emph{Econometrica}, 55, 703–708.

Ramey, V.A., and Zubairy, S. (2018). "Government Spending Multipliers in Good Times
and in Bad: Evidence from US Historical Data." \emph{Journal of Political Economy},
126(2): 850 - 901.

Schwarz, Gideon E. (1978). "Estimating the dimension of a model", \emph{Annals of Statistics}, 6 (2): 461–464.
}
\seealso{
\url{https://adaemmerp.github.io/lpirfs/README_docs.html}
}
\author{
Philipp Adämmer
}
