/**
 api_build_pedigrees.cpp
 Purpose: Infers pedigrees from individuals and their relation to each other.
 Details: API between R user and C++ logic.
 
 @author Mikkel Meyer Andersen
 */

#include <RcppArmadillo.h>
//#include <Rcpp.h>

// [[Rcpp::depends(RcppProgress)]]
#include <progress.hpp>

#include "malan_types.h"

using namespace Rcpp;

// Compare sizes of pedigrees. 
// Can for example be used to sort list of pedigrees according to size.
bool pedigree_size_comparator(Pedigree* p1, Pedigree* p2) { 
  return (p1->get_all_individuals()->size() > p2->get_all_individuals()->size());
}

//' Build pedigrees from (individuals in) a population.
//' 
//' In a newly simulated population, each individual only knows its father and children. 
//' Using this information, this function builds pedigrees.
//' This makes it easier to e.g. population haplotypes, find path between two individuals 
//' (if they are not in the same pedigree, they are not connected).
//' 
//' @param population Population generated by [sample_geneology()] or [sample_geneology_varying_size()].
//' @param progress Show progress.
//' 
//' @examples
//' sim <- sample_geneology(100, 10)
//' str(sim, 1)
//' sim$population
//' peds <- build_pedigrees(sim$population)
//' peds
//' 
//' @return An object with class `malan_pedigreelist` (an internal list of external pointers to pedigrees).
//' 
//' @seealso [sample_geneology()] and [sample_geneology_varying_size()] for simulating populations.
//'
//' @export
// [[Rcpp::export]]
Rcpp::XPtr< std::vector<Pedigree*> > build_pedigrees(Rcpp::XPtr<Population> population, bool progress = true) {
  std::unordered_map<int, Individual*> pop = *(population->get_population());
  
  // Check if peds are already built ->  
  for (auto it = pop.begin(); it != pop.end(); ++it) {
    if (it->second == nullptr) {
      continue;
    }
    
    if (it->second->pedigree_is_set()) {
      Rcpp::stop("It looks like pedigrees are already built for this population, so will not do it again.");
    }
    
    break;
  }
  // <- Check if peds are already built
  
  
  //Rcpp::Rcout << "Bulding pedigrees!" << std::endl;
  
  // Construct pedigrees
  //std::cout << "Starts giving pedigrees ids..." << std::endl;

  std::vector<Pedigree*>* pedigrees = new std::vector<Pedigree*>();
  Rcpp::XPtr< std::vector<Pedigree*> > res(pedigrees, RCPP_XPTR_2ND_ARG);
  res.attr("class") = CharacterVector::create("malan_pedigreelist", "externalptr");

  int pedigree_id = 1;
  Pedigree* ped;
  
  int N = pop.size();
  int k = 0;
  Progress p(N, progress);

  ped = new Pedigree(pedigree_id);
  
  for (auto it = pop.begin(); it != pop.end(); ++it) {
    if (it->second->pedigree_is_set()) {
      continue;
    }
    
    int ped_size = 0;
    it->second->set_pedigree_id(pedigree_id, ped, &ped_size);

    pedigree_id += 1;
    
    pedigrees->push_back(ped);
    ped = new Pedigree(pedigree_id);
    
    if (k % CHECK_ABORT_EVERY == 0 && Progress::check_abort() ) {
      stop("Aborted");
    }
    
    if (progress) {
      p.increment();
    }
    
    ++k;
  }
  
  if (ped->get_all_individuals()->size() > 0) {
    pedigrees->push_back(ped);
  } else {
    delete ped;
  }
  
  std::sort(pedigrees->begin(), pedigrees->end(), pedigree_size_comparator);
  
  return res;
}


