% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/marginaleffects.R
\name{marginaleffects}
\alias{marginaleffects}
\title{Marginal Effects}
\usage{
marginaleffects(
  model,
  newdata = NULL,
  variables = NULL,
  vcov = TRUE,
  type = "response",
  ...
)
}
\arguments{
\item{model}{Model object}

\item{newdata}{A dataset over which to compute marginal effects. \code{NULL} uses
the original data used to fit the model.}

\item{variables}{Variables to consider (character vector). \code{NULL}
calculates marginal effects for all terms in the model object.}

\item{vcov}{Matrix or boolean
\itemize{
\item FALSE: does not compute unit-level standard errors. This can speed up computation considerably.
\item TRUE: computes unit-level standard errors using the default \code{vcov(model)} variance-covariance matrix.
\item Named square matrix: computes standard errors with a user-supplied variance-covariance matrix. This matrix must be square and have dimensions equal to the number of coefficients in \code{get_coef(model)}.
}}

\item{type}{Type(s) of prediction as string or character vector. This can
differ based on the model type, but will typically be a string such as:
"response", "link", "probs", or "zero".}

\item{...}{Additional arguments are passed to the \code{predict()} method used to
compute adjusted predictions. These arguments are particularly useful for
mixed-effects or bayesian models (see the online vignettes on the
\code{marginaleffects} website). Available arguments can vary from model to
model, depending on the range of supported arguments by each modeling
package. See the "Model-Specific Arguments" section of the
\code{?marginaleffects} document for a non-exhaustive list of available
arguments.}
}
\value{
A \code{data.frame} with one row per observation (per term/group) and several columns:
\itemize{
\item \code{rowid}: row number of the \code{newdata} data frame
\item \code{type}: prediction type, as defined by the \code{type} argument
\item \code{group}: (optional) value of the grouped outcome (e.g., categorical outcome models)
\item \code{term}: the variable whose marginal effect is computed
\item \code{dydx}: marginal effect of the term on the outcome for a given combination of regressor values
\item \code{std.error}: standard errors computed by via the delta method.
}
}
\description{
This function calculates marginal effects (slopes) for each row of the
dataset. The resulting object can processed by the \code{tidy()} or \code{summary()}
functions, which compute Average Marginal Effects (AME). The \code{datagrid()}
function and the \code{newdata} argument can be used to calculate Marginal
Effects at the Mean (MEM) or Marginal Effects at User-Specified values (aka
Marginal Effects at Representative values, MER). Additional information can
be found in the Details and Examples sections below, and in the vignette on
the \code{marginaleffects} website.
}
\details{
A "marginal effect" is the partial derivative of the regression equation
with respect to a variable in the model. This function uses automatic
differentiation to compute marginal effects for a vast array of models,
including non-linear models with transformations (e.g., polynomials).
Uncertainty estimates are computed using the delta method.

A detailed vignette on marginal effects and a list of supported models can
be found on the package website:

https://vincentarelbundock.github.io/marginaleffects/
}
\section{Model-Specific Arguments}{


Some model types allow model-specific arguments to modify the nature of
marginal effects, predictions, marginal means, and contrasts.\tabular{llll}{
   Package \tab Class \tab Argument \tab Documentation \cr
   \code{brms} \tab \code{brmsfit} \tab \code{ndraws} \tab \link[brms:posterior_predict.brmsfit]{brms::posterior_predict} \cr
    \tab  \tab \code{re_formula} \tab  \cr
   \code{lme4} \tab \code{merMod} \tab \code{include_random} \tab \link[insight:get_predicted]{insight::get_predicted} \cr
    \tab  \tab \code{re.form} \tab \link[lme4:predict.merMod]{lme4::predict.merMod} \cr
    \tab  \tab \code{allow.new.levels} \tab \link[lme4:predict.merMod]{lme4::predict.merMod} \cr
    \tab  \tab \code{random.only} \tab \link[lme4:predict.merMod]{lme4::predict.merMod} \cr
   \code{mgcv} \tab \code{bam} \tab \code{exclude} \tab \link[mgcv:predict.bam]{mgcv::predict.bam} \cr
}
}

\examples{
\dontshow{if (interactive()) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
\dontshow{\}) # examplesIf}

mod <- glm(am ~ hp * wt, data = mtcars, family = binomial)
mfx <- marginaleffects(mod)
head(mfx)

# Average Marginal Effect (AME)
summary(mfx)
tidy(mfx)
plot(mfx)

# Marginal Effect at the Mean (MEM)
marginaleffects(mod, newdata = datagrid())

# Marginal Effect at User-Specified Values
# Variables not explicitly included in `datagrid()` are held at their means
marginaleffects(mod,
                newdata = datagrid(hp = c(100, 110)))

# Marginal Effects at User-Specified Values (counterfactual)
# Variables not explicitly included in `datagrid()` are held at their
# original values, and the whole dataset is duplicated once for each
# combination of the values in `datagrid()`
mfx <- marginaleffects(mod,
                       newdata = datagrid(hp = c(100, 110),
                                          grid.type = "counterfactual"))
head(mfx)

# Heteroskedasticity robust standard errors
marginaleffects(mod, vcov = sandwich::vcovHC(mod))

}
