% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{hydrology_soilWaterBalance}
\alias{hydrology_soilWaterBalance}
\title{Soil water balance}
\usage{
hydrology_soilWaterBalance(
  soil,
  soilFunctions,
  rainfallInput,
  rainfallIntensity,
  snowmelt,
  sourceSink,
  runon = 0,
  lateralFlows = NULL,
  waterTableDepth = NA_real_,
  infiltrationMode = "GreenAmpt1911",
  infiltrationCorrection = 5,
  soilDomains = "single",
  nsteps = 24L,
  max_nsubsteps = 3600L,
  modifySoil = TRUE
)
}
\arguments{
\item{soil}{Object of class \code{\link{soil}}.}

\item{soilFunctions}{Soil water retention curve and conductivity functions, either 'SX' (for Saxton) or 'VG' (for Van Genuchten).}

\item{rainfallInput}{Amount of water from rainfall event (after excluding interception), in mm.}

\item{rainfallIntensity}{Rainfall intensity, in mm/h.}

\item{snowmelt}{Amount of water originated from snow melt, in mm.}

\item{sourceSink}{Local source/sink term for each soil layer (from soil evaporation or plant transpiration/redistribution)
as mm/day.}

\item{runon}{Surface water amount running on the target area from upslope (in mm).}

\item{lateralFlows}{Lateral source/sink terms for each soil layer (interflow/to from adjacent locations) as mm/day.}

\item{waterTableDepth}{Water table depth (in mm). When not missing, capillarity rise will be allowed if lower than total soil depth.}

\item{infiltrationMode}{Infiltration model, either "GreenAmpt1911" or "Boughton1989"}

\item{infiltrationCorrection}{Correction for saturated conductivity, to account for increased infiltration due to macropore presence}

\item{soilDomains}{Either "single" (for single-domain) or "dual" (for dual-permeability).}

\item{nsteps}{Number of time steps per day}

\item{max_nsubsteps}{Maximum number of substeps per time step}

\item{modifySoil}{Boolean flag to indicate that the input \code{soil} object should be modified during the simulation.}
}
\value{
Returns a named vector with different elements, depending on \code{soilDomains}. If
  \code{soilDomains == "single"}:
  \itemize{
    \item{\code{Snowmelt}: Snowmelt input (mm).}
    \item{\code{Source/sinks}: Sum of source/sink input across layers (mm).}
    \item{\code{Infiltration}: Water infiltrated into the soil (mm).}
    \item{\code{InfiltrationExcess}: Excess infiltration in the topmost layer (mm) leading to an increase in runoff.}
    \item{\code{SaturationExcess}: Excess saturation in the topmost layer (mm) leading to an increase in runoff.}
    \item{\code{Runoff}: Surface runoff generated by saturation excess or infiltration excess (mm).}
    \item{\code{DeepDrainage}: Water draining from the bottom layer (mm). This quantity is corrected to close the water balance.}
    \item{\code{CapillarityRise}: Water entering the soil via capillarity rise (mm) from the water table, if \code{waterTableDepth} is supplied.}
    \item{\code{Correction}: Amount of water (mm) added to deep drainage to correct the water balance.}
    \item{\code{VolumeChange}: Change in soil water volume (mm).}
    \item{\code{Substep}: Time step of the moisture solving (seconds).}
  }
 If \code{soilDomains == "dual"} the named vector contains the following additional elements:
  \itemize{
    \item{\code{Lateral flows}: Sum of water circulating between micropores and macropores, positive when filling micropores (mm).}
    \item{\code{InfiltrationMatrix}: Water infiltrated into the soil matrix (mm).}
    \item{\code{InfiltrationMacropores}: Water infiltrated into the soil macropore domain (mm).}
    \item{\code{InfiltrationExcessMatrix/InfiltrationExcessMacropores}: Excess infiltration in the topmost layer (mm) leading to an increase in runoff.}
    \item{\code{SaturationExcessMatrix/SaturationExcessMacropores}: Excess saturation in the topmost layer (mm) leading to an increase in runoff.}
    \item{\code{DrainageMatrix}: Water draining from the bottom layer of the matrix domain (mm). This quantity is corrected to close water balance in the micropore domain.}
    \item{\code{DrainageMacropores}: Water draining from the bottom layer of the macropore domain (mm). This quantity is corrected to close the water balance in the macropore domain.}
    \item{\code{CorrectionMatrix}: Amount of water (mm) added to deep drainage of soil matrix to correct the water balance.}
    \item{\code{CorrectionMacropores}: Amount of water (mm) added to deep drainage of macropores to correct the water balance.}
    \item{\code{MatrixVolumeChange}: Change in soil water volume in the soil matrix domain (mm).}
    \item{\code{MacroporeVolumeChange}: Change in soil water volume in the macropore domain (mm).}
  }
}
\description{
Function \code{hydrology_soilWaterBalance} estimates water balance of soil layers given water inputs/outputs, including the simulation of water movement within the soil.
}
\details{
The single-domain model simulates water flows by solving Richards's equation using the predictor-corrector method, as described in 
Bonan et al. (2019).

The dual-permeability model is an implementation of the model MACRO 5.0 (Jarvis et al. 1991; Larsbo et al. 2005).
}
\examples{
# Define soil parameters
spar <- defaultSoilParams(4)

# Initializes soil hydraulic parameters
examplesoil <- soil(spar)

# Water balance in a single-domain simulation (Richards equation)
hydrology_soilWaterBalance(examplesoil, "VG", 10, 5, 0, c(-1,-1,-1,-1), 
                           soilDomains = "single", modifySoil = FALSE)
                    
# Water balance in a dual-permeability model (MACRO)
hydrology_soilWaterBalance(examplesoil, "VG", 10, 5, 0, c(-1,-1,-1,-1), 
                           soilDomains = "dual", modifySoil = FALSE)
  
}
\references{
Bonan, G. (2019). Climate change and terrestrial ecosystem modeling. Cambridge University Press, Cambridge, UK. 

Jarvis, N.J., Jansson, P‐E., Dik, P.E. & Messing, I. (1991). Modelling water and solute transport in macroporous soil. I. Model description and sensitivity analysis. Journal of Soil Science, 42, 59–70. 

Larsbo, M., Roulier, S., Stenemo, F., Kasteel, R. & Jarvis, N. (2005). An Improved Dual‐Permeability Model of Water Flow and Solute Transport in the Vadose Zone. Vadose Zone Journal, 4, 398–406.
}
\seealso{
\code{\link{spwb}}, \code{\link{hydrology_waterInputs}}, \code{\link{hydrology_infiltration}}
}
\author{
Miquel De \enc{Cáceres}{Caceres} Ainsa, CREAF

\enc{María González Sanchís}{Maria Gonzalez Sanchis}, UPV-CTFC
}
