% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/BP_estim.R
\name{BP_estim}
\alias{BP_estim}
\title{Breakpoint estimators for a change in persistence}
\usage{
BP_estim(x, trend = c("none", "linear"), tau = 0.2, type = c("BT",
  "LKT", "LKSN", "MR"), direction = c("01", "10"), d_estim = c("ELW",
  "GPH"), d_bw = 0.7, m = 0, serial = c(FALSE, TRUE))
}
\arguments{
\item{x}{the univariate numeric vector to be investigated. Missing values are not allowed.}

\item{trend}{whether the time series exhibits a trend, \code{"none"} implies no trend and \code{"linear"} implies a linear trend.}

\item{tau}{the function searches in the interval \code{[T*tau,T*(1-tau)]} for a break in persistence with T being the length of the time series. It must hold that \code{0<tau<0.5}, default is \code{tau=0.2} as commonly used in the literature. Note that if \code{type="BT"} and \code{T*tau<=2+ as.numeric(trend=="linear")}, \code{type="LT"} and \code{T*tau<=2+ as.numeric(trend=="linear") + (m>3)*(m-3)}, \code{type="LKSN"} and \code{T*tau<=10}, or \code{type="MR"} and \code{T*tau<=2+(p>1)*p}the break point cannot be found.}

\item{type}{which type of break point estimator should be used, \code{"LKSN"} for the estimator by Leybourne, Kim, Smith, and Newbold (2003), \code{"BT"} for the estimator by Busetti and Taylor (2004), \code{"LKT"} for the estimator by Leybourne, Kim, and Taylor (2006),
and \code{MR} for the estimator by Martins and Rodrigues (2014). See details.}

\item{direction}{direction of the change in persistence, \code{"01"} implies an increase in persistence over time and \code{"10"} a decrease. See details.}

\item{d_estim}{which estimator should be used to determine the order of integration in the two regimes, \code{"GPH"} corresponds to the estimator by Geweke and Porter-Hudak (1983) and \code{"ELW"} corresponds to the exact local Whittle estimator by Shimotsu and Phillips (2005).}

\item{d_bw}{bandwidth used for estimating the order of integration d. Default is \code{d_bw=0.7}. Note that the estimation of the memory parameter can only be performed for 0<d_bw<=1 and it is even recommended to choose 0.5<=d_bw<=0.8 as otherwise the estimators might be inconsistent.")}

\item{m}{Number of covariances used for the estimation of the long run variance when considering the LKT estimator. Default is \code{m=0}.}

\item{serial}{boolean, indicating whether to account for serial correlation of the errors when considering the MR estimator. Default is \code{serial=FALSE} implying no correction for serial correlation.}
}
\value{
Returns a list that contains break point, estimate of the order of integration in the two regimes (the memory parameter d) and standard deviation of this estimate.
}
\description{
This function estimates the location where the investigated time series exhibits a break in persistence. It requires
knowledge of the direction of the break, i.e. an increase or decrease in persistence. This
needs to be determined beforehand using one of the various persistence change tests provided in this package.
}
\details{
The estimators BT and LKSN are only consistent for changes from I(0) to I(1) or vice versa, the LKT estimator is consistent for changes from stationary to nonstationary memory or vice versa (cf. also Sibbertsen and Kruse (2009)), and the MR estimator is consistent for changes in d in general.
}
\examples{
set.seed(410)

# generate dummy-data
series <- c(rnorm(200), cumsum(rnorm(200)))

# estimate the break point
BP_estim(series, trend="none", type="BT", direction="01", d_estim="ELW")
}
\references{
Leybourne, S., Kim, T., Smith, V., and Newbold, P. (2003): Tests for a change in persistence against the null of difference-stationarity. Econometrics Journal, 6, pp. 291-311.

Busetti, F. and Taylor, R. (2004): Tests of stationarity against a change in persistence. Journal of Econometrics, 123, pp. 33-66.

Leybourne, S., Kim, T., and Taylor, R. (2007): Cusum of squares-based tests for a change in persistence. Journal of Time Series Analysis, 28, pp. 408-433.

Martins, L.. and Rodrigues, P. (2014): Testing for persistence change in fractionally integrated models: An application to world inflation rates Cusum of squares-based tests for a change in persistence. Computational Statistics and Data Analysis, 76, pp. 502-522.
}
\seealso{
\code{\link{cusum_test}}, \code{\link{LKSN_test}}, \code{\link{MR_test}}, \code{\link{ratio_test}}.
}
\author{
Janis Becker
}
