% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/penultimate.R
\name{smith.penult.fn}
\alias{smith.penult.fn}
\title{Smith's third penultimate approximation}
\usage{
smith.penult.fn(
  loc,
  scale,
  shape,
  eps,
  rho = NULL,
  method = c("bm", "pot"),
  mdaGumbel = FALSE,
  ...
)
}
\arguments{
\item{loc}{location parameter returned by \code{\link{smith.penult}} or threshold vector}

\item{scale}{scale parameter returned by \code{\link{smith.penult}}}

\item{shape}{shape parameter returned by \code{\link{smith.penult}}}

\item{eps}{parameter vector, see \strong{Details}.}

\item{rho}{second-order parameter, model dependent}

\item{method}{one of \code{pot} for the generalized Pareto or \code{bm} for the generalized extreme value distribution}

\item{mdaGumbel}{logical indicating whether the function \eqn{H_{\rho}}{H(x;\rho)} should be replaced by \eqn{x^3/6}; see \strong{Details}.}

\item{...}{additional parameters, currently ignored. These are used for backward compatibility due to a change in the names of the arguments.}
}
\description{
This function returns the density and distribution functions
of the 3rd penultimate approximation for extremes of Smith (1987). It requires
knowledge of the exact constants \eqn{\epsilon} and \eqn{\rho} described in the paper.
}
\details{
Let \eqn{F}, \eqn{f} denote respectively the distribution and density functions and define the function \eqn{\phi(x)}  as
\deqn{\phi(x)=-\frac{F(x)\log F(x)}{f(x)}}{\phi(x)=-F(x)log F(x)/f(x)}
for block maxima.
The sequence \code{loc} corresponds to \eqn{b_n} otherwise, defined as the solution of \eqn{F(b_n)=\exp(-1/n)}{F(b_n)=exp(-1/n)}.

The \code{scale} is given by \eqn{a_n=\phi(b_n)}, the \code{shape} as \eqn{\gamma_n=\phi'(b_n)}. These are returned by a call to \link{smith.penult}.

For threshold exceedances, \eqn{b_n} is replaced by the sequence of thresholds \eqn{u} and we
take instead \eqn{\phi(x)} to be the reciprocal hazard function \eqn{\phi(x)=(1-F(x))/f(x)}{\phi(x)=(1-F(x))/f(x)}.

In cases where the distribution function is in the maximum domain of
attraction of the Gumbel distribution, \eqn{\rho} is possibly undetermined and
\eqn{\epsilon} can be equal to \eqn{\phi(b_n)\phi''(b_n)}.

For distributions in the maximum domain of
attraction of the Gumbel distribution and that are class N, it is also possible to abstract from the \eqn{\rho} parameter by substituting the function \eqn{H_{\rho}}{H(x;\rho)} by \eqn{x^3/6} without affecting the rate of convergence. This can be done by setting \code{mdaGumbel=TRUE} in the function call.
}
\section{Warning}{

The third penultimate approximation does not yield a valid distribution function over the whole range of the original distribution, but is rather valid in a neighborhood of the true support of the distribution of maxima/threshold exceedance.
The function handles the most standard failure (decreasing distribution function and negative densities), but any oscillatory behaviour will not necessarily be captured.
This is inherent to the method and can be resolved by `not' evaluating the functions \eqn{F} and \eqn{f} at the faulty points.
}

\examples{
#Normal maxima example from Smith (1987)
m <- 100 #block of size 100
p <- smith.penult(family='norm',
   ddensF=function(x){-x*dnorm(x)}, method='bm', m=m, returnList=FALSE)
approx <- smith.penult.fn(loc=p[1], scale=p[2], shape=p[3],
   eps=p[3]^2+p[3]+p[2]^2, mdaGumbel=TRUE, method='bm')
x <- seq(0.5,6,by=0.001)
#First penultimate approximation
plot(x, exp(m*pnorm(x, log.p=TRUE)),type='l', ylab='CDF',
main='Distribution of the maxima of\n 100 standard normal variates')
lines(x, evd::pgev(x,loc=p[1], scale=p[2], shape=0),col=2)
lines(x, evd::pgev(x,loc=p[1], scale=p[2], shape=p[3]),col=3)
lines(x, approx$F(x),col=4)
legend(x='bottomright',lty=c(1,1,1,1),col=c(1,2,3,4),
   legend=c('Exact','1st approx.','2nd approx.','3rd approx'),bty='n')
plot(x, m*dnorm(x)*exp((m-1)*pnorm(x,log.p=TRUE)),type='l', ylab='Density',
main='Distribution of the maxima of\n 100 standard normal variates')
lines(x, evd::dgev(x,loc=p[1], scale=p[2], shape=0),col=2)
lines(x, evd::dgev(x,loc=p[1], scale=p[2], shape=p[3]),col=3)
lines(x, approx$f(x),col=4)
legend(x='topright',lty=c(1,1,1,1),col=c(1,2,3,4),
 legend=c('Exact','1st approx.','2nd approx.','3rd approx'),bty='n')

#Threshold exceedances
par <- smith.penult(family = "norm", ddensF=function(x){-x*dnorm(x)},
method='pot', u=4, returnList=FALSE)
approx <- smith.penult.fn(loc=par[1], scale=par[2], shape=par[3],
 eps=par[3]^2+par[3]+par[2]^2, mdaGumbel=TRUE, method='pot')
x <- seq(4.01,7,by=0.01)
#Distribution function
plot(x, 1-(1-pnorm(x))/(1-pnorm(par[1])),type='l', ylab='Conditional CDF',
main='Exceedances over 4\n for standard normal variates')
lines(x, evd::pgpd(x, loc=par[1], scale=par[2], shape=0),col=2)
lines(x, evd::pgpd(x, loc=par[1], scale=par[2], shape=par[3]),col=3)
lines(x, approx$F(x),col=4)
#Density
plot(x, dnorm(x)/(1-pnorm(par[1])),type='l', ylab='Conditional density',
main='Exceedances over 4\n for standard normal variates')
lines(x, evd::dgpd(x, loc=par[1], scale=par[2], shape=0),col=2)
lines(x, evd::dgpd(x, loc=par[1], scale=par[2], shape=par[3]),col=3)
lines(x, approx$f(x),col=4)
}
\references{
Smith, R.L. (1987). Approximations in extreme value theory. \emph{Technical report 205}, Center for Stochastic Process, University of North Carolina, 1--34.
}
