% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/blasting.R
\name{blastAllAll}
\alias{blastAllAll}
\title{Making BLAST search all against all genomes}
\usage{
blastAllAll(prot.files, out.folder, e.value = 1, job = 1, threads = 1,
  verbose = T)
}
\arguments{
\item{prot.files}{A text vector with the names of the FASTA files where the protein sequences of
each genome is found.}

\item{out.folder}{The name of the folder where the result files should end up.}

\item{e.value}{The chosen E-value threshold in BLAST. Default is \samp{e.value=1}, a smaller value
will speed up the search at the cost of less sensitivity.}

\item{job}{An integer to separate multiple jobs. You may want to run several jobs in parallell,
and each job should have different number here to avoid confusion on databases. Default is
\samp{job=1}.}

\item{threads}{The number of CPU's to use.}

\item{verbose}{Logical, if \code{TRUE} some text output is produced to monitor the progress.}
}
\value{
The function produces \emph{N*N} result files if \samp{prot.files} lists \emph{N} sequence files.
These result files are located in \code{out.folder}. Existing files are never overwritten by
\code{\link{blastAllAll}}, if you want to re-compute something, delete the corresponding result files first.
}
\description{
Runs a reciprocal all-against-all BLAST search to look for similarity of proteins
within and across genomes. The main job is done by the BLAST+ software.
}
\details{
A basic step in pangenomics and many other comparative studies is to cluster proteins into
groups or families. One commonly used approach is based on reciprocal BLASTing. This function uses the
BLAST+ software available for free from NCBI (Camacho et al, 2009). 

A vector listing FASTA files of protein sequences is given as input in \samp{prot.files}. These files
must have the GID-tag in the first token of every header, and in their filenames as well, i.e. all input
files should first be prepared by \code{\link{panPrep}} to ensure this. Note that only protein sequences
are considered here. If your coding genes are stored as DNA, please translate them to protein prior to
using this function, see \code{\link[microseq]{translate}}.

A BLAST database is made from each genome in turn. Then all genomes are queried against this database,
and for every pair of genomes a result file is produced. If two genomes have GID-tags \samp{GID111},
and \samp{GID222} then both result file \samp{GID111_vs_GID222.txt} and \samp{GID222_vs_GID111.txt} will
be found in \samp{out.folder} after the completion of this search. This reciprocal (two-way) search is
required because of the heuristics of BLAST.

The \samp{out.folder} is scanned for already existing result files, and \code{\link{blastAllAll}} never
overwrites an existing result file. If a file with the name \samp{GID111_vs_GID222.txt} already exists in
the \samp{out.folder}, this particular search is skipped. This makes it possible to run multiple jobs in
parallell, writing to the same \samp{out.folder}. It also makes it possible to add new genomes, and only
BLAST the new combinations without repeating previous comparisons. 

This search can be slow if the genomes contain many proteins and it scales quadratically in the number of
input files. It is best suited for the study of a smaller number of genomes (less than say 100). By
starting multiple R sessions, you can speed up the search by running \code{\link{blastAllAll}} from each R
session, using the same \samp{out.folder} but different integers for the \code{job} option. If you are
using a computing cluster you can also increase the number of CPUs by increasing \code{threads}.

The result files are text files, and can be read into R using \code{\link{readBlastTable}}, but more
commonly they are used as input to \code{\link{bDist}} to compute distances between sequences for subsequent
clustering.
}
\note{
The BLAST+ software must be installed on the system for this function to work, i.e. the command
\samp{system("makeblastdb -help")} must be recognized as valid commands if you
run them in the Console window.
}
\examples{
\dontrun{
# This example requires the external BLAST+ software
# Using protein files in this package
xpth <- file.path(path.package("micropan"),"extdata")
prot.files <- file.path(xpth,c("Example_proteins_GID1.fasta.xz",
                               "Example_proteins_GID2.fasta.xz",
                               "Example_proteins_GID3.fasta.xz"))

# We need to uncompress them first...
tf <- tempfile(fileext=c("GID1.fasta.xz","GID2.fasta.xz","GID3.fasta.xz"))
s <- file.copy(prot.files,tf)
tf <- unlist(lapply(tf,xzuncompress))

# Blasting all versus all...(requires BLAST+)
tmp.dir <- tempdir()
blastAllAll(tf,out.folder=tmp.dir)

# Reading results, and computing blast.distances
blast.files <- dir(tmp.dir,pattern="GID[0-9]+_vs_GID[0-9]+.txt")
blast.distances <- bDist(file.path(tmp.dir,blast.files))

# ...and cleaning tmp.dir...
s <- file.remove(tf)
s <- file.remove(file.path(tmp.dir,blast.files))
}

}
\references{
Camacho, C., Coulouris, G., Avagyan, V., Ma, N., Papadopoulos, J., Bealer, K., Madden, T.L.
(2009). BLAST+: architecture and applications. BMC Bioinformatics, 10:421.
}
\seealso{
\code{\link{panPrep}}, \code{\link{readBlastTable}}, \code{\link{bDist}}.
}
\author{
Lars Snipen and Kristian Hovde Liland.
}
