% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/OptimizerHyperband.R
\name{mlr_optimizers_hyperband}
\alias{mlr_optimizers_hyperband}
\alias{OptimizerHyperband}
\title{Optimizer Using the Hyperband Algorithm}
\source{
Li L, Jamieson K, DeSalvo G, Rostamizadeh A, Talwalkar A (2018).
\dQuote{Hyperband: A Novel Bandit-Based Approach to Hyperparameter Optimization.}
\emph{Journal of Machine Learning Research}, \bold{18}(185), 1-52.
\url{https://jmlr.org/papers/v18/16-558.html}.
}
\description{
\code{OptimizerHyperband} class that implements hyperband optimization (HB).
HB repeatedly calls SHA (\link{OptimizerSuccessiveHalving}) with different
numbers of starting points. A larger number of starting points corresponds to
a smaller budget allocated in the base stage. Each run of SHA within HB is
called a bracket. HB considers \code{s_max + 1} brackets with \verb{s_max = floor(log(r_max / r_min, eta)}. The most explorative bracket \code{s = s_max}
constructs \code{s_max + 1} stages and allocates the minimum budget (\code{r_min}) in
the base stage. The minimum budget is increased in each bracket by a factor
of \code{eta} and the number of starting points is computed so that each bracket
approximately spends the same budget. Use \code{\link[=hyperband_schedule]{hyperband_schedule()}} to get a
preview of the bracket layout.\tabular{rlrrlrrlrrlrr}{
   \code{s} \tab  \tab  \tab 3 \tab  \tab  \tab 2 \tab  \tab  \tab 1 \tab  \tab  \tab 0 \cr
   \code{i} \tab  \tab \code{n_i} \tab \code{r_i} \tab  \tab \code{n_i} \tab \code{r_i} \tab  \tab \code{n_i} \tab \code{r_i} \tab  \tab \code{n_i} \tab \code{r_i} \cr
   0 \tab  \tab 8 \tab 1 \tab  \tab 6 \tab 2 \tab  \tab 4 \tab 4 \tab  \tab 8 \tab 4 \cr
   1 \tab  \tab 4 \tab 2 \tab  \tab 3 \tab 4 \tab  \tab 2 \tab 8 \tab  \tab  \tab  \cr
   2 \tab  \tab 2 \tab 4 \tab  \tab 1 \tab 8 \tab  \tab  \tab  \tab  \tab  \tab  \cr
   3 \tab  \tab 1 \tab 8 \tab  \tab  \tab  \tab  \tab  \tab  \tab  \tab  \tab  \cr
}


\code{s} is the bracket number, \code{i} is stage number, \code{n_i} is the number of
configurations and \code{r_i} is the budget allocated to a single configuration.

The budget hyperparameter must be tagged with \code{"budget"} in the search space.
The minimum budget (\code{r_min}) which is allocated in the base stage of the most
explorative bracket, is set by the lower bound of the budget parameter. The
upper bound defines the maximum budget (\code{r_max}) which which is allocated to
the candidates in the last stages.
}
\section{Dictionary}{

This \link{Optimizer} can be instantiated via the \link[mlr3misc:Dictionary]{dictionary}
\link{mlr_optimizers} or with the associated sugar function \code{\link[=opt]{opt()}}:

\if{html}{\out{<div class="sourceCode">}}\preformatted{mlr_optimizers$get("hyperband")
opt("hyperband")
}\if{html}{\out{</div>}}
}

\section{Parameters}{

\describe{
\item{\code{eta}}{\code{numeric(1)}\cr
With every stage, the budget is increased by a factor of \code{eta}
and only the best \code{1 / eta} points are promoted to the next stage.
Non-integer values are supported, but \code{eta} is not allowed to be less or
equal 1.
}
\item{\code{sampler}}{\link[paradox:Sampler]{paradox::Sampler}\cr
Object defining how the samples of the parameter space should be drawn in the
base stage of each bracket. The default is uniform sampling.
}
\item{\code{repetitions}}{\code{integer(1)}\cr
If \code{1} (default), optimization is stopped once all brackets are evaluated.
Otherwise, optimization is stopped after \code{repetitions} runs of hyperband. The
\link[bbotk:Terminator]{bbotk::Terminator} might stop the optimization before all repetitions are
executed.
}}
}

\section{Archive}{

The \link[bbotk:Archive]{bbotk::Archive} holds the following additional columns that are specific
to the hyperband algorithm:
\itemize{
\item \code{bracket} (\code{integer(1)})\cr
The bracket index. Counts down to 0.
\item \code{stage} (\verb{integer(1))}\cr
The stages of each bracket. Starts counting at 0.
\item \code{repetition} (\verb{integer(1))}\cr
Repetition index. Start counting at 1.
}
}

\section{Custom Sampler}{

Hyperband supports custom \link[paradox:Sampler]{paradox::Sampler} object for initial
configurations in each bracket.
A custom sampler may look like this (the full example is given in the
\emph{examples} section):

\if{html}{\out{<div class="sourceCode">}}\preformatted{# - beta distribution with alpha = 2 and beta = 5
# - categorical distribution with custom probabilities
sampler = SamplerJointIndep$new(list(
  Sampler1DRfun$new(params[[2]], function(n) rbeta(n, 2, 5)),
  Sampler1DCateg$new(params[[3]], prob = c(0.2, 0.3, 0.5))
))
}\if{html}{\out{</div>}}
}

\section{Progress Bars}{

\verb{$optimize()} supports progress bars via the package \CRANpkg{progressr}
combined with a \link{Terminator}. Simply wrap the function in
\code{progressr::with_progress()} to enable them. We recommend to use package
\CRANpkg{progress} as backend; enable with \code{progressr::handlers("progress")}.
}

\section{Logging}{

Hyperband uses a logger (as implemented in \CRANpkg{lgr}) from package
\CRANpkg{bbotk}.
Use \code{lgr::get_logger("bbotk")} to access and control the logger.
}

\examples{
library(bbotk)
library(data.table)

# set search space
search_space = domain = ps(
  x1 = p_dbl(-5, 10),
  x2 = p_dbl(0, 15),
  fidelity = p_dbl(1e-2, 1, tags = "budget")
)

# objective with modified branin function, see `bbotk::branin()`
objective = ObjectiveRFun$new(
  fun = branin,
  domain = domain,
  codomain = ps(y = p_dbl(tags = "minimize"))
)

# initialize instance and optimizer
instance = OptimInstanceSingleCrit$new(
  objective = objective,
  search_space = search_space,
  terminator = trm("evals", n_evals = 50)
)

optimizer = opt("hyperband")

# optimize branin function
optimizer$optimize(instance)

# best scoring evaluation
instance$result

# all evaluations
as.data.table(instance$archive)
}
\section{Super class}{
\code{\link[bbotk:Optimizer]{bbotk::Optimizer}} -> \code{OptimizerHyperband}
}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-OptimizerHyperband-new}{\code{OptimizerHyperband$new()}}
\item \href{#method-OptimizerHyperband-clone}{\code{OptimizerHyperband$clone()}}
}
}
\if{html}{\out{
<details open><summary>Inherited methods</summary>
<ul>
<li><span class="pkg-link" data-pkg="bbotk" data-topic="Optimizer" data-id="format"><a href='../../bbotk/html/Optimizer.html#method-Optimizer-format'><code>bbotk::Optimizer$format()</code></a></span></li>
<li><span class="pkg-link" data-pkg="bbotk" data-topic="Optimizer" data-id="help"><a href='../../bbotk/html/Optimizer.html#method-Optimizer-help'><code>bbotk::Optimizer$help()</code></a></span></li>
<li><span class="pkg-link" data-pkg="bbotk" data-topic="Optimizer" data-id="optimize"><a href='../../bbotk/html/Optimizer.html#method-Optimizer-optimize'><code>bbotk::Optimizer$optimize()</code></a></span></li>
<li><span class="pkg-link" data-pkg="bbotk" data-topic="Optimizer" data-id="print"><a href='../../bbotk/html/Optimizer.html#method-Optimizer-print'><code>bbotk::Optimizer$print()</code></a></span></li>
</ul>
</details>
}}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-OptimizerHyperband-new"></a>}}
\if{latex}{\out{\hypertarget{method-OptimizerHyperband-new}{}}}
\subsection{Method \code{new()}}{
Creates a new instance of this \link[R6:R6Class]{R6} class.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{OptimizerHyperband$new()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-OptimizerHyperband-clone"></a>}}
\if{latex}{\out{\hypertarget{method-OptimizerHyperband-clone}{}}}
\subsection{Method \code{clone()}}{
The objects of this class are cloneable with this method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{OptimizerHyperband$clone(deep = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{deep}}{Whether to make a deep clone.}
}
\if{html}{\out{</div>}}
}
}
}
