% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simData.R
\name{simData}
\alias{simData}
\title{Simulation tool}
\usage{
simData(nbAnimals = 1, nbStates = 2, dist, Par, beta = NULL,
  formula = ~1, covs = NULL, nbCovs = 0, spatialCovs = NULL,
  zeroInflation = NULL, oneInflation = NULL, circularAngleMean = NULL,
  centers = NULL, obsPerAnimal = c(500, 1500), initialPosition = c(0, 0),
  DM = NULL, cons = NULL, userBounds = NULL, workcons = NULL,
  stateNames = NULL, model = NULL, states = FALSE, retrySims = 0,
  lambda = NULL, errorEllipse = NULL)
}
\arguments{
\item{nbAnimals}{Number of observed individuals to simulate.}

\item{nbStates}{Number of behavioural states to simulate.}

\item{dist}{A named list indicating the probability distributions of the data streams. Currently
supported distributions are 'gamma','weibull','exp','lnorm','beta','pois','wrpcauchy', and 'vm'. For example,
\code{dist=list(step='gamma', angle='vm', dives='pois')} indicates 3 data streams ('step', 'angle', and 'dives')
and their respective probability distributions ('gamma', 'vm', and 'pois').}

\item{Par}{A named list containing vectors of initial state-dependent probability distribution parameters for 
each data stream specified in \code{dist}. The parameters should be in the order expected by the pdfs of \code{dist}, 
and any zero-mass and/or one-mass parameters should be the last (if both are present, then zero-mass parameters must preceed one-mass parameters). 

If \code{DM} is not specified for a given data stream, then \code{Par} 
is on the natural (i.e., real) scale of the parameters. However, if \code{DM} is specified for a given data stream, then 
\code{Par} must be on the working (i.e., beta) scale of the parameters, and the length of \code{Par} must match the number 
of columns in the design matrix. See details below.}

\item{beta}{Matrix of regression parameters for the transition probabilities (more information
in "Details").}

\item{formula}{Regression formula for the transition probability covariates. Default: \code{~1} (no covariate effect). In addition to allowing standard functions in R formulas
(e.g., \code{cos(cov)}, \code{cov1*cov2}, \code{I(cov^2)}), special functions include \code{cosinor(cov,period)} for modeling cyclical patterns, spline functions 
(\code{\link[splines]{bs}}, \code{\link[splines]{ns}}, \code{\link[splines2]{bSpline}}, \code{\link[splines2]{cSpline}}, \code{\link[splines2]{iSpline}}, and \code{\link[splines2]{mSpline}}), 
and state- or parameter-specific formulas (see details).
Any formula terms that are not state- or parameter-specific are included on all of the transition probabilities.}

\item{covs}{Covariate values to include in the simulated data, as a dataframe. The names of any covariates specified by \code{covs} can
be included in \code{formula} and/or \code{DM}. Covariates can also be simulated according to a standard normal distribution, by setting
\code{covs} to \code{NULL} (the default), and specifying \code{nbCovs>0}.}

\item{nbCovs}{Number of covariates to simulate (0 by default). Does not need to be specified if
\code{covs} is specified. Simulated covariates are provided generic names (e.g., 'cov1' and 'cov2' for \code{nbCovs=2}) and can be included in \code{formula} and/or \code{DM}.}

\item{spatialCovs}{List of \code{\link[raster]{RasterLayer-class}} objects for spatially-referenced covariates. Covariates specified by \code{spatialCovs} are
extracted from the raster layer(s) based on the simulated location data for each time step (if applicable). The names of the raster layer(s) can be included in 
\code{formula} and/or \code{DM}.  Note that \code{simData} usually takes longer to generate simulated data when \code{spatialCovs} is specified.}

\item{zeroInflation}{A named list of logicals indicating whether the probability distributions of the data streams should be zero-inflated. If \code{zeroInflation} is \code{TRUE} 
for a given data stream, then values for the zero-mass parameters should be
included in the corresponding element of \code{Par}.}

\item{oneInflation}{A named list of logicals indicating whether the probability distributions of the data streams should be one-inflated. If \code{oneInflation} is \code{TRUE} 
for a given data stream, then values for the one-mass parameters should be
included in the corresponding element of \code{Par}.}

\item{circularAngleMean}{An optional named list indicating whether to use circular-linear (FALSE) or circular-circular (TRUE) 
regression on the mean of circular distributions ('vm' and 'wrpcauchy') for turning angles.  For example, 
\code{circularAngleMean=list(angle=TRUE)} indicates the angle mean is be estimated for 'angle' using circular-circular 
regression.  Whenever circular-circular regression is used for an angular data stream, a corresponding design matrix (\code{DM}) 
must be specified for the data stream, and the previous movement direction (i.e., a turning angle of zero) is automatically used 
as the reference angle (i.e., the intercept). Default is \code{NULL}, which assumes circular-linear regression is 
used for any angular distributions. Any \code{circularAngleMean} elements 
corresponding to data streams that do not have angular distributions are ignored.}

\item{centers}{2-column matrix providing the x-coordinates (column 1) and y-coordinates (column 2) for any activity centers (e.g., potential 
centers of attraction or repulsion) from which distance and angle covariates will be calculated based on the simulated location data. These distance and angle 
covariates can be included in \code{formula} and \code{DM} using the row names of \code{centers}.  If no row names are provided, then generic names are generated 
for the distance and angle covariates (e.g., 'center1.dist', 'center1.angle', 'center2.dist', 'center2.angle'); otherwise the covariate names are derived from the row names
of \code{centers} as \code{paste0(rep(rownames(centers),each=2),c(".dist",".angle"))}. Note that the angle covariates for each activity center are calculated relative to 
the previous movement direction instead of standard directions relative to the x-axis; this is to allow turning angles to be simulated as a function of these covariates using circular-circular regression.}

\item{obsPerAnimal}{Either the number of the number of observations per animal (if single value) or the bounds of the number of observations per animal (if vector of two values). In the latter case, 
the numbers of obervations generated for each animal are uniformously picked from this interval. Alternatively, \code{obsPerAnimal} can be specified as
a list of length \code{nbAnimals} with each element providing the number of observations (if single value) or the bounds (if vector of two values) for each individual.
Default: \code{c(500,1500)}.}

\item{initialPosition}{2-vector providing the x- and y-coordinates of the initial position for all animals. Default: \code{c(0,0)}.}

\item{DM}{An optional named list indicating the design matrices to be used for the probability distribution parameters of each data 
stream. Each element of \code{DM} can either be a named list of regression formulas or a ``pseudo'' design matrix.  For example, for a 2-state 
model using the gamma distribution for a data stream named 'step', \code{DM=list(step=list(mean=~cov1, sd=~1))} specifies the mean 
parameters as a function of the covariate 'cov1' for each state.  This model could equivalently be specified as a 4x6 ``pseudo'' design matrix using 
character strings for the covariate: 
\code{DM=list(step=matrix(c(1,0,0,0,'cov1',0,0,0,0,1,0,0,0,'cov1',0,0,0,0,1,0,0,0,0,1),4,6))}
where the 4 rows correspond to the state-dependent paramaters (mean_1,mean_2,sd_1,sd_2) and the 6 columns correspond to the regression 
coefficients. 

Design matrices specified using formulas allow standard functions in R formulas
(e.g., \code{cos(cov)}, \code{cov1*cov2}, \code{I(cov^2)}).  Special formula functions include \code{cosinor(cov,period)} for modeling cyclical patterns, spline functions 
(\code{\link[splines]{bs}}, \code{\link[splines]{ns}}, \code{\link[splines2]{bSpline}}, \code{\link[splines2]{cSpline}}, \code{\link[splines2]{iSpline}}, and \code{\link[splines2]{mSpline}}), 
and state-specific formulas (see details). Any formula terms that are not state-specific are included on the parameters for all \code{nbStates} states.}

\item{cons}{An optional named list of vectors specifying a power to raise parameters corresponding to each column of the design matrix 
for each data stream. While there could be other uses, primarily intended to constrain specific parameters to be positive. For example, 
\code{cons=list(step=c(1,2,1,1))} raises the second parameter to the second power. Default=NULL, which simply raises all parameters to 
the power of 1. \code{cons} is ignored for any given data stream unless \code{DM} is specified.}

\item{userBounds}{An optional named list of 2-column matrices specifying bounds on the natural (i.e, real) scale of the probability 
distribution parameters for each data stream. For example, for a 2-state model using the wrapped Cauchy ('wrpcauchy') distribution for 
a data stream named 'angle' with \code{estAngleMean$angle=TRUE)}, \code{userBounds=list(angle=matrix(c(-pi,-pi,-1,-1,pi,pi,1,1),4,2,dimnames=list(c("mean_1",
"mean_2","concentration_1","concentration_2"))))} 
specifies (-1,1) bounds for the concentration parameters instead of the default [0,1) bounds.}

\item{workcons}{An optional named list of vectors specifying constants to add to the regression coefficients on the working scale for 
each data stream. Warning: use of \code{workcons} is recommended only for advanced users implementing unusual parameter constraints 
through a combination of \code{DM}, \code{cons}, and \code{workcons}. \code{workcons} is ignored for any given data stream unless \code{DM} is specified.}

\item{stateNames}{Optional character vector of length nbStates indicating state names.}

\item{model}{A \code{\link{momentuHMM}}, \code{\link{miHMM}}, or \code{\link{miSum}} object. This option can be used to simulate from a fitted model.  Default: NULL.
Note that, if this argument is specified, most other arguments will be ignored -- except for \code{nbAnimals},
\code{obsPerAnimal}, \code{states}, \code{initialPosition}, \code{lambda}, \code{errorEllipse}, and, if covariate values different from those in the data should be specified, 
\code{covs}, \code{spatialCovs}, and \code{centers}.}

\item{states}{\code{TRUE} if the simulated states should be returned, \code{FALSE} otherwise (default).}

\item{retrySims}{Number of times to attempt to simulate data within the spatial extent of \code{spatialCovs}. If \code{retrySims=0} (the default), an
error is returned if the simulated tracks(s) move beyond the extent(s) of the raster layer(s). Instead of relying on \code{retrySims}, in many cases
it might be better to simply expand the extent of the raster layer(s) and/or adjust the step length and turning angle probability distributions. 
Ignored if \code{spatialCovs=NULL}.}

\item{lambda}{Observation rate for location data. If \code{NULL} (the default), location data are obtained at regular intervals. Otherwise 
\code{lambda} is the rate parameter of the exponential distribution for the waiting times between successive location observations, i.e., 
\code{1/lambda} is the expected time between successive location observations. Only the 'step' and 'angle' data streams are subject to temporal irregularity;
any other data streams are observed at temporally-regular intervals.  Ignored unless a valid distribution for the 'step' data stream is specified.}

\item{errorEllipse}{List providing the upper bound for the semi-major axis (\code{M}; on scale of x- and y-coordinates), semi-minor axis (\code{m}; 
on scale of x- and y-coordinates), and orientation (\code{r}; in degrees) of location error ellipses. If \code{NULL} (the default), no location 
measurement error is simulated. If \code{errorEllipse} is specified, then each observed location is subject to bivariate normal errors as described 
in McClintock et al. (2015), where the components of the error ellipse for each location are randomly drawn from \code{runif(1,min(errorEllipse$M),max(errorEllipse$M))}, 
\code{runif(1,min(errorEllipse$m),max(errorEllipse$m))}, and \code{runif(1,min(errorEllipse$r),max(errorEllipse$r))}. If only a single value is provided for any of the 
error ellipse elements, then the corresponding component is fixed to this value for each location. Only the 'step' and 'angle' data streams are subject to location measurement error;
any other data streams are observed without error.  Ignored unless a valid distribution for the 'step' data stream is specified.}
}
\value{
If the simulated data are temporally regular (i.e., \code{lambda=NULL}) with no measurement error (i.e., \code{errorEllipse=NULL}), an object \code{\link{momentuHMMData}}, 
i.e., a dataframe of:
\item{ID}{The ID(s) of the observed animal(s)}
\item{...}{Data streams as specified by \code{dist}}
\item{x}{Either easting or longitude (if data streams include valid non-negative distribution for 'step')}
\item{y}{Either norting or latitude (if data streams include valid non-negative distribution for 'step')}
\item{...}{Covariates (if any)}

If simulated location data are temporally irregular (i.e., \code{lambda>0}) and/or include measurement error (i.e., \code{errorEllipse!=NULL}), a dataframe of:
\item{time}{Numeric time of each observed (and missing) observation}
\item{ID}{The ID(s) of the observed animal(s)}
\item{x}{Either easting or longitude observed location}
\item{y}{Either norting or latitude observed location}
\item{...}{Data streams that are not derived from location (if applicable)}
\item{...}{Covariates at temporally-regular true (\code{mux},\code{muy}) locations (if any)}
\item{mux}{Either easting or longitude true location}
\item{muy}{Either norting or latitude true location}
\item{error_semimajor_axis}{error ellipse semi-major axis (if applicable)}
\item{error_semiminor_axis}{error ellipse semi-minor axis (if applicable)}
\item{error_ellipse_orientation}{error ellipse orientation (if applicable)}
\item{ln.sd.x}{log of the square root of the x-variance of bivariate normal error (if applicable; required for error ellipse models in \code{\link{crawlWrap}})}
\item{ln.sd.y}{log of the square root of the y-variance of bivariate normal error (if applicable; required for error ellipse models in \code{\link{crawlWrap}})}
\item{error.corr}{correlation term of bivariate normal error (if applicable; required for error ellipse models in \code{\link{crawlWrap}})}
}
\description{
Simulates data from a (multivariate) hidden Markov model. Movement data can be generated with or without observation error attributable to temporal irregularity or location measurement error.
}
\details{
\itemize{
\item x- and y-coordinate location data are generated only if valid 'step' and 'angle' data streams are specified.  Vaild distributions for 'step' include 
'gamma', 'weibull', 'exp', and 'lnorm'.  Valid distributions for 'angle' include 'vm' and 'wrpcauchy'.  If only a valid 'step' data stream is specified, then only x-coordinates
are generated.

\item If \code{DM} is specified for a particular data stream, then the initial values are specified on 
the working (i.e., beta) scale of the parameters. The working scale of each parameter is determined by the link function used.
The function \code{\link{getParDM}} is intended to help with obtaining initial values on the working scale when specifying a design matrix and other 
parameter constraints. 

\item Simulated data that are temporally regular (i.e., \code{lambda=NULL}) and without location measurement error (i.e., \code{errorEllipse=NULL}) are returned
as a \code{\link{momentuHMMData}} object suitable for analysis using \code{\link{fitHMM}}.

\item Simulated location data that are temporally-irregular (i.e., \code{lambda>0}) and/or with location measurement error (i.e., \code{errorEllipse!=NULL}) are returned
as a data frame suitable for analysis using \code{\link{crawlWrap}}.

\item The matrix \code{beta} of regression coefficients for the transition probabilities has
one row for the intercept, plus one row for each covariate, and one column for
each non-diagonal element of the transition probability matrix. For example, in a 3-state
HMM with 2 covariates, the matrix \code{beta} has three rows (intercept + two covariates)
and six columns (six non-diagonal elements in the 3x3 transition probability matrix - filled in
row-wise).
In a covariate-free model (default), \code{beta} has one row, for the intercept.

\item State-specific formulas can be specified in \code{DM} using special formula functions. These special functions can take
the names \code{paste0("state",1:nbStates)} (where the integer indicates the state-specific formula).  For example, 
\code{DM=list(step=list(mean=~cov1+state1(cov2),sd=~cov2+state2(cov1)))} includes \code{cov1} on the mean parameter for all states, \code{cov2}
on the mean parameter for state 1, \code{cov2} on the sd parameter for all states, and \code{cov1} on the sd parameter for state 2.

\item State- and parameter-specific formulas can be specified for transition probabilities in \code{formula} using special formula functions.
These special functions can take the names \code{paste0("state",1:nbStates)} (where the integer indicates the current state from which transitions occur),
or \code{paste0("betaCol",nbStates*(nbStates-1))} (where the integer indicates the column of the \code{beta} matrix).  For example with \code{nbStates=3},
\code{formula=~cov1+betaCol1(cov2)+state3(cov3)} includes \code{cov1} on all transition probability parameters, \code{cov2} on the \code{beta} column corresponding
to the transition from state 1->2, and \code{cov3} on transition probabilities from state 3 (i.e., \code{beta} columns corresponding to state transitions 3->1 and 3->2).

\item Cyclical relationships (e.g., hourly, monthly) may be simulated using the \code{consinor(x,period)} special formula function for covariate \code{x}
and sine curve period of time length \code{period}. For example, if 
the data are hourly, a 24-hour cycle can be simulated using \code{~cosinor(cov1,24)}, where the covariate \code{cov1} is a repeating series
of integers \code{0,1,...,23,0,1,...,23,0,1,...} (note that \code{simData} will not do this for you, the appropriate covariate must be specified using the \code{covs} argument; see example below). 
The \code{cosinor(x,period)} function converts \code{x} to 2 covariates
\code{cosinorCos(x)=cos(2*pi*x/period)} and \code{consinorSin(x)=sin(2*pi*x/period} for inclusion in the model (i.e., 2 additional parameters per state). The amplitude of the sine wave
is thus \code{sqrt(B_cos^2 + B_sin^2)}, where \code{B_cos} and \code{B_sin} are the working parameters correponding to \code{cosinorCos(x)} and \code{cosinorSin(x)}, respectively (e.g., see Cornelissen 2014).

\item If the length of covariate values passed (either through 'covs', or 'model') is not the same
as the number of observations suggested by 'nbAnimals' and 'obsPerAnimal', then the series of
covariates is either shortened (removing last values - if too long) or extended (starting
over from the first values - if too short).
}
}
\examples{
# 1. Pass a fitted model to simulate from
# (m is a momentuHMM object - as returned by fitHMM - automatically loaded with the package)
# We keep the default nbAnimals=1.
m <- example$m
obsPerAnimal=c(50,100)
data <- simData(model=m,obsPerAnimal=obsPerAnimal)

# 2. Pass the parameters of the model to simulate from
stepPar <- c(1,10,1,5,0.2,0.3) # mean_1, mean_2, sd_1, sd_2, zeromass_1, zeromass_2
anglePar <- c(pi,0,0.5,2) # mean_1, mean_2, concentration_1, concentration_2
omegaPar <- c(1,10,10,1) # shape1_1, shape1_2, shape2_1, shape2_2
stepDist <- "gamma"
angleDist <- "vm"
omegaDist <- "beta"
data <- simData(nbAnimals=4,nbStates=2,dist=list(step=stepDist,angle=angleDist,omega=omegaDist),
                Par=list(step=stepPar,angle=anglePar,omega=omegaPar),nbCovs=2,
                zeroInflation=list(step=TRUE),
                obsPerAnimal=obsPerAnimal)

# 3. Include covariates
# (note that it is useless to specify "nbCovs", which are overruled
# by the number of columns of "cov")
cov <- data.frame(temp=log(rnorm(500,20,5)))
stepPar <- c(log(10),0.1,log(100),-0.1,log(5),log(25)) # working scale parameters for step DM
anglePar <- c(pi,0,0.5,2) # mean_1, mean_2, concentration_1, concentration_2
stepDist <- "gamma"
angleDist <- "vm"
data <- simData(nbAnimals=2,nbStates=2,dist=list(step=stepDist,angle=angleDist),
                Par=list(step=stepPar,angle=anglePar),
                DM=list(step=list(mean=~temp,sd=~1)),
                covs=cov,
                obsPerAnimal=obsPerAnimal)
                
# 4. Include example 'forest' spatial covariate raster layer
# nbAnimals and obsPerAnimal kept small to reduce example run time
spatialCov<-list(forest=forest)
data <- simData(nbAnimals=1,nbStates=2,dist=list(step=stepDist,angle=angleDist),
                Par=list(step=c(100,1000,50,100),angle=c(0,0,0.1,5)),
                beta=matrix(c(5,-10,-25,50),nrow=2,ncol=2,byrow=TRUE),
                formula=~forest,spatialCovs=spatialCov,
                obsPerAnimal=250,states=TRUE)
                
# 5. Specify design matrix for 'omega' data stream
# natural scale parameters for step and angle
stepPar <- c(1,10,1,5) # shape_1, shape_2, scale_1, scale_2
anglePar <- c(pi,0,0.5,0.7) # mean_1, mean_2, concentration_1, concentration_2

# working scale parameters for omega DM
omegaPar <- c(log(1),0.1,log(10),-0.1,log(10),-0.1,log(1),0.1)

stepDist <- "weibull"
angleDist <- "wrpcauchy"
omegaDist <- "beta"

data <- simData(nbStates=2,dist=list(step=stepDist,angle=angleDist,omega=omegaDist),
                Par=list(step=stepPar,angle=anglePar,omega=omegaPar),nbCovs=2,
                DM=list(omega=list(shape1=~cov1,shape2=~cov2)),
                obsPerAnimal=obsPerAnimal,states=TRUE)
                
# 6. Include temporal irregularity and location measurement error
lambda <- 2 # expect 2 observations per time step
errorEllipse <- list(M=50,m=25,r=180)
obsData <- simData(model=m,obsPerAnimal=obsPerAnimal,
                   lambda=lambda, errorEllipse=errorEllipse)
                   
# 7. Cosinor and state-dependent formulas
nbStates<-2
dist<-list(step="gamma")
Par<-list(step=c(100,1000,50,100))

# include 24-hour cycle on all transition probabilities
# include 12-hour cycle on transitions from state 2
formula=~cosinor(hour24,24)+state2(cosinor(hour12,12))

# specify appropriate covariates
covs<-data.frame(hour24=0:23,hour12=0:11)

beta<-matrix(c(-1.5,1,1,NA,NA,-1.5,-1,-1,1,1),5,2)
# row names for beta not required but can be helpful
rownames(beta)<-c("(Intercept)",
                  "cosinorCos(hour24, 24)",
                  "cosinorSin(hour24, 24)",
                  "cosinorCos(hour12, 12)",
                  "cosinorSin(hour12, 12)")
data.cos<-simData(nbStates=nbStates,dist=dist,Par=Par,
                  beta=beta,formula=formula,covs=covs)     
                  
# 8. Piecewise constant B-spline on step length mean and angle concentration
nObs <- 1000 # length of simulated track
cov <- data.frame(time=1:nObs) # time covariate for splines
dist <- list(step="gamma",angle="vm")
stepDM <- list(mean=~bSpline(time,df=3,degree=0),sd=~1)
angleDM <- list(mean=~1,concentration=~bSpline(time,df=3,degree=0))
DM <- list(step=stepDM,angle=angleDM)
Par <- list(step=c(log(1000),1,-1,log(100)),angle=c(0,log(10),2,-5))

data.spline<-simData(obsPerAnimal=nObs,nbStates=1,dist=dist,Par=Par,DM=DM,covs=cov)              
                
}
\references{
Cornelissen, G. 2014. Cosinor-based rhythmometry. Theoretical Biology and Medical Modelling 11:16.

McClintock BT, London JM, Cameron MF, Boveng PL. 2015. Modelling animal movement using the Argos satellite telemetry location error ellipse. 
Methods in Ecology and Evolution 6(3):266-277.
}
\seealso{
\code{\link{prepData}}, \code{\link{simObsData}}
}
