% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mtar.R
\name{simtar}
\alias{simtar}
\title{Simulation of multivariate time series according to a TAR model}
\usage{
simtar(
  n,
  k = 2,
  ars = list(p = 1),
  Intercept = TRUE,
  parms,
  delay = 0,
  thresholds = 0,
  t.series,
  ex.series,
  dist = "gaussian",
  extra
)
}
\arguments{
\item{n}{a positive integer value indicating the length of the desired output series.}

\item{k}{a positive integer value indicating the dimension of the desired output series.}

\item{ars}{a list composed of three objects, namely: \code{p}, \code{q} and \code{d},
each of which corresponds to a vector of \eqn{l} non-negative integers, where \eqn{l} represents the number of regimes in the TAR model.}

\item{Intercept}{an (optional) logical variable. If \code{TRUE}, then the model includes an intercept.}

\item{parms}{a list with as many sublists as regimes in the user-specified TAR model. Each sublist is composed of two matrices. The first corresponds
to location parameters, while the second corresponds to scale parameters.}

\item{delay}{an (optional) non-negative integer value indicating the delay in the threshold series.}

\item{thresholds}{a vector with \eqn{l-1} real values sorted ascendingly.}

\item{t.series}{a matrix with the values of the threshold series.}

\item{ex.series}{a matrix with the values of the multivariate exogenous series.}

\item{dist}{an (optional) character string which allows the user to specify the multivariate
distribution to be used to describe the behavior of the noise process. The
available options are: Gaussian ("Gaussian"), Student-\eqn{t} ("Student-t"),
Slash ("Slash"), Symmetric Hyperbolic ("Hyperbolic"), Laplace ("Laplace"), and
contaminated normal ("Contaminated normal"). As default, \code{dist} is set to
"Gaussian".}

\item{extra}{a value indicating the value of the extra parameter of the noise process distribution, if any.}
}
\value{
a \code{data.frame} containing the output series, threshold series (if any), and multivariate exogenous series (if any).
}
\description{
This function simulates multivariate time series according to a user-specified TAR model.
}
\examples{
###### Simulation of a trivariate TAR model with two regimes
n <- 2000
k <- 3
ars <- list(p=c(1,2))
Z <- as.matrix(arima.sim(n=n+max(ars$p),list(ar=c(0.5))))
Intercept <- TRUE
parms <- list()
for(i in 1:length(ars$p)){
   np <- Intercept + ars$p[i]*k
   parms[[i]] <- list()
   parms[[i]]$location <- c(ifelse(runif(np*k)<=0.5,1,-1)*rbeta(np*k,shape1=4,shape2=16))
   parms[[i]]$location <- matrix(parms[[i]]$location,np,k)
   parms[[i]]$scale <- rgamma(k,shape=1,scale=1)*diag(k)
}
thresholds <- quantile(Z,probs=seq(1,length(ars$p)-1)/length(ars$p))
out1 <- simtar(n=n,k=k,ars=ars,Intercept=Intercept,parms=parms,
               thresholds=thresholds,t.series=Z,dist="Student-t",extra=6)
str(out1)

###### Simulation of a trivariate VAR model
n <- 2000
k <- 3
ars <- list(p=2)
Intercept <- TRUE
parms <- list()
for(i in 1:length(ars$p)){
   np <- Intercept + ars$p[i]*k
   parms[[i]] <- list()
   parms[[i]]$location <- c(ifelse(runif(np*k)<=0.5,1,-1)*rbeta(np*k,shape1=4,shape2=16))
   parms[[i]]$location <- matrix(parms[[i]]$location,np,k)
   parms[[i]]$scale <- rgamma(k,shape=1,scale=1)*diag(k)
}
out2 <- simtar(n=n,k=k,ars=ars,Intercept=Intercept,parms=parms,
               dist="Slash",extra=2)
str(out2)

}
