% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/check_balance.R
\name{check_balance}
\alias{check_balance}
\title{Check covariate balance of the control and treated groups}
\usage{
check_balance(
  z,
  X,
  st,
  selected,
  denom_variance = "treated",
  plot = FALSE,
  message = TRUE
)
}
\arguments{
\item{z}{a treatment indicator vector with \code{i}th entry equal to 0 if
unit \code{i} is a control and equal to 1 if unit \code{i} is treated.}

\item{X}{a data frame containing the covariates in the columns over which balance is desired. The number
of rows should equal the length of \code{z}.}

\item{st}{a stratum vector with the \code{i}th entry equal to the
stratum of unit \code{i}. This should have the same order of units and length
as \code{z}.}

\item{selected}{a boolean vector including whether each unit was selected as part of the treated and control
groups for analysis. Should be the same length as \code{z} and typically comes from the results of
\code{\link{optimize_controls}()}.}

\item{denom_variance}{character stating what variance to use in the standardization:
either the default "treated", meaning the standardization will use the
treated variance (across all strata), or "pooled", meaning
the standardization will use the average of the treated and control variances.}

\item{plot}{a boolean denoting whether to generate love plots for the standardized differences.}

\item{message}{a boolean denoting whether to print a message about the level of balance achieved}
}
\value{
List containing:
\describe{
  \item{sd_across}{matrix with one row per covariate and two columns: one for the
  standardized difference before a subset of controls were selected and one for after.}
  \item{sd_strata}{matrix similar to \code{sd_across}, but with separate
  standardized differences for each stratum for each covariate.}
  \item{sd_strata_avg}{matrix similar to \code{sd_across}, but taking the
  average of the standardized differences within the strata, weighted by stratum size.}
  \item{plot_across}{ggplot object plotting \code{sd_across}, only exists if \code{plot = TRUE}.}
  \item{plot_strata}{a named list of ggplot objects plotting \code{sd_strata},
  one for each stratum, only exists if \code{plot = TRUE}.}
  \item{plot_strata_avg}{ggplot object plotting \code{sd_strata_avg}, only exists if \code{plot = TRUE}.}
  \item{plot_pair}{ggplot object with two facets displaying \code{sd_across} and
  \code{sd_strata_avg} with one y-axis, only exists if \code{plot = TRUE}.}
}
}
\description{
Reports standardized differences in means between the treated and
control group before and after choosing a subset of controls.
These differences are reported both across strata and within strata.
This function can also generate love plots of the same quantities.
}
\examples{

data('nh0506')

# Create strata
age_cat <- cut(nh0506$age,
               breaks = c(19, 39, 50, 85),
               labels = c('< 40 years', '40 - 50 years', '> 50 years'))
strata <- age_cat : nh0506$sex

# Balance age, race, education, poverty ratio, and bmi both across and within the levels of strata
constraints <- generate_constraints(
                 balance_formulas = list(age + race + education + povertyr + bmi ~ 1 + strata),
                 z = nh0506$z,
                 data = nh0506)

# Choose one control for every treated unit in each stratum,
# balancing the covariates as described by the constraints
results <- optimize_controls(z = nh0506$z,
                             X = constraints$X,
                             st = strata,
                             importances = constraints$importances,
                             ratio = 1)

cov_data <- nh0506[, c('sex', 'age', 'race', 'education', 'povertyr', 'bmi')]

# Check balance
stand_diffs <- check_balance(z = nh0506$z,
                             X = cov_data,
                             st = strata,
                             selected = results$selected,
                             plot = TRUE)
}
