\name{netcomb}

\alias{netcomb}

\title{Additive network meta-analysis for combinations of treatments}

\description{
  Some treatments in a network meta-analysis may be combinations of
  other treatments or have common components. The influence of
  individual components can be evaluated in an additive network
  meta-analysis model assuming that the effect of treatment
  combinations is the sum of the effects of its components. This
  function implements this additive model in a frequentist way.
}

\usage{
netcomb(x, inactive = NULL, sep.comps = "+", C.matrix,
        comb.fixed = x$comb.fixed,
        comb.random = x$comb.random | !is.null(tau.preset),
        tau.preset = NULL)
}

\arguments{
  \item{x}{An object of class \code{netmeta}.}
  \item{inactive}{A character string defining the inactive treatment
    (see Details).}
  \item{sep.comps}{A single character to define separator between
    treatment components.}
  \item{C.matrix}{C matrix (see Details).}
  \item{comb.fixed}{A logical indicating whether a fixed effects (common
    effects) network meta-analysis should be conducted.}
  \item{comb.random}{A logical indicating whether a random effects
    network meta-analysis should be conducted.}
  \item{tau.preset}{An optional value for the square-root of the
    between-study variance \eqn{\tau^2}.}
}

\details{
Treatments in network meta-analysis (NMA) can be complex
interventions. Some treatments may be combinations of others or have
common components. The standard analysis provided by
\code{\link{netmeta}} is a NMA where all existing (single or combined)
treatments are considered as different nodes in the network. Exploiting
the fact that some treatments are combinations of common components, an
additive component network meta-analysis (CNMA) model can be used to
evaluate the influence of individual components. This model assumes that
the effect of a treatment combination is the sum of the effects of its
components which implies that common components cancel out in
comparisons.

The additive CNMA model has been implemented using Bayesian methods
(Mills et al., 2012; Welton et al., 2013). This function implements the
additive model in a frequentist way (Rücker et al., 2018).

The underlying multivariate model is given by

\deqn{\bold{\delta} = \bold{B} \bold{\theta}, \bold{\theta} = \bold{C} \bold{\beta}}

with
\itemize{
\item[\eqn{\bold{\delta}}] vector of true treatment effects
(differences) from individual studies,
\item[\eqn{\bold{B}}] is a design matrix describing the structure of
the network,
\item[\eqn{\bold{\theta}}] parameter vector that represents the
existing combined treatments,
\item[\eqn{\bold{C}}] matrix describing how the treatments are
composed,
\item[\eqn{\bold{\beta}}] is a parameter vector representing the
treatment components.
}
All parameters are estimated using weighted least squares regression.

Argument \code{inactive} can be used to specify a single component
that does not have any therapeutic value. Accordingly, it is assumed
that the treatment effect of the combination of this component with an
additional treatment component is equal to the treatment effect of the
additional component alone.

Argument \code{sep.comps} can be used to specify the separator between
individual components. By default, the matrix \strong{C} is calculated
internally from treatment names. However, it is possible to specify a
different matrix using argument \code{C.matrix}.
}                                       

\value{
  An object of class \code{netcomb} with corresponding \code{print},
  \code{summary}, and \code{forest} functions. The object is a list
  containing the following components:
  \item{x, inactive, sep.comps, C.matrix}{As defined above.}  
  \item{comb.fixed, comb.random, tau.preset}{As defined above.}  
  \item{k}{Total number of studies.}
  \item{m}{Total number of pairwise comparisons.}
  \item{n}{Total number of treatments.}
  \item{c}{Total number of components.}
  \item{comparisons.fixed, comparisons.random}{Lists with components
    studlab, treat1, treat2, TE, seTE, lower, upper, z, p level, df
    (fixed and random effects model).}
  \item{components.fixed, components.random}{Lists with components TE,
    seTE, lower, upper, z, p level, df (fixed and random effects
    model).}
  \item{combinations.fixed, combinations.random}{Lists with components
    TE, seTE, lower, upper, z, p level, df (fixed and random effects
    model).}
  \item{sm}{Summary measure.}
  \item{level.comb}{Level for confidence intervals.}
  \item{Q.additive}{Overall heterogeneity / inconsistency statistic
    (additive model).}
  \item{df.Q.additive}{Degrees of freedom for test of heterogeneity /
    inconsistency (additive model).}
  \item{pval.Q.additive}{P-value for test of heterogeneity /
    inconsistency (additive model).}
  \item{Q.standard}{Overall heterogeneity / inconsistency statistic
    (standard model).}
  \item{df.Q.standard}{Degrees of freedom for test of heterogeneity /
    inconsistency (standard model).}
  \item{pval.Q.standard}{P-value for test of heterogeneity /
    inconsistency (standard model).}
  \item{Q.diff.fixed, Q.diff.random}{Test statistic for difference in
    goodness of fit between standard and additive model (fixed and
    random effects model).}
  \item{df.Q.diff.fixed, df.Q.diff.random}{Degrees of freedom for
    difference in goodness of fit between standard and additive model
    (fixed and random effects model).}
  \item{pval.Q.diff.fixed, pval.Q.diff.random}{P-value for difference
    in goodness of fit between standard and additive model (fixed and
    random effects model).}
  \item{backtransf}{A logical indicating whether results should be
    back transformed in printouts and forest plots.}
  \item{nchar.trts}{A numeric defining the minimum number of characters
    used to create unique treatment and component names.}
  \item{title}{Title of meta-analysis / systematic review.}
  \item{call}{Function call.}
  \item{version}{Version of R package netmeta used to create object.}
}

\references{
  König J, Krahn U, Binder H (2013):
  Visualizing the flow of evidence in network meta-analysis and
  characterizing mixed treatment comparisons.
  \emph{Statistics in Medicine},
  \bold{32}, 5414--29

  Mills EJ, Thorlund K, Ioannidis JP (2012):
  Calculating additive treatment effects from multiple randomized
  trials provides useful estimates of combination therapies.
  \emph{Journal of Clinical Epidemiology},
  \bold{65}, 1282--8

  Rücker G, Petropoulou M, Schwarzer G (2018):
  Network meta-analysis of multicomponent interventions.
  \emph{Manuscript submitted for publication}.

  Welton NJ, Caldwell DM, Adamopoulos E, Vedhara K (2009):
  Mixed treatment comparison meta-analysis of complex interventions:
  psychological interventions in coronary heart disease.
  \emph{American Journal of Epidemiology},
  \bold{169}: 1158--65
}

\author{Gerta Rücker \email{ruecker@imbi.uni-freiburg.de}, Guido
Schwarzer \email{sc@imbi.uni-freiburg.de}}

\seealso{\link{netmeta}, \link{forest.netcomb}, \link{print.netcomb}, \link{discomb}}

\examples{
data(Linde2016)

# Only consider studies including Face-to-face PST (to reduce runtime
# of example)
#
face <- subset(Linde2016, id \%in\% c(16, 24, 49, 118))

# Conduct random effects network meta-analysis
#
net1 <- netmeta(lnOR, selnOR, treat1, treat2, id,
                data = face, ref = "placebo",
                sm = "OR", comb.fixed = FALSE)
#
summary(net1)
forest(net1, xlim = c(0.2, 50))

# Additive model for treatment components (with placebo as inactive
# treatment)
#
nc1 <- netcomb(net1, inactive = "placebo")
#
summary(nc1)
forest(nc1, xlim = c(0.2, 50))

\dontrun{
# Specify, order of treatments
#
trts <- c("TCA", "SSRI", "SNRI", "NRI", "Low-dose SARI", "NaSSa",
          "rMAO-A", "Ind drug", "Hypericum", "Face-to-face CBT",
          "Face-to-face PST", "Face-to-face interpsy", "Face-to-face psychodyn",
          "Other face-to-face", "Remote CBT", "Self-help CBT", "No contact CBT",
          "Face-to-face CBT + SSRI", "Face-to-face interpsy + SSRI",
          "Face-to-face PST + SSRI", "UC", "Placebo")
#
# Note, three treatments are actually combinations of 'SSRI' with
# other components:
#
# "Face-to-face CBT + SSRI",
# "Face-to-face interpsy + SSRI",
# "Face-to-face PST + SSRI"

# Conduct random effects network meta-analysis
#
net1 <- netmeta(lnOR, selnOR, treat1, treat2, id,
                data = Linde2016, ref = "placebo",
                seq = trts,
                sm = "OR", comb.fixed = FALSE)
#
summary(net1)
forest(net1, xlim = c(0.2, 50))

# Additive model for treatment components (with placebo as inactive
# treatment)
#
nc1 <- netcomb(net1, inactive = "placebo")
#
summary(nc1)
forest(nc1, xlim = c(0.2, 50))
}
}                                                          

\keyword{Network meta-analysis}
\keyword{Additive network meta-analysis}
