% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/f_EM_hyperEstimation.R
\name{hyperEM}
\alias{hyperEM}
\title{Estimate hyperparameters using an EM algorithm}
\usage{
hyperEM(
  data,
  theta_init_vec,
  squashed = TRUE,
  zeroes = FALSE,
  N_star = 1,
  method = c("score", "nlminb"),
  profile = c("parameter", "distribution"),
  LL_tol = 1e-04,
  consecutive = 100,
  param_lower = 1e-05,
  param_upper = 20,
  print_level = 2,
  max_iter = 5000,
  conf_ints = FALSE,
  conf_level = c("95", "80", "90", "99"),
  track = FALSE
)
}
\arguments{
\item{data}{A data frame from \code{\link{processRaw}} or
\code{\link{squashData}} containing columns named \emph{N}, \emph{E}, and
(if squashed) \emph{weight}.}

\item{theta_init_vec}{A numeric vector of initial hyperparameter guesses
ordered as: \eqn{\alpha_1, \beta_1, \alpha_2, \beta_2, P}.}

\item{squashed}{A scalar logical (\code{TRUE} or \code{FALSE}) indicating
whether or not data squashing was used.}

\item{zeroes}{A scalar logical specifying if zero counts are included.}

\item{N_star}{A positive scalar whole number value for the minimum count
size to be used for hyperparameter estimation. If zeroes are used, set
\code{N_star} to \code{NULL}.}

\item{method}{A scalar string indicating which method (i.e. score functions
or log-likelihood function) to use for the maximization steps. Possible
values are \code{"score"} and \code{"nlminb"}.}

\item{profile}{A scalar string indicating which method to use to optimize the
log-likelihood function if \code{method = "nlminb"} (ignored if
\code{method = "score"}).  \code{profile = "parameter"} optimizes one
parameter (\eqn{\alpha} or \eqn{\beta}) from the log-likelihood function at
a time.  \code{profile = "distribution"} optimizes one distribution from
the mixture at a time (\eqn{\alpha} and \eqn{\beta} simultaneously).}

\item{LL_tol}{A scalar numeric value for the tolerance used for determining
when the change in log-likelihood at each iteration is sufficiently small.
Used for convergence assessment.}

\item{consecutive}{A positive scalar whole number value for the number of
consecutive iterations the change in log-likelihood must be below
\code{LL_tol} in order to reach convergence. Larger values reduce the
chance of getting stuck in a flat region of the curve.}

\item{param_lower}{A scalar numeric value for the smallest acceptable value
for each \eqn{\alpha} and \eqn{\beta} estimate.}

\item{param_upper}{A scalar numeric value for the largest acceptable value
for each \eqn{\alpha} and \eqn{\beta} estimate.}

\item{print_level}{A value that determines how much information is printed
during execution. Possible value are \code{0} for no printing, \code{1} for
minimal information, and \code{2} for maximal information.}

\item{max_iter}{A positive scalar whole number value for the maximum number
of iterations to use.}

\item{conf_ints}{A scalar logical indicating if confidence intervals and
standard errors should be returned.}

\item{conf_level}{A scalar string for the confidence level used if confidence
intervals are requested.}

\item{track}{A scalar logical indicating whether or not to retain the
hyperparameter estimates and log-likelihood value at each iteration. Can be
used for plotting to better understand the algorithm's behavior.}
}
\value{
A list including the following:
  \itemize{
    \item{\emph{estimates}: }{The maximum likelihood estimate (MLE) of the
      hyperparameter, \eqn{\theta}.}
    \item{\emph{conf_int}: }{A data frame including the standard errors and
      confidence limits for \code{estimates}. Only included if
      \code{conf_ints = TRUE}.}
    \item{\emph{maximum}: }{The log-likelihood function evaluated at
      \code{estimates}.}
    \item{\emph{method}: }{The method used in the maximization step.}
    \item{\emph{elapsed}: }{The elapsed function execution time in seconds.}
    \item{\emph{iters}: }{The number of iterations used.}
    \item{\emph{score}: }{The score functions (i.e. score vector) evaluated
      at \code{estimates}. All elements should be close to zero.}
    \item{\emph{score_norm}: }{The Euclidean norm of the score vector
      evaluated at \code{estimates}. Should be close to zero.}
    \item{\emph{tracking}: }{The estimates of \eqn{\theta} at each iteration
      and the log-likelihood function evaluated at those estimates. Unless
      \code{track = TRUE}, only shows the starting point of the algorithm.}
  }
}
\description{
\code{hyperEM} finds hyperparameter estimates using a variation on the
Expectation-Maximization (EM) algorithm known as the Expectation/Conditional
Maximization (ECM) algorithm (Meng et al, 1993). The algorithm estimates each
element of the hyperparameter vector, \eqn{\theta}, while holding fixed
(conditioning on) the other parameters in the vector. Alternatively, it can
estimate both parameters for each distribution in the mixture while holding
the parameters from the other distribution and the mixing fraction fixed.
}
\details{
If \code{method = "score"}, the maximization step finds a root
  of the score function. If \code{method = "nlminb"},
  \code{\link[stats]{nlminb}} is used to find a minimum of the negative
  log-likelihood function.

If \code{method = "score"} and \code{zeroes = FALSE}, then
  \code{'N_star'} must equal \code{1}.

If \code{method = "score"}, the model is reparameterized. The
  parameters are transformed to force the parameter space to include all real
  numbers. This approach addresses numerical issues at the edge of the
  parameter space. The reparameterization follows:
  \eqn{\alpha_{prime} = log(\alpha)}, \eqn{\beta_{prime} = log(\beta)}, and
  \eqn{P_{prime} = tan(pi * P - pi / 2)}. However, the values returned in
  \code{estimates} are on the original scale (back-transformed).

On every 100th iteration, the procedure described in Millar (2011)
  is used to accelerate the estimate of \eqn{\theta}.

The score vector and its Euclidean norm should be close to zero at
  a local maximum and can therefore be used to help assess the reliability of
  the results. A local maximum might not be the global MLE, however.

Asymptotic normal confidence intervals, if requested, use standard
  errors calculated from the observed Fisher information matrix as discussed
  in DuMouchel (1999).
}
\examples{
data(caers)
proc <- processRaw(caers)
squashed <- squashData(proc, bin_size = 100, keep_pts = 0)
squashed <- squashData(squashed, count = 2, bin_size = 12, keep_pts = 24)
hyperEM(squashed, theta_init_vec = c(1, 1, 2, 2, .3), consecutive = 10)

}
\references{
DuMouchel W (1999). "Bayesian Data Mining in Large Frequency
  Tables, With an Application to the FDA Spontaneous Reporting System."
  \emph{The American Statistician}, 53(3), 177-190.

Meng X-L, Rubin D (1993). "Maximum likelihood estimation via the
  ECM algorithm: A general framework", \emph{Biometrika}, 80(2), 267-278.

Millar, Russell B (2011). "Maximum Likelihood Estimation and
  Inference", \emph{John Wiley & Sons, Ltd}, 111-112.
}
\seealso{
\code{\link[stats]{uniroot}} for finding a zero of the score
  function and \code{\link[stats]{nlminb}} for minimizing the negative
  log-likelihood function

Other hyperparameter estimation functions: 
\code{\link{autoHyper}()},
\code{\link{exploreHypers}()}
}
\concept{hyperparameter estimation functions}
\keyword{openEBGM}
