% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/outliertree.R
\name{unpack.outlier.tree}
\alias{unpack.outlier.tree}
\title{Unpack Outlier Tree model after de-serializing}
\usage{
unpack.outlier.tree(model)
}
\arguments{
\item{model}{An Outlier Tree object as returned by `outlier.tree`, which has been just loaded from a disk
file through `readRDS`, `load`, or a session restart.}
}
\value{
No return value. Object is modified in-place.
}
\description{
After persisting an outlier tree model object through `saveRDS`, `save`, or restarting a session, the
underlying C++ objects that constitute the outlier tree model and which live only on the C++ heap memory are not saved along,
thus not restored after loading a saved model through `readRDS` or `load`.

The model object however keeps serialized versions of the C++ objects as raw bytes, from which the C++ objects can be
reconstructed, and are done so automatically after calling `predict`, `print`, or `summary` on the
freshly-loaded object from `readRDS` or `load`.

But due to R's environments system (as opposed to other systems such as Python which can use pass-by-reference), they will
only be re-constructed in the environment that is calling `predict`, `print`, etc. and not in higher-up environments
(i.e. if you call `predict` on the object from inside different functions, each function will have to reconstruct the
C++ objects independently and they will only live within the function that called `predict`).

This function serves as an environment-level unpacker that will reconstruct the C++ object in the environment in which
it is called (i.e. if you need to call `predict` from inside multiple functions, use this function before passing the
freshly-loaded model object to those other functions, and then they will not need to reconstruct the C++ objects anymore),
in the same way as `predict` or `print`, but without producing any outputs or messages.
}
\examples{
### Warning: this example will generate a temporary .Rds
### file in your temp folder, and will then delete it
library(outliertree)
set.seed(1)
df <- as.data.frame(matrix(rnorm(1000), nrow = 250))
otree <- outlier.tree(df, outliers_print=0, nthreads=1)
temp_file <- file.path(tempdir(), "otree.Rds")
saveRDS(otree, temp_file)
otree2 <- readRDS(temp_file)
file.remove(temp_file)

### will de-serialize inside, but object is short-lived
wrap_predict <- function(model, data) {
    pred <- predict(model, data, outliers_print = 0)
    cat("pointer inside function is this: ")
    print(model$obj_from_cpp$ptr_model)
    return(pred)
}
temp <- wrap_predict(otree2, df)
cat("pointer outside function is this: \\n")
print(otree2$obj_from_cpp$ptr_model) ### pointer to the C++ object

### now unpack the C++ object beforehand
unpack.outlier.tree(otree2)
print("after unpacking beforehand")
temp <- wrap_predict(otree2, df)
cat("pointer outside function is this: \\n")
print(otree2$obj_from_cpp$ptr_model)
}
