% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/outliertree.R
\name{unpack.outlier.tree}
\alias{unpack.outlier.tree}
\title{Unpack Outlier Tree model after de-serializing}
\usage{
unpack.outlier.tree(model)
}
\arguments{
\item{model}{An Outlier Tree object as returned by `outlier.tree`, which has
been just loaded from a disk file through `readRDS`, `load`, or a session restart.}
}
\value{
No return value. Object is modified in-place.
}
\description{
After persisting an outlier tree model object through `saveRDS`, `save`,
or restarting a session, the underlying C++ objects that constitute the outlier tree
model and which live only on the C++ heap memory are not saved along, thus not
restored after loading a saved model through `readRDS` or `load`.

The model object however keeps serialized versions of the C++ objects as raw bytes,
from which the C++ objects can be reconstructed, and are done so automatically after
calling `predict`, `print`, or `summary` on the freshly-loaded object from
`readRDS` or `load`.

This function allows de-serializing the object bytes without invoking any extra
side effects or computations, akin to XGBoost's `xgb.Booster.complete` or
CatBoost's `catboost.restore_handle`.
}
\details{
If the model is going to be used in a production system, it's possible
after de-serialization to delete the raw bytes in order to save memory (e.g.
`otree$obj_from_cpp$serialized_obj <- NULL`). The memory will however not be
freed automatically, as it's managed by R's garbage collector.
}
\examples{
### Warning: this example will generate a temporary .Rds
### file in your temp folder, and will then delete it
library(outliertree)
set.seed(1)
df <- as.data.frame(matrix(rnorm(1000), nrow = 250))
otree <- outlier.tree(df, outliers_print=0, nthreads=1)
temp_file <- file.path(tempdir(), "otree.Rds")
saveRDS(otree, temp_file)
otree2 <- readRDS(temp_file)
file.remove(temp_file)

cat("Pointer after loading model is this: \n")
print(otree2$obj_from_cpp$ptr_model)

### now unpack the raw bytes
unpack.outlier.tree(otree2)
cat("Pointer after unpacking is this: \n")
print(otree2$obj_from_cpp$ptr_model)
}
