#' Read the grammar.output file generated by bison
#' 
#' When bison is used with the \code{-v} switch, the \code{grammar.output} file
#' is generated. This function extracts symbols, terminal and non-terminal,
#' from the file and structure them in a data frame
#' 
#' 
#' @return A data frame. See \link{symbols}
#' @author Romain Francois <romain@@r-enthusiasts.com>
#' @keywords manip
#' @examples
#' 
#' \dontrun{
#' # the bison generated grammar output
#' system.file( "grammar", "gram.output", package = "parser" )
#' 
#' grammar.symbols()
#' }
#' @export
grammar.symbols <- function(gram.output.file){
    # the output file from bison
    # gram.output.file <- "inst/grammar/gram.output"
    rl <- readLines( gram.output.file ) 
    
    .extract <- function( start.rx, end.rx, type = "terminal" ){
        start <- grep( start.rx, rl )[1] + 1L
        end <- grep( end.rx, rl)[1] - 1L
        
        rl <- rl[ start:end ]
        rx <- "(^.*) \\((\\d+)\\).*"
        rl <- grep( rx, rl, perl = T, value = T )
        desc   <- gsub( rx, "\\1", rl, perl = TRUE )
        token  <- as.integer( gsub( rx, "\\2", rl, perl = TRUE ) )
        data.frame( desc = desc, token = token, 
                    terminal = rep.int( type , length(token) ), 
                    stringsAsFactors = FALSE )
    }
    rbind( 
        .extract( 
            "^Terminals, with rules where they appear",
            "^Nonterminals, with rules where they appear", 
            TRUE ), 
        .extract( 
            "^Nonterminals, with rules where they appear",
            "^state 0", 
            FALSE )
    )
}

infile <- commandArgs(T)[[1]]
outfile <- commandArgs(T)[[2]]

cat(sprintf("Reading bison outpuf from '%s'", normalizePath(infile)), "\n")
cat(sprintf("Writing symbol table to '%s'", normalizePath(outfile, mustWork=FALSE)), "\n")

grammar_symbols <- grammar.symbols(infile)
save(grammar_symbols, file=outfile)
