#' AWS App Runner
#'
#' @description
#' App Runner
#'
#' App Runner is an application service that provides a fast, simple, and
#' cost-effective way to go directly from an existing container image or
#' source code to a running service in the Amazon Web Services Cloud in
#' seconds. You don't need to learn new technologies, decide which compute
#' service to use, or understand how to provision and configure Amazon Web
#' Services resources.
#'
#' App Runner connects directly to your container registry or source code
#' repository. It provides an automatic delivery pipeline with fully
#' managed operations, high performance, scalability, and security.
#'
#' For more information about App Runner, see the [App Runner Developer
#' Guide](https://docs.aws.amazon.com/apprunner/latest/dg/). For release
#' information, see the [App Runner Release
#' Notes](https://docs.aws.amazon.com/apprunner/latest/relnotes/).
#'
#' To install the Software Development Kits (SDKs), Integrated Development
#' Environment (IDE) Toolkits, and command line tools that you can use to
#' access the API, see [Tools for Amazon Web
#' Services](https://aws.amazon.com/developer/tools/).
#'
#' **Endpoints**
#'
#' For a list of Region-specific endpoints that App Runner supports, see
#' [App Runner endpoints and
#' quotas](https://docs.aws.amazon.com/general/latest/gr/apprunner.html) in
#' the *Amazon Web Services General Reference*.
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- apprunner(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- apprunner()
#' svc$associate_custom_domain(
#'   Foo = 123
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.compute:apprunner_associate_custom_domain]{associate_custom_domain} \tab Associate your own domain name with the App Runner subdomain URL of your App Runner service\cr
#'  \link[paws.compute:apprunner_create_auto_scaling_configuration]{create_auto_scaling_configuration} \tab Create an App Runner automatic scaling configuration resource\cr
#'  \link[paws.compute:apprunner_create_connection]{create_connection} \tab Create an App Runner connection resource\cr
#'  \link[paws.compute:apprunner_create_observability_configuration]{create_observability_configuration} \tab Create an App Runner observability configuration resource\cr
#'  \link[paws.compute:apprunner_create_service]{create_service} \tab Create an App Runner service\cr
#'  \link[paws.compute:apprunner_create_vpc_connector]{create_vpc_connector} \tab Create an App Runner VPC connector resource\cr
#'  \link[paws.compute:apprunner_create_vpc_ingress_connection]{create_vpc_ingress_connection} \tab Create an App Runner VPC Ingress Connection resource\cr
#'  \link[paws.compute:apprunner_delete_auto_scaling_configuration]{delete_auto_scaling_configuration} \tab Delete an App Runner automatic scaling configuration resource\cr
#'  \link[paws.compute:apprunner_delete_connection]{delete_connection} \tab Delete an App Runner connection\cr
#'  \link[paws.compute:apprunner_delete_observability_configuration]{delete_observability_configuration} \tab Delete an App Runner observability configuration resource\cr
#'  \link[paws.compute:apprunner_delete_service]{delete_service} \tab Delete an App Runner service\cr
#'  \link[paws.compute:apprunner_delete_vpc_connector]{delete_vpc_connector} \tab Delete an App Runner VPC connector resource\cr
#'  \link[paws.compute:apprunner_delete_vpc_ingress_connection]{delete_vpc_ingress_connection} \tab Delete an App Runner VPC Ingress Connection resource that's associated with an App Runner service\cr
#'  \link[paws.compute:apprunner_describe_auto_scaling_configuration]{describe_auto_scaling_configuration} \tab Return a full description of an App Runner automatic scaling configuration resource\cr
#'  \link[paws.compute:apprunner_describe_custom_domains]{describe_custom_domains} \tab Return a description of custom domain names that are associated with an App Runner service\cr
#'  \link[paws.compute:apprunner_describe_observability_configuration]{describe_observability_configuration} \tab Return a full description of an App Runner observability configuration resource\cr
#'  \link[paws.compute:apprunner_describe_service]{describe_service} \tab Return a full description of an App Runner service\cr
#'  \link[paws.compute:apprunner_describe_vpc_connector]{describe_vpc_connector} \tab Return a description of an App Runner VPC connector resource\cr
#'  \link[paws.compute:apprunner_describe_vpc_ingress_connection]{describe_vpc_ingress_connection} \tab Return a full description of an App Runner VPC Ingress Connection resource\cr
#'  \link[paws.compute:apprunner_disassociate_custom_domain]{disassociate_custom_domain} \tab Disassociate a custom domain name from an App Runner service\cr
#'  \link[paws.compute:apprunner_list_auto_scaling_configurations]{list_auto_scaling_configurations} \tab Returns a list of active App Runner automatic scaling configurations in your Amazon Web Services account\cr
#'  \link[paws.compute:apprunner_list_connections]{list_connections} \tab Returns a list of App Runner connections that are associated with your Amazon Web Services account\cr
#'  \link[paws.compute:apprunner_list_observability_configurations]{list_observability_configurations} \tab Returns a list of active App Runner observability configurations in your Amazon Web Services account\cr
#'  \link[paws.compute:apprunner_list_operations]{list_operations} \tab Return a list of operations that occurred on an App Runner service\cr
#'  \link[paws.compute:apprunner_list_services]{list_services} \tab Returns a list of running App Runner services in your Amazon Web Services account\cr
#'  \link[paws.compute:apprunner_list_tags_for_resource]{list_tags_for_resource} \tab List tags that are associated with for an App Runner resource\cr
#'  \link[paws.compute:apprunner_list_vpc_connectors]{list_vpc_connectors} \tab Returns a list of App Runner VPC connectors in your Amazon Web Services account\cr
#'  \link[paws.compute:apprunner_list_vpc_ingress_connections]{list_vpc_ingress_connections} \tab Return a list of App Runner VPC Ingress Connections in your Amazon Web Services account\cr
#'  \link[paws.compute:apprunner_pause_service]{pause_service} \tab Pause an active App Runner service\cr
#'  \link[paws.compute:apprunner_resume_service]{resume_service} \tab Resume an active App Runner service\cr
#'  \link[paws.compute:apprunner_start_deployment]{start_deployment} \tab Initiate a manual deployment of the latest commit in a source code repository or the latest image in a source image repository to an App Runner service\cr
#'  \link[paws.compute:apprunner_tag_resource]{tag_resource} \tab Add tags to, or update the tag values of, an App Runner resource\cr
#'  \link[paws.compute:apprunner_untag_resource]{untag_resource} \tab Remove tags from an App Runner resource\cr
#'  \link[paws.compute:apprunner_update_service]{update_service} \tab Update an App Runner service\cr
#'  \link[paws.compute:apprunner_update_vpc_ingress_connection]{update_vpc_ingress_connection} \tab Update an existing App Runner VPC Ingress Connection resource
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname apprunner
#' @export
apprunner <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.compute::apprunner(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' AWS Batch
#'
#' @description
#' Batch
#'
#' Using Batch, you can run batch computing workloads on the Amazon Web
#' Services Cloud. Batch computing is a common means for developers,
#' scientists, and engineers to access large amounts of compute resources.
#' Batch uses the advantages of the batch computing to remove the
#' undifferentiated heavy lifting of configuring and managing required
#' infrastructure. At the same time, it also adopts a familiar batch
#' computing software approach. You can use Batch to efficiently provision
#' resources d, and work toward eliminating capacity constraints, reducing
#' your overall compute costs, and delivering results more quickly.
#'
#' As a fully managed service, Batch can run batch computing workloads of
#' any scale. Batch automatically provisions compute resources and
#' optimizes workload distribution based on the quantity and scale of your
#' specific workloads. With Batch, there's no need to install or manage
#' batch computing software. This means that you can focus on analyzing
#' results and solving your specific problems instead.
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- batch(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- batch()
#' # This example cancels a job with the specified job ID.
#' svc$cancel_job(
#'   jobId = "1d828f65-7a4d-42e8-996d-3b900ed59dc4",
#'   reason = "Cancelling job."
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.compute:batch_cancel_job]{cancel_job} \tab Cancels a job in an Batch job queue\cr
#'  \link[paws.compute:batch_create_compute_environment]{create_compute_environment} \tab Creates an Batch compute environment\cr
#'  \link[paws.compute:batch_create_job_queue]{create_job_queue} \tab Creates an Batch job queue\cr
#'  \link[paws.compute:batch_create_scheduling_policy]{create_scheduling_policy} \tab Creates an Batch scheduling policy\cr
#'  \link[paws.compute:batch_delete_compute_environment]{delete_compute_environment} \tab Deletes an Batch compute environment\cr
#'  \link[paws.compute:batch_delete_job_queue]{delete_job_queue} \tab Deletes the specified job queue\cr
#'  \link[paws.compute:batch_delete_scheduling_policy]{delete_scheduling_policy} \tab Deletes the specified scheduling policy\cr
#'  \link[paws.compute:batch_deregister_job_definition]{deregister_job_definition} \tab Deregisters an Batch job definition\cr
#'  \link[paws.compute:batch_describe_compute_environments]{describe_compute_environments} \tab Describes one or more of your compute environments\cr
#'  \link[paws.compute:batch_describe_job_definitions]{describe_job_definitions} \tab Describes a list of job definitions\cr
#'  \link[paws.compute:batch_describe_job_queues]{describe_job_queues} \tab Describes one or more of your job queues\cr
#'  \link[paws.compute:batch_describe_jobs]{describe_jobs} \tab Describes a list of Batch jobs\cr
#'  \link[paws.compute:batch_describe_scheduling_policies]{describe_scheduling_policies} \tab Describes one or more of your scheduling policies\cr
#'  \link[paws.compute:batch_list_jobs]{list_jobs} \tab Returns a list of Batch jobs\cr
#'  \link[paws.compute:batch_list_scheduling_policies]{list_scheduling_policies} \tab Returns a list of Batch scheduling policies\cr
#'  \link[paws.compute:batch_list_tags_for_resource]{list_tags_for_resource} \tab Lists the tags for an Batch resource\cr
#'  \link[paws.compute:batch_register_job_definition]{register_job_definition} \tab Registers an Batch job definition\cr
#'  \link[paws.compute:batch_submit_job]{submit_job} \tab Submits an Batch job from a job definition\cr
#'  \link[paws.compute:batch_tag_resource]{tag_resource} \tab Associates the specified tags to a resource with the specified resourceArn\cr
#'  \link[paws.compute:batch_terminate_job]{terminate_job} \tab Terminates a job in a job queue\cr
#'  \link[paws.compute:batch_untag_resource]{untag_resource} \tab Deletes specified tags from an Batch resource\cr
#'  \link[paws.compute:batch_update_compute_environment]{update_compute_environment} \tab Updates an Batch compute environment\cr
#'  \link[paws.compute:batch_update_job_queue]{update_job_queue} \tab Updates a job queue\cr
#'  \link[paws.compute:batch_update_scheduling_policy]{update_scheduling_policy} \tab Updates a scheduling policy
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname batch
#' @export
batch <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.compute::batch(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' Braket
#'
#' @description
#' The Amazon Braket API Reference provides information about the
#' operations and structures supported in Amazon Braket.
#'
#' Additional Resources:
#'
#' -   [Amazon Braket Developer
#'     Guide](https://docs.aws.amazon.com/braket/latest/developerguide/what-is-braket.html)
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- braket(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- braket()
#' svc$cancel_job(
#'   Foo = 123
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.compute:braket_cancel_job]{cancel_job} \tab Cancels an Amazon Braket job\cr
#'  \link[paws.compute:braket_cancel_quantum_task]{cancel_quantum_task} \tab Cancels the specified task\cr
#'  \link[paws.compute:braket_create_job]{create_job} \tab Creates an Amazon Braket job\cr
#'  \link[paws.compute:braket_create_quantum_task]{create_quantum_task} \tab Creates a quantum task\cr
#'  \link[paws.compute:braket_get_device]{get_device} \tab Retrieves the devices available in Amazon Braket\cr
#'  \link[paws.compute:braket_get_job]{get_job} \tab Retrieves the specified Amazon Braket job\cr
#'  \link[paws.compute:braket_get_quantum_task]{get_quantum_task} \tab Retrieves the specified quantum task\cr
#'  \link[paws.compute:braket_list_tags_for_resource]{list_tags_for_resource} \tab Shows the tags associated with this resource\cr
#'  \link[paws.compute:braket_search_devices]{search_devices} \tab Searches for devices using the specified filters\cr
#'  \link[paws.compute:braket_search_jobs]{search_jobs} \tab Searches for Amazon Braket jobs that match the specified filter values\cr
#'  \link[paws.compute:braket_search_quantum_tasks]{search_quantum_tasks} \tab Searches for tasks that match the specified filter values\cr
#'  \link[paws.compute:braket_tag_resource]{tag_resource} \tab Add a tag to the specified resource\cr
#'  \link[paws.compute:braket_untag_resource]{untag_resource} \tab Remove tags from a resource
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname braket
#' @export
braket <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.compute::braket(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' AWS Compute Optimizer
#'
#' @description
#' Compute Optimizer is a service that analyzes the configuration and
#' utilization metrics of your Amazon Web Services compute resources, such
#' as Amazon EC2 instances, Amazon EC2 Auto Scaling groups, Lambda
#' functions, Amazon EBS volumes, and Amazon ECS services on Fargate. It
#' reports whether your resources are optimal, and generates optimization
#' recommendations to reduce the cost and improve the performance of your
#' workloads. Compute Optimizer also provides recent utilization metric
#' data, in addition to projected utilization metric data for the
#' recommendations, which you can use to evaluate which recommendation
#' provides the best price-performance trade-off. The analysis of your
#' usage patterns can help you decide when to move or resize your running
#' resources, and still meet your performance and capacity requirements.
#' For more information about Compute Optimizer, including the required
#' permissions to use the service, see the [Compute Optimizer User
#' Guide](https://docs.aws.amazon.com/compute-optimizer/latest/ug/).
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- computeoptimizer(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- computeoptimizer()
#' svc$delete_recommendation_preferences(
#'   Foo = 123
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.compute:computeoptimizer_delete_recommendation_preferences]{delete_recommendation_preferences} \tab Deletes a recommendation preference, such as enhanced infrastructure metrics\cr
#'  \link[paws.compute:computeoptimizer_describe_recommendation_export_jobs]{describe_recommendation_export_jobs} \tab Describes recommendation export jobs created in the last seven days\cr
#'  \link[paws.compute:computeoptimizer_export_auto_scaling_group_recommendations]{export_auto_scaling_group_recommendations} \tab Exports optimization recommendations for Auto Scaling groups\cr
#'  \link[paws.compute:computeoptimizer_export_ebs_volume_recommendations]{export_ebs_volume_recommendations} \tab Exports optimization recommendations for Amazon EBS volumes\cr
#'  \link[paws.compute:computeoptimizer_export_ec2_instance_recommendations]{export_ec2_instance_recommendations} \tab Exports optimization recommendations for Amazon EC2 instances\cr
#'  \link[paws.compute:computeoptimizer_export_ecs_service_recommendations]{export_ecs_service_recommendations} \tab Exports optimization recommendations for Amazon ECS services on Fargate\cr
#'  \link[paws.compute:computeoptimizer_export_lambda_function_recommendations]{export_lambda_function_recommendations} \tab Exports optimization recommendations for Lambda functions\cr
#'  \link[paws.compute:computeoptimizer_export_license_recommendations]{export_license_recommendations} \tab Export optimization recommendations for your licenses\cr
#'  \link[paws.compute:computeoptimizer_get_auto_scaling_group_recommendations]{get_auto_scaling_group_recommendations} \tab Returns Auto Scaling group recommendations\cr
#'  \link[paws.compute:computeoptimizer_get_ebs_volume_recommendations]{get_ebs_volume_recommendations} \tab Returns Amazon Elastic Block Store (Amazon EBS) volume recommendations\cr
#'  \link[paws.compute:computeoptimizer_get_ec2_instance_recommendations]{get_ec2_instance_recommendations} \tab Returns Amazon EC2 instance recommendations\cr
#'  \link[paws.compute:computeoptimizer_get_ec2_recommendation_projected_metrics]{get_ec2_recommendation_projected_metrics} \tab Returns the projected utilization metrics of Amazon EC2 instance recommendations\cr
#'  \link[paws.compute:computeoptimizer_get_ecs_service_recommendation_projected_metrics]{get_ecs_service_recommendation_projected_metrics} \tab Returns the projected metrics of Amazon ECS service recommendations\cr
#'  \link[paws.compute:computeoptimizer_get_ecs_service_recommendations]{get_ecs_service_recommendations} \tab Returns Amazon ECS service recommendations\cr
#'  \link[paws.compute:computeoptimizer_get_effective_recommendation_preferences]{get_effective_recommendation_preferences} \tab Returns the recommendation preferences that are in effect for a given resource, such as enhanced infrastructure metrics\cr
#'  \link[paws.compute:computeoptimizer_get_enrollment_status]{get_enrollment_status} \tab Returns the enrollment (opt in) status of an account to the Compute Optimizer service\cr
#'  \link[paws.compute:computeoptimizer_get_enrollment_statuses_for_organization]{get_enrollment_statuses_for_organization} \tab Returns the Compute Optimizer enrollment (opt-in) status of organization member accounts, if your account is an organization management account\cr
#'  \link[paws.compute:computeoptimizer_get_lambda_function_recommendations]{get_lambda_function_recommendations} \tab Returns Lambda function recommendations\cr
#'  \link[paws.compute:computeoptimizer_get_license_recommendations]{get_license_recommendations} \tab Returns license recommendations for Amazon EC2 instances that run on a specific license\cr
#'  \link[paws.compute:computeoptimizer_get_recommendation_preferences]{get_recommendation_preferences} \tab Returns existing recommendation preferences, such as enhanced infrastructure metrics\cr
#'  \link[paws.compute:computeoptimizer_get_recommendation_summaries]{get_recommendation_summaries} \tab Returns the optimization findings for an account\cr
#'  \link[paws.compute:computeoptimizer_put_recommendation_preferences]{put_recommendation_preferences} \tab Creates a new recommendation preference or updates an existing recommendation preference, such as enhanced infrastructure metrics\cr
#'  \link[paws.compute:computeoptimizer_update_enrollment_status]{update_enrollment_status} \tab Updates the enrollment (opt in and opt out) status of an account to the Compute Optimizer service
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname computeoptimizer
#' @export
computeoptimizer <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.compute::computeoptimizer(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' Amazon Elastic Compute Cloud
#'
#' @description
#' Amazon Elastic Compute Cloud (Amazon EC2) provides secure and resizable
#' computing capacity in the Amazon Web Services Cloud. Using Amazon EC2
#' eliminates the need to invest in hardware up front, so you can develop
#' and deploy applications faster. Amazon Virtual Private Cloud (Amazon
#' VPC) enables you to provision a logically isolated section of the Amazon
#' Web Services Cloud where you can launch Amazon Web Services resources in
#' a virtual network that you've defined. Amazon Elastic Block Store
#' (Amazon EBS) provides block level storage volumes for use with EC2
#' instances. EBS volumes are highly available and reliable storage volumes
#' that can be attached to any running instance and used like a hard drive.
#'
#' To learn more, see the following resources:
#'
#' -   Amazon EC2: [Amazon EC2 product page](https://aws.amazon.com/ec2/),
#'     [Amazon EC2 documentation](https://docs.aws.amazon.com/ec2/)
#'
#' -   Amazon EBS: [Amazon EBS product page](https://aws.amazon.com/ebs/),
#'     [Amazon EBS documentation](https://docs.aws.amazon.com/ebs/)
#'
#' -   Amazon VPC: [Amazon VPC product page](https://aws.amazon.com/vpc/),
#'     [Amazon VPC documentation](https://docs.aws.amazon.com/vpc/)
#'
#' -   VPN: [VPN product page](https://aws.amazon.com/vpn/), [VPN
#'     documentation](https://docs.aws.amazon.com/vpn/)
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- ec2(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- ec2()
#' # This example allocates an Elastic IP address.
#' svc$allocate_address()
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.compute:ec2_accept_address_transfer]{accept_address_transfer} \tab Accepts an Elastic IP address transfer\cr
#'  \link[paws.compute:ec2_accept_reserved_instances_exchange_quote]{accept_reserved_instances_exchange_quote} \tab Accepts the Convertible Reserved Instance exchange quote described in the GetReservedInstancesExchangeQuote call\cr
#'  \link[paws.compute:ec2_accept_transit_gateway_multicast_domain_associations]{accept_transit_gateway_multicast_domain_associations} \tab Accepts a request to associate subnets with a transit gateway multicast domain\cr
#'  \link[paws.compute:ec2_accept_transit_gateway_peering_attachment]{accept_transit_gateway_peering_attachment} \tab Accepts a transit gateway peering attachment request\cr
#'  \link[paws.compute:ec2_accept_transit_gateway_vpc_attachment]{accept_transit_gateway_vpc_attachment} \tab Accepts a request to attach a VPC to a transit gateway\cr
#'  \link[paws.compute:ec2_accept_vpc_endpoint_connections]{accept_vpc_endpoint_connections} \tab Accepts connection requests to your VPC endpoint service\cr
#'  \link[paws.compute:ec2_accept_vpc_peering_connection]{accept_vpc_peering_connection} \tab Accept a VPC peering connection request\cr
#'  \link[paws.compute:ec2_advertise_byoip_cidr]{advertise_byoip_cidr} \tab Advertises an IPv4 or IPv6 address range that is provisioned for use with your Amazon Web Services resources through bring your own IP addresses (BYOIP)\cr
#'  \link[paws.compute:ec2_allocate_address]{allocate_address} \tab Allocates an Elastic IP address to your Amazon Web Services account\cr
#'  \link[paws.compute:ec2_allocate_hosts]{allocate_hosts} \tab Allocates a Dedicated Host to your account\cr
#'  \link[paws.compute:ec2_allocate_ipam_pool_cidr]{allocate_ipam_pool_cidr} \tab Allocate a CIDR from an IPAM pool\cr
#'  \link[paws.compute:ec2_apply_security_groups_to_client_vpn_target_network]{apply_security_groups_to_client_vpn_target_network} \tab Applies a security group to the association between the target network and the Client VPN endpoint\cr
#'  \link[paws.compute:ec2_assign_ipv_6_addresses]{assign_ipv_6_addresses} \tab Assigns one or more IPv6 addresses to the specified network interface\cr
#'  \link[paws.compute:ec2_assign_private_ip_addresses]{assign_private_ip_addresses} \tab Assigns one or more secondary private IP addresses to the specified network interface\cr
#'  \link[paws.compute:ec2_assign_private_nat_gateway_address]{assign_private_nat_gateway_address} \tab Assigns one or more private IPv4 addresses to a private NAT gateway\cr
#'  \link[paws.compute:ec2_associate_address]{associate_address} \tab Associates an Elastic IP address, or carrier IP address (for instances that are in subnets in Wavelength Zones) with an instance or a network interface\cr
#'  \link[paws.compute:ec2_associate_client_vpn_target_network]{associate_client_vpn_target_network} \tab Associates a target network with a Client VPN endpoint\cr
#'  \link[paws.compute:ec2_associate_dhcp_options]{associate_dhcp_options} \tab Associates a set of DHCP options (that you've previously created) with the specified VPC, or associates no DHCP options with the VPC\cr
#'  \link[paws.compute:ec2_associate_enclave_certificate_iam_role]{associate_enclave_certificate_iam_role} \tab Associates an Identity and Access Management (IAM) role with an Certificate Manager (ACM) certificate\cr
#'  \link[paws.compute:ec2_associate_iam_instance_profile]{associate_iam_instance_profile} \tab Associates an IAM instance profile with a running or stopped instance\cr
#'  \link[paws.compute:ec2_associate_instance_event_window]{associate_instance_event_window} \tab Associates one or more targets with an event window\cr
#'  \link[paws.compute:ec2_associate_ipam_resource_discovery]{associate_ipam_resource_discovery} \tab Associates an IPAM resource discovery with an Amazon VPC IPAM\cr
#'  \link[paws.compute:ec2_associate_nat_gateway_address]{associate_nat_gateway_address} \tab Associates Elastic IP addresses (EIPs) and private IPv4 addresses with a public NAT gateway\cr
#'  \link[paws.compute:ec2_associate_route_table]{associate_route_table} \tab Associates a subnet in your VPC or an internet gateway or virtual private gateway attached to your VPC with a route table in your VPC\cr
#'  \link[paws.compute:ec2_associate_subnet_cidr_block]{associate_subnet_cidr_block} \tab Associates a CIDR block with your subnet\cr
#'  \link[paws.compute:ec2_associate_transit_gateway_multicast_domain]{associate_transit_gateway_multicast_domain} \tab Associates the specified subnets and transit gateway attachments with the specified transit gateway multicast domain\cr
#'  \link[paws.compute:ec2_associate_transit_gateway_policy_table]{associate_transit_gateway_policy_table} \tab Associates the specified transit gateway attachment with a transit gateway policy table\cr
#'  \link[paws.compute:ec2_associate_transit_gateway_route_table]{associate_transit_gateway_route_table} \tab Associates the specified attachment with the specified transit gateway route table\cr
#'  \link[paws.compute:ec2_associate_trunk_interface]{associate_trunk_interface} \tab This API action is currently in limited preview only\cr
#'  \link[paws.compute:ec2_associate_vpc_cidr_block]{associate_vpc_cidr_block} \tab Associates a CIDR block with your VPC\cr
#'  \link[paws.compute:ec2_attach_classic_link_vpc]{attach_classic_link_vpc} \tab This action is deprecated\cr
#'  \link[paws.compute:ec2_attach_internet_gateway]{attach_internet_gateway} \tab Attaches an internet gateway or a virtual private gateway to a VPC, enabling connectivity between the internet and the VPC\cr
#'  \link[paws.compute:ec2_attach_network_interface]{attach_network_interface} \tab Attaches a network interface to an instance\cr
#'  \link[paws.compute:ec2_attach_verified_access_trust_provider]{attach_verified_access_trust_provider} \tab Attaches the specified Amazon Web Services Verified Access trust provider to the specified Amazon Web Services Verified Access instance\cr
#'  \link[paws.compute:ec2_attach_volume]{attach_volume} \tab Attaches an EBS volume to a running or stopped instance and exposes it to the instance with the specified device name\cr
#'  \link[paws.compute:ec2_attach_vpn_gateway]{attach_vpn_gateway} \tab Attaches a virtual private gateway to a VPC\cr
#'  \link[paws.compute:ec2_authorize_client_vpn_ingress]{authorize_client_vpn_ingress} \tab Adds an ingress authorization rule to a Client VPN endpoint\cr
#'  \link[paws.compute:ec2_authorize_security_group_egress]{authorize_security_group_egress} \tab Adds the specified outbound (egress) rules to a security group for use with a VPC\cr
#'  \link[paws.compute:ec2_authorize_security_group_ingress]{authorize_security_group_ingress} \tab Adds the specified inbound (ingress) rules to a security group\cr
#'  \link[paws.compute:ec2_bundle_instance]{bundle_instance} \tab Bundles an Amazon instance store-backed Windows instance\cr
#'  \link[paws.compute:ec2_cancel_bundle_task]{cancel_bundle_task} \tab Cancels a bundling operation for an instance store-backed Windows instance\cr
#'  \link[paws.compute:ec2_cancel_capacity_reservation]{cancel_capacity_reservation} \tab Cancels the specified Capacity Reservation, releases the reserved capacity, and changes the Capacity Reservation's state to cancelled\cr
#'  \link[paws.compute:ec2_cancel_capacity_reservation_fleets]{cancel_capacity_reservation_fleets} \tab Cancels one or more Capacity Reservation Fleets\cr
#'  \link[paws.compute:ec2_cancel_conversion_task]{cancel_conversion_task} \tab Cancels an active conversion task\cr
#'  \link[paws.compute:ec2_cancel_export_task]{cancel_export_task} \tab Cancels an active export task\cr
#'  \link[paws.compute:ec2_cancel_image_launch_permission]{cancel_image_launch_permission} \tab Removes your Amazon Web Services account from the launch permissions for the specified AMI\cr
#'  \link[paws.compute:ec2_cancel_import_task]{cancel_import_task} \tab Cancels an in-process import virtual machine or import snapshot task\cr
#'  \link[paws.compute:ec2_cancel_reserved_instances_listing]{cancel_reserved_instances_listing} \tab Cancels the specified Reserved Instance listing in the Reserved Instance Marketplace\cr
#'  \link[paws.compute:ec2_cancel_spot_fleet_requests]{cancel_spot_fleet_requests} \tab Cancels the specified Spot Fleet requests\cr
#'  \link[paws.compute:ec2_cancel_spot_instance_requests]{cancel_spot_instance_requests} \tab Cancels one or more Spot Instance requests\cr
#'  \link[paws.compute:ec2_confirm_product_instance]{confirm_product_instance} \tab Determines whether a product code is associated with an instance\cr
#'  \link[paws.compute:ec2_copy_fpga_image]{copy_fpga_image} \tab Copies the specified Amazon FPGA Image (AFI) to the current Region\cr
#'  \link[paws.compute:ec2_copy_image]{copy_image} \tab Initiates the copy of an AMI\cr
#'  \link[paws.compute:ec2_copy_snapshot]{copy_snapshot} \tab Copies a point-in-time snapshot of an EBS volume and stores it in Amazon S3\cr
#'  \link[paws.compute:ec2_create_capacity_reservation]{create_capacity_reservation} \tab Creates a new Capacity Reservation with the specified attributes\cr
#'  \link[paws.compute:ec2_create_capacity_reservation_fleet]{create_capacity_reservation_fleet} \tab Creates a Capacity Reservation Fleet\cr
#'  \link[paws.compute:ec2_create_carrier_gateway]{create_carrier_gateway} \tab Creates a carrier gateway\cr
#'  \link[paws.compute:ec2_create_client_vpn_endpoint]{create_client_vpn_endpoint} \tab Creates a Client VPN endpoint\cr
#'  \link[paws.compute:ec2_create_client_vpn_route]{create_client_vpn_route} \tab Adds a route to a network to a Client VPN endpoint\cr
#'  \link[paws.compute:ec2_create_coip_cidr]{create_coip_cidr} \tab Creates a range of customer-owned IP addresses\cr
#'  \link[paws.compute:ec2_create_coip_pool]{create_coip_pool} \tab Creates a pool of customer-owned IP (CoIP) addresses\cr
#'  \link[paws.compute:ec2_create_customer_gateway]{create_customer_gateway} \tab Provides information to Amazon Web Services about your customer gateway device\cr
#'  \link[paws.compute:ec2_create_default_subnet]{create_default_subnet} \tab Creates a default subnet with a size /20 IPv4 CIDR block in the specified Availability Zone in your default VPC\cr
#'  \link[paws.compute:ec2_create_default_vpc]{create_default_vpc} \tab Creates a default VPC with a size /16 IPv4 CIDR block and a default subnet in each Availability Zone\cr
#'  \link[paws.compute:ec2_create_dhcp_options]{create_dhcp_options} \tab Creates a set of DHCP options for your VPC\cr
#'  \link[paws.compute:ec2_create_egress_only_internet_gateway]{create_egress_only_internet_gateway} \tab &#91;IPv6 only&#93; Creates an egress-only internet gateway for your VPC\cr
#'  \link[paws.compute:ec2_create_fleet]{create_fleet} \tab Creates an EC2 Fleet that contains the configuration information for On-Demand Instances and Spot Instances\cr
#'  \link[paws.compute:ec2_create_flow_logs]{create_flow_logs} \tab Creates one or more flow logs to capture information about IP traffic for a specific network interface, subnet, or VPC\cr
#'  \link[paws.compute:ec2_create_fpga_image]{create_fpga_image} \tab Creates an Amazon FPGA Image (AFI) from the specified design checkpoint (DCP)\cr
#'  \link[paws.compute:ec2_create_image]{create_image} \tab Creates an Amazon EBS-backed AMI from an Amazon EBS-backed instance that is either running or stopped\cr
#'  \link[paws.compute:ec2_create_instance_connect_endpoint]{create_instance_connect_endpoint} \tab Creates an EC2 Instance Connect Endpoint\cr
#'  \link[paws.compute:ec2_create_instance_event_window]{create_instance_event_window} \tab Creates an event window in which scheduled events for the associated Amazon EC2 instances can run\cr
#'  \link[paws.compute:ec2_create_instance_export_task]{create_instance_export_task} \tab Exports a running or stopped instance to an Amazon S3 bucket\cr
#'  \link[paws.compute:ec2_create_internet_gateway]{create_internet_gateway} \tab Creates an internet gateway for use with a VPC\cr
#'  \link[paws.compute:ec2_create_ipam]{create_ipam} \tab Create an IPAM\cr
#'  \link[paws.compute:ec2_create_ipam_pool]{create_ipam_pool} \tab Create an IP address pool for Amazon VPC IP Address Manager (IPAM)\cr
#'  \link[paws.compute:ec2_create_ipam_resource_discovery]{create_ipam_resource_discovery} \tab Creates an IPAM resource discovery\cr
#'  \link[paws.compute:ec2_create_ipam_scope]{create_ipam_scope} \tab Create an IPAM scope\cr
#'  \link[paws.compute:ec2_create_key_pair]{create_key_pair} \tab Creates an ED25519 or 2048-bit RSA key pair with the specified name and in the specified PEM or PPK format\cr
#'  \link[paws.compute:ec2_create_launch_template]{create_launch_template} \tab Creates a launch template\cr
#'  \link[paws.compute:ec2_create_launch_template_version]{create_launch_template_version} \tab Creates a new version of a launch template\cr
#'  \link[paws.compute:ec2_create_local_gateway_route]{create_local_gateway_route} \tab Creates a static route for the specified local gateway route table\cr
#'  \link[paws.compute:ec2_create_local_gateway_route_table]{create_local_gateway_route_table} \tab Creates a local gateway route table\cr
#'  \link[paws.compute:ec2_create_local_gateway_route_table_virtual_interface_group_association]{create_local_gateway_route_table_virtual_interface_group_association} \tab Creates a local gateway route table virtual interface group association\cr
#'  \link[paws.compute:ec2_create_local_gateway_route_table_vpc_association]{create_local_gateway_route_table_vpc_association} \tab Associates the specified VPC with the specified local gateway route table\cr
#'  \link[paws.compute:ec2_create_managed_prefix_list]{create_managed_prefix_list} \tab Creates a managed prefix list\cr
#'  \link[paws.compute:ec2_create_nat_gateway]{create_nat_gateway} \tab Creates a NAT gateway in the specified subnet\cr
#'  \link[paws.compute:ec2_create_network_acl]{create_network_acl} \tab Creates a network ACL in a VPC\cr
#'  \link[paws.compute:ec2_create_network_acl_entry]{create_network_acl_entry} \tab Creates an entry (a rule) in a network ACL with the specified rule number\cr
#'  \link[paws.compute:ec2_create_network_insights_access_scope]{create_network_insights_access_scope} \tab Creates a Network Access Scope\cr
#'  \link[paws.compute:ec2_create_network_insights_path]{create_network_insights_path} \tab Creates a path to analyze for reachability\cr
#'  \link[paws.compute:ec2_create_network_interface]{create_network_interface} \tab Creates a network interface in the specified subnet\cr
#'  \link[paws.compute:ec2_create_network_interface_permission]{create_network_interface_permission} \tab Grants an Amazon Web Services-authorized account permission to attach the specified network interface to an instance in their account\cr
#'  \link[paws.compute:ec2_create_placement_group]{create_placement_group} \tab Creates a placement group in which to launch instances\cr
#'  \link[paws.compute:ec2_create_public_ipv_4_pool]{create_public_ipv_4_pool} \tab Creates a public IPv4 address pool\cr
#'  \link[paws.compute:ec2_create_replace_root_volume_task]{create_replace_root_volume_task} \tab Replaces the EBS-backed root volume for a running instance with a new volume that is restored to the original root volume's launch state, that is restored to a specific snapshot taken from the original root volume, or that is restored from an AMI that has the same key characteristics as that of the instance\cr
#'  \link[paws.compute:ec2_create_reserved_instances_listing]{create_reserved_instances_listing} \tab Creates a listing for Amazon EC2 Standard Reserved Instances to be sold in the Reserved Instance Marketplace\cr
#'  \link[paws.compute:ec2_create_restore_image_task]{create_restore_image_task} \tab Starts a task that restores an AMI from an Amazon S3 object that was previously created by using CreateStoreImageTask\cr
#'  \link[paws.compute:ec2_create_route]{create_route} \tab Creates a route in a route table within a VPC\cr
#'  \link[paws.compute:ec2_create_route_table]{create_route_table} \tab Creates a route table for the specified VPC\cr
#'  \link[paws.compute:ec2_create_security_group]{create_security_group} \tab Creates a security group\cr
#'  \link[paws.compute:ec2_create_snapshot]{create_snapshot} \tab Creates a snapshot of an EBS volume and stores it in Amazon S3\cr
#'  \link[paws.compute:ec2_create_snapshots]{create_snapshots} \tab Creates crash-consistent snapshots of multiple EBS volumes and stores the data in S3\cr
#'  \link[paws.compute:ec2_create_spot_datafeed_subscription]{create_spot_datafeed_subscription} \tab Creates a data feed for Spot Instances, enabling you to view Spot Instance usage logs\cr
#'  \link[paws.compute:ec2_create_store_image_task]{create_store_image_task} \tab Stores an AMI as a single object in an Amazon S3 bucket\cr
#'  \link[paws.compute:ec2_create_subnet]{create_subnet} \tab Creates a subnet in the specified VPC\cr
#'  \link[paws.compute:ec2_create_subnet_cidr_reservation]{create_subnet_cidr_reservation} \tab Creates a subnet CIDR reservation\cr
#'  \link[paws.compute:ec2_create_tags]{create_tags} \tab Adds or overwrites only the specified tags for the specified Amazon EC2 resource or resources\cr
#'  \link[paws.compute:ec2_create_traffic_mirror_filter]{create_traffic_mirror_filter} \tab Creates a Traffic Mirror filter\cr
#'  \link[paws.compute:ec2_create_traffic_mirror_filter_rule]{create_traffic_mirror_filter_rule} \tab Creates a Traffic Mirror filter rule\cr
#'  \link[paws.compute:ec2_create_traffic_mirror_session]{create_traffic_mirror_session} \tab Creates a Traffic Mirror session\cr
#'  \link[paws.compute:ec2_create_traffic_mirror_target]{create_traffic_mirror_target} \tab Creates a target for your Traffic Mirror session\cr
#'  \link[paws.compute:ec2_create_transit_gateway]{create_transit_gateway} \tab Creates a transit gateway\cr
#'  \link[paws.compute:ec2_create_transit_gateway_connect]{create_transit_gateway_connect} \tab Creates a Connect attachment from a specified transit gateway attachment\cr
#'  \link[paws.compute:ec2_create_transit_gateway_connect_peer]{create_transit_gateway_connect_peer} \tab Creates a Connect peer for a specified transit gateway Connect attachment between a transit gateway and an appliance\cr
#'  \link[paws.compute:ec2_create_transit_gateway_multicast_domain]{create_transit_gateway_multicast_domain} \tab Creates a multicast domain using the specified transit gateway\cr
#'  \link[paws.compute:ec2_create_transit_gateway_peering_attachment]{create_transit_gateway_peering_attachment} \tab Requests a transit gateway peering attachment between the specified transit gateway (requester) and a peer transit gateway (accepter)\cr
#'  \link[paws.compute:ec2_create_transit_gateway_policy_table]{create_transit_gateway_policy_table} \tab Creates a transit gateway policy table\cr
#'  \link[paws.compute:ec2_create_transit_gateway_prefix_list_reference]{create_transit_gateway_prefix_list_reference} \tab Creates a reference (route) to a prefix list in a specified transit gateway route table\cr
#'  \link[paws.compute:ec2_create_transit_gateway_route]{create_transit_gateway_route} \tab Creates a static route for the specified transit gateway route table\cr
#'  \link[paws.compute:ec2_create_transit_gateway_route_table]{create_transit_gateway_route_table} \tab Creates a route table for the specified transit gateway\cr
#'  \link[paws.compute:ec2_create_transit_gateway_route_table_announcement]{create_transit_gateway_route_table_announcement} \tab Advertises a new transit gateway route table\cr
#'  \link[paws.compute:ec2_create_transit_gateway_vpc_attachment]{create_transit_gateway_vpc_attachment} \tab Attaches the specified VPC to the specified transit gateway\cr
#'  \link[paws.compute:ec2_create_verified_access_endpoint]{create_verified_access_endpoint} \tab An Amazon Web Services Verified Access endpoint is where you define your application along with an optional endpoint-level access policy\cr
#'  \link[paws.compute:ec2_create_verified_access_group]{create_verified_access_group} \tab An Amazon Web Services Verified Access group is a collection of Amazon Web Services Verified Access endpoints who's associated applications have similar security requirements\cr
#'  \link[paws.compute:ec2_create_verified_access_instance]{create_verified_access_instance} \tab An Amazon Web Services Verified Access instance is a regional entity that evaluates application requests and grants access only when your security requirements are met\cr
#'  \link[paws.compute:ec2_create_verified_access_trust_provider]{create_verified_access_trust_provider} \tab A trust provider is a third-party entity that creates, maintains, and manages identity information for users and devices\cr
#'  \link[paws.compute:ec2_create_volume]{create_volume} \tab Creates an EBS volume that can be attached to an instance in the same Availability Zone\cr
#'  \link[paws.compute:ec2_create_vpc]{create_vpc} \tab Creates a VPC with the specified CIDR blocks\cr
#'  \link[paws.compute:ec2_create_vpc_endpoint]{create_vpc_endpoint} \tab Creates a VPC endpoint\cr
#'  \link[paws.compute:ec2_create_vpc_endpoint_connection_notification]{create_vpc_endpoint_connection_notification} \tab Creates a connection notification for a specified VPC endpoint or VPC endpoint service\cr
#'  \link[paws.compute:ec2_create_vpc_endpoint_service_configuration]{create_vpc_endpoint_service_configuration} \tab Creates a VPC endpoint service to which service consumers (Amazon Web Services accounts, users, and IAM roles) can connect\cr
#'  \link[paws.compute:ec2_create_vpc_peering_connection]{create_vpc_peering_connection} \tab Requests a VPC peering connection between two VPCs: a requester VPC that you own and an accepter VPC with which to create the connection\cr
#'  \link[paws.compute:ec2_create_vpn_connection]{create_vpn_connection} \tab Creates a VPN connection between an existing virtual private gateway or transit gateway and a customer gateway\cr
#'  \link[paws.compute:ec2_create_vpn_connection_route]{create_vpn_connection_route} \tab Creates a static route associated with a VPN connection between an existing virtual private gateway and a VPN customer gateway\cr
#'  \link[paws.compute:ec2_create_vpn_gateway]{create_vpn_gateway} \tab Creates a virtual private gateway\cr
#'  \link[paws.compute:ec2_delete_carrier_gateway]{delete_carrier_gateway} \tab Deletes a carrier gateway\cr
#'  \link[paws.compute:ec2_delete_client_vpn_endpoint]{delete_client_vpn_endpoint} \tab Deletes the specified Client VPN endpoint\cr
#'  \link[paws.compute:ec2_delete_client_vpn_route]{delete_client_vpn_route} \tab Deletes a route from a Client VPN endpoint\cr
#'  \link[paws.compute:ec2_delete_coip_cidr]{delete_coip_cidr} \tab Deletes a range of customer-owned IP addresses\cr
#'  \link[paws.compute:ec2_delete_coip_pool]{delete_coip_pool} \tab Deletes a pool of customer-owned IP (CoIP) addresses\cr
#'  \link[paws.compute:ec2_delete_customer_gateway]{delete_customer_gateway} \tab Deletes the specified customer gateway\cr
#'  \link[paws.compute:ec2_delete_dhcp_options]{delete_dhcp_options} \tab Deletes the specified set of DHCP options\cr
#'  \link[paws.compute:ec2_delete_egress_only_internet_gateway]{delete_egress_only_internet_gateway} \tab Deletes an egress-only internet gateway\cr
#'  \link[paws.compute:ec2_delete_fleets]{delete_fleets} \tab Deletes the specified EC2 Fleets\cr
#'  \link[paws.compute:ec2_delete_flow_logs]{delete_flow_logs} \tab Deletes one or more flow logs\cr
#'  \link[paws.compute:ec2_delete_fpga_image]{delete_fpga_image} \tab Deletes the specified Amazon FPGA Image (AFI)\cr
#'  \link[paws.compute:ec2_delete_instance_connect_endpoint]{delete_instance_connect_endpoint} \tab Deletes the specified EC2 Instance Connect Endpoint\cr
#'  \link[paws.compute:ec2_delete_instance_event_window]{delete_instance_event_window} \tab Deletes the specified event window\cr
#'  \link[paws.compute:ec2_delete_internet_gateway]{delete_internet_gateway} \tab Deletes the specified internet gateway\cr
#'  \link[paws.compute:ec2_delete_ipam]{delete_ipam} \tab Delete an IPAM\cr
#'  \link[paws.compute:ec2_delete_ipam_pool]{delete_ipam_pool} \tab Delete an IPAM pool\cr
#'  \link[paws.compute:ec2_delete_ipam_resource_discovery]{delete_ipam_resource_discovery} \tab Deletes an IPAM resource discovery\cr
#'  \link[paws.compute:ec2_delete_ipam_scope]{delete_ipam_scope} \tab Delete the scope for an IPAM\cr
#'  \link[paws.compute:ec2_delete_key_pair]{delete_key_pair} \tab Deletes the specified key pair, by removing the public key from Amazon EC2\cr
#'  \link[paws.compute:ec2_delete_launch_template]{delete_launch_template} \tab Deletes a launch template\cr
#'  \link[paws.compute:ec2_delete_launch_template_versions]{delete_launch_template_versions} \tab Deletes one or more versions of a launch template\cr
#'  \link[paws.compute:ec2_delete_local_gateway_route]{delete_local_gateway_route} \tab Deletes the specified route from the specified local gateway route table\cr
#'  \link[paws.compute:ec2_delete_local_gateway_route_table]{delete_local_gateway_route_table} \tab Deletes a local gateway route table\cr
#'  \link[paws.compute:ec2_delete_local_gateway_route_table_virtual_interface_group_association]{delete_local_gateway_route_table_virtual_interface_group_association} \tab Deletes a local gateway route table virtual interface group association\cr
#'  \link[paws.compute:ec2_delete_local_gateway_route_table_vpc_association]{delete_local_gateway_route_table_vpc_association} \tab Deletes the specified association between a VPC and local gateway route table\cr
#'  \link[paws.compute:ec2_delete_managed_prefix_list]{delete_managed_prefix_list} \tab Deletes the specified managed prefix list\cr
#'  \link[paws.compute:ec2_delete_nat_gateway]{delete_nat_gateway} \tab Deletes the specified NAT gateway\cr
#'  \link[paws.compute:ec2_delete_network_acl]{delete_network_acl} \tab Deletes the specified network ACL\cr
#'  \link[paws.compute:ec2_delete_network_acl_entry]{delete_network_acl_entry} \tab Deletes the specified ingress or egress entry (rule) from the specified network ACL\cr
#'  \link[paws.compute:ec2_delete_network_insights_access_scope]{delete_network_insights_access_scope} \tab Deletes the specified Network Access Scope\cr
#'  \link[paws.compute:ec2_delete_network_insights_access_scope_analysis]{delete_network_insights_access_scope_analysis} \tab Deletes the specified Network Access Scope analysis\cr
#'  \link[paws.compute:ec2_delete_network_insights_analysis]{delete_network_insights_analysis} \tab Deletes the specified network insights analysis\cr
#'  \link[paws.compute:ec2_delete_network_insights_path]{delete_network_insights_path} \tab Deletes the specified path\cr
#'  \link[paws.compute:ec2_delete_network_interface]{delete_network_interface} \tab Deletes the specified network interface\cr
#'  \link[paws.compute:ec2_delete_network_interface_permission]{delete_network_interface_permission} \tab Deletes a permission for a network interface\cr
#'  \link[paws.compute:ec2_delete_placement_group]{delete_placement_group} \tab Deletes the specified placement group\cr
#'  \link[paws.compute:ec2_delete_public_ipv_4_pool]{delete_public_ipv_4_pool} \tab Delete a public IPv4 pool\cr
#'  \link[paws.compute:ec2_delete_queued_reserved_instances]{delete_queued_reserved_instances} \tab Deletes the queued purchases for the specified Reserved Instances\cr
#'  \link[paws.compute:ec2_delete_route]{delete_route} \tab Deletes the specified route from the specified route table\cr
#'  \link[paws.compute:ec2_delete_route_table]{delete_route_table} \tab Deletes the specified route table\cr
#'  \link[paws.compute:ec2_delete_security_group]{delete_security_group} \tab Deletes a security group\cr
#'  \link[paws.compute:ec2_delete_snapshot]{delete_snapshot} \tab Deletes the specified snapshot\cr
#'  \link[paws.compute:ec2_delete_spot_datafeed_subscription]{delete_spot_datafeed_subscription} \tab Deletes the data feed for Spot Instances\cr
#'  \link[paws.compute:ec2_delete_subnet]{delete_subnet} \tab Deletes the specified subnet\cr
#'  \link[paws.compute:ec2_delete_subnet_cidr_reservation]{delete_subnet_cidr_reservation} \tab Deletes a subnet CIDR reservation\cr
#'  \link[paws.compute:ec2_delete_tags]{delete_tags} \tab Deletes the specified set of tags from the specified set of resources\cr
#'  \link[paws.compute:ec2_delete_traffic_mirror_filter]{delete_traffic_mirror_filter} \tab Deletes the specified Traffic Mirror filter\cr
#'  \link[paws.compute:ec2_delete_traffic_mirror_filter_rule]{delete_traffic_mirror_filter_rule} \tab Deletes the specified Traffic Mirror rule\cr
#'  \link[paws.compute:ec2_delete_traffic_mirror_session]{delete_traffic_mirror_session} \tab Deletes the specified Traffic Mirror session\cr
#'  \link[paws.compute:ec2_delete_traffic_mirror_target]{delete_traffic_mirror_target} \tab Deletes the specified Traffic Mirror target\cr
#'  \link[paws.compute:ec2_delete_transit_gateway]{delete_transit_gateway} \tab Deletes the specified transit gateway\cr
#'  \link[paws.compute:ec2_delete_transit_gateway_connect]{delete_transit_gateway_connect} \tab Deletes the specified Connect attachment\cr
#'  \link[paws.compute:ec2_delete_transit_gateway_connect_peer]{delete_transit_gateway_connect_peer} \tab Deletes the specified Connect peer\cr
#'  \link[paws.compute:ec2_delete_transit_gateway_multicast_domain]{delete_transit_gateway_multicast_domain} \tab Deletes the specified transit gateway multicast domain\cr
#'  \link[paws.compute:ec2_delete_transit_gateway_peering_attachment]{delete_transit_gateway_peering_attachment} \tab Deletes a transit gateway peering attachment\cr
#'  \link[paws.compute:ec2_delete_transit_gateway_policy_table]{delete_transit_gateway_policy_table} \tab Deletes the specified transit gateway policy table\cr
#'  \link[paws.compute:ec2_delete_transit_gateway_prefix_list_reference]{delete_transit_gateway_prefix_list_reference} \tab Deletes a reference (route) to a prefix list in a specified transit gateway route table\cr
#'  \link[paws.compute:ec2_delete_transit_gateway_route]{delete_transit_gateway_route} \tab Deletes the specified route from the specified transit gateway route table\cr
#'  \link[paws.compute:ec2_delete_transit_gateway_route_table]{delete_transit_gateway_route_table} \tab Deletes the specified transit gateway route table\cr
#'  \link[paws.compute:ec2_delete_transit_gateway_route_table_announcement]{delete_transit_gateway_route_table_announcement} \tab Advertises to the transit gateway that a transit gateway route table is deleted\cr
#'  \link[paws.compute:ec2_delete_transit_gateway_vpc_attachment]{delete_transit_gateway_vpc_attachment} \tab Deletes the specified VPC attachment\cr
#'  \link[paws.compute:ec2_delete_verified_access_endpoint]{delete_verified_access_endpoint} \tab Delete an Amazon Web Services Verified Access endpoint\cr
#'  \link[paws.compute:ec2_delete_verified_access_group]{delete_verified_access_group} \tab Delete an Amazon Web Services Verified Access group\cr
#'  \link[paws.compute:ec2_delete_verified_access_instance]{delete_verified_access_instance} \tab Delete an Amazon Web Services Verified Access instance\cr
#'  \link[paws.compute:ec2_delete_verified_access_trust_provider]{delete_verified_access_trust_provider} \tab Delete an Amazon Web Services Verified Access trust provider\cr
#'  \link[paws.compute:ec2_delete_volume]{delete_volume} \tab Deletes the specified EBS volume\cr
#'  \link[paws.compute:ec2_delete_vpc]{delete_vpc} \tab Deletes the specified VPC\cr
#'  \link[paws.compute:ec2_delete_vpc_endpoint_connection_notifications]{delete_vpc_endpoint_connection_notifications} \tab Deletes the specified VPC endpoint connection notifications\cr
#'  \link[paws.compute:ec2_delete_vpc_endpoints]{delete_vpc_endpoints} \tab Deletes the specified VPC endpoints\cr
#'  \link[paws.compute:ec2_delete_vpc_endpoint_service_configurations]{delete_vpc_endpoint_service_configurations} \tab Deletes the specified VPC endpoint service configurations\cr
#'  \link[paws.compute:ec2_delete_vpc_peering_connection]{delete_vpc_peering_connection} \tab Deletes a VPC peering connection\cr
#'  \link[paws.compute:ec2_delete_vpn_connection]{delete_vpn_connection} \tab Deletes the specified VPN connection\cr
#'  \link[paws.compute:ec2_delete_vpn_connection_route]{delete_vpn_connection_route} \tab Deletes the specified static route associated with a VPN connection between an existing virtual private gateway and a VPN customer gateway\cr
#'  \link[paws.compute:ec2_delete_vpn_gateway]{delete_vpn_gateway} \tab Deletes the specified virtual private gateway\cr
#'  \link[paws.compute:ec2_deprovision_byoip_cidr]{deprovision_byoip_cidr} \tab Releases the specified address range that you provisioned for use with your Amazon Web Services resources through bring your own IP addresses (BYOIP) and deletes the corresponding address pool\cr
#'  \link[paws.compute:ec2_deprovision_ipam_pool_cidr]{deprovision_ipam_pool_cidr} \tab Deprovision a CIDR provisioned from an IPAM pool\cr
#'  \link[paws.compute:ec2_deprovision_public_ipv_4_pool_cidr]{deprovision_public_ipv_4_pool_cidr} \tab Deprovision a CIDR from a public IPv4 pool\cr
#'  \link[paws.compute:ec2_deregister_image]{deregister_image} \tab Deregisters the specified AMI\cr
#'  \link[paws.compute:ec2_deregister_instance_event_notification_attributes]{deregister_instance_event_notification_attributes} \tab Deregisters tag keys to prevent tags that have the specified tag keys from being included in scheduled event notifications for resources in the Region\cr
#'  \link[paws.compute:ec2_deregister_transit_gateway_multicast_group_members]{deregister_transit_gateway_multicast_group_members} \tab Deregisters the specified members (network interfaces) from the transit gateway multicast group\cr
#'  \link[paws.compute:ec2_deregister_transit_gateway_multicast_group_sources]{deregister_transit_gateway_multicast_group_sources} \tab Deregisters the specified sources (network interfaces) from the transit gateway multicast group\cr
#'  \link[paws.compute:ec2_describe_account_attributes]{describe_account_attributes} \tab Describes attributes of your Amazon Web Services account\cr
#'  \link[paws.compute:ec2_describe_addresses]{describe_addresses} \tab Describes the specified Elastic IP addresses or all of your Elastic IP addresses\cr
#'  \link[paws.compute:ec2_describe_addresses_attribute]{describe_addresses_attribute} \tab Describes the attributes of the specified Elastic IP addresses\cr
#'  \link[paws.compute:ec2_describe_address_transfers]{describe_address_transfers} \tab Describes an Elastic IP address transfer\cr
#'  \link[paws.compute:ec2_describe_aggregate_id_format]{describe_aggregate_id_format} \tab Describes the longer ID format settings for all resource types in a specific Region\cr
#'  \link[paws.compute:ec2_describe_availability_zones]{describe_availability_zones} \tab Describes the Availability Zones, Local Zones, and Wavelength Zones that are available to you\cr
#'  \link[paws.compute:ec2_describe_aws_network_performance_metric_subscriptions]{describe_aws_network_performance_metric_subscriptions} \tab Describes the current Infrastructure Performance metric subscriptions\cr
#'  \link[paws.compute:ec2_describe_bundle_tasks]{describe_bundle_tasks} \tab Describes the specified bundle tasks or all of your bundle tasks\cr
#'  \link[paws.compute:ec2_describe_byoip_cidrs]{describe_byoip_cidrs} \tab Describes the IP address ranges that were specified in calls to ProvisionByoipCidr\cr
#'  \link[paws.compute:ec2_describe_capacity_reservation_fleets]{describe_capacity_reservation_fleets} \tab Describes one or more Capacity Reservation Fleets\cr
#'  \link[paws.compute:ec2_describe_capacity_reservations]{describe_capacity_reservations} \tab Describes one or more of your Capacity Reservations\cr
#'  \link[paws.compute:ec2_describe_carrier_gateways]{describe_carrier_gateways} \tab Describes one or more of your carrier gateways\cr
#'  \link[paws.compute:ec2_describe_classic_link_instances]{describe_classic_link_instances} \tab This action is deprecated\cr
#'  \link[paws.compute:ec2_describe_client_vpn_authorization_rules]{describe_client_vpn_authorization_rules} \tab Describes the authorization rules for a specified Client VPN endpoint\cr
#'  \link[paws.compute:ec2_describe_client_vpn_connections]{describe_client_vpn_connections} \tab Describes active client connections and connections that have been terminated within the last 60 minutes for the specified Client VPN endpoint\cr
#'  \link[paws.compute:ec2_describe_client_vpn_endpoints]{describe_client_vpn_endpoints} \tab Describes one or more Client VPN endpoints in the account\cr
#'  \link[paws.compute:ec2_describe_client_vpn_routes]{describe_client_vpn_routes} \tab Describes the routes for the specified Client VPN endpoint\cr
#'  \link[paws.compute:ec2_describe_client_vpn_target_networks]{describe_client_vpn_target_networks} \tab Describes the target networks associated with the specified Client VPN endpoint\cr
#'  \link[paws.compute:ec2_describe_coip_pools]{describe_coip_pools} \tab Describes the specified customer-owned address pools or all of your customer-owned address pools\cr
#'  \link[paws.compute:ec2_describe_conversion_tasks]{describe_conversion_tasks} \tab Describes the specified conversion tasks or all your conversion tasks\cr
#'  \link[paws.compute:ec2_describe_customer_gateways]{describe_customer_gateways} \tab Describes one or more of your VPN customer gateways\cr
#'  \link[paws.compute:ec2_describe_dhcp_options]{describe_dhcp_options} \tab Describes one or more of your DHCP options sets\cr
#'  \link[paws.compute:ec2_describe_egress_only_internet_gateways]{describe_egress_only_internet_gateways} \tab Describes one or more of your egress-only internet gateways\cr
#'  \link[paws.compute:ec2_describe_elastic_gpus]{describe_elastic_gpus} \tab Describes the Elastic Graphics accelerator associated with your instances\cr
#'  \link[paws.compute:ec2_describe_export_image_tasks]{describe_export_image_tasks} \tab Describes the specified export image tasks or all of your export image tasks\cr
#'  \link[paws.compute:ec2_describe_export_tasks]{describe_export_tasks} \tab Describes the specified export instance tasks or all of your export instance tasks\cr
#'  \link[paws.compute:ec2_describe_fast_launch_images]{describe_fast_launch_images} \tab Describe details for Windows AMIs that are configured for faster launching\cr
#'  \link[paws.compute:ec2_describe_fast_snapshot_restores]{describe_fast_snapshot_restores} \tab Describes the state of fast snapshot restores for your snapshots\cr
#'  \link[paws.compute:ec2_describe_fleet_history]{describe_fleet_history} \tab Describes the events for the specified EC2 Fleet during the specified time\cr
#'  \link[paws.compute:ec2_describe_fleet_instances]{describe_fleet_instances} \tab Describes the running instances for the specified EC2 Fleet\cr
#'  \link[paws.compute:ec2_describe_fleets]{describe_fleets} \tab Describes the specified EC2 Fleets or all of your EC2 Fleets\cr
#'  \link[paws.compute:ec2_describe_flow_logs]{describe_flow_logs} \tab Describes one or more flow logs\cr
#'  \link[paws.compute:ec2_describe_fpga_image_attribute]{describe_fpga_image_attribute} \tab Describes the specified attribute of the specified Amazon FPGA Image (AFI)\cr
#'  \link[paws.compute:ec2_describe_fpga_images]{describe_fpga_images} \tab Describes the Amazon FPGA Images (AFIs) available to you\cr
#'  \link[paws.compute:ec2_describe_host_reservation_offerings]{describe_host_reservation_offerings} \tab Describes the Dedicated Host reservations that are available to purchase\cr
#'  \link[paws.compute:ec2_describe_host_reservations]{describe_host_reservations} \tab Describes reservations that are associated with Dedicated Hosts in your account\cr
#'  \link[paws.compute:ec2_describe_hosts]{describe_hosts} \tab Describes the specified Dedicated Hosts or all your Dedicated Hosts\cr
#'  \link[paws.compute:ec2_describe_iam_instance_profile_associations]{describe_iam_instance_profile_associations} \tab Describes your IAM instance profile associations\cr
#'  \link[paws.compute:ec2_describe_identity_id_format]{describe_identity_id_format} \tab Describes the ID format settings for resources for the specified IAM user, IAM role, or root user\cr
#'  \link[paws.compute:ec2_describe_id_format]{describe_id_format} \tab Describes the ID format settings for your resources on a per-Region basis, for example, to view which resource types are enabled for longer IDs\cr
#'  \link[paws.compute:ec2_describe_image_attribute]{describe_image_attribute} \tab Describes the specified attribute of the specified AMI\cr
#'  \link[paws.compute:ec2_describe_images]{describe_images} \tab Describes the specified images (AMIs, AKIs, and ARIs) available to you or all of the images available to you\cr
#'  \link[paws.compute:ec2_describe_import_image_tasks]{describe_import_image_tasks} \tab Displays details about an import virtual machine or import snapshot tasks that are already created\cr
#'  \link[paws.compute:ec2_describe_import_snapshot_tasks]{describe_import_snapshot_tasks} \tab Describes your import snapshot tasks\cr
#'  \link[paws.compute:ec2_describe_instance_attribute]{describe_instance_attribute} \tab Describes the specified attribute of the specified instance\cr
#'  \link[paws.compute:ec2_describe_instance_connect_endpoints]{describe_instance_connect_endpoints} \tab Describes the specified EC2 Instance Connect Endpoints or all EC2 Instance Connect Endpoints\cr
#'  \link[paws.compute:ec2_describe_instance_credit_specifications]{describe_instance_credit_specifications} \tab Describes the credit option for CPU usage of the specified burstable performance instances\cr
#'  \link[paws.compute:ec2_describe_instance_event_notification_attributes]{describe_instance_event_notification_attributes} \tab Describes the tag keys that are registered to appear in scheduled event notifications for resources in the current Region\cr
#'  \link[paws.compute:ec2_describe_instance_event_windows]{describe_instance_event_windows} \tab Describes the specified event windows or all event windows\cr
#'  \link[paws.compute:ec2_describe_instances]{describe_instances} \tab Describes the specified instances or all instances\cr
#'  \link[paws.compute:ec2_describe_instance_status]{describe_instance_status} \tab Describes the status of the specified instances or all of your instances\cr
#'  \link[paws.compute:ec2_describe_instance_type_offerings]{describe_instance_type_offerings} \tab Returns a list of all instance types offered\cr
#'  \link[paws.compute:ec2_describe_instance_types]{describe_instance_types} \tab Describes the details of the instance types that are offered in a location\cr
#'  \link[paws.compute:ec2_describe_internet_gateways]{describe_internet_gateways} \tab Describes one or more of your internet gateways\cr
#'  \link[paws.compute:ec2_describe_ipam_pools]{describe_ipam_pools} \tab Get information about your IPAM pools\cr
#'  \link[paws.compute:ec2_describe_ipam_resource_discoveries]{describe_ipam_resource_discoveries} \tab Describes IPAM resource discoveries\cr
#'  \link[paws.compute:ec2_describe_ipam_resource_discovery_associations]{describe_ipam_resource_discovery_associations} \tab Describes resource discovery association with an Amazon VPC IPAM\cr
#'  \link[paws.compute:ec2_describe_ipams]{describe_ipams} \tab Get information about your IPAM pools\cr
#'  \link[paws.compute:ec2_describe_ipam_scopes]{describe_ipam_scopes} \tab Get information about your IPAM scopes\cr
#'  \link[paws.compute:ec2_describe_ipv_6_pools]{describe_ipv_6_pools} \tab Describes your IPv6 address pools\cr
#'  \link[paws.compute:ec2_describe_key_pairs]{describe_key_pairs} \tab Describes the specified key pairs or all of your key pairs\cr
#'  \link[paws.compute:ec2_describe_launch_templates]{describe_launch_templates} \tab Describes one or more launch templates\cr
#'  \link[paws.compute:ec2_describe_launch_template_versions]{describe_launch_template_versions} \tab Describes one or more versions of a specified launch template\cr
#'  \link[paws.compute:ec2_describe_local_gateway_route_tables]{describe_local_gateway_route_tables} \tab Describes one or more local gateway route tables\cr
#'  \link[paws.compute:ec2_describe_local_gateway_route_table_virtual_interface_group_associations]{describe_local_gateway_route_table_virtual_interface_group_associations} \tab Describes the associations between virtual interface groups and local gateway route tables\cr
#'  \link[paws.compute:ec2_describe_local_gateway_route_table_vpc_associations]{describe_local_gateway_route_table_vpc_associations} \tab Describes the specified associations between VPCs and local gateway route tables\cr
#'  \link[paws.compute:ec2_describe_local_gateways]{describe_local_gateways} \tab Describes one or more local gateways\cr
#'  \link[paws.compute:ec2_describe_local_gateway_virtual_interface_groups]{describe_local_gateway_virtual_interface_groups} \tab Describes the specified local gateway virtual interface groups\cr
#'  \link[paws.compute:ec2_describe_local_gateway_virtual_interfaces]{describe_local_gateway_virtual_interfaces} \tab Describes the specified local gateway virtual interfaces\cr
#'  \link[paws.compute:ec2_describe_managed_prefix_lists]{describe_managed_prefix_lists} \tab Describes your managed prefix lists and any Amazon Web Services-managed prefix lists\cr
#'  \link[paws.compute:ec2_describe_moving_addresses]{describe_moving_addresses} \tab This action is deprecated\cr
#'  \link[paws.compute:ec2_describe_nat_gateways]{describe_nat_gateways} \tab Describes one or more of your NAT gateways\cr
#'  \link[paws.compute:ec2_describe_network_acls]{describe_network_acls} \tab Describes one or more of your network ACLs\cr
#'  \link[paws.compute:ec2_describe_network_insights_access_scope_analyses]{describe_network_insights_access_scope_analyses} \tab Describes the specified Network Access Scope analyses\cr
#'  \link[paws.compute:ec2_describe_network_insights_access_scopes]{describe_network_insights_access_scopes} \tab Describes the specified Network Access Scopes\cr
#'  \link[paws.compute:ec2_describe_network_insights_analyses]{describe_network_insights_analyses} \tab Describes one or more of your network insights analyses\cr
#'  \link[paws.compute:ec2_describe_network_insights_paths]{describe_network_insights_paths} \tab Describes one or more of your paths\cr
#'  \link[paws.compute:ec2_describe_network_interface_attribute]{describe_network_interface_attribute} \tab Describes a network interface attribute\cr
#'  \link[paws.compute:ec2_describe_network_interface_permissions]{describe_network_interface_permissions} \tab Describes the permissions for your network interfaces\cr
#'  \link[paws.compute:ec2_describe_network_interfaces]{describe_network_interfaces} \tab Describes one or more of your network interfaces\cr
#'  \link[paws.compute:ec2_describe_placement_groups]{describe_placement_groups} \tab Describes the specified placement groups or all of your placement groups\cr
#'  \link[paws.compute:ec2_describe_prefix_lists]{describe_prefix_lists} \tab Describes available Amazon Web Services services in a prefix list format, which includes the prefix list name and prefix list ID of the service and the IP address range for the service\cr
#'  \link[paws.compute:ec2_describe_principal_id_format]{describe_principal_id_format} \tab Describes the ID format settings for the root user and all IAM roles and IAM users that have explicitly specified a longer ID (17-character ID) preference\cr
#'  \link[paws.compute:ec2_describe_public_ipv_4_pools]{describe_public_ipv_4_pools} \tab Describes the specified IPv4 address pools\cr
#'  \link[paws.compute:ec2_describe_regions]{describe_regions} \tab Describes the Regions that are enabled for your account, or all Regions\cr
#'  \link[paws.compute:ec2_describe_replace_root_volume_tasks]{describe_replace_root_volume_tasks} \tab Describes a root volume replacement task\cr
#'  \link[paws.compute:ec2_describe_reserved_instances]{describe_reserved_instances} \tab Describes one or more of the Reserved Instances that you purchased\cr
#'  \link[paws.compute:ec2_describe_reserved_instances_listings]{describe_reserved_instances_listings} \tab Describes your account's Reserved Instance listings in the Reserved Instance Marketplace\cr
#'  \link[paws.compute:ec2_describe_reserved_instances_modifications]{describe_reserved_instances_modifications} \tab Describes the modifications made to your Reserved Instances\cr
#'  \link[paws.compute:ec2_describe_reserved_instances_offerings]{describe_reserved_instances_offerings} \tab Describes Reserved Instance offerings that are available for purchase\cr
#'  \link[paws.compute:ec2_describe_route_tables]{describe_route_tables} \tab Describes one or more of your route tables\cr
#'  \link[paws.compute:ec2_describe_scheduled_instance_availability]{describe_scheduled_instance_availability} \tab Finds available schedules that meet the specified criteria\cr
#'  \link[paws.compute:ec2_describe_scheduled_instances]{describe_scheduled_instances} \tab Describes the specified Scheduled Instances or all your Scheduled Instances\cr
#'  \link[paws.compute:ec2_describe_security_group_references]{describe_security_group_references} \tab Describes the VPCs on the other side of a VPC peering connection that are referencing the security groups you've specified in this request\cr
#'  \link[paws.compute:ec2_describe_security_group_rules]{describe_security_group_rules} \tab Describes one or more of your security group rules\cr
#'  \link[paws.compute:ec2_describe_security_groups]{describe_security_groups} \tab Describes the specified security groups or all of your security groups\cr
#'  \link[paws.compute:ec2_describe_snapshot_attribute]{describe_snapshot_attribute} \tab Describes the specified attribute of the specified snapshot\cr
#'  \link[paws.compute:ec2_describe_snapshots]{describe_snapshots} \tab Describes the specified EBS snapshots available to you or all of the EBS snapshots available to you\cr
#'  \link[paws.compute:ec2_describe_snapshot_tier_status]{describe_snapshot_tier_status} \tab Describes the storage tier status of one or more Amazon EBS snapshots\cr
#'  \link[paws.compute:ec2_describe_spot_datafeed_subscription]{describe_spot_datafeed_subscription} \tab Describes the data feed for Spot Instances\cr
#'  \link[paws.compute:ec2_describe_spot_fleet_instances]{describe_spot_fleet_instances} \tab Describes the running instances for the specified Spot Fleet\cr
#'  \link[paws.compute:ec2_describe_spot_fleet_request_history]{describe_spot_fleet_request_history} \tab Describes the events for the specified Spot Fleet request during the specified time\cr
#'  \link[paws.compute:ec2_describe_spot_fleet_requests]{describe_spot_fleet_requests} \tab Describes your Spot Fleet requests\cr
#'  \link[paws.compute:ec2_describe_spot_instance_requests]{describe_spot_instance_requests} \tab Describes the specified Spot Instance requests\cr
#'  \link[paws.compute:ec2_describe_spot_price_history]{describe_spot_price_history} \tab Describes the Spot price history\cr
#'  \link[paws.compute:ec2_describe_stale_security_groups]{describe_stale_security_groups} \tab Describes the stale security group rules for security groups in a specified VPC\cr
#'  \link[paws.compute:ec2_describe_store_image_tasks]{describe_store_image_tasks} \tab Describes the progress of the AMI store tasks\cr
#'  \link[paws.compute:ec2_describe_subnets]{describe_subnets} \tab Describes one or more of your subnets\cr
#'  \link[paws.compute:ec2_describe_tags]{describe_tags} \tab Describes the specified tags for your EC2 resources\cr
#'  \link[paws.compute:ec2_describe_traffic_mirror_filters]{describe_traffic_mirror_filters} \tab Describes one or more Traffic Mirror filters\cr
#'  \link[paws.compute:ec2_describe_traffic_mirror_sessions]{describe_traffic_mirror_sessions} \tab Describes one or more Traffic Mirror sessions\cr
#'  \link[paws.compute:ec2_describe_traffic_mirror_targets]{describe_traffic_mirror_targets} \tab Information about one or more Traffic Mirror targets\cr
#'  \link[paws.compute:ec2_describe_transit_gateway_attachments]{describe_transit_gateway_attachments} \tab Describes one or more attachments between resources and transit gateways\cr
#'  \link[paws.compute:ec2_describe_transit_gateway_connect_peers]{describe_transit_gateway_connect_peers} \tab Describes one or more Connect peers\cr
#'  \link[paws.compute:ec2_describe_transit_gateway_connects]{describe_transit_gateway_connects} \tab Describes one or more Connect attachments\cr
#'  \link[paws.compute:ec2_describe_transit_gateway_multicast_domains]{describe_transit_gateway_multicast_domains} \tab Describes one or more transit gateway multicast domains\cr
#'  \link[paws.compute:ec2_describe_transit_gateway_peering_attachments]{describe_transit_gateway_peering_attachments} \tab Describes your transit gateway peering attachments\cr
#'  \link[paws.compute:ec2_describe_transit_gateway_policy_tables]{describe_transit_gateway_policy_tables} \tab Describes one or more transit gateway route policy tables\cr
#'  \link[paws.compute:ec2_describe_transit_gateway_route_table_announcements]{describe_transit_gateway_route_table_announcements} \tab Describes one or more transit gateway route table advertisements\cr
#'  \link[paws.compute:ec2_describe_transit_gateway_route_tables]{describe_transit_gateway_route_tables} \tab Describes one or more transit gateway route tables\cr
#'  \link[paws.compute:ec2_describe_transit_gateways]{describe_transit_gateways} \tab Describes one or more transit gateways\cr
#'  \link[paws.compute:ec2_describe_transit_gateway_vpc_attachments]{describe_transit_gateway_vpc_attachments} \tab Describes one or more VPC attachments\cr
#'  \link[paws.compute:ec2_describe_trunk_interface_associations]{describe_trunk_interface_associations} \tab This API action is currently in limited preview only\cr
#'  \link[paws.compute:ec2_describe_verified_access_endpoints]{describe_verified_access_endpoints} \tab Describes the specified Amazon Web Services Verified Access endpoints\cr
#'  \link[paws.compute:ec2_describe_verified_access_groups]{describe_verified_access_groups} \tab Describes the specified Verified Access groups\cr
#'  \link[paws.compute:ec2_describe_verified_access_instance_logging_configurations]{describe_verified_access_instance_logging_configurations} \tab Describes the specified Amazon Web Services Verified Access instances\cr
#'  \link[paws.compute:ec2_describe_verified_access_instances]{describe_verified_access_instances} \tab Describes the specified Amazon Web Services Verified Access instances\cr
#'  \link[paws.compute:ec2_describe_verified_access_trust_providers]{describe_verified_access_trust_providers} \tab Describes the specified Amazon Web Services Verified Access trust providers\cr
#'  \link[paws.compute:ec2_describe_volume_attribute]{describe_volume_attribute} \tab Describes the specified attribute of the specified volume\cr
#'  \link[paws.compute:ec2_describe_volumes]{describe_volumes} \tab Describes the specified EBS volumes or all of your EBS volumes\cr
#'  \link[paws.compute:ec2_describe_volumes_modifications]{describe_volumes_modifications} \tab Describes the most recent volume modification request for the specified EBS volumes\cr
#'  \link[paws.compute:ec2_describe_volume_status]{describe_volume_status} \tab Describes the status of the specified volumes\cr
#'  \link[paws.compute:ec2_describe_vpc_attribute]{describe_vpc_attribute} \tab Describes the specified attribute of the specified VPC\cr
#'  \link[paws.compute:ec2_describe_vpc_classic_link]{describe_vpc_classic_link} \tab This action is deprecated\cr
#'  \link[paws.compute:ec2_describe_vpc_classic_link_dns_support]{describe_vpc_classic_link_dns_support} \tab This action is deprecated\cr
#'  \link[paws.compute:ec2_describe_vpc_endpoint_connection_notifications]{describe_vpc_endpoint_connection_notifications} \tab Describes the connection notifications for VPC endpoints and VPC endpoint services\cr
#'  \link[paws.compute:ec2_describe_vpc_endpoint_connections]{describe_vpc_endpoint_connections} \tab Describes the VPC endpoint connections to your VPC endpoint services, including any endpoints that are pending your acceptance\cr
#'  \link[paws.compute:ec2_describe_vpc_endpoints]{describe_vpc_endpoints} \tab Describes your VPC endpoints\cr
#'  \link[paws.compute:ec2_describe_vpc_endpoint_service_configurations]{describe_vpc_endpoint_service_configurations} \tab Describes the VPC endpoint service configurations in your account (your services)\cr
#'  \link[paws.compute:ec2_describe_vpc_endpoint_service_permissions]{describe_vpc_endpoint_service_permissions} \tab Describes the principals (service consumers) that are permitted to discover your VPC endpoint service\cr
#'  \link[paws.compute:ec2_describe_vpc_endpoint_services]{describe_vpc_endpoint_services} \tab Describes available services to which you can create a VPC endpoint\cr
#'  \link[paws.compute:ec2_describe_vpc_peering_connections]{describe_vpc_peering_connections} \tab Describes one or more of your VPC peering connections\cr
#'  \link[paws.compute:ec2_describe_vpcs]{describe_vpcs} \tab Describes one or more of your VPCs\cr
#'  \link[paws.compute:ec2_describe_vpn_connections]{describe_vpn_connections} \tab Describes one or more of your VPN connections\cr
#'  \link[paws.compute:ec2_describe_vpn_gateways]{describe_vpn_gateways} \tab Describes one or more of your virtual private gateways\cr
#'  \link[paws.compute:ec2_detach_classic_link_vpc]{detach_classic_link_vpc} \tab This action is deprecated\cr
#'  \link[paws.compute:ec2_detach_internet_gateway]{detach_internet_gateway} \tab Detaches an internet gateway from a VPC, disabling connectivity between the internet and the VPC\cr
#'  \link[paws.compute:ec2_detach_network_interface]{detach_network_interface} \tab Detaches a network interface from an instance\cr
#'  \link[paws.compute:ec2_detach_verified_access_trust_provider]{detach_verified_access_trust_provider} \tab Detaches the specified Amazon Web Services Verified Access trust provider from the specified Amazon Web Services Verified Access instance\cr
#'  \link[paws.compute:ec2_detach_volume]{detach_volume} \tab Detaches an EBS volume from an instance\cr
#'  \link[paws.compute:ec2_detach_vpn_gateway]{detach_vpn_gateway} \tab Detaches a virtual private gateway from a VPC\cr
#'  \link[paws.compute:ec2_disable_address_transfer]{disable_address_transfer} \tab Disables Elastic IP address transfer\cr
#'  \link[paws.compute:ec2_disable_aws_network_performance_metric_subscription]{disable_aws_network_performance_metric_subscription} \tab Disables Infrastructure Performance metric subscriptions\cr
#'  \link[paws.compute:ec2_disable_ebs_encryption_by_default]{disable_ebs_encryption_by_default} \tab Disables EBS encryption by default for your account in the current Region\cr
#'  \link[paws.compute:ec2_disable_fast_launch]{disable_fast_launch} \tab Discontinue faster launching for a Windows AMI, and clean up existing pre-provisioned snapshots\cr
#'  \link[paws.compute:ec2_disable_fast_snapshot_restores]{disable_fast_snapshot_restores} \tab Disables fast snapshot restores for the specified snapshots in the specified Availability Zones\cr
#'  \link[paws.compute:ec2_disable_image_deprecation]{disable_image_deprecation} \tab Cancels the deprecation of the specified AMI\cr
#'  \link[paws.compute:ec2_disable_ipam_organization_admin_account]{disable_ipam_organization_admin_account} \tab Disable the IPAM account\cr
#'  \link[paws.compute:ec2_disable_serial_console_access]{disable_serial_console_access} \tab Disables access to the EC2 serial console of all instances for your account\cr
#'  \link[paws.compute:ec2_disable_transit_gateway_route_table_propagation]{disable_transit_gateway_route_table_propagation} \tab Disables the specified resource attachment from propagating routes to the specified propagation route table\cr
#'  \link[paws.compute:ec2_disable_vgw_route_propagation]{disable_vgw_route_propagation} \tab Disables a virtual private gateway (VGW) from propagating routes to a specified route table of a VPC\cr
#'  \link[paws.compute:ec2_disable_vpc_classic_link]{disable_vpc_classic_link} \tab This action is deprecated\cr
#'  \link[paws.compute:ec2_disable_vpc_classic_link_dns_support]{disable_vpc_classic_link_dns_support} \tab This action is deprecated\cr
#'  \link[paws.compute:ec2_disassociate_address]{disassociate_address} \tab Disassociates an Elastic IP address from the instance or network interface it's associated with\cr
#'  \link[paws.compute:ec2_disassociate_client_vpn_target_network]{disassociate_client_vpn_target_network} \tab Disassociates a target network from the specified Client VPN endpoint\cr
#'  \link[paws.compute:ec2_disassociate_enclave_certificate_iam_role]{disassociate_enclave_certificate_iam_role} \tab Disassociates an IAM role from an Certificate Manager (ACM) certificate\cr
#'  \link[paws.compute:ec2_disassociate_iam_instance_profile]{disassociate_iam_instance_profile} \tab Disassociates an IAM instance profile from a running or stopped instance\cr
#'  \link[paws.compute:ec2_disassociate_instance_event_window]{disassociate_instance_event_window} \tab Disassociates one or more targets from an event window\cr
#'  \link[paws.compute:ec2_disassociate_ipam_resource_discovery]{disassociate_ipam_resource_discovery} \tab Disassociates a resource discovery from an Amazon VPC IPAM\cr
#'  \link[paws.compute:ec2_disassociate_nat_gateway_address]{disassociate_nat_gateway_address} \tab Disassociates secondary Elastic IP addresses (EIPs) from a public NAT gateway\cr
#'  \link[paws.compute:ec2_disassociate_route_table]{disassociate_route_table} \tab Disassociates a subnet or gateway from a route table\cr
#'  \link[paws.compute:ec2_disassociate_subnet_cidr_block]{disassociate_subnet_cidr_block} \tab Disassociates a CIDR block from a subnet\cr
#'  \link[paws.compute:ec2_disassociate_transit_gateway_multicast_domain]{disassociate_transit_gateway_multicast_domain} \tab Disassociates the specified subnets from the transit gateway multicast domain\cr
#'  \link[paws.compute:ec2_disassociate_transit_gateway_policy_table]{disassociate_transit_gateway_policy_table} \tab Removes the association between an an attachment and a policy table\cr
#'  \link[paws.compute:ec2_disassociate_transit_gateway_route_table]{disassociate_transit_gateway_route_table} \tab Disassociates a resource attachment from a transit gateway route table\cr
#'  \link[paws.compute:ec2_disassociate_trunk_interface]{disassociate_trunk_interface} \tab This API action is currently in limited preview only\cr
#'  \link[paws.compute:ec2_disassociate_vpc_cidr_block]{disassociate_vpc_cidr_block} \tab Disassociates a CIDR block from a VPC\cr
#'  \link[paws.compute:ec2_enable_address_transfer]{enable_address_transfer} \tab Enables Elastic IP address transfer\cr
#'  \link[paws.compute:ec2_enable_aws_network_performance_metric_subscription]{enable_aws_network_performance_metric_subscription} \tab Enables Infrastructure Performance subscriptions\cr
#'  \link[paws.compute:ec2_enable_ebs_encryption_by_default]{enable_ebs_encryption_by_default} \tab Enables EBS encryption by default for your account in the current Region\cr
#'  \link[paws.compute:ec2_enable_fast_launch]{enable_fast_launch} \tab When you enable faster launching for a Windows AMI, images are pre-provisioned, using snapshots to launch instances up to 65% faster\cr
#'  \link[paws.compute:ec2_enable_fast_snapshot_restores]{enable_fast_snapshot_restores} \tab Enables fast snapshot restores for the specified snapshots in the specified Availability Zones\cr
#'  \link[paws.compute:ec2_enable_image_deprecation]{enable_image_deprecation} \tab Enables deprecation of the specified AMI at the specified date and time\cr
#'  \link[paws.compute:ec2_enable_ipam_organization_admin_account]{enable_ipam_organization_admin_account} \tab Enable an Organizations member account as the IPAM admin account\cr
#'  \link[paws.compute:ec2_enable_reachability_analyzer_organization_sharing]{enable_reachability_analyzer_organization_sharing} \tab Establishes a trust relationship between Reachability Analyzer and Organizations\cr
#'  \link[paws.compute:ec2_enable_serial_console_access]{enable_serial_console_access} \tab Enables access to the EC2 serial console of all instances for your account\cr
#'  \link[paws.compute:ec2_enable_transit_gateway_route_table_propagation]{enable_transit_gateway_route_table_propagation} \tab Enables the specified attachment to propagate routes to the specified propagation route table\cr
#'  \link[paws.compute:ec2_enable_vgw_route_propagation]{enable_vgw_route_propagation} \tab Enables a virtual private gateway (VGW) to propagate routes to the specified route table of a VPC\cr
#'  \link[paws.compute:ec2_enable_volume_io]{enable_volume_io} \tab Enables I/O operations for a volume that had I/O operations disabled because the data on the volume was potentially inconsistent\cr
#'  \link[paws.compute:ec2_enable_vpc_classic_link]{enable_vpc_classic_link} \tab This action is deprecated\cr
#'  \link[paws.compute:ec2_enable_vpc_classic_link_dns_support]{enable_vpc_classic_link_dns_support} \tab This action is deprecated\cr
#'  \link[paws.compute:ec2_export_client_vpn_client_certificate_revocation_list]{export_client_vpn_client_certificate_revocation_list} \tab Downloads the client certificate revocation list for the specified Client VPN endpoint\cr
#'  \link[paws.compute:ec2_export_client_vpn_client_configuration]{export_client_vpn_client_configuration} \tab Downloads the contents of the Client VPN endpoint configuration file for the specified Client VPN endpoint\cr
#'  \link[paws.compute:ec2_export_image]{export_image} \tab Exports an Amazon Machine Image (AMI) to a VM file\cr
#'  \link[paws.compute:ec2_export_transit_gateway_routes]{export_transit_gateway_routes} \tab Exports routes from the specified transit gateway route table to the specified S3 bucket\cr
#'  \link[paws.compute:ec2_get_associated_enclave_certificate_iam_roles]{get_associated_enclave_certificate_iam_roles} \tab Returns the IAM roles that are associated with the specified ACM (ACM) certificate\cr
#'  \link[paws.compute:ec2_get_associated_ipv_6_pool_cidrs]{get_associated_ipv_6_pool_cidrs} \tab Gets information about the IPv6 CIDR block associations for a specified IPv6 address pool\cr
#'  \link[paws.compute:ec2_get_aws_network_performance_data]{get_aws_network_performance_data} \tab Gets network performance data\cr
#'  \link[paws.compute:ec2_get_capacity_reservation_usage]{get_capacity_reservation_usage} \tab Gets usage information about a Capacity Reservation\cr
#'  \link[paws.compute:ec2_get_coip_pool_usage]{get_coip_pool_usage} \tab Describes the allocations from the specified customer-owned address pool\cr
#'  \link[paws.compute:ec2_get_console_output]{get_console_output} \tab Gets the console output for the specified instance\cr
#'  \link[paws.compute:ec2_get_console_screenshot]{get_console_screenshot} \tab Retrieve a JPG-format screenshot of a running instance to help with troubleshooting\cr
#'  \link[paws.compute:ec2_get_default_credit_specification]{get_default_credit_specification} \tab Describes the default credit option for CPU usage of a burstable performance instance family\cr
#'  \link[paws.compute:ec2_get_ebs_default_kms_key_id]{get_ebs_default_kms_key_id} \tab Describes the default KMS key for EBS encryption by default for your account in this Region\cr
#'  \link[paws.compute:ec2_get_ebs_encryption_by_default]{get_ebs_encryption_by_default} \tab Describes whether EBS encryption by default is enabled for your account in the current Region\cr
#'  \link[paws.compute:ec2_get_flow_logs_integration_template]{get_flow_logs_integration_template} \tab Generates a CloudFormation template that streamlines and automates the integration of VPC flow logs with Amazon Athena\cr
#'  \link[paws.compute:ec2_get_groups_for_capacity_reservation]{get_groups_for_capacity_reservation} \tab Lists the resource groups to which a Capacity Reservation has been added\cr
#'  \link[paws.compute:ec2_get_host_reservation_purchase_preview]{get_host_reservation_purchase_preview} \tab Preview a reservation purchase with configurations that match those of your Dedicated Host\cr
#'  \link[paws.compute:ec2_get_instance_types_from_instance_requirements]{get_instance_types_from_instance_requirements} \tab Returns a list of instance types with the specified instance attributes\cr
#'  \link[paws.compute:ec2_get_instance_uefi_data]{get_instance_uefi_data} \tab A binary representation of the UEFI variable store\cr
#'  \link[paws.compute:ec2_get_ipam_address_history]{get_ipam_address_history} \tab Retrieve historical information about a CIDR within an IPAM scope\cr
#'  \link[paws.compute:ec2_get_ipam_discovered_accounts]{get_ipam_discovered_accounts} \tab Gets IPAM discovered accounts\cr
#'  \link[paws.compute:ec2_get_ipam_discovered_resource_cidrs]{get_ipam_discovered_resource_cidrs} \tab Returns the resource CIDRs that are monitored as part of a resource discovery\cr
#'  \link[paws.compute:ec2_get_ipam_pool_allocations]{get_ipam_pool_allocations} \tab Get a list of all the CIDR allocations in an IPAM pool\cr
#'  \link[paws.compute:ec2_get_ipam_pool_cidrs]{get_ipam_pool_cidrs} \tab Get the CIDRs provisioned to an IPAM pool\cr
#'  \link[paws.compute:ec2_get_ipam_resource_cidrs]{get_ipam_resource_cidrs} \tab Returns resource CIDRs managed by IPAM in a given scope\cr
#'  \link[paws.compute:ec2_get_launch_template_data]{get_launch_template_data} \tab Retrieves the configuration data of the specified instance\cr
#'  \link[paws.compute:ec2_get_managed_prefix_list_associations]{get_managed_prefix_list_associations} \tab Gets information about the resources that are associated with the specified managed prefix list\cr
#'  \link[paws.compute:ec2_get_managed_prefix_list_entries]{get_managed_prefix_list_entries} \tab Gets information about the entries for a specified managed prefix list\cr
#'  \link[paws.compute:ec2_get_network_insights_access_scope_analysis_findings]{get_network_insights_access_scope_analysis_findings} \tab Gets the findings for the specified Network Access Scope analysis\cr
#'  \link[paws.compute:ec2_get_network_insights_access_scope_content]{get_network_insights_access_scope_content} \tab Gets the content for the specified Network Access Scope\cr
#'  \link[paws.compute:ec2_get_password_data]{get_password_data} \tab Retrieves the encrypted administrator password for a running Windows instance\cr
#'  \link[paws.compute:ec2_get_reserved_instances_exchange_quote]{get_reserved_instances_exchange_quote} \tab Returns a quote and exchange information for exchanging one or more specified Convertible Reserved Instances for a new Convertible Reserved Instance\cr
#'  \link[paws.compute:ec2_get_serial_console_access_status]{get_serial_console_access_status} \tab Retrieves the access status of your account to the EC2 serial console of all instances\cr
#'  \link[paws.compute:ec2_get_spot_placement_scores]{get_spot_placement_scores} \tab Calculates the Spot placement score for a Region or Availability Zone based on the specified target capacity and compute requirements\cr
#'  \link[paws.compute:ec2_get_subnet_cidr_reservations]{get_subnet_cidr_reservations} \tab Gets information about the subnet CIDR reservations\cr
#'  \link[paws.compute:ec2_get_transit_gateway_attachment_propagations]{get_transit_gateway_attachment_propagations} \tab Lists the route tables to which the specified resource attachment propagates routes\cr
#'  \link[paws.compute:ec2_get_transit_gateway_multicast_domain_associations]{get_transit_gateway_multicast_domain_associations} \tab Gets information about the associations for the transit gateway multicast domain\cr
#'  \link[paws.compute:ec2_get_transit_gateway_policy_table_associations]{get_transit_gateway_policy_table_associations} \tab Gets a list of the transit gateway policy table associations\cr
#'  \link[paws.compute:ec2_get_transit_gateway_policy_table_entries]{get_transit_gateway_policy_table_entries} \tab Returns a list of transit gateway policy table entries\cr
#'  \link[paws.compute:ec2_get_transit_gateway_prefix_list_references]{get_transit_gateway_prefix_list_references} \tab Gets information about the prefix list references in a specified transit gateway route table\cr
#'  \link[paws.compute:ec2_get_transit_gateway_route_table_associations]{get_transit_gateway_route_table_associations} \tab Gets information about the associations for the specified transit gateway route table\cr
#'  \link[paws.compute:ec2_get_transit_gateway_route_table_propagations]{get_transit_gateway_route_table_propagations} \tab Gets information about the route table propagations for the specified transit gateway route table\cr
#'  \link[paws.compute:ec2_get_verified_access_endpoint_policy]{get_verified_access_endpoint_policy} \tab Get the Verified Access policy associated with the endpoint\cr
#'  \link[paws.compute:ec2_get_verified_access_group_policy]{get_verified_access_group_policy} \tab Shows the contents of the Verified Access policy associated with the group\cr
#'  \link[paws.compute:ec2_get_vpn_connection_device_sample_configuration]{get_vpn_connection_device_sample_configuration} \tab Download an Amazon Web Services-provided sample configuration file to be used with the customer gateway device specified for your Site-to-Site VPN connection\cr
#'  \link[paws.compute:ec2_get_vpn_connection_device_types]{get_vpn_connection_device_types} \tab Obtain a list of customer gateway devices for which sample configuration files can be provided\cr
#'  \link[paws.compute:ec2_get_vpn_tunnel_replacement_status]{get_vpn_tunnel_replacement_status} \tab Get details of available tunnel endpoint maintenance\cr
#'  \link[paws.compute:ec2_import_client_vpn_client_certificate_revocation_list]{import_client_vpn_client_certificate_revocation_list} \tab Uploads a client certificate revocation list to the specified Client VPN endpoint\cr
#'  \link[paws.compute:ec2_import_image]{import_image} \tab To import your virtual machines (VMs) with a console-based experience, you can use the Import virtual machine images to Amazon Web Services template in the Migration Hub Orchestrator console\cr
#'  \link[paws.compute:ec2_import_instance]{import_instance} \tab We recommend that you use the ImportImage API\cr
#'  \link[paws.compute:ec2_import_key_pair]{import_key_pair} \tab Imports the public key from an RSA or ED25519 key pair that you created with a third-party tool\cr
#'  \link[paws.compute:ec2_import_snapshot]{import_snapshot} \tab Imports a disk into an EBS snapshot\cr
#'  \link[paws.compute:ec2_import_volume]{import_volume} \tab Creates an import volume task using metadata from the specified disk image\cr
#'  \link[paws.compute:ec2_list_images_in_recycle_bin]{list_images_in_recycle_bin} \tab Lists one or more AMIs that are currently in the Recycle Bin\cr
#'  \link[paws.compute:ec2_list_snapshots_in_recycle_bin]{list_snapshots_in_recycle_bin} \tab Lists one or more snapshots that are currently in the Recycle Bin\cr
#'  \link[paws.compute:ec2_modify_address_attribute]{modify_address_attribute} \tab Modifies an attribute of the specified Elastic IP address\cr
#'  \link[paws.compute:ec2_modify_availability_zone_group]{modify_availability_zone_group} \tab Changes the opt-in status of the Local Zone and Wavelength Zone group for your account\cr
#'  \link[paws.compute:ec2_modify_capacity_reservation]{modify_capacity_reservation} \tab Modifies a Capacity Reservation's capacity and the conditions under which it is to be released\cr
#'  \link[paws.compute:ec2_modify_capacity_reservation_fleet]{modify_capacity_reservation_fleet} \tab Modifies a Capacity Reservation Fleet\cr
#'  \link[paws.compute:ec2_modify_client_vpn_endpoint]{modify_client_vpn_endpoint} \tab Modifies the specified Client VPN endpoint\cr
#'  \link[paws.compute:ec2_modify_default_credit_specification]{modify_default_credit_specification} \tab Modifies the default credit option for CPU usage of burstable performance instances\cr
#'  \link[paws.compute:ec2_modify_ebs_default_kms_key_id]{modify_ebs_default_kms_key_id} \tab Changes the default KMS key for EBS encryption by default for your account in this Region\cr
#'  \link[paws.compute:ec2_modify_fleet]{modify_fleet} \tab Modifies the specified EC2 Fleet\cr
#'  \link[paws.compute:ec2_modify_fpga_image_attribute]{modify_fpga_image_attribute} \tab Modifies the specified attribute of the specified Amazon FPGA Image (AFI)\cr
#'  \link[paws.compute:ec2_modify_hosts]{modify_hosts} \tab Modify the auto-placement setting of a Dedicated Host\cr
#'  \link[paws.compute:ec2_modify_identity_id_format]{modify_identity_id_format} \tab Modifies the ID format of a resource for a specified IAM user, IAM role, or the root user for an account; or all IAM users, IAM roles, and the root user for an account\cr
#'  \link[paws.compute:ec2_modify_id_format]{modify_id_format} \tab Modifies the ID format for the specified resource on a per-Region basis\cr
#'  \link[paws.compute:ec2_modify_image_attribute]{modify_image_attribute} \tab Modifies the specified attribute of the specified AMI\cr
#'  \link[paws.compute:ec2_modify_instance_attribute]{modify_instance_attribute} \tab Modifies the specified attribute of the specified instance\cr
#'  \link[paws.compute:ec2_modify_instance_capacity_reservation_attributes]{modify_instance_capacity_reservation_attributes} \tab Modifies the Capacity Reservation settings for a stopped instance\cr
#'  \link[paws.compute:ec2_modify_instance_credit_specification]{modify_instance_credit_specification} \tab Modifies the credit option for CPU usage on a running or stopped burstable performance instance\cr
#'  \link[paws.compute:ec2_modify_instance_event_start_time]{modify_instance_event_start_time} \tab Modifies the start time for a scheduled Amazon EC2 instance event\cr
#'  \link[paws.compute:ec2_modify_instance_event_window]{modify_instance_event_window} \tab Modifies the specified event window\cr
#'  \link[paws.compute:ec2_modify_instance_maintenance_options]{modify_instance_maintenance_options} \tab Modifies the recovery behavior of your instance to disable simplified automatic recovery or set the recovery behavior to default\cr
#'  \link[paws.compute:ec2_modify_instance_metadata_options]{modify_instance_metadata_options} \tab Modify the instance metadata parameters on a running or stopped instance\cr
#'  \link[paws.compute:ec2_modify_instance_placement]{modify_instance_placement} \tab Modifies the placement attributes for a specified instance\cr
#'  \link[paws.compute:ec2_modify_ipam]{modify_ipam} \tab Modify the configurations of an IPAM\cr
#'  \link[paws.compute:ec2_modify_ipam_pool]{modify_ipam_pool} \tab Modify the configurations of an IPAM pool\cr
#'  \link[paws.compute:ec2_modify_ipam_resource_cidr]{modify_ipam_resource_cidr} \tab Modify a resource CIDR\cr
#'  \link[paws.compute:ec2_modify_ipam_resource_discovery]{modify_ipam_resource_discovery} \tab Modifies a resource discovery\cr
#'  \link[paws.compute:ec2_modify_ipam_scope]{modify_ipam_scope} \tab Modify an IPAM scope\cr
#'  \link[paws.compute:ec2_modify_launch_template]{modify_launch_template} \tab Modifies a launch template\cr
#'  \link[paws.compute:ec2_modify_local_gateway_route]{modify_local_gateway_route} \tab Modifies the specified local gateway route\cr
#'  \link[paws.compute:ec2_modify_managed_prefix_list]{modify_managed_prefix_list} \tab Modifies the specified managed prefix list\cr
#'  \link[paws.compute:ec2_modify_network_interface_attribute]{modify_network_interface_attribute} \tab Modifies the specified network interface attribute\cr
#'  \link[paws.compute:ec2_modify_private_dns_name_options]{modify_private_dns_name_options} \tab Modifies the options for instance hostnames for the specified instance\cr
#'  \link[paws.compute:ec2_modify_reserved_instances]{modify_reserved_instances} \tab Modifies the configuration of your Reserved Instances, such as the Availability Zone, instance count, or instance type\cr
#'  \link[paws.compute:ec2_modify_security_group_rules]{modify_security_group_rules} \tab Modifies the rules of a security group\cr
#'  \link[paws.compute:ec2_modify_snapshot_attribute]{modify_snapshot_attribute} \tab Adds or removes permission settings for the specified snapshot\cr
#'  \link[paws.compute:ec2_modify_snapshot_tier]{modify_snapshot_tier} \tab Archives an Amazon EBS snapshot\cr
#'  \link[paws.compute:ec2_modify_spot_fleet_request]{modify_spot_fleet_request} \tab Modifies the specified Spot Fleet request\cr
#'  \link[paws.compute:ec2_modify_subnet_attribute]{modify_subnet_attribute} \tab Modifies a subnet attribute\cr
#'  \link[paws.compute:ec2_modify_traffic_mirror_filter_network_services]{modify_traffic_mirror_filter_network_services} \tab Allows or restricts mirroring network services\cr
#'  \link[paws.compute:ec2_modify_traffic_mirror_filter_rule]{modify_traffic_mirror_filter_rule} \tab Modifies the specified Traffic Mirror rule\cr
#'  \link[paws.compute:ec2_modify_traffic_mirror_session]{modify_traffic_mirror_session} \tab Modifies a Traffic Mirror session\cr
#'  \link[paws.compute:ec2_modify_transit_gateway]{modify_transit_gateway} \tab Modifies the specified transit gateway\cr
#'  \link[paws.compute:ec2_modify_transit_gateway_prefix_list_reference]{modify_transit_gateway_prefix_list_reference} \tab Modifies a reference (route) to a prefix list in a specified transit gateway route table\cr
#'  \link[paws.compute:ec2_modify_transit_gateway_vpc_attachment]{modify_transit_gateway_vpc_attachment} \tab Modifies the specified VPC attachment\cr
#'  \link[paws.compute:ec2_modify_verified_access_endpoint]{modify_verified_access_endpoint} \tab Modifies the configuration of the specified Amazon Web Services Verified Access endpoint\cr
#'  \link[paws.compute:ec2_modify_verified_access_endpoint_policy]{modify_verified_access_endpoint_policy} \tab Modifies the specified Amazon Web Services Verified Access endpoint policy\cr
#'  \link[paws.compute:ec2_modify_verified_access_group]{modify_verified_access_group} \tab Modifies the specified Amazon Web Services Verified Access group configuration\cr
#'  \link[paws.compute:ec2_modify_verified_access_group_policy]{modify_verified_access_group_policy} \tab Modifies the specified Amazon Web Services Verified Access group policy\cr
#'  \link[paws.compute:ec2_modify_verified_access_instance]{modify_verified_access_instance} \tab Modifies the configuration of the specified Amazon Web Services Verified Access instance\cr
#'  \link[paws.compute:ec2_modify_verified_access_instance_logging_configuration]{modify_verified_access_instance_logging_configuration} \tab Modifies the logging configuration for the specified Amazon Web Services Verified Access instance\cr
#'  \link[paws.compute:ec2_modify_verified_access_trust_provider]{modify_verified_access_trust_provider} \tab Modifies the configuration of the specified Amazon Web Services Verified Access trust provider\cr
#'  \link[paws.compute:ec2_modify_volume]{modify_volume} \tab You can modify several parameters of an existing EBS volume, including volume size, volume type, and IOPS capacity\cr
#'  \link[paws.compute:ec2_modify_volume_attribute]{modify_volume_attribute} \tab Modifies a volume attribute\cr
#'  \link[paws.compute:ec2_modify_vpc_attribute]{modify_vpc_attribute} \tab Modifies the specified attribute of the specified VPC\cr
#'  \link[paws.compute:ec2_modify_vpc_endpoint]{modify_vpc_endpoint} \tab Modifies attributes of a specified VPC endpoint\cr
#'  \link[paws.compute:ec2_modify_vpc_endpoint_connection_notification]{modify_vpc_endpoint_connection_notification} \tab Modifies a connection notification for VPC endpoint or VPC endpoint service\cr
#'  \link[paws.compute:ec2_modify_vpc_endpoint_service_configuration]{modify_vpc_endpoint_service_configuration} \tab Modifies the attributes of your VPC endpoint service configuration\cr
#'  \link[paws.compute:ec2_modify_vpc_endpoint_service_payer_responsibility]{modify_vpc_endpoint_service_payer_responsibility} \tab Modifies the payer responsibility for your VPC endpoint service\cr
#'  \link[paws.compute:ec2_modify_vpc_endpoint_service_permissions]{modify_vpc_endpoint_service_permissions} \tab Modifies the permissions for your VPC endpoint service\cr
#'  \link[paws.compute:ec2_modify_vpc_peering_connection_options]{modify_vpc_peering_connection_options} \tab Modifies the VPC peering connection options on one side of a VPC peering connection\cr
#'  \link[paws.compute:ec2_modify_vpc_tenancy]{modify_vpc_tenancy} \tab Modifies the instance tenancy attribute of the specified VPC\cr
#'  \link[paws.compute:ec2_modify_vpn_connection]{modify_vpn_connection} \tab Modifies the customer gateway or the target gateway of an Amazon Web Services Site-to-Site VPN connection\cr
#'  \link[paws.compute:ec2_modify_vpn_connection_options]{modify_vpn_connection_options} \tab Modifies the connection options for your Site-to-Site VPN connection\cr
#'  \link[paws.compute:ec2_modify_vpn_tunnel_certificate]{modify_vpn_tunnel_certificate} \tab Modifies the VPN tunnel endpoint certificate\cr
#'  \link[paws.compute:ec2_modify_vpn_tunnel_options]{modify_vpn_tunnel_options} \tab Modifies the options for a VPN tunnel in an Amazon Web Services Site-to-Site VPN connection\cr
#'  \link[paws.compute:ec2_monitor_instances]{monitor_instances} \tab Enables detailed monitoring for a running instance\cr
#'  \link[paws.compute:ec2_move_address_to_vpc]{move_address_to_vpc} \tab This action is deprecated\cr
#'  \link[paws.compute:ec2_move_byoip_cidr_to_ipam]{move_byoip_cidr_to_ipam} \tab Move a BYOIPv4 CIDR to IPAM from a public IPv4 pool\cr
#'  \link[paws.compute:ec2_provision_byoip_cidr]{provision_byoip_cidr} \tab Provisions an IPv4 or IPv6 address range for use with your Amazon Web Services resources through bring your own IP addresses (BYOIP) and creates a corresponding address pool\cr
#'  \link[paws.compute:ec2_provision_ipam_pool_cidr]{provision_ipam_pool_cidr} \tab Provision a CIDR to an IPAM pool\cr
#'  \link[paws.compute:ec2_provision_public_ipv_4_pool_cidr]{provision_public_ipv_4_pool_cidr} \tab Provision a CIDR to a public IPv4 pool\cr
#'  \link[paws.compute:ec2_purchase_host_reservation]{purchase_host_reservation} \tab Purchase a reservation with configurations that match those of your Dedicated Host\cr
#'  \link[paws.compute:ec2_purchase_reserved_instances_offering]{purchase_reserved_instances_offering} \tab Purchases a Reserved Instance for use with your account\cr
#'  \link[paws.compute:ec2_purchase_scheduled_instances]{purchase_scheduled_instances} \tab You can no longer purchase Scheduled Instances\cr
#'  \link[paws.compute:ec2_reboot_instances]{reboot_instances} \tab Requests a reboot of the specified instances\cr
#'  \link[paws.compute:ec2_register_image]{register_image} \tab Registers an AMI\cr
#'  \link[paws.compute:ec2_register_instance_event_notification_attributes]{register_instance_event_notification_attributes} \tab Registers a set of tag keys to include in scheduled event notifications for your resources\cr
#'  \link[paws.compute:ec2_register_transit_gateway_multicast_group_members]{register_transit_gateway_multicast_group_members} \tab Registers members (network interfaces) with the transit gateway multicast group\cr
#'  \link[paws.compute:ec2_register_transit_gateway_multicast_group_sources]{register_transit_gateway_multicast_group_sources} \tab Registers sources (network interfaces) with the specified transit gateway multicast group\cr
#'  \link[paws.compute:ec2_reject_transit_gateway_multicast_domain_associations]{reject_transit_gateway_multicast_domain_associations} \tab Rejects a request to associate cross-account subnets with a transit gateway multicast domain\cr
#'  \link[paws.compute:ec2_reject_transit_gateway_peering_attachment]{reject_transit_gateway_peering_attachment} \tab Rejects a transit gateway peering attachment request\cr
#'  \link[paws.compute:ec2_reject_transit_gateway_vpc_attachment]{reject_transit_gateway_vpc_attachment} \tab Rejects a request to attach a VPC to a transit gateway\cr
#'  \link[paws.compute:ec2_reject_vpc_endpoint_connections]{reject_vpc_endpoint_connections} \tab Rejects VPC endpoint connection requests to your VPC endpoint service\cr
#'  \link[paws.compute:ec2_reject_vpc_peering_connection]{reject_vpc_peering_connection} \tab Rejects a VPC peering connection request\cr
#'  \link[paws.compute:ec2_release_address]{release_address} \tab Releases the specified Elastic IP address\cr
#'  \link[paws.compute:ec2_release_hosts]{release_hosts} \tab When you no longer want to use an On-Demand Dedicated Host it can be released\cr
#'  \link[paws.compute:ec2_release_ipam_pool_allocation]{release_ipam_pool_allocation} \tab Release an allocation within an IPAM pool\cr
#'  \link[paws.compute:ec2_replace_iam_instance_profile_association]{replace_iam_instance_profile_association} \tab Replaces an IAM instance profile for the specified running instance\cr
#'  \link[paws.compute:ec2_replace_network_acl_association]{replace_network_acl_association} \tab Changes which network ACL a subnet is associated with\cr
#'  \link[paws.compute:ec2_replace_network_acl_entry]{replace_network_acl_entry} \tab Replaces an entry (rule) in a network ACL\cr
#'  \link[paws.compute:ec2_replace_route]{replace_route} \tab Replaces an existing route within a route table in a VPC\cr
#'  \link[paws.compute:ec2_replace_route_table_association]{replace_route_table_association} \tab Changes the route table associated with a given subnet, internet gateway, or virtual private gateway in a VPC\cr
#'  \link[paws.compute:ec2_replace_transit_gateway_route]{replace_transit_gateway_route} \tab Replaces the specified route in the specified transit gateway route table\cr
#'  \link[paws.compute:ec2_replace_vpn_tunnel]{replace_vpn_tunnel} \tab Trigger replacement of specified VPN tunnel\cr
#'  \link[paws.compute:ec2_report_instance_status]{report_instance_status} \tab Submits feedback about the status of an instance\cr
#'  \link[paws.compute:ec2_request_spot_fleet]{request_spot_fleet} \tab Creates a Spot Fleet request\cr
#'  \link[paws.compute:ec2_request_spot_instances]{request_spot_instances} \tab Creates a Spot Instance request\cr
#'  \link[paws.compute:ec2_reset_address_attribute]{reset_address_attribute} \tab Resets the attribute of the specified IP address\cr
#'  \link[paws.compute:ec2_reset_ebs_default_kms_key_id]{reset_ebs_default_kms_key_id} \tab Resets the default KMS key for EBS encryption for your account in this Region to the Amazon Web Services managed KMS key for EBS\cr
#'  \link[paws.compute:ec2_reset_fpga_image_attribute]{reset_fpga_image_attribute} \tab Resets the specified attribute of the specified Amazon FPGA Image (AFI) to its default value\cr
#'  \link[paws.compute:ec2_reset_image_attribute]{reset_image_attribute} \tab Resets an attribute of an AMI to its default value\cr
#'  \link[paws.compute:ec2_reset_instance_attribute]{reset_instance_attribute} \tab Resets an attribute of an instance to its default value\cr
#'  \link[paws.compute:ec2_reset_network_interface_attribute]{reset_network_interface_attribute} \tab Resets a network interface attribute\cr
#'  \link[paws.compute:ec2_reset_snapshot_attribute]{reset_snapshot_attribute} \tab Resets permission settings for the specified snapshot\cr
#'  \link[paws.compute:ec2_restore_address_to_classic]{restore_address_to_classic} \tab This action is deprecated\cr
#'  \link[paws.compute:ec2_restore_image_from_recycle_bin]{restore_image_from_recycle_bin} \tab Restores an AMI from the Recycle Bin\cr
#'  \link[paws.compute:ec2_restore_managed_prefix_list_version]{restore_managed_prefix_list_version} \tab Restores the entries from a previous version of a managed prefix list to a new version of the prefix list\cr
#'  \link[paws.compute:ec2_restore_snapshot_from_recycle_bin]{restore_snapshot_from_recycle_bin} \tab Restores a snapshot from the Recycle Bin\cr
#'  \link[paws.compute:ec2_restore_snapshot_tier]{restore_snapshot_tier} \tab Restores an archived Amazon EBS snapshot for use temporarily or permanently, or modifies the restore period or restore type for a snapshot that was previously temporarily restored\cr
#'  \link[paws.compute:ec2_revoke_client_vpn_ingress]{revoke_client_vpn_ingress} \tab Removes an ingress authorization rule from a Client VPN endpoint\cr
#'  \link[paws.compute:ec2_revoke_security_group_egress]{revoke_security_group_egress} \tab Removes the specified outbound (egress) rules from the specified security group\cr
#'  \link[paws.compute:ec2_revoke_security_group_ingress]{revoke_security_group_ingress} \tab Removes the specified inbound (ingress) rules from a security group\cr
#'  \link[paws.compute:ec2_run_instances]{run_instances} \tab Launches the specified number of instances using an AMI for which you have permissions\cr
#'  \link[paws.compute:ec2_run_scheduled_instances]{run_scheduled_instances} \tab Launches the specified Scheduled Instances\cr
#'  \link[paws.compute:ec2_search_local_gateway_routes]{search_local_gateway_routes} \tab Searches for routes in the specified local gateway route table\cr
#'  \link[paws.compute:ec2_search_transit_gateway_multicast_groups]{search_transit_gateway_multicast_groups} \tab Searches one or more transit gateway multicast groups and returns the group membership information\cr
#'  \link[paws.compute:ec2_search_transit_gateway_routes]{search_transit_gateway_routes} \tab Searches for routes in the specified transit gateway route table\cr
#'  \link[paws.compute:ec2_send_diagnostic_interrupt]{send_diagnostic_interrupt} \tab Sends a diagnostic interrupt to the specified Amazon EC2 instance to trigger a kernel panic (on Linux instances), or a blue screen/stop error (on Windows instances)\cr
#'  \link[paws.compute:ec2_start_instances]{start_instances} \tab Starts an Amazon EBS-backed instance that you've previously stopped\cr
#'  \link[paws.compute:ec2_start_network_insights_access_scope_analysis]{start_network_insights_access_scope_analysis} \tab Starts analyzing the specified Network Access Scope\cr
#'  \link[paws.compute:ec2_start_network_insights_analysis]{start_network_insights_analysis} \tab Starts analyzing the specified path\cr
#'  \link[paws.compute:ec2_start_vpc_endpoint_service_private_dns_verification]{start_vpc_endpoint_service_private_dns_verification} \tab Initiates the verification process to prove that the service provider owns the private DNS name domain for the endpoint service\cr
#'  \link[paws.compute:ec2_stop_instances]{stop_instances} \tab Stops an Amazon EBS-backed instance\cr
#'  \link[paws.compute:ec2_terminate_client_vpn_connections]{terminate_client_vpn_connections} \tab Terminates active Client VPN endpoint connections\cr
#'  \link[paws.compute:ec2_terminate_instances]{terminate_instances} \tab Shuts down the specified instances\cr
#'  \link[paws.compute:ec2_unassign_ipv_6_addresses]{unassign_ipv_6_addresses} \tab Unassigns one or more IPv6 addresses IPv4 Prefix Delegation prefixes from a network interface\cr
#'  \link[paws.compute:ec2_unassign_private_ip_addresses]{unassign_private_ip_addresses} \tab Unassigns one or more secondary private IP addresses, or IPv4 Prefix Delegation prefixes from a network interface\cr
#'  \link[paws.compute:ec2_unassign_private_nat_gateway_address]{unassign_private_nat_gateway_address} \tab Unassigns secondary private IPv4 addresses from a private NAT gateway\cr
#'  \link[paws.compute:ec2_unmonitor_instances]{unmonitor_instances} \tab Disables detailed monitoring for a running instance\cr
#'  \link[paws.compute:ec2_update_security_group_rule_descriptions_egress]{update_security_group_rule_descriptions_egress} \tab Updates the description of an egress (outbound) security group rule\cr
#'  \link[paws.compute:ec2_update_security_group_rule_descriptions_ingress]{update_security_group_rule_descriptions_ingress} \tab Updates the description of an ingress (inbound) security group rule\cr
#'  \link[paws.compute:ec2_withdraw_byoip_cidr]{withdraw_byoip_cidr} \tab Stops advertising an address range that is provisioned as an address pool
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname ec2
#' @export
ec2 <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.compute::ec2(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' AWS EC2 Instance Connect
#'
#' @description
#' Amazon EC2 Instance Connect enables system administrators to publish
#' one-time use SSH public keys to EC2, providing users a simple and secure
#' way to connect to their instances.
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- ec2instanceconnect(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- ec2instanceconnect()
#' # The following example pushes a sample SSH public key to the EC2 instance
#' # i-abcd1234 in AZ us-west-2b for use by the instance OS user ec2-user.
#' svc$send_ssh_public_key(
#'   AvailabilityZone = "us-west-2a",
#'   InstanceId = "i-abcd1234",
#'   InstanceOSUser = "ec2-user",
#'   SSHPublicKey = "ssh-rsa AAAAB3NzaC1yc2EAAAADAQABAAABAQC3FlHqj2eqCdrGHuA6d..."
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.compute:ec2instanceconnect_send_serial_console_ssh_public_key]{send_serial_console_ssh_public_key} \tab Pushes an SSH public key to the specified EC2 instance\cr
#'  \link[paws.compute:ec2instanceconnect_send_ssh_public_key]{send_ssh_public_key} \tab Pushes an SSH public key to the specified EC2 instance for use by the specified user
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname ec2instanceconnect
#' @export
ec2instanceconnect <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.compute::ec2instanceconnect(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' Amazon EC2 Container Registry
#'
#' @description
#' Amazon Elastic Container Registry
#'
#' Amazon Elastic Container Registry (Amazon ECR) is a managed container
#' image registry service. Customers can use the familiar Docker CLI, or
#' their preferred client, to push, pull, and manage images. Amazon ECR
#' provides a secure, scalable, and reliable registry for your Docker or
#' Open Container Initiative (OCI) images. Amazon ECR supports private
#' repositories with resource-based permissions using IAM so that specific
#' users or Amazon EC2 instances can access repositories and images.
#'
#' Amazon ECR has service endpoints in each supported Region. For more
#' information, see [Amazon ECR
#' endpoints](https://docs.aws.amazon.com/general/latest/gr/ecr.html) in
#' the *Amazon Web Services General Reference*.
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- ecr(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- ecr()
#' # This example deletes images with the tags precise and trusty in a
#' # repository called ubuntu in the default registry for an account.
#' svc$batch_delete_image(
#'   imageIds = list(
#'     list(
#'       imageTag = "precise"
#'     )
#'   ),
#'   repositoryName = "ubuntu"
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.compute:ecr_batch_check_layer_availability]{batch_check_layer_availability} \tab Checks the availability of one or more image layers in a repository\cr
#'  \link[paws.compute:ecr_batch_delete_image]{batch_delete_image} \tab Deletes a list of specified images within a repository\cr
#'  \link[paws.compute:ecr_batch_get_image]{batch_get_image} \tab Gets detailed information for an image\cr
#'  \link[paws.compute:ecr_batch_get_repository_scanning_configuration]{batch_get_repository_scanning_configuration} \tab Gets the scanning configuration for one or more repositories\cr
#'  \link[paws.compute:ecr_complete_layer_upload]{complete_layer_upload} \tab Informs Amazon ECR that the image layer upload has completed for a specified registry, repository name, and upload ID\cr
#'  \link[paws.compute:ecr_create_pull_through_cache_rule]{create_pull_through_cache_rule} \tab Creates a pull through cache rule\cr
#'  \link[paws.compute:ecr_create_repository]{create_repository} \tab Creates a repository\cr
#'  \link[paws.compute:ecr_delete_lifecycle_policy]{delete_lifecycle_policy} \tab Deletes the lifecycle policy associated with the specified repository\cr
#'  \link[paws.compute:ecr_delete_pull_through_cache_rule]{delete_pull_through_cache_rule} \tab Deletes a pull through cache rule\cr
#'  \link[paws.compute:ecr_delete_registry_policy]{delete_registry_policy} \tab Deletes the registry permissions policy\cr
#'  \link[paws.compute:ecr_delete_repository]{delete_repository} \tab Deletes a repository\cr
#'  \link[paws.compute:ecr_delete_repository_policy]{delete_repository_policy} \tab Deletes the repository policy associated with the specified repository\cr
#'  \link[paws.compute:ecr_describe_image_replication_status]{describe_image_replication_status} \tab Returns the replication status for a specified image\cr
#'  \link[paws.compute:ecr_describe_images]{describe_images} \tab Returns metadata about the images in a repository\cr
#'  \link[paws.compute:ecr_describe_image_scan_findings]{describe_image_scan_findings} \tab Returns the scan findings for the specified image\cr
#'  \link[paws.compute:ecr_describe_pull_through_cache_rules]{describe_pull_through_cache_rules} \tab Returns the pull through cache rules for a registry\cr
#'  \link[paws.compute:ecr_describe_registry]{describe_registry} \tab Describes the settings for a registry\cr
#'  \link[paws.compute:ecr_describe_repositories]{describe_repositories} \tab Describes image repositories in a registry\cr
#'  \link[paws.compute:ecr_get_authorization_token]{get_authorization_token} \tab Retrieves an authorization token\cr
#'  \link[paws.compute:ecr_get_download_url_for_layer]{get_download_url_for_layer} \tab Retrieves the pre-signed Amazon S3 download URL corresponding to an image layer\cr
#'  \link[paws.compute:ecr_get_lifecycle_policy]{get_lifecycle_policy} \tab Retrieves the lifecycle policy for the specified repository\cr
#'  \link[paws.compute:ecr_get_lifecycle_policy_preview]{get_lifecycle_policy_preview} \tab Retrieves the results of the lifecycle policy preview request for the specified repository\cr
#'  \link[paws.compute:ecr_get_registry_policy]{get_registry_policy} \tab Retrieves the permissions policy for a registry\cr
#'  \link[paws.compute:ecr_get_registry_scanning_configuration]{get_registry_scanning_configuration} \tab Retrieves the scanning configuration for a registry\cr
#'  \link[paws.compute:ecr_get_repository_policy]{get_repository_policy} \tab Retrieves the repository policy for the specified repository\cr
#'  \link[paws.compute:ecr_initiate_layer_upload]{initiate_layer_upload} \tab Notifies Amazon ECR that you intend to upload an image layer\cr
#'  \link[paws.compute:ecr_list_images]{list_images} \tab Lists all the image IDs for the specified repository\cr
#'  \link[paws.compute:ecr_list_tags_for_resource]{list_tags_for_resource} \tab List the tags for an Amazon ECR resource\cr
#'  \link[paws.compute:ecr_put_image]{put_image} \tab Creates or updates the image manifest and tags associated with an image\cr
#'  \link[paws.compute:ecr_put_image_scanning_configuration]{put_image_scanning_configuration} \tab The PutImageScanningConfiguration API is being deprecated, in favor of specifying the image scanning configuration at the registry level\cr
#'  \link[paws.compute:ecr_put_image_tag_mutability]{put_image_tag_mutability} \tab Updates the image tag mutability settings for the specified repository\cr
#'  \link[paws.compute:ecr_put_lifecycle_policy]{put_lifecycle_policy} \tab Creates or updates the lifecycle policy for the specified repository\cr
#'  \link[paws.compute:ecr_put_registry_policy]{put_registry_policy} \tab Creates or updates the permissions policy for your registry\cr
#'  \link[paws.compute:ecr_put_registry_scanning_configuration]{put_registry_scanning_configuration} \tab Creates or updates the scanning configuration for your private registry\cr
#'  \link[paws.compute:ecr_put_replication_configuration]{put_replication_configuration} \tab Creates or updates the replication configuration for a registry\cr
#'  \link[paws.compute:ecr_set_repository_policy]{set_repository_policy} \tab Applies a repository policy to the specified repository to control access permissions\cr
#'  \link[paws.compute:ecr_start_image_scan]{start_image_scan} \tab Starts an image vulnerability scan\cr
#'  \link[paws.compute:ecr_start_lifecycle_policy_preview]{start_lifecycle_policy_preview} \tab Starts a preview of a lifecycle policy for the specified repository\cr
#'  \link[paws.compute:ecr_tag_resource]{tag_resource} \tab Adds specified tags to a resource with the specified ARN\cr
#'  \link[paws.compute:ecr_untag_resource]{untag_resource} \tab Deletes specified tags from a resource\cr
#'  \link[paws.compute:ecr_upload_layer_part]{upload_layer_part} \tab Uploads an image layer part to Amazon ECR
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname ecr
#' @export
ecr <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.compute::ecr(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' Amazon Elastic Container Registry Public
#'
#' @description
#' Amazon Elastic Container Registry Public (Amazon ECR Public) is a
#' managed container image registry service. Amazon ECR provides both
#' public and private registries to host your container images. You can use
#' the Docker CLI or your preferred client to push, pull, and manage
#' images. Amazon ECR provides a secure, scalable, and reliable registry
#' for your Docker or Open Container Initiative (OCI) images. Amazon ECR
#' supports public repositories with this API. For information about the
#' Amazon ECR API for private repositories, see [Amazon Elastic Container
#' Registry API
#' Reference](https://docs.aws.amazon.com/AmazonECR/latest/APIReference/Welcome.html).
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- ecrpublic(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- ecrpublic()
#' svc$batch_check_layer_availability(
#'   Foo = 123
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.compute:ecrpublic_batch_check_layer_availability]{batch_check_layer_availability} \tab Checks the availability of one or more image layers that are within a repository in a public registry\cr
#'  \link[paws.compute:ecrpublic_batch_delete_image]{batch_delete_image} \tab Deletes a list of specified images that are within a repository in a public registry\cr
#'  \link[paws.compute:ecrpublic_complete_layer_upload]{complete_layer_upload} \tab Informs Amazon ECR that the image layer upload is complete for a specified public registry, repository name, and upload ID\cr
#'  \link[paws.compute:ecrpublic_create_repository]{create_repository} \tab Creates a repository in a public registry\cr
#'  \link[paws.compute:ecrpublic_delete_repository]{delete_repository} \tab Deletes a repository in a public registry\cr
#'  \link[paws.compute:ecrpublic_delete_repository_policy]{delete_repository_policy} \tab Deletes the repository policy that's associated with the specified repository\cr
#'  \link[paws.compute:ecrpublic_describe_images]{describe_images} \tab Returns metadata that's related to the images in a repository in a public registry\cr
#'  \link[paws.compute:ecrpublic_describe_image_tags]{describe_image_tags} \tab Returns the image tag details for a repository in a public registry\cr
#'  \link[paws.compute:ecrpublic_describe_registries]{describe_registries} \tab Returns details for a public registry\cr
#'  \link[paws.compute:ecrpublic_describe_repositories]{describe_repositories} \tab Describes repositories that are in a public registry\cr
#'  \link[paws.compute:ecrpublic_get_authorization_token]{get_authorization_token} \tab Retrieves an authorization token\cr
#'  \link[paws.compute:ecrpublic_get_registry_catalog_data]{get_registry_catalog_data} \tab Retrieves catalog metadata for a public registry\cr
#'  \link[paws.compute:ecrpublic_get_repository_catalog_data]{get_repository_catalog_data} \tab Retrieve catalog metadata for a repository in a public registry\cr
#'  \link[paws.compute:ecrpublic_get_repository_policy]{get_repository_policy} \tab Retrieves the repository policy for the specified repository\cr
#'  \link[paws.compute:ecrpublic_initiate_layer_upload]{initiate_layer_upload} \tab Notifies Amazon ECR that you intend to upload an image layer\cr
#'  \link[paws.compute:ecrpublic_list_tags_for_resource]{list_tags_for_resource} \tab List the tags for an Amazon ECR Public resource\cr
#'  \link[paws.compute:ecrpublic_put_image]{put_image} \tab Creates or updates the image manifest and tags that are associated with an image\cr
#'  \link[paws.compute:ecrpublic_put_registry_catalog_data]{put_registry_catalog_data} \tab Create or update the catalog data for a public registry\cr
#'  \link[paws.compute:ecrpublic_put_repository_catalog_data]{put_repository_catalog_data} \tab Creates or updates the catalog data for a repository in a public registry\cr
#'  \link[paws.compute:ecrpublic_set_repository_policy]{set_repository_policy} \tab Applies a repository policy to the specified public repository to control access permissions\cr
#'  \link[paws.compute:ecrpublic_tag_resource]{tag_resource} \tab Associates the specified tags to a resource with the specified resourceArn\cr
#'  \link[paws.compute:ecrpublic_untag_resource]{untag_resource} \tab Deletes specified tags from a resource\cr
#'  \link[paws.compute:ecrpublic_upload_layer_part]{upload_layer_part} \tab Uploads an image layer part to Amazon ECR
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname ecrpublic
#' @export
ecrpublic <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.compute::ecrpublic(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' Amazon EC2 Container Service
#'
#' @description
#' Amazon Elastic Container Service
#'
#' Amazon Elastic Container Service (Amazon ECS) is a highly scalable,
#' fast, container management service. It makes it easy to run, stop, and
#' manage Docker containers. You can host your cluster on a serverless
#' infrastructure that's managed by Amazon ECS by launching your services
#' or tasks on Fargate. For more control, you can host your tasks on a
#' cluster of Amazon Elastic Compute Cloud (Amazon EC2) or External
#' (on-premises) instances that you manage.
#'
#' Amazon ECS makes it easy to launch and stop container-based applications
#' with simple API calls. This makes it easy to get the state of your
#' cluster from a centralized service, and gives you access to many
#' familiar Amazon EC2 features.
#'
#' You can use Amazon ECS to schedule the placement of containers across
#' your cluster based on your resource needs, isolation policies, and
#' availability requirements. With Amazon ECS, you don't need to operate
#' your own cluster management and configuration management systems. You
#' also don't need to worry about scaling your management infrastructure.
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- ecs(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- ecs()
#' # This example creates a cluster in your default region.
#' svc$create_cluster(
#'   clusterName = "my_cluster"
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.compute:ecs_create_capacity_provider]{create_capacity_provider} \tab Creates a new capacity provider\cr
#'  \link[paws.compute:ecs_create_cluster]{create_cluster} \tab Creates a new Amazon ECS cluster\cr
#'  \link[paws.compute:ecs_create_service]{create_service} \tab Runs and maintains your desired number of tasks from a specified task definition\cr
#'  \link[paws.compute:ecs_create_task_set]{create_task_set} \tab Create a task set in the specified cluster and service\cr
#'  \link[paws.compute:ecs_delete_account_setting]{delete_account_setting} \tab Disables an account setting for a specified user, role, or the root user for an account\cr
#'  \link[paws.compute:ecs_delete_attributes]{delete_attributes} \tab Deletes one or more custom attributes from an Amazon ECS resource\cr
#'  \link[paws.compute:ecs_delete_capacity_provider]{delete_capacity_provider} \tab Deletes the specified capacity provider\cr
#'  \link[paws.compute:ecs_delete_cluster]{delete_cluster} \tab Deletes the specified cluster\cr
#'  \link[paws.compute:ecs_delete_service]{delete_service} \tab Deletes a specified service within a cluster\cr
#'  \link[paws.compute:ecs_delete_task_definitions]{delete_task_definitions} \tab Deletes one or more task definitions\cr
#'  \link[paws.compute:ecs_delete_task_set]{delete_task_set} \tab Deletes a specified task set within a service\cr
#'  \link[paws.compute:ecs_deregister_container_instance]{deregister_container_instance} \tab Deregisters an Amazon ECS container instance from the specified cluster\cr
#'  \link[paws.compute:ecs_deregister_task_definition]{deregister_task_definition} \tab Deregisters the specified task definition by family and revision\cr
#'  \link[paws.compute:ecs_describe_capacity_providers]{describe_capacity_providers} \tab Describes one or more of your capacity providers\cr
#'  \link[paws.compute:ecs_describe_clusters]{describe_clusters} \tab Describes one or more of your clusters\cr
#'  \link[paws.compute:ecs_describe_container_instances]{describe_container_instances} \tab Describes one or more container instances\cr
#'  \link[paws.compute:ecs_describe_services]{describe_services} \tab Describes the specified services running in your cluster\cr
#'  \link[paws.compute:ecs_describe_task_definition]{describe_task_definition} \tab Describes a task definition\cr
#'  \link[paws.compute:ecs_describe_tasks]{describe_tasks} \tab Describes a specified task or tasks\cr
#'  \link[paws.compute:ecs_describe_task_sets]{describe_task_sets} \tab Describes the task sets in the specified cluster and service\cr
#'  \link[paws.compute:ecs_discover_poll_endpoint]{discover_poll_endpoint} \tab This action is only used by the Amazon ECS agent, and it is not intended for use outside of the agent\cr
#'  \link[paws.compute:ecs_execute_command]{execute_command} \tab Runs a command remotely on a container within a task\cr
#'  \link[paws.compute:ecs_get_task_protection]{get_task_protection} \tab Retrieves the protection status of tasks in an Amazon ECS service\cr
#'  \link[paws.compute:ecs_list_account_settings]{list_account_settings} \tab Lists the account settings for a specified principal\cr
#'  \link[paws.compute:ecs_list_attributes]{list_attributes} \tab Lists the attributes for Amazon ECS resources within a specified target type and cluster\cr
#'  \link[paws.compute:ecs_list_clusters]{list_clusters} \tab Returns a list of existing clusters\cr
#'  \link[paws.compute:ecs_list_container_instances]{list_container_instances} \tab Returns a list of container instances in a specified cluster\cr
#'  \link[paws.compute:ecs_list_services]{list_services} \tab Returns a list of services\cr
#'  \link[paws.compute:ecs_list_services_by_namespace]{list_services_by_namespace} \tab This operation lists all of the services that are associated with a Cloud Map namespace\cr
#'  \link[paws.compute:ecs_list_tags_for_resource]{list_tags_for_resource} \tab List the tags for an Amazon ECS resource\cr
#'  \link[paws.compute:ecs_list_task_definition_families]{list_task_definition_families} \tab Returns a list of task definition families that are registered to your account\cr
#'  \link[paws.compute:ecs_list_task_definitions]{list_task_definitions} \tab Returns a list of task definitions that are registered to your account\cr
#'  \link[paws.compute:ecs_list_tasks]{list_tasks} \tab Returns a list of tasks\cr
#'  \link[paws.compute:ecs_put_account_setting]{put_account_setting} \tab Modifies an account setting\cr
#'  \link[paws.compute:ecs_put_account_setting_default]{put_account_setting_default} \tab Modifies an account setting for all users on an account for whom no individual account setting has been specified\cr
#'  \link[paws.compute:ecs_put_attributes]{put_attributes} \tab Create or update an attribute on an Amazon ECS resource\cr
#'  \link[paws.compute:ecs_put_cluster_capacity_providers]{put_cluster_capacity_providers} \tab Modifies the available capacity providers and the default capacity provider strategy for a cluster\cr
#'  \link[paws.compute:ecs_register_container_instance]{register_container_instance} \tab This action is only used by the Amazon ECS agent, and it is not intended for use outside of the agent\cr
#'  \link[paws.compute:ecs_register_task_definition]{register_task_definition} \tab Registers a new task definition from the supplied family and containerDefinitions\cr
#'  \link[paws.compute:ecs_run_task]{run_task} \tab Starts a new task using the specified task definition\cr
#'  \link[paws.compute:ecs_start_task]{start_task} \tab Starts a new task from the specified task definition on the specified container instance or instances\cr
#'  \link[paws.compute:ecs_stop_task]{stop_task} \tab Stops a running task\cr
#'  \link[paws.compute:ecs_submit_attachment_state_changes]{submit_attachment_state_changes} \tab This action is only used by the Amazon ECS agent, and it is not intended for use outside of the agent\cr
#'  \link[paws.compute:ecs_submit_container_state_change]{submit_container_state_change} \tab This action is only used by the Amazon ECS agent, and it is not intended for use outside of the agent\cr
#'  \link[paws.compute:ecs_submit_task_state_change]{submit_task_state_change} \tab This action is only used by the Amazon ECS agent, and it is not intended for use outside of the agent\cr
#'  \link[paws.compute:ecs_tag_resource]{tag_resource} \tab Associates the specified tags to a resource with the specified resourceArn\cr
#'  \link[paws.compute:ecs_untag_resource]{untag_resource} \tab Deletes specified tags from a resource\cr
#'  \link[paws.compute:ecs_update_capacity_provider]{update_capacity_provider} \tab Modifies the parameters for a capacity provider\cr
#'  \link[paws.compute:ecs_update_cluster]{update_cluster} \tab Updates the cluster\cr
#'  \link[paws.compute:ecs_update_cluster_settings]{update_cluster_settings} \tab Modifies the settings to use for a cluster\cr
#'  \link[paws.compute:ecs_update_container_agent]{update_container_agent} \tab Updates the Amazon ECS container agent on a specified container instance\cr
#'  \link[paws.compute:ecs_update_container_instances_state]{update_container_instances_state} \tab Modifies the status of an Amazon ECS container instance\cr
#'  \link[paws.compute:ecs_update_service]{update_service} \tab Modifies the parameters of a service\cr
#'  \link[paws.compute:ecs_update_service_primary_task_set]{update_service_primary_task_set} \tab Modifies which task set in a service is the primary task set\cr
#'  \link[paws.compute:ecs_update_task_protection]{update_task_protection} \tab Updates the protection status of a task\cr
#'  \link[paws.compute:ecs_update_task_set]{update_task_set} \tab Modifies a task set
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname ecs
#' @export
ecs <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.compute::ecs(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' Amazon Elastic Kubernetes Service
#'
#' @description
#' Amazon Elastic Kubernetes Service (Amazon EKS) is a managed service that
#' makes it easy for you to run Kubernetes on Amazon Web Services without
#' needing to stand up or maintain your own Kubernetes control plane.
#' Kubernetes is an open-source system for automating the deployment,
#' scaling, and management of containerized applications.
#'
#' Amazon EKS runs up-to-date versions of the open-source Kubernetes
#' software, so you can use all the existing plugins and tooling from the
#' Kubernetes community. Applications running on Amazon EKS are fully
#' compatible with applications running on any standard Kubernetes
#' environment, whether running in on-premises data centers or public
#' clouds. This means that you can easily migrate any standard Kubernetes
#' application to Amazon EKS without any code modification required.
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- eks(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- eks()
#' # The following example creates an Amazon EKS cluster called prod.
#' svc$create_cluster(
#'   version = "1.10",
#'   name = "prod",
#'   clientRequestToken = "1d2129a1-3d38-460a-9756-e5b91fddb951",
#'   resourcesVpcConfig = list(
#'     securityGroupIds = list(
#'       "sg-6979fe18"
#'     ),
#'     subnetIds = list(
#'       "subnet-6782e71e",
#'       "subnet-e7e761ac"
#'     )
#'   ),
#'   roleArn = "arn:aws:iam::012345678910:role/eks-service-role-AWSServiceRole..."
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.compute:eks_associate_encryption_config]{associate_encryption_config} \tab Associate encryption configuration to an existing cluster\cr
#'  \link[paws.compute:eks_associate_identity_provider_config]{associate_identity_provider_config} \tab Associate an identity provider configuration to a cluster\cr
#'  \link[paws.compute:eks_create_addon]{create_addon} \tab Creates an Amazon EKS add-on\cr
#'  \link[paws.compute:eks_create_cluster]{create_cluster} \tab Creates an Amazon EKS control plane\cr
#'  \link[paws.compute:eks_create_fargate_profile]{create_fargate_profile} \tab Creates an Fargate profile for your Amazon EKS cluster\cr
#'  \link[paws.compute:eks_create_nodegroup]{create_nodegroup} \tab Creates a managed node group for an Amazon EKS cluster\cr
#'  \link[paws.compute:eks_delete_addon]{delete_addon} \tab Delete an Amazon EKS add-on\cr
#'  \link[paws.compute:eks_delete_cluster]{delete_cluster} \tab Deletes the Amazon EKS cluster control plane\cr
#'  \link[paws.compute:eks_delete_fargate_profile]{delete_fargate_profile} \tab Deletes an Fargate profile\cr
#'  \link[paws.compute:eks_delete_nodegroup]{delete_nodegroup} \tab Deletes an Amazon EKS node group for a cluster\cr
#'  \link[paws.compute:eks_deregister_cluster]{deregister_cluster} \tab Deregisters a connected cluster to remove it from the Amazon EKS control plane\cr
#'  \link[paws.compute:eks_describe_addon]{describe_addon} \tab Describes an Amazon EKS add-on\cr
#'  \link[paws.compute:eks_describe_addon_configuration]{describe_addon_configuration} \tab Returns configuration options\cr
#'  \link[paws.compute:eks_describe_addon_versions]{describe_addon_versions} \tab Describes the versions for an add-on\cr
#'  \link[paws.compute:eks_describe_cluster]{describe_cluster} \tab Returns descriptive information about an Amazon EKS cluster\cr
#'  \link[paws.compute:eks_describe_fargate_profile]{describe_fargate_profile} \tab Returns descriptive information about an Fargate profile\cr
#'  \link[paws.compute:eks_describe_identity_provider_config]{describe_identity_provider_config} \tab Returns descriptive information about an identity provider configuration\cr
#'  \link[paws.compute:eks_describe_nodegroup]{describe_nodegroup} \tab Returns descriptive information about an Amazon EKS node group\cr
#'  \link[paws.compute:eks_describe_update]{describe_update} \tab Returns descriptive information about an update against your Amazon EKS cluster or associated managed node group or Amazon EKS add-on\cr
#'  \link[paws.compute:eks_disassociate_identity_provider_config]{disassociate_identity_provider_config} \tab Disassociates an identity provider configuration from a cluster\cr
#'  \link[paws.compute:eks_list_addons]{list_addons} \tab Lists the available add-ons\cr
#'  \link[paws.compute:eks_list_clusters]{list_clusters} \tab Lists the Amazon EKS clusters in your Amazon Web Services account in the specified Region\cr
#'  \link[paws.compute:eks_list_fargate_profiles]{list_fargate_profiles} \tab Lists the Fargate profiles associated with the specified cluster in your Amazon Web Services account in the specified Region\cr
#'  \link[paws.compute:eks_list_identity_provider_configs]{list_identity_provider_configs} \tab A list of identity provider configurations\cr
#'  \link[paws.compute:eks_list_nodegroups]{list_nodegroups} \tab Lists the Amazon EKS managed node groups associated with the specified cluster in your Amazon Web Services account in the specified Region\cr
#'  \link[paws.compute:eks_list_tags_for_resource]{list_tags_for_resource} \tab List the tags for an Amazon EKS resource\cr
#'  \link[paws.compute:eks_list_updates]{list_updates} \tab Lists the updates associated with an Amazon EKS cluster or managed node group in your Amazon Web Services account, in the specified Region\cr
#'  \link[paws.compute:eks_register_cluster]{register_cluster} \tab Connects a Kubernetes cluster to the Amazon EKS control plane\cr
#'  \link[paws.compute:eks_tag_resource]{tag_resource} \tab Associates the specified tags to a resource with the specified resourceArn\cr
#'  \link[paws.compute:eks_untag_resource]{untag_resource} \tab Deletes specified tags from a resource\cr
#'  \link[paws.compute:eks_update_addon]{update_addon} \tab Updates an Amazon EKS add-on\cr
#'  \link[paws.compute:eks_update_cluster_config]{update_cluster_config} \tab Updates an Amazon EKS cluster configuration\cr
#'  \link[paws.compute:eks_update_cluster_version]{update_cluster_version} \tab Updates an Amazon EKS cluster to the specified Kubernetes version\cr
#'  \link[paws.compute:eks_update_nodegroup_config]{update_nodegroup_config} \tab Updates an Amazon EKS managed node group configuration\cr
#'  \link[paws.compute:eks_update_nodegroup_version]{update_nodegroup_version} \tab Updates the Kubernetes version or AMI version of an Amazon EKS managed node group
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname eks
#' @export
eks <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.compute::eks(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' AWS Elastic Beanstalk
#'
#' @description
#' AWS Elastic Beanstalk makes it easy for you to create, deploy, and
#' manage scalable, fault-tolerant applications running on the Amazon Web
#' Services cloud.
#'
#' For more information about this product, go to the [AWS Elastic
#' Beanstalk](https://aws.amazon.com/elasticbeanstalk/) details page. The
#' location of the latest AWS Elastic Beanstalk WSDL is
#' <https://elasticbeanstalk.s3.amazonaws.com/doc/2010-12-01/AWSElasticBeanstalk.wsdl>.
#' To install the Software Development Kits (SDKs), Integrated Development
#' Environment (IDE) Toolkits, and command line tools that enable you to
#' access the API, go to [Tools for Amazon Web
#' Services](https://aws.amazon.com/developer/tools/).
#'
#' **Endpoints**
#'
#' For a list of region-specific endpoints that AWS Elastic Beanstalk
#' supports, go to [Regions and
#' Endpoints](https://docs.aws.amazon.com/general/latest/gr/rande.html#elasticbeanstalk_region)
#' in the *Amazon Web Services Glossary*.
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- elasticbeanstalk(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- elasticbeanstalk()
#' # The following code aborts a running application version deployment for
#' # an environment named my-env:
#' svc$abort_environment_update(
#'   EnvironmentName = "my-env"
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.compute:elasticbeanstalk_abort_environment_update]{abort_environment_update} \tab Cancels in-progress environment configuration update or application version deployment\cr
#'  \link[paws.compute:elasticbeanstalk_apply_environment_managed_action]{apply_environment_managed_action} \tab Applies a scheduled managed action immediately\cr
#'  \link[paws.compute:elasticbeanstalk_associate_environment_operations_role]{associate_environment_operations_role} \tab Add or change the operations role used by an environment\cr
#'  \link[paws.compute:elasticbeanstalk_check_dns_availability]{check_dns_availability} \tab Checks if the specified CNAME is available\cr
#'  \link[paws.compute:elasticbeanstalk_compose_environments]{compose_environments} \tab Create or update a group of environments that each run a separate component of a single application\cr
#'  \link[paws.compute:elasticbeanstalk_create_application]{create_application} \tab Creates an application that has one configuration template named default and no application versions\cr
#'  \link[paws.compute:elasticbeanstalk_create_application_version]{create_application_version} \tab Creates an application version for the specified application\cr
#'  \link[paws.compute:elasticbeanstalk_create_configuration_template]{create_configuration_template} \tab Creates an AWS Elastic Beanstalk configuration template, associated with a specific Elastic Beanstalk application\cr
#'  \link[paws.compute:elasticbeanstalk_create_environment]{create_environment} \tab Launches an AWS Elastic Beanstalk environment for the specified application using the specified configuration\cr
#'  \link[paws.compute:elasticbeanstalk_create_platform_version]{create_platform_version} \tab Create a new version of your custom platform\cr
#'  \link[paws.compute:elasticbeanstalk_create_storage_location]{create_storage_location} \tab Creates a bucket in Amazon S3 to store application versions, logs, and other files used by Elastic Beanstalk environments\cr
#'  \link[paws.compute:elasticbeanstalk_delete_application]{delete_application} \tab Deletes the specified application along with all associated versions and configurations\cr
#'  \link[paws.compute:elasticbeanstalk_delete_application_version]{delete_application_version} \tab Deletes the specified version from the specified application\cr
#'  \link[paws.compute:elasticbeanstalk_delete_configuration_template]{delete_configuration_template} \tab Deletes the specified configuration template\cr
#'  \link[paws.compute:elasticbeanstalk_delete_environment_configuration]{delete_environment_configuration} \tab Deletes the draft configuration associated with the running environment\cr
#'  \link[paws.compute:elasticbeanstalk_delete_platform_version]{delete_platform_version} \tab Deletes the specified version of a custom platform\cr
#'  \link[paws.compute:elasticbeanstalk_describe_account_attributes]{describe_account_attributes} \tab Returns attributes related to AWS Elastic Beanstalk that are associated with the calling AWS account\cr
#'  \link[paws.compute:elasticbeanstalk_describe_applications]{describe_applications} \tab Returns the descriptions of existing applications\cr
#'  \link[paws.compute:elasticbeanstalk_describe_application_versions]{describe_application_versions} \tab Retrieve a list of application versions\cr
#'  \link[paws.compute:elasticbeanstalk_describe_configuration_options]{describe_configuration_options} \tab Describes the configuration options that are used in a particular configuration template or environment, or that a specified solution stack defines\cr
#'  \link[paws.compute:elasticbeanstalk_describe_configuration_settings]{describe_configuration_settings} \tab Returns a description of the settings for the specified configuration set, that is, either a configuration template or the configuration set associated with a running environment\cr
#'  \link[paws.compute:elasticbeanstalk_describe_environment_health]{describe_environment_health} \tab Returns information about the overall health of the specified environment\cr
#'  \link[paws.compute:elasticbeanstalk_describe_environment_managed_action_history]{describe_environment_managed_action_history} \tab Lists an environment's completed and failed managed actions\cr
#'  \link[paws.compute:elasticbeanstalk_describe_environment_managed_actions]{describe_environment_managed_actions} \tab Lists an environment's upcoming and in-progress managed actions\cr
#'  \link[paws.compute:elasticbeanstalk_describe_environment_resources]{describe_environment_resources} \tab Returns AWS resources for this environment\cr
#'  \link[paws.compute:elasticbeanstalk_describe_environments]{describe_environments} \tab Returns descriptions for existing environments\cr
#'  \link[paws.compute:elasticbeanstalk_describe_events]{describe_events} \tab Returns list of event descriptions matching criteria up to the last 6 weeks\cr
#'  \link[paws.compute:elasticbeanstalk_describe_instances_health]{describe_instances_health} \tab Retrieves detailed information about the health of instances in your AWS Elastic Beanstalk\cr
#'  \link[paws.compute:elasticbeanstalk_describe_platform_version]{describe_platform_version} \tab Describes a platform version\cr
#'  \link[paws.compute:elasticbeanstalk_disassociate_environment_operations_role]{disassociate_environment_operations_role} \tab Disassociate the operations role from an environment\cr
#'  \link[paws.compute:elasticbeanstalk_list_available_solution_stacks]{list_available_solution_stacks} \tab Returns a list of the available solution stack names, with the public version first and then in reverse chronological order\cr
#'  \link[paws.compute:elasticbeanstalk_list_platform_branches]{list_platform_branches} \tab Lists the platform branches available for your account in an AWS Region\cr
#'  \link[paws.compute:elasticbeanstalk_list_platform_versions]{list_platform_versions} \tab Lists the platform versions available for your account in an AWS Region\cr
#'  \link[paws.compute:elasticbeanstalk_list_tags_for_resource]{list_tags_for_resource} \tab Return the tags applied to an AWS Elastic Beanstalk resource\cr
#'  \link[paws.compute:elasticbeanstalk_rebuild_environment]{rebuild_environment} \tab Deletes and recreates all of the AWS resources (for example: the Auto Scaling group, load balancer, etc\cr
#'  \link[paws.compute:elasticbeanstalk_request_environment_info]{request_environment_info} \tab Initiates a request to compile the specified type of information of the deployed environment\cr
#'  \link[paws.compute:elasticbeanstalk_restart_app_server]{restart_app_server} \tab Causes the environment to restart the application container server running on each Amazon EC2 instance\cr
#'  \link[paws.compute:elasticbeanstalk_retrieve_environment_info]{retrieve_environment_info} \tab Retrieves the compiled information from a RequestEnvironmentInfo request\cr
#'  \link[paws.compute:elasticbeanstalk_swap_environment_cnam_es]{swap_environment_cnam_es} \tab Swaps the CNAMEs of two environments\cr
#'  \link[paws.compute:elasticbeanstalk_terminate_environment]{terminate_environment} \tab Terminates the specified environment\cr
#'  \link[paws.compute:elasticbeanstalk_update_application]{update_application} \tab Updates the specified application to have the specified properties\cr
#'  \link[paws.compute:elasticbeanstalk_update_application_resource_lifecycle]{update_application_resource_lifecycle} \tab Modifies lifecycle settings for an application\cr
#'  \link[paws.compute:elasticbeanstalk_update_application_version]{update_application_version} \tab Updates the specified application version to have the specified properties\cr
#'  \link[paws.compute:elasticbeanstalk_update_configuration_template]{update_configuration_template} \tab Updates the specified configuration template to have the specified properties or configuration option values\cr
#'  \link[paws.compute:elasticbeanstalk_update_environment]{update_environment} \tab Updates the environment description, deploys a new application version, updates the configuration settings to an entirely new configuration template, or updates select configuration option values in the running environment\cr
#'  \link[paws.compute:elasticbeanstalk_update_tags_for_resource]{update_tags_for_resource} \tab Update the list of tags applied to an AWS Elastic Beanstalk resource\cr
#'  \link[paws.compute:elasticbeanstalk_validate_configuration_settings]{validate_configuration_settings} \tab Takes a set of configuration settings and either a configuration template or environment, and determines whether those values are valid
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname elasticbeanstalk
#' @export
elasticbeanstalk <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.compute::elasticbeanstalk(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' Amazon EMR Containers
#'
#' @description
#' Amazon EMR on EKS provides a deployment option for Amazon EMR that
#' allows you to run open-source big data frameworks on Amazon Elastic
#' Kubernetes Service (Amazon EKS). With this deployment option, you can
#' focus on running analytics workloads while Amazon EMR on EKS builds,
#' configures, and manages containers for open-source applications. For
#' more information about Amazon EMR on EKS concepts and tasks, see [What
#' is shared
#' id="EMR-EKS"/\>](https://docs.aws.amazon.com/emr/latest/EMR-on-EKS-DevelopmentGuide/emr-eks.html).
#'
#' *Amazon EMR containers* is the API name for Amazon EMR on EKS. The
#' `emr-containers` prefix is used in the following scenarios:
#'
#' -   It is the prefix in the CLI commands for Amazon EMR on EKS. For
#'     example, `aws emr-containers start-job-run`.
#'
#' -   It is the prefix before IAM policy actions for Amazon EMR on EKS.
#'     For example, `"Action": [ "emr-containers:StartJobRun"]`. For more
#'     information, see [Policy actions for Amazon EMR on
#'     EKS](https://docs.aws.amazon.com/emr/latest/EMR-on-EKS-DevelopmentGuide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-actions).
#'
#' -   It is the prefix used in Amazon EMR on EKS service endpoints. For
#'     example, `emr-containers.us-east-2.amazonaws.com`. For more
#'     information, see [Amazon EMR on EKSService
#'     Endpoints](https://docs.aws.amazon.com/emr/latest/EMR-on-EKS-DevelopmentGuide/service-quotas.html#service-endpoints).
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- emrcontainers(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- emrcontainers()
#' svc$cancel_job_run(
#'   Foo = 123
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.compute:emrcontainers_cancel_job_run]{cancel_job_run} \tab Cancels a job run\cr
#'  \link[paws.compute:emrcontainers_create_job_template]{create_job_template} \tab Creates a job template\cr
#'  \link[paws.compute:emrcontainers_create_managed_endpoint]{create_managed_endpoint} \tab Creates a managed endpoint\cr
#'  \link[paws.compute:emrcontainers_create_virtual_cluster]{create_virtual_cluster} \tab Creates a virtual cluster\cr
#'  \link[paws.compute:emrcontainers_delete_job_template]{delete_job_template} \tab Deletes a job template\cr
#'  \link[paws.compute:emrcontainers_delete_managed_endpoint]{delete_managed_endpoint} \tab Deletes a managed endpoint\cr
#'  \link[paws.compute:emrcontainers_delete_virtual_cluster]{delete_virtual_cluster} \tab Deletes a virtual cluster\cr
#'  \link[paws.compute:emrcontainers_describe_job_run]{describe_job_run} \tab Displays detailed information about a job run\cr
#'  \link[paws.compute:emrcontainers_describe_job_template]{describe_job_template} \tab Displays detailed information about a specified job template\cr
#'  \link[paws.compute:emrcontainers_describe_managed_endpoint]{describe_managed_endpoint} \tab Displays detailed information about a managed endpoint\cr
#'  \link[paws.compute:emrcontainers_describe_virtual_cluster]{describe_virtual_cluster} \tab Displays detailed information about a specified virtual cluster\cr
#'  \link[paws.compute:emrcontainers_get_managed_endpoint_session_credentials]{get_managed_endpoint_session_credentials} \tab Generate a session token to connect to a managed endpoint\cr
#'  \link[paws.compute:emrcontainers_list_job_runs]{list_job_runs} \tab Lists job runs based on a set of parameters\cr
#'  \link[paws.compute:emrcontainers_list_job_templates]{list_job_templates} \tab Lists job templates based on a set of parameters\cr
#'  \link[paws.compute:emrcontainers_list_managed_endpoints]{list_managed_endpoints} \tab Lists managed endpoints based on a set of parameters\cr
#'  \link[paws.compute:emrcontainers_list_tags_for_resource]{list_tags_for_resource} \tab Lists the tags assigned to the resources\cr
#'  \link[paws.compute:emrcontainers_list_virtual_clusters]{list_virtual_clusters} \tab Lists information about the specified virtual cluster\cr
#'  \link[paws.compute:emrcontainers_start_job_run]{start_job_run} \tab Starts a job run\cr
#'  \link[paws.compute:emrcontainers_tag_resource]{tag_resource} \tab Assigns tags to resources\cr
#'  \link[paws.compute:emrcontainers_untag_resource]{untag_resource} \tab Removes tags from resources
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname emrcontainers
#' @export
emrcontainers <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.compute::emrcontainers(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' EMR Serverless
#'
#' @description
#' Amazon EMR Serverless is a new deployment option for Amazon EMR. Amazon
#' EMR Serverless provides a serverless runtime environment that simplifies
#' running analytics applications using the latest open source frameworks
#' such as Apache Spark and Apache Hive. With Amazon EMR Serverless, you
#' don’t have to configure, optimize, secure, or operate clusters to run
#' applications with these frameworks.
#'
#' The API reference to Amazon EMR Serverless is `emr-serverless`. The
#' `emr-serverless` prefix is used in the following scenarios:
#'
#' -   It is the prefix in the CLI commands for Amazon EMR Serverless. For
#'     example, `aws emr-serverless start-job-run`.
#'
#' -   It is the prefix before IAM policy actions for Amazon EMR
#'     Serverless. For example, `"Action": ["emr-serverless:StartJobRun"]`.
#'     For more information, see [Policy actions for Amazon EMR
#'     Serverless](https://docs.aws.amazon.com/emr/latest/EMR-Serverless-UserGuide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-actions).
#'
#' -   It is the prefix used in Amazon EMR Serverless service endpoints.
#'     For example, `emr-serverless.us-east-2.amazonaws.com`.
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- emrserverless(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- emrserverless()
#' svc$cancel_job_run(
#'   Foo = 123
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.compute:emrserverless_cancel_job_run]{cancel_job_run} \tab Cancels a job run\cr
#'  \link[paws.compute:emrserverless_create_application]{create_application} \tab Creates an application\cr
#'  \link[paws.compute:emrserverless_delete_application]{delete_application} \tab Deletes an application\cr
#'  \link[paws.compute:emrserverless_get_application]{get_application} \tab Displays detailed information about a specified application\cr
#'  \link[paws.compute:emrserverless_get_dashboard_for_job_run]{get_dashboard_for_job_run} \tab Creates and returns a URL that you can use to access the application UIs for a job run\cr
#'  \link[paws.compute:emrserverless_get_job_run]{get_job_run} \tab Displays detailed information about a job run\cr
#'  \link[paws.compute:emrserverless_list_applications]{list_applications} \tab Lists applications based on a set of parameters\cr
#'  \link[paws.compute:emrserverless_list_job_runs]{list_job_runs} \tab Lists job runs based on a set of parameters\cr
#'  \link[paws.compute:emrserverless_list_tags_for_resource]{list_tags_for_resource} \tab Lists the tags assigned to the resources\cr
#'  \link[paws.compute:emrserverless_start_application]{start_application} \tab Starts a specified application and initializes initial capacity if configured\cr
#'  \link[paws.compute:emrserverless_start_job_run]{start_job_run} \tab Starts a job run\cr
#'  \link[paws.compute:emrserverless_stop_application]{stop_application} \tab Stops a specified application and releases initial capacity if configured\cr
#'  \link[paws.compute:emrserverless_tag_resource]{tag_resource} \tab Assigns tags to resources\cr
#'  \link[paws.compute:emrserverless_untag_resource]{untag_resource} \tab Removes tags from resources\cr
#'  \link[paws.compute:emrserverless_update_application]{update_application} \tab Updates a specified application
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname emrserverless
#' @export
emrserverless <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.compute::emrserverless(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' EC2 Image Builder
#'
#' @description
#' EC2 Image Builder is a fully managed Amazon Web Services service that
#' makes it easier to automate the creation, management, and deployment of
#' customized, secure, and up-to-date "golden" server images that are
#' pre-installed and pre-configured with software and settings to meet
#' specific IT standards.
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- imagebuilder(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- imagebuilder()
#' svc$cancel_image_creation(
#'   Foo = 123
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.compute:imagebuilder_cancel_image_creation]{cancel_image_creation} \tab CancelImageCreation cancels the creation of Image\cr
#'  \link[paws.compute:imagebuilder_create_component]{create_component} \tab Creates a new component that can be used to build, validate, test, and assess your image\cr
#'  \link[paws.compute:imagebuilder_create_container_recipe]{create_container_recipe} \tab Creates a new container recipe\cr
#'  \link[paws.compute:imagebuilder_create_distribution_configuration]{create_distribution_configuration} \tab Creates a new distribution configuration\cr
#'  \link[paws.compute:imagebuilder_create_image]{create_image} \tab Creates a new image\cr
#'  \link[paws.compute:imagebuilder_create_image_pipeline]{create_image_pipeline} \tab Creates a new image pipeline\cr
#'  \link[paws.compute:imagebuilder_create_image_recipe]{create_image_recipe} \tab Creates a new image recipe\cr
#'  \link[paws.compute:imagebuilder_create_infrastructure_configuration]{create_infrastructure_configuration} \tab Creates a new infrastructure configuration\cr
#'  \link[paws.compute:imagebuilder_delete_component]{delete_component} \tab Deletes a component build version\cr
#'  \link[paws.compute:imagebuilder_delete_container_recipe]{delete_container_recipe} \tab Deletes a container recipe\cr
#'  \link[paws.compute:imagebuilder_delete_distribution_configuration]{delete_distribution_configuration} \tab Deletes a distribution configuration\cr
#'  \link[paws.compute:imagebuilder_delete_image]{delete_image} \tab Deletes an Image Builder image resource\cr
#'  \link[paws.compute:imagebuilder_delete_image_pipeline]{delete_image_pipeline} \tab Deletes an image pipeline\cr
#'  \link[paws.compute:imagebuilder_delete_image_recipe]{delete_image_recipe} \tab Deletes an image recipe\cr
#'  \link[paws.compute:imagebuilder_delete_infrastructure_configuration]{delete_infrastructure_configuration} \tab Deletes an infrastructure configuration\cr
#'  \link[paws.compute:imagebuilder_get_component]{get_component} \tab Gets a component object\cr
#'  \link[paws.compute:imagebuilder_get_component_policy]{get_component_policy} \tab Gets a component policy\cr
#'  \link[paws.compute:imagebuilder_get_container_recipe]{get_container_recipe} \tab Retrieves a container recipe\cr
#'  \link[paws.compute:imagebuilder_get_container_recipe_policy]{get_container_recipe_policy} \tab Retrieves the policy for a container recipe\cr
#'  \link[paws.compute:imagebuilder_get_distribution_configuration]{get_distribution_configuration} \tab Gets a distribution configuration\cr
#'  \link[paws.compute:imagebuilder_get_image]{get_image} \tab Gets an image\cr
#'  \link[paws.compute:imagebuilder_get_image_pipeline]{get_image_pipeline} \tab Gets an image pipeline\cr
#'  \link[paws.compute:imagebuilder_get_image_policy]{get_image_policy} \tab Gets an image policy\cr
#'  \link[paws.compute:imagebuilder_get_image_recipe]{get_image_recipe} \tab Gets an image recipe\cr
#'  \link[paws.compute:imagebuilder_get_image_recipe_policy]{get_image_recipe_policy} \tab Gets an image recipe policy\cr
#'  \link[paws.compute:imagebuilder_get_infrastructure_configuration]{get_infrastructure_configuration} \tab Gets an infrastructure configuration\cr
#'  \link[paws.compute:imagebuilder_get_workflow_execution]{get_workflow_execution} \tab Get the runtime information that was logged for a specific runtime instance of the workflow\cr
#'  \link[paws.compute:imagebuilder_get_workflow_step_execution]{get_workflow_step_execution} \tab Get the runtime information that was logged for a specific runtime instance of the workflow step\cr
#'  \link[paws.compute:imagebuilder_import_component]{import_component} \tab Imports a component and transforms its data into a component document\cr
#'  \link[paws.compute:imagebuilder_import_vm_image]{import_vm_image} \tab When you export your virtual machine (VM) from its virtualization environment, that process creates a set of one or more disk container files that act as snapshots of your VM’s environment, settings, and data\cr
#'  \link[paws.compute:imagebuilder_list_component_build_versions]{list_component_build_versions} \tab Returns the list of component build versions for the specified semantic version\cr
#'  \link[paws.compute:imagebuilder_list_components]{list_components} \tab Returns the list of components that can be filtered by name, or by using the listed filters to streamline results\cr
#'  \link[paws.compute:imagebuilder_list_container_recipes]{list_container_recipes} \tab Returns a list of container recipes\cr
#'  \link[paws.compute:imagebuilder_list_distribution_configurations]{list_distribution_configurations} \tab Returns a list of distribution configurations\cr
#'  \link[paws.compute:imagebuilder_list_image_build_versions]{list_image_build_versions} \tab Returns a list of image build versions\cr
#'  \link[paws.compute:imagebuilder_list_image_packages]{list_image_packages} \tab List the Packages that are associated with an Image Build Version, as determined by Amazon Web Services Systems Manager Inventory at build time\cr
#'  \link[paws.compute:imagebuilder_list_image_pipeline_images]{list_image_pipeline_images} \tab Returns a list of images created by the specified pipeline\cr
#'  \link[paws.compute:imagebuilder_list_image_pipelines]{list_image_pipelines} \tab Returns a list of image pipelines\cr
#'  \link[paws.compute:imagebuilder_list_image_recipes]{list_image_recipes} \tab Returns a list of image recipes\cr
#'  \link[paws.compute:imagebuilder_list_images]{list_images} \tab Returns the list of images that you have access to\cr
#'  \link[paws.compute:imagebuilder_list_image_scan_finding_aggregations]{list_image_scan_finding_aggregations} \tab Returns a list of image scan aggregations for your account\cr
#'  \link[paws.compute:imagebuilder_list_image_scan_findings]{list_image_scan_findings} \tab Returns a list of image scan findings for your account\cr
#'  \link[paws.compute:imagebuilder_list_infrastructure_configurations]{list_infrastructure_configurations} \tab Returns a list of infrastructure configurations\cr
#'  \link[paws.compute:imagebuilder_list_tags_for_resource]{list_tags_for_resource} \tab Returns the list of tags for the specified resource\cr
#'  \link[paws.compute:imagebuilder_list_workflow_executions]{list_workflow_executions} \tab Returns a list of workflow runtime instance metadata objects for a specific image build version\cr
#'  \link[paws.compute:imagebuilder_list_workflow_step_executions]{list_workflow_step_executions} \tab Shows runtime data for each step in a runtime instance of the workflow that you specify in the request\cr
#'  \link[paws.compute:imagebuilder_put_component_policy]{put_component_policy} \tab Applies a policy to a component\cr
#'  \link[paws.compute:imagebuilder_put_container_recipe_policy]{put_container_recipe_policy} \tab Applies a policy to a container image\cr
#'  \link[paws.compute:imagebuilder_put_image_policy]{put_image_policy} \tab Applies a policy to an image\cr
#'  \link[paws.compute:imagebuilder_put_image_recipe_policy]{put_image_recipe_policy} \tab Applies a policy to an image recipe\cr
#'  \link[paws.compute:imagebuilder_start_image_pipeline_execution]{start_image_pipeline_execution} \tab Manually triggers a pipeline to create an image\cr
#'  \link[paws.compute:imagebuilder_tag_resource]{tag_resource} \tab Adds a tag to a resource\cr
#'  \link[paws.compute:imagebuilder_untag_resource]{untag_resource} \tab Removes a tag from a resource\cr
#'  \link[paws.compute:imagebuilder_update_distribution_configuration]{update_distribution_configuration} \tab Updates a new distribution configuration\cr
#'  \link[paws.compute:imagebuilder_update_image_pipeline]{update_image_pipeline} \tab Updates an image pipeline\cr
#'  \link[paws.compute:imagebuilder_update_infrastructure_configuration]{update_infrastructure_configuration} \tab Updates a new infrastructure configuration
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname imagebuilder
#' @export
imagebuilder <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.compute::imagebuilder(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' AWS Lambda
#'
#' @description
#' Lambda
#'
#' **Overview**
#'
#' Lambda is a compute service that lets you run code without provisioning
#' or managing servers. Lambda runs your code on a high-availability
#' compute infrastructure and performs all of the administration of the
#' compute resources, including server and operating system maintenance,
#' capacity provisioning and automatic scaling, code monitoring and
#' logging. With Lambda, you can run code for virtually any type of
#' application or backend service. For more information about the Lambda
#' service, see [What is
#' Lambda](https://docs.aws.amazon.com/lambda/latest/dg/welcome.html) in
#' the **Lambda Developer Guide**.
#'
#' The *Lambda API Reference* provides information about each of the API
#' methods, including details about the parameters in each API request and
#' response.
#'
#' You can use Software Development Kits (SDKs), Integrated Development
#' Environment (IDE) Toolkits, and command line tools to access the API.
#' For installation instructions, see [Tools for Amazon Web
#' Services](https://aws.amazon.com/developer/tools/).
#'
#' For a list of Region-specific endpoints that Lambda supports, see Lambda
#' endpoints and quotas in the *Amazon Web Services General Reference.*.
#'
#' When making the API calls, you will need to authenticate your request by
#' providing a signature. Lambda supports signature version 4. For more
#' information, see [Signature Version 4 signing
#' process](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_aws-signing.html)
#' in the *Amazon Web Services General Reference.*.
#'
#' **CA certificates**
#'
#' Because Amazon Web Services SDKs use the CA certificates from your
#' computer, changes to the certificates on the Amazon Web Services servers
#' can cause connection failures when you attempt to use an SDK. You can
#' prevent these failures by keeping your computer's CA certificates and
#' operating system up-to-date. If you encounter this issue in a corporate
#' environment and do not manage your own computer, you might need to ask
#' an administrator to assist with the update process. The following list
#' shows minimum operating system and Java versions:
#'
#' -   Microsoft Windows versions that have updates from January 2005 or
#'     later installed contain at least one of the required CAs in their
#'     trust list.
#'
#' -   Mac OS X 10.4 with Java for Mac OS X 10.4 Release 5 (February 2007),
#'     Mac OS X 10.5 (October 2007), and later versions contain at least
#'     one of the required CAs in their trust list.
#'
#' -   Red Hat Enterprise Linux 5 (March 2007), 6, and 7 and CentOS 5, 6,
#'     and 7 all contain at least one of the required CAs in their default
#'     trusted CA list.
#'
#' -   Java 1.4.2_12 (May 2006), 5 Update 2 (March 2005), and all later
#'     versions, including Java 6 (December 2006), 7, and 8, contain at
#'     least one of the required CAs in their default trusted CA list.
#'
#' When accessing the Lambda management console or Lambda API endpoints,
#' whether through browsers or programmatically, you will need to ensure
#' your client machines support any of the following CAs:
#'
#' -   Amazon Root CA 1
#'
#' -   Starfield Services Root Certificate Authority - G2
#'
#' -   Starfield Class 2 Certification Authority
#'
#' Root certificates from the first two authorities are available from
#' [Amazon trust services](https://www.amazontrust.com/repository/), but
#' keeping your computer up-to-date is the more straightforward solution.
#' To learn more about ACM-provided certificates, see [Amazon Web Services
#' Certificate Manager
#' FAQs.](https://aws.amazon.com/certificate-manager/faqs/#certificates)
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- lambda(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- lambda()
#' svc$add_layer_version_permission(
#'   Foo = 123
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.compute:lambda_add_layer_version_permission]{add_layer_version_permission} \tab Adds permissions to the resource-based policy of a version of an Lambda layer\cr
#'  \link[paws.compute:lambda_add_permission]{add_permission} \tab Grants an Amazon Web Service, Amazon Web Services account, or Amazon Web Services organization permission to use a function\cr
#'  \link[paws.compute:lambda_create_alias]{create_alias} \tab Creates an alias for a Lambda function version\cr
#'  \link[paws.compute:lambda_create_code_signing_config]{create_code_signing_config} \tab Creates a code signing configuration\cr
#'  \link[paws.compute:lambda_create_event_source_mapping]{create_event_source_mapping} \tab Creates a mapping between an event source and an Lambda function\cr
#'  \link[paws.compute:lambda_create_function]{create_function} \tab Creates a Lambda function\cr
#'  \link[paws.compute:lambda_create_function_url_config]{create_function_url_config} \tab Creates a Lambda function URL with the specified configuration parameters\cr
#'  \link[paws.compute:lambda_delete_alias]{delete_alias} \tab Deletes a Lambda function alias\cr
#'  \link[paws.compute:lambda_delete_code_signing_config]{delete_code_signing_config} \tab Deletes the code signing configuration\cr
#'  \link[paws.compute:lambda_delete_event_source_mapping]{delete_event_source_mapping} \tab Deletes an event source mapping\cr
#'  \link[paws.compute:lambda_delete_function]{delete_function} \tab Deletes a Lambda function\cr
#'  \link[paws.compute:lambda_delete_function_code_signing_config]{delete_function_code_signing_config} \tab Removes the code signing configuration from the function\cr
#'  \link[paws.compute:lambda_delete_function_concurrency]{delete_function_concurrency} \tab Removes a concurrent execution limit from a function\cr
#'  \link[paws.compute:lambda_delete_function_event_invoke_config]{delete_function_event_invoke_config} \tab Deletes the configuration for asynchronous invocation for a function, version, or alias\cr
#'  \link[paws.compute:lambda_delete_function_url_config]{delete_function_url_config} \tab Deletes a Lambda function URL\cr
#'  \link[paws.compute:lambda_delete_layer_version]{delete_layer_version} \tab Deletes a version of an Lambda layer\cr
#'  \link[paws.compute:lambda_delete_provisioned_concurrency_config]{delete_provisioned_concurrency_config} \tab Deletes the provisioned concurrency configuration for a function\cr
#'  \link[paws.compute:lambda_get_account_settings]{get_account_settings} \tab Retrieves details about your account's limits and usage in an Amazon Web Services Region\cr
#'  \link[paws.compute:lambda_get_alias]{get_alias} \tab Returns details about a Lambda function alias\cr
#'  \link[paws.compute:lambda_get_code_signing_config]{get_code_signing_config} \tab Returns information about the specified code signing configuration\cr
#'  \link[paws.compute:lambda_get_event_source_mapping]{get_event_source_mapping} \tab Returns details about an event source mapping\cr
#'  \link[paws.compute:lambda_get_function]{get_function} \tab Returns information about the function or function version, with a link to download the deployment package that's valid for 10 minutes\cr
#'  \link[paws.compute:lambda_get_function_code_signing_config]{get_function_code_signing_config} \tab Returns the code signing configuration for the specified function\cr
#'  \link[paws.compute:lambda_get_function_concurrency]{get_function_concurrency} \tab Returns details about the reserved concurrency configuration for a function\cr
#'  \link[paws.compute:lambda_get_function_configuration]{get_function_configuration} \tab Returns the version-specific settings of a Lambda function or version\cr
#'  \link[paws.compute:lambda_get_function_event_invoke_config]{get_function_event_invoke_config} \tab Retrieves the configuration for asynchronous invocation for a function, version, or alias\cr
#'  \link[paws.compute:lambda_get_function_url_config]{get_function_url_config} \tab Returns details about a Lambda function URL\cr
#'  \link[paws.compute:lambda_get_layer_version]{get_layer_version} \tab Returns information about a version of an Lambda layer, with a link to download the layer archive that's valid for 10 minutes\cr
#'  \link[paws.compute:lambda_get_layer_version_by_arn]{get_layer_version_by_arn} \tab Returns information about a version of an Lambda layer, with a link to download the layer archive that's valid for 10 minutes\cr
#'  \link[paws.compute:lambda_get_layer_version_policy]{get_layer_version_policy} \tab Returns the permission policy for a version of an Lambda layer\cr
#'  \link[paws.compute:lambda_get_policy]{get_policy} \tab Returns the resource-based IAM policy for a function, version, or alias\cr
#'  \link[paws.compute:lambda_get_provisioned_concurrency_config]{get_provisioned_concurrency_config} \tab Retrieves the provisioned concurrency configuration for a function's alias or version\cr
#'  \link[paws.compute:lambda_get_runtime_management_config]{get_runtime_management_config} \tab Retrieves the runtime management configuration for a function's version\cr
#'  \link[paws.compute:lambda_invoke]{invoke} \tab Invokes a Lambda function\cr
#'  \link[paws.compute:lambda_invoke_async]{invoke_async} \tab For asynchronous function invocation, use Invoke\cr
#'  \link[paws.compute:lambda_invoke_with_response_stream]{invoke_with_response_stream} \tab Configure your Lambda functions to stream response payloads back to clients\cr
#'  \link[paws.compute:lambda_list_aliases]{list_aliases} \tab Returns a list of aliases for a Lambda function\cr
#'  \link[paws.compute:lambda_list_code_signing_configs]{list_code_signing_configs} \tab Returns a list of code signing configurations\cr
#'  \link[paws.compute:lambda_list_event_source_mappings]{list_event_source_mappings} \tab Lists event source mappings\cr
#'  \link[paws.compute:lambda_list_function_event_invoke_configs]{list_function_event_invoke_configs} \tab Retrieves a list of configurations for asynchronous invocation for a function\cr
#'  \link[paws.compute:lambda_list_functions]{list_functions} \tab Returns a list of Lambda functions, with the version-specific configuration of each\cr
#'  \link[paws.compute:lambda_list_functions_by_code_signing_config]{list_functions_by_code_signing_config} \tab List the functions that use the specified code signing configuration\cr
#'  \link[paws.compute:lambda_list_function_url_configs]{list_function_url_configs} \tab Returns a list of Lambda function URLs for the specified function\cr
#'  \link[paws.compute:lambda_list_layers]{list_layers} \tab Lists Lambda layers and shows information about the latest version of each\cr
#'  \link[paws.compute:lambda_list_layer_versions]{list_layer_versions} \tab Lists the versions of an Lambda layer\cr
#'  \link[paws.compute:lambda_list_provisioned_concurrency_configs]{list_provisioned_concurrency_configs} \tab Retrieves a list of provisioned concurrency configurations for a function\cr
#'  \link[paws.compute:lambda_list_tags]{list_tags} \tab Returns a function's tags\cr
#'  \link[paws.compute:lambda_list_versions_by_function]{list_versions_by_function} \tab Returns a list of versions, with the version-specific configuration of each\cr
#'  \link[paws.compute:lambda_publish_layer_version]{publish_layer_version} \tab Creates an Lambda layer from a ZIP archive\cr
#'  \link[paws.compute:lambda_publish_version]{publish_version} \tab Creates a version from the current code and configuration of a function\cr
#'  \link[paws.compute:lambda_put_function_code_signing_config]{put_function_code_signing_config} \tab Update the code signing configuration for the function\cr
#'  \link[paws.compute:lambda_put_function_concurrency]{put_function_concurrency} \tab Sets the maximum number of simultaneous executions for a function, and reserves capacity for that concurrency level\cr
#'  \link[paws.compute:lambda_put_function_event_invoke_config]{put_function_event_invoke_config} \tab Configures options for asynchronous invocation on a function, version, or alias\cr
#'  \link[paws.compute:lambda_put_provisioned_concurrency_config]{put_provisioned_concurrency_config} \tab Adds a provisioned concurrency configuration to a function's alias or version\cr
#'  \link[paws.compute:lambda_put_runtime_management_config]{put_runtime_management_config} \tab Sets the runtime management configuration for a function's version\cr
#'  \link[paws.compute:lambda_remove_layer_version_permission]{remove_layer_version_permission} \tab Removes a statement from the permissions policy for a version of an Lambda layer\cr
#'  \link[paws.compute:lambda_remove_permission]{remove_permission} \tab Revokes function-use permission from an Amazon Web Service or another Amazon Web Services account\cr
#'  \link[paws.compute:lambda_tag_resource]{tag_resource} \tab Adds tags to a function\cr
#'  \link[paws.compute:lambda_untag_resource]{untag_resource} \tab Removes tags from a function\cr
#'  \link[paws.compute:lambda_update_alias]{update_alias} \tab Updates the configuration of a Lambda function alias\cr
#'  \link[paws.compute:lambda_update_code_signing_config]{update_code_signing_config} \tab Update the code signing configuration\cr
#'  \link[paws.compute:lambda_update_event_source_mapping]{update_event_source_mapping} \tab Updates an event source mapping\cr
#'  \link[paws.compute:lambda_update_function_code]{update_function_code} \tab Updates a Lambda function's code\cr
#'  \link[paws.compute:lambda_update_function_configuration]{update_function_configuration} \tab Modify the version-specific settings of a Lambda function\cr
#'  \link[paws.compute:lambda_update_function_event_invoke_config]{update_function_event_invoke_config} \tab Updates the configuration for asynchronous invocation for a function, version, or alias\cr
#'  \link[paws.compute:lambda_update_function_url_config]{update_function_url_config} \tab Updates the configuration for a Lambda function URL
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname lambda
#' @export
lambda <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.compute::lambda(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' Amazon Lightsail
#'
#' @description
#' Amazon Lightsail is the easiest way to get started with Amazon Web
#' Services (Amazon Web Services) for developers who need to build websites
#' or web applications. It includes everything you need to launch your
#' project quickly - instances (virtual private servers), container
#' services, storage buckets, managed databases, SSD-based block storage,
#' static IP addresses, load balancers, content delivery network (CDN)
#' distributions, DNS management of registered domains, and resource
#' snapshots (backups) - for a low, predictable monthly price.
#'
#' You can manage your Lightsail resources using the Lightsail console,
#' Lightsail API, Command Line Interface (CLI), or SDKs. For more
#' information about Lightsail concepts and tasks, see the Amazon Lightsail
#' Developer Guide.
#'
#' This API Reference provides detailed information about the actions, data
#' types, parameters, and errors of the Lightsail service. For more
#' information about the supported Amazon Web Services Regions, endpoints,
#' and service quotas of the Lightsail service, see [Amazon Lightsail
#' Endpoints and
#' Quotas](https://docs.aws.amazon.com/general/latest/gr/lightsail.html) in
#' the *Amazon Web Services General Reference*.
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- lightsail(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- lightsail()
#' svc$allocate_static_ip(
#'   Foo = 123
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.compute:lightsail_allocate_static_ip]{allocate_static_ip} \tab Allocates a static IP address\cr
#'  \link[paws.compute:lightsail_attach_certificate_to_distribution]{attach_certificate_to_distribution} \tab Attaches an SSL/TLS certificate to your Amazon Lightsail content delivery network (CDN) distribution\cr
#'  \link[paws.compute:lightsail_attach_disk]{attach_disk} \tab Attaches a block storage disk to a running or stopped Lightsail instance and exposes it to the instance with the specified disk name\cr
#'  \link[paws.compute:lightsail_attach_instances_to_load_balancer]{attach_instances_to_load_balancer} \tab Attaches one or more Lightsail instances to a load balancer\cr
#'  \link[paws.compute:lightsail_attach_load_balancer_tls_certificate]{attach_load_balancer_tls_certificate} \tab Attaches a Transport Layer Security (TLS) certificate to your load balancer\cr
#'  \link[paws.compute:lightsail_attach_static_ip]{attach_static_ip} \tab Attaches a static IP address to a specific Amazon Lightsail instance\cr
#'  \link[paws.compute:lightsail_close_instance_public_ports]{close_instance_public_ports} \tab Closes ports for a specific Amazon Lightsail instance\cr
#'  \link[paws.compute:lightsail_copy_snapshot]{copy_snapshot} \tab Copies a manual snapshot of an instance or disk as another manual snapshot, or copies an automatic snapshot of an instance or disk as a manual snapshot\cr
#'  \link[paws.compute:lightsail_create_bucket]{create_bucket} \tab Creates an Amazon Lightsail bucket\cr
#'  \link[paws.compute:lightsail_create_bucket_access_key]{create_bucket_access_key} \tab Creates a new access key for the specified Amazon Lightsail bucket\cr
#'  \link[paws.compute:lightsail_create_certificate]{create_certificate} \tab Creates an SSL/TLS certificate for an Amazon Lightsail content delivery network (CDN) distribution and a container service\cr
#'  \link[paws.compute:lightsail_create_cloud_formation_stack]{create_cloud_formation_stack} \tab Creates an AWS CloudFormation stack, which creates a new Amazon EC2 instance from an exported Amazon Lightsail snapshot\cr
#'  \link[paws.compute:lightsail_create_contact_method]{create_contact_method} \tab Creates an email or SMS text message contact method\cr
#'  \link[paws.compute:lightsail_create_container_service]{create_container_service} \tab Creates an Amazon Lightsail container service\cr
#'  \link[paws.compute:lightsail_create_container_service_deployment]{create_container_service_deployment} \tab Creates a deployment for your Amazon Lightsail container service\cr
#'  \link[paws.compute:lightsail_create_container_service_registry_login]{create_container_service_registry_login} \tab Creates a temporary set of log in credentials that you can use to log in to the Docker process on your local machine\cr
#'  \link[paws.compute:lightsail_create_disk]{create_disk} \tab Creates a block storage disk that can be attached to an Amazon Lightsail instance in the same Availability Zone (e\cr
#'  \link[paws.compute:lightsail_create_disk_from_snapshot]{create_disk_from_snapshot} \tab Creates a block storage disk from a manual or automatic snapshot of a disk\cr
#'  \link[paws.compute:lightsail_create_disk_snapshot]{create_disk_snapshot} \tab Creates a snapshot of a block storage disk\cr
#'  \link[paws.compute:lightsail_create_distribution]{create_distribution} \tab Creates an Amazon Lightsail content delivery network (CDN) distribution\cr
#'  \link[paws.compute:lightsail_create_domain]{create_domain} \tab Creates a domain resource for the specified domain (e\cr
#'  \link[paws.compute:lightsail_create_domain_entry]{create_domain_entry} \tab Creates one of the following domain name system (DNS) records in a domain DNS zone: Address (A), canonical name (CNAME), mail exchanger (MX), name server (NS), start of authority (SOA), service locator (SRV), or text (TXT)\cr
#'  \link[paws.compute:lightsail_create_gui_session_access_details]{create_gui_session_access_details} \tab Creates two URLs that are used to access a virtual computer’s graphical user interface (GUI) session\cr
#'  \link[paws.compute:lightsail_create_instances]{create_instances} \tab Creates one or more Amazon Lightsail instances\cr
#'  \link[paws.compute:lightsail_create_instances_from_snapshot]{create_instances_from_snapshot} \tab Creates one or more new instances from a manual or automatic snapshot of an instance\cr
#'  \link[paws.compute:lightsail_create_instance_snapshot]{create_instance_snapshot} \tab Creates a snapshot of a specific virtual private server, or instance\cr
#'  \link[paws.compute:lightsail_create_key_pair]{create_key_pair} \tab Creates a custom SSH key pair that you can use with an Amazon Lightsail instance\cr
#'  \link[paws.compute:lightsail_create_load_balancer]{create_load_balancer} \tab Creates a Lightsail load balancer\cr
#'  \link[paws.compute:lightsail_create_load_balancer_tls_certificate]{create_load_balancer_tls_certificate} \tab Creates an SSL/TLS certificate for an Amazon Lightsail load balancer\cr
#'  \link[paws.compute:lightsail_create_relational_database]{create_relational_database} \tab Creates a new database in Amazon Lightsail\cr
#'  \link[paws.compute:lightsail_create_relational_database_from_snapshot]{create_relational_database_from_snapshot} \tab Creates a new database from an existing database snapshot in Amazon Lightsail\cr
#'  \link[paws.compute:lightsail_create_relational_database_snapshot]{create_relational_database_snapshot} \tab Creates a snapshot of your database in Amazon Lightsail\cr
#'  \link[paws.compute:lightsail_delete_alarm]{delete_alarm} \tab Deletes an alarm\cr
#'  \link[paws.compute:lightsail_delete_auto_snapshot]{delete_auto_snapshot} \tab Deletes an automatic snapshot of an instance or disk\cr
#'  \link[paws.compute:lightsail_delete_bucket]{delete_bucket} \tab Deletes a Amazon Lightsail bucket\cr
#'  \link[paws.compute:lightsail_delete_bucket_access_key]{delete_bucket_access_key} \tab Deletes an access key for the specified Amazon Lightsail bucket\cr
#'  \link[paws.compute:lightsail_delete_certificate]{delete_certificate} \tab Deletes an SSL/TLS certificate for your Amazon Lightsail content delivery network (CDN) distribution\cr
#'  \link[paws.compute:lightsail_delete_contact_method]{delete_contact_method} \tab Deletes a contact method\cr
#'  \link[paws.compute:lightsail_delete_container_image]{delete_container_image} \tab Deletes a container image that is registered to your Amazon Lightsail container service\cr
#'  \link[paws.compute:lightsail_delete_container_service]{delete_container_service} \tab Deletes your Amazon Lightsail container service\cr
#'  \link[paws.compute:lightsail_delete_disk]{delete_disk} \tab Deletes the specified block storage disk\cr
#'  \link[paws.compute:lightsail_delete_disk_snapshot]{delete_disk_snapshot} \tab Deletes the specified disk snapshot\cr
#'  \link[paws.compute:lightsail_delete_distribution]{delete_distribution} \tab Deletes your Amazon Lightsail content delivery network (CDN) distribution\cr
#'  \link[paws.compute:lightsail_delete_domain]{delete_domain} \tab Deletes the specified domain recordset and all of its domain records\cr
#'  \link[paws.compute:lightsail_delete_domain_entry]{delete_domain_entry} \tab Deletes a specific domain entry\cr
#'  \link[paws.compute:lightsail_delete_instance]{delete_instance} \tab Deletes an Amazon Lightsail instance\cr
#'  \link[paws.compute:lightsail_delete_instance_snapshot]{delete_instance_snapshot} \tab Deletes a specific snapshot of a virtual private server (or instance)\cr
#'  \link[paws.compute:lightsail_delete_key_pair]{delete_key_pair} \tab Deletes the specified key pair by removing the public key from Amazon Lightsail\cr
#'  \link[paws.compute:lightsail_delete_known_host_keys]{delete_known_host_keys} \tab Deletes the known host key or certificate used by the Amazon Lightsail browser-based SSH or RDP clients to authenticate an instance\cr
#'  \link[paws.compute:lightsail_delete_load_balancer]{delete_load_balancer} \tab Deletes a Lightsail load balancer and all its associated SSL/TLS certificates\cr
#'  \link[paws.compute:lightsail_delete_load_balancer_tls_certificate]{delete_load_balancer_tls_certificate} \tab Deletes an SSL/TLS certificate associated with a Lightsail load balancer\cr
#'  \link[paws.compute:lightsail_delete_relational_database]{delete_relational_database} \tab Deletes a database in Amazon Lightsail\cr
#'  \link[paws.compute:lightsail_delete_relational_database_snapshot]{delete_relational_database_snapshot} \tab Deletes a database snapshot in Amazon Lightsail\cr
#'  \link[paws.compute:lightsail_detach_certificate_from_distribution]{detach_certificate_from_distribution} \tab Detaches an SSL/TLS certificate from your Amazon Lightsail content delivery network (CDN) distribution\cr
#'  \link[paws.compute:lightsail_detach_disk]{detach_disk} \tab Detaches a stopped block storage disk from a Lightsail instance\cr
#'  \link[paws.compute:lightsail_detach_instances_from_load_balancer]{detach_instances_from_load_balancer} \tab Detaches the specified instances from a Lightsail load balancer\cr
#'  \link[paws.compute:lightsail_detach_static_ip]{detach_static_ip} \tab Detaches a static IP from the Amazon Lightsail instance to which it is attached\cr
#'  \link[paws.compute:lightsail_disable_add_on]{disable_add_on} \tab Disables an add-on for an Amazon Lightsail resource\cr
#'  \link[paws.compute:lightsail_download_default_key_pair]{download_default_key_pair} \tab Downloads the regional Amazon Lightsail default key pair\cr
#'  \link[paws.compute:lightsail_enable_add_on]{enable_add_on} \tab Enables or modifies an add-on for an Amazon Lightsail resource\cr
#'  \link[paws.compute:lightsail_export_snapshot]{export_snapshot} \tab Exports an Amazon Lightsail instance or block storage disk snapshot to Amazon Elastic Compute Cloud (Amazon EC2)\cr
#'  \link[paws.compute:lightsail_get_active_names]{get_active_names} \tab Returns the names of all active (not deleted) resources\cr
#'  \link[paws.compute:lightsail_get_alarms]{get_alarms} \tab Returns information about the configured alarms\cr
#'  \link[paws.compute:lightsail_get_auto_snapshots]{get_auto_snapshots} \tab Returns the available automatic snapshots for an instance or disk\cr
#'  \link[paws.compute:lightsail_get_blueprints]{get_blueprints} \tab Returns the list of available instance images, or blueprints\cr
#'  \link[paws.compute:lightsail_get_bucket_access_keys]{get_bucket_access_keys} \tab Returns the existing access key IDs for the specified Amazon Lightsail bucket\cr
#'  \link[paws.compute:lightsail_get_bucket_bundles]{get_bucket_bundles} \tab Returns the bundles that you can apply to a Amazon Lightsail bucket\cr
#'  \link[paws.compute:lightsail_get_bucket_metric_data]{get_bucket_metric_data} \tab Returns the data points of a specific metric for an Amazon Lightsail bucket\cr
#'  \link[paws.compute:lightsail_get_buckets]{get_buckets} \tab Returns information about one or more Amazon Lightsail buckets\cr
#'  \link[paws.compute:lightsail_get_bundles]{get_bundles} \tab Returns the bundles that you can apply to an Amazon Lightsail instance when you create it\cr
#'  \link[paws.compute:lightsail_get_certificates]{get_certificates} \tab Returns information about one or more Amazon Lightsail SSL/TLS certificates\cr
#'  \link[paws.compute:lightsail_get_cloud_formation_stack_records]{get_cloud_formation_stack_records} \tab Returns the CloudFormation stack record created as a result of the create cloud formation stack operation\cr
#'  \link[paws.compute:lightsail_get_contact_methods]{get_contact_methods} \tab Returns information about the configured contact methods\cr
#'  \link[paws.compute:lightsail_get_container_api_metadata]{get_container_api_metadata} \tab Returns information about Amazon Lightsail containers, such as the current version of the Lightsail Control (lightsailctl) plugin\cr
#'  \link[paws.compute:lightsail_get_container_images]{get_container_images} \tab Returns the container images that are registered to your Amazon Lightsail container service\cr
#'  \link[paws.compute:lightsail_get_container_log]{get_container_log} \tab Returns the log events of a container of your Amazon Lightsail container service\cr
#'  \link[paws.compute:lightsail_get_container_service_deployments]{get_container_service_deployments} \tab Returns the deployments for your Amazon Lightsail container service\cr
#'  \link[paws.compute:lightsail_get_container_service_metric_data]{get_container_service_metric_data} \tab Returns the data points of a specific metric of your Amazon Lightsail container service\cr
#'  \link[paws.compute:lightsail_get_container_service_powers]{get_container_service_powers} \tab Returns the list of powers that can be specified for your Amazon Lightsail container services\cr
#'  \link[paws.compute:lightsail_get_container_services]{get_container_services} \tab Returns information about one or more of your Amazon Lightsail container services\cr
#'  \link[paws.compute:lightsail_get_cost_estimate]{get_cost_estimate} \tab Retrieves information about the cost estimate for a specified resource\cr
#'  \link[paws.compute:lightsail_get_disk]{get_disk} \tab Returns information about a specific block storage disk\cr
#'  \link[paws.compute:lightsail_get_disks]{get_disks} \tab Returns information about all block storage disks in your AWS account and region\cr
#'  \link[paws.compute:lightsail_get_disk_snapshot]{get_disk_snapshot} \tab Returns information about a specific block storage disk snapshot\cr
#'  \link[paws.compute:lightsail_get_disk_snapshots]{get_disk_snapshots} \tab Returns information about all block storage disk snapshots in your AWS account and region\cr
#'  \link[paws.compute:lightsail_get_distribution_bundles]{get_distribution_bundles} \tab Returns the bundles that can be applied to your Amazon Lightsail content delivery network (CDN) distributions\cr
#'  \link[paws.compute:lightsail_get_distribution_latest_cache_reset]{get_distribution_latest_cache_reset} \tab Returns the timestamp and status of the last cache reset of a specific Amazon Lightsail content delivery network (CDN) distribution\cr
#'  \link[paws.compute:lightsail_get_distribution_metric_data]{get_distribution_metric_data} \tab Returns the data points of a specific metric for an Amazon Lightsail content delivery network (CDN) distribution\cr
#'  \link[paws.compute:lightsail_get_distributions]{get_distributions} \tab Returns information about one or more of your Amazon Lightsail content delivery network (CDN) distributions\cr
#'  \link[paws.compute:lightsail_get_domain]{get_domain} \tab Returns information about a specific domain recordset\cr
#'  \link[paws.compute:lightsail_get_domains]{get_domains} \tab Returns a list of all domains in the user's account\cr
#'  \link[paws.compute:lightsail_get_export_snapshot_records]{get_export_snapshot_records} \tab Returns all export snapshot records created as a result of the export snapshot operation\cr
#'  \link[paws.compute:lightsail_get_instance]{get_instance} \tab Returns information about a specific Amazon Lightsail instance, which is a virtual private server\cr
#'  \link[paws.compute:lightsail_get_instance_access_details]{get_instance_access_details} \tab Returns temporary SSH keys you can use to connect to a specific virtual private server, or instance\cr
#'  \link[paws.compute:lightsail_get_instance_metric_data]{get_instance_metric_data} \tab Returns the data points for the specified Amazon Lightsail instance metric, given an instance name\cr
#'  \link[paws.compute:lightsail_get_instance_port_states]{get_instance_port_states} \tab Returns the firewall port states for a specific Amazon Lightsail instance, the IP addresses allowed to connect to the instance through the ports, and the protocol\cr
#'  \link[paws.compute:lightsail_get_instances]{get_instances} \tab Returns information about all Amazon Lightsail virtual private servers, or instances\cr
#'  \link[paws.compute:lightsail_get_instance_snapshot]{get_instance_snapshot} \tab Returns information about a specific instance snapshot\cr
#'  \link[paws.compute:lightsail_get_instance_snapshots]{get_instance_snapshots} \tab Returns all instance snapshots for the user's account\cr
#'  \link[paws.compute:lightsail_get_instance_state]{get_instance_state} \tab Returns the state of a specific instance\cr
#'  \link[paws.compute:lightsail_get_key_pair]{get_key_pair} \tab Returns information about a specific key pair\cr
#'  \link[paws.compute:lightsail_get_key_pairs]{get_key_pairs} \tab Returns information about all key pairs in the user's account\cr
#'  \link[paws.compute:lightsail_get_load_balancer]{get_load_balancer} \tab Returns information about the specified Lightsail load balancer\cr
#'  \link[paws.compute:lightsail_get_load_balancer_metric_data]{get_load_balancer_metric_data} \tab Returns information about health metrics for your Lightsail load balancer\cr
#'  \link[paws.compute:lightsail_get_load_balancers]{get_load_balancers} \tab Returns information about all load balancers in an account\cr
#'  \link[paws.compute:lightsail_get_load_balancer_tls_certificates]{get_load_balancer_tls_certificates} \tab Returns information about the TLS certificates that are associated with the specified Lightsail load balancer\cr
#'  \link[paws.compute:lightsail_get_load_balancer_tls_policies]{get_load_balancer_tls_policies} \tab Returns a list of TLS security policies that you can apply to Lightsail load balancers\cr
#'  \link[paws.compute:lightsail_get_operation]{get_operation} \tab Returns information about a specific operation\cr
#'  \link[paws.compute:lightsail_get_operations]{get_operations} \tab Returns information about all operations\cr
#'  \link[paws.compute:lightsail_get_operations_for_resource]{get_operations_for_resource} \tab Gets operations for a specific resource (e\cr
#'  \link[paws.compute:lightsail_get_regions]{get_regions} \tab Returns a list of all valid regions for Amazon Lightsail\cr
#'  \link[paws.compute:lightsail_get_relational_database]{get_relational_database} \tab Returns information about a specific database in Amazon Lightsail\cr
#'  \link[paws.compute:lightsail_get_relational_database_blueprints]{get_relational_database_blueprints} \tab Returns a list of available database blueprints in Amazon Lightsail\cr
#'  \link[paws.compute:lightsail_get_relational_database_bundles]{get_relational_database_bundles} \tab Returns the list of bundles that are available in Amazon Lightsail\cr
#'  \link[paws.compute:lightsail_get_relational_database_events]{get_relational_database_events} \tab Returns a list of events for a specific database in Amazon Lightsail\cr
#'  \link[paws.compute:lightsail_get_relational_database_log_events]{get_relational_database_log_events} \tab Returns a list of log events for a database in Amazon Lightsail\cr
#'  \link[paws.compute:lightsail_get_relational_database_log_streams]{get_relational_database_log_streams} \tab Returns a list of available log streams for a specific database in Amazon Lightsail\cr
#'  \link[paws.compute:lightsail_get_relational_database_master_user_password]{get_relational_database_master_user_password} \tab Returns the current, previous, or pending versions of the master user password for a Lightsail database\cr
#'  \link[paws.compute:lightsail_get_relational_database_metric_data]{get_relational_database_metric_data} \tab Returns the data points of the specified metric for a database in Amazon Lightsail\cr
#'  \link[paws.compute:lightsail_get_relational_database_parameters]{get_relational_database_parameters} \tab Returns all of the runtime parameters offered by the underlying database software, or engine, for a specific database in Amazon Lightsail\cr
#'  \link[paws.compute:lightsail_get_relational_databases]{get_relational_databases} \tab Returns information about all of your databases in Amazon Lightsail\cr
#'  \link[paws.compute:lightsail_get_relational_database_snapshot]{get_relational_database_snapshot} \tab Returns information about a specific database snapshot in Amazon Lightsail\cr
#'  \link[paws.compute:lightsail_get_relational_database_snapshots]{get_relational_database_snapshots} \tab Returns information about all of your database snapshots in Amazon Lightsail\cr
#'  \link[paws.compute:lightsail_get_static_ip]{get_static_ip} \tab Returns information about an Amazon Lightsail static IP\cr
#'  \link[paws.compute:lightsail_get_static_ips]{get_static_ips} \tab Returns information about all static IPs in the user's account\cr
#'  \link[paws.compute:lightsail_import_key_pair]{import_key_pair} \tab Imports a public SSH key from a specific key pair\cr
#'  \link[paws.compute:lightsail_is_vpc_peered]{is_vpc_peered} \tab Returns a Boolean value indicating whether your Lightsail VPC is peered\cr
#'  \link[paws.compute:lightsail_open_instance_public_ports]{open_instance_public_ports} \tab Opens ports for a specific Amazon Lightsail instance, and specifies the IP addresses allowed to connect to the instance through the ports, and the protocol\cr
#'  \link[paws.compute:lightsail_peer_vpc]{peer_vpc} \tab Peers the Lightsail VPC with the user's default VPC\cr
#'  \link[paws.compute:lightsail_put_alarm]{put_alarm} \tab Creates or updates an alarm, and associates it with the specified metric\cr
#'  \link[paws.compute:lightsail_put_instance_public_ports]{put_instance_public_ports} \tab Opens ports for a specific Amazon Lightsail instance, and specifies the IP addresses allowed to connect to the instance through the ports, and the protocol\cr
#'  \link[paws.compute:lightsail_reboot_instance]{reboot_instance} \tab Restarts a specific instance\cr
#'  \link[paws.compute:lightsail_reboot_relational_database]{reboot_relational_database} \tab Restarts a specific database in Amazon Lightsail\cr
#'  \link[paws.compute:lightsail_register_container_image]{register_container_image} \tab Registers a container image to your Amazon Lightsail container service\cr
#'  \link[paws.compute:lightsail_release_static_ip]{release_static_ip} \tab Deletes a specific static IP from your account\cr
#'  \link[paws.compute:lightsail_reset_distribution_cache]{reset_distribution_cache} \tab Deletes currently cached content from your Amazon Lightsail content delivery network (CDN) distribution\cr
#'  \link[paws.compute:lightsail_send_contact_method_verification]{send_contact_method_verification} \tab Sends a verification request to an email contact method to ensure it's owned by the requester\cr
#'  \link[paws.compute:lightsail_set_ip_address_type]{set_ip_address_type} \tab Sets the IP address type for an Amazon Lightsail resource\cr
#'  \link[paws.compute:lightsail_set_resource_access_for_bucket]{set_resource_access_for_bucket} \tab Sets the Amazon Lightsail resources that can access the specified Lightsail bucket\cr
#'  \link[paws.compute:lightsail_start_gui_session]{start_gui_session} \tab Initiates a graphical user interface (GUI) session that’s used to access a virtual computer’s operating system and application\cr
#'  \link[paws.compute:lightsail_start_instance]{start_instance} \tab Starts a specific Amazon Lightsail instance from a stopped state\cr
#'  \link[paws.compute:lightsail_start_relational_database]{start_relational_database} \tab Starts a specific database from a stopped state in Amazon Lightsail\cr
#'  \link[paws.compute:lightsail_stop_gui_session]{stop_gui_session} \tab Terminates a web-based NICE DCV session that’s used to access a virtual computer’s operating system or application\cr
#'  \link[paws.compute:lightsail_stop_instance]{stop_instance} \tab Stops a specific Amazon Lightsail instance that is currently running\cr
#'  \link[paws.compute:lightsail_stop_relational_database]{stop_relational_database} \tab Stops a specific database that is currently running in Amazon Lightsail\cr
#'  \link[paws.compute:lightsail_tag_resource]{tag_resource} \tab Adds one or more tags to the specified Amazon Lightsail resource\cr
#'  \link[paws.compute:lightsail_test_alarm]{test_alarm} \tab Tests an alarm by displaying a banner on the Amazon Lightsail console\cr
#'  \link[paws.compute:lightsail_unpeer_vpc]{unpeer_vpc} \tab Unpeers the Lightsail VPC from the user's default VPC\cr
#'  \link[paws.compute:lightsail_untag_resource]{untag_resource} \tab Deletes the specified set of tag keys and their values from the specified Amazon Lightsail resource\cr
#'  \link[paws.compute:lightsail_update_bucket]{update_bucket} \tab Updates an existing Amazon Lightsail bucket\cr
#'  \link[paws.compute:lightsail_update_bucket_bundle]{update_bucket_bundle} \tab Updates the bundle, or storage plan, of an existing Amazon Lightsail bucket\cr
#'  \link[paws.compute:lightsail_update_container_service]{update_container_service} \tab Updates the configuration of your Amazon Lightsail container service, such as its power, scale, and public domain names\cr
#'  \link[paws.compute:lightsail_update_distribution]{update_distribution} \tab Updates an existing Amazon Lightsail content delivery network (CDN) distribution\cr
#'  \link[paws.compute:lightsail_update_distribution_bundle]{update_distribution_bundle} \tab Updates the bundle of your Amazon Lightsail content delivery network (CDN) distribution\cr
#'  \link[paws.compute:lightsail_update_domain_entry]{update_domain_entry} \tab Updates a domain recordset after it is created\cr
#'  \link[paws.compute:lightsail_update_instance_metadata_options]{update_instance_metadata_options} \tab Modifies the Amazon Lightsail instance metadata parameters on a running or stopped instance\cr
#'  \link[paws.compute:lightsail_update_load_balancer_attribute]{update_load_balancer_attribute} \tab Updates the specified attribute for a load balancer\cr
#'  \link[paws.compute:lightsail_update_relational_database]{update_relational_database} \tab Allows the update of one or more attributes of a database in Amazon Lightsail\cr
#'  \link[paws.compute:lightsail_update_relational_database_parameters]{update_relational_database_parameters} \tab Allows the update of one or more parameters of a database in Amazon Lightsail
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname lightsail
#' @export
lightsail <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.compute::lightsail(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' AWS Proton
#'
#' @description
#' This is the Proton Service API Reference. It provides descriptions,
#' syntax and usage examples for each of the
#' [actions](https://docs.aws.amazon.com/proton/latest/APIReference/API_Operations.html)
#' and [data
#' types](https://docs.aws.amazon.com/proton/latest/APIReference/API_Types.html)
#' for the Proton service.
#'
#' The documentation for each action shows the Query API request parameters
#' and the XML response.
#'
#' Alternatively, you can use the Amazon Web Services CLI to access an API.
#' For more information, see the [Amazon Web Services Command Line
#' Interface User
#' Guide](https://docs.aws.amazon.com/cli/latest/userguide/cli-chap-welcome.html).
#'
#' The Proton service is a two-pronged automation framework. Administrators
#' create service templates to provide standardized infrastructure and
#' deployment tooling for serverless and container based applications.
#' Developers, in turn, select from the available service templates to
#' automate their application or service deployments.
#'
#' Because administrators define the infrastructure and tooling that Proton
#' deploys and manages, they need permissions to use all of the listed API
#' operations.
#'
#' When developers select a specific infrastructure and tooling set, Proton
#' deploys their applications. To monitor their applications that are
#' running on Proton, developers need permissions to the service *create*,
#' *list*, *update* and *delete* API operations and the service instance
#' *list* and *update* API operations.
#'
#' To learn more about Proton, see the [Proton User
#' Guide](https://docs.aws.amazon.com/proton/latest/userguide/Welcome.html).
#'
#' **Ensuring Idempotency**
#'
#' When you make a mutating API request, the request typically returns a
#' result before the asynchronous workflows of the operation are complete.
#' Operations might also time out or encounter other server issues before
#' they're complete, even if the request already returned a result. This
#' might make it difficult to determine whether the request succeeded.
#' Moreover, you might need to retry the request multiple times to ensure
#' that the operation completes successfully. However, if the original
#' request and the subsequent retries are successful, the operation occurs
#' multiple times. This means that you might create more resources than you
#' intended.
#'
#' *Idempotency* ensures that an API request action completes no more than
#' one time. With an idempotent request, if the original request action
#' completes successfully, any subsequent retries complete successfully
#' without performing any further actions. However, the result might
#' contain updated information, such as the current creation status.
#'
#' The following lists of APIs are grouped according to methods that ensure
#' idempotency.
#'
#' **Idempotent create APIs with a client token**
#'
#' The API actions in this list support idempotency with the use of a
#' *client token*. The corresponding Amazon Web Services CLI commands also
#' support idempotency using a client token. A client token is a unique,
#' case-sensitive string of up to 64 ASCII characters. To make an
#' idempotent API request using one of these actions, specify a client
#' token in the request. We recommend that you *don't* reuse the same
#' client token for other API requests. If you don’t provide a client token
#' for these APIs, a default client token is automatically provided by
#' SDKs.
#'
#' Given a request action that has succeeded:
#'
#' If you retry the request using the same client token and the same
#' parameters, the retry succeeds without performing any further actions
#' other than returning the original resource detail data in the response.
#'
#' If you retry the request using the same client token, but one or more of
#' the parameters are different, the retry throws a `ValidationException`
#' with an `IdempotentParameterMismatch` error.
#'
#' Client tokens expire eight hours after a request is made. If you retry
#' the request with the expired token, a new resource is created.
#'
#' If the original resource is deleted and you retry the request, a new
#' resource is created.
#'
#' Idempotent create APIs with a client token:
#'
#' -   CreateEnvironmentTemplateVersion
#'
#' -   CreateServiceTemplateVersion
#'
#' -   CreateEnvironmentAccountConnection
#'
#' **Idempotent create APIs**
#'
#' Given a request action that has succeeded:
#'
#' If you retry the request with an API from this group, and the original
#' resource *hasn't* been modified, the retry succeeds without performing
#' any further actions other than returning the original resource detail
#' data in the response.
#'
#' If the original resource has been modified, the retry throws a
#' `ConflictException`.
#'
#' If you retry with different input parameters, the retry throws a
#' `ValidationException` with an `IdempotentParameterMismatch` error.
#'
#' Idempotent create APIs:
#'
#' -   CreateEnvironmentTemplate
#'
#' -   CreateServiceTemplate
#'
#' -   CreateEnvironment
#'
#' -   CreateService
#'
#' **Idempotent delete APIs**
#'
#' Given a request action that has succeeded:
#'
#' When you retry the request with an API from this group and the resource
#' was deleted, its metadata is returned in the response.
#'
#' If you retry and the resource doesn't exist, the response is empty.
#'
#' In both cases, the retry succeeds.
#'
#' Idempotent delete APIs:
#'
#' -   DeleteEnvironmentTemplate
#'
#' -   DeleteEnvironmentTemplateVersion
#'
#' -   DeleteServiceTemplate
#'
#' -   DeleteServiceTemplateVersion
#'
#' -   DeleteEnvironmentAccountConnection
#'
#' **Asynchronous idempotent delete APIs**
#'
#' Given a request action that has succeeded:
#'
#' If you retry the request with an API from this group, if the original
#' request delete operation status is `DELETE_IN_PROGRESS`, the retry
#' returns the resource detail data in the response without performing any
#' further actions.
#'
#' If the original request delete operation is complete, a retry returns an
#' empty response.
#'
#' Asynchronous idempotent delete APIs:
#'
#' -   DeleteEnvironment
#'
#' -   DeleteService
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- proton(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- proton()
#' svc$accept_environment_account_connection(
#'   Foo = 123
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.compute:proton_accept_environment_account_connection]{accept_environment_account_connection} \tab In a management account, an environment account connection request is accepted\cr
#'  \link[paws.compute:proton_cancel_component_deployment]{cancel_component_deployment} \tab Attempts to cancel a component deployment (for a component that is in the IN_PROGRESS deployment status)\cr
#'  \link[paws.compute:proton_cancel_environment_deployment]{cancel_environment_deployment} \tab Attempts to cancel an environment deployment on an UpdateEnvironment action, if the deployment is IN_PROGRESS\cr
#'  \link[paws.compute:proton_cancel_service_instance_deployment]{cancel_service_instance_deployment} \tab Attempts to cancel a service instance deployment on an UpdateServiceInstance action, if the deployment is IN_PROGRESS\cr
#'  \link[paws.compute:proton_cancel_service_pipeline_deployment]{cancel_service_pipeline_deployment} \tab Attempts to cancel a service pipeline deployment on an UpdateServicePipeline action, if the deployment is IN_PROGRESS\cr
#'  \link[paws.compute:proton_create_component]{create_component} \tab Create an Proton component\cr
#'  \link[paws.compute:proton_create_environment]{create_environment} \tab Deploy a new environment\cr
#'  \link[paws.compute:proton_create_environment_account_connection]{create_environment_account_connection} \tab Create an environment account connection in an environment account so that environment infrastructure resources can be provisioned in the environment account from a management account\cr
#'  \link[paws.compute:proton_create_environment_template]{create_environment_template} \tab Create an environment template for Proton\cr
#'  \link[paws.compute:proton_create_environment_template_version]{create_environment_template_version} \tab Create a new major or minor version of an environment template\cr
#'  \link[paws.compute:proton_create_repository]{create_repository} \tab Create and register a link to a repository\cr
#'  \link[paws.compute:proton_create_service]{create_service} \tab Create an Proton service\cr
#'  \link[paws.compute:proton_create_service_instance]{create_service_instance} \tab Create a service instance\cr
#'  \link[paws.compute:proton_create_service_sync_config]{create_service_sync_config} \tab Create the Proton Ops configuration file\cr
#'  \link[paws.compute:proton_create_service_template]{create_service_template} \tab Create a service template\cr
#'  \link[paws.compute:proton_create_service_template_version]{create_service_template_version} \tab Create a new major or minor version of a service template\cr
#'  \link[paws.compute:proton_create_template_sync_config]{create_template_sync_config} \tab Set up a template to create new template versions automatically by tracking a linked repository\cr
#'  \link[paws.compute:proton_delete_component]{delete_component} \tab Delete an Proton component resource\cr
#'  \link[paws.compute:proton_delete_deployment]{delete_deployment} \tab Delete the deployment\cr
#'  \link[paws.compute:proton_delete_environment]{delete_environment} \tab Delete an environment\cr
#'  \link[paws.compute:proton_delete_environment_account_connection]{delete_environment_account_connection} \tab In an environment account, delete an environment account connection\cr
#'  \link[paws.compute:proton_delete_environment_template]{delete_environment_template} \tab If no other major or minor versions of an environment template exist, delete the environment template\cr
#'  \link[paws.compute:proton_delete_environment_template_version]{delete_environment_template_version} \tab If no other minor versions of an environment template exist, delete a major version of the environment template if it's not the Recommended version\cr
#'  \link[paws.compute:proton_delete_repository]{delete_repository} \tab De-register and unlink your repository\cr
#'  \link[paws.compute:proton_delete_service]{delete_service} \tab Delete a service, with its instances and pipeline\cr
#'  \link[paws.compute:proton_delete_service_sync_config]{delete_service_sync_config} \tab Delete the Proton Ops file\cr
#'  \link[paws.compute:proton_delete_service_template]{delete_service_template} \tab If no other major or minor versions of the service template exist, delete the service template\cr
#'  \link[paws.compute:proton_delete_service_template_version]{delete_service_template_version} \tab If no other minor versions of a service template exist, delete a major version of the service template if it's not the Recommended version\cr
#'  \link[paws.compute:proton_delete_template_sync_config]{delete_template_sync_config} \tab Delete a template sync configuration\cr
#'  \link[paws.compute:proton_get_account_settings]{get_account_settings} \tab Get detail data for Proton account-wide settings\cr
#'  \link[paws.compute:proton_get_component]{get_component} \tab Get detailed data for a component\cr
#'  \link[paws.compute:proton_get_deployment]{get_deployment} \tab Get detailed data for a deployment\cr
#'  \link[paws.compute:proton_get_environment]{get_environment} \tab Get detailed data for an environment\cr
#'  \link[paws.compute:proton_get_environment_account_connection]{get_environment_account_connection} \tab In an environment account, get the detailed data for an environment account connection\cr
#'  \link[paws.compute:proton_get_environment_template]{get_environment_template} \tab Get detailed data for an environment template\cr
#'  \link[paws.compute:proton_get_environment_template_version]{get_environment_template_version} \tab Get detailed data for a major or minor version of an environment template\cr
#'  \link[paws.compute:proton_get_repository]{get_repository} \tab Get detail data for a linked repository\cr
#'  \link[paws.compute:proton_get_repository_sync_status]{get_repository_sync_status} \tab Get the sync status of a repository used for Proton template sync\cr
#'  \link[paws.compute:proton_get_resources_summary]{get_resources_summary} \tab Get counts of Proton resources\cr
#'  \link[paws.compute:proton_get_service]{get_service} \tab Get detailed data for a service\cr
#'  \link[paws.compute:proton_get_service_instance]{get_service_instance} \tab Get detailed data for a service instance\cr
#'  \link[paws.compute:proton_get_service_instance_sync_status]{get_service_instance_sync_status} \tab Get the status of the synced service instance\cr
#'  \link[paws.compute:proton_get_service_sync_blocker_summary]{get_service_sync_blocker_summary} \tab Get detailed data for the service sync blocker summary\cr
#'  \link[paws.compute:proton_get_service_sync_config]{get_service_sync_config} \tab Get detailed information for the service sync configuration\cr
#'  \link[paws.compute:proton_get_service_template]{get_service_template} \tab Get detailed data for a service template\cr
#'  \link[paws.compute:proton_get_service_template_version]{get_service_template_version} \tab Get detailed data for a major or minor version of a service template\cr
#'  \link[paws.compute:proton_get_template_sync_config]{get_template_sync_config} \tab Get detail data for a template sync configuration\cr
#'  \link[paws.compute:proton_get_template_sync_status]{get_template_sync_status} \tab Get the status of a template sync\cr
#'  \link[paws.compute:proton_list_component_outputs]{list_component_outputs} \tab Get a list of component Infrastructure as Code (IaC) outputs\cr
#'  \link[paws.compute:proton_list_component_provisioned_resources]{list_component_provisioned_resources} \tab List provisioned resources for a component with details\cr
#'  \link[paws.compute:proton_list_components]{list_components} \tab List components with summary data\cr
#'  \link[paws.compute:proton_list_deployments]{list_deployments} \tab List deployments\cr
#'  \link[paws.compute:proton_list_environment_account_connections]{list_environment_account_connections} \tab View a list of environment account connections\cr
#'  \link[paws.compute:proton_list_environment_outputs]{list_environment_outputs} \tab List the infrastructure as code outputs for your environment\cr
#'  \link[paws.compute:proton_list_environment_provisioned_resources]{list_environment_provisioned_resources} \tab List the provisioned resources for your environment\cr
#'  \link[paws.compute:proton_list_environments]{list_environments} \tab List environments with detail data summaries\cr
#'  \link[paws.compute:proton_list_environment_templates]{list_environment_templates} \tab List environment templates\cr
#'  \link[paws.compute:proton_list_environment_template_versions]{list_environment_template_versions} \tab List major or minor versions of an environment template with detail data\cr
#'  \link[paws.compute:proton_list_repositories]{list_repositories} \tab List linked repositories with detail data\cr
#'  \link[paws.compute:proton_list_repository_sync_definitions]{list_repository_sync_definitions} \tab List repository sync definitions with detail data\cr
#'  \link[paws.compute:proton_list_service_instance_outputs]{list_service_instance_outputs} \tab Get a list service of instance Infrastructure as Code (IaC) outputs\cr
#'  \link[paws.compute:proton_list_service_instance_provisioned_resources]{list_service_instance_provisioned_resources} \tab List provisioned resources for a service instance with details\cr
#'  \link[paws.compute:proton_list_service_instances]{list_service_instances} \tab List service instances with summary data\cr
#'  \link[paws.compute:proton_list_service_pipeline_outputs]{list_service_pipeline_outputs} \tab Get a list of service pipeline Infrastructure as Code (IaC) outputs\cr
#'  \link[paws.compute:proton_list_service_pipeline_provisioned_resources]{list_service_pipeline_provisioned_resources} \tab List provisioned resources for a service and pipeline with details\cr
#'  \link[paws.compute:proton_list_services]{list_services} \tab List services with summaries of detail data\cr
#'  \link[paws.compute:proton_list_service_templates]{list_service_templates} \tab List service templates with detail data\cr
#'  \link[paws.compute:proton_list_service_template_versions]{list_service_template_versions} \tab List major or minor versions of a service template with detail data\cr
#'  \link[paws.compute:proton_list_tags_for_resource]{list_tags_for_resource} \tab List tags for a resource\cr
#'  \link[paws.compute:proton_notify_resource_deployment_status_change]{notify_resource_deployment_status_change} \tab Notify Proton of status changes to a provisioned resource when you use self-managed provisioning\cr
#'  \link[paws.compute:proton_reject_environment_account_connection]{reject_environment_account_connection} \tab In a management account, reject an environment account connection from another environment account\cr
#'  \link[paws.compute:proton_tag_resource]{tag_resource} \tab Tag a resource\cr
#'  \link[paws.compute:proton_untag_resource]{untag_resource} \tab Remove a customer tag from a resource\cr
#'  \link[paws.compute:proton_update_account_settings]{update_account_settings} \tab Update Proton settings that are used for multiple services in the Amazon Web Services account\cr
#'  \link[paws.compute:proton_update_component]{update_component} \tab Update a component\cr
#'  \link[paws.compute:proton_update_environment]{update_environment} \tab Update an environment\cr
#'  \link[paws.compute:proton_update_environment_account_connection]{update_environment_account_connection} \tab In an environment account, update an environment account connection to use a new IAM role\cr
#'  \link[paws.compute:proton_update_environment_template]{update_environment_template} \tab Update an environment template\cr
#'  \link[paws.compute:proton_update_environment_template_version]{update_environment_template_version} \tab Update a major or minor version of an environment template\cr
#'  \link[paws.compute:proton_update_service]{update_service} \tab Edit a service description or use a spec to add and delete service instances\cr
#'  \link[paws.compute:proton_update_service_instance]{update_service_instance} \tab Update a service instance\cr
#'  \link[paws.compute:proton_update_service_pipeline]{update_service_pipeline} \tab Update the service pipeline\cr
#'  \link[paws.compute:proton_update_service_sync_blocker]{update_service_sync_blocker} \tab Update the service sync blocker by resolving it\cr
#'  \link[paws.compute:proton_update_service_sync_config]{update_service_sync_config} \tab Update the Proton Ops config file\cr
#'  \link[paws.compute:proton_update_service_template]{update_service_template} \tab Update a service template\cr
#'  \link[paws.compute:proton_update_service_template_version]{update_service_template_version} \tab Update a major or minor version of a service template\cr
#'  \link[paws.compute:proton_update_template_sync_config]{update_template_sync_config} \tab Update template sync configuration parameters, except for the templateName and templateType
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname proton
#' @export
proton <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.compute::proton(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' AWSServerlessApplicationRepository
#'
#' @description
#' The AWS Serverless Application Repository makes it easy for developers
#' and enterprises to quickly find and deploy serverless applications in
#' the AWS Cloud. For more information about serverless applications, see
#' Serverless Computing and Applications on the AWS website.
#'
#' The AWS Serverless Application Repository is deeply integrated with the
#' AWS Lambda console, so that developers of all levels can get started
#' with serverless computing without needing to learn anything new. You can
#' use category keywords to browse for applications such as web and mobile
#' backends, data processing applications, or chatbots. You can also search
#' for applications by name, publisher, or event source. To use an
#' application, you simply choose it, configure any required fields, and
#' deploy it with a few clicks.
#'
#' You can also easily publish applications, sharing them publicly with the
#' community at large, or privately within your team or across your
#' organization. To publish a serverless application (or app), you can use
#' the AWS Management Console, AWS Command Line Interface (AWS CLI), or AWS
#' SDKs to upload the code. Along with the code, you upload a simple
#' manifest file, also known as the AWS Serverless Application Model (AWS
#' SAM) template. For more information about AWS SAM, see AWS Serverless
#' Application Model (AWS SAM) on the AWS Labs GitHub repository.
#'
#' The AWS Serverless Application Repository Developer Guide contains more
#' information about the two developer experiences available:
#'
#' -   Consuming Applications – Browse for applications and view
#'     information about them, including source code and readme files. Also
#'     install, configure, and deploy applications of your choosing.
#'
#'     Publishing Applications – Configure and upload applications to make
#'     them available to other developers, and publish new versions of
#'     applications.
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- serverlessapplicationrepository(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- serverlessapplicationrepository()
#' svc$create_application(
#'   Foo = 123
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.compute:serverlessapplicationrepository_create_application]{create_application} \tab Creates an application, optionally including an AWS SAM file to create the first application version in the same call\cr
#'  \link[paws.compute:serverlessapplicationrepository_create_application_version]{create_application_version} \tab Creates an application version\cr
#'  \link[paws.compute:serverlessapplicationrepository_create_cloud_formation_change_set]{create_cloud_formation_change_set} \tab Creates an AWS CloudFormation change set for the given application\cr
#'  \link[paws.compute:serverlessapplicationrepository_create_cloud_formation_template]{create_cloud_formation_template} \tab Creates an AWS CloudFormation template\cr
#'  \link[paws.compute:serverlessapplicationrepository_delete_application]{delete_application} \tab Deletes the specified application\cr
#'  \link[paws.compute:serverlessapplicationrepository_get_application]{get_application} \tab Gets the specified application\cr
#'  \link[paws.compute:serverlessapplicationrepository_get_application_policy]{get_application_policy} \tab Retrieves the policy for the application\cr
#'  \link[paws.compute:serverlessapplicationrepository_get_cloud_formation_template]{get_cloud_formation_template} \tab Gets the specified AWS CloudFormation template\cr
#'  \link[paws.compute:serverlessapplicationrepository_list_application_dependencies]{list_application_dependencies} \tab Retrieves the list of applications nested in the containing application\cr
#'  \link[paws.compute:serverlessapplicationrepository_list_applications]{list_applications} \tab Lists applications owned by the requester\cr
#'  \link[paws.compute:serverlessapplicationrepository_list_application_versions]{list_application_versions} \tab Lists versions for the specified application\cr
#'  \link[paws.compute:serverlessapplicationrepository_put_application_policy]{put_application_policy} \tab Sets the permission policy for an application\cr
#'  \link[paws.compute:serverlessapplicationrepository_unshare_application]{unshare_application} \tab Unshares an application from an AWS Organization\cr
#'  \link[paws.compute:serverlessapplicationrepository_update_application]{update_application} \tab Updates the specified application
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname serverlessapplicationrepository
#' @export
serverlessapplicationrepository <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.compute::serverlessapplicationrepository(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' AWS Backup
#'
#' @description
#' Backup
#'
#' Backup is a unified backup service designed to protect Amazon Web
#' Services services and their associated data. Backup simplifies the
#' creation, migration, restoration, and deletion of backups, while also
#' providing reporting and auditing.
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- backup(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- backup()
#' svc$cancel_legal_hold(
#'   Foo = 123
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.storage:backup_cancel_legal_hold]{cancel_legal_hold} \tab This action removes the specified legal hold on a recovery point\cr
#'  \link[paws.storage:backup_create_backup_plan]{create_backup_plan} \tab Creates a backup plan using a backup plan name and backup rules\cr
#'  \link[paws.storage:backup_create_backup_selection]{create_backup_selection} \tab Creates a JSON document that specifies a set of resources to assign to a backup plan\cr
#'  \link[paws.storage:backup_create_backup_vault]{create_backup_vault} \tab Creates a logical container where backups are stored\cr
#'  \link[paws.storage:backup_create_framework]{create_framework} \tab Creates a framework with one or more controls\cr
#'  \link[paws.storage:backup_create_legal_hold]{create_legal_hold} \tab This action creates a legal hold on a recovery point (backup)\cr
#'  \link[paws.storage:backup_create_logically_air_gapped_backup_vault]{create_logically_air_gapped_backup_vault} \tab This request creates a logical container to where backups may be copied\cr
#'  \link[paws.storage:backup_create_report_plan]{create_report_plan} \tab Creates a report plan\cr
#'  \link[paws.storage:backup_delete_backup_plan]{delete_backup_plan} \tab Deletes a backup plan\cr
#'  \link[paws.storage:backup_delete_backup_selection]{delete_backup_selection} \tab Deletes the resource selection associated with a backup plan that is specified by the SelectionId\cr
#'  \link[paws.storage:backup_delete_backup_vault]{delete_backup_vault} \tab Deletes the backup vault identified by its name\cr
#'  \link[paws.storage:backup_delete_backup_vault_access_policy]{delete_backup_vault_access_policy} \tab Deletes the policy document that manages permissions on a backup vault\cr
#'  \link[paws.storage:backup_delete_backup_vault_lock_configuration]{delete_backup_vault_lock_configuration} \tab Deletes Backup Vault Lock from a backup vault specified by a backup vault name\cr
#'  \link[paws.storage:backup_delete_backup_vault_notifications]{delete_backup_vault_notifications} \tab Deletes event notifications for the specified backup vault\cr
#'  \link[paws.storage:backup_delete_framework]{delete_framework} \tab Deletes the framework specified by a framework name\cr
#'  \link[paws.storage:backup_delete_recovery_point]{delete_recovery_point} \tab Deletes the recovery point specified by a recovery point ID\cr
#'  \link[paws.storage:backup_delete_report_plan]{delete_report_plan} \tab Deletes the report plan specified by a report plan name\cr
#'  \link[paws.storage:backup_describe_backup_job]{describe_backup_job} \tab Returns backup job details for the specified BackupJobId\cr
#'  \link[paws.storage:backup_describe_backup_vault]{describe_backup_vault} \tab Returns metadata about a backup vault specified by its name\cr
#'  \link[paws.storage:backup_describe_copy_job]{describe_copy_job} \tab Returns metadata associated with creating a copy of a resource\cr
#'  \link[paws.storage:backup_describe_framework]{describe_framework} \tab Returns the framework details for the specified FrameworkName\cr
#'  \link[paws.storage:backup_describe_global_settings]{describe_global_settings} \tab Describes whether the Amazon Web Services account is opted in to cross-account backup\cr
#'  \link[paws.storage:backup_describe_protected_resource]{describe_protected_resource} \tab Returns information about a saved resource, including the last time it was backed up, its Amazon Resource Name (ARN), and the Amazon Web Services service type of the saved resource\cr
#'  \link[paws.storage:backup_describe_recovery_point]{describe_recovery_point} \tab Returns metadata associated with a recovery point, including ID, status, encryption, and lifecycle\cr
#'  \link[paws.storage:backup_describe_region_settings]{describe_region_settings} \tab Returns the current service opt-in settings for the Region\cr
#'  \link[paws.storage:backup_describe_report_job]{describe_report_job} \tab Returns the details associated with creating a report as specified by its ReportJobId\cr
#'  \link[paws.storage:backup_describe_report_plan]{describe_report_plan} \tab Returns a list of all report plans for an Amazon Web Services account and Amazon Web Services Region\cr
#'  \link[paws.storage:backup_describe_restore_job]{describe_restore_job} \tab Returns metadata associated with a restore job that is specified by a job ID\cr
#'  \link[paws.storage:backup_disassociate_recovery_point]{disassociate_recovery_point} \tab Deletes the specified continuous backup recovery point from Backup and releases control of that continuous backup to the source service, such as Amazon RDS\cr
#'  \link[paws.storage:backup_disassociate_recovery_point_from_parent]{disassociate_recovery_point_from_parent} \tab This action to a specific child (nested) recovery point removes the relationship between the specified recovery point and its parent (composite) recovery point\cr
#'  \link[paws.storage:backup_export_backup_plan_template]{export_backup_plan_template} \tab Returns the backup plan that is specified by the plan ID as a backup template\cr
#'  \link[paws.storage:backup_get_backup_plan]{get_backup_plan} \tab Returns BackupPlan details for the specified BackupPlanId\cr
#'  \link[paws.storage:backup_get_backup_plan_from_json]{get_backup_plan_from_json} \tab Returns a valid JSON document specifying a backup plan or an error\cr
#'  \link[paws.storage:backup_get_backup_plan_from_template]{get_backup_plan_from_template} \tab Returns the template specified by its templateId as a backup plan\cr
#'  \link[paws.storage:backup_get_backup_selection]{get_backup_selection} \tab Returns selection metadata and a document in JSON format that specifies a list of resources that are associated with a backup plan\cr
#'  \link[paws.storage:backup_get_backup_vault_access_policy]{get_backup_vault_access_policy} \tab Returns the access policy document that is associated with the named backup vault\cr
#'  \link[paws.storage:backup_get_backup_vault_notifications]{get_backup_vault_notifications} \tab Returns event notifications for the specified backup vault\cr
#'  \link[paws.storage:backup_get_legal_hold]{get_legal_hold} \tab This action returns details for a specified legal hold\cr
#'  \link[paws.storage:backup_get_recovery_point_restore_metadata]{get_recovery_point_restore_metadata} \tab Returns a set of metadata key-value pairs that were used to create the backup\cr
#'  \link[paws.storage:backup_get_supported_resource_types]{get_supported_resource_types} \tab Returns the Amazon Web Services resource types supported by Backup\cr
#'  \link[paws.storage:backup_list_backup_jobs]{list_backup_jobs} \tab Returns a list of existing backup jobs for an authenticated account for the last 30 days\cr
#'  \link[paws.storage:backup_list_backup_plans]{list_backup_plans} \tab Returns a list of all active backup plans for an authenticated account\cr
#'  \link[paws.storage:backup_list_backup_plan_templates]{list_backup_plan_templates} \tab Returns metadata of your saved backup plan templates, including the template ID, name, and the creation and deletion dates\cr
#'  \link[paws.storage:backup_list_backup_plan_versions]{list_backup_plan_versions} \tab Returns version metadata of your backup plans, including Amazon Resource Names (ARNs), backup plan IDs, creation and deletion dates, plan names, and version IDs\cr
#'  \link[paws.storage:backup_list_backup_selections]{list_backup_selections} \tab Returns an array containing metadata of the resources associated with the target backup plan\cr
#'  \link[paws.storage:backup_list_backup_vaults]{list_backup_vaults} \tab Returns a list of recovery point storage containers along with information about them\cr
#'  \link[paws.storage:backup_list_copy_jobs]{list_copy_jobs} \tab Returns metadata about your copy jobs\cr
#'  \link[paws.storage:backup_list_frameworks]{list_frameworks} \tab Returns a list of all frameworks for an Amazon Web Services account and Amazon Web Services Region\cr
#'  \link[paws.storage:backup_list_legal_holds]{list_legal_holds} \tab This action returns metadata about active and previous legal holds\cr
#'  \link[paws.storage:backup_list_protected_resources]{list_protected_resources} \tab Returns an array of resources successfully backed up by Backup, including the time the resource was saved, an Amazon Resource Name (ARN) of the resource, and a resource type\cr
#'  \link[paws.storage:backup_list_protected_resources_by_backup_vault]{list_protected_resources_by_backup_vault} \tab This request lists the protected resources corresponding to each backup vault\cr
#'  \link[paws.storage:backup_list_recovery_points_by_backup_vault]{list_recovery_points_by_backup_vault} \tab Returns detailed information about the recovery points stored in a backup vault\cr
#'  \link[paws.storage:backup_list_recovery_points_by_legal_hold]{list_recovery_points_by_legal_hold} \tab This action returns recovery point ARNs (Amazon Resource Names) of the specified legal hold\cr
#'  \link[paws.storage:backup_list_recovery_points_by_resource]{list_recovery_points_by_resource} \tab Returns detailed information about all the recovery points of the type specified by a resource Amazon Resource Name (ARN)\cr
#'  \link[paws.storage:backup_list_report_jobs]{list_report_jobs} \tab Returns details about your report jobs\cr
#'  \link[paws.storage:backup_list_report_plans]{list_report_plans} \tab Returns a list of your report plans\cr
#'  \link[paws.storage:backup_list_restore_jobs]{list_restore_jobs} \tab Returns a list of jobs that Backup initiated to restore a saved resource, including details about the recovery process\cr
#'  \link[paws.storage:backup_list_tags]{list_tags} \tab Returns a list of key-value pairs assigned to a target recovery point, backup plan, or backup vault\cr
#'  \link[paws.storage:backup_put_backup_vault_access_policy]{put_backup_vault_access_policy} \tab Sets a resource-based policy that is used to manage access permissions on the target backup vault\cr
#'  \link[paws.storage:backup_put_backup_vault_lock_configuration]{put_backup_vault_lock_configuration} \tab Applies Backup Vault Lock to a backup vault, preventing attempts to delete any recovery point stored in or created in a backup vault\cr
#'  \link[paws.storage:backup_put_backup_vault_notifications]{put_backup_vault_notifications} \tab Turns on notifications on a backup vault for the specified topic and events\cr
#'  \link[paws.storage:backup_start_backup_job]{start_backup_job} \tab Starts an on-demand backup job for the specified resource\cr
#'  \link[paws.storage:backup_start_copy_job]{start_copy_job} \tab Starts a job to create a one-time copy of the specified resource\cr
#'  \link[paws.storage:backup_start_report_job]{start_report_job} \tab Starts an on-demand report job for the specified report plan\cr
#'  \link[paws.storage:backup_start_restore_job]{start_restore_job} \tab Recovers the saved resource identified by an Amazon Resource Name (ARN)\cr
#'  \link[paws.storage:backup_stop_backup_job]{stop_backup_job} \tab Attempts to cancel a job to create a one-time backup of a resource\cr
#'  \link[paws.storage:backup_tag_resource]{tag_resource} \tab Assigns a set of key-value pairs to a recovery point, backup plan, or backup vault identified by an Amazon Resource Name (ARN)\cr
#'  \link[paws.storage:backup_untag_resource]{untag_resource} \tab Removes a set of key-value pairs from a recovery point, backup plan, or backup vault identified by an Amazon Resource Name (ARN)\cr
#'  \link[paws.storage:backup_update_backup_plan]{update_backup_plan} \tab Updates an existing backup plan identified by its backupPlanId with the input document in JSON format\cr
#'  \link[paws.storage:backup_update_framework]{update_framework} \tab Updates an existing framework identified by its FrameworkName with the input document in JSON format\cr
#'  \link[paws.storage:backup_update_global_settings]{update_global_settings} \tab Updates whether the Amazon Web Services account is opted in to cross-account backup\cr
#'  \link[paws.storage:backup_update_recovery_point_lifecycle]{update_recovery_point_lifecycle} \tab Sets the transition lifecycle of a recovery point\cr
#'  \link[paws.storage:backup_update_region_settings]{update_region_settings} \tab Updates the current service opt-in settings for the Region\cr
#'  \link[paws.storage:backup_update_report_plan]{update_report_plan} \tab Updates an existing report plan identified by its ReportPlanName with the input document in JSON format
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname backup
#' @export
backup <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.storage::backup(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' AWS Backup Storage
#'
#' @description
#' The frontend service for Cryo Storage.
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- backupstorage(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- backupstorage()
#' svc$delete_object(
#'   Foo = 123
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.storage:backupstorage_delete_object]{delete_object} \tab Delete Object from the incremental base Backup\cr
#'  \link[paws.storage:backupstorage_get_chunk]{get_chunk} \tab Gets the specified object's chunk\cr
#'  \link[paws.storage:backupstorage_get_object_metadata]{get_object_metadata} \tab Get metadata associated with an Object\cr
#'  \link[paws.storage:backupstorage_list_chunks]{list_chunks} \tab List chunks in a given Object\cr
#'  \link[paws.storage:backupstorage_list_objects]{list_objects} \tab List all Objects in a given Backup\cr
#'  \link[paws.storage:backupstorage_notify_object_complete]{notify_object_complete} \tab Complete upload\cr
#'  \link[paws.storage:backupstorage_put_chunk]{put_chunk} \tab Upload chunk\cr
#'  \link[paws.storage:backupstorage_put_object]{put_object} \tab Upload object that can store object metadata String and data blob in single API call using inline chunk field\cr
#'  \link[paws.storage:backupstorage_start_object]{start_object} \tab Start upload containing one or many chunks
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname backupstorage
#' @export
backupstorage <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.storage::backupstorage(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' Amazon Data Lifecycle Manager
#'
#' @description
#' With Amazon Data Lifecycle Manager, you can manage the lifecycle of your
#' Amazon Web Services resources. You create lifecycle policies, which are
#' used to automate operations on the specified resources.
#'
#' Amazon Data Lifecycle Manager supports Amazon EBS volumes and snapshots.
#' For information about using Amazon Data Lifecycle Manager with Amazon
#' EBS, see [Amazon Data Lifecycle
#' Manager](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/snapshot-lifecycle.html)
#' in the *Amazon EC2 User Guide*.
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- dlm(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- dlm()
#' svc$create_lifecycle_policy(
#'   Foo = 123
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.storage:dlm_create_lifecycle_policy]{create_lifecycle_policy} \tab Creates a policy to manage the lifecycle of the specified Amazon Web Services resources\cr
#'  \link[paws.storage:dlm_delete_lifecycle_policy]{delete_lifecycle_policy} \tab Deletes the specified lifecycle policy and halts the automated operations that the policy specified\cr
#'  \link[paws.storage:dlm_get_lifecycle_policies]{get_lifecycle_policies} \tab Gets summary information about all or the specified data lifecycle policies\cr
#'  \link[paws.storage:dlm_get_lifecycle_policy]{get_lifecycle_policy} \tab Gets detailed information about the specified lifecycle policy\cr
#'  \link[paws.storage:dlm_list_tags_for_resource]{list_tags_for_resource} \tab Lists the tags for the specified resource\cr
#'  \link[paws.storage:dlm_tag_resource]{tag_resource} \tab Adds the specified tags to the specified resource\cr
#'  \link[paws.storage:dlm_untag_resource]{untag_resource} \tab Removes the specified tags from the specified resource\cr
#'  \link[paws.storage:dlm_update_lifecycle_policy]{update_lifecycle_policy} \tab Updates the specified lifecycle policy
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname dlm
#' @export
dlm <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.storage::dlm(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' Amazon Elastic Block Store
#'
#' @description
#' You can use the Amazon Elastic Block Store (Amazon EBS) direct APIs to
#' create Amazon EBS snapshots, write data directly to your snapshots, read
#' data on your snapshots, and identify the differences or changes between
#' two snapshots. If you’re an independent software vendor (ISV) who offers
#' backup services for Amazon EBS, the EBS direct APIs make it more
#' efficient and cost-effective to track incremental changes on your Amazon
#' EBS volumes through snapshots. This can be done without having to create
#' new volumes from snapshots, and then use Amazon Elastic Compute Cloud
#' (Amazon EC2) instances to compare the differences.
#'
#' You can create incremental snapshots directly from data on-premises into
#' volumes and the cloud to use for quick disaster recovery. With the
#' ability to write and read snapshots, you can write your on-premises data
#' to an snapshot during a disaster. Then after recovery, you can restore
#' it back to Amazon Web Services or on-premises from the snapshot. You no
#' longer need to build and maintain complex mechanisms to copy data to and
#' from Amazon EBS.
#'
#' This API reference provides detailed information about the actions, data
#' types, parameters, and errors of the EBS direct APIs. For more
#' information about the elements that make up the EBS direct APIs, and
#' examples of how to use them effectively, see [Accessing the Contents of
#' an Amazon EBS
#' Snapshot](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-accessing-snapshot.html)
#' in the *Amazon Elastic Compute Cloud User Guide*. For more information
#' about the supported Amazon Web Services Regions, endpoints, and service
#' quotas for the EBS direct APIs, see [Amazon Elastic Block Store
#' Endpoints and
#' Quotas](https://docs.aws.amazon.com/general/latest/gr/ebs-service.html)
#' in the *Amazon Web Services General Reference*.
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- ebs(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- ebs()
#' svc$complete_snapshot(
#'   Foo = 123
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.storage:ebs_complete_snapshot]{complete_snapshot} \tab Seals and completes the snapshot after all of the required blocks of data have been written to it\cr
#'  \link[paws.storage:ebs_get_snapshot_block]{get_snapshot_block} \tab Returns the data in a block in an Amazon Elastic Block Store snapshot\cr
#'  \link[paws.storage:ebs_list_changed_blocks]{list_changed_blocks} \tab Returns information about the blocks that are different between two Amazon Elastic Block Store snapshots of the same volume/snapshot lineage\cr
#'  \link[paws.storage:ebs_list_snapshot_blocks]{list_snapshot_blocks} \tab Returns information about the blocks in an Amazon Elastic Block Store snapshot\cr
#'  \link[paws.storage:ebs_put_snapshot_block]{put_snapshot_block} \tab Writes a block of data to a snapshot\cr
#'  \link[paws.storage:ebs_start_snapshot]{start_snapshot} \tab Creates a new Amazon EBS snapshot
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname ebs
#' @export
ebs <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.storage::ebs(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' Amazon Elastic File System
#'
#' @description
#' Amazon Elastic File System (Amazon EFS) provides simple, scalable file
#' storage for use with Amazon EC2 Linux and Mac instances in the Amazon
#' Web Services Cloud. With Amazon EFS, storage capacity is elastic,
#' growing and shrinking automatically as you add and remove files, so that
#' your applications have the storage they need, when they need it. For
#' more information, see the [Amazon Elastic File System API
#' Reference](https://docs.aws.amazon.com/efs/latest/ug/api-reference.html)
#' and the [Amazon Elastic File System User
#' Guide](https://docs.aws.amazon.com/efs/latest/ug/whatisefs.html).
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- efs(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- efs()
#' # This operation creates a new, encrypted file system with automatic
#' # backups enabled, and the default generalpurpose performance mode.
#' svc$create_file_system(
#'   Backup = TRUE,
#'   CreationToken = "tokenstring",
#'   Encrypted = TRUE,
#'   PerformanceMode = "generalPurpose",
#'   Tags = list(
#'     list(
#'       Key = "Name",
#'       Value = "MyFileSystem"
#'     )
#'   )
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.storage:efs_create_access_point]{create_access_point} \tab Creates an EFS access point\cr
#'  \link[paws.storage:efs_create_file_system]{create_file_system} \tab Creates a new, empty file system\cr
#'  \link[paws.storage:efs_create_mount_target]{create_mount_target} \tab Creates a mount target for a file system\cr
#'  \link[paws.storage:efs_create_replication_configuration]{create_replication_configuration} \tab Creates a replication configuration that replicates an existing EFS file system to a new, read-only file system\cr
#'  \link[paws.storage:efs_create_tags]{create_tags} \tab DEPRECATED - CreateTags is deprecated and not maintained\cr
#'  \link[paws.storage:efs_delete_access_point]{delete_access_point} \tab Deletes the specified access point\cr
#'  \link[paws.storage:efs_delete_file_system]{delete_file_system} \tab Deletes a file system, permanently severing access to its contents\cr
#'  \link[paws.storage:efs_delete_file_system_policy]{delete_file_system_policy} \tab Deletes the FileSystemPolicy for the specified file system\cr
#'  \link[paws.storage:efs_delete_mount_target]{delete_mount_target} \tab Deletes the specified mount target\cr
#'  \link[paws.storage:efs_delete_replication_configuration]{delete_replication_configuration} \tab Deletes an existing replication configuration\cr
#'  \link[paws.storage:efs_delete_tags]{delete_tags} \tab DEPRECATED - DeleteTags is deprecated and not maintained\cr
#'  \link[paws.storage:efs_describe_access_points]{describe_access_points} \tab Returns the description of a specific Amazon EFS access point if the AccessPointId is provided\cr
#'  \link[paws.storage:efs_describe_account_preferences]{describe_account_preferences} \tab Returns the account preferences settings for the Amazon Web Services account associated with the user making the request, in the current Amazon Web Services Region\cr
#'  \link[paws.storage:efs_describe_backup_policy]{describe_backup_policy} \tab Returns the backup policy for the specified EFS file system\cr
#'  \link[paws.storage:efs_describe_file_system_policy]{describe_file_system_policy} \tab Returns the FileSystemPolicy for the specified EFS file system\cr
#'  \link[paws.storage:efs_describe_file_systems]{describe_file_systems} \tab Returns the description of a specific Amazon EFS file system if either the file system CreationToken or the FileSystemId is provided\cr
#'  \link[paws.storage:efs_describe_lifecycle_configuration]{describe_lifecycle_configuration} \tab Returns the current LifecycleConfiguration object for the specified Amazon EFS file system\cr
#'  \link[paws.storage:efs_describe_mount_targets]{describe_mount_targets} \tab Returns the descriptions of all the current mount targets, or a specific mount target, for a file system\cr
#'  \link[paws.storage:efs_describe_mount_target_security_groups]{describe_mount_target_security_groups} \tab Returns the security groups currently in effect for a mount target\cr
#'  \link[paws.storage:efs_describe_replication_configurations]{describe_replication_configurations} \tab Retrieves the replication configuration for a specific file system\cr
#'  \link[paws.storage:efs_describe_tags]{describe_tags} \tab DEPRECATED - The DescribeTags action is deprecated and not maintained\cr
#'  \link[paws.storage:efs_list_tags_for_resource]{list_tags_for_resource} \tab Lists all tags for a top-level EFS resource\cr
#'  \link[paws.storage:efs_modify_mount_target_security_groups]{modify_mount_target_security_groups} \tab Modifies the set of security groups in effect for a mount target\cr
#'  \link[paws.storage:efs_put_account_preferences]{put_account_preferences} \tab Use this operation to set the account preference in the current Amazon Web Services Region to use long 17 character (63 bit) or short 8 character (32 bit) resource IDs for new EFS file system and mount target resources\cr
#'  \link[paws.storage:efs_put_backup_policy]{put_backup_policy} \tab Updates the file system's backup policy\cr
#'  \link[paws.storage:efs_put_file_system_policy]{put_file_system_policy} \tab Applies an Amazon EFS FileSystemPolicy to an Amazon EFS file system\cr
#'  \link[paws.storage:efs_put_lifecycle_configuration]{put_lifecycle_configuration} \tab Use this action to manage EFS lifecycle management and EFS Intelligent-Tiering\cr
#'  \link[paws.storage:efs_tag_resource]{tag_resource} \tab Creates a tag for an EFS resource\cr
#'  \link[paws.storage:efs_untag_resource]{untag_resource} \tab Removes tags from an EFS resource\cr
#'  \link[paws.storage:efs_update_file_system]{update_file_system} \tab Updates the throughput mode or the amount of provisioned throughput of an existing file system
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname efs
#' @export
efs <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.storage::efs(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' FinSpace Public API
#'
#' @description
#' The FinSpace APIs let you take actions inside the FinSpace.
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- finspacedata(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- finspacedata()
#' svc$associate_user_to_permission_group(
#'   Foo = 123
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.storage:finspacedata_associate_user_to_permission_group]{associate_user_to_permission_group} \tab Adds a user account to a permission group to grant permissions for actions a user can perform in FinSpace\cr
#'  \link[paws.storage:finspacedata_create_changeset]{create_changeset} \tab Creates a new Changeset in a FinSpace Dataset\cr
#'  \link[paws.storage:finspacedata_create_dataset]{create_dataset} \tab Creates a new FinSpace Dataset\cr
#'  \link[paws.storage:finspacedata_create_data_view]{create_data_view} \tab Creates a Dataview for a Dataset\cr
#'  \link[paws.storage:finspacedata_create_permission_group]{create_permission_group} \tab Creates a group of permissions for various actions that a user can perform in FinSpace\cr
#'  \link[paws.storage:finspacedata_create_user]{create_user} \tab Creates a new user in FinSpace\cr
#'  \link[paws.storage:finspacedata_delete_dataset]{delete_dataset} \tab Deletes a FinSpace Dataset\cr
#'  \link[paws.storage:finspacedata_delete_permission_group]{delete_permission_group} \tab Deletes a permission group\cr
#'  \link[paws.storage:finspacedata_disable_user]{disable_user} \tab Denies access to the FinSpace web application and API for the specified user\cr
#'  \link[paws.storage:finspacedata_disassociate_user_from_permission_group]{disassociate_user_from_permission_group} \tab Removes a user account from a permission group\cr
#'  \link[paws.storage:finspacedata_enable_user]{enable_user} \tab Allows the specified user to access the FinSpace web application and API\cr
#'  \link[paws.storage:finspacedata_get_changeset]{get_changeset} \tab Get information about a Changeset\cr
#'  \link[paws.storage:finspacedata_get_dataset]{get_dataset} \tab Returns information about a Dataset\cr
#'  \link[paws.storage:finspacedata_get_data_view]{get_data_view} \tab Gets information about a Dataview\cr
#'  \link[paws.storage:finspacedata_get_external_data_view_access_details]{get_external_data_view_access_details} \tab Returns the credentials to access the external Dataview from an S3 location\cr
#'  \link[paws.storage:finspacedata_get_permission_group]{get_permission_group} \tab Retrieves the details of a specific permission group\cr
#'  \link[paws.storage:finspacedata_get_programmatic_access_credentials]{get_programmatic_access_credentials} \tab Request programmatic credentials to use with FinSpace SDK\cr
#'  \link[paws.storage:finspacedata_get_user]{get_user} \tab Retrieves details for a specific user\cr
#'  \link[paws.storage:finspacedata_get_working_location]{get_working_location} \tab A temporary Amazon S3 location, where you can copy your files from a source location to stage or use as a scratch space in FinSpace notebook\cr
#'  \link[paws.storage:finspacedata_list_changesets]{list_changesets} \tab Lists the FinSpace Changesets for a Dataset\cr
#'  \link[paws.storage:finspacedata_list_datasets]{list_datasets} \tab Lists all of the active Datasets that a user has access to\cr
#'  \link[paws.storage:finspacedata_list_data_views]{list_data_views} \tab Lists all available Dataviews for a Dataset\cr
#'  \link[paws.storage:finspacedata_list_permission_groups]{list_permission_groups} \tab Lists all available permission groups in FinSpace\cr
#'  \link[paws.storage:finspacedata_list_permission_groups_by_user]{list_permission_groups_by_user} \tab Lists all the permission groups that are associated with a specific user account\cr
#'  \link[paws.storage:finspacedata_list_users]{list_users} \tab Lists all available user accounts in FinSpace\cr
#'  \link[paws.storage:finspacedata_list_users_by_permission_group]{list_users_by_permission_group} \tab Lists details of all the users in a specific permission group\cr
#'  \link[paws.storage:finspacedata_reset_user_password]{reset_user_password} \tab Resets the password for a specified user ID and generates a temporary one\cr
#'  \link[paws.storage:finspacedata_update_changeset]{update_changeset} \tab Updates a FinSpace Changeset\cr
#'  \link[paws.storage:finspacedata_update_dataset]{update_dataset} \tab Updates a FinSpace Dataset\cr
#'  \link[paws.storage:finspacedata_update_permission_group]{update_permission_group} \tab Modifies the details of a permission group\cr
#'  \link[paws.storage:finspacedata_update_user]{update_user} \tab Modifies the details of the specified user account
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname finspacedata
#' @export
finspacedata <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.storage::finspacedata(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' Amazon FSx
#'
#' @description
#' Amazon FSx is a fully managed service that makes it easy for storage and
#' application administrators to launch and use shared file storage.
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- fsx(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- fsx()
#' # This operation copies an Amazon FSx backup.
#' svc$copy_backup(
#'   SourceBackupId = "backup-03e3c82e0183b7b6b",
#'   SourceRegion = "us-east-2"
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.storage:fsx_associate_file_system_aliases]{associate_file_system_aliases} \tab Use this action to associate one or more Domain Name Server (DNS) aliases with an existing Amazon FSx for Windows File Server file system\cr
#'  \link[paws.storage:fsx_cancel_data_repository_task]{cancel_data_repository_task} \tab Cancels an existing Amazon FSx for Lustre data repository task if that task is in either the PENDING or EXECUTING state\cr
#'  \link[paws.storage:fsx_copy_backup]{copy_backup} \tab Copies an existing backup within the same Amazon Web Services account to another Amazon Web Services Region (cross-Region copy) or within the same Amazon Web Services Region (in-Region copy)\cr
#'  \link[paws.storage:fsx_create_backup]{create_backup} \tab Creates a backup of an existing Amazon FSx for Windows File Server file system, Amazon FSx for Lustre file system, Amazon FSx for NetApp ONTAP volume, or Amazon FSx for OpenZFS file system\cr
#'  \link[paws.storage:fsx_create_data_repository_association]{create_data_repository_association} \tab Creates an Amazon FSx for Lustre data repository association (DRA)\cr
#'  \link[paws.storage:fsx_create_data_repository_task]{create_data_repository_task} \tab Creates an Amazon FSx for Lustre data repository task\cr
#'  \link[paws.storage:fsx_create_file_cache]{create_file_cache} \tab Creates a new Amazon File Cache resource\cr
#'  \link[paws.storage:fsx_create_file_system]{create_file_system} \tab Creates a new, empty Amazon FSx file system\cr
#'  \link[paws.storage:fsx_create_file_system_from_backup]{create_file_system_from_backup} \tab Creates a new Amazon FSx for Lustre, Amazon FSx for Windows File Server, or Amazon FSx for OpenZFS file system from an existing Amazon FSx backup\cr
#'  \link[paws.storage:fsx_create_snapshot]{create_snapshot} \tab Creates a snapshot of an existing Amazon FSx for OpenZFS volume\cr
#'  \link[paws.storage:fsx_create_storage_virtual_machine]{create_storage_virtual_machine} \tab Creates a storage virtual machine (SVM) for an Amazon FSx for ONTAP file system\cr
#'  \link[paws.storage:fsx_create_volume]{create_volume} \tab Creates an FSx for ONTAP or Amazon FSx for OpenZFS storage volume\cr
#'  \link[paws.storage:fsx_create_volume_from_backup]{create_volume_from_backup} \tab Creates a new Amazon FSx for NetApp ONTAP volume from an existing Amazon FSx volume backup\cr
#'  \link[paws.storage:fsx_delete_backup]{delete_backup} \tab Deletes an Amazon FSx backup\cr
#'  \link[paws.storage:fsx_delete_data_repository_association]{delete_data_repository_association} \tab Deletes a data repository association on an Amazon FSx for Lustre file system\cr
#'  \link[paws.storage:fsx_delete_file_cache]{delete_file_cache} \tab Deletes an Amazon File Cache resource\cr
#'  \link[paws.storage:fsx_delete_file_system]{delete_file_system} \tab Deletes a file system\cr
#'  \link[paws.storage:fsx_delete_snapshot]{delete_snapshot} \tab Deletes an Amazon FSx for OpenZFS snapshot\cr
#'  \link[paws.storage:fsx_delete_storage_virtual_machine]{delete_storage_virtual_machine} \tab Deletes an existing Amazon FSx for ONTAP storage virtual machine (SVM)\cr
#'  \link[paws.storage:fsx_delete_volume]{delete_volume} \tab Deletes an Amazon FSx for NetApp ONTAP or Amazon FSx for OpenZFS volume\cr
#'  \link[paws.storage:fsx_describe_backups]{describe_backups} \tab Returns the description of a specific Amazon FSx backup, if a BackupIds value is provided for that backup\cr
#'  \link[paws.storage:fsx_describe_data_repository_associations]{describe_data_repository_associations} \tab Returns the description of specific Amazon FSx for Lustre or Amazon File Cache data repository associations, if one or more AssociationIds values are provided in the request, or if filters are used in the request\cr
#'  \link[paws.storage:fsx_describe_data_repository_tasks]{describe_data_repository_tasks} \tab Returns the description of specific Amazon FSx for Lustre or Amazon File Cache data repository tasks, if one or more TaskIds values are provided in the request, or if filters are used in the request\cr
#'  \link[paws.storage:fsx_describe_file_caches]{describe_file_caches} \tab Returns the description of a specific Amazon File Cache resource, if a FileCacheIds value is provided for that cache\cr
#'  \link[paws.storage:fsx_describe_file_system_aliases]{describe_file_system_aliases} \tab Returns the DNS aliases that are associated with the specified Amazon FSx for Windows File Server file system\cr
#'  \link[paws.storage:fsx_describe_file_systems]{describe_file_systems} \tab Returns the description of specific Amazon FSx file systems, if a FileSystemIds value is provided for that file system\cr
#'  \link[paws.storage:fsx_describe_snapshots]{describe_snapshots} \tab Returns the description of specific Amazon FSx for OpenZFS snapshots, if a SnapshotIds value is provided\cr
#'  \link[paws.storage:fsx_describe_storage_virtual_machines]{describe_storage_virtual_machines} \tab Describes one or more Amazon FSx for NetApp ONTAP storage virtual machines (SVMs)\cr
#'  \link[paws.storage:fsx_describe_volumes]{describe_volumes} \tab Describes one or more Amazon FSx for NetApp ONTAP or Amazon FSx for OpenZFS volumes\cr
#'  \link[paws.storage:fsx_disassociate_file_system_aliases]{disassociate_file_system_aliases} \tab Use this action to disassociate, or remove, one or more Domain Name Service (DNS) aliases from an Amazon FSx for Windows File Server file system\cr
#'  \link[paws.storage:fsx_list_tags_for_resource]{list_tags_for_resource} \tab Lists tags for Amazon FSx resources\cr
#'  \link[paws.storage:fsx_release_file_system_nfs_v3_locks]{release_file_system_nfs_v3_locks} \tab Releases the file system lock from an Amazon FSx for OpenZFS file system\cr
#'  \link[paws.storage:fsx_restore_volume_from_snapshot]{restore_volume_from_snapshot} \tab Returns an Amazon FSx for OpenZFS volume to the state saved by the specified snapshot\cr
#'  \link[paws.storage:fsx_tag_resource]{tag_resource} \tab Tags an Amazon FSx resource\cr
#'  \link[paws.storage:fsx_untag_resource]{untag_resource} \tab This action removes a tag from an Amazon FSx resource\cr
#'  \link[paws.storage:fsx_update_data_repository_association]{update_data_repository_association} \tab Updates the configuration of an existing data repository association on an Amazon FSx for Lustre file system\cr
#'  \link[paws.storage:fsx_update_file_cache]{update_file_cache} \tab Updates the configuration of an existing Amazon File Cache resource\cr
#'  \link[paws.storage:fsx_update_file_system]{update_file_system} \tab Use this operation to update the configuration of an existing Amazon FSx file system\cr
#'  \link[paws.storage:fsx_update_snapshot]{update_snapshot} \tab Updates the name of an Amazon FSx for OpenZFS snapshot\cr
#'  \link[paws.storage:fsx_update_storage_virtual_machine]{update_storage_virtual_machine} \tab Updates an FSx for ONTAP storage virtual machine (SVM)\cr
#'  \link[paws.storage:fsx_update_volume]{update_volume} \tab Updates the configuration of an Amazon FSx for NetApp ONTAP or Amazon FSx for OpenZFS volume
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname fsx
#' @export
fsx <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.storage::fsx(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' Amazon Glacier
#'
#' @description
#' Amazon S3 Glacier (Glacier) is a storage solution for "cold data."
#'
#' Glacier is an extremely low-cost storage service that provides secure,
#' durable, and easy-to-use storage for data backup and archival. With
#' Glacier, customers can store their data cost effectively for months,
#' years, or decades. Glacier also enables customers to offload the
#' administrative burdens of operating and scaling storage to AWS, so they
#' don't have to worry about capacity planning, hardware provisioning, data
#' replication, hardware failure and recovery, or time-consuming hardware
#' migrations.
#'
#' Glacier is a great storage choice when low storage cost is paramount and
#' your data is rarely retrieved. If your application requires fast or
#' frequent access to your data, consider using Amazon S3. For more
#' information, see [Amazon Simple Storage Service (Amazon
#' S3)](https://aws.amazon.com/s3/).
#'
#' You can store any kind of data in any format. There is no maximum limit
#' on the total amount of data you can store in Glacier.
#'
#' If you are a first-time user of Glacier, we recommend that you begin by
#' reading the following sections in the *Amazon S3 Glacier Developer
#' Guide*:
#'
#' -   [What is Amazon S3
#'     Glacier](https://docs.aws.amazon.com/amazonglacier/latest/dev/introduction.html) -
#'     This section of the Developer Guide describes the underlying data
#'     model, the operations it supports, and the AWS SDKs that you can use
#'     to interact with the service.
#'
#' -   [Getting Started with Amazon S3
#'     Glacier](https://docs.aws.amazon.com/amazonglacier/latest/dev/amazon-glacier-getting-started.html) -
#'     The Getting Started section walks you through the process of
#'     creating a vault, uploading archives, creating jobs to download
#'     archives, retrieving the job output, and deleting archives.
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- glacier(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- glacier()
#' # The example deletes an in-progress multipart upload to a vault named
#' # my-vault:
#' svc$abort_multipart_upload(
#'   accountId = "-",
#'   uploadId = "19gaRezEXAMPLES6Ry5YYdqthHOC_kGRCT03L9yetr220UmPtBYKk-OssZtLq...",
#'   vaultName = "my-vault"
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.storage:glacier_abort_multipart_upload]{abort_multipart_upload} \tab This operation aborts a multipart upload identified by the upload ID\cr
#'  \link[paws.storage:glacier_abort_vault_lock]{abort_vault_lock} \tab This operation aborts the vault locking process if the vault lock is not in the Locked state\cr
#'  \link[paws.storage:glacier_add_tags_to_vault]{add_tags_to_vault} \tab This operation adds the specified tags to a vault\cr
#'  \link[paws.storage:glacier_complete_multipart_upload]{complete_multipart_upload} \tab You call this operation to inform Amazon S3 Glacier (Glacier) that all the archive parts have been uploaded and that Glacier can now assemble the archive from the uploaded parts\cr
#'  \link[paws.storage:glacier_complete_vault_lock]{complete_vault_lock} \tab This operation completes the vault locking process by transitioning the vault lock from the InProgress state to the Locked state, which causes the vault lock policy to become unchangeable\cr
#'  \link[paws.storage:glacier_create_vault]{create_vault} \tab This operation creates a new vault with the specified name\cr
#'  \link[paws.storage:glacier_delete_archive]{delete_archive} \tab This operation deletes an archive from a vault\cr
#'  \link[paws.storage:glacier_delete_vault]{delete_vault} \tab This operation deletes a vault\cr
#'  \link[paws.storage:glacier_delete_vault_access_policy]{delete_vault_access_policy} \tab This operation deletes the access policy associated with the specified vault\cr
#'  \link[paws.storage:glacier_delete_vault_notifications]{delete_vault_notifications} \tab This operation deletes the notification configuration set for a vault\cr
#'  \link[paws.storage:glacier_describe_job]{describe_job} \tab This operation returns information about a job you previously initiated, including the job initiation date, the user who initiated the job, the job status code/message and the Amazon SNS topic to notify after Amazon S3 Glacier (Glacier) completes the job\cr
#'  \link[paws.storage:glacier_describe_vault]{describe_vault} \tab This operation returns information about a vault, including the vault's Amazon Resource Name (ARN), the date the vault was created, the number of archives it contains, and the total size of all the archives in the vault\cr
#'  \link[paws.storage:glacier_get_data_retrieval_policy]{get_data_retrieval_policy} \tab This operation returns the current data retrieval policy for the account and region specified in the GET request\cr
#'  \link[paws.storage:glacier_get_job_output]{get_job_output} \tab This operation downloads the output of the job you initiated using InitiateJob\cr
#'  \link[paws.storage:glacier_get_vault_access_policy]{get_vault_access_policy} \tab This operation retrieves the access-policy subresource set on the vault; for more information on setting this subresource, see Set Vault Access Policy (PUT access-policy)\cr
#'  \link[paws.storage:glacier_get_vault_lock]{get_vault_lock} \tab This operation retrieves the following attributes from the lock-policy subresource set on the specified vault:\cr
#'  \link[paws.storage:glacier_get_vault_notifications]{get_vault_notifications} \tab This operation retrieves the notification-configuration subresource of the specified vault\cr
#'  \link[paws.storage:glacier_initiate_job]{initiate_job} \tab This operation initiates a job of the specified type, which can be a select, an archival retrieval, or a vault retrieval\cr
#'  \link[paws.storage:glacier_initiate_multipart_upload]{initiate_multipart_upload} \tab This operation initiates a multipart upload\cr
#'  \link[paws.storage:glacier_initiate_vault_lock]{initiate_vault_lock} \tab This operation initiates the vault locking process by doing the following:\cr
#'  \link[paws.storage:glacier_list_jobs]{list_jobs} \tab This operation lists jobs for a vault, including jobs that are in-progress and jobs that have recently finished\cr
#'  \link[paws.storage:glacier_list_multipart_uploads]{list_multipart_uploads} \tab This operation lists in-progress multipart uploads for the specified vault\cr
#'  \link[paws.storage:glacier_list_parts]{list_parts} \tab This operation lists the parts of an archive that have been uploaded in a specific multipart upload\cr
#'  \link[paws.storage:glacier_list_provisioned_capacity]{list_provisioned_capacity} \tab This operation lists the provisioned capacity units for the specified AWS account\cr
#'  \link[paws.storage:glacier_list_tags_for_vault]{list_tags_for_vault} \tab This operation lists all the tags attached to a vault\cr
#'  \link[paws.storage:glacier_list_vaults]{list_vaults} \tab This operation lists all vaults owned by the calling user's account\cr
#'  \link[paws.storage:glacier_purchase_provisioned_capacity]{purchase_provisioned_capacity} \tab This operation purchases a provisioned capacity unit for an AWS account\cr
#'  \link[paws.storage:glacier_remove_tags_from_vault]{remove_tags_from_vault} \tab This operation removes one or more tags from the set of tags attached to a vault\cr
#'  \link[paws.storage:glacier_set_data_retrieval_policy]{set_data_retrieval_policy} \tab This operation sets and then enacts a data retrieval policy in the region specified in the PUT request\cr
#'  \link[paws.storage:glacier_set_vault_access_policy]{set_vault_access_policy} \tab This operation configures an access policy for a vault and will overwrite an existing policy\cr
#'  \link[paws.storage:glacier_set_vault_notifications]{set_vault_notifications} \tab This operation configures notifications that will be sent when specific events happen to a vault\cr
#'  \link[paws.storage:glacier_upload_archive]{upload_archive} \tab This operation adds an archive to a vault\cr
#'  \link[paws.storage:glacier_upload_multipart_part]{upload_multipart_part} \tab This operation uploads a part of an archive
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname glacier
#' @export
glacier <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.storage::glacier(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' Amazon Recycle Bin
#'
#' @description
#' This is the *Recycle Bin API Reference*. This documentation provides
#' descriptions and syntax for each of the actions and data types in
#' Recycle Bin.
#'
#' Recycle Bin is a resource recovery feature that enables you to restore
#' accidentally deleted snapshots and EBS-backed AMIs. When using Recycle
#' Bin, if your resources are deleted, they are retained in the Recycle Bin
#' for a time period that you specify.
#'
#' You can restore a resource from the Recycle Bin at any time before its
#' retention period expires. After you restore a resource from the Recycle
#' Bin, the resource is removed from the Recycle Bin, and you can then use
#' it in the same way you use any other resource of that type in your
#' account. If the retention period expires and the resource is not
#' restored, the resource is permanently deleted from the Recycle Bin and
#' is no longer available for recovery. For more information about Recycle
#' Bin, see [Recycle
#' Bin](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/) in the
#' *Amazon Elastic Compute Cloud User Guide*.
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- recyclebin(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- recyclebin()
#' svc$create_rule(
#'   Foo = 123
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.storage:recyclebin_create_rule]{create_rule} \tab Creates a Recycle Bin retention rule\cr
#'  \link[paws.storage:recyclebin_delete_rule]{delete_rule} \tab Deletes a Recycle Bin retention rule\cr
#'  \link[paws.storage:recyclebin_get_rule]{get_rule} \tab Gets information about a Recycle Bin retention rule\cr
#'  \link[paws.storage:recyclebin_list_rules]{list_rules} \tab Lists the Recycle Bin retention rules in the Region\cr
#'  \link[paws.storage:recyclebin_list_tags_for_resource]{list_tags_for_resource} \tab Lists the tags assigned to a retention rule\cr
#'  \link[paws.storage:recyclebin_lock_rule]{lock_rule} \tab Locks a retention rule\cr
#'  \link[paws.storage:recyclebin_tag_resource]{tag_resource} \tab Assigns tags to the specified retention rule\cr
#'  \link[paws.storage:recyclebin_unlock_rule]{unlock_rule} \tab Unlocks a retention rule\cr
#'  \link[paws.storage:recyclebin_untag_resource]{untag_resource} \tab Unassigns a tag from a retention rule\cr
#'  \link[paws.storage:recyclebin_update_rule]{update_rule} \tab Updates an existing Recycle Bin retention rule
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname recyclebin
#' @export
recyclebin <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.storage::recyclebin(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' Amazon Simple Storage Service
#'
#'
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- s3(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- s3()
#' # The following example aborts a multipart upload.
#' svc$abort_multipart_upload(
#'   Bucket = "examplebucket",
#'   Key = "bigobject",
#'   UploadId = "xadcOB_7YPBOJuoFiQ9cz4P3Pe6FIZwO4f7wN93uHsNBEw97pl5eNwzExg0LA..."
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.storage:s3_abort_multipart_upload]{abort_multipart_upload} \tab This action aborts a multipart upload\cr
#'  \link[paws.storage:s3_complete_multipart_upload]{complete_multipart_upload} \tab Completes a multipart upload by assembling previously uploaded parts\cr
#'  \link[paws.storage:s3_copy_object]{copy_object} \tab Creates a copy of an object that is already stored in Amazon S3\cr
#'  \link[paws.storage:s3_create_bucket]{create_bucket} \tab Creates a new S3 bucket\cr
#'  \link[paws.storage:s3_create_multipart_upload]{create_multipart_upload} \tab This action initiates a multipart upload and returns an upload ID\cr
#'  \link[paws.storage:s3_delete_bucket]{delete_bucket} \tab Deletes the S3 bucket\cr
#'  \link[paws.storage:s3_delete_bucket_analytics_configuration]{delete_bucket_analytics_configuration} \tab Deletes an analytics configuration for the bucket (specified by the analytics configuration ID)\cr
#'  \link[paws.storage:s3_delete_bucket_cors]{delete_bucket_cors} \tab Deletes the cors configuration information set for the bucket\cr
#'  \link[paws.storage:s3_delete_bucket_encryption]{delete_bucket_encryption} \tab This implementation of the DELETE action resets the default encryption for the bucket as server-side encryption with Amazon S3 managed keys (SSE-S3)\cr
#'  \link[paws.storage:s3_delete_bucket_intelligent_tiering_configuration]{delete_bucket_intelligent_tiering_configuration} \tab Deletes the S3 Intelligent-Tiering configuration from the specified bucket\cr
#'  \link[paws.storage:s3_delete_bucket_inventory_configuration]{delete_bucket_inventory_configuration} \tab Deletes an inventory configuration (identified by the inventory ID) from the bucket\cr
#'  \link[paws.storage:s3_delete_bucket_lifecycle]{delete_bucket_lifecycle} \tab Deletes the lifecycle configuration from the specified bucket\cr
#'  \link[paws.storage:s3_delete_bucket_metrics_configuration]{delete_bucket_metrics_configuration} \tab Deletes a metrics configuration for the Amazon CloudWatch request metrics (specified by the metrics configuration ID) from the bucket\cr
#'  \link[paws.storage:s3_delete_bucket_ownership_controls]{delete_bucket_ownership_controls} \tab Removes OwnershipControls for an Amazon S3 bucket\cr
#'  \link[paws.storage:s3_delete_bucket_policy]{delete_bucket_policy} \tab This implementation of the DELETE action uses the policy subresource to delete the policy of a specified bucket\cr
#'  \link[paws.storage:s3_delete_bucket_replication]{delete_bucket_replication} \tab Deletes the replication configuration from the bucket\cr
#'  \link[paws.storage:s3_delete_bucket_tagging]{delete_bucket_tagging} \tab Deletes the tags from the bucket\cr
#'  \link[paws.storage:s3_delete_bucket_website]{delete_bucket_website} \tab This action removes the website configuration for a bucket\cr
#'  \link[paws.storage:s3_delete_object]{delete_object} \tab Removes the null version (if there is one) of an object and inserts a delete marker, which becomes the latest version of the object\cr
#'  \link[paws.storage:s3_delete_objects]{delete_objects} \tab This action enables you to delete multiple objects from a bucket using a single HTTP request\cr
#'  \link[paws.storage:s3_delete_object_tagging]{delete_object_tagging} \tab Removes the entire tag set from the specified object\cr
#'  \link[paws.storage:s3_delete_public_access_block]{delete_public_access_block} \tab Removes the PublicAccessBlock configuration for an Amazon S3 bucket\cr
#'  \link[paws.storage:s3_download_file]{download_file} \tab Download a file from S3 and store it at a specified file location\cr
#'  \link[paws.storage:s3_generate_presigned_url]{generate_presigned_url} \tab @title Generate a presigned url given a client, its method, and arguments\cr
#'  \link[paws.storage:s3_get_bucket_accelerate_configuration]{get_bucket_accelerate_configuration} \tab This implementation of the GET action uses the accelerate subresource to return the Transfer Acceleration state of a bucket, which is either Enabled or Suspended\cr
#'  \link[paws.storage:s3_get_bucket_acl]{get_bucket_acl} \tab This implementation of the GET action uses the acl subresource to return the access control list (ACL) of a bucket\cr
#'  \link[paws.storage:s3_get_bucket_analytics_configuration]{get_bucket_analytics_configuration} \tab This implementation of the GET action returns an analytics configuration (identified by the analytics configuration ID) from the bucket\cr
#'  \link[paws.storage:s3_get_bucket_cors]{get_bucket_cors} \tab Returns the Cross-Origin Resource Sharing (CORS) configuration information set for the bucket\cr
#'  \link[paws.storage:s3_get_bucket_encryption]{get_bucket_encryption} \tab Returns the default encryption configuration for an Amazon S3 bucket\cr
#'  \link[paws.storage:s3_get_bucket_intelligent_tiering_configuration]{get_bucket_intelligent_tiering_configuration} \tab Gets the S3 Intelligent-Tiering configuration from the specified bucket\cr
#'  \link[paws.storage:s3_get_bucket_inventory_configuration]{get_bucket_inventory_configuration} \tab Returns an inventory configuration (identified by the inventory configuration ID) from the bucket\cr
#'  \link[paws.storage:s3_get_bucket_lifecycle]{get_bucket_lifecycle} \tab For an updated version of this API, see GetBucketLifecycleConfiguration\cr
#'  \link[paws.storage:s3_get_bucket_lifecycle_configuration]{get_bucket_lifecycle_configuration} \tab Bucket lifecycle configuration now supports specifying a lifecycle rule using an object key name prefix, one or more object tags, or a combination of both\cr
#'  \link[paws.storage:s3_get_bucket_location]{get_bucket_location} \tab Returns the Region the bucket resides in\cr
#'  \link[paws.storage:s3_get_bucket_logging]{get_bucket_logging} \tab Returns the logging status of a bucket and the permissions users have to view and modify that status\cr
#'  \link[paws.storage:s3_get_bucket_metrics_configuration]{get_bucket_metrics_configuration} \tab Gets a metrics configuration (specified by the metrics configuration ID) from the bucket\cr
#'  \link[paws.storage:s3_get_bucket_notification]{get_bucket_notification} \tab No longer used, see GetBucketNotificationConfiguration\cr
#'  \link[paws.storage:s3_get_bucket_notification_configuration]{get_bucket_notification_configuration} \tab Returns the notification configuration of a bucket\cr
#'  \link[paws.storage:s3_get_bucket_ownership_controls]{get_bucket_ownership_controls} \tab Retrieves OwnershipControls for an Amazon S3 bucket\cr
#'  \link[paws.storage:s3_get_bucket_policy]{get_bucket_policy} \tab Returns the policy of a specified bucket\cr
#'  \link[paws.storage:s3_get_bucket_policy_status]{get_bucket_policy_status} \tab Retrieves the policy status for an Amazon S3 bucket, indicating whether the bucket is public\cr
#'  \link[paws.storage:s3_get_bucket_replication]{get_bucket_replication} \tab Returns the replication configuration of a bucket\cr
#'  \link[paws.storage:s3_get_bucket_request_payment]{get_bucket_request_payment} \tab Returns the request payment configuration of a bucket\cr
#'  \link[paws.storage:s3_get_bucket_tagging]{get_bucket_tagging} \tab Returns the tag set associated with the bucket\cr
#'  \link[paws.storage:s3_get_bucket_versioning]{get_bucket_versioning} \tab Returns the versioning state of a bucket\cr
#'  \link[paws.storage:s3_get_bucket_website]{get_bucket_website} \tab Returns the website configuration for a bucket\cr
#'  \link[paws.storage:s3_get_object]{get_object} \tab Retrieves objects from Amazon S3\cr
#'  \link[paws.storage:s3_get_object_acl]{get_object_acl} \tab Returns the access control list (ACL) of an object\cr
#'  \link[paws.storage:s3_get_object_attributes]{get_object_attributes} \tab Retrieves all the metadata from an object without returning the object itself\cr
#'  \link[paws.storage:s3_get_object_legal_hold]{get_object_legal_hold} \tab Gets an object's current legal hold status\cr
#'  \link[paws.storage:s3_get_object_lock_configuration]{get_object_lock_configuration} \tab Gets the Object Lock configuration for a bucket\cr
#'  \link[paws.storage:s3_get_object_retention]{get_object_retention} \tab Retrieves an object's retention settings\cr
#'  \link[paws.storage:s3_get_object_tagging]{get_object_tagging} \tab Returns the tag-set of an object\cr
#'  \link[paws.storage:s3_get_object_torrent]{get_object_torrent} \tab Returns torrent files from a bucket\cr
#'  \link[paws.storage:s3_get_public_access_block]{get_public_access_block} \tab Retrieves the PublicAccessBlock configuration for an Amazon S3 bucket\cr
#'  \link[paws.storage:s3_head_bucket]{head_bucket} \tab This action is useful to determine if a bucket exists and you have permission to access it\cr
#'  \link[paws.storage:s3_head_object]{head_object} \tab The HEAD action retrieves metadata from an object without returning the object itself\cr
#'  \link[paws.storage:s3_list_bucket_analytics_configurations]{list_bucket_analytics_configurations} \tab Lists the analytics configurations for the bucket\cr
#'  \link[paws.storage:s3_list_bucket_intelligent_tiering_configurations]{list_bucket_intelligent_tiering_configurations} \tab Lists the S3 Intelligent-Tiering configuration from the specified bucket\cr
#'  \link[paws.storage:s3_list_bucket_inventory_configurations]{list_bucket_inventory_configurations} \tab Returns a list of inventory configurations for the bucket\cr
#'  \link[paws.storage:s3_list_bucket_metrics_configurations]{list_bucket_metrics_configurations} \tab Lists the metrics configurations for the bucket\cr
#'  \link[paws.storage:s3_list_buckets]{list_buckets} \tab Returns a list of all buckets owned by the authenticated sender of the request\cr
#'  \link[paws.storage:s3_list_multipart_uploads]{list_multipart_uploads} \tab This action lists in-progress multipart uploads\cr
#'  \link[paws.storage:s3_list_objects]{list_objects} \tab Returns some or all (up to 1,000) of the objects in a bucket\cr
#'  \link[paws.storage:s3_list_objects_v2]{list_objects_v2} \tab Returns some or all (up to 1,000) of the objects in a bucket with each request\cr
#'  \link[paws.storage:s3_list_object_versions]{list_object_versions} \tab Returns metadata about all versions of the objects in a bucket\cr
#'  \link[paws.storage:s3_list_parts]{list_parts} \tab Lists the parts that have been uploaded for a specific multipart upload\cr
#'  \link[paws.storage:s3_put_bucket_accelerate_configuration]{put_bucket_accelerate_configuration} \tab Sets the accelerate configuration of an existing bucket\cr
#'  \link[paws.storage:s3_put_bucket_acl]{put_bucket_acl} \tab Sets the permissions on an existing bucket using access control lists (ACL)\cr
#'  \link[paws.storage:s3_put_bucket_analytics_configuration]{put_bucket_analytics_configuration} \tab Sets an analytics configuration for the bucket (specified by the analytics configuration ID)\cr
#'  \link[paws.storage:s3_put_bucket_cors]{put_bucket_cors} \tab Sets the cors configuration for your bucket\cr
#'  \link[paws.storage:s3_put_bucket_encryption]{put_bucket_encryption} \tab This action uses the encryption subresource to configure default encryption and Amazon S3 Bucket Keys for an existing bucket\cr
#'  \link[paws.storage:s3_put_bucket_intelligent_tiering_configuration]{put_bucket_intelligent_tiering_configuration} \tab Puts a S3 Intelligent-Tiering configuration to the specified bucket\cr
#'  \link[paws.storage:s3_put_bucket_inventory_configuration]{put_bucket_inventory_configuration} \tab This implementation of the PUT action adds an inventory configuration (identified by the inventory ID) to the bucket\cr
#'  \link[paws.storage:s3_put_bucket_lifecycle]{put_bucket_lifecycle} \tab For an updated version of this API, see PutBucketLifecycleConfiguration\cr
#'  \link[paws.storage:s3_put_bucket_lifecycle_configuration]{put_bucket_lifecycle_configuration} \tab Creates a new lifecycle configuration for the bucket or replaces an existing lifecycle configuration\cr
#'  \link[paws.storage:s3_put_bucket_logging]{put_bucket_logging} \tab Set the logging parameters for a bucket and to specify permissions for who can view and modify the logging parameters\cr
#'  \link[paws.storage:s3_put_bucket_metrics_configuration]{put_bucket_metrics_configuration} \tab Sets a metrics configuration (specified by the metrics configuration ID) for the bucket\cr
#'  \link[paws.storage:s3_put_bucket_notification]{put_bucket_notification} \tab No longer used, see the PutBucketNotificationConfiguration operation\cr
#'  \link[paws.storage:s3_put_bucket_notification_configuration]{put_bucket_notification_configuration} \tab Enables notifications of specified events for a bucket\cr
#'  \link[paws.storage:s3_put_bucket_ownership_controls]{put_bucket_ownership_controls} \tab Creates or modifies OwnershipControls for an Amazon S3 bucket\cr
#'  \link[paws.storage:s3_put_bucket_policy]{put_bucket_policy} \tab Applies an Amazon S3 bucket policy to an Amazon S3 bucket\cr
#'  \link[paws.storage:s3_put_bucket_replication]{put_bucket_replication} \tab Creates a replication configuration or replaces an existing one\cr
#'  \link[paws.storage:s3_put_bucket_request_payment]{put_bucket_request_payment} \tab Sets the request payment configuration for a bucket\cr
#'  \link[paws.storage:s3_put_bucket_tagging]{put_bucket_tagging} \tab Sets the tags for a bucket\cr
#'  \link[paws.storage:s3_put_bucket_versioning]{put_bucket_versioning} \tab Sets the versioning state of an existing bucket\cr
#'  \link[paws.storage:s3_put_bucket_website]{put_bucket_website} \tab Sets the configuration of the website that is specified in the website subresource\cr
#'  \link[paws.storage:s3_put_object]{put_object} \tab Adds an object to a bucket\cr
#'  \link[paws.storage:s3_put_object_acl]{put_object_acl} \tab Uses the acl subresource to set the access control list (ACL) permissions for a new or existing object in an S3 bucket\cr
#'  \link[paws.storage:s3_put_object_legal_hold]{put_object_legal_hold} \tab Applies a legal hold configuration to the specified object\cr
#'  \link[paws.storage:s3_put_object_lock_configuration]{put_object_lock_configuration} \tab Places an Object Lock configuration on the specified bucket\cr
#'  \link[paws.storage:s3_put_object_retention]{put_object_retention} \tab Places an Object Retention configuration on an object\cr
#'  \link[paws.storage:s3_put_object_tagging]{put_object_tagging} \tab Sets the supplied tag-set to an object that already exists in a bucket\cr
#'  \link[paws.storage:s3_put_public_access_block]{put_public_access_block} \tab Creates or modifies the PublicAccessBlock configuration for an Amazon S3 bucket\cr
#'  \link[paws.storage:s3_restore_object]{restore_object} \tab Restores an archived copy of an object back into Amazon S3\cr
#'  \link[paws.storage:s3_select_object_content]{select_object_content} \tab This action filters the contents of an Amazon S3 object based on a simple structured query language (SQL) statement\cr
#'  \link[paws.storage:s3_upload_part]{upload_part} \tab Uploads a part in a multipart upload\cr
#'  \link[paws.storage:s3_upload_part_copy]{upload_part_copy} \tab Uploads a part by copying data from an existing object as data source\cr
#'  \link[paws.storage:s3_write_get_object_response]{write_get_object_response} \tab Passes transformed objects to a GetObject operation when using Object Lambda access points
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname s3
#' @export
s3 <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.storage::s3(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' AWS S3 Control
#'
#' @description
#' Amazon Web Services S3 Control provides access to Amazon S3 control
#' plane actions.
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- s3control(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- s3control()
#' svc$create_access_point(
#'   Foo = 123
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.storage:s3control_create_access_point]{create_access_point} \tab Creates an access point and associates it with the specified bucket\cr
#'  \link[paws.storage:s3control_create_access_point_for_object_lambda]{create_access_point_for_object_lambda} \tab Creates an Object Lambda Access Point\cr
#'  \link[paws.storage:s3control_create_bucket]{create_bucket} \tab This action creates an Amazon S3 on Outposts bucket\cr
#'  \link[paws.storage:s3control_create_job]{create_job} \tab You can use S3 Batch Operations to perform large-scale batch actions on Amazon S3 objects\cr
#'  \link[paws.storage:s3control_create_multi_region_access_point]{create_multi_region_access_point} \tab Creates a Multi-Region Access Point and associates it with the specified buckets\cr
#'  \link[paws.storage:s3control_delete_access_point]{delete_access_point} \tab Deletes the specified access point\cr
#'  \link[paws.storage:s3control_delete_access_point_for_object_lambda]{delete_access_point_for_object_lambda} \tab Deletes the specified Object Lambda Access Point\cr
#'  \link[paws.storage:s3control_delete_access_point_policy]{delete_access_point_policy} \tab Deletes the access point policy for the specified access point\cr
#'  \link[paws.storage:s3control_delete_access_point_policy_for_object_lambda]{delete_access_point_policy_for_object_lambda} \tab Removes the resource policy for an Object Lambda Access Point\cr
#'  \link[paws.storage:s3control_delete_bucket]{delete_bucket} \tab This action deletes an Amazon S3 on Outposts bucket\cr
#'  \link[paws.storage:s3control_delete_bucket_lifecycle_configuration]{delete_bucket_lifecycle_configuration} \tab This action deletes an Amazon S3 on Outposts bucket's lifecycle configuration\cr
#'  \link[paws.storage:s3control_delete_bucket_policy]{delete_bucket_policy} \tab This action deletes an Amazon S3 on Outposts bucket policy\cr
#'  \link[paws.storage:s3control_delete_bucket_replication]{delete_bucket_replication} \tab This operation deletes an Amazon S3 on Outposts bucket's replication configuration\cr
#'  \link[paws.storage:s3control_delete_bucket_tagging]{delete_bucket_tagging} \tab This action deletes an Amazon S3 on Outposts bucket's tags\cr
#'  \link[paws.storage:s3control_delete_job_tagging]{delete_job_tagging} \tab Removes the entire tag set from the specified S3 Batch Operations job\cr
#'  \link[paws.storage:s3control_delete_multi_region_access_point]{delete_multi_region_access_point} \tab Deletes a Multi-Region Access Point\cr
#'  \link[paws.storage:s3control_delete_public_access_block]{delete_public_access_block} \tab Removes the PublicAccessBlock configuration for an Amazon Web Services account\cr
#'  \link[paws.storage:s3control_delete_storage_lens_configuration]{delete_storage_lens_configuration} \tab Deletes the Amazon S3 Storage Lens configuration\cr
#'  \link[paws.storage:s3control_delete_storage_lens_configuration_tagging]{delete_storage_lens_configuration_tagging} \tab Deletes the Amazon S3 Storage Lens configuration tags\cr
#'  \link[paws.storage:s3control_describe_job]{describe_job} \tab Retrieves the configuration parameters and status for a Batch Operations job\cr
#'  \link[paws.storage:s3control_describe_multi_region_access_point_operation]{describe_multi_region_access_point_operation} \tab Retrieves the status of an asynchronous request to manage a Multi-Region Access Point\cr
#'  \link[paws.storage:s3control_get_access_point]{get_access_point} \tab Returns configuration information about the specified access point\cr
#'  \link[paws.storage:s3control_get_access_point_configuration_for_object_lambda]{get_access_point_configuration_for_object_lambda} \tab Returns configuration for an Object Lambda Access Point\cr
#'  \link[paws.storage:s3control_get_access_point_for_object_lambda]{get_access_point_for_object_lambda} \tab Returns configuration information about the specified Object Lambda Access Point\cr
#'  \link[paws.storage:s3control_get_access_point_policy]{get_access_point_policy} \tab Returns the access point policy associated with the specified access point\cr
#'  \link[paws.storage:s3control_get_access_point_policy_for_object_lambda]{get_access_point_policy_for_object_lambda} \tab Returns the resource policy for an Object Lambda Access Point\cr
#'  \link[paws.storage:s3control_get_access_point_policy_status]{get_access_point_policy_status} \tab Indicates whether the specified access point currently has a policy that allows public access\cr
#'  \link[paws.storage:s3control_get_access_point_policy_status_for_object_lambda]{get_access_point_policy_status_for_object_lambda} \tab Returns the status of the resource policy associated with an Object Lambda Access Point\cr
#'  \link[paws.storage:s3control_get_bucket]{get_bucket} \tab Gets an Amazon S3 on Outposts bucket\cr
#'  \link[paws.storage:s3control_get_bucket_lifecycle_configuration]{get_bucket_lifecycle_configuration} \tab This action gets an Amazon S3 on Outposts bucket's lifecycle configuration\cr
#'  \link[paws.storage:s3control_get_bucket_policy]{get_bucket_policy} \tab This action gets a bucket policy for an Amazon S3 on Outposts bucket\cr
#'  \link[paws.storage:s3control_get_bucket_replication]{get_bucket_replication} \tab This operation gets an Amazon S3 on Outposts bucket's replication configuration\cr
#'  \link[paws.storage:s3control_get_bucket_tagging]{get_bucket_tagging} \tab This action gets an Amazon S3 on Outposts bucket's tags\cr
#'  \link[paws.storage:s3control_get_bucket_versioning]{get_bucket_versioning} \tab This operation returns the versioning state for S3 on Outposts buckets only\cr
#'  \link[paws.storage:s3control_get_job_tagging]{get_job_tagging} \tab Returns the tags on an S3 Batch Operations job\cr
#'  \link[paws.storage:s3control_get_multi_region_access_point]{get_multi_region_access_point} \tab Returns configuration information about the specified Multi-Region Access Point\cr
#'  \link[paws.storage:s3control_get_multi_region_access_point_policy]{get_multi_region_access_point_policy} \tab Returns the access control policy of the specified Multi-Region Access Point\cr
#'  \link[paws.storage:s3control_get_multi_region_access_point_policy_status]{get_multi_region_access_point_policy_status} \tab Indicates whether the specified Multi-Region Access Point has an access control policy that allows public access\cr
#'  \link[paws.storage:s3control_get_multi_region_access_point_routes]{get_multi_region_access_point_routes} \tab Returns the routing configuration for a Multi-Region Access Point, indicating which Regions are active or passive\cr
#'  \link[paws.storage:s3control_get_public_access_block]{get_public_access_block} \tab Retrieves the PublicAccessBlock configuration for an Amazon Web Services account\cr
#'  \link[paws.storage:s3control_get_storage_lens_configuration]{get_storage_lens_configuration} \tab Gets the Amazon S3 Storage Lens configuration\cr
#'  \link[paws.storage:s3control_get_storage_lens_configuration_tagging]{get_storage_lens_configuration_tagging} \tab Gets the tags of Amazon S3 Storage Lens configuration\cr
#'  \link[paws.storage:s3control_list_access_points]{list_access_points} \tab Returns a list of the access points that are owned by the current account that's associated with the specified bucket\cr
#'  \link[paws.storage:s3control_list_access_points_for_object_lambda]{list_access_points_for_object_lambda} \tab Returns some or all (up to 1,000) access points associated with the Object Lambda Access Point per call\cr
#'  \link[paws.storage:s3control_list_jobs]{list_jobs} \tab Lists current S3 Batch Operations jobs and jobs that have ended within the last 30 days for the Amazon Web Services account making the request\cr
#'  \link[paws.storage:s3control_list_multi_region_access_points]{list_multi_region_access_points} \tab Returns a list of the Multi-Region Access Points currently associated with the specified Amazon Web Services account\cr
#'  \link[paws.storage:s3control_list_regional_buckets]{list_regional_buckets} \tab Returns a list of all Outposts buckets in an Outpost that are owned by the authenticated sender of the request\cr
#'  \link[paws.storage:s3control_list_storage_lens_configurations]{list_storage_lens_configurations} \tab Gets a list of Amazon S3 Storage Lens configurations\cr
#'  \link[paws.storage:s3control_put_access_point_configuration_for_object_lambda]{put_access_point_configuration_for_object_lambda} \tab Replaces configuration for an Object Lambda Access Point\cr
#'  \link[paws.storage:s3control_put_access_point_policy]{put_access_point_policy} \tab Associates an access policy with the specified access point\cr
#'  \link[paws.storage:s3control_put_access_point_policy_for_object_lambda]{put_access_point_policy_for_object_lambda} \tab Creates or replaces resource policy for an Object Lambda Access Point\cr
#'  \link[paws.storage:s3control_put_bucket_lifecycle_configuration]{put_bucket_lifecycle_configuration} \tab This action puts a lifecycle configuration to an Amazon S3 on Outposts bucket\cr
#'  \link[paws.storage:s3control_put_bucket_policy]{put_bucket_policy} \tab This action puts a bucket policy to an Amazon S3 on Outposts bucket\cr
#'  \link[paws.storage:s3control_put_bucket_replication]{put_bucket_replication} \tab This action creates an Amazon S3 on Outposts bucket's replication configuration\cr
#'  \link[paws.storage:s3control_put_bucket_tagging]{put_bucket_tagging} \tab This action puts tags on an Amazon S3 on Outposts bucket\cr
#'  \link[paws.storage:s3control_put_bucket_versioning]{put_bucket_versioning} \tab This operation sets the versioning state for S3 on Outposts buckets only\cr
#'  \link[paws.storage:s3control_put_job_tagging]{put_job_tagging} \tab Sets the supplied tag-set on an S3 Batch Operations job\cr
#'  \link[paws.storage:s3control_put_multi_region_access_point_policy]{put_multi_region_access_point_policy} \tab Associates an access control policy with the specified Multi-Region Access Point\cr
#'  \link[paws.storage:s3control_put_public_access_block]{put_public_access_block} \tab Creates or modifies the PublicAccessBlock configuration for an Amazon Web Services account\cr
#'  \link[paws.storage:s3control_put_storage_lens_configuration]{put_storage_lens_configuration} \tab Puts an Amazon S3 Storage Lens configuration\cr
#'  \link[paws.storage:s3control_put_storage_lens_configuration_tagging]{put_storage_lens_configuration_tagging} \tab Put or replace tags on an existing Amazon S3 Storage Lens configuration\cr
#'  \link[paws.storage:s3control_submit_multi_region_access_point_routes]{submit_multi_region_access_point_routes} \tab Submits an updated route configuration for a Multi-Region Access Point\cr
#'  \link[paws.storage:s3control_update_job_priority]{update_job_priority} \tab Updates an existing S3 Batch Operations job's priority\cr
#'  \link[paws.storage:s3control_update_job_status]{update_job_status} \tab Updates the status for the specified job
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname s3control
#' @export
s3control <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.storage::s3control(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' Amazon S3 on Outposts
#'
#' @description
#' Amazon S3 on Outposts provides access to S3 on Outposts operations.
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- s3outposts(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- s3outposts()
#' svc$create_endpoint(
#'   Foo = 123
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.storage:s3outposts_create_endpoint]{create_endpoint} \tab Creates an endpoint and associates it with the specified Outpost\cr
#'  \link[paws.storage:s3outposts_delete_endpoint]{delete_endpoint} \tab Deletes an endpoint\cr
#'  \link[paws.storage:s3outposts_list_endpoints]{list_endpoints} \tab Lists endpoints associated with the specified Outpost\cr
#'  \link[paws.storage:s3outposts_list_outposts_with_s3]{list_outposts_with_s3} \tab Lists the Outposts with S3 on Outposts capacity for your Amazon Web Services account\cr
#'  \link[paws.storage:s3outposts_list_shared_endpoints]{list_shared_endpoints} \tab Lists all endpoints associated with an Outpost that has been shared by Amazon Web Services Resource Access Manager (RAM)
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname s3outposts
#' @export
s3outposts <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.storage::s3outposts(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' AWS Storage Gateway
#'
#' @description
#' Storage Gateway Service
#'
#' Storage Gateway is the service that connects an on-premises software
#' appliance with cloud-based storage to provide seamless and secure
#' integration between an organization's on-premises IT environment and the
#' Amazon Web Services storage infrastructure. The service enables you to
#' securely upload data to the Amazon Web Services Cloud for cost effective
#' backup and rapid disaster recovery.
#'
#' Use the following links to get started using the *Storage Gateway
#' Service API Reference*:
#'
#' -   [Storage Gateway required request
#'     headers](https://docs.aws.amazon.com/storagegateway/#AWSStorageGatewayHTTPRequestsHeaders):
#'     Describes the required headers that you must send with every POST
#'     request to Storage Gateway.
#'
#' -   [Signing
#'     requests](https://docs.aws.amazon.com/storagegateway/#AWSStorageGatewaySigningRequests):
#'     Storage Gateway requires that you authenticate every request you
#'     send; this topic describes how sign such a request.
#'
#' -   [Error
#'     responses](https://docs.aws.amazon.com/storagegateway/#APIErrorResponses):
#'     Provides reference information about Storage Gateway errors.
#'
#' -   [Operations in Storage
#'     Gateway](https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_Operations.html):
#'     Contains detailed descriptions of all Storage Gateway operations,
#'     their request parameters, response elements, possible errors, and
#'     examples of requests and responses.
#'
#' -   [Storage Gateway endpoints and
#'     quotas](https://docs.aws.amazon.com/general/latest/gr/sg.html):
#'     Provides a list of each Amazon Web Services Region and the endpoints
#'     available for use with Storage Gateway.
#'
#' Storage Gateway resource IDs are in uppercase. When you use these
#' resource IDs with the Amazon EC2 API, EC2 expects resource IDs in
#' lowercase. You must change your resource ID to lowercase to use it with
#' the EC2 API. For example, in Storage Gateway the ID for a volume might
#' be `vol-AA22BB012345DAF670`. When you use this ID with the EC2 API, you
#' must change it to `vol-aa22bb012345daf670`. Otherwise, the EC2 API might
#' not behave as expected.
#'
#' IDs for Storage Gateway volumes and Amazon EBS snapshots created from
#' gateway volumes are changing to a longer format. Starting in December
#' 2016, all new volumes and snapshots will be created with a 17-character
#' string. Starting in April 2016, you will be able to use these longer IDs
#' so you can test your systems with the new format. For more information,
#' see [Longer EC2 and EBS resource
#' IDs](https://aws.amazon.com/ec2/faqs/#longer-ids).
#'
#' For example, a volume Amazon Resource Name (ARN) with the longer volume
#' ID format looks like the following:
#'
#' `arn:aws:storagegateway:us-west-2:111122223333:gateway/sgw-12A3456B/volume/vol-1122AABBCCDDEEFFG`.
#'
#' A snapshot ID with the longer ID format looks like the following:
#' `snap-78e226633445566ee`.
#'
#' For more information, see Announcement: Heads-up – Longer Storage
#' Gateway volume and snapshot IDs coming in 2016.
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- storagegateway(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- storagegateway()
#' # Activates the gateway you previously deployed on your host.
#' svc$activate_gateway(
#'   ActivationKey = "29AV1-3OFV9-VVIUB-NKT0I-LRO6V",
#'   GatewayName = "My_Gateway",
#'   GatewayRegion = "us-east-1",
#'   GatewayTimezone = "GMT-12:00",
#'   GatewayType = "STORED",
#'   MediumChangerType = "AWS-Gateway-VTL",
#'   TapeDriveType = "IBM-ULT3580-TD5"
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.storage:storagegateway_activate_gateway]{activate_gateway} \tab Activates the gateway you previously deployed on your host\cr
#'  \link[paws.storage:storagegateway_add_cache]{add_cache} \tab Configures one or more gateway local disks as cache for a gateway\cr
#'  \link[paws.storage:storagegateway_add_tags_to_resource]{add_tags_to_resource} \tab Adds one or more tags to the specified resource\cr
#'  \link[paws.storage:storagegateway_add_upload_buffer]{add_upload_buffer} \tab Configures one or more gateway local disks as upload buffer for a specified gateway\cr
#'  \link[paws.storage:storagegateway_add_working_storage]{add_working_storage} \tab Configures one or more gateway local disks as working storage for a gateway\cr
#'  \link[paws.storage:storagegateway_assign_tape_pool]{assign_tape_pool} \tab Assigns a tape to a tape pool for archiving\cr
#'  \link[paws.storage:storagegateway_associate_file_system]{associate_file_system} \tab Associate an Amazon FSx file system with the FSx File Gateway\cr
#'  \link[paws.storage:storagegateway_attach_volume]{attach_volume} \tab Connects a volume to an iSCSI connection and then attaches the volume to the specified gateway\cr
#'  \link[paws.storage:storagegateway_cancel_archival]{cancel_archival} \tab Cancels archiving of a virtual tape to the virtual tape shelf (VTS) after the archiving process is initiated\cr
#'  \link[paws.storage:storagegateway_cancel_retrieval]{cancel_retrieval} \tab Cancels retrieval of a virtual tape from the virtual tape shelf (VTS) to a gateway after the retrieval process is initiated\cr
#'  \link[paws.storage:storagegateway_create_cachedi_scsi_volume]{create_cachedi_scsi_volume} \tab Creates a cached volume on a specified cached volume gateway\cr
#'  \link[paws.storage:storagegateway_create_nfs_file_share]{create_nfs_file_share} \tab Creates a Network File System (NFS) file share on an existing S3 File Gateway\cr
#'  \link[paws.storage:storagegateway_create_smb_file_share]{create_smb_file_share} \tab Creates a Server Message Block (SMB) file share on an existing S3 File Gateway\cr
#'  \link[paws.storage:storagegateway_create_snapshot]{create_snapshot} \tab Initiates a snapshot of a volume\cr
#'  \link[paws.storage:storagegateway_create_snapshot_from_volume_recovery_point]{create_snapshot_from_volume_recovery_point} \tab Initiates a snapshot of a gateway from a volume recovery point\cr
#'  \link[paws.storage:storagegateway_create_storedi_scsi_volume]{create_storedi_scsi_volume} \tab Creates a volume on a specified gateway\cr
#'  \link[paws.storage:storagegateway_create_tape_pool]{create_tape_pool} \tab Creates a new custom tape pool\cr
#'  \link[paws.storage:storagegateway_create_tapes]{create_tapes} \tab Creates one or more virtual tapes\cr
#'  \link[paws.storage:storagegateway_create_tape_with_barcode]{create_tape_with_barcode} \tab Creates a virtual tape by using your own barcode\cr
#'  \link[paws.storage:storagegateway_delete_automatic_tape_creation_policy]{delete_automatic_tape_creation_policy} \tab Deletes the automatic tape creation policy of a gateway\cr
#'  \link[paws.storage:storagegateway_delete_bandwidth_rate_limit]{delete_bandwidth_rate_limit} \tab Deletes the bandwidth rate limits of a gateway\cr
#'  \link[paws.storage:storagegateway_delete_chap_credentials]{delete_chap_credentials} \tab Deletes Challenge-Handshake Authentication Protocol (CHAP) credentials for a specified iSCSI target and initiator pair\cr
#'  \link[paws.storage:storagegateway_delete_file_share]{delete_file_share} \tab Deletes a file share from an S3 File Gateway\cr
#'  \link[paws.storage:storagegateway_delete_gateway]{delete_gateway} \tab Deletes a gateway\cr
#'  \link[paws.storage:storagegateway_delete_snapshot_schedule]{delete_snapshot_schedule} \tab Deletes a snapshot of a volume\cr
#'  \link[paws.storage:storagegateway_delete_tape]{delete_tape} \tab Deletes the specified virtual tape\cr
#'  \link[paws.storage:storagegateway_delete_tape_archive]{delete_tape_archive} \tab Deletes the specified virtual tape from the virtual tape shelf (VTS)\cr
#'  \link[paws.storage:storagegateway_delete_tape_pool]{delete_tape_pool} \tab Delete a custom tape pool\cr
#'  \link[paws.storage:storagegateway_delete_volume]{delete_volume} \tab Deletes the specified storage volume that you previously created using the CreateCachediSCSIVolume or CreateStorediSCSIVolume API\cr
#'  \link[paws.storage:storagegateway_describe_availability_monitor_test]{describe_availability_monitor_test} \tab Returns information about the most recent high availability monitoring test that was performed on the host in a cluster\cr
#'  \link[paws.storage:storagegateway_describe_bandwidth_rate_limit]{describe_bandwidth_rate_limit} \tab Returns the bandwidth rate limits of a gateway\cr
#'  \link[paws.storage:storagegateway_describe_bandwidth_rate_limit_schedule]{describe_bandwidth_rate_limit_schedule} \tab Returns information about the bandwidth rate limit schedule of a gateway\cr
#'  \link[paws.storage:storagegateway_describe_cache]{describe_cache} \tab Returns information about the cache of a gateway\cr
#'  \link[paws.storage:storagegateway_describe_cachedi_scsi_volumes]{describe_cachedi_scsi_volumes} \tab Returns a description of the gateway volumes specified in the request\cr
#'  \link[paws.storage:storagegateway_describe_chap_credentials]{describe_chap_credentials} \tab Returns an array of Challenge-Handshake Authentication Protocol (CHAP) credentials information for a specified iSCSI target, one for each target-initiator pair\cr
#'  \link[paws.storage:storagegateway_describe_file_system_associations]{describe_file_system_associations} \tab Gets the file system association information\cr
#'  \link[paws.storage:storagegateway_describe_gateway_information]{describe_gateway_information} \tab Returns metadata about a gateway such as its name, network interfaces, configured time zone, and the state (whether the gateway is running or not)\cr
#'  \link[paws.storage:storagegateway_describe_maintenance_start_time]{describe_maintenance_start_time} \tab Returns your gateway's weekly maintenance start time including the day and time of the week\cr
#'  \link[paws.storage:storagegateway_describe_nfs_file_shares]{describe_nfs_file_shares} \tab Gets a description for one or more Network File System (NFS) file shares from an S3 File Gateway\cr
#'  \link[paws.storage:storagegateway_describe_smb_file_shares]{describe_smb_file_shares} \tab Gets a description for one or more Server Message Block (SMB) file shares from a S3 File Gateway\cr
#'  \link[paws.storage:storagegateway_describe_smb_settings]{describe_smb_settings} \tab Gets a description of a Server Message Block (SMB) file share settings from a file gateway\cr
#'  \link[paws.storage:storagegateway_describe_snapshot_schedule]{describe_snapshot_schedule} \tab Describes the snapshot schedule for the specified gateway volume\cr
#'  \link[paws.storage:storagegateway_describe_storedi_scsi_volumes]{describe_storedi_scsi_volumes} \tab Returns the description of the gateway volumes specified in the request\cr
#'  \link[paws.storage:storagegateway_describe_tape_archives]{describe_tape_archives} \tab Returns a description of specified virtual tapes in the virtual tape shelf (VTS)\cr
#'  \link[paws.storage:storagegateway_describe_tape_recovery_points]{describe_tape_recovery_points} \tab Returns a list of virtual tape recovery points that are available for the specified tape gateway\cr
#'  \link[paws.storage:storagegateway_describe_tapes]{describe_tapes} \tab Returns a description of the specified Amazon Resource Name (ARN) of virtual tapes\cr
#'  \link[paws.storage:storagegateway_describe_upload_buffer]{describe_upload_buffer} \tab Returns information about the upload buffer of a gateway\cr
#'  \link[paws.storage:storagegateway_describe_vtl_devices]{describe_vtl_devices} \tab Returns a description of virtual tape library (VTL) devices for the specified tape gateway\cr
#'  \link[paws.storage:storagegateway_describe_working_storage]{describe_working_storage} \tab Returns information about the working storage of a gateway\cr
#'  \link[paws.storage:storagegateway_detach_volume]{detach_volume} \tab Disconnects a volume from an iSCSI connection and then detaches the volume from the specified gateway\cr
#'  \link[paws.storage:storagegateway_disable_gateway]{disable_gateway} \tab Disables a tape gateway when the gateway is no longer functioning\cr
#'  \link[paws.storage:storagegateway_disassociate_file_system]{disassociate_file_system} \tab Disassociates an Amazon FSx file system from the specified gateway\cr
#'  \link[paws.storage:storagegateway_join_domain]{join_domain} \tab Adds a file gateway to an Active Directory domain\cr
#'  \link[paws.storage:storagegateway_list_automatic_tape_creation_policies]{list_automatic_tape_creation_policies} \tab Lists the automatic tape creation policies for a gateway\cr
#'  \link[paws.storage:storagegateway_list_file_shares]{list_file_shares} \tab Gets a list of the file shares for a specific S3 File Gateway, or the list of file shares that belong to the calling user account\cr
#'  \link[paws.storage:storagegateway_list_file_system_associations]{list_file_system_associations} \tab Gets a list of FileSystemAssociationSummary objects\cr
#'  \link[paws.storage:storagegateway_list_gateways]{list_gateways} \tab Lists gateways owned by an Amazon Web Services account in an Amazon Web Services Region specified in the request\cr
#'  \link[paws.storage:storagegateway_list_local_disks]{list_local_disks} \tab Returns a list of the gateway's local disks\cr
#'  \link[paws.storage:storagegateway_list_tags_for_resource]{list_tags_for_resource} \tab Lists the tags that have been added to the specified resource\cr
#'  \link[paws.storage:storagegateway_list_tape_pools]{list_tape_pools} \tab Lists custom tape pools\cr
#'  \link[paws.storage:storagegateway_list_tapes]{list_tapes} \tab Lists virtual tapes in your virtual tape library (VTL) and your virtual tape shelf (VTS)\cr
#'  \link[paws.storage:storagegateway_list_volume_initiators]{list_volume_initiators} \tab Lists iSCSI initiators that are connected to a volume\cr
#'  \link[paws.storage:storagegateway_list_volume_recovery_points]{list_volume_recovery_points} \tab Lists the recovery points for a specified gateway\cr
#'  \link[paws.storage:storagegateway_list_volumes]{list_volumes} \tab Lists the iSCSI stored volumes of a gateway\cr
#'  \link[paws.storage:storagegateway_notify_when_uploaded]{notify_when_uploaded} \tab Sends you notification through CloudWatch Events when all files written to your file share have been uploaded to S3\cr
#'  \link[paws.storage:storagegateway_refresh_cache]{refresh_cache} \tab Refreshes the cached inventory of objects for the specified file share\cr
#'  \link[paws.storage:storagegateway_remove_tags_from_resource]{remove_tags_from_resource} \tab Removes one or more tags from the specified resource\cr
#'  \link[paws.storage:storagegateway_reset_cache]{reset_cache} \tab Resets all cache disks that have encountered an error and makes the disks available for reconfiguration as cache storage\cr
#'  \link[paws.storage:storagegateway_retrieve_tape_archive]{retrieve_tape_archive} \tab Retrieves an archived virtual tape from the virtual tape shelf (VTS) to a tape gateway\cr
#'  \link[paws.storage:storagegateway_retrieve_tape_recovery_point]{retrieve_tape_recovery_point} \tab Retrieves the recovery point for the specified virtual tape\cr
#'  \link[paws.storage:storagegateway_set_local_console_password]{set_local_console_password} \tab Sets the password for your VM local console\cr
#'  \link[paws.storage:storagegateway_set_smb_guest_password]{set_smb_guest_password} \tab Sets the password for the guest user smbguest\cr
#'  \link[paws.storage:storagegateway_shutdown_gateway]{shutdown_gateway} \tab Shuts down a gateway\cr
#'  \link[paws.storage:storagegateway_start_availability_monitor_test]{start_availability_monitor_test} \tab Start a test that verifies that the specified gateway is configured for High Availability monitoring in your host environment\cr
#'  \link[paws.storage:storagegateway_start_gateway]{start_gateway} \tab Starts a gateway that you previously shut down (see ShutdownGateway)\cr
#'  \link[paws.storage:storagegateway_update_automatic_tape_creation_policy]{update_automatic_tape_creation_policy} \tab Updates the automatic tape creation policy of a gateway\cr
#'  \link[paws.storage:storagegateway_update_bandwidth_rate_limit]{update_bandwidth_rate_limit} \tab Updates the bandwidth rate limits of a gateway\cr
#'  \link[paws.storage:storagegateway_update_bandwidth_rate_limit_schedule]{update_bandwidth_rate_limit_schedule} \tab Updates the bandwidth rate limit schedule for a specified gateway\cr
#'  \link[paws.storage:storagegateway_update_chap_credentials]{update_chap_credentials} \tab Updates the Challenge-Handshake Authentication Protocol (CHAP) credentials for a specified iSCSI target\cr
#'  \link[paws.storage:storagegateway_update_file_system_association]{update_file_system_association} \tab Updates a file system association\cr
#'  \link[paws.storage:storagegateway_update_gateway_information]{update_gateway_information} \tab Updates a gateway's metadata, which includes the gateway's name and time zone\cr
#'  \link[paws.storage:storagegateway_update_gateway_software_now]{update_gateway_software_now} \tab Updates the gateway virtual machine (VM) software\cr
#'  \link[paws.storage:storagegateway_update_maintenance_start_time]{update_maintenance_start_time} \tab Updates a gateway's weekly maintenance start time information, including day and time of the week\cr
#'  \link[paws.storage:storagegateway_update_nfs_file_share]{update_nfs_file_share} \tab Updates a Network File System (NFS) file share\cr
#'  \link[paws.storage:storagegateway_update_smb_file_share]{update_smb_file_share} \tab Updates a Server Message Block (SMB) file share\cr
#'  \link[paws.storage:storagegateway_update_smb_file_share_visibility]{update_smb_file_share_visibility} \tab Controls whether the shares on an S3 File Gateway are visible in a net view or browse list\cr
#'  \link[paws.storage:storagegateway_update_smb_local_groups]{update_smb_local_groups} \tab Updates the list of Active Directory users and groups that have special permissions for SMB file shares on the gateway\cr
#'  \link[paws.storage:storagegateway_update_smb_security_strategy]{update_smb_security_strategy} \tab Updates the SMB security strategy on a file gateway\cr
#'  \link[paws.storage:storagegateway_update_snapshot_schedule]{update_snapshot_schedule} \tab Updates a snapshot schedule configured for a gateway volume\cr
#'  \link[paws.storage:storagegateway_update_vtl_device_type]{update_vtl_device_type} \tab Updates the type of medium changer in a tape gateway
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname storagegateway
#' @export
storagegateway <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.storage::storagegateway(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' Amazon DynamoDB Accelerator (DAX)
#'
#' @description
#' DAX is a managed caching service engineered for Amazon DynamoDB. DAX
#' dramatically speeds up database reads by caching frequently-accessed
#' data from DynamoDB, so applications can access that data with
#' sub-millisecond latency. You can create a DAX cluster easily, using the
#' AWS Management Console. With a few simple modifications to your code,
#' your application can begin taking advantage of the DAX cluster and
#' realize significant improvements in read performance.
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- dax(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- dax()
#' svc$create_cluster(
#'   Foo = 123
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.database:dax_create_cluster]{create_cluster} \tab Creates a DAX cluster\cr
#'  \link[paws.database:dax_create_parameter_group]{create_parameter_group} \tab Creates a new parameter group\cr
#'  \link[paws.database:dax_create_subnet_group]{create_subnet_group} \tab Creates a new subnet group\cr
#'  \link[paws.database:dax_decrease_replication_factor]{decrease_replication_factor} \tab Removes one or more nodes from a DAX cluster\cr
#'  \link[paws.database:dax_delete_cluster]{delete_cluster} \tab Deletes a previously provisioned DAX cluster\cr
#'  \link[paws.database:dax_delete_parameter_group]{delete_parameter_group} \tab Deletes the specified parameter group\cr
#'  \link[paws.database:dax_delete_subnet_group]{delete_subnet_group} \tab Deletes a subnet group\cr
#'  \link[paws.database:dax_describe_clusters]{describe_clusters} \tab Returns information about all provisioned DAX clusters if no cluster identifier is specified, or about a specific DAX cluster if a cluster identifier is supplied\cr
#'  \link[paws.database:dax_describe_default_parameters]{describe_default_parameters} \tab Returns the default system parameter information for the DAX caching software\cr
#'  \link[paws.database:dax_describe_events]{describe_events} \tab Returns events related to DAX clusters and parameter groups\cr
#'  \link[paws.database:dax_describe_parameter_groups]{describe_parameter_groups} \tab Returns a list of parameter group descriptions\cr
#'  \link[paws.database:dax_describe_parameters]{describe_parameters} \tab Returns the detailed parameter list for a particular parameter group\cr
#'  \link[paws.database:dax_describe_subnet_groups]{describe_subnet_groups} \tab Returns a list of subnet group descriptions\cr
#'  \link[paws.database:dax_increase_replication_factor]{increase_replication_factor} \tab Adds one or more nodes to a DAX cluster\cr
#'  \link[paws.database:dax_list_tags]{list_tags} \tab List all of the tags for a DAX cluster\cr
#'  \link[paws.database:dax_reboot_node]{reboot_node} \tab Reboots a single node of a DAX cluster\cr
#'  \link[paws.database:dax_tag_resource]{tag_resource} \tab Associates a set of tags with a DAX resource\cr
#'  \link[paws.database:dax_untag_resource]{untag_resource} \tab Removes the association of tags from a DAX resource\cr
#'  \link[paws.database:dax_update_cluster]{update_cluster} \tab Modifies the settings for a DAX cluster\cr
#'  \link[paws.database:dax_update_parameter_group]{update_parameter_group} \tab Modifies the parameters of a parameter group\cr
#'  \link[paws.database:dax_update_subnet_group]{update_subnet_group} \tab Modifies an existing subnet group
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname dax
#' @export
dax <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.database::dax(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' Amazon DocumentDB with MongoDB compatibility
#'
#' @description
#' Amazon DocumentDB is a fast, reliable, and fully managed database
#' service. Amazon DocumentDB makes it easy to set up, operate, and scale
#' MongoDB-compatible databases in the cloud. With Amazon DocumentDB, you
#' can run the same application code and use the same drivers and tools
#' that you use with MongoDB.
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- docdb(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- docdb()
#' svc$add_source_identifier_to_subscription(
#'   Foo = 123
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.database:docdb_add_source_identifier_to_subscription]{add_source_identifier_to_subscription} \tab Adds a source identifier to an existing event notification subscription\cr
#'  \link[paws.database:docdb_add_tags_to_resource]{add_tags_to_resource} \tab Adds metadata tags to an Amazon DocumentDB resource\cr
#'  \link[paws.database:docdb_apply_pending_maintenance_action]{apply_pending_maintenance_action} \tab Applies a pending maintenance action to a resource (for example, to an Amazon DocumentDB instance)\cr
#'  \link[paws.database:docdb_copy_db_cluster_parameter_group]{copy_db_cluster_parameter_group} \tab Copies the specified cluster parameter group\cr
#'  \link[paws.database:docdb_copy_db_cluster_snapshot]{copy_db_cluster_snapshot} \tab Copies a snapshot of a cluster\cr
#'  \link[paws.database:docdb_create_db_cluster]{create_db_cluster} \tab Creates a new Amazon DocumentDB cluster\cr
#'  \link[paws.database:docdb_create_db_cluster_parameter_group]{create_db_cluster_parameter_group} \tab Creates a new cluster parameter group\cr
#'  \link[paws.database:docdb_create_db_cluster_snapshot]{create_db_cluster_snapshot} \tab Creates a snapshot of a cluster\cr
#'  \link[paws.database:docdb_create_db_instance]{create_db_instance} \tab Creates a new instance\cr
#'  \link[paws.database:docdb_create_db_subnet_group]{create_db_subnet_group} \tab Creates a new subnet group\cr
#'  \link[paws.database:docdb_create_event_subscription]{create_event_subscription} \tab Creates an Amazon DocumentDB event notification subscription\cr
#'  \link[paws.database:docdb_create_global_cluster]{create_global_cluster} \tab Creates an Amazon DocumentDB global cluster that can span multiple multiple Amazon Web Services Regions\cr
#'  \link[paws.database:docdb_delete_db_cluster]{delete_db_cluster} \tab Deletes a previously provisioned cluster\cr
#'  \link[paws.database:docdb_delete_db_cluster_parameter_group]{delete_db_cluster_parameter_group} \tab Deletes a specified cluster parameter group\cr
#'  \link[paws.database:docdb_delete_db_cluster_snapshot]{delete_db_cluster_snapshot} \tab Deletes a cluster snapshot\cr
#'  \link[paws.database:docdb_delete_db_instance]{delete_db_instance} \tab Deletes a previously provisioned instance\cr
#'  \link[paws.database:docdb_delete_db_subnet_group]{delete_db_subnet_group} \tab Deletes a subnet group\cr
#'  \link[paws.database:docdb_delete_event_subscription]{delete_event_subscription} \tab Deletes an Amazon DocumentDB event notification subscription\cr
#'  \link[paws.database:docdb_delete_global_cluster]{delete_global_cluster} \tab Deletes a global cluster\cr
#'  \link[paws.database:docdb_describe_certificates]{describe_certificates} \tab Returns a list of certificate authority (CA) certificates provided by Amazon DocumentDB for this Amazon Web Services account\cr
#'  \link[paws.database:docdb_describe_db_cluster_parameter_groups]{describe_db_cluster_parameter_groups} \tab Returns a list of DBClusterParameterGroup descriptions\cr
#'  \link[paws.database:docdb_describe_db_cluster_parameters]{describe_db_cluster_parameters} \tab Returns the detailed parameter list for a particular cluster parameter group\cr
#'  \link[paws.database:docdb_describe_db_clusters]{describe_db_clusters} \tab Returns information about provisioned Amazon DocumentDB clusters\cr
#'  \link[paws.database:docdb_describe_db_cluster_snapshot_attributes]{describe_db_cluster_snapshot_attributes} \tab Returns a list of cluster snapshot attribute names and values for a manual DB cluster snapshot\cr
#'  \link[paws.database:docdb_describe_db_cluster_snapshots]{describe_db_cluster_snapshots} \tab Returns information about cluster snapshots\cr
#'  \link[paws.database:docdb_describe_db_engine_versions]{describe_db_engine_versions} \tab Returns a list of the available engines\cr
#'  \link[paws.database:docdb_describe_db_instances]{describe_db_instances} \tab Returns information about provisioned Amazon DocumentDB instances\cr
#'  \link[paws.database:docdb_describe_db_subnet_groups]{describe_db_subnet_groups} \tab Returns a list of DBSubnetGroup descriptions\cr
#'  \link[paws.database:docdb_describe_engine_default_cluster_parameters]{describe_engine_default_cluster_parameters} \tab Returns the default engine and system parameter information for the cluster database engine\cr
#'  \link[paws.database:docdb_describe_event_categories]{describe_event_categories} \tab Displays a list of categories for all event source types, or, if specified, for a specified source type\cr
#'  \link[paws.database:docdb_describe_events]{describe_events} \tab Returns events related to instances, security groups, snapshots, and DB parameter groups for the past 14 days\cr
#'  \link[paws.database:docdb_describe_event_subscriptions]{describe_event_subscriptions} \tab Lists all the subscription descriptions for a customer account\cr
#'  \link[paws.database:docdb_describe_global_clusters]{describe_global_clusters} \tab Returns information about Amazon DocumentDB global clusters\cr
#'  \link[paws.database:docdb_describe_orderable_db_instance_options]{describe_orderable_db_instance_options} \tab Returns a list of orderable instance options for the specified engine\cr
#'  \link[paws.database:docdb_describe_pending_maintenance_actions]{describe_pending_maintenance_actions} \tab Returns a list of resources (for example, instances) that have at least one pending maintenance action\cr
#'  \link[paws.database:docdb_failover_db_cluster]{failover_db_cluster} \tab Forces a failover for a cluster\cr
#'  \link[paws.database:docdb_list_tags_for_resource]{list_tags_for_resource} \tab Lists all tags on an Amazon DocumentDB resource\cr
#'  \link[paws.database:docdb_modify_db_cluster]{modify_db_cluster} \tab Modifies a setting for an Amazon DocumentDB cluster\cr
#'  \link[paws.database:docdb_modify_db_cluster_parameter_group]{modify_db_cluster_parameter_group} \tab Modifies the parameters of a cluster parameter group\cr
#'  \link[paws.database:docdb_modify_db_cluster_snapshot_attribute]{modify_db_cluster_snapshot_attribute} \tab Adds an attribute and values to, or removes an attribute and values from, a manual cluster snapshot\cr
#'  \link[paws.database:docdb_modify_db_instance]{modify_db_instance} \tab Modifies settings for an instance\cr
#'  \link[paws.database:docdb_modify_db_subnet_group]{modify_db_subnet_group} \tab Modifies an existing subnet group\cr
#'  \link[paws.database:docdb_modify_event_subscription]{modify_event_subscription} \tab Modifies an existing Amazon DocumentDB event notification subscription\cr
#'  \link[paws.database:docdb_modify_global_cluster]{modify_global_cluster} \tab Modify a setting for an Amazon DocumentDB global cluster\cr
#'  \link[paws.database:docdb_reboot_db_instance]{reboot_db_instance} \tab You might need to reboot your instance, usually for maintenance reasons\cr
#'  \link[paws.database:docdb_remove_from_global_cluster]{remove_from_global_cluster} \tab Detaches an Amazon DocumentDB secondary cluster from a global cluster\cr
#'  \link[paws.database:docdb_remove_source_identifier_from_subscription]{remove_source_identifier_from_subscription} \tab Removes a source identifier from an existing Amazon DocumentDB event notification subscription\cr
#'  \link[paws.database:docdb_remove_tags_from_resource]{remove_tags_from_resource} \tab Removes metadata tags from an Amazon DocumentDB resource\cr
#'  \link[paws.database:docdb_reset_db_cluster_parameter_group]{reset_db_cluster_parameter_group} \tab Modifies the parameters of a cluster parameter group to the default value\cr
#'  \link[paws.database:docdb_restore_db_cluster_from_snapshot]{restore_db_cluster_from_snapshot} \tab Creates a new cluster from a snapshot or cluster snapshot\cr
#'  \link[paws.database:docdb_restore_db_cluster_to_point_in_time]{restore_db_cluster_to_point_in_time} \tab Restores a cluster to an arbitrary point in time\cr
#'  \link[paws.database:docdb_start_db_cluster]{start_db_cluster} \tab Restarts the stopped cluster that is specified by DBClusterIdentifier\cr
#'  \link[paws.database:docdb_stop_db_cluster]{stop_db_cluster} \tab Stops the running cluster that is specified by DBClusterIdentifier
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname docdb
#' @export
docdb <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.database::docdb(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' Amazon DynamoDB
#'
#' @description
#' Amazon DynamoDB is a fully managed NoSQL database service that provides
#' fast and predictable performance with seamless scalability. DynamoDB
#' lets you offload the administrative burdens of operating and scaling a
#' distributed database, so that you don't have to worry about hardware
#' provisioning, setup and configuration, replication, software patching,
#' or cluster scaling.
#'
#' With DynamoDB, you can create database tables that can store and
#' retrieve any amount of data, and serve any level of request traffic. You
#' can scale up or scale down your tables' throughput capacity without
#' downtime or performance degradation, and use the Amazon Web Services
#' Management Console to monitor resource utilization and performance
#' metrics.
#'
#' DynamoDB automatically spreads the data and traffic for your tables over
#' a sufficient number of servers to handle your throughput and storage
#' requirements, while maintaining consistent and fast performance. All of
#' your data is stored on solid state disks (SSDs) and automatically
#' replicated across multiple Availability Zones in an Amazon Web Services
#' Region, providing built-in high availability and data durability.
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- dynamodb(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- dynamodb()
#' # This example reads multiple items from the Music table using a batch of
#' # three GetItem requests.  Only the AlbumTitle attribute is returned.
#' svc$batch_get_item(
#'   RequestItems = list(
#'     Music = list(
#'       Keys = list(
#'         list(
#'           Artist = list(
#'             S = "No One You Know"
#'           ),
#'           SongTitle = list(
#'             S = "Call Me Today"
#'           )
#'         ),
#'         list(
#'           Artist = list(
#'             S = "Acme Band"
#'           ),
#'           SongTitle = list(
#'             S = "Happy Day"
#'           )
#'         ),
#'         list(
#'           Artist = list(
#'             S = "No One You Know"
#'           ),
#'           SongTitle = list(
#'             S = "Scared of My Shadow"
#'           )
#'         )
#'       ),
#'       ProjectionExpression = "AlbumTitle"
#'     )
#'   )
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.database:dynamodb_batch_execute_statement]{batch_execute_statement} \tab This operation allows you to perform batch reads or writes on data stored in DynamoDB, using PartiQL\cr
#'  \link[paws.database:dynamodb_batch_get_item]{batch_get_item} \tab The BatchGetItem operation returns the attributes of one or more items from one or more tables\cr
#'  \link[paws.database:dynamodb_batch_write_item]{batch_write_item} \tab The BatchWriteItem operation puts or deletes multiple items in one or more tables\cr
#'  \link[paws.database:dynamodb_create_backup]{create_backup} \tab Creates a backup for an existing table\cr
#'  \link[paws.database:dynamodb_create_global_table]{create_global_table} \tab Creates a global table from an existing table\cr
#'  \link[paws.database:dynamodb_create_table]{create_table} \tab The CreateTable operation adds a new table to your account\cr
#'  \link[paws.database:dynamodb_delete_backup]{delete_backup} \tab Deletes an existing backup of a table\cr
#'  \link[paws.database:dynamodb_delete_item]{delete_item} \tab Deletes a single item in a table by primary key\cr
#'  \link[paws.database:dynamodb_delete_table]{delete_table} \tab The DeleteTable operation deletes a table and all of its items\cr
#'  \link[paws.database:dynamodb_describe_backup]{describe_backup} \tab Describes an existing backup of a table\cr
#'  \link[paws.database:dynamodb_describe_continuous_backups]{describe_continuous_backups} \tab Checks the status of continuous backups and point in time recovery on the specified table\cr
#'  \link[paws.database:dynamodb_describe_contributor_insights]{describe_contributor_insights} \tab Returns information about contributor insights for a given table or global secondary index\cr
#'  \link[paws.database:dynamodb_describe_endpoints]{describe_endpoints} \tab Returns the regional endpoint information\cr
#'  \link[paws.database:dynamodb_describe_export]{describe_export} \tab Describes an existing table export\cr
#'  \link[paws.database:dynamodb_describe_global_table]{describe_global_table} \tab Returns information about the specified global table\cr
#'  \link[paws.database:dynamodb_describe_global_table_settings]{describe_global_table_settings} \tab Describes Region-specific settings for a global table\cr
#'  \link[paws.database:dynamodb_describe_import]{describe_import} \tab Represents the properties of the import\cr
#'  \link[paws.database:dynamodb_describe_kinesis_streaming_destination]{describe_kinesis_streaming_destination} \tab Returns information about the status of Kinesis streaming\cr
#'  \link[paws.database:dynamodb_describe_limits]{describe_limits} \tab Returns the current provisioned-capacity quotas for your Amazon Web Services account in a Region, both for the Region as a whole and for any one DynamoDB table that you create there\cr
#'  \link[paws.database:dynamodb_describe_table]{describe_table} \tab Returns information about the table, including the current status of the table, when it was created, the primary key schema, and any indexes on the table\cr
#'  \link[paws.database:dynamodb_describe_table_replica_auto_scaling]{describe_table_replica_auto_scaling} \tab Describes auto scaling settings across replicas of the global table at once\cr
#'  \link[paws.database:dynamodb_describe_time_to_live]{describe_time_to_live} \tab Gives a description of the Time to Live (TTL) status on the specified table\cr
#'  \link[paws.database:dynamodb_disable_kinesis_streaming_destination]{disable_kinesis_streaming_destination} \tab Stops replication from the DynamoDB table to the Kinesis data stream\cr
#'  \link[paws.database:dynamodb_enable_kinesis_streaming_destination]{enable_kinesis_streaming_destination} \tab Starts table data replication to the specified Kinesis data stream at a timestamp chosen during the enable workflow\cr
#'  \link[paws.database:dynamodb_execute_statement]{execute_statement} \tab This operation allows you to perform reads and singleton writes on data stored in DynamoDB, using PartiQL\cr
#'  \link[paws.database:dynamodb_execute_transaction]{execute_transaction} \tab This operation allows you to perform transactional reads or writes on data stored in DynamoDB, using PartiQL\cr
#'  \link[paws.database:dynamodb_export_table_to_point_in_time]{export_table_to_point_in_time} \tab Exports table data to an S3 bucket\cr
#'  \link[paws.database:dynamodb_get_item]{get_item} \tab The GetItem operation returns a set of attributes for the item with the given primary key\cr
#'  \link[paws.database:dynamodb_import_table]{import_table} \tab Imports table data from an S3 bucket\cr
#'  \link[paws.database:dynamodb_list_backups]{list_backups} \tab List backups associated with an Amazon Web Services account\cr
#'  \link[paws.database:dynamodb_list_contributor_insights]{list_contributor_insights} \tab Returns a list of ContributorInsightsSummary for a table and all its global secondary indexes\cr
#'  \link[paws.database:dynamodb_list_exports]{list_exports} \tab Lists completed exports within the past 90 days\cr
#'  \link[paws.database:dynamodb_list_global_tables]{list_global_tables} \tab Lists all global tables that have a replica in the specified Region\cr
#'  \link[paws.database:dynamodb_list_imports]{list_imports} \tab Lists completed imports within the past 90 days\cr
#'  \link[paws.database:dynamodb_list_tables]{list_tables} \tab Returns an array of table names associated with the current account and endpoint\cr
#'  \link[paws.database:dynamodb_list_tags_of_resource]{list_tags_of_resource} \tab List all tags on an Amazon DynamoDB resource\cr
#'  \link[paws.database:dynamodb_put_item]{put_item} \tab Creates a new item, or replaces an old item with a new item\cr
#'  \link[paws.database:dynamodb_query]{query} \tab You must provide the name of the partition key attribute and a single value for that attribute\cr
#'  \link[paws.database:dynamodb_restore_table_from_backup]{restore_table_from_backup} \tab Creates a new table from an existing backup\cr
#'  \link[paws.database:dynamodb_restore_table_to_point_in_time]{restore_table_to_point_in_time} \tab Restores the specified table to the specified point in time within EarliestRestorableDateTime and LatestRestorableDateTime\cr
#'  \link[paws.database:dynamodb_scan]{scan} \tab The Scan operation returns one or more items and item attributes by accessing every item in a table or a secondary index\cr
#'  \link[paws.database:dynamodb_tag_resource]{tag_resource} \tab Associate a set of tags with an Amazon DynamoDB resource\cr
#'  \link[paws.database:dynamodb_transact_get_items]{transact_get_items} \tab TransactGetItems is a synchronous operation that atomically retrieves multiple items from one or more tables (but not from indexes) in a single account and Region\cr
#'  \link[paws.database:dynamodb_transact_write_items]{transact_write_items} \tab TransactWriteItems is a synchronous write operation that groups up to 100 action requests\cr
#'  \link[paws.database:dynamodb_untag_resource]{untag_resource} \tab Removes the association of tags from an Amazon DynamoDB resource\cr
#'  \link[paws.database:dynamodb_update_continuous_backups]{update_continuous_backups} \tab UpdateContinuousBackups enables or disables point in time recovery for the specified table\cr
#'  \link[paws.database:dynamodb_update_contributor_insights]{update_contributor_insights} \tab Updates the status for contributor insights for a specific table or index\cr
#'  \link[paws.database:dynamodb_update_global_table]{update_global_table} \tab Adds or removes replicas in the specified global table\cr
#'  \link[paws.database:dynamodb_update_global_table_settings]{update_global_table_settings} \tab Updates settings for a global table\cr
#'  \link[paws.database:dynamodb_update_item]{update_item} \tab Edits an existing item's attributes, or adds a new item to the table if it does not already exist\cr
#'  \link[paws.database:dynamodb_update_table]{update_table} \tab Modifies the provisioned throughput settings, global secondary indexes, or DynamoDB Streams settings for a given table\cr
#'  \link[paws.database:dynamodb_update_table_replica_auto_scaling]{update_table_replica_auto_scaling} \tab Updates auto scaling settings on your global tables at once\cr
#'  \link[paws.database:dynamodb_update_time_to_live]{update_time_to_live} \tab The UpdateTimeToLive method enables or disables Time to Live (TTL) for the specified table
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname dynamodb
#' @export
dynamodb <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.database::dynamodb(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' Amazon DynamoDB Streams
#'
#' @description
#' Amazon DynamoDB
#'
#' Amazon DynamoDB Streams provides API actions for accessing streams and
#' processing stream records. To learn more about application development
#' with Streams, see [Capturing Table Activity with DynamoDB
#' Streams](https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Streams.html)
#' in the Amazon DynamoDB Developer Guide.
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- dynamodbstreams(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- dynamodbstreams()
#' # The following example describes a stream with a given stream ARN.
#' svc$describe_stream(
#'   StreamArn = "arn:aws:dynamodb:us-west-2:111122223333:table/Forum/stream/2..."
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.database:dynamodbstreams_describe_stream]{describe_stream} \tab Returns information about a stream, including the current status of the stream, its Amazon Resource Name (ARN), the composition of its shards, and its corresponding DynamoDB table\cr
#'  \link[paws.database:dynamodbstreams_get_records]{get_records} \tab Retrieves the stream records from a given shard\cr
#'  \link[paws.database:dynamodbstreams_get_shard_iterator]{get_shard_iterator} \tab Returns a shard iterator\cr
#'  \link[paws.database:dynamodbstreams_list_streams]{list_streams} \tab Returns an array of stream ARNs associated with the current account and endpoint
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname dynamodbstreams
#' @export
dynamodbstreams <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.database::dynamodbstreams(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' Amazon ElastiCache
#'
#' @description
#' Amazon ElastiCache is a web service that makes it easier to set up,
#' operate, and scale a distributed cache in the cloud.
#'
#' With ElastiCache, customers get all of the benefits of a
#' high-performance, in-memory cache with less of the administrative burden
#' involved in launching and managing a distributed cache. The service
#' makes setup, scaling, and cluster failure handling much simpler than in
#' a self-managed cache deployment.
#'
#' In addition, through integration with Amazon CloudWatch, customers get
#' enhanced visibility into the key performance statistics associated with
#' their cache and can receive alarms if a part of their cache runs hot.
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- elasticache(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- elasticache()
#' svc$add_tags_to_resource(
#'   Foo = 123
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.database:elasticache_add_tags_to_resource]{add_tags_to_resource} \tab A tag is a key-value pair where the key and value are case-sensitive\cr
#'  \link[paws.database:elasticache_authorize_cache_security_group_ingress]{authorize_cache_security_group_ingress} \tab Allows network ingress to a cache security group\cr
#'  \link[paws.database:elasticache_batch_apply_update_action]{batch_apply_update_action} \tab Apply the service update\cr
#'  \link[paws.database:elasticache_batch_stop_update_action]{batch_stop_update_action} \tab Stop the service update\cr
#'  \link[paws.database:elasticache_complete_migration]{complete_migration} \tab Complete the migration of data\cr
#'  \link[paws.database:elasticache_copy_snapshot]{copy_snapshot} \tab Makes a copy of an existing snapshot\cr
#'  \link[paws.database:elasticache_create_cache_cluster]{create_cache_cluster} \tab Creates a cluster\cr
#'  \link[paws.database:elasticache_create_cache_parameter_group]{create_cache_parameter_group} \tab Creates a new Amazon ElastiCache cache parameter group\cr
#'  \link[paws.database:elasticache_create_cache_security_group]{create_cache_security_group} \tab Creates a new cache security group\cr
#'  \link[paws.database:elasticache_create_cache_subnet_group]{create_cache_subnet_group} \tab Creates a new cache subnet group\cr
#'  \link[paws.database:elasticache_create_global_replication_group]{create_global_replication_group} \tab Global Datastore for Redis offers fully managed, fast, reliable and secure cross-region replication\cr
#'  \link[paws.database:elasticache_create_replication_group]{create_replication_group} \tab Creates a Redis (cluster mode disabled) or a Redis (cluster mode enabled) replication group\cr
#'  \link[paws.database:elasticache_create_snapshot]{create_snapshot} \tab Creates a copy of an entire cluster or replication group at a specific moment in time\cr
#'  \link[paws.database:elasticache_create_user]{create_user} \tab For Redis engine version 6\cr
#'  \link[paws.database:elasticache_create_user_group]{create_user_group} \tab For Redis engine version 6\cr
#'  \link[paws.database:elasticache_decrease_node_groups_in_global_replication_group]{decrease_node_groups_in_global_replication_group} \tab Decreases the number of node groups in a Global datastore\cr
#'  \link[paws.database:elasticache_decrease_replica_count]{decrease_replica_count} \tab Dynamically decreases the number of replicas in a Redis (cluster mode disabled) replication group or the number of replica nodes in one or more node groups (shards) of a Redis (cluster mode enabled) replication group\cr
#'  \link[paws.database:elasticache_delete_cache_cluster]{delete_cache_cluster} \tab Deletes a previously provisioned cluster\cr
#'  \link[paws.database:elasticache_delete_cache_parameter_group]{delete_cache_parameter_group} \tab Deletes the specified cache parameter group\cr
#'  \link[paws.database:elasticache_delete_cache_security_group]{delete_cache_security_group} \tab Deletes a cache security group\cr
#'  \link[paws.database:elasticache_delete_cache_subnet_group]{delete_cache_subnet_group} \tab Deletes a cache subnet group\cr
#'  \link[paws.database:elasticache_delete_global_replication_group]{delete_global_replication_group} \tab Deleting a Global datastore is a two-step process:\cr
#'  \link[paws.database:elasticache_delete_replication_group]{delete_replication_group} \tab Deletes an existing replication group\cr
#'  \link[paws.database:elasticache_delete_snapshot]{delete_snapshot} \tab Deletes an existing snapshot\cr
#'  \link[paws.database:elasticache_delete_user]{delete_user} \tab For Redis engine version 6\cr
#'  \link[paws.database:elasticache_delete_user_group]{delete_user_group} \tab For Redis engine version 6\cr
#'  \link[paws.database:elasticache_describe_cache_clusters]{describe_cache_clusters} \tab Returns information about all provisioned clusters if no cluster identifier is specified, or about a specific cache cluster if a cluster identifier is supplied\cr
#'  \link[paws.database:elasticache_describe_cache_engine_versions]{describe_cache_engine_versions} \tab Returns a list of the available cache engines and their versions\cr
#'  \link[paws.database:elasticache_describe_cache_parameter_groups]{describe_cache_parameter_groups} \tab Returns a list of cache parameter group descriptions\cr
#'  \link[paws.database:elasticache_describe_cache_parameters]{describe_cache_parameters} \tab Returns the detailed parameter list for a particular cache parameter group\cr
#'  \link[paws.database:elasticache_describe_cache_security_groups]{describe_cache_security_groups} \tab Returns a list of cache security group descriptions\cr
#'  \link[paws.database:elasticache_describe_cache_subnet_groups]{describe_cache_subnet_groups} \tab Returns a list of cache subnet group descriptions\cr
#'  \link[paws.database:elasticache_describe_engine_default_parameters]{describe_engine_default_parameters} \tab Returns the default engine and system parameter information for the specified cache engine\cr
#'  \link[paws.database:elasticache_describe_events]{describe_events} \tab Returns events related to clusters, cache security groups, and cache parameter groups\cr
#'  \link[paws.database:elasticache_describe_global_replication_groups]{describe_global_replication_groups} \tab Returns information about a particular global replication group\cr
#'  \link[paws.database:elasticache_describe_replication_groups]{describe_replication_groups} \tab Returns information about a particular replication group\cr
#'  \link[paws.database:elasticache_describe_reserved_cache_nodes]{describe_reserved_cache_nodes} \tab Returns information about reserved cache nodes for this account, or about a specified reserved cache node\cr
#'  \link[paws.database:elasticache_describe_reserved_cache_nodes_offerings]{describe_reserved_cache_nodes_offerings} \tab Lists available reserved cache node offerings\cr
#'  \link[paws.database:elasticache_describe_service_updates]{describe_service_updates} \tab Returns details of the service updates\cr
#'  \link[paws.database:elasticache_describe_snapshots]{describe_snapshots} \tab Returns information about cluster or replication group snapshots\cr
#'  \link[paws.database:elasticache_describe_update_actions]{describe_update_actions} \tab Returns details of the update actions\cr
#'  \link[paws.database:elasticache_describe_user_groups]{describe_user_groups} \tab Returns a list of user groups\cr
#'  \link[paws.database:elasticache_describe_users]{describe_users} \tab Returns a list of users\cr
#'  \link[paws.database:elasticache_disassociate_global_replication_group]{disassociate_global_replication_group} \tab Remove a secondary cluster from the Global datastore using the Global datastore name\cr
#'  \link[paws.database:elasticache_failover_global_replication_group]{failover_global_replication_group} \tab Used to failover the primary region to a secondary region\cr
#'  \link[paws.database:elasticache_increase_node_groups_in_global_replication_group]{increase_node_groups_in_global_replication_group} \tab Increase the number of node groups in the Global datastore\cr
#'  \link[paws.database:elasticache_increase_replica_count]{increase_replica_count} \tab Dynamically increases the number of replicas in a Redis (cluster mode disabled) replication group or the number of replica nodes in one or more node groups (shards) of a Redis (cluster mode enabled) replication group\cr
#'  \link[paws.database:elasticache_list_allowed_node_type_modifications]{list_allowed_node_type_modifications} \tab Lists all available node types that you can scale your Redis cluster's or replication group's current node type\cr
#'  \link[paws.database:elasticache_list_tags_for_resource]{list_tags_for_resource} \tab Lists all tags currently on a named resource\cr
#'  \link[paws.database:elasticache_modify_cache_cluster]{modify_cache_cluster} \tab Modifies the settings for a cluster\cr
#'  \link[paws.database:elasticache_modify_cache_parameter_group]{modify_cache_parameter_group} \tab Modifies the parameters of a cache parameter group\cr
#'  \link[paws.database:elasticache_modify_cache_subnet_group]{modify_cache_subnet_group} \tab Modifies an existing cache subnet group\cr
#'  \link[paws.database:elasticache_modify_global_replication_group]{modify_global_replication_group} \tab Modifies the settings for a Global datastore\cr
#'  \link[paws.database:elasticache_modify_replication_group]{modify_replication_group} \tab Modifies the settings for a replication group\cr
#'  \link[paws.database:elasticache_modify_replication_group_shard_configuration]{modify_replication_group_shard_configuration} \tab Modifies a replication group's shards (node groups) by allowing you to add shards, remove shards, or rebalance the keyspaces among existing shards\cr
#'  \link[paws.database:elasticache_modify_user]{modify_user} \tab Changes user password(s) and/or access string\cr
#'  \link[paws.database:elasticache_modify_user_group]{modify_user_group} \tab Changes the list of users that belong to the user group\cr
#'  \link[paws.database:elasticache_purchase_reserved_cache_nodes_offering]{purchase_reserved_cache_nodes_offering} \tab Allows you to purchase a reserved cache node offering\cr
#'  \link[paws.database:elasticache_rebalance_slots_in_global_replication_group]{rebalance_slots_in_global_replication_group} \tab Redistribute slots to ensure uniform distribution across existing shards in the cluster\cr
#'  \link[paws.database:elasticache_reboot_cache_cluster]{reboot_cache_cluster} \tab Reboots some, or all, of the cache nodes within a provisioned cluster\cr
#'  \link[paws.database:elasticache_remove_tags_from_resource]{remove_tags_from_resource} \tab Removes the tags identified by the TagKeys list from the named resource\cr
#'  \link[paws.database:elasticache_reset_cache_parameter_group]{reset_cache_parameter_group} \tab Modifies the parameters of a cache parameter group to the engine or system default value\cr
#'  \link[paws.database:elasticache_revoke_cache_security_group_ingress]{revoke_cache_security_group_ingress} \tab Revokes ingress from a cache security group\cr
#'  \link[paws.database:elasticache_start_migration]{start_migration} \tab Start the migration of data\cr
#'  \link[paws.database:elasticache_test_failover]{test_failover} \tab Represents the input of a TestFailover operation which test automatic failover on a specified node group (called shard in the console) in a replication group (called cluster in the console)\cr
#'  \link[paws.database:elasticache_test_migration]{test_migration} \tab Async API to test connection between source and target replication group
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname elasticache
#' @export
elasticache <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.database::elasticache(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' Amazon Keyspaces
#'
#' @description
#' Amazon Keyspaces (for Apache Cassandra) is a scalable, highly available,
#' and managed Apache Cassandra-compatible database service. Amazon
#' Keyspaces makes it easy to migrate, run, and scale Cassandra workloads
#' in the Amazon Web Services Cloud. With just a few clicks on the Amazon
#' Web Services Management Console or a few lines of code, you can create
#' keyspaces and tables in Amazon Keyspaces, without deploying any
#' infrastructure or installing software.
#'
#' In addition to supporting Cassandra Query Language (CQL) requests via
#' open-source Cassandra drivers, Amazon Keyspaces supports data definition
#' language (DDL) operations to manage keyspaces and tables using the
#' Amazon Web Services SDK and CLI, as well as infrastructure as code (IaC)
#' services and tools such as CloudFormation and Terraform. This API
#' reference describes the supported DDL operations in detail.
#'
#' For the list of all supported CQL APIs, see [Supported Cassandra APIs,
#' operations, and data types in Amazon
#' Keyspaces](https://docs.aws.amazon.com/keyspaces/latest/devguide/cassandra-apis.html)
#' in the *Amazon Keyspaces Developer Guide*.
#'
#' To learn how Amazon Keyspaces API actions are recorded with CloudTrail,
#' see [Amazon Keyspaces information in
#' CloudTrail](https://docs.aws.amazon.com/keyspaces/latest/devguide/logging-using-cloudtrail.html#service-name-info-in-cloudtrail)
#' in the *Amazon Keyspaces Developer Guide*.
#'
#' For more information about Amazon Web Services APIs, for example how to
#' implement retry logic or how to sign Amazon Web Services API requests,
#' see [Amazon Web Services
#' APIs](https://docs.aws.amazon.com/general/latest/gr/) in the *General
#' Reference*.
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- keyspaces(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- keyspaces()
#' svc$create_keyspace(
#'   Foo = 123
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.database:keyspaces_create_keyspace]{create_keyspace} \tab The CreateKeyspace operation adds a new keyspace to your account\cr
#'  \link[paws.database:keyspaces_create_table]{create_table} \tab The CreateTable operation adds a new table to the specified keyspace\cr
#'  \link[paws.database:keyspaces_delete_keyspace]{delete_keyspace} \tab The DeleteKeyspace operation deletes a keyspace and all of its tables\cr
#'  \link[paws.database:keyspaces_delete_table]{delete_table} \tab The DeleteTable operation deletes a table and all of its data\cr
#'  \link[paws.database:keyspaces_get_keyspace]{get_keyspace} \tab Returns the name and the Amazon Resource Name (ARN) of the specified table\cr
#'  \link[paws.database:keyspaces_get_table]{get_table} \tab Returns information about the table, including the table's name and current status, the keyspace name, configuration settings, and metadata\cr
#'  \link[paws.database:keyspaces_list_keyspaces]{list_keyspaces} \tab Returns a list of keyspaces\cr
#'  \link[paws.database:keyspaces_list_tables]{list_tables} \tab Returns a list of tables for a specified keyspace\cr
#'  \link[paws.database:keyspaces_list_tags_for_resource]{list_tags_for_resource} \tab Returns a list of all tags associated with the specified Amazon Keyspaces resource\cr
#'  \link[paws.database:keyspaces_restore_table]{restore_table} \tab Restores the specified table to the specified point in time within the earliest_restorable_timestamp and the current time\cr
#'  \link[paws.database:keyspaces_tag_resource]{tag_resource} \tab Associates a set of tags with a Amazon Keyspaces resource\cr
#'  \link[paws.database:keyspaces_untag_resource]{untag_resource} \tab Removes the association of tags from a Amazon Keyspaces resource\cr
#'  \link[paws.database:keyspaces_update_table]{update_table} \tab Adds new columns to the table or updates one of the table's settings, for example capacity mode, encryption, point-in-time recovery, or ttl settings
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname keyspaces
#' @export
keyspaces <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.database::keyspaces(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' AWS Lake Formation
#'
#' @description
#' Lake Formation
#'
#' Defines the public endpoint for the Lake Formation service.
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- lakeformation(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- lakeformation()
#' svc$add_lf_tags_to_resource(
#'   Foo = 123
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.database:lakeformation_add_lf_tags_to_resource]{add_lf_tags_to_resource} \tab Attaches one or more LF-tags to an existing resource\cr
#'  \link[paws.database:lakeformation_assume_decorated_role_with_saml]{assume_decorated_role_with_saml} \tab Allows a caller to assume an IAM role decorated as the SAML user specified in the SAML assertion included in the request\cr
#'  \link[paws.database:lakeformation_batch_grant_permissions]{batch_grant_permissions} \tab Batch operation to grant permissions to the principal\cr
#'  \link[paws.database:lakeformation_batch_revoke_permissions]{batch_revoke_permissions} \tab Batch operation to revoke permissions from the principal\cr
#'  \link[paws.database:lakeformation_cancel_transaction]{cancel_transaction} \tab Attempts to cancel the specified transaction\cr
#'  \link[paws.database:lakeformation_commit_transaction]{commit_transaction} \tab Attempts to commit the specified transaction\cr
#'  \link[paws.database:lakeformation_create_data_cells_filter]{create_data_cells_filter} \tab Creates a data cell filter to allow one to grant access to certain columns on certain rows\cr
#'  \link[paws.database:lakeformation_create_lf_tag]{create_lf_tag} \tab Creates an LF-tag with the specified name and values\cr
#'  \link[paws.database:lakeformation_delete_data_cells_filter]{delete_data_cells_filter} \tab Deletes a data cell filter\cr
#'  \link[paws.database:lakeformation_delete_lf_tag]{delete_lf_tag} \tab Deletes the specified LF-tag given a key name\cr
#'  \link[paws.database:lakeformation_delete_objects_on_cancel]{delete_objects_on_cancel} \tab For a specific governed table, provides a list of Amazon S3 objects that will be written during the current transaction and that can be automatically deleted if the transaction is canceled\cr
#'  \link[paws.database:lakeformation_deregister_resource]{deregister_resource} \tab Deregisters the resource as managed by the Data Catalog\cr
#'  \link[paws.database:lakeformation_describe_resource]{describe_resource} \tab Retrieves the current data access role for the given resource registered in Lake Formation\cr
#'  \link[paws.database:lakeformation_describe_transaction]{describe_transaction} \tab Returns the details of a single transaction\cr
#'  \link[paws.database:lakeformation_extend_transaction]{extend_transaction} \tab Indicates to the service that the specified transaction is still active and should not be treated as idle and aborted\cr
#'  \link[paws.database:lakeformation_get_data_cells_filter]{get_data_cells_filter} \tab Returns a data cells filter\cr
#'  \link[paws.database:lakeformation_get_data_lake_settings]{get_data_lake_settings} \tab Retrieves the list of the data lake administrators of a Lake Formation-managed data lake\cr
#'  \link[paws.database:lakeformation_get_effective_permissions_for_path]{get_effective_permissions_for_path} \tab Returns the Lake Formation permissions for a specified table or database resource located at a path in Amazon S3\cr
#'  \link[paws.database:lakeformation_get_lf_tag]{get_lf_tag} \tab Returns an LF-tag definition\cr
#'  \link[paws.database:lakeformation_get_query_state]{get_query_state} \tab Returns the state of a query previously submitted\cr
#'  \link[paws.database:lakeformation_get_query_statistics]{get_query_statistics} \tab Retrieves statistics on the planning and execution of a query\cr
#'  \link[paws.database:lakeformation_get_resource_lf_tags]{get_resource_lf_tags} \tab Returns the LF-tags applied to a resource\cr
#'  \link[paws.database:lakeformation_get_table_objects]{get_table_objects} \tab Returns the set of Amazon S3 objects that make up the specified governed table\cr
#'  \link[paws.database:lakeformation_get_temporary_glue_partition_credentials]{get_temporary_glue_partition_credentials} \tab This API is identical to GetTemporaryTableCredentials except that this is used when the target Data Catalog resource is of type Partition\cr
#'  \link[paws.database:lakeformation_get_temporary_glue_table_credentials]{get_temporary_glue_table_credentials} \tab Allows a caller in a secure environment to assume a role with permission to access Amazon S3\cr
#'  \link[paws.database:lakeformation_get_work_unit_results]{get_work_unit_results} \tab Returns the work units resulting from the query\cr
#'  \link[paws.database:lakeformation_get_work_units]{get_work_units} \tab Retrieves the work units generated by the StartQueryPlanning operation\cr
#'  \link[paws.database:lakeformation_grant_permissions]{grant_permissions} \tab Grants permissions to the principal to access metadata in the Data Catalog and data organized in underlying data storage such as Amazon S3\cr
#'  \link[paws.database:lakeformation_list_data_cells_filter]{list_data_cells_filter} \tab Lists all the data cell filters on a table\cr
#'  \link[paws.database:lakeformation_list_lf_tags]{list_lf_tags} \tab Lists LF-tags that the requester has permission to view\cr
#'  \link[paws.database:lakeformation_list_permissions]{list_permissions} \tab Returns a list of the principal permissions on the resource, filtered by the permissions of the caller\cr
#'  \link[paws.database:lakeformation_list_resources]{list_resources} \tab Lists the resources registered to be managed by the Data Catalog\cr
#'  \link[paws.database:lakeformation_list_table_storage_optimizers]{list_table_storage_optimizers} \tab Returns the configuration of all storage optimizers associated with a specified table\cr
#'  \link[paws.database:lakeformation_list_transactions]{list_transactions} \tab Returns metadata about transactions and their status\cr
#'  \link[paws.database:lakeformation_put_data_lake_settings]{put_data_lake_settings} \tab Sets the list of data lake administrators who have admin privileges on all resources managed by Lake Formation\cr
#'  \link[paws.database:lakeformation_register_resource]{register_resource} \tab Registers the resource as managed by the Data Catalog\cr
#'  \link[paws.database:lakeformation_remove_lf_tags_from_resource]{remove_lf_tags_from_resource} \tab Removes an LF-tag from the resource\cr
#'  \link[paws.database:lakeformation_revoke_permissions]{revoke_permissions} \tab Revokes permissions to the principal to access metadata in the Data Catalog and data organized in underlying data storage such as Amazon S3\cr
#'  \link[paws.database:lakeformation_search_databases_by_lf_tags]{search_databases_by_lf_tags} \tab This operation allows a search on DATABASE resources by TagCondition\cr
#'  \link[paws.database:lakeformation_search_tables_by_lf_tags]{search_tables_by_lf_tags} \tab This operation allows a search on TABLE resources by LFTags\cr
#'  \link[paws.database:lakeformation_start_query_planning]{start_query_planning} \tab Submits a request to process a query statement\cr
#'  \link[paws.database:lakeformation_start_transaction]{start_transaction} \tab Starts a new transaction and returns its transaction ID\cr
#'  \link[paws.database:lakeformation_update_data_cells_filter]{update_data_cells_filter} \tab Updates a data cell filter\cr
#'  \link[paws.database:lakeformation_update_lf_tag]{update_lf_tag} \tab Updates the list of possible values for the specified LF-tag key\cr
#'  \link[paws.database:lakeformation_update_resource]{update_resource} \tab Updates the data access role used for vending access to the given (registered) resource in Lake Formation\cr
#'  \link[paws.database:lakeformation_update_table_objects]{update_table_objects} \tab Updates the manifest of Amazon S3 objects that make up the specified governed table\cr
#'  \link[paws.database:lakeformation_update_table_storage_optimizer]{update_table_storage_optimizer} \tab Updates the configuration of the storage optimizers for a table
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname lakeformation
#' @export
lakeformation <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.database::lakeformation(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' Amazon MemoryDB
#'
#' @description
#' MemoryDB for Redis is a fully managed, Redis-compatible, in-memory
#' database that delivers ultra-fast performance and Multi-AZ durability
#' for modern applications built using microservices architectures.
#' MemoryDB stores the entire database in-memory, enabling low latency and
#' high throughput data access. It is compatible with Redis, a popular open
#' source data store, enabling you to leverage Redis’ flexible and friendly
#' data structures, APIs, and commands.
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- memorydb(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- memorydb()
#' svc$batch_update_cluster(
#'   Foo = 123
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.database:memorydb_batch_update_cluster]{batch_update_cluster} \tab Apply the service update to a list of clusters supplied\cr
#'  \link[paws.database:memorydb_copy_snapshot]{copy_snapshot} \tab Makes a copy of an existing snapshot\cr
#'  \link[paws.database:memorydb_create_acl]{create_acl} \tab Creates an Access Control List\cr
#'  \link[paws.database:memorydb_create_cluster]{create_cluster} \tab Creates a cluster\cr
#'  \link[paws.database:memorydb_create_parameter_group]{create_parameter_group} \tab Creates a new MemoryDB parameter group\cr
#'  \link[paws.database:memorydb_create_snapshot]{create_snapshot} \tab Creates a copy of an entire cluster at a specific moment in time\cr
#'  \link[paws.database:memorydb_create_subnet_group]{create_subnet_group} \tab Creates a subnet group\cr
#'  \link[paws.database:memorydb_create_user]{create_user} \tab Creates a MemoryDB user\cr
#'  \link[paws.database:memorydb_delete_acl]{delete_acl} \tab Deletes an Access Control List\cr
#'  \link[paws.database:memorydb_delete_cluster]{delete_cluster} \tab Deletes a cluster\cr
#'  \link[paws.database:memorydb_delete_parameter_group]{delete_parameter_group} \tab Deletes the specified parameter group\cr
#'  \link[paws.database:memorydb_delete_snapshot]{delete_snapshot} \tab Deletes an existing snapshot\cr
#'  \link[paws.database:memorydb_delete_subnet_group]{delete_subnet_group} \tab Deletes a subnet group\cr
#'  \link[paws.database:memorydb_delete_user]{delete_user} \tab Deletes a user\cr
#'  \link[paws.database:memorydb_describe_ac_ls]{describe_ac_ls} \tab Returns a list of ACLs\cr
#'  \link[paws.database:memorydb_describe_clusters]{describe_clusters} \tab Returns information about all provisioned clusters if no cluster identifier is specified, or about a specific cluster if a cluster name is supplied\cr
#'  \link[paws.database:memorydb_describe_engine_versions]{describe_engine_versions} \tab Returns a list of the available Redis engine versions\cr
#'  \link[paws.database:memorydb_describe_events]{describe_events} \tab Returns events related to clusters, security groups, and parameter groups\cr
#'  \link[paws.database:memorydb_describe_parameter_groups]{describe_parameter_groups} \tab Returns a list of parameter group descriptions\cr
#'  \link[paws.database:memorydb_describe_parameters]{describe_parameters} \tab Returns the detailed parameter list for a particular parameter group\cr
#'  \link[paws.database:memorydb_describe_reserved_nodes]{describe_reserved_nodes} \tab Returns information about reserved nodes for this account, or about a specified reserved node\cr
#'  \link[paws.database:memorydb_describe_reserved_nodes_offerings]{describe_reserved_nodes_offerings} \tab Lists available reserved node offerings\cr
#'  \link[paws.database:memorydb_describe_service_updates]{describe_service_updates} \tab Returns details of the service updates\cr
#'  \link[paws.database:memorydb_describe_snapshots]{describe_snapshots} \tab Returns information about cluster snapshots\cr
#'  \link[paws.database:memorydb_describe_subnet_groups]{describe_subnet_groups} \tab Returns a list of subnet group descriptions\cr
#'  \link[paws.database:memorydb_describe_users]{describe_users} \tab Returns a list of users\cr
#'  \link[paws.database:memorydb_failover_shard]{failover_shard} \tab Used to failover a shard\cr
#'  \link[paws.database:memorydb_list_allowed_node_type_updates]{list_allowed_node_type_updates} \tab Lists all available node types that you can scale to from your cluster's current node type\cr
#'  \link[paws.database:memorydb_list_tags]{list_tags} \tab Lists all tags currently on a named resource\cr
#'  \link[paws.database:memorydb_purchase_reserved_nodes_offering]{purchase_reserved_nodes_offering} \tab Allows you to purchase a reserved node offering\cr
#'  \link[paws.database:memorydb_reset_parameter_group]{reset_parameter_group} \tab Modifies the parameters of a parameter group to the engine or system default value\cr
#'  \link[paws.database:memorydb_tag_resource]{tag_resource} \tab A tag is a key-value pair where the key and value are case-sensitive\cr
#'  \link[paws.database:memorydb_untag_resource]{untag_resource} \tab Use this operation to remove tags on a resource\cr
#'  \link[paws.database:memorydb_update_acl]{update_acl} \tab Changes the list of users that belong to the Access Control List\cr
#'  \link[paws.database:memorydb_update_cluster]{update_cluster} \tab Modifies the settings for a cluster\cr
#'  \link[paws.database:memorydb_update_parameter_group]{update_parameter_group} \tab Updates the parameters of a parameter group\cr
#'  \link[paws.database:memorydb_update_subnet_group]{update_subnet_group} \tab Updates a subnet group\cr
#'  \link[paws.database:memorydb_update_user]{update_user} \tab Changes user password(s) and/or access string
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname memorydb
#' @export
memorydb <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.database::memorydb(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' Amazon Neptune
#'
#' @description
#' Amazon Neptune is a fast, reliable, fully-managed graph database service
#' that makes it easy to build and run applications that work with highly
#' connected datasets. The core of Amazon Neptune is a purpose-built,
#' high-performance graph database engine optimized for storing billions of
#' relationships and querying the graph with milliseconds latency. Amazon
#' Neptune supports popular graph models Property Graph and W3C's RDF, and
#' their respective query languages Apache TinkerPop Gremlin and SPARQL,
#' allowing you to easily build queries that efficiently navigate highly
#' connected datasets. Neptune powers graph use cases such as
#' recommendation engines, fraud detection, knowledge graphs, drug
#' discovery, and network security.
#'
#' This interface reference for Amazon Neptune contains documentation for a
#' programming or command line interface you can use to manage Amazon
#' Neptune. Note that Amazon Neptune is asynchronous, which means that some
#' interfaces might require techniques such as polling or callback
#' functions to determine when a command has been applied. In this
#' reference, the parameter descriptions indicate whether a command is
#' applied immediately, on the next instance reboot, or during the
#' maintenance window. The reference structure is as follows, and we list
#' following some related topics from the user guide.
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- neptune(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- neptune()
#' svc$add_role_to_db_cluster(
#'   Foo = 123
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.database:neptune_add_role_to_db_cluster]{add_role_to_db_cluster} \tab Associates an Identity and Access Management (IAM) role with an Neptune DB cluster\cr
#'  \link[paws.database:neptune_add_source_identifier_to_subscription]{add_source_identifier_to_subscription} \tab Adds a source identifier to an existing event notification subscription\cr
#'  \link[paws.database:neptune_add_tags_to_resource]{add_tags_to_resource} \tab Adds metadata tags to an Amazon Neptune resource\cr
#'  \link[paws.database:neptune_apply_pending_maintenance_action]{apply_pending_maintenance_action} \tab Applies a pending maintenance action to a resource (for example, to a DB instance)\cr
#'  \link[paws.database:neptune_copy_db_cluster_parameter_group]{copy_db_cluster_parameter_group} \tab Copies the specified DB cluster parameter group\cr
#'  \link[paws.database:neptune_copy_db_cluster_snapshot]{copy_db_cluster_snapshot} \tab Copies a snapshot of a DB cluster\cr
#'  \link[paws.database:neptune_copy_db_parameter_group]{copy_db_parameter_group} \tab Copies the specified DB parameter group\cr
#'  \link[paws.database:neptune_create_db_cluster]{create_db_cluster} \tab Creates a new Amazon Neptune DB cluster\cr
#'  \link[paws.database:neptune_create_db_cluster_endpoint]{create_db_cluster_endpoint} \tab Creates a new custom endpoint and associates it with an Amazon Neptune DB cluster\cr
#'  \link[paws.database:neptune_create_db_cluster_parameter_group]{create_db_cluster_parameter_group} \tab Creates a new DB cluster parameter group\cr
#'  \link[paws.database:neptune_create_db_cluster_snapshot]{create_db_cluster_snapshot} \tab Creates a snapshot of a DB cluster\cr
#'  \link[paws.database:neptune_create_db_instance]{create_db_instance} \tab Creates a new DB instance\cr
#'  \link[paws.database:neptune_create_db_parameter_group]{create_db_parameter_group} \tab Creates a new DB parameter group\cr
#'  \link[paws.database:neptune_create_db_subnet_group]{create_db_subnet_group} \tab Creates a new DB subnet group\cr
#'  \link[paws.database:neptune_create_event_subscription]{create_event_subscription} \tab Creates an event notification subscription\cr
#'  \link[paws.database:neptune_create_global_cluster]{create_global_cluster} \tab Creates a Neptune global database spread across multiple Amazon Regions\cr
#'  \link[paws.database:neptune_delete_db_cluster]{delete_db_cluster} \tab The DeleteDBCluster action deletes a previously provisioned DB cluster\cr
#'  \link[paws.database:neptune_delete_db_cluster_endpoint]{delete_db_cluster_endpoint} \tab Deletes a custom endpoint and removes it from an Amazon Neptune DB cluster\cr
#'  \link[paws.database:neptune_delete_db_cluster_parameter_group]{delete_db_cluster_parameter_group} \tab Deletes a specified DB cluster parameter group\cr
#'  \link[paws.database:neptune_delete_db_cluster_snapshot]{delete_db_cluster_snapshot} \tab Deletes a DB cluster snapshot\cr
#'  \link[paws.database:neptune_delete_db_instance]{delete_db_instance} \tab The DeleteDBInstance action deletes a previously provisioned DB instance\cr
#'  \link[paws.database:neptune_delete_db_parameter_group]{delete_db_parameter_group} \tab Deletes a specified DBParameterGroup\cr
#'  \link[paws.database:neptune_delete_db_subnet_group]{delete_db_subnet_group} \tab Deletes a DB subnet group\cr
#'  \link[paws.database:neptune_delete_event_subscription]{delete_event_subscription} \tab Deletes an event notification subscription\cr
#'  \link[paws.database:neptune_delete_global_cluster]{delete_global_cluster} \tab Deletes a global database\cr
#'  \link[paws.database:neptune_describe_db_cluster_endpoints]{describe_db_cluster_endpoints} \tab Returns information about endpoints for an Amazon Neptune DB cluster\cr
#'  \link[paws.database:neptune_describe_db_cluster_parameter_groups]{describe_db_cluster_parameter_groups} \tab Returns a list of DBClusterParameterGroup descriptions\cr
#'  \link[paws.database:neptune_describe_db_cluster_parameters]{describe_db_cluster_parameters} \tab Returns the detailed parameter list for a particular DB cluster parameter group\cr
#'  \link[paws.database:neptune_describe_db_clusters]{describe_db_clusters} \tab Returns information about provisioned DB clusters, and supports pagination\cr
#'  \link[paws.database:neptune_describe_db_cluster_snapshot_attributes]{describe_db_cluster_snapshot_attributes} \tab Returns a list of DB cluster snapshot attribute names and values for a manual DB cluster snapshot\cr
#'  \link[paws.database:neptune_describe_db_cluster_snapshots]{describe_db_cluster_snapshots} \tab Returns information about DB cluster snapshots\cr
#'  \link[paws.database:neptune_describe_db_engine_versions]{describe_db_engine_versions} \tab Returns a list of the available DB engines\cr
#'  \link[paws.database:neptune_describe_db_instances]{describe_db_instances} \tab Returns information about provisioned instances, and supports pagination\cr
#'  \link[paws.database:neptune_describe_db_parameter_groups]{describe_db_parameter_groups} \tab Returns a list of DBParameterGroup descriptions\cr
#'  \link[paws.database:neptune_describe_db_parameters]{describe_db_parameters} \tab Returns the detailed parameter list for a particular DB parameter group\cr
#'  \link[paws.database:neptune_describe_db_subnet_groups]{describe_db_subnet_groups} \tab Returns a list of DBSubnetGroup descriptions\cr
#'  \link[paws.database:neptune_describe_engine_default_cluster_parameters]{describe_engine_default_cluster_parameters} \tab Returns the default engine and system parameter information for the cluster database engine\cr
#'  \link[paws.database:neptune_describe_engine_default_parameters]{describe_engine_default_parameters} \tab Returns the default engine and system parameter information for the specified database engine\cr
#'  \link[paws.database:neptune_describe_event_categories]{describe_event_categories} \tab Displays a list of categories for all event source types, or, if specified, for a specified source type\cr
#'  \link[paws.database:neptune_describe_events]{describe_events} \tab Returns events related to DB instances, DB security groups, DB snapshots, and DB parameter groups for the past 14 days\cr
#'  \link[paws.database:neptune_describe_event_subscriptions]{describe_event_subscriptions} \tab Lists all the subscription descriptions for a customer account\cr
#'  \link[paws.database:neptune_describe_global_clusters]{describe_global_clusters} \tab Returns information about Neptune global database clusters\cr
#'  \link[paws.database:neptune_describe_orderable_db_instance_options]{describe_orderable_db_instance_options} \tab Returns a list of orderable DB instance options for the specified engine\cr
#'  \link[paws.database:neptune_describe_pending_maintenance_actions]{describe_pending_maintenance_actions} \tab Returns a list of resources (for example, DB instances) that have at least one pending maintenance action\cr
#'  \link[paws.database:neptune_describe_valid_db_instance_modifications]{describe_valid_db_instance_modifications} \tab You can call DescribeValidDBInstanceModifications to learn what modifications you can make to your DB instance\cr
#'  \link[paws.database:neptune_failover_db_cluster]{failover_db_cluster} \tab Forces a failover for a DB cluster\cr
#'  \link[paws.database:neptune_failover_global_cluster]{failover_global_cluster} \tab Initiates the failover process for a Neptune global database\cr
#'  \link[paws.database:neptune_list_tags_for_resource]{list_tags_for_resource} \tab Lists all tags on an Amazon Neptune resource\cr
#'  \link[paws.database:neptune_modify_db_cluster]{modify_db_cluster} \tab Modify a setting for a DB cluster\cr
#'  \link[paws.database:neptune_modify_db_cluster_endpoint]{modify_db_cluster_endpoint} \tab Modifies the properties of an endpoint in an Amazon Neptune DB cluster\cr
#'  \link[paws.database:neptune_modify_db_cluster_parameter_group]{modify_db_cluster_parameter_group} \tab Modifies the parameters of a DB cluster parameter group\cr
#'  \link[paws.database:neptune_modify_db_cluster_snapshot_attribute]{modify_db_cluster_snapshot_attribute} \tab Adds an attribute and values to, or removes an attribute and values from, a manual DB cluster snapshot\cr
#'  \link[paws.database:neptune_modify_db_instance]{modify_db_instance} \tab Modifies settings for a DB instance\cr
#'  \link[paws.database:neptune_modify_db_parameter_group]{modify_db_parameter_group} \tab Modifies the parameters of a DB parameter group\cr
#'  \link[paws.database:neptune_modify_db_subnet_group]{modify_db_subnet_group} \tab Modifies an existing DB subnet group\cr
#'  \link[paws.database:neptune_modify_event_subscription]{modify_event_subscription} \tab Modifies an existing event notification subscription\cr
#'  \link[paws.database:neptune_modify_global_cluster]{modify_global_cluster} \tab Modify a setting for an Amazon Neptune global cluster\cr
#'  \link[paws.database:neptune_promote_read_replica_db_cluster]{promote_read_replica_db_cluster} \tab Not supported\cr
#'  \link[paws.database:neptune_reboot_db_instance]{reboot_db_instance} \tab You might need to reboot your DB instance, usually for maintenance reasons\cr
#'  \link[paws.database:neptune_remove_from_global_cluster]{remove_from_global_cluster} \tab Detaches a Neptune DB cluster from a Neptune global database\cr
#'  \link[paws.database:neptune_remove_role_from_db_cluster]{remove_role_from_db_cluster} \tab Disassociates an Identity and Access Management (IAM) role from a DB cluster\cr
#'  \link[paws.database:neptune_remove_source_identifier_from_subscription]{remove_source_identifier_from_subscription} \tab Removes a source identifier from an existing event notification subscription\cr
#'  \link[paws.database:neptune_remove_tags_from_resource]{remove_tags_from_resource} \tab Removes metadata tags from an Amazon Neptune resource\cr
#'  \link[paws.database:neptune_reset_db_cluster_parameter_group]{reset_db_cluster_parameter_group} \tab Modifies the parameters of a DB cluster parameter group to the default value\cr
#'  \link[paws.database:neptune_reset_db_parameter_group]{reset_db_parameter_group} \tab Modifies the parameters of a DB parameter group to the engine/system default value\cr
#'  \link[paws.database:neptune_restore_db_cluster_from_snapshot]{restore_db_cluster_from_snapshot} \tab Creates a new DB cluster from a DB snapshot or DB cluster snapshot\cr
#'  \link[paws.database:neptune_restore_db_cluster_to_point_in_time]{restore_db_cluster_to_point_in_time} \tab Restores a DB cluster to an arbitrary point in time\cr
#'  \link[paws.database:neptune_start_db_cluster]{start_db_cluster} \tab Starts an Amazon Neptune DB cluster that was stopped using the Amazon console, the Amazon CLI stop-db-cluster command, or the StopDBCluster API\cr
#'  \link[paws.database:neptune_stop_db_cluster]{stop_db_cluster} \tab Stops an Amazon Neptune DB cluster
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname neptune
#' @export
neptune <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.database::neptune(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' Amazon QLDB
#'
#' @description
#' The resource management API for Amazon QLDB
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- qldb(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- qldb()
#' svc$cancel_journal_kinesis_stream(
#'   Foo = 123
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.database:qldb_cancel_journal_kinesis_stream]{cancel_journal_kinesis_stream} \tab Ends a given Amazon QLDB journal stream\cr
#'  \link[paws.database:qldb_create_ledger]{create_ledger} \tab Creates a new ledger in your Amazon Web Services account in the current Region\cr
#'  \link[paws.database:qldb_delete_ledger]{delete_ledger} \tab Deletes a ledger and all of its contents\cr
#'  \link[paws.database:qldb_describe_journal_kinesis_stream]{describe_journal_kinesis_stream} \tab Returns detailed information about a given Amazon QLDB journal stream\cr
#'  \link[paws.database:qldb_describe_journal_s3_export]{describe_journal_s3_export} \tab Returns information about a journal export job, including the ledger name, export ID, creation time, current status, and the parameters of the original export creation request\cr
#'  \link[paws.database:qldb_describe_ledger]{describe_ledger} \tab Returns information about a ledger, including its state, permissions mode, encryption at rest settings, and when it was created\cr
#'  \link[paws.database:qldb_export_journal_to_s3]{export_journal_to_s3} \tab Exports journal contents within a date and time range from a ledger into a specified Amazon Simple Storage Service (Amazon S3) bucket\cr
#'  \link[paws.database:qldb_get_block]{get_block} \tab Returns a block object at a specified address in a journal\cr
#'  \link[paws.database:qldb_get_digest]{get_digest} \tab Returns the digest of a ledger at the latest committed block in the journal\cr
#'  \link[paws.database:qldb_get_revision]{get_revision} \tab Returns a revision data object for a specified document ID and block address\cr
#'  \link[paws.database:qldb_list_journal_kinesis_streams_for_ledger]{list_journal_kinesis_streams_for_ledger} \tab Returns all Amazon QLDB journal streams for a given ledger\cr
#'  \link[paws.database:qldb_list_journal_s3_exports]{list_journal_s3_exports} \tab Returns all journal export jobs for all ledgers that are associated with the current Amazon Web Services account and Region\cr
#'  \link[paws.database:qldb_list_journal_s3_exports_for_ledger]{list_journal_s3_exports_for_ledger} \tab Returns all journal export jobs for a specified ledger\cr
#'  \link[paws.database:qldb_list_ledgers]{list_ledgers} \tab Returns all ledgers that are associated with the current Amazon Web Services account and Region\cr
#'  \link[paws.database:qldb_list_tags_for_resource]{list_tags_for_resource} \tab Returns all tags for a specified Amazon QLDB resource\cr
#'  \link[paws.database:qldb_stream_journal_to_kinesis]{stream_journal_to_kinesis} \tab Creates a journal stream for a given Amazon QLDB ledger\cr
#'  \link[paws.database:qldb_tag_resource]{tag_resource} \tab Adds one or more tags to a specified Amazon QLDB resource\cr
#'  \link[paws.database:qldb_untag_resource]{untag_resource} \tab Removes one or more tags from a specified Amazon QLDB resource\cr
#'  \link[paws.database:qldb_update_ledger]{update_ledger} \tab Updates properties on a ledger\cr
#'  \link[paws.database:qldb_update_ledger_permissions_mode]{update_ledger_permissions_mode} \tab Updates the permissions mode of a ledger
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname qldb
#' @export
qldb <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.database::qldb(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' Amazon QLDB Session
#'
#' @description
#' The transactional data APIs for Amazon QLDB
#'
#' Instead of interacting directly with this API, we recommend using the
#' QLDB driver or the QLDB shell to execute data transactions on a ledger.
#'
#' -   If you are working with an AWS SDK, use the QLDB driver. The driver
#'     provides a high-level abstraction layer above this *QLDB Session*
#'     data plane and manages `send_command`
#'     API calls for you. For information and a list of supported
#'     programming languages, see [Getting started with the
#'     driver](https://docs.aws.amazon.com/qldb/latest/developerguide/getting-started-driver.html)
#'     in the *Amazon QLDB Developer Guide*.
#'
#' -   If you are working with the AWS Command Line Interface (AWS CLI),
#'     use the QLDB shell. The shell is a command line interface that uses
#'     the QLDB driver to interact with a ledger. For information, see
#'     [Accessing Amazon QLDB using the QLDB
#'     shell](https://docs.aws.amazon.com/qldb/latest/developerguide/data-shell.html).
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- qldbsession(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- qldbsession()
#' svc$send_command(
#'   Foo = 123
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.database:qldbsession_send_command]{send_command} \tab Sends a command to an Amazon QLDB ledger
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname qldbsession
#' @export
qldbsession <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.database::qldbsession(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' Amazon Relational Database Service
#'
#' @description
#' Amazon Relational Database Service (Amazon RDS) is a web service that
#' makes it easier to set up, operate, and scale a relational database in
#' the cloud. It provides cost-efficient, resizeable capacity for an
#' industry-standard relational database and manages common database
#' administration tasks, freeing up developers to focus on what makes their
#' applications and businesses unique.
#'
#' Amazon RDS gives you access to the capabilities of a MySQL, MariaDB,
#' PostgreSQL, Microsoft SQL Server, Oracle, or Amazon Aurora database
#' server. These capabilities mean that the code, applications, and tools
#' you already use today with your existing databases work with Amazon RDS
#' without modification. Amazon RDS automatically backs up your database
#' and maintains the database software that powers your DB instance. Amazon
#' RDS is flexible: you can scale your DB instance's compute resources and
#' storage capacity to meet your application's demand. As with all Amazon
#' Web Services, there are no up-front investments, and you pay only for
#' the resources you use.
#'
#' This interface reference for Amazon RDS contains documentation for a
#' programming or command line interface you can use to manage Amazon RDS.
#' Amazon RDS is asynchronous, which means that some interfaces might
#' require techniques such as polling or callback functions to determine
#' when a command has been applied. In this reference, the parameter
#' descriptions indicate whether a command is applied immediately, on the
#' next instance reboot, or during the maintenance window. The reference
#' structure is as follows, and we list following some related topics from
#' the user guide.
#'
#' **Amazon RDS API Reference**
#'
#' -   For the alphabetical list of API actions, see [API
#'     Actions](https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_Operations.html).
#'
#' -   For the alphabetical list of data types, see [Data
#'     Types](https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_Types.html).
#'
#' -   For a list of common query parameters, see [Common
#'     Parameters](https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/CommonParameters.html).
#'
#' -   For descriptions of the error codes, see [Common
#'     Errors](https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/CommonErrors.html).
#'
#' **Amazon RDS User Guide**
#'
#' -   For a summary of the Amazon RDS interfaces, see [Available RDS
#'     Interfaces](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Welcome.html#Welcome.Interfaces).
#'
#' -   For more information about how to use the Query API, see [Using the
#'     Query
#'     API](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Using_the_Query_API.html).
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- rds(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- rds()
#' svc$add_role_to_db_cluster(
#'   Foo = 123
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.database:rds_add_role_to_db_cluster]{add_role_to_db_cluster} \tab Associates an Identity and Access Management (IAM) role with a DB cluster\cr
#'  \link[paws.database:rds_add_role_to_db_instance]{add_role_to_db_instance} \tab Associates an Amazon Web Services Identity and Access Management (IAM) role with a DB instance\cr
#'  \link[paws.database:rds_add_source_identifier_to_subscription]{add_source_identifier_to_subscription} \tab Adds a source identifier to an existing RDS event notification subscription\cr
#'  \link[paws.database:rds_add_tags_to_resource]{add_tags_to_resource} \tab Adds metadata tags to an Amazon RDS resource\cr
#'  \link[paws.database:rds_apply_pending_maintenance_action]{apply_pending_maintenance_action} \tab Applies a pending maintenance action to a resource (for example, to a DB instance)\cr
#'  \link[paws.database:rds_authorize_db_security_group_ingress]{authorize_db_security_group_ingress} \tab Enables ingress to a DBSecurityGroup using one of two forms of authorization\cr
#'  \link[paws.database:rds_backtrack_db_cluster]{backtrack_db_cluster} \tab Backtracks a DB cluster to a specific time, without creating a new DB cluster\cr
#'  \link[paws.database:rds_build_auth_token]{build_auth_token} \tab Return an authentication token for a database connection\cr
#'  \link[paws.database:rds_cancel_export_task]{cancel_export_task} \tab Cancels an export task in progress that is exporting a snapshot or cluster to Amazon S3\cr
#'  \link[paws.database:rds_copy_db_cluster_parameter_group]{copy_db_cluster_parameter_group} \tab Copies the specified DB cluster parameter group\cr
#'  \link[paws.database:rds_copy_db_cluster_snapshot]{copy_db_cluster_snapshot} \tab Copies a snapshot of a DB cluster\cr
#'  \link[paws.database:rds_copy_db_parameter_group]{copy_db_parameter_group} \tab Copies the specified DB parameter group\cr
#'  \link[paws.database:rds_copy_db_snapshot]{copy_db_snapshot} \tab Copies the specified DB snapshot\cr
#'  \link[paws.database:rds_copy_option_group]{copy_option_group} \tab Copies the specified option group\cr
#'  \link[paws.database:rds_create_blue_green_deployment]{create_blue_green_deployment} \tab Creates a blue/green deployment\cr
#'  \link[paws.database:rds_create_custom_db_engine_version]{create_custom_db_engine_version} \tab Creates a custom DB engine version (CEV)\cr
#'  \link[paws.database:rds_create_db_cluster]{create_db_cluster} \tab Creates a new Amazon Aurora DB cluster or Multi-AZ DB cluster\cr
#'  \link[paws.database:rds_create_db_cluster_endpoint]{create_db_cluster_endpoint} \tab Creates a new custom endpoint and associates it with an Amazon Aurora DB cluster\cr
#'  \link[paws.database:rds_create_db_cluster_parameter_group]{create_db_cluster_parameter_group} \tab Creates a new DB cluster parameter group\cr
#'  \link[paws.database:rds_create_db_cluster_snapshot]{create_db_cluster_snapshot} \tab Creates a snapshot of a DB cluster\cr
#'  \link[paws.database:rds_create_db_instance]{create_db_instance} \tab Creates a new DB instance\cr
#'  \link[paws.database:rds_create_db_instance_read_replica]{create_db_instance_read_replica} \tab Creates a new DB instance that acts as a read replica for an existing source DB instance or Multi-AZ DB cluster\cr
#'  \link[paws.database:rds_create_db_parameter_group]{create_db_parameter_group} \tab Creates a new DB parameter group\cr
#'  \link[paws.database:rds_create_db_proxy]{create_db_proxy} \tab Creates a new DB proxy\cr
#'  \link[paws.database:rds_create_db_proxy_endpoint]{create_db_proxy_endpoint} \tab Creates a DBProxyEndpoint\cr
#'  \link[paws.database:rds_create_db_security_group]{create_db_security_group} \tab Creates a new DB security group\cr
#'  \link[paws.database:rds_create_db_snapshot]{create_db_snapshot} \tab Creates a snapshot of a DB instance\cr
#'  \link[paws.database:rds_create_db_subnet_group]{create_db_subnet_group} \tab Creates a new DB subnet group\cr
#'  \link[paws.database:rds_create_event_subscription]{create_event_subscription} \tab Creates an RDS event notification subscription\cr
#'  \link[paws.database:rds_create_global_cluster]{create_global_cluster} \tab Creates an Aurora global database spread across multiple Amazon Web Services Regions\cr
#'  \link[paws.database:rds_create_option_group]{create_option_group} \tab Creates a new option group\cr
#'  \link[paws.database:rds_delete_blue_green_deployment]{delete_blue_green_deployment} \tab Deletes a blue/green deployment\cr
#'  \link[paws.database:rds_delete_custom_db_engine_version]{delete_custom_db_engine_version} \tab Deletes a custom engine version\cr
#'  \link[paws.database:rds_delete_db_cluster]{delete_db_cluster} \tab The DeleteDBCluster action deletes a previously provisioned DB cluster\cr
#'  \link[paws.database:rds_delete_db_cluster_automated_backup]{delete_db_cluster_automated_backup} \tab Deletes automated backups using the DbClusterResourceId value of the source DB cluster or the Amazon Resource Name (ARN) of the automated backups\cr
#'  \link[paws.database:rds_delete_db_cluster_endpoint]{delete_db_cluster_endpoint} \tab Deletes a custom endpoint and removes it from an Amazon Aurora DB cluster\cr
#'  \link[paws.database:rds_delete_db_cluster_parameter_group]{delete_db_cluster_parameter_group} \tab Deletes a specified DB cluster parameter group\cr
#'  \link[paws.database:rds_delete_db_cluster_snapshot]{delete_db_cluster_snapshot} \tab Deletes a DB cluster snapshot\cr
#'  \link[paws.database:rds_delete_db_instance]{delete_db_instance} \tab The DeleteDBInstance action deletes a previously provisioned DB instance\cr
#'  \link[paws.database:rds_delete_db_instance_automated_backup]{delete_db_instance_automated_backup} \tab Deletes automated backups using the DbiResourceId value of the source DB instance or the Amazon Resource Name (ARN) of the automated backups\cr
#'  \link[paws.database:rds_delete_db_parameter_group]{delete_db_parameter_group} \tab Deletes a specified DB parameter group\cr
#'  \link[paws.database:rds_delete_db_proxy]{delete_db_proxy} \tab Deletes an existing DB proxy\cr
#'  \link[paws.database:rds_delete_db_proxy_endpoint]{delete_db_proxy_endpoint} \tab Deletes a DBProxyEndpoint\cr
#'  \link[paws.database:rds_delete_db_security_group]{delete_db_security_group} \tab Deletes a DB security group\cr
#'  \link[paws.database:rds_delete_db_snapshot]{delete_db_snapshot} \tab Deletes a DB snapshot\cr
#'  \link[paws.database:rds_delete_db_subnet_group]{delete_db_subnet_group} \tab Deletes a DB subnet group\cr
#'  \link[paws.database:rds_delete_event_subscription]{delete_event_subscription} \tab Deletes an RDS event notification subscription\cr
#'  \link[paws.database:rds_delete_global_cluster]{delete_global_cluster} \tab Deletes a global database cluster\cr
#'  \link[paws.database:rds_delete_option_group]{delete_option_group} \tab Deletes an existing option group\cr
#'  \link[paws.database:rds_deregister_db_proxy_targets]{deregister_db_proxy_targets} \tab Remove the association between one or more DBProxyTarget data structures and a DBProxyTargetGroup\cr
#'  \link[paws.database:rds_describe_account_attributes]{describe_account_attributes} \tab Lists all of the attributes for a customer account\cr
#'  \link[paws.database:rds_describe_blue_green_deployments]{describe_blue_green_deployments} \tab Describes one or more blue/green deployments\cr
#'  \link[paws.database:rds_describe_certificates]{describe_certificates} \tab Lists the set of CA certificates provided by Amazon RDS for this Amazon Web Services account\cr
#'  \link[paws.database:rds_describe_db_cluster_automated_backups]{describe_db_cluster_automated_backups} \tab Displays backups for both current and deleted DB clusters\cr
#'  \link[paws.database:rds_describe_db_cluster_backtracks]{describe_db_cluster_backtracks} \tab Returns information about backtracks for a DB cluster\cr
#'  \link[paws.database:rds_describe_db_cluster_endpoints]{describe_db_cluster_endpoints} \tab Returns information about endpoints for an Amazon Aurora DB cluster\cr
#'  \link[paws.database:rds_describe_db_cluster_parameter_groups]{describe_db_cluster_parameter_groups} \tab Returns a list of DBClusterParameterGroup descriptions\cr
#'  \link[paws.database:rds_describe_db_cluster_parameters]{describe_db_cluster_parameters} \tab Returns the detailed parameter list for a particular DB cluster parameter group\cr
#'  \link[paws.database:rds_describe_db_clusters]{describe_db_clusters} \tab Describes existing Amazon Aurora DB clusters and Multi-AZ DB clusters\cr
#'  \link[paws.database:rds_describe_db_cluster_snapshot_attributes]{describe_db_cluster_snapshot_attributes} \tab Returns a list of DB cluster snapshot attribute names and values for a manual DB cluster snapshot\cr
#'  \link[paws.database:rds_describe_db_cluster_snapshots]{describe_db_cluster_snapshots} \tab Returns information about DB cluster snapshots\cr
#'  \link[paws.database:rds_describe_db_engine_versions]{describe_db_engine_versions} \tab Returns a list of the available DB engines\cr
#'  \link[paws.database:rds_describe_db_instance_automated_backups]{describe_db_instance_automated_backups} \tab Displays backups for both current and deleted instances\cr
#'  \link[paws.database:rds_describe_db_instances]{describe_db_instances} \tab Describes provisioned RDS instances\cr
#'  \link[paws.database:rds_describe_db_log_files]{describe_db_log_files} \tab Returns a list of DB log files for the DB instance\cr
#'  \link[paws.database:rds_describe_db_parameter_groups]{describe_db_parameter_groups} \tab Returns a list of DBParameterGroup descriptions\cr
#'  \link[paws.database:rds_describe_db_parameters]{describe_db_parameters} \tab Returns the detailed parameter list for a particular DB parameter group\cr
#'  \link[paws.database:rds_describe_db_proxies]{describe_db_proxies} \tab Returns information about DB proxies\cr
#'  \link[paws.database:rds_describe_db_proxy_endpoints]{describe_db_proxy_endpoints} \tab Returns information about DB proxy endpoints\cr
#'  \link[paws.database:rds_describe_db_proxy_target_groups]{describe_db_proxy_target_groups} \tab Returns information about DB proxy target groups, represented by DBProxyTargetGroup data structures\cr
#'  \link[paws.database:rds_describe_db_proxy_targets]{describe_db_proxy_targets} \tab Returns information about DBProxyTarget objects\cr
#'  \link[paws.database:rds_describe_db_security_groups]{describe_db_security_groups} \tab Returns a list of DBSecurityGroup descriptions\cr
#'  \link[paws.database:rds_describe_db_snapshot_attributes]{describe_db_snapshot_attributes} \tab Returns a list of DB snapshot attribute names and values for a manual DB snapshot\cr
#'  \link[paws.database:rds_describe_db_snapshots]{describe_db_snapshots} \tab Returns information about DB snapshots\cr
#'  \link[paws.database:rds_describe_db_subnet_groups]{describe_db_subnet_groups} \tab Returns a list of DBSubnetGroup descriptions\cr
#'  \link[paws.database:rds_describe_engine_default_cluster_parameters]{describe_engine_default_cluster_parameters} \tab Returns the default engine and system parameter information for the cluster database engine\cr
#'  \link[paws.database:rds_describe_engine_default_parameters]{describe_engine_default_parameters} \tab Returns the default engine and system parameter information for the specified database engine\cr
#'  \link[paws.database:rds_describe_event_categories]{describe_event_categories} \tab Displays a list of categories for all event source types, or, if specified, for a specified source type\cr
#'  \link[paws.database:rds_describe_events]{describe_events} \tab Returns events related to DB instances, DB clusters, DB parameter groups, DB security groups, DB snapshots, DB cluster snapshots, and RDS Proxies for the past 14 days\cr
#'  \link[paws.database:rds_describe_event_subscriptions]{describe_event_subscriptions} \tab Lists all the subscription descriptions for a customer account\cr
#'  \link[paws.database:rds_describe_export_tasks]{describe_export_tasks} \tab Returns information about a snapshot or cluster export to Amazon S3\cr
#'  \link[paws.database:rds_describe_global_clusters]{describe_global_clusters} \tab Returns information about Aurora global database clusters\cr
#'  \link[paws.database:rds_describe_option_group_options]{describe_option_group_options} \tab Describes all available options\cr
#'  \link[paws.database:rds_describe_option_groups]{describe_option_groups} \tab Describes the available option groups\cr
#'  \link[paws.database:rds_describe_orderable_db_instance_options]{describe_orderable_db_instance_options} \tab Returns a list of orderable DB instance options for the specified DB engine, DB engine version, and DB instance class\cr
#'  \link[paws.database:rds_describe_pending_maintenance_actions]{describe_pending_maintenance_actions} \tab Returns a list of resources (for example, DB instances) that have at least one pending maintenance action\cr
#'  \link[paws.database:rds_describe_reserved_db_instances]{describe_reserved_db_instances} \tab Returns information about reserved DB instances for this account, or about a specified reserved DB instance\cr
#'  \link[paws.database:rds_describe_reserved_db_instances_offerings]{describe_reserved_db_instances_offerings} \tab Lists available reserved DB instance offerings\cr
#'  \link[paws.database:rds_describe_source_regions]{describe_source_regions} \tab Returns a list of the source Amazon Web Services Regions where the current Amazon Web Services Region can create a read replica, copy a DB snapshot from, or replicate automated backups from\cr
#'  \link[paws.database:rds_describe_valid_db_instance_modifications]{describe_valid_db_instance_modifications} \tab You can call DescribeValidDBInstanceModifications to learn what modifications you can make to your DB instance\cr
#'  \link[paws.database:rds_download_db_log_file_portion]{download_db_log_file_portion} \tab Downloads all or a portion of the specified log file, up to 1 MB in size\cr
#'  \link[paws.database:rds_failover_db_cluster]{failover_db_cluster} \tab Forces a failover for a DB cluster\cr
#'  \link[paws.database:rds_failover_global_cluster]{failover_global_cluster} \tab Promotes the specified secondary DB cluster to be the primary DB cluster in the global database cluster to fail over or switch over a global database\cr
#'  \link[paws.database:rds_list_tags_for_resource]{list_tags_for_resource} \tab Lists all tags on an Amazon RDS resource\cr
#'  \link[paws.database:rds_modify_activity_stream]{modify_activity_stream} \tab Changes the audit policy state of a database activity stream to either locked (default) or unlocked\cr
#'  \link[paws.database:rds_modify_certificates]{modify_certificates} \tab Override the system-default Secure Sockets Layer/Transport Layer Security (SSL/TLS) certificate for Amazon RDS for new DB instances, or remove the override\cr
#'  \link[paws.database:rds_modify_current_db_cluster_capacity]{modify_current_db_cluster_capacity} \tab Set the capacity of an Aurora Serverless v1 DB cluster to a specific value\cr
#'  \link[paws.database:rds_modify_custom_db_engine_version]{modify_custom_db_engine_version} \tab Modifies the status of a custom engine version (CEV)\cr
#'  \link[paws.database:rds_modify_db_cluster]{modify_db_cluster} \tab Modifies the settings of an Amazon Aurora DB cluster or a Multi-AZ DB cluster\cr
#'  \link[paws.database:rds_modify_db_cluster_endpoint]{modify_db_cluster_endpoint} \tab Modifies the properties of an endpoint in an Amazon Aurora DB cluster\cr
#'  \link[paws.database:rds_modify_db_cluster_parameter_group]{modify_db_cluster_parameter_group} \tab Modifies the parameters of a DB cluster parameter group\cr
#'  \link[paws.database:rds_modify_db_cluster_snapshot_attribute]{modify_db_cluster_snapshot_attribute} \tab Adds an attribute and values to, or removes an attribute and values from, a manual DB cluster snapshot\cr
#'  \link[paws.database:rds_modify_db_instance]{modify_db_instance} \tab Modifies settings for a DB instance\cr
#'  \link[paws.database:rds_modify_db_parameter_group]{modify_db_parameter_group} \tab Modifies the parameters of a DB parameter group\cr
#'  \link[paws.database:rds_modify_db_proxy]{modify_db_proxy} \tab Changes the settings for an existing DB proxy\cr
#'  \link[paws.database:rds_modify_db_proxy_endpoint]{modify_db_proxy_endpoint} \tab Changes the settings for an existing DB proxy endpoint\cr
#'  \link[paws.database:rds_modify_db_proxy_target_group]{modify_db_proxy_target_group} \tab Modifies the properties of a DBProxyTargetGroup\cr
#'  \link[paws.database:rds_modify_db_snapshot]{modify_db_snapshot} \tab Updates a manual DB snapshot with a new engine version\cr
#'  \link[paws.database:rds_modify_db_snapshot_attribute]{modify_db_snapshot_attribute} \tab Adds an attribute and values to, or removes an attribute and values from, a manual DB snapshot\cr
#'  \link[paws.database:rds_modify_db_subnet_group]{modify_db_subnet_group} \tab Modifies an existing DB subnet group\cr
#'  \link[paws.database:rds_modify_event_subscription]{modify_event_subscription} \tab Modifies an existing RDS event notification subscription\cr
#'  \link[paws.database:rds_modify_global_cluster]{modify_global_cluster} \tab Modifies a setting for an Amazon Aurora global database cluster\cr
#'  \link[paws.database:rds_modify_option_group]{modify_option_group} \tab Modifies an existing option group\cr
#'  \link[paws.database:rds_promote_read_replica]{promote_read_replica} \tab Promotes a read replica DB instance to a standalone DB instance\cr
#'  \link[paws.database:rds_promote_read_replica_db_cluster]{promote_read_replica_db_cluster} \tab Promotes a read replica DB cluster to a standalone DB cluster\cr
#'  \link[paws.database:rds_purchase_reserved_db_instances_offering]{purchase_reserved_db_instances_offering} \tab Purchases a reserved DB instance offering\cr
#'  \link[paws.database:rds_reboot_db_cluster]{reboot_db_cluster} \tab You might need to reboot your DB cluster, usually for maintenance reasons\cr
#'  \link[paws.database:rds_reboot_db_instance]{reboot_db_instance} \tab You might need to reboot your DB instance, usually for maintenance reasons\cr
#'  \link[paws.database:rds_register_db_proxy_targets]{register_db_proxy_targets} \tab Associate one or more DBProxyTarget data structures with a DBProxyTargetGroup\cr
#'  \link[paws.database:rds_remove_from_global_cluster]{remove_from_global_cluster} \tab Detaches an Aurora secondary cluster from an Aurora global database cluster\cr
#'  \link[paws.database:rds_remove_role_from_db_cluster]{remove_role_from_db_cluster} \tab Removes the asssociation of an Amazon Web Services Identity and Access Management (IAM) role from a DB cluster\cr
#'  \link[paws.database:rds_remove_role_from_db_instance]{remove_role_from_db_instance} \tab Disassociates an Amazon Web Services Identity and Access Management (IAM) role from a DB instance\cr
#'  \link[paws.database:rds_remove_source_identifier_from_subscription]{remove_source_identifier_from_subscription} \tab Removes a source identifier from an existing RDS event notification subscription\cr
#'  \link[paws.database:rds_remove_tags_from_resource]{remove_tags_from_resource} \tab Removes metadata tags from an Amazon RDS resource\cr
#'  \link[paws.database:rds_reset_db_cluster_parameter_group]{reset_db_cluster_parameter_group} \tab Modifies the parameters of a DB cluster parameter group to the default value\cr
#'  \link[paws.database:rds_reset_db_parameter_group]{reset_db_parameter_group} \tab Modifies the parameters of a DB parameter group to the engine/system default value\cr
#'  \link[paws.database:rds_restore_db_cluster_from_s3]{restore_db_cluster_from_s3} \tab Creates an Amazon Aurora DB cluster from MySQL data stored in an Amazon S3 bucket\cr
#'  \link[paws.database:rds_restore_db_cluster_from_snapshot]{restore_db_cluster_from_snapshot} \tab Creates a new DB cluster from a DB snapshot or DB cluster snapshot\cr
#'  \link[paws.database:rds_restore_db_cluster_to_point_in_time]{restore_db_cluster_to_point_in_time} \tab Restores a DB cluster to an arbitrary point in time\cr
#'  \link[paws.database:rds_restore_db_instance_from_db_snapshot]{restore_db_instance_from_db_snapshot} \tab Creates a new DB instance from a DB snapshot\cr
#'  \link[paws.database:rds_restore_db_instance_from_s3]{restore_db_instance_from_s3} \tab Amazon Relational Database Service (Amazon RDS) supports importing MySQL databases by using backup files\cr
#'  \link[paws.database:rds_restore_db_instance_to_point_in_time]{restore_db_instance_to_point_in_time} \tab Restores a DB instance to an arbitrary point in time\cr
#'  \link[paws.database:rds_revoke_db_security_group_ingress]{revoke_db_security_group_ingress} \tab Revokes ingress from a DBSecurityGroup for previously authorized IP ranges or EC2 or VPC security groups\cr
#'  \link[paws.database:rds_start_activity_stream]{start_activity_stream} \tab Starts a database activity stream to monitor activity on the database\cr
#'  \link[paws.database:rds_start_db_cluster]{start_db_cluster} \tab Starts an Amazon Aurora DB cluster that was stopped using the Amazon Web Services console, the stop-db-cluster CLI command, or the StopDBCluster action\cr
#'  \link[paws.database:rds_start_db_instance]{start_db_instance} \tab Starts an Amazon RDS DB instance that was stopped using the Amazon Web Services console, the stop-db-instance CLI command, or the StopDBInstance action\cr
#'  \link[paws.database:rds_start_db_instance_automated_backups_replication]{start_db_instance_automated_backups_replication} \tab Enables replication of automated backups to a different Amazon Web Services Region\cr
#'  \link[paws.database:rds_start_export_task]{start_export_task} \tab Starts an export of DB snapshot or DB cluster data to Amazon S3\cr
#'  \link[paws.database:rds_stop_activity_stream]{stop_activity_stream} \tab Stops a database activity stream that was started using the Amazon Web Services console, the start-activity-stream CLI command, or the StartActivityStream action\cr
#'  \link[paws.database:rds_stop_db_cluster]{stop_db_cluster} \tab Stops an Amazon Aurora DB cluster\cr
#'  \link[paws.database:rds_stop_db_instance]{stop_db_instance} \tab Stops an Amazon RDS DB instance\cr
#'  \link[paws.database:rds_stop_db_instance_automated_backups_replication]{stop_db_instance_automated_backups_replication} \tab Stops automated backup replication for a DB instance\cr
#'  \link[paws.database:rds_switchover_blue_green_deployment]{switchover_blue_green_deployment} \tab Switches over a blue/green deployment\cr
#'  \link[paws.database:rds_switchover_global_cluster]{switchover_global_cluster} \tab Switches over the specified secondary DB cluster to be the new primary DB cluster in the global database cluster\cr
#'  \link[paws.database:rds_switchover_read_replica]{switchover_read_replica} \tab Switches over an Oracle standby database in an Oracle Data Guard environment, making it the new primary database
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname rds
#' @export
rds <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.database::rds(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' AWS RDS DataService
#'
#' @description
#' Amazon RDS Data Service
#'
#' Amazon RDS provides an HTTP endpoint to run SQL statements on an Amazon
#' Aurora Serverless v1 DB cluster. To run these statements, you work with
#' the Data Service API.
#'
#' The Data Service API isn't supported on Amazon Aurora Serverless v2 DB
#' clusters.
#'
#' For more information about the Data Service API, see [Using the Data
#' API](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/data-api.html)
#' in the *Amazon Aurora User Guide*.
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- rdsdataservice(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- rdsdataservice()
#' svc$batch_execute_statement(
#'   Foo = 123
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.database:rdsdataservice_batch_execute_statement]{batch_execute_statement} \tab Runs a batch SQL statement over an array of data\cr
#'  \link[paws.database:rdsdataservice_begin_transaction]{begin_transaction} \tab Starts a SQL transaction\cr
#'  \link[paws.database:rdsdataservice_commit_transaction]{commit_transaction} \tab Ends a SQL transaction started with the BeginTransaction operation and commits the changes\cr
#'  \link[paws.database:rdsdataservice_execute_sql]{execute_sql} \tab Runs one or more SQL statements\cr
#'  \link[paws.database:rdsdataservice_execute_statement]{execute_statement} \tab Runs a SQL statement against a database\cr
#'  \link[paws.database:rdsdataservice_rollback_transaction]{rollback_transaction} \tab Performs a rollback of a transaction
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname rdsdataservice
#' @export
rdsdataservice <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.database::rdsdataservice(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' Amazon Redshift
#'
#' @description
#' **Overview**
#'
#' This is an interface reference for Amazon Redshift. It contains
#' documentation for one of the programming or command line interfaces you
#' can use to manage Amazon Redshift clusters. Note that Amazon Redshift is
#' asynchronous, which means that some interfaces may require techniques,
#' such as polling or asynchronous callback handlers, to determine when a
#' command has been applied. In this reference, the parameter descriptions
#' indicate whether a change is applied immediately, on the next instance
#' reboot, or during the next maintenance window. For a summary of the
#' Amazon Redshift cluster management interfaces, go to [Using the Amazon
#' Redshift Management
#' Interfaces](https://docs.aws.amazon.com/redshift/latest/mgmt/using-aws-sdk.html).
#'
#' Amazon Redshift manages all the work of setting up, operating, and
#' scaling a data warehouse: provisioning capacity, monitoring and backing
#' up the cluster, and applying patches and upgrades to the Amazon Redshift
#' engine. You can focus on using your data to acquire new insights for
#' your business and customers.
#'
#' If you are a first-time user of Amazon Redshift, we recommend that you
#' begin by reading the [Amazon Redshift Getting Started
#' Guide](https://docs.aws.amazon.com/redshift/latest/gsg/getting-started.html).
#'
#' If you are a database developer, the [Amazon Redshift Database Developer
#' Guide](https://docs.aws.amazon.com/redshift/latest/dg/welcome.html)
#' explains how to design, build, query, and maintain the databases that
#' make up your data warehouse.
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- redshift(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- redshift()
#' svc$accept_reserved_node_exchange(
#'   Foo = 123
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.database:redshift_accept_reserved_node_exchange]{accept_reserved_node_exchange} \tab Exchanges a DC1 Reserved Node for a DC2 Reserved Node with no changes to the configuration (term, payment type, or number of nodes) and no additional costs\cr
#'  \link[paws.database:redshift_add_partner]{add_partner} \tab Adds a partner integration to a cluster\cr
#'  \link[paws.database:redshift_associate_data_share_consumer]{associate_data_share_consumer} \tab From a datashare consumer account, associates a datashare with the account (AssociateEntireAccount) or the specified namespace (ConsumerArn)\cr
#'  \link[paws.database:redshift_authorize_cluster_security_group_ingress]{authorize_cluster_security_group_ingress} \tab Adds an inbound (ingress) rule to an Amazon Redshift security group\cr
#'  \link[paws.database:redshift_authorize_data_share]{authorize_data_share} \tab From a data producer account, authorizes the sharing of a datashare with one or more consumer accounts or managing entities\cr
#'  \link[paws.database:redshift_authorize_endpoint_access]{authorize_endpoint_access} \tab Grants access to a cluster\cr
#'  \link[paws.database:redshift_authorize_snapshot_access]{authorize_snapshot_access} \tab Authorizes the specified Amazon Web Services account to restore the specified snapshot\cr
#'  \link[paws.database:redshift_batch_delete_cluster_snapshots]{batch_delete_cluster_snapshots} \tab Deletes a set of cluster snapshots\cr
#'  \link[paws.database:redshift_batch_modify_cluster_snapshots]{batch_modify_cluster_snapshots} \tab Modifies the settings for a set of cluster snapshots\cr
#'  \link[paws.database:redshift_cancel_resize]{cancel_resize} \tab Cancels a resize operation for a cluster\cr
#'  \link[paws.database:redshift_copy_cluster_snapshot]{copy_cluster_snapshot} \tab Copies the specified automated cluster snapshot to a new manual cluster snapshot\cr
#'  \link[paws.database:redshift_create_authentication_profile]{create_authentication_profile} \tab Creates an authentication profile with the specified parameters\cr
#'  \link[paws.database:redshift_create_cluster]{create_cluster} \tab Creates a new cluster with the specified parameters\cr
#'  \link[paws.database:redshift_create_cluster_parameter_group]{create_cluster_parameter_group} \tab Creates an Amazon Redshift parameter group\cr
#'  \link[paws.database:redshift_create_cluster_security_group]{create_cluster_security_group} \tab Creates a new Amazon Redshift security group\cr
#'  \link[paws.database:redshift_create_cluster_snapshot]{create_cluster_snapshot} \tab Creates a manual snapshot of the specified cluster\cr
#'  \link[paws.database:redshift_create_cluster_subnet_group]{create_cluster_subnet_group} \tab Creates a new Amazon Redshift subnet group\cr
#'  \link[paws.database:redshift_create_custom_domain_association]{create_custom_domain_association} \tab Used to create a custom domain name for a cluster\cr
#'  \link[paws.database:redshift_create_endpoint_access]{create_endpoint_access} \tab Creates a Redshift-managed VPC endpoint\cr
#'  \link[paws.database:redshift_create_event_subscription]{create_event_subscription} \tab Creates an Amazon Redshift event notification subscription\cr
#'  \link[paws.database:redshift_create_hsm_client_certificate]{create_hsm_client_certificate} \tab Creates an HSM client certificate that an Amazon Redshift cluster will use to connect to the client's HSM in order to store and retrieve the keys used to encrypt the cluster databases\cr
#'  \link[paws.database:redshift_create_hsm_configuration]{create_hsm_configuration} \tab Creates an HSM configuration that contains the information required by an Amazon Redshift cluster to store and use database encryption keys in a Hardware Security Module (HSM)\cr
#'  \link[paws.database:redshift_create_scheduled_action]{create_scheduled_action} \tab Creates a scheduled action\cr
#'  \link[paws.database:redshift_create_snapshot_copy_grant]{create_snapshot_copy_grant} \tab Creates a snapshot copy grant that permits Amazon Redshift to use an encrypted symmetric key from Key Management Service (KMS) to encrypt copied snapshots in a destination region\cr
#'  \link[paws.database:redshift_create_snapshot_schedule]{create_snapshot_schedule} \tab Create a snapshot schedule that can be associated to a cluster and which overrides the default system backup schedule\cr
#'  \link[paws.database:redshift_create_tags]{create_tags} \tab Adds tags to a cluster\cr
#'  \link[paws.database:redshift_create_usage_limit]{create_usage_limit} \tab Creates a usage limit for a specified Amazon Redshift feature on a cluster\cr
#'  \link[paws.database:redshift_deauthorize_data_share]{deauthorize_data_share} \tab From a datashare producer account, removes authorization from the specified datashare\cr
#'  \link[paws.database:redshift_delete_authentication_profile]{delete_authentication_profile} \tab Deletes an authentication profile\cr
#'  \link[paws.database:redshift_delete_cluster]{delete_cluster} \tab Deletes a previously provisioned cluster without its final snapshot being created\cr
#'  \link[paws.database:redshift_delete_cluster_parameter_group]{delete_cluster_parameter_group} \tab Deletes a specified Amazon Redshift parameter group\cr
#'  \link[paws.database:redshift_delete_cluster_security_group]{delete_cluster_security_group} \tab Deletes an Amazon Redshift security group\cr
#'  \link[paws.database:redshift_delete_cluster_snapshot]{delete_cluster_snapshot} \tab Deletes the specified manual snapshot\cr
#'  \link[paws.database:redshift_delete_cluster_subnet_group]{delete_cluster_subnet_group} \tab Deletes the specified cluster subnet group\cr
#'  \link[paws.database:redshift_delete_custom_domain_association]{delete_custom_domain_association} \tab Contains information about deleting a custom domain association for a cluster\cr
#'  \link[paws.database:redshift_delete_endpoint_access]{delete_endpoint_access} \tab Deletes a Redshift-managed VPC endpoint\cr
#'  \link[paws.database:redshift_delete_event_subscription]{delete_event_subscription} \tab Deletes an Amazon Redshift event notification subscription\cr
#'  \link[paws.database:redshift_delete_hsm_client_certificate]{delete_hsm_client_certificate} \tab Deletes the specified HSM client certificate\cr
#'  \link[paws.database:redshift_delete_hsm_configuration]{delete_hsm_configuration} \tab Deletes the specified Amazon Redshift HSM configuration\cr
#'  \link[paws.database:redshift_delete_partner]{delete_partner} \tab Deletes a partner integration from a cluster\cr
#'  \link[paws.database:redshift_delete_scheduled_action]{delete_scheduled_action} \tab Deletes a scheduled action\cr
#'  \link[paws.database:redshift_delete_snapshot_copy_grant]{delete_snapshot_copy_grant} \tab Deletes the specified snapshot copy grant\cr
#'  \link[paws.database:redshift_delete_snapshot_schedule]{delete_snapshot_schedule} \tab Deletes a snapshot schedule\cr
#'  \link[paws.database:redshift_delete_tags]{delete_tags} \tab Deletes tags from a resource\cr
#'  \link[paws.database:redshift_delete_usage_limit]{delete_usage_limit} \tab Deletes a usage limit from a cluster\cr
#'  \link[paws.database:redshift_describe_account_attributes]{describe_account_attributes} \tab Returns a list of attributes attached to an account\cr
#'  \link[paws.database:redshift_describe_authentication_profiles]{describe_authentication_profiles} \tab Describes an authentication profile\cr
#'  \link[paws.database:redshift_describe_cluster_db_revisions]{describe_cluster_db_revisions} \tab Returns an array of ClusterDbRevision objects\cr
#'  \link[paws.database:redshift_describe_cluster_parameter_groups]{describe_cluster_parameter_groups} \tab Returns a list of Amazon Redshift parameter groups, including parameter groups you created and the default parameter group\cr
#'  \link[paws.database:redshift_describe_cluster_parameters]{describe_cluster_parameters} \tab Returns a detailed list of parameters contained within the specified Amazon Redshift parameter group\cr
#'  \link[paws.database:redshift_describe_clusters]{describe_clusters} \tab Returns properties of provisioned clusters including general cluster properties, cluster database properties, maintenance and backup properties, and security and access properties\cr
#'  \link[paws.database:redshift_describe_cluster_security_groups]{describe_cluster_security_groups} \tab Returns information about Amazon Redshift security groups\cr
#'  \link[paws.database:redshift_describe_cluster_snapshots]{describe_cluster_snapshots} \tab Returns one or more snapshot objects, which contain metadata about your cluster snapshots\cr
#'  \link[paws.database:redshift_describe_cluster_subnet_groups]{describe_cluster_subnet_groups} \tab Returns one or more cluster subnet group objects, which contain metadata about your cluster subnet groups\cr
#'  \link[paws.database:redshift_describe_cluster_tracks]{describe_cluster_tracks} \tab Returns a list of all the available maintenance tracks\cr
#'  \link[paws.database:redshift_describe_cluster_versions]{describe_cluster_versions} \tab Returns descriptions of the available Amazon Redshift cluster versions\cr
#'  \link[paws.database:redshift_describe_custom_domain_associations]{describe_custom_domain_associations} \tab Contains information for custom domain associations for a cluster\cr
#'  \link[paws.database:redshift_describe_data_shares]{describe_data_shares} \tab Shows the status of any inbound or outbound datashares available in the specified account\cr
#'  \link[paws.database:redshift_describe_data_shares_for_consumer]{describe_data_shares_for_consumer} \tab Returns a list of datashares where the account identifier being called is a consumer account identifier\cr
#'  \link[paws.database:redshift_describe_data_shares_for_producer]{describe_data_shares_for_producer} \tab Returns a list of datashares when the account identifier being called is a producer account identifier\cr
#'  \link[paws.database:redshift_describe_default_cluster_parameters]{describe_default_cluster_parameters} \tab Returns a list of parameter settings for the specified parameter group family\cr
#'  \link[paws.database:redshift_describe_endpoint_access]{describe_endpoint_access} \tab Describes a Redshift-managed VPC endpoint\cr
#'  \link[paws.database:redshift_describe_endpoint_authorization]{describe_endpoint_authorization} \tab Describes an endpoint authorization\cr
#'  \link[paws.database:redshift_describe_event_categories]{describe_event_categories} \tab Displays a list of event categories for all event source types, or for a specified source type\cr
#'  \link[paws.database:redshift_describe_events]{describe_events} \tab Returns events related to clusters, security groups, snapshots, and parameter groups for the past 14 days\cr
#'  \link[paws.database:redshift_describe_event_subscriptions]{describe_event_subscriptions} \tab Lists descriptions of all the Amazon Redshift event notification subscriptions for a customer account\cr
#'  \link[paws.database:redshift_describe_hsm_client_certificates]{describe_hsm_client_certificates} \tab Returns information about the specified HSM client certificate\cr
#'  \link[paws.database:redshift_describe_hsm_configurations]{describe_hsm_configurations} \tab Returns information about the specified Amazon Redshift HSM configuration\cr
#'  \link[paws.database:redshift_describe_logging_status]{describe_logging_status} \tab Describes whether information, such as queries and connection attempts, is being logged for the specified Amazon Redshift cluster\cr
#'  \link[paws.database:redshift_describe_node_configuration_options]{describe_node_configuration_options} \tab Returns properties of possible node configurations such as node type, number of nodes, and disk usage for the specified action type\cr
#'  \link[paws.database:redshift_describe_orderable_cluster_options]{describe_orderable_cluster_options} \tab Returns a list of orderable cluster options\cr
#'  \link[paws.database:redshift_describe_partners]{describe_partners} \tab Returns information about the partner integrations defined for a cluster\cr
#'  \link[paws.database:redshift_describe_reserved_node_exchange_status]{describe_reserved_node_exchange_status} \tab Returns exchange status details and associated metadata for a reserved-node exchange\cr
#'  \link[paws.database:redshift_describe_reserved_node_offerings]{describe_reserved_node_offerings} \tab Returns a list of the available reserved node offerings by Amazon Redshift with their descriptions including the node type, the fixed and recurring costs of reserving the node and duration the node will be reserved for you\cr
#'  \link[paws.database:redshift_describe_reserved_nodes]{describe_reserved_nodes} \tab Returns the descriptions of the reserved nodes\cr
#'  \link[paws.database:redshift_describe_resize]{describe_resize} \tab Returns information about the last resize operation for the specified cluster\cr
#'  \link[paws.database:redshift_describe_scheduled_actions]{describe_scheduled_actions} \tab Describes properties of scheduled actions\cr
#'  \link[paws.database:redshift_describe_snapshot_copy_grants]{describe_snapshot_copy_grants} \tab Returns a list of snapshot copy grants owned by the Amazon Web Services account in the destination region\cr
#'  \link[paws.database:redshift_describe_snapshot_schedules]{describe_snapshot_schedules} \tab Returns a list of snapshot schedules\cr
#'  \link[paws.database:redshift_describe_storage]{describe_storage} \tab Returns account level backups storage size and provisional storage\cr
#'  \link[paws.database:redshift_describe_table_restore_status]{describe_table_restore_status} \tab Lists the status of one or more table restore requests made using the RestoreTableFromClusterSnapshot API action\cr
#'  \link[paws.database:redshift_describe_tags]{describe_tags} \tab Returns a list of tags\cr
#'  \link[paws.database:redshift_describe_usage_limits]{describe_usage_limits} \tab Shows usage limits on a cluster\cr
#'  \link[paws.database:redshift_disable_logging]{disable_logging} \tab Stops logging information, such as queries and connection attempts, for the specified Amazon Redshift cluster\cr
#'  \link[paws.database:redshift_disable_snapshot_copy]{disable_snapshot_copy} \tab Disables the automatic copying of snapshots from one region to another region for a specified cluster\cr
#'  \link[paws.database:redshift_disassociate_data_share_consumer]{disassociate_data_share_consumer} \tab From a datashare consumer account, remove association for the specified datashare\cr
#'  \link[paws.database:redshift_enable_logging]{enable_logging} \tab Starts logging information, such as queries and connection attempts, for the specified Amazon Redshift cluster\cr
#'  \link[paws.database:redshift_enable_snapshot_copy]{enable_snapshot_copy} \tab Enables the automatic copy of snapshots from one region to another region for a specified cluster\cr
#'  \link[paws.database:redshift_get_cluster_credentials]{get_cluster_credentials} \tab Returns a database user name and temporary password with temporary authorization to log on to an Amazon Redshift database\cr
#'  \link[paws.database:redshift_get_cluster_credentials_with_iam]{get_cluster_credentials_with_iam} \tab Returns a database user name and temporary password with temporary authorization to log in to an Amazon Redshift database\cr
#'  \link[paws.database:redshift_get_reserved_node_exchange_configuration_options]{get_reserved_node_exchange_configuration_options} \tab Gets the configuration options for the reserved-node exchange\cr
#'  \link[paws.database:redshift_get_reserved_node_exchange_offerings]{get_reserved_node_exchange_offerings} \tab Returns an array of DC2 ReservedNodeOfferings that matches the payment type, term, and usage price of the given DC1 reserved node\cr
#'  \link[paws.database:redshift_modify_aqua_configuration]{modify_aqua_configuration} \tab This operation is retired\cr
#'  \link[paws.database:redshift_modify_authentication_profile]{modify_authentication_profile} \tab Modifies an authentication profile\cr
#'  \link[paws.database:redshift_modify_cluster]{modify_cluster} \tab Modifies the settings for a cluster\cr
#'  \link[paws.database:redshift_modify_cluster_db_revision]{modify_cluster_db_revision} \tab Modifies the database revision of a cluster\cr
#'  \link[paws.database:redshift_modify_cluster_iam_roles]{modify_cluster_iam_roles} \tab Modifies the list of Identity and Access Management (IAM) roles that can be used by the cluster to access other Amazon Web Services services\cr
#'  \link[paws.database:redshift_modify_cluster_maintenance]{modify_cluster_maintenance} \tab Modifies the maintenance settings of a cluster\cr
#'  \link[paws.database:redshift_modify_cluster_parameter_group]{modify_cluster_parameter_group} \tab Modifies the parameters of a parameter group\cr
#'  \link[paws.database:redshift_modify_cluster_snapshot]{modify_cluster_snapshot} \tab Modifies the settings for a snapshot\cr
#'  \link[paws.database:redshift_modify_cluster_snapshot_schedule]{modify_cluster_snapshot_schedule} \tab Modifies a snapshot schedule for a cluster\cr
#'  \link[paws.database:redshift_modify_cluster_subnet_group]{modify_cluster_subnet_group} \tab Modifies a cluster subnet group to include the specified list of VPC subnets\cr
#'  \link[paws.database:redshift_modify_custom_domain_association]{modify_custom_domain_association} \tab Contains information for changing a custom domain association\cr
#'  \link[paws.database:redshift_modify_endpoint_access]{modify_endpoint_access} \tab Modifies a Redshift-managed VPC endpoint\cr
#'  \link[paws.database:redshift_modify_event_subscription]{modify_event_subscription} \tab Modifies an existing Amazon Redshift event notification subscription\cr
#'  \link[paws.database:redshift_modify_scheduled_action]{modify_scheduled_action} \tab Modifies a scheduled action\cr
#'  \link[paws.database:redshift_modify_snapshot_copy_retention_period]{modify_snapshot_copy_retention_period} \tab Modifies the number of days to retain snapshots in the destination Amazon Web Services Region after they are copied from the source Amazon Web Services Region\cr
#'  \link[paws.database:redshift_modify_snapshot_schedule]{modify_snapshot_schedule} \tab Modifies a snapshot schedule\cr
#'  \link[paws.database:redshift_modify_usage_limit]{modify_usage_limit} \tab Modifies a usage limit in a cluster\cr
#'  \link[paws.database:redshift_pause_cluster]{pause_cluster} \tab Pauses a cluster\cr
#'  \link[paws.database:redshift_purchase_reserved_node_offering]{purchase_reserved_node_offering} \tab Allows you to purchase reserved nodes\cr
#'  \link[paws.database:redshift_reboot_cluster]{reboot_cluster} \tab Reboots a cluster\cr
#'  \link[paws.database:redshift_reject_data_share]{reject_data_share} \tab From a datashare consumer account, rejects the specified datashare\cr
#'  \link[paws.database:redshift_reset_cluster_parameter_group]{reset_cluster_parameter_group} \tab Sets one or more parameters of the specified parameter group to their default values and sets the source values of the parameters to "engine-default"\cr
#'  \link[paws.database:redshift_resize_cluster]{resize_cluster} \tab Changes the size of the cluster\cr
#'  \link[paws.database:redshift_restore_from_cluster_snapshot]{restore_from_cluster_snapshot} \tab Creates a new cluster from a snapshot\cr
#'  \link[paws.database:redshift_restore_table_from_cluster_snapshot]{restore_table_from_cluster_snapshot} \tab Creates a new table from a table in an Amazon Redshift cluster snapshot\cr
#'  \link[paws.database:redshift_resume_cluster]{resume_cluster} \tab Resumes a paused cluster\cr
#'  \link[paws.database:redshift_revoke_cluster_security_group_ingress]{revoke_cluster_security_group_ingress} \tab Revokes an ingress rule in an Amazon Redshift security group for a previously authorized IP range or Amazon EC2 security group\cr
#'  \link[paws.database:redshift_revoke_endpoint_access]{revoke_endpoint_access} \tab Revokes access to a cluster\cr
#'  \link[paws.database:redshift_revoke_snapshot_access]{revoke_snapshot_access} \tab Removes the ability of the specified Amazon Web Services account to restore the specified snapshot\cr
#'  \link[paws.database:redshift_rotate_encryption_key]{rotate_encryption_key} \tab Rotates the encryption keys for a cluster\cr
#'  \link[paws.database:redshift_update_partner_status]{update_partner_status} \tab Updates the status of a partner integration
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname redshift
#' @export
redshift <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.database::redshift(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' Redshift Data API Service
#'
#' @description
#' You can use the Amazon Redshift Data API to run queries on Amazon
#' Redshift tables. You can run SQL statements, which are committed if the
#' statement succeeds.
#'
#' For more information about the Amazon Redshift Data API and CLI usage
#' examples, see [Using the Amazon Redshift Data
#' API](https://docs.aws.amazon.com/redshift/latest/mgmt/data-api.html) in
#' the *Amazon Redshift Management Guide*.
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- redshiftdataapiservice(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- redshiftdataapiservice()
#' svc$batch_execute_statement(
#'   Foo = 123
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.database:redshiftdataapiservice_batch_execute_statement]{batch_execute_statement} \tab Runs one or more SQL statements, which can be data manipulation language (DML) or data definition language (DDL)\cr
#'  \link[paws.database:redshiftdataapiservice_cancel_statement]{cancel_statement} \tab Cancels a running query\cr
#'  \link[paws.database:redshiftdataapiservice_describe_statement]{describe_statement} \tab Describes the details about a specific instance when a query was run by the Amazon Redshift Data API\cr
#'  \link[paws.database:redshiftdataapiservice_describe_table]{describe_table} \tab Describes the detailed information about a table from metadata in the cluster\cr
#'  \link[paws.database:redshiftdataapiservice_execute_statement]{execute_statement} \tab Runs an SQL statement, which can be data manipulation language (DML) or data definition language (DDL)\cr
#'  \link[paws.database:redshiftdataapiservice_get_statement_result]{get_statement_result} \tab Fetches the temporarily cached result of an SQL statement\cr
#'  \link[paws.database:redshiftdataapiservice_list_databases]{list_databases} \tab List the databases in a cluster\cr
#'  \link[paws.database:redshiftdataapiservice_list_schemas]{list_schemas} \tab Lists the schemas in a database\cr
#'  \link[paws.database:redshiftdataapiservice_list_statements]{list_statements} \tab List of SQL statements\cr
#'  \link[paws.database:redshiftdataapiservice_list_tables]{list_tables} \tab List the tables in a database
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname redshiftdataapiservice
#' @export
redshiftdataapiservice <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.database::redshiftdataapiservice(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' Redshift Serverless
#'
#' @description
#' This is an interface reference for Amazon Redshift Serverless. It
#' contains documentation for one of the programming or command line
#' interfaces you can use to manage Amazon Redshift Serverless.
#'
#' Amazon Redshift Serverless automatically provisions data warehouse
#' capacity and intelligently scales the underlying resources based on
#' workload demands. Amazon Redshift Serverless adjusts capacity in seconds
#' to deliver consistently high performance and simplified operations for
#' even the most demanding and volatile workloads. Amazon Redshift
#' Serverless lets you focus on using your data to acquire new insights for
#' your business and customers.
#'
#' To learn more about Amazon Redshift Serverless, see [What is Amazon
#' Redshift
#' Serverless](https://docs.aws.amazon.com/redshift/latest/mgmt/serverless-whatis.html).
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- redshiftserverless(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- redshiftserverless()
#' svc$convert_recovery_point_to_snapshot(
#'   Foo = 123
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.database:redshiftserverless_convert_recovery_point_to_snapshot]{convert_recovery_point_to_snapshot} \tab Converts a recovery point to a snapshot\cr
#'  \link[paws.database:redshiftserverless_create_endpoint_access]{create_endpoint_access} \tab Creates an Amazon Redshift Serverless managed VPC endpoint\cr
#'  \link[paws.database:redshiftserverless_create_namespace]{create_namespace} \tab Creates a namespace in Amazon Redshift Serverless\cr
#'  \link[paws.database:redshiftserverless_create_snapshot]{create_snapshot} \tab Creates a snapshot of all databases in a namespace\cr
#'  \link[paws.database:redshiftserverless_create_usage_limit]{create_usage_limit} \tab Creates a usage limit for a specified Amazon Redshift Serverless usage type\cr
#'  \link[paws.database:redshiftserverless_create_workgroup]{create_workgroup} \tab Creates an workgroup in Amazon Redshift Serverless\cr
#'  \link[paws.database:redshiftserverless_delete_endpoint_access]{delete_endpoint_access} \tab Deletes an Amazon Redshift Serverless managed VPC endpoint\cr
#'  \link[paws.database:redshiftserverless_delete_namespace]{delete_namespace} \tab Deletes a namespace from Amazon Redshift Serverless\cr
#'  \link[paws.database:redshiftserverless_delete_resource_policy]{delete_resource_policy} \tab Deletes the specified resource policy\cr
#'  \link[paws.database:redshiftserverless_delete_snapshot]{delete_snapshot} \tab Deletes a snapshot from Amazon Redshift Serverless\cr
#'  \link[paws.database:redshiftserverless_delete_usage_limit]{delete_usage_limit} \tab Deletes a usage limit from Amazon Redshift Serverless\cr
#'  \link[paws.database:redshiftserverless_delete_workgroup]{delete_workgroup} \tab Deletes a workgroup\cr
#'  \link[paws.database:redshiftserverless_get_credentials]{get_credentials} \tab Returns a database user name and temporary password with temporary authorization to log in to Amazon Redshift Serverless\cr
#'  \link[paws.database:redshiftserverless_get_endpoint_access]{get_endpoint_access} \tab Returns information, such as the name, about a VPC endpoint\cr
#'  \link[paws.database:redshiftserverless_get_namespace]{get_namespace} \tab Returns information about a namespace in Amazon Redshift Serverless\cr
#'  \link[paws.database:redshiftserverless_get_recovery_point]{get_recovery_point} \tab Returns information about a recovery point\cr
#'  \link[paws.database:redshiftserverless_get_resource_policy]{get_resource_policy} \tab Returns a resource policy\cr
#'  \link[paws.database:redshiftserverless_get_snapshot]{get_snapshot} \tab Returns information about a specific snapshot\cr
#'  \link[paws.database:redshiftserverless_get_table_restore_status]{get_table_restore_status} \tab Returns information about a TableRestoreStatus object\cr
#'  \link[paws.database:redshiftserverless_get_usage_limit]{get_usage_limit} \tab Returns information about a usage limit\cr
#'  \link[paws.database:redshiftserverless_get_workgroup]{get_workgroup} \tab Returns information about a specific workgroup\cr
#'  \link[paws.database:redshiftserverless_list_endpoint_access]{list_endpoint_access} \tab Returns an array of EndpointAccess objects and relevant information\cr
#'  \link[paws.database:redshiftserverless_list_namespaces]{list_namespaces} \tab Returns information about a list of specified namespaces\cr
#'  \link[paws.database:redshiftserverless_list_recovery_points]{list_recovery_points} \tab Returns an array of recovery points\cr
#'  \link[paws.database:redshiftserverless_list_snapshots]{list_snapshots} \tab Returns a list of snapshots\cr
#'  \link[paws.database:redshiftserverless_list_table_restore_status]{list_table_restore_status} \tab Returns information about an array of TableRestoreStatus objects\cr
#'  \link[paws.database:redshiftserverless_list_tags_for_resource]{list_tags_for_resource} \tab Lists the tags assigned to a resource\cr
#'  \link[paws.database:redshiftserverless_list_usage_limits]{list_usage_limits} \tab Lists all usage limits within Amazon Redshift Serverless\cr
#'  \link[paws.database:redshiftserverless_list_workgroups]{list_workgroups} \tab Returns information about a list of specified workgroups\cr
#'  \link[paws.database:redshiftserverless_put_resource_policy]{put_resource_policy} \tab Creates or updates a resource policy\cr
#'  \link[paws.database:redshiftserverless_restore_from_recovery_point]{restore_from_recovery_point} \tab Restore the data from a recovery point\cr
#'  \link[paws.database:redshiftserverless_restore_from_snapshot]{restore_from_snapshot} \tab Restores a namespace from a snapshot\cr
#'  \link[paws.database:redshiftserverless_restore_table_from_snapshot]{restore_table_from_snapshot} \tab Restores a table from a snapshot to your Amazon Redshift Serverless instance\cr
#'  \link[paws.database:redshiftserverless_tag_resource]{tag_resource} \tab Assigns one or more tags to a resource\cr
#'  \link[paws.database:redshiftserverless_untag_resource]{untag_resource} \tab Removes a tag or set of tags from a resource\cr
#'  \link[paws.database:redshiftserverless_update_endpoint_access]{update_endpoint_access} \tab Updates an Amazon Redshift Serverless managed endpoint\cr
#'  \link[paws.database:redshiftserverless_update_namespace]{update_namespace} \tab Updates a namespace with the specified settings\cr
#'  \link[paws.database:redshiftserverless_update_snapshot]{update_snapshot} \tab Updates a snapshot\cr
#'  \link[paws.database:redshiftserverless_update_usage_limit]{update_usage_limit} \tab Update a usage limit in Amazon Redshift Serverless\cr
#'  \link[paws.database:redshiftserverless_update_workgroup]{update_workgroup} \tab Updates a workgroup with the specified configuration settings
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname redshiftserverless
#' @export
redshiftserverless <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.database::redshiftserverless(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' Amazon SimpleDB
#'
#' @description
#' Amazon SimpleDB is a web service providing the core database functions
#' of data indexing and querying in the cloud. By offloading the time and
#' effort associated with building and operating a web-scale database,
#' SimpleDB provides developers the freedom to focus on application
#' development.
#'
#' A traditional, clustered relational database requires a sizable upfront
#' capital outlay, is complex to design, and often requires extensive and
#' repetitive database administration. Amazon SimpleDB is dramatically
#' simpler, requiring no schema, automatically indexing your data and
#' providing a simple API for storage and access. This approach eliminates
#' the administrative burden of data modeling, index maintenance, and
#' performance tuning. Developers gain access to this functionality within
#' Amazon's proven computing environment, are able to scale instantly, and
#' pay only for what they use.
#'
#' Visit
#' [http://aws.amazon.com/simpledb/](https://aws.amazon.com/simpledb/) for
#' more information.
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- simpledb(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- simpledb()
#' svc$batch_delete_attributes(
#'   Foo = 123
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.database:simpledb_batch_delete_attributes]{batch_delete_attributes} \tab Performs multiple DeleteAttributes operations in a single call, which reduces round trips and latencies\cr
#'  \link[paws.database:simpledb_batch_put_attributes]{batch_put_attributes} \tab The BatchPutAttributes operation creates or replaces attributes within one or more items\cr
#'  \link[paws.database:simpledb_create_domain]{create_domain} \tab The CreateDomain operation creates a new domain\cr
#'  \link[paws.database:simpledb_delete_attributes]{delete_attributes} \tab Deletes one or more attributes associated with an item\cr
#'  \link[paws.database:simpledb_delete_domain]{delete_domain} \tab The DeleteDomain operation deletes a domain\cr
#'  \link[paws.database:simpledb_domain_metadata]{domain_metadata} \tab Returns information about the domain, including when the domain was created, the number of items and attributes in the domain, and the size of the attribute names and values\cr
#'  \link[paws.database:simpledb_get_attributes]{get_attributes} \tab Returns all of the attributes associated with the specified item\cr
#'  \link[paws.database:simpledb_list_domains]{list_domains} \tab The ListDomains operation lists all domains associated with the Access Key ID\cr
#'  \link[paws.database:simpledb_put_attributes]{put_attributes} \tab The PutAttributes operation creates or replaces attributes in an item\cr
#'  \link[paws.database:simpledb_select]{select} \tab The Select operation returns a set of attributes for ItemNames that match the select expression
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname simpledb
#' @export
simpledb <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.database::simpledb(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' Amazon Timestream Query
#'
#' @description
#' Amazon Timestream Query
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- timestreamquery(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- timestreamquery()
#' svc$cancel_query(
#'   Foo = 123
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.database:timestreamquery_cancel_query]{cancel_query} \tab Cancels a query that has been issued\cr
#'  \link[paws.database:timestreamquery_create_scheduled_query]{create_scheduled_query} \tab Create a scheduled query that will be run on your behalf at the configured schedule\cr
#'  \link[paws.database:timestreamquery_delete_scheduled_query]{delete_scheduled_query} \tab Deletes a given scheduled query\cr
#'  \link[paws.database:timestreamquery_describe_endpoints]{describe_endpoints} \tab DescribeEndpoints returns a list of available endpoints to make Timestream API calls against\cr
#'  \link[paws.database:timestreamquery_describe_scheduled_query]{describe_scheduled_query} \tab Provides detailed information about a scheduled query\cr
#'  \link[paws.database:timestreamquery_execute_scheduled_query]{execute_scheduled_query} \tab You can use this API to run a scheduled query manually\cr
#'  \link[paws.database:timestreamquery_list_scheduled_queries]{list_scheduled_queries} \tab Gets a list of all scheduled queries in the caller's Amazon account and Region\cr
#'  \link[paws.database:timestreamquery_list_tags_for_resource]{list_tags_for_resource} \tab List all tags on a Timestream query resource\cr
#'  \link[paws.database:timestreamquery_prepare_query]{prepare_query} \tab A synchronous operation that allows you to submit a query with parameters to be stored by Timestream for later running\cr
#'  \link[paws.database:timestreamquery_query]{query} \tab Query is a synchronous operation that enables you to run a query against your Amazon Timestream data\cr
#'  \link[paws.database:timestreamquery_tag_resource]{tag_resource} \tab Associate a set of tags with a Timestream resource\cr
#'  \link[paws.database:timestreamquery_untag_resource]{untag_resource} \tab Removes the association of tags from a Timestream query resource\cr
#'  \link[paws.database:timestreamquery_update_scheduled_query]{update_scheduled_query} \tab Update a scheduled query
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname timestreamquery
#' @export
timestreamquery <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.database::timestreamquery(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' Amazon Timestream Write
#'
#' @description
#' Amazon Timestream is a fast, scalable, fully managed time-series
#' database service that makes it easy to store and analyze trillions of
#' time-series data points per day. With Timestream, you can easily store
#' and analyze IoT sensor data to derive insights from your IoT
#' applications. You can analyze industrial telemetry to streamline
#' equipment management and maintenance. You can also store and analyze log
#' data and metrics to improve the performance and availability of your
#' applications.
#'
#' Timestream is built from the ground up to effectively ingest, process,
#' and store time-series data. It organizes data to optimize query
#' processing. It automatically scales based on the volume of data ingested
#' and on the query volume to ensure you receive optimal performance while
#' inserting and querying data. As your data grows over time, Timestream’s
#' adaptive query processing engine spans across storage tiers to provide
#' fast analysis while reducing costs.
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- timestreamwrite(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- timestreamwrite()
#' svc$create_batch_load_task(
#'   Foo = 123
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.database:timestreamwrite_create_batch_load_task]{create_batch_load_task} \tab Creates a new Timestream batch load task\cr
#'  \link[paws.database:timestreamwrite_create_database]{create_database} \tab Creates a new Timestream database\cr
#'  \link[paws.database:timestreamwrite_create_table]{create_table} \tab Adds a new table to an existing database in your account\cr
#'  \link[paws.database:timestreamwrite_delete_database]{delete_database} \tab Deletes a given Timestream database\cr
#'  \link[paws.database:timestreamwrite_delete_table]{delete_table} \tab Deletes a given Timestream table\cr
#'  \link[paws.database:timestreamwrite_describe_batch_load_task]{describe_batch_load_task} \tab Returns information about the batch load task, including configurations, mappings, progress, and other details\cr
#'  \link[paws.database:timestreamwrite_describe_database]{describe_database} \tab Returns information about the database, including the database name, time that the database was created, and the total number of tables found within the database\cr
#'  \link[paws.database:timestreamwrite_describe_endpoints]{describe_endpoints} \tab Returns a list of available endpoints to make Timestream API calls against\cr
#'  \link[paws.database:timestreamwrite_describe_table]{describe_table} \tab Returns information about the table, including the table name, database name, retention duration of the memory store and the magnetic store\cr
#'  \link[paws.database:timestreamwrite_list_batch_load_tasks]{list_batch_load_tasks} \tab Provides a list of batch load tasks, along with the name, status, when the task is resumable until, and other details\cr
#'  \link[paws.database:timestreamwrite_list_databases]{list_databases} \tab Returns a list of your Timestream databases\cr
#'  \link[paws.database:timestreamwrite_list_tables]{list_tables} \tab Provides a list of tables, along with the name, status, and retention properties of each table\cr
#'  \link[paws.database:timestreamwrite_list_tags_for_resource]{list_tags_for_resource} \tab Lists all tags on a Timestream resource\cr
#'  \link[paws.database:timestreamwrite_resume_batch_load_task]{resume_batch_load_task} \tab Resume batch load task\cr
#'  \link[paws.database:timestreamwrite_tag_resource]{tag_resource} \tab Associates a set of tags with a Timestream resource\cr
#'  \link[paws.database:timestreamwrite_untag_resource]{untag_resource} \tab Removes the association of tags from a Timestream resource\cr
#'  \link[paws.database:timestreamwrite_update_database]{update_database} \tab Modifies the KMS key for an existing database\cr
#'  \link[paws.database:timestreamwrite_update_table]{update_table} \tab Modifies the retention duration of the memory store and magnetic store for your Timestream table\cr
#'  \link[paws.database:timestreamwrite_write_records]{write_records} \tab Enables you to write your time-series data into Timestream
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname timestreamwrite
#' @export
timestreamwrite <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.database::timestreamwrite(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' Amazon API Gateway
#'
#' @description
#' Amazon API Gateway helps developers deliver robust, secure, and scalable
#' mobile and web application back ends. API Gateway allows developers to
#' securely connect mobile and web applications to APIs that run on Lambda,
#' Amazon EC2, or other publicly addressable web services that are hosted
#' outside of AWS.
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- apigateway(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- apigateway()
#' svc$create_api_key(
#'   Foo = 123
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.networking:apigateway_create_api_key]{create_api_key} \tab Create an ApiKey resource\cr
#'  \link[paws.networking:apigateway_create_authorizer]{create_authorizer} \tab Adds a new Authorizer resource to an existing RestApi resource\cr
#'  \link[paws.networking:apigateway_create_base_path_mapping]{create_base_path_mapping} \tab Creates a new BasePathMapping resource\cr
#'  \link[paws.networking:apigateway_create_deployment]{create_deployment} \tab Creates a Deployment resource, which makes a specified RestApi callable over the internet\cr
#'  \link[paws.networking:apigateway_create_documentation_part]{create_documentation_part} \tab Creates a documentation part\cr
#'  \link[paws.networking:apigateway_create_documentation_version]{create_documentation_version} \tab Creates a documentation version\cr
#'  \link[paws.networking:apigateway_create_domain_name]{create_domain_name} \tab Creates a new domain name\cr
#'  \link[paws.networking:apigateway_create_model]{create_model} \tab Adds a new Model resource to an existing RestApi resource\cr
#'  \link[paws.networking:apigateway_create_request_validator]{create_request_validator} \tab Creates a RequestValidator of a given RestApi\cr
#'  \link[paws.networking:apigateway_create_resource]{create_resource} \tab Creates a Resource resource\cr
#'  \link[paws.networking:apigateway_create_rest_api]{create_rest_api} \tab Creates a new RestApi resource\cr
#'  \link[paws.networking:apigateway_create_stage]{create_stage} \tab Creates a new Stage resource that references a pre-existing Deployment for the API\cr
#'  \link[paws.networking:apigateway_create_usage_plan]{create_usage_plan} \tab Creates a usage plan with the throttle and quota limits, as well as the associated API stages, specified in the payload\cr
#'  \link[paws.networking:apigateway_create_usage_plan_key]{create_usage_plan_key} \tab Creates a usage plan key for adding an existing API key to a usage plan\cr
#'  \link[paws.networking:apigateway_create_vpc_link]{create_vpc_link} \tab Creates a VPC link, under the caller's account in a selected region, in an asynchronous operation that typically takes 2-4 minutes to complete and become operational\cr
#'  \link[paws.networking:apigateway_delete_api_key]{delete_api_key} \tab Deletes the ApiKey resource\cr
#'  \link[paws.networking:apigateway_delete_authorizer]{delete_authorizer} \tab Deletes an existing Authorizer resource\cr
#'  \link[paws.networking:apigateway_delete_base_path_mapping]{delete_base_path_mapping} \tab Deletes the BasePathMapping resource\cr
#'  \link[paws.networking:apigateway_delete_client_certificate]{delete_client_certificate} \tab Deletes the ClientCertificate resource\cr
#'  \link[paws.networking:apigateway_delete_deployment]{delete_deployment} \tab Deletes a Deployment resource\cr
#'  \link[paws.networking:apigateway_delete_documentation_part]{delete_documentation_part} \tab Deletes a documentation part\cr
#'  \link[paws.networking:apigateway_delete_documentation_version]{delete_documentation_version} \tab Deletes a documentation version\cr
#'  \link[paws.networking:apigateway_delete_domain_name]{delete_domain_name} \tab Deletes the DomainName resource\cr
#'  \link[paws.networking:apigateway_delete_gateway_response]{delete_gateway_response} \tab Clears any customization of a GatewayResponse of a specified response type on the given RestApi and resets it with the default settings\cr
#'  \link[paws.networking:apigateway_delete_integration]{delete_integration} \tab Represents a delete integration\cr
#'  \link[paws.networking:apigateway_delete_integration_response]{delete_integration_response} \tab Represents a delete integration response\cr
#'  \link[paws.networking:apigateway_delete_method]{delete_method} \tab Deletes an existing Method resource\cr
#'  \link[paws.networking:apigateway_delete_method_response]{delete_method_response} \tab Deletes an existing MethodResponse resource\cr
#'  \link[paws.networking:apigateway_delete_model]{delete_model} \tab Deletes a model\cr
#'  \link[paws.networking:apigateway_delete_request_validator]{delete_request_validator} \tab Deletes a RequestValidator of a given RestApi\cr
#'  \link[paws.networking:apigateway_delete_resource]{delete_resource} \tab Deletes a Resource resource\cr
#'  \link[paws.networking:apigateway_delete_rest_api]{delete_rest_api} \tab Deletes the specified API\cr
#'  \link[paws.networking:apigateway_delete_stage]{delete_stage} \tab Deletes a Stage resource\cr
#'  \link[paws.networking:apigateway_delete_usage_plan]{delete_usage_plan} \tab Deletes a usage plan of a given plan Id\cr
#'  \link[paws.networking:apigateway_delete_usage_plan_key]{delete_usage_plan_key} \tab Deletes a usage plan key and remove the underlying API key from the associated usage plan\cr
#'  \link[paws.networking:apigateway_delete_vpc_link]{delete_vpc_link} \tab Deletes an existing VpcLink of a specified identifier\cr
#'  \link[paws.networking:apigateway_flush_stage_authorizers_cache]{flush_stage_authorizers_cache} \tab Flushes all authorizer cache entries on a stage\cr
#'  \link[paws.networking:apigateway_flush_stage_cache]{flush_stage_cache} \tab Flushes a stage's cache\cr
#'  \link[paws.networking:apigateway_generate_client_certificate]{generate_client_certificate} \tab Generates a ClientCertificate resource\cr
#'  \link[paws.networking:apigateway_get_account]{get_account} \tab Gets information about the current Account resource\cr
#'  \link[paws.networking:apigateway_get_api_key]{get_api_key} \tab Gets information about the current ApiKey resource\cr
#'  \link[paws.networking:apigateway_get_api_keys]{get_api_keys} \tab Gets information about the current ApiKeys resource\cr
#'  \link[paws.networking:apigateway_get_authorizer]{get_authorizer} \tab Describe an existing Authorizer resource\cr
#'  \link[paws.networking:apigateway_get_authorizers]{get_authorizers} \tab Describe an existing Authorizers resource\cr
#'  \link[paws.networking:apigateway_get_base_path_mapping]{get_base_path_mapping} \tab Describe a BasePathMapping resource\cr
#'  \link[paws.networking:apigateway_get_base_path_mappings]{get_base_path_mappings} \tab Represents a collection of BasePathMapping resources\cr
#'  \link[paws.networking:apigateway_get_client_certificate]{get_client_certificate} \tab Gets information about the current ClientCertificate resource\cr
#'  \link[paws.networking:apigateway_get_client_certificates]{get_client_certificates} \tab Gets a collection of ClientCertificate resources\cr
#'  \link[paws.networking:apigateway_get_deployment]{get_deployment} \tab Gets information about a Deployment resource\cr
#'  \link[paws.networking:apigateway_get_deployments]{get_deployments} \tab Gets information about a Deployments collection\cr
#'  \link[paws.networking:apigateway_get_documentation_part]{get_documentation_part} \tab Gets a documentation part\cr
#'  \link[paws.networking:apigateway_get_documentation_parts]{get_documentation_parts} \tab Gets documentation parts\cr
#'  \link[paws.networking:apigateway_get_documentation_version]{get_documentation_version} \tab Gets a documentation version\cr
#'  \link[paws.networking:apigateway_get_documentation_versions]{get_documentation_versions} \tab Gets documentation versions\cr
#'  \link[paws.networking:apigateway_get_domain_name]{get_domain_name} \tab Represents a domain name that is contained in a simpler, more intuitive URL that can be called\cr
#'  \link[paws.networking:apigateway_get_domain_names]{get_domain_names} \tab Represents a collection of DomainName resources\cr
#'  \link[paws.networking:apigateway_get_export]{get_export} \tab Exports a deployed version of a RestApi in a specified format\cr
#'  \link[paws.networking:apigateway_get_gateway_response]{get_gateway_response} \tab Gets a GatewayResponse of a specified response type on the given RestApi\cr
#'  \link[paws.networking:apigateway_get_gateway_responses]{get_gateway_responses} \tab Gets the GatewayResponses collection on the given RestApi\cr
#'  \link[paws.networking:apigateway_get_integration]{get_integration} \tab Get the integration settings\cr
#'  \link[paws.networking:apigateway_get_integration_response]{get_integration_response} \tab Represents a get integration response\cr
#'  \link[paws.networking:apigateway_get_method]{get_method} \tab Describe an existing Method resource\cr
#'  \link[paws.networking:apigateway_get_method_response]{get_method_response} \tab Describes a MethodResponse resource\cr
#'  \link[paws.networking:apigateway_get_model]{get_model} \tab Describes an existing model defined for a RestApi resource\cr
#'  \link[paws.networking:apigateway_get_models]{get_models} \tab Describes existing Models defined for a RestApi resource\cr
#'  \link[paws.networking:apigateway_get_model_template]{get_model_template} \tab Generates a sample mapping template that can be used to transform a payload into the structure of a model\cr
#'  \link[paws.networking:apigateway_get_request_validator]{get_request_validator} \tab Gets a RequestValidator of a given RestApi\cr
#'  \link[paws.networking:apigateway_get_request_validators]{get_request_validators} \tab Gets the RequestValidators collection of a given RestApi\cr
#'  \link[paws.networking:apigateway_get_resource]{get_resource} \tab Lists information about a resource\cr
#'  \link[paws.networking:apigateway_get_resources]{get_resources} \tab Lists information about a collection of Resource resources\cr
#'  \link[paws.networking:apigateway_get_rest_api]{get_rest_api} \tab Lists the RestApi resource in the collection\cr
#'  \link[paws.networking:apigateway_get_rest_apis]{get_rest_apis} \tab Lists the RestApis resources for your collection\cr
#'  \link[paws.networking:apigateway_get_sdk]{get_sdk} \tab Generates a client SDK for a RestApi and Stage\cr
#'  \link[paws.networking:apigateway_get_sdk_type]{get_sdk_type} \tab Gets an SDK type\cr
#'  \link[paws.networking:apigateway_get_sdk_types]{get_sdk_types} \tab Gets SDK types\cr
#'  \link[paws.networking:apigateway_get_stage]{get_stage} \tab Gets information about a Stage resource\cr
#'  \link[paws.networking:apigateway_get_stages]{get_stages} \tab Gets information about one or more Stage resources\cr
#'  \link[paws.networking:apigateway_get_tags]{get_tags} \tab Gets the Tags collection for a given resource\cr
#'  \link[paws.networking:apigateway_get_usage]{get_usage} \tab Gets the usage data of a usage plan in a specified time interval\cr
#'  \link[paws.networking:apigateway_get_usage_plan]{get_usage_plan} \tab Gets a usage plan of a given plan identifier\cr
#'  \link[paws.networking:apigateway_get_usage_plan_key]{get_usage_plan_key} \tab Gets a usage plan key of a given key identifier\cr
#'  \link[paws.networking:apigateway_get_usage_plan_keys]{get_usage_plan_keys} \tab Gets all the usage plan keys representing the API keys added to a specified usage plan\cr
#'  \link[paws.networking:apigateway_get_usage_plans]{get_usage_plans} \tab Gets all the usage plans of the caller's account\cr
#'  \link[paws.networking:apigateway_get_vpc_link]{get_vpc_link} \tab Gets a specified VPC link under the caller's account in a region\cr
#'  \link[paws.networking:apigateway_get_vpc_links]{get_vpc_links} \tab Gets the VpcLinks collection under the caller's account in a selected region\cr
#'  \link[paws.networking:apigateway_import_api_keys]{import_api_keys} \tab Import API keys from an external source, such as a CSV-formatted file\cr
#'  \link[paws.networking:apigateway_import_documentation_parts]{import_documentation_parts} \tab Imports documentation parts\cr
#'  \link[paws.networking:apigateway_import_rest_api]{import_rest_api} \tab A feature of the API Gateway control service for creating a new API from an external API definition file\cr
#'  \link[paws.networking:apigateway_put_gateway_response]{put_gateway_response} \tab Creates a customization of a GatewayResponse of a specified response type and status code on the given RestApi\cr
#'  \link[paws.networking:apigateway_put_integration]{put_integration} \tab Sets up a method's integration\cr
#'  \link[paws.networking:apigateway_put_integration_response]{put_integration_response} \tab Represents a put integration\cr
#'  \link[paws.networking:apigateway_put_method]{put_method} \tab Add a method to an existing Resource resource\cr
#'  \link[paws.networking:apigateway_put_method_response]{put_method_response} \tab Adds a MethodResponse to an existing Method resource\cr
#'  \link[paws.networking:apigateway_put_rest_api]{put_rest_api} \tab A feature of the API Gateway control service for updating an existing API with an input of external API definitions\cr
#'  \link[paws.networking:apigateway_tag_resource]{tag_resource} \tab Adds or updates a tag on a given resource\cr
#'  \link[paws.networking:apigateway_test_invoke_authorizer]{test_invoke_authorizer} \tab Simulate the execution of an Authorizer in your RestApi with headers, parameters, and an incoming request body\cr
#'  \link[paws.networking:apigateway_test_invoke_method]{test_invoke_method} \tab Simulate the invocation of a Method in your RestApi with headers, parameters, and an incoming request body\cr
#'  \link[paws.networking:apigateway_untag_resource]{untag_resource} \tab Removes a tag from a given resource\cr
#'  \link[paws.networking:apigateway_update_account]{update_account} \tab Changes information about the current Account resource\cr
#'  \link[paws.networking:apigateway_update_api_key]{update_api_key} \tab Changes information about an ApiKey resource\cr
#'  \link[paws.networking:apigateway_update_authorizer]{update_authorizer} \tab Updates an existing Authorizer resource\cr
#'  \link[paws.networking:apigateway_update_base_path_mapping]{update_base_path_mapping} \tab Changes information about the BasePathMapping resource\cr
#'  \link[paws.networking:apigateway_update_client_certificate]{update_client_certificate} \tab Changes information about an ClientCertificate resource\cr
#'  \link[paws.networking:apigateway_update_deployment]{update_deployment} \tab Changes information about a Deployment resource\cr
#'  \link[paws.networking:apigateway_update_documentation_part]{update_documentation_part} \tab Updates a documentation part\cr
#'  \link[paws.networking:apigateway_update_documentation_version]{update_documentation_version} \tab Updates a documentation version\cr
#'  \link[paws.networking:apigateway_update_domain_name]{update_domain_name} \tab Changes information about the DomainName resource\cr
#'  \link[paws.networking:apigateway_update_gateway_response]{update_gateway_response} \tab Updates a GatewayResponse of a specified response type on the given RestApi\cr
#'  \link[paws.networking:apigateway_update_integration]{update_integration} \tab Represents an update integration\cr
#'  \link[paws.networking:apigateway_update_integration_response]{update_integration_response} \tab Represents an update integration response\cr
#'  \link[paws.networking:apigateway_update_method]{update_method} \tab Updates an existing Method resource\cr
#'  \link[paws.networking:apigateway_update_method_response]{update_method_response} \tab Updates an existing MethodResponse resource\cr
#'  \link[paws.networking:apigateway_update_model]{update_model} \tab Changes information about a model\cr
#'  \link[paws.networking:apigateway_update_request_validator]{update_request_validator} \tab Updates a RequestValidator of a given RestApi\cr
#'  \link[paws.networking:apigateway_update_resource]{update_resource} \tab Changes information about a Resource resource\cr
#'  \link[paws.networking:apigateway_update_rest_api]{update_rest_api} \tab Changes information about the specified API\cr
#'  \link[paws.networking:apigateway_update_stage]{update_stage} \tab Changes information about a Stage resource\cr
#'  \link[paws.networking:apigateway_update_usage]{update_usage} \tab Grants a temporary extension to the remaining quota of a usage plan associated with a specified API key\cr
#'  \link[paws.networking:apigateway_update_usage_plan]{update_usage_plan} \tab Updates a usage plan of a given plan Id\cr
#'  \link[paws.networking:apigateway_update_vpc_link]{update_vpc_link} \tab Updates an existing VpcLink of a specified identifier
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname apigateway
#' @export
apigateway <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.networking::apigateway(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' AmazonApiGatewayManagementApi
#'
#' @description
#' The Amazon API Gateway Management API allows you to directly manage
#' runtime aspects of your deployed APIs. To use it, you must explicitly
#' set the SDK's endpoint to point to the endpoint of your deployed API.
#' The endpoint will be of the form
#' https://\{api-id\}.execute-api.\{region\}.amazonaws.com/\{stage\},
#' or will be the endpoint corresponding to your API's custom domain and
#' base path, if applicable.
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- apigatewaymanagementapi(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- apigatewaymanagementapi()
#' svc$delete_connection(
#'   Foo = 123
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.networking:apigatewaymanagementapi_delete_connection]{delete_connection} \tab Delete the connection with the provided id\cr
#'  \link[paws.networking:apigatewaymanagementapi_get_connection]{get_connection} \tab Get information about the connection with the provided id\cr
#'  \link[paws.networking:apigatewaymanagementapi_post_to_connection]{post_to_connection} \tab Sends the provided data to the specified connection
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname apigatewaymanagementapi
#' @export
apigatewaymanagementapi <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.networking::apigatewaymanagementapi(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' AmazonApiGatewayV2
#'
#' @description
#' Amazon API Gateway V2
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- apigatewayv2(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- apigatewayv2()
#' svc$create_api(
#'   Foo = 123
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.networking:apigatewayv2_create_api]{create_api} \tab Creates an Api resource\cr
#'  \link[paws.networking:apigatewayv2_create_api_mapping]{create_api_mapping} \tab Creates an API mapping\cr
#'  \link[paws.networking:apigatewayv2_create_authorizer]{create_authorizer} \tab Creates an Authorizer for an API\cr
#'  \link[paws.networking:apigatewayv2_create_deployment]{create_deployment} \tab Creates a Deployment for an API\cr
#'  \link[paws.networking:apigatewayv2_create_domain_name]{create_domain_name} \tab Creates a domain name\cr
#'  \link[paws.networking:apigatewayv2_create_integration]{create_integration} \tab Creates an Integration\cr
#'  \link[paws.networking:apigatewayv2_create_integration_response]{create_integration_response} \tab Creates an IntegrationResponses\cr
#'  \link[paws.networking:apigatewayv2_create_model]{create_model} \tab Creates a Model for an API\cr
#'  \link[paws.networking:apigatewayv2_create_route]{create_route} \tab Creates a Route for an API\cr
#'  \link[paws.networking:apigatewayv2_create_route_response]{create_route_response} \tab Creates a RouteResponse for a Route\cr
#'  \link[paws.networking:apigatewayv2_create_stage]{create_stage} \tab Creates a Stage for an API\cr
#'  \link[paws.networking:apigatewayv2_create_vpc_link]{create_vpc_link} \tab Creates a VPC link\cr
#'  \link[paws.networking:apigatewayv2_delete_access_log_settings]{delete_access_log_settings} \tab Deletes the AccessLogSettings for a Stage\cr
#'  \link[paws.networking:apigatewayv2_delete_api]{delete_api} \tab Deletes an Api resource\cr
#'  \link[paws.networking:apigatewayv2_delete_api_mapping]{delete_api_mapping} \tab Deletes an API mapping\cr
#'  \link[paws.networking:apigatewayv2_delete_authorizer]{delete_authorizer} \tab Deletes an Authorizer\cr
#'  \link[paws.networking:apigatewayv2_delete_cors_configuration]{delete_cors_configuration} \tab Deletes a CORS configuration\cr
#'  \link[paws.networking:apigatewayv2_delete_deployment]{delete_deployment} \tab Deletes a Deployment\cr
#'  \link[paws.networking:apigatewayv2_delete_domain_name]{delete_domain_name} \tab Deletes a domain name\cr
#'  \link[paws.networking:apigatewayv2_delete_integration]{delete_integration} \tab Deletes an Integration\cr
#'  \link[paws.networking:apigatewayv2_delete_integration_response]{delete_integration_response} \tab Deletes an IntegrationResponses\cr
#'  \link[paws.networking:apigatewayv2_delete_model]{delete_model} \tab Deletes a Model\cr
#'  \link[paws.networking:apigatewayv2_delete_route]{delete_route} \tab Deletes a Route\cr
#'  \link[paws.networking:apigatewayv2_delete_route_request_parameter]{delete_route_request_parameter} \tab Deletes a route request parameter\cr
#'  \link[paws.networking:apigatewayv2_delete_route_response]{delete_route_response} \tab Deletes a RouteResponse\cr
#'  \link[paws.networking:apigatewayv2_delete_route_settings]{delete_route_settings} \tab Deletes the RouteSettings for a stage\cr
#'  \link[paws.networking:apigatewayv2_delete_stage]{delete_stage} \tab Deletes a Stage\cr
#'  \link[paws.networking:apigatewayv2_delete_vpc_link]{delete_vpc_link} \tab Deletes a VPC link\cr
#'  \link[paws.networking:apigatewayv2_export_api]{export_api} \tab Export api\cr
#'  \link[paws.networking:apigatewayv2_get_api]{get_api} \tab Gets an Api resource\cr
#'  \link[paws.networking:apigatewayv2_get_api_mapping]{get_api_mapping} \tab Gets an API mapping\cr
#'  \link[paws.networking:apigatewayv2_get_api_mappings]{get_api_mappings} \tab Gets API mappings\cr
#'  \link[paws.networking:apigatewayv2_get_apis]{get_apis} \tab Gets a collection of Api resources\cr
#'  \link[paws.networking:apigatewayv2_get_authorizer]{get_authorizer} \tab Gets an Authorizer\cr
#'  \link[paws.networking:apigatewayv2_get_authorizers]{get_authorizers} \tab Gets the Authorizers for an API\cr
#'  \link[paws.networking:apigatewayv2_get_deployment]{get_deployment} \tab Gets a Deployment\cr
#'  \link[paws.networking:apigatewayv2_get_deployments]{get_deployments} \tab Gets the Deployments for an API\cr
#'  \link[paws.networking:apigatewayv2_get_domain_name]{get_domain_name} \tab Gets a domain name\cr
#'  \link[paws.networking:apigatewayv2_get_domain_names]{get_domain_names} \tab Gets the domain names for an AWS account\cr
#'  \link[paws.networking:apigatewayv2_get_integration]{get_integration} \tab Gets an Integration\cr
#'  \link[paws.networking:apigatewayv2_get_integration_response]{get_integration_response} \tab Gets an IntegrationResponses\cr
#'  \link[paws.networking:apigatewayv2_get_integration_responses]{get_integration_responses} \tab Gets the IntegrationResponses for an Integration\cr
#'  \link[paws.networking:apigatewayv2_get_integrations]{get_integrations} \tab Gets the Integrations for an API\cr
#'  \link[paws.networking:apigatewayv2_get_model]{get_model} \tab Gets a Model\cr
#'  \link[paws.networking:apigatewayv2_get_models]{get_models} \tab Gets the Models for an API\cr
#'  \link[paws.networking:apigatewayv2_get_model_template]{get_model_template} \tab Gets a model template\cr
#'  \link[paws.networking:apigatewayv2_get_route]{get_route} \tab Gets a Route\cr
#'  \link[paws.networking:apigatewayv2_get_route_response]{get_route_response} \tab Gets a RouteResponse\cr
#'  \link[paws.networking:apigatewayv2_get_route_responses]{get_route_responses} \tab Gets the RouteResponses for a Route\cr
#'  \link[paws.networking:apigatewayv2_get_routes]{get_routes} \tab Gets the Routes for an API\cr
#'  \link[paws.networking:apigatewayv2_get_stage]{get_stage} \tab Gets a Stage\cr
#'  \link[paws.networking:apigatewayv2_get_stages]{get_stages} \tab Gets the Stages for an API\cr
#'  \link[paws.networking:apigatewayv2_get_tags]{get_tags} \tab Gets a collection of Tag resources\cr
#'  \link[paws.networking:apigatewayv2_get_vpc_link]{get_vpc_link} \tab Gets a VPC link\cr
#'  \link[paws.networking:apigatewayv2_get_vpc_links]{get_vpc_links} \tab Gets a collection of VPC links\cr
#'  \link[paws.networking:apigatewayv2_import_api]{import_api} \tab Imports an API\cr
#'  \link[paws.networking:apigatewayv2_reimport_api]{reimport_api} \tab Puts an Api resource\cr
#'  \link[paws.networking:apigatewayv2_reset_authorizers_cache]{reset_authorizers_cache} \tab Resets all authorizer cache entries on a stage\cr
#'  \link[paws.networking:apigatewayv2_tag_resource]{tag_resource} \tab Creates a new Tag resource to represent a tag\cr
#'  \link[paws.networking:apigatewayv2_untag_resource]{untag_resource} \tab Deletes a Tag\cr
#'  \link[paws.networking:apigatewayv2_update_api]{update_api} \tab Updates an Api resource\cr
#'  \link[paws.networking:apigatewayv2_update_api_mapping]{update_api_mapping} \tab The API mapping\cr
#'  \link[paws.networking:apigatewayv2_update_authorizer]{update_authorizer} \tab Updates an Authorizer\cr
#'  \link[paws.networking:apigatewayv2_update_deployment]{update_deployment} \tab Updates a Deployment\cr
#'  \link[paws.networking:apigatewayv2_update_domain_name]{update_domain_name} \tab Updates a domain name\cr
#'  \link[paws.networking:apigatewayv2_update_integration]{update_integration} \tab Updates an Integration\cr
#'  \link[paws.networking:apigatewayv2_update_integration_response]{update_integration_response} \tab Updates an IntegrationResponses\cr
#'  \link[paws.networking:apigatewayv2_update_model]{update_model} \tab Updates a Model\cr
#'  \link[paws.networking:apigatewayv2_update_route]{update_route} \tab Updates a Route\cr
#'  \link[paws.networking:apigatewayv2_update_route_response]{update_route_response} \tab Updates a RouteResponse\cr
#'  \link[paws.networking:apigatewayv2_update_stage]{update_stage} \tab Updates a Stage\cr
#'  \link[paws.networking:apigatewayv2_update_vpc_link]{update_vpc_link} \tab Updates a VPC link
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname apigatewayv2
#' @export
apigatewayv2 <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.networking::apigatewayv2(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' AWS App Mesh
#'
#' @description
#' App Mesh is a service mesh based on the Envoy proxy that makes it easy
#' to monitor and control microservices. App Mesh standardizes how your
#' microservices communicate, giving you end-to-end visibility and helping
#' to ensure high availability for your applications.
#'
#' App Mesh gives you consistent visibility and network traffic controls
#' for every microservice in an application. You can use App Mesh with
#' Amazon Web Services Fargate, Amazon ECS, Amazon EKS, Kubernetes on
#' Amazon Web Services, and Amazon EC2.
#'
#' App Mesh supports microservice applications that use service discovery
#' naming for their components. For more information about service
#' discovery on Amazon ECS, see [Service
#' Discovery](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-discovery.html)
#' in the *Amazon Elastic Container Service Developer Guide*. Kubernetes
#' `kube-dns` and `coredns` are supported. For more information, see [DNS
#' for Services and
#' Pods](https://kubernetes.io/docs/concepts/services-networking/dns-pod-service/)
#' in the Kubernetes documentation.
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- appmesh(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- appmesh()
#' svc$create_gateway_route(
#'   Foo = 123
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.networking:appmesh_create_gateway_route]{create_gateway_route} \tab Creates a gateway route\cr
#'  \link[paws.networking:appmesh_create_mesh]{create_mesh} \tab Creates a service mesh\cr
#'  \link[paws.networking:appmesh_create_route]{create_route} \tab Creates a route that is associated with a virtual router\cr
#'  \link[paws.networking:appmesh_create_virtual_gateway]{create_virtual_gateway} \tab Creates a virtual gateway\cr
#'  \link[paws.networking:appmesh_create_virtual_node]{create_virtual_node} \tab Creates a virtual node within a service mesh\cr
#'  \link[paws.networking:appmesh_create_virtual_router]{create_virtual_router} \tab Creates a virtual router within a service mesh\cr
#'  \link[paws.networking:appmesh_create_virtual_service]{create_virtual_service} \tab Creates a virtual service within a service mesh\cr
#'  \link[paws.networking:appmesh_delete_gateway_route]{delete_gateway_route} \tab Deletes an existing gateway route\cr
#'  \link[paws.networking:appmesh_delete_mesh]{delete_mesh} \tab Deletes an existing service mesh\cr
#'  \link[paws.networking:appmesh_delete_route]{delete_route} \tab Deletes an existing route\cr
#'  \link[paws.networking:appmesh_delete_virtual_gateway]{delete_virtual_gateway} \tab Deletes an existing virtual gateway\cr
#'  \link[paws.networking:appmesh_delete_virtual_node]{delete_virtual_node} \tab Deletes an existing virtual node\cr
#'  \link[paws.networking:appmesh_delete_virtual_router]{delete_virtual_router} \tab Deletes an existing virtual router\cr
#'  \link[paws.networking:appmesh_delete_virtual_service]{delete_virtual_service} \tab Deletes an existing virtual service\cr
#'  \link[paws.networking:appmesh_describe_gateway_route]{describe_gateway_route} \tab Describes an existing gateway route\cr
#'  \link[paws.networking:appmesh_describe_mesh]{describe_mesh} \tab Describes an existing service mesh\cr
#'  \link[paws.networking:appmesh_describe_route]{describe_route} \tab Describes an existing route\cr
#'  \link[paws.networking:appmesh_describe_virtual_gateway]{describe_virtual_gateway} \tab Describes an existing virtual gateway\cr
#'  \link[paws.networking:appmesh_describe_virtual_node]{describe_virtual_node} \tab Describes an existing virtual node\cr
#'  \link[paws.networking:appmesh_describe_virtual_router]{describe_virtual_router} \tab Describes an existing virtual router\cr
#'  \link[paws.networking:appmesh_describe_virtual_service]{describe_virtual_service} \tab Describes an existing virtual service\cr
#'  \link[paws.networking:appmesh_list_gateway_routes]{list_gateway_routes} \tab Returns a list of existing gateway routes that are associated to a virtual gateway\cr
#'  \link[paws.networking:appmesh_list_meshes]{list_meshes} \tab Returns a list of existing service meshes\cr
#'  \link[paws.networking:appmesh_list_routes]{list_routes} \tab Returns a list of existing routes in a service mesh\cr
#'  \link[paws.networking:appmesh_list_tags_for_resource]{list_tags_for_resource} \tab List the tags for an App Mesh resource\cr
#'  \link[paws.networking:appmesh_list_virtual_gateways]{list_virtual_gateways} \tab Returns a list of existing virtual gateways in a service mesh\cr
#'  \link[paws.networking:appmesh_list_virtual_nodes]{list_virtual_nodes} \tab Returns a list of existing virtual nodes\cr
#'  \link[paws.networking:appmesh_list_virtual_routers]{list_virtual_routers} \tab Returns a list of existing virtual routers in a service mesh\cr
#'  \link[paws.networking:appmesh_list_virtual_services]{list_virtual_services} \tab Returns a list of existing virtual services in a service mesh\cr
#'  \link[paws.networking:appmesh_tag_resource]{tag_resource} \tab Associates the specified tags to a resource with the specified resourceArn\cr
#'  \link[paws.networking:appmesh_untag_resource]{untag_resource} \tab Deletes specified tags from a resource\cr
#'  \link[paws.networking:appmesh_update_gateway_route]{update_gateway_route} \tab Updates an existing gateway route that is associated to a specified virtual gateway in a service mesh\cr
#'  \link[paws.networking:appmesh_update_mesh]{update_mesh} \tab Updates an existing service mesh\cr
#'  \link[paws.networking:appmesh_update_route]{update_route} \tab Updates an existing route for a specified service mesh and virtual router\cr
#'  \link[paws.networking:appmesh_update_virtual_gateway]{update_virtual_gateway} \tab Updates an existing virtual gateway in a specified service mesh\cr
#'  \link[paws.networking:appmesh_update_virtual_node]{update_virtual_node} \tab Updates an existing virtual node in a specified service mesh\cr
#'  \link[paws.networking:appmesh_update_virtual_router]{update_virtual_router} \tab Updates an existing virtual router in a specified service mesh\cr
#'  \link[paws.networking:appmesh_update_virtual_service]{update_virtual_service} \tab Updates an existing virtual service in a specified service mesh
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname appmesh
#' @export
appmesh <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.networking::appmesh(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' AWS Backup Gateway
#'
#' @description
#' Backup gateway
#'
#' Backup gateway connects Backup to your hypervisor, so you can create,
#' store, and restore backups of your virtual machines (VMs) anywhere,
#' whether on-premises or in the VMware Cloud (VMC) on Amazon Web Services.
#'
#' Add on-premises resources by connecting to a hypervisor through a
#' gateway. Backup will automatically discover the resources in your
#' hypervisor.
#'
#' Use Backup to assign virtual or on-premises resources to a backup plan,
#' or run on-demand backups. Once you have backed up your resources, you
#' can view them and restore them like any resource supported by Backup.
#'
#' To download the Amazon Web Services software to get started, navigate to
#' the Backup console, choose **Gateways**, then choose **Create gateway**.
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- backupgateway(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- backupgateway()
#' svc$associate_gateway_to_server(
#'   Foo = 123
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.networking:backupgateway_associate_gateway_to_server]{associate_gateway_to_server} \tab Associates a backup gateway with your server\cr
#'  \link[paws.networking:backupgateway_create_gateway]{create_gateway} \tab Creates a backup gateway\cr
#'  \link[paws.networking:backupgateway_delete_gateway]{delete_gateway} \tab Deletes a backup gateway\cr
#'  \link[paws.networking:backupgateway_delete_hypervisor]{delete_hypervisor} \tab Deletes a hypervisor\cr
#'  \link[paws.networking:backupgateway_disassociate_gateway_from_server]{disassociate_gateway_from_server} \tab Disassociates a backup gateway from the specified server\cr
#'  \link[paws.networking:backupgateway_get_bandwidth_rate_limit_schedule]{get_bandwidth_rate_limit_schedule} \tab Retrieves the bandwidth rate limit schedule for a specified gateway\cr
#'  \link[paws.networking:backupgateway_get_gateway]{get_gateway} \tab By providing the ARN (Amazon Resource Name), this API returns the gateway\cr
#'  \link[paws.networking:backupgateway_get_hypervisor]{get_hypervisor} \tab This action requests information about the specified hypervisor to which the gateway will connect\cr
#'  \link[paws.networking:backupgateway_get_hypervisor_property_mappings]{get_hypervisor_property_mappings} \tab This action retrieves the property mappings for the specified hypervisor\cr
#'  \link[paws.networking:backupgateway_get_virtual_machine]{get_virtual_machine} \tab By providing the ARN (Amazon Resource Name), this API returns the virtual machine\cr
#'  \link[paws.networking:backupgateway_import_hypervisor_configuration]{import_hypervisor_configuration} \tab Connect to a hypervisor by importing its configuration\cr
#'  \link[paws.networking:backupgateway_list_gateways]{list_gateways} \tab Lists backup gateways owned by an Amazon Web Services account in an Amazon Web Services Region\cr
#'  \link[paws.networking:backupgateway_list_hypervisors]{list_hypervisors} \tab Lists your hypervisors\cr
#'  \link[paws.networking:backupgateway_list_tags_for_resource]{list_tags_for_resource} \tab Lists the tags applied to the resource identified by its Amazon Resource Name (ARN)\cr
#'  \link[paws.networking:backupgateway_list_virtual_machines]{list_virtual_machines} \tab Lists your virtual machines\cr
#'  \link[paws.networking:backupgateway_put_bandwidth_rate_limit_schedule]{put_bandwidth_rate_limit_schedule} \tab This action sets the bandwidth rate limit schedule for a specified gateway\cr
#'  \link[paws.networking:backupgateway_put_hypervisor_property_mappings]{put_hypervisor_property_mappings} \tab This action sets the property mappings for the specified hypervisor\cr
#'  \link[paws.networking:backupgateway_put_maintenance_start_time]{put_maintenance_start_time} \tab Set the maintenance start time for a gateway\cr
#'  \link[paws.networking:backupgateway_start_virtual_machines_metadata_sync]{start_virtual_machines_metadata_sync} \tab This action sends a request to sync metadata across the specified virtual machines\cr
#'  \link[paws.networking:backupgateway_tag_resource]{tag_resource} \tab Tag the resource\cr
#'  \link[paws.networking:backupgateway_test_hypervisor_configuration]{test_hypervisor_configuration} \tab Tests your hypervisor configuration to validate that backup gateway can connect with the hypervisor and its resources\cr
#'  \link[paws.networking:backupgateway_untag_resource]{untag_resource} \tab Removes tags from the resource\cr
#'  \link[paws.networking:backupgateway_update_gateway_information]{update_gateway_information} \tab Updates a gateway's name\cr
#'  \link[paws.networking:backupgateway_update_gateway_software_now]{update_gateway_software_now} \tab Updates the gateway virtual machine (VM) software\cr
#'  \link[paws.networking:backupgateway_update_hypervisor]{update_hypervisor} \tab Updates a hypervisor metadata, including its host, username, and password
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname backupgateway
#' @export
backupgateway <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.networking::backupgateway(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' Amazon CloudFront
#'
#' @description
#' This is the *Amazon CloudFront API Reference*. This guide is for
#' developers who need detailed information about CloudFront API actions,
#' data types, and errors. For detailed information about CloudFront
#' features, see the *Amazon CloudFront Developer Guide*.
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- cloudfront(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- cloudfront()
#' svc$associate_alias(
#'   Foo = 123
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.networking:cloudfront_associate_alias]{associate_alias} \tab Associates an alias (also known as a CNAME or an alternate domain name) with a CloudFront distribution\cr
#'  \link[paws.networking:cloudfront_copy_distribution]{copy_distribution} \tab Creates a staging distribution using the configuration of the provided primary distribution\cr
#'  \link[paws.networking:cloudfront_create_cache_policy]{create_cache_policy} \tab Creates a cache policy\cr
#'  \link[paws.networking:cloudfront_create_cloud_front_origin_access_identity]{create_cloud_front_origin_access_identity} \tab Creates a new origin access identity\cr
#'  \link[paws.networking:cloudfront_create_continuous_deployment_policy]{create_continuous_deployment_policy} \tab Creates a continuous deployment policy that distributes traffic for a custom domain name to two different CloudFront distributions\cr
#'  \link[paws.networking:cloudfront_create_distribution]{create_distribution} \tab Creates a CloudFront distribution\cr
#'  \link[paws.networking:cloudfront_create_distribution_with_tags]{create_distribution_with_tags} \tab Create a new distribution with tags\cr
#'  \link[paws.networking:cloudfront_create_field_level_encryption_config]{create_field_level_encryption_config} \tab Create a new field-level encryption configuration\cr
#'  \link[paws.networking:cloudfront_create_field_level_encryption_profile]{create_field_level_encryption_profile} \tab Create a field-level encryption profile\cr
#'  \link[paws.networking:cloudfront_create_function]{create_function} \tab Creates a CloudFront function\cr
#'  \link[paws.networking:cloudfront_create_invalidation]{create_invalidation} \tab Create a new invalidation\cr
#'  \link[paws.networking:cloudfront_create_key_group]{create_key_group} \tab Creates a key group that you can use with CloudFront signed URLs and signed cookies\cr
#'  \link[paws.networking:cloudfront_create_monitoring_subscription]{create_monitoring_subscription} \tab Enables additional CloudWatch metrics for the specified CloudFront distribution\cr
#'  \link[paws.networking:cloudfront_create_origin_access_control]{create_origin_access_control} \tab Creates a new origin access control in CloudFront\cr
#'  \link[paws.networking:cloudfront_create_origin_request_policy]{create_origin_request_policy} \tab Creates an origin request policy\cr
#'  \link[paws.networking:cloudfront_create_public_key]{create_public_key} \tab Uploads a public key to CloudFront that you can use with signed URLs and signed cookies, or with field-level encryption\cr
#'  \link[paws.networking:cloudfront_create_realtime_log_config]{create_realtime_log_config} \tab Creates a real-time log configuration\cr
#'  \link[paws.networking:cloudfront_create_response_headers_policy]{create_response_headers_policy} \tab Creates a response headers policy\cr
#'  \link[paws.networking:cloudfront_create_streaming_distribution]{create_streaming_distribution} \tab This API is deprecated\cr
#'  \link[paws.networking:cloudfront_create_streaming_distribution_with_tags]{create_streaming_distribution_with_tags} \tab This API is deprecated\cr
#'  \link[paws.networking:cloudfront_delete_cache_policy]{delete_cache_policy} \tab Deletes a cache policy\cr
#'  \link[paws.networking:cloudfront_delete_cloud_front_origin_access_identity]{delete_cloud_front_origin_access_identity} \tab Delete an origin access identity\cr
#'  \link[paws.networking:cloudfront_delete_continuous_deployment_policy]{delete_continuous_deployment_policy} \tab Deletes a continuous deployment policy\cr
#'  \link[paws.networking:cloudfront_delete_distribution]{delete_distribution} \tab Delete a distribution\cr
#'  \link[paws.networking:cloudfront_delete_field_level_encryption_config]{delete_field_level_encryption_config} \tab Remove a field-level encryption configuration\cr
#'  \link[paws.networking:cloudfront_delete_field_level_encryption_profile]{delete_field_level_encryption_profile} \tab Remove a field-level encryption profile\cr
#'  \link[paws.networking:cloudfront_delete_function]{delete_function} \tab Deletes a CloudFront function\cr
#'  \link[paws.networking:cloudfront_delete_key_group]{delete_key_group} \tab Deletes a key group\cr
#'  \link[paws.networking:cloudfront_delete_monitoring_subscription]{delete_monitoring_subscription} \tab Disables additional CloudWatch metrics for the specified CloudFront distribution\cr
#'  \link[paws.networking:cloudfront_delete_origin_access_control]{delete_origin_access_control} \tab Deletes a CloudFront origin access control\cr
#'  \link[paws.networking:cloudfront_delete_origin_request_policy]{delete_origin_request_policy} \tab Deletes an origin request policy\cr
#'  \link[paws.networking:cloudfront_delete_public_key]{delete_public_key} \tab Remove a public key you previously added to CloudFront\cr
#'  \link[paws.networking:cloudfront_delete_realtime_log_config]{delete_realtime_log_config} \tab Deletes a real-time log configuration\cr
#'  \link[paws.networking:cloudfront_delete_response_headers_policy]{delete_response_headers_policy} \tab Deletes a response headers policy\cr
#'  \link[paws.networking:cloudfront_delete_streaming_distribution]{delete_streaming_distribution} \tab Delete a streaming distribution\cr
#'  \link[paws.networking:cloudfront_describe_function]{describe_function} \tab Gets configuration information and metadata about a CloudFront function, but not the function's code\cr
#'  \link[paws.networking:cloudfront_get_cache_policy]{get_cache_policy} \tab Gets a cache policy, including the following metadata:\cr
#'  \link[paws.networking:cloudfront_get_cache_policy_config]{get_cache_policy_config} \tab Gets a cache policy configuration\cr
#'  \link[paws.networking:cloudfront_get_cloud_front_origin_access_identity]{get_cloud_front_origin_access_identity} \tab Get the information about an origin access identity\cr
#'  \link[paws.networking:cloudfront_get_cloud_front_origin_access_identity_config]{get_cloud_front_origin_access_identity_config} \tab Get the configuration information about an origin access identity\cr
#'  \link[paws.networking:cloudfront_get_continuous_deployment_policy]{get_continuous_deployment_policy} \tab Gets a continuous deployment policy, including metadata (the policy's identifier and the date and time when the policy was last modified)\cr
#'  \link[paws.networking:cloudfront_get_continuous_deployment_policy_config]{get_continuous_deployment_policy_config} \tab Gets configuration information about a continuous deployment policy\cr
#'  \link[paws.networking:cloudfront_get_distribution]{get_distribution} \tab Get the information about a distribution\cr
#'  \link[paws.networking:cloudfront_get_distribution_config]{get_distribution_config} \tab Get the configuration information about a distribution\cr
#'  \link[paws.networking:cloudfront_get_field_level_encryption]{get_field_level_encryption} \tab Get the field-level encryption configuration information\cr
#'  \link[paws.networking:cloudfront_get_field_level_encryption_config]{get_field_level_encryption_config} \tab Get the field-level encryption configuration information\cr
#'  \link[paws.networking:cloudfront_get_field_level_encryption_profile]{get_field_level_encryption_profile} \tab Get the field-level encryption profile information\cr
#'  \link[paws.networking:cloudfront_get_field_level_encryption_profile_config]{get_field_level_encryption_profile_config} \tab Get the field-level encryption profile configuration information\cr
#'  \link[paws.networking:cloudfront_get_function]{get_function} \tab Gets the code of a CloudFront function\cr
#'  \link[paws.networking:cloudfront_get_invalidation]{get_invalidation} \tab Get the information about an invalidation\cr
#'  \link[paws.networking:cloudfront_get_key_group]{get_key_group} \tab Gets a key group, including the date and time when the key group was last modified\cr
#'  \link[paws.networking:cloudfront_get_key_group_config]{get_key_group_config} \tab Gets a key group configuration\cr
#'  \link[paws.networking:cloudfront_get_monitoring_subscription]{get_monitoring_subscription} \tab Gets information about whether additional CloudWatch metrics are enabled for the specified CloudFront distribution\cr
#'  \link[paws.networking:cloudfront_get_origin_access_control]{get_origin_access_control} \tab Gets a CloudFront origin access control, including its unique identifier\cr
#'  \link[paws.networking:cloudfront_get_origin_access_control_config]{get_origin_access_control_config} \tab Gets a CloudFront origin access control configuration\cr
#'  \link[paws.networking:cloudfront_get_origin_request_policy]{get_origin_request_policy} \tab Gets an origin request policy, including the following metadata:\cr
#'  \link[paws.networking:cloudfront_get_origin_request_policy_config]{get_origin_request_policy_config} \tab Gets an origin request policy configuration\cr
#'  \link[paws.networking:cloudfront_get_public_key]{get_public_key} \tab Gets a public key\cr
#'  \link[paws.networking:cloudfront_get_public_key_config]{get_public_key_config} \tab Gets a public key configuration\cr
#'  \link[paws.networking:cloudfront_get_realtime_log_config]{get_realtime_log_config} \tab Gets a real-time log configuration\cr
#'  \link[paws.networking:cloudfront_get_response_headers_policy]{get_response_headers_policy} \tab Gets a response headers policy, including metadata (the policy's identifier and the date and time when the policy was last modified)\cr
#'  \link[paws.networking:cloudfront_get_response_headers_policy_config]{get_response_headers_policy_config} \tab Gets a response headers policy configuration\cr
#'  \link[paws.networking:cloudfront_get_streaming_distribution]{get_streaming_distribution} \tab Gets information about a specified RTMP distribution, including the distribution configuration\cr
#'  \link[paws.networking:cloudfront_get_streaming_distribution_config]{get_streaming_distribution_config} \tab Get the configuration information about a streaming distribution\cr
#'  \link[paws.networking:cloudfront_list_cache_policies]{list_cache_policies} \tab Gets a list of cache policies\cr
#'  \link[paws.networking:cloudfront_list_cloud_front_origin_access_identities]{list_cloud_front_origin_access_identities} \tab Lists origin access identities\cr
#'  \link[paws.networking:cloudfront_list_conflicting_aliases]{list_conflicting_aliases} \tab Gets a list of aliases (also called CNAMEs or alternate domain names) that conflict or overlap with the provided alias, and the associated CloudFront distributions and Amazon Web Services accounts for each conflicting alias\cr
#'  \link[paws.networking:cloudfront_list_continuous_deployment_policies]{list_continuous_deployment_policies} \tab Gets a list of the continuous deployment policies in your Amazon Web Services account\cr
#'  \link[paws.networking:cloudfront_list_distributions]{list_distributions} \tab List CloudFront distributions\cr
#'  \link[paws.networking:cloudfront_list_distributions_by_cache_policy_id]{list_distributions_by_cache_policy_id} \tab Gets a list of distribution IDs for distributions that have a cache behavior that's associated with the specified cache policy\cr
#'  \link[paws.networking:cloudfront_list_distributions_by_key_group]{list_distributions_by_key_group} \tab Gets a list of distribution IDs for distributions that have a cache behavior that references the specified key group\cr
#'  \link[paws.networking:cloudfront_list_distributions_by_origin_request_policy_id]{list_distributions_by_origin_request_policy_id} \tab Gets a list of distribution IDs for distributions that have a cache behavior that's associated with the specified origin request policy\cr
#'  \link[paws.networking:cloudfront_list_distributions_by_realtime_log_config]{list_distributions_by_realtime_log_config} \tab Gets a list of distributions that have a cache behavior that's associated with the specified real-time log configuration\cr
#'  \link[paws.networking:cloudfront_list_distributions_by_response_headers_policy_id]{list_distributions_by_response_headers_policy_id} \tab Gets a list of distribution IDs for distributions that have a cache behavior that's associated with the specified response headers policy\cr
#'  \link[paws.networking:cloudfront_list_distributions_by_web_acl_id]{list_distributions_by_web_acl_id} \tab List the distributions that are associated with a specified WAF web ACL\cr
#'  \link[paws.networking:cloudfront_list_field_level_encryption_configs]{list_field_level_encryption_configs} \tab List all field-level encryption configurations that have been created in CloudFront for this account\cr
#'  \link[paws.networking:cloudfront_list_field_level_encryption_profiles]{list_field_level_encryption_profiles} \tab Request a list of field-level encryption profiles that have been created in CloudFront for this account\cr
#'  \link[paws.networking:cloudfront_list_functions]{list_functions} \tab Gets a list of all CloudFront functions in your Amazon Web Services account\cr
#'  \link[paws.networking:cloudfront_list_invalidations]{list_invalidations} \tab Lists invalidation batches\cr
#'  \link[paws.networking:cloudfront_list_key_groups]{list_key_groups} \tab Gets a list of key groups\cr
#'  \link[paws.networking:cloudfront_list_origin_access_controls]{list_origin_access_controls} \tab Gets the list of CloudFront origin access controls in this Amazon Web Services account\cr
#'  \link[paws.networking:cloudfront_list_origin_request_policies]{list_origin_request_policies} \tab Gets a list of origin request policies\cr
#'  \link[paws.networking:cloudfront_list_public_keys]{list_public_keys} \tab List all public keys that have been added to CloudFront for this account\cr
#'  \link[paws.networking:cloudfront_list_realtime_log_configs]{list_realtime_log_configs} \tab Gets a list of real-time log configurations\cr
#'  \link[paws.networking:cloudfront_list_response_headers_policies]{list_response_headers_policies} \tab Gets a list of response headers policies\cr
#'  \link[paws.networking:cloudfront_list_streaming_distributions]{list_streaming_distributions} \tab List streaming distributions\cr
#'  \link[paws.networking:cloudfront_list_tags_for_resource]{list_tags_for_resource} \tab List tags for a CloudFront resource\cr
#'  \link[paws.networking:cloudfront_publish_function]{publish_function} \tab Publishes a CloudFront function by copying the function code from the DEVELOPMENT stage to LIVE\cr
#'  \link[paws.networking:cloudfront_tag_resource]{tag_resource} \tab Add tags to a CloudFront resource\cr
#'  \link[paws.networking:cloudfront_test_function]{test_function} \tab Tests a CloudFront function\cr
#'  \link[paws.networking:cloudfront_untag_resource]{untag_resource} \tab Remove tags from a CloudFront resource\cr
#'  \link[paws.networking:cloudfront_update_cache_policy]{update_cache_policy} \tab Updates a cache policy configuration\cr
#'  \link[paws.networking:cloudfront_update_cloud_front_origin_access_identity]{update_cloud_front_origin_access_identity} \tab Update an origin access identity\cr
#'  \link[paws.networking:cloudfront_update_continuous_deployment_policy]{update_continuous_deployment_policy} \tab Updates a continuous deployment policy\cr
#'  \link[paws.networking:cloudfront_update_distribution]{update_distribution} \tab Updates the configuration for a CloudFront distribution\cr
#'  \link[paws.networking:cloudfront_update_distribution_with_staging_config]{update_distribution_with_staging_config} \tab Copies the staging distribution's configuration to its corresponding primary distribution\cr
#'  \link[paws.networking:cloudfront_update_field_level_encryption_config]{update_field_level_encryption_config} \tab Update a field-level encryption configuration\cr
#'  \link[paws.networking:cloudfront_update_field_level_encryption_profile]{update_field_level_encryption_profile} \tab Update a field-level encryption profile\cr
#'  \link[paws.networking:cloudfront_update_function]{update_function} \tab Updates a CloudFront function\cr
#'  \link[paws.networking:cloudfront_update_key_group]{update_key_group} \tab Updates a key group\cr
#'  \link[paws.networking:cloudfront_update_origin_access_control]{update_origin_access_control} \tab Updates a CloudFront origin access control\cr
#'  \link[paws.networking:cloudfront_update_origin_request_policy]{update_origin_request_policy} \tab Updates an origin request policy configuration\cr
#'  \link[paws.networking:cloudfront_update_public_key]{update_public_key} \tab Update public key information\cr
#'  \link[paws.networking:cloudfront_update_realtime_log_config]{update_realtime_log_config} \tab Updates a real-time log configuration\cr
#'  \link[paws.networking:cloudfront_update_response_headers_policy]{update_response_headers_policy} \tab Updates a response headers policy\cr
#'  \link[paws.networking:cloudfront_update_streaming_distribution]{update_streaming_distribution} \tab Update a streaming distribution
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname cloudfront
#' @export
cloudfront <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.networking::cloudfront(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' AWS Direct Connect
#'
#' @description
#' Direct Connect links your internal network to an Direct Connect location
#' over a standard Ethernet fiber-optic cable. One end of the cable is
#' connected to your router, the other to an Direct Connect router. With
#' this connection in place, you can create virtual interfaces directly to
#' the Amazon Web Services Cloud (for example, to Amazon EC2 and Amazon S3)
#' and to Amazon VPC, bypassing Internet service providers in your network
#' path. A connection provides access to all Amazon Web Services Regions
#' except the China (Beijing) and (China) Ningxia Regions. Amazon Web
#' Services resources in the China Regions can only be accessed through
#' locations associated with those Regions.
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- directconnect(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- directconnect()
#' svc$accept_direct_connect_gateway_association_proposal(
#'   Foo = 123
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.networking:directconnect_accept_direct_connect_gateway_association_proposal]{accept_direct_connect_gateway_association_proposal} \tab Accepts a proposal request to attach a virtual private gateway or transit gateway to a Direct Connect gateway\cr
#'  \link[paws.networking:directconnect_allocate_connection_on_interconnect]{allocate_connection_on_interconnect} \tab Deprecated\cr
#'  \link[paws.networking:directconnect_allocate_hosted_connection]{allocate_hosted_connection} \tab Creates a hosted connection on the specified interconnect or a link aggregation group (LAG) of interconnects\cr
#'  \link[paws.networking:directconnect_allocate_private_virtual_interface]{allocate_private_virtual_interface} \tab Provisions a private virtual interface to be owned by the specified Amazon Web Services account\cr
#'  \link[paws.networking:directconnect_allocate_public_virtual_interface]{allocate_public_virtual_interface} \tab Provisions a public virtual interface to be owned by the specified Amazon Web Services account\cr
#'  \link[paws.networking:directconnect_allocate_transit_virtual_interface]{allocate_transit_virtual_interface} \tab Provisions a transit virtual interface to be owned by the specified Amazon Web Services account\cr
#'  \link[paws.networking:directconnect_associate_connection_with_lag]{associate_connection_with_lag} \tab Associates an existing connection with a link aggregation group (LAG)\cr
#'  \link[paws.networking:directconnect_associate_hosted_connection]{associate_hosted_connection} \tab Associates a hosted connection and its virtual interfaces with a link aggregation group (LAG) or interconnect\cr
#'  \link[paws.networking:directconnect_associate_mac_sec_key]{associate_mac_sec_key} \tab Associates a MAC Security (MACsec) Connection Key Name (CKN)/ Connectivity Association Key (CAK) pair with an Direct Connect dedicated connection\cr
#'  \link[paws.networking:directconnect_associate_virtual_interface]{associate_virtual_interface} \tab Associates a virtual interface with a specified link aggregation group (LAG) or connection\cr
#'  \link[paws.networking:directconnect_confirm_connection]{confirm_connection} \tab Confirms the creation of the specified hosted connection on an interconnect\cr
#'  \link[paws.networking:directconnect_confirm_customer_agreement]{confirm_customer_agreement} \tab The confirmation of the terms of agreement when creating the connection/link aggregation group (LAG)\cr
#'  \link[paws.networking:directconnect_confirm_private_virtual_interface]{confirm_private_virtual_interface} \tab Accepts ownership of a private virtual interface created by another Amazon Web Services account\cr
#'  \link[paws.networking:directconnect_confirm_public_virtual_interface]{confirm_public_virtual_interface} \tab Accepts ownership of a public virtual interface created by another Amazon Web Services account\cr
#'  \link[paws.networking:directconnect_confirm_transit_virtual_interface]{confirm_transit_virtual_interface} \tab Accepts ownership of a transit virtual interface created by another Amazon Web Services account\cr
#'  \link[paws.networking:directconnect_create_bgp_peer]{create_bgp_peer} \tab Creates a BGP peer on the specified virtual interface\cr
#'  \link[paws.networking:directconnect_create_connection]{create_connection} \tab Creates a connection between a customer network and a specific Direct Connect location\cr
#'  \link[paws.networking:directconnect_create_direct_connect_gateway]{create_direct_connect_gateway} \tab Creates a Direct Connect gateway, which is an intermediate object that enables you to connect a set of virtual interfaces and virtual private gateways\cr
#'  \link[paws.networking:directconnect_create_direct_connect_gateway_association]{create_direct_connect_gateway_association} \tab Creates an association between a Direct Connect gateway and a virtual private gateway\cr
#'  \link[paws.networking:directconnect_create_direct_connect_gateway_association_proposal]{create_direct_connect_gateway_association_proposal} \tab Creates a proposal to associate the specified virtual private gateway or transit gateway with the specified Direct Connect gateway\cr
#'  \link[paws.networking:directconnect_create_interconnect]{create_interconnect} \tab Creates an interconnect between an Direct Connect Partner's network and a specific Direct Connect location\cr
#'  \link[paws.networking:directconnect_create_lag]{create_lag} \tab Creates a link aggregation group (LAG) with the specified number of bundled physical dedicated connections between the customer network and a specific Direct Connect location\cr
#'  \link[paws.networking:directconnect_create_private_virtual_interface]{create_private_virtual_interface} \tab Creates a private virtual interface\cr
#'  \link[paws.networking:directconnect_create_public_virtual_interface]{create_public_virtual_interface} \tab Creates a public virtual interface\cr
#'  \link[paws.networking:directconnect_create_transit_virtual_interface]{create_transit_virtual_interface} \tab Creates a transit virtual interface\cr
#'  \link[paws.networking:directconnect_delete_bgp_peer]{delete_bgp_peer} \tab Deletes the specified BGP peer on the specified virtual interface with the specified customer address and ASN\cr
#'  \link[paws.networking:directconnect_delete_connection]{delete_connection} \tab Deletes the specified connection\cr
#'  \link[paws.networking:directconnect_delete_direct_connect_gateway]{delete_direct_connect_gateway} \tab Deletes the specified Direct Connect gateway\cr
#'  \link[paws.networking:directconnect_delete_direct_connect_gateway_association]{delete_direct_connect_gateway_association} \tab Deletes the association between the specified Direct Connect gateway and virtual private gateway\cr
#'  \link[paws.networking:directconnect_delete_direct_connect_gateway_association_proposal]{delete_direct_connect_gateway_association_proposal} \tab Deletes the association proposal request between the specified Direct Connect gateway and virtual private gateway or transit gateway\cr
#'  \link[paws.networking:directconnect_delete_interconnect]{delete_interconnect} \tab Deletes the specified interconnect\cr
#'  \link[paws.networking:directconnect_delete_lag]{delete_lag} \tab Deletes the specified link aggregation group (LAG)\cr
#'  \link[paws.networking:directconnect_delete_virtual_interface]{delete_virtual_interface} \tab Deletes a virtual interface\cr
#'  \link[paws.networking:directconnect_describe_connection_loa]{describe_connection_loa} \tab Deprecated\cr
#'  \link[paws.networking:directconnect_describe_connections]{describe_connections} \tab Displays the specified connection or all connections in this Region\cr
#'  \link[paws.networking:directconnect_describe_connections_on_interconnect]{describe_connections_on_interconnect} \tab Deprecated\cr
#'  \link[paws.networking:directconnect_describe_customer_metadata]{describe_customer_metadata} \tab Get and view a list of customer agreements, along with their signed status and whether the customer is an NNIPartner, NNIPartnerV2, or a nonPartner\cr
#'  \link[paws.networking:directconnect_describe_direct_connect_gateway_association_proposals]{describe_direct_connect_gateway_association_proposals} \tab Describes one or more association proposals for connection between a virtual private gateway or transit gateway and a Direct Connect gateway\cr
#'  \link[paws.networking:directconnect_describe_direct_connect_gateway_associations]{describe_direct_connect_gateway_associations} \tab Lists the associations between your Direct Connect gateways and virtual private gateways and transit gateways\cr
#'  \link[paws.networking:directconnect_describe_direct_connect_gateway_attachments]{describe_direct_connect_gateway_attachments} \tab Lists the attachments between your Direct Connect gateways and virtual interfaces\cr
#'  \link[paws.networking:directconnect_describe_direct_connect_gateways]{describe_direct_connect_gateways} \tab Lists all your Direct Connect gateways or only the specified Direct Connect gateway\cr
#'  \link[paws.networking:directconnect_describe_hosted_connections]{describe_hosted_connections} \tab Lists the hosted connections that have been provisioned on the specified interconnect or link aggregation group (LAG)\cr
#'  \link[paws.networking:directconnect_describe_interconnect_loa]{describe_interconnect_loa} \tab Deprecated\cr
#'  \link[paws.networking:directconnect_describe_interconnects]{describe_interconnects} \tab Lists the interconnects owned by the Amazon Web Services account or only the specified interconnect\cr
#'  \link[paws.networking:directconnect_describe_lags]{describe_lags} \tab Describes all your link aggregation groups (LAG) or the specified LAG\cr
#'  \link[paws.networking:directconnect_describe_loa]{describe_loa} \tab Gets the LOA-CFA for a connection, interconnect, or link aggregation group (LAG)\cr
#'  \link[paws.networking:directconnect_describe_locations]{describe_locations} \tab Lists the Direct Connect locations in the current Amazon Web Services Region\cr
#'  \link[paws.networking:directconnect_describe_router_configuration]{describe_router_configuration} \tab Details about the router\cr
#'  \link[paws.networking:directconnect_describe_tags]{describe_tags} \tab Describes the tags associated with the specified Direct Connect resources\cr
#'  \link[paws.networking:directconnect_describe_virtual_gateways]{describe_virtual_gateways} \tab Lists the virtual private gateways owned by the Amazon Web Services account\cr
#'  \link[paws.networking:directconnect_describe_virtual_interfaces]{describe_virtual_interfaces} \tab Displays all virtual interfaces for an Amazon Web Services account\cr
#'  \link[paws.networking:directconnect_disassociate_connection_from_lag]{disassociate_connection_from_lag} \tab Disassociates a connection from a link aggregation group (LAG)\cr
#'  \link[paws.networking:directconnect_disassociate_mac_sec_key]{disassociate_mac_sec_key} \tab Removes the association between a MAC Security (MACsec) security key and an Direct Connect dedicated connection\cr
#'  \link[paws.networking:directconnect_list_virtual_interface_test_history]{list_virtual_interface_test_history} \tab Lists the virtual interface failover test history\cr
#'  \link[paws.networking:directconnect_start_bgp_failover_test]{start_bgp_failover_test} \tab Starts the virtual interface failover test that verifies your configuration meets your resiliency requirements by placing the BGP peering session in the DOWN state\cr
#'  \link[paws.networking:directconnect_stop_bgp_failover_test]{stop_bgp_failover_test} \tab Stops the virtual interface failover test\cr
#'  \link[paws.networking:directconnect_tag_resource]{tag_resource} \tab Adds the specified tags to the specified Direct Connect resource\cr
#'  \link[paws.networking:directconnect_untag_resource]{untag_resource} \tab Removes one or more tags from the specified Direct Connect resource\cr
#'  \link[paws.networking:directconnect_update_connection]{update_connection} \tab Updates the Direct Connect dedicated connection configuration\cr
#'  \link[paws.networking:directconnect_update_direct_connect_gateway]{update_direct_connect_gateway} \tab Updates the name of a current Direct Connect gateway\cr
#'  \link[paws.networking:directconnect_update_direct_connect_gateway_association]{update_direct_connect_gateway_association} \tab Updates the specified attributes of the Direct Connect gateway association\cr
#'  \link[paws.networking:directconnect_update_lag]{update_lag} \tab Updates the attributes of the specified link aggregation group (LAG)\cr
#'  \link[paws.networking:directconnect_update_virtual_interface_attributes]{update_virtual_interface_attributes} \tab Updates the specified attributes of the specified virtual private interface
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname directconnect
#' @export
directconnect <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.networking::directconnect(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' Elastic Load Balancing
#'
#' @description
#' A load balancer can distribute incoming traffic across your EC2
#' instances. This enables you to increase the availability of your
#' application. The load balancer also monitors the health of its
#' registered instances and ensures that it routes traffic only to healthy
#' instances. You configure your load balancer to accept incoming traffic
#' by specifying one or more listeners, which are configured with a
#' protocol and port number for connections from clients to the load
#' balancer and a protocol and port number for connections from the load
#' balancer to the instances.
#'
#' Elastic Load Balancing supports three types of load balancers:
#' Application Load Balancers, Network Load Balancers, and Classic Load
#' Balancers. You can select a load balancer based on your application
#' needs. For more information, see the [Elastic Load Balancing User
#' Guide](https://docs.aws.amazon.com/elasticloadbalancing/latest/userguide/).
#'
#' This reference covers the 2012-06-01 API, which supports Classic Load
#' Balancers. The 2015-12-01 API supports Application Load Balancers and
#' Network Load Balancers.
#'
#' To get started, create a load balancer with one or more listeners using
#' `create_load_balancer`. Register your
#' instances with the load balancer using
#' `register_instances_with_load_balancer`.
#'
#' All Elastic Load Balancing operations are *idempotent*, which means that
#' they complete at most one time. If you repeat an operation, it succeeds
#' with a 200 OK response code.
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- elb(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- elb()
#' # This example adds two tags to the specified load balancer.
#' svc$add_tags(
#'   LoadBalancerNames = list(
#'     "my-load-balancer"
#'   ),
#'   Tags = list(
#'     list(
#'       Key = "project",
#'       Value = "lima"
#'     ),
#'     list(
#'       Key = "department",
#'       Value = "digital-media"
#'     )
#'   )
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.networking:elb_add_tags]{add_tags} \tab Adds the specified tags to the specified load balancer\cr
#'  \link[paws.networking:elb_apply_security_groups_to_load_balancer]{apply_security_groups_to_load_balancer} \tab Associates one or more security groups with your load balancer in a virtual private cloud (VPC)\cr
#'  \link[paws.networking:elb_attach_load_balancer_to_subnets]{attach_load_balancer_to_subnets} \tab Adds one or more subnets to the set of configured subnets for the specified load balancer\cr
#'  \link[paws.networking:elb_configure_health_check]{configure_health_check} \tab Specifies the health check settings to use when evaluating the health state of your EC2 instances\cr
#'  \link[paws.networking:elb_create_app_cookie_stickiness_policy]{create_app_cookie_stickiness_policy} \tab Generates a stickiness policy with sticky session lifetimes that follow that of an application-generated cookie\cr
#'  \link[paws.networking:elb_create_lb_cookie_stickiness_policy]{create_lb_cookie_stickiness_policy} \tab Generates a stickiness policy with sticky session lifetimes controlled by the lifetime of the browser (user-agent) or a specified expiration period\cr
#'  \link[paws.networking:elb_create_load_balancer]{create_load_balancer} \tab Creates a Classic Load Balancer\cr
#'  \link[paws.networking:elb_create_load_balancer_listeners]{create_load_balancer_listeners} \tab Creates one or more listeners for the specified load balancer\cr
#'  \link[paws.networking:elb_create_load_balancer_policy]{create_load_balancer_policy} \tab Creates a policy with the specified attributes for the specified load balancer\cr
#'  \link[paws.networking:elb_delete_load_balancer]{delete_load_balancer} \tab Deletes the specified load balancer\cr
#'  \link[paws.networking:elb_delete_load_balancer_listeners]{delete_load_balancer_listeners} \tab Deletes the specified listeners from the specified load balancer\cr
#'  \link[paws.networking:elb_delete_load_balancer_policy]{delete_load_balancer_policy} \tab Deletes the specified policy from the specified load balancer\cr
#'  \link[paws.networking:elb_deregister_instances_from_load_balancer]{deregister_instances_from_load_balancer} \tab Deregisters the specified instances from the specified load balancer\cr
#'  \link[paws.networking:elb_describe_account_limits]{describe_account_limits} \tab Describes the current Elastic Load Balancing resource limits for your AWS account\cr
#'  \link[paws.networking:elb_describe_instance_health]{describe_instance_health} \tab Describes the state of the specified instances with respect to the specified load balancer\cr
#'  \link[paws.networking:elb_describe_load_balancer_attributes]{describe_load_balancer_attributes} \tab Describes the attributes for the specified load balancer\cr
#'  \link[paws.networking:elb_describe_load_balancer_policies]{describe_load_balancer_policies} \tab Describes the specified policies\cr
#'  \link[paws.networking:elb_describe_load_balancer_policy_types]{describe_load_balancer_policy_types} \tab Describes the specified load balancer policy types or all load balancer policy types\cr
#'  \link[paws.networking:elb_describe_load_balancers]{describe_load_balancers} \tab Describes the specified the load balancers\cr
#'  \link[paws.networking:elb_describe_tags]{describe_tags} \tab Describes the tags associated with the specified load balancers\cr
#'  \link[paws.networking:elb_detach_load_balancer_from_subnets]{detach_load_balancer_from_subnets} \tab Removes the specified subnets from the set of configured subnets for the load balancer\cr
#'  \link[paws.networking:elb_disable_availability_zones_for_load_balancer]{disable_availability_zones_for_load_balancer} \tab Removes the specified Availability Zones from the set of Availability Zones for the specified load balancer in EC2-Classic or a default VPC\cr
#'  \link[paws.networking:elb_enable_availability_zones_for_load_balancer]{enable_availability_zones_for_load_balancer} \tab Adds the specified Availability Zones to the set of Availability Zones for the specified load balancer in EC2-Classic or a default VPC\cr
#'  \link[paws.networking:elb_modify_load_balancer_attributes]{modify_load_balancer_attributes} \tab Modifies the attributes of the specified load balancer\cr
#'  \link[paws.networking:elb_register_instances_with_load_balancer]{register_instances_with_load_balancer} \tab Adds the specified instances to the specified load balancer\cr
#'  \link[paws.networking:elb_remove_tags]{remove_tags} \tab Removes one or more tags from the specified load balancer\cr
#'  \link[paws.networking:elb_set_load_balancer_listener_ssl_certificate]{set_load_balancer_listener_ssl_certificate} \tab Sets the certificate that terminates the specified listener's SSL connections\cr
#'  \link[paws.networking:elb_set_load_balancer_policies_for_backend_server]{set_load_balancer_policies_for_backend_server} \tab Replaces the set of policies associated with the specified port on which the EC2 instance is listening with a new set of policies\cr
#'  \link[paws.networking:elb_set_load_balancer_policies_of_listener]{set_load_balancer_policies_of_listener} \tab Replaces the current set of policies for the specified load balancer port with the specified set of policies
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname elb
#' @export
elb <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.networking::elb(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' Elastic Load Balancing
#'
#' @description
#' A load balancer distributes incoming traffic across targets, such as
#' your EC2 instances. This enables you to increase the availability of
#' your application. The load balancer also monitors the health of its
#' registered targets and ensures that it routes traffic only to healthy
#' targets. You configure your load balancer to accept incoming traffic by
#' specifying one or more listeners, which are configured with a protocol
#' and port number for connections from clients to the load balancer. You
#' configure a target group with a protocol and port number for connections
#' from the load balancer to the targets, and with health check settings to
#' be used when checking the health status of the targets.
#'
#' Elastic Load Balancing supports the following types of load balancers:
#' Application Load Balancers, Network Load Balancers, Gateway Load
#' Balancers, and Classic Load Balancers. This reference covers the
#' following load balancer types:
#'
#' -   Application Load Balancer - Operates at the application layer
#'     (layer 7) and supports HTTP and HTTPS.
#'
#' -   Network Load Balancer - Operates at the transport layer (layer 4)
#'     and supports TCP, TLS, and UDP.
#'
#' -   Gateway Load Balancer - Operates at the network layer (layer 3).
#'
#' For more information, see the [Elastic Load Balancing User
#' Guide](https://docs.aws.amazon.com/elasticloadbalancing/latest/userguide/).
#'
#' All Elastic Load Balancing operations are idempotent, which means that
#' they complete at most one time. If you repeat an operation, it succeeds.
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- elbv2(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- elbv2()
#' # This example adds the specified tags to the specified load balancer.
#' svc$add_tags(
#'   ResourceArns = list(
#'     "arn:aws:elasticloadbalancing:us-west-2:123456789012:loadbalancer/app/m..."
#'   ),
#'   Tags = list(
#'     list(
#'       Key = "project",
#'       Value = "lima"
#'     ),
#'     list(
#'       Key = "department",
#'       Value = "digital-media"
#'     )
#'   )
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.networking:elbv2_add_listener_certificates]{add_listener_certificates} \tab Adds the specified SSL server certificate to the certificate list for the specified HTTPS or TLS listener\cr
#'  \link[paws.networking:elbv2_add_tags]{add_tags} \tab Adds the specified tags to the specified Elastic Load Balancing resource\cr
#'  \link[paws.networking:elbv2_create_listener]{create_listener} \tab Creates a listener for the specified Application Load Balancer, Network Load Balancer, or Gateway Load Balancer\cr
#'  \link[paws.networking:elbv2_create_load_balancer]{create_load_balancer} \tab Creates an Application Load Balancer, Network Load Balancer, or Gateway Load Balancer\cr
#'  \link[paws.networking:elbv2_create_rule]{create_rule} \tab Creates a rule for the specified listener\cr
#'  \link[paws.networking:elbv2_create_target_group]{create_target_group} \tab Creates a target group\cr
#'  \link[paws.networking:elbv2_delete_listener]{delete_listener} \tab Deletes the specified listener\cr
#'  \link[paws.networking:elbv2_delete_load_balancer]{delete_load_balancer} \tab Deletes the specified Application Load Balancer, Network Load Balancer, or Gateway Load Balancer\cr
#'  \link[paws.networking:elbv2_delete_rule]{delete_rule} \tab Deletes the specified rule\cr
#'  \link[paws.networking:elbv2_delete_target_group]{delete_target_group} \tab Deletes the specified target group\cr
#'  \link[paws.networking:elbv2_deregister_targets]{deregister_targets} \tab Deregisters the specified targets from the specified target group\cr
#'  \link[paws.networking:elbv2_describe_account_limits]{describe_account_limits} \tab Describes the current Elastic Load Balancing resource limits for your Amazon Web Services account\cr
#'  \link[paws.networking:elbv2_describe_listener_certificates]{describe_listener_certificates} \tab Describes the default certificate and the certificate list for the specified HTTPS or TLS listener\cr
#'  \link[paws.networking:elbv2_describe_listeners]{describe_listeners} \tab Describes the specified listeners or the listeners for the specified Application Load Balancer, Network Load Balancer, or Gateway Load Balancer\cr
#'  \link[paws.networking:elbv2_describe_load_balancer_attributes]{describe_load_balancer_attributes} \tab Describes the attributes for the specified Application Load Balancer, Network Load Balancer, or Gateway Load Balancer\cr
#'  \link[paws.networking:elbv2_describe_load_balancers]{describe_load_balancers} \tab Describes the specified load balancers or all of your load balancers\cr
#'  \link[paws.networking:elbv2_describe_rules]{describe_rules} \tab Describes the specified rules or the rules for the specified listener\cr
#'  \link[paws.networking:elbv2_describe_ssl_policies]{describe_ssl_policies} \tab Describes the specified policies or all policies used for SSL negotiation\cr
#'  \link[paws.networking:elbv2_describe_tags]{describe_tags} \tab Describes the tags for the specified Elastic Load Balancing resources\cr
#'  \link[paws.networking:elbv2_describe_target_group_attributes]{describe_target_group_attributes} \tab Describes the attributes for the specified target group\cr
#'  \link[paws.networking:elbv2_describe_target_groups]{describe_target_groups} \tab Describes the specified target groups or all of your target groups\cr
#'  \link[paws.networking:elbv2_describe_target_health]{describe_target_health} \tab Describes the health of the specified targets or all of your targets\cr
#'  \link[paws.networking:elbv2_modify_listener]{modify_listener} \tab Replaces the specified properties of the specified listener\cr
#'  \link[paws.networking:elbv2_modify_load_balancer_attributes]{modify_load_balancer_attributes} \tab Modifies the specified attributes of the specified Application Load Balancer, Network Load Balancer, or Gateway Load Balancer\cr
#'  \link[paws.networking:elbv2_modify_rule]{modify_rule} \tab Replaces the specified properties of the specified rule\cr
#'  \link[paws.networking:elbv2_modify_target_group]{modify_target_group} \tab Modifies the health checks used when evaluating the health state of the targets in the specified target group\cr
#'  \link[paws.networking:elbv2_modify_target_group_attributes]{modify_target_group_attributes} \tab Modifies the specified attributes of the specified target group\cr
#'  \link[paws.networking:elbv2_register_targets]{register_targets} \tab Registers the specified targets with the specified target group\cr
#'  \link[paws.networking:elbv2_remove_listener_certificates]{remove_listener_certificates} \tab Removes the specified certificate from the certificate list for the specified HTTPS or TLS listener\cr
#'  \link[paws.networking:elbv2_remove_tags]{remove_tags} \tab Removes the specified tags from the specified Elastic Load Balancing resources\cr
#'  \link[paws.networking:elbv2_set_ip_address_type]{set_ip_address_type} \tab Sets the type of IP addresses used by the subnets of the specified load balancer\cr
#'  \link[paws.networking:elbv2_set_rule_priorities]{set_rule_priorities} \tab Sets the priorities of the specified rules\cr
#'  \link[paws.networking:elbv2_set_security_groups]{set_security_groups} \tab Associates the specified security groups with the specified Application Load Balancer or Network Load Balancer\cr
#'  \link[paws.networking:elbv2_set_subnets]{set_subnets} \tab Enables the Availability Zones for the specified public subnets for the specified Application Load Balancer or Network Load Balancer
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname elbv2
#' @export
elbv2 <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.networking::elbv2(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' AWS Global Accelerator
#'
#' @description
#' Global Accelerator
#'
#' This is the *Global Accelerator API Reference*. This guide is for
#' developers who need detailed information about Global Accelerator API
#' actions, data types, and errors. For more information about Global
#' Accelerator features, see the [Global Accelerator Developer
#' Guide](https://docs.aws.amazon.com/global-accelerator/latest/dg/what-is-global-accelerator.html).
#'
#' Global Accelerator is a service in which you create *accelerators* to
#' improve the performance of your applications for local and global users.
#' Depending on the type of accelerator you choose, you can gain additional
#' benefits.
#'
#' -   By using a standard accelerator, you can improve availability of
#'     your internet applications that are used by a global audience. With
#'     a standard accelerator, Global Accelerator directs traffic to
#'     optimal endpoints over the Amazon Web Services global network.
#'
#' -   For other scenarios, you might choose a custom routing accelerator.
#'     With a custom routing accelerator, you can use application logic to
#'     directly map one or more users to a specific endpoint among many
#'     endpoints.
#'
#' Global Accelerator is a global service that supports endpoints in
#' multiple Amazon Web Services Regions but you must specify the US West
#' (Oregon) Region to create, update, or otherwise work with accelerators.
#' That is, for example, specify `--region us-west-2` on Amazon Web
#' Services CLI commands.
#'
#' By default, Global Accelerator provides you with static IP addresses
#' that you associate with your accelerator. The static IP addresses are
#' anycast from the Amazon Web Services edge network. For IPv4, Global
#' Accelerator provides two static IPv4 addresses. For dual-stack, Global
#' Accelerator provides a total of four addresses: two static IPv4
#' addresses and two static IPv6 addresses. With a standard accelerator for
#' IPv4, instead of using the addresses that Global Accelerator provides,
#' you can configure these entry points to be IPv4 addresses from your own
#' IP address ranges that you bring toGlobal Accelerator (BYOIP).
#'
#' For a standard accelerator, they distribute incoming application traffic
#' across multiple endpoint resources in multiple Amazon Web Services
#' Regions , which increases the availability of your applications.
#' Endpoints for standard accelerators can be Network Load Balancers,
#' Application Load Balancers, Amazon EC2 instances, or Elastic IP
#' addresses that are located in one Amazon Web Services Region or multiple
#' Amazon Web Services Regions. For custom routing accelerators, you map
#' traffic that arrives to the static IP addresses to specific Amazon EC2
#' servers in endpoints that are virtual private cloud (VPC) subnets.
#'
#' The static IP addresses remain assigned to your accelerator for as long
#' as it exists, even if you disable the accelerator and it no longer
#' accepts or routes traffic. However, when you *delete* an accelerator,
#' you lose the static IP addresses that are assigned to it, so you can no
#' longer route traffic by using them. You can use IAM policies like
#' tag-based permissions with Global Accelerator to limit the users who
#' have permissions to delete an accelerator. For more information, see
#' [Tag-based
#' policies](https://docs.aws.amazon.com/global-accelerator/latest/dg/).
#'
#' For standard accelerators, Global Accelerator uses the Amazon Web
#' Services global network to route traffic to the optimal regional
#' endpoint based on health, client location, and policies that you
#' configure. The service reacts instantly to changes in health or
#' configuration to ensure that internet traffic from clients is always
#' directed to healthy endpoints.
#'
#' For more information about understanding and using Global Accelerator,
#' see the [Global Accelerator Developer
#' Guide](https://docs.aws.amazon.com/global-accelerator/latest/dg/what-is-global-accelerator.html).
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- globalaccelerator(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- globalaccelerator()
#' svc$add_custom_routing_endpoints(
#'   Foo = 123
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.networking:globalaccelerator_add_custom_routing_endpoints]{add_custom_routing_endpoints} \tab Associate a virtual private cloud (VPC) subnet endpoint with your custom routing accelerator\cr
#'  \link[paws.networking:globalaccelerator_add_endpoints]{add_endpoints} \tab Add endpoints to an endpoint group\cr
#'  \link[paws.networking:globalaccelerator_advertise_byoip_cidr]{advertise_byoip_cidr} \tab Advertises an IPv4 address range that is provisioned for use with your Amazon Web Services resources through bring your own IP addresses (BYOIP)\cr
#'  \link[paws.networking:globalaccelerator_allow_custom_routing_traffic]{allow_custom_routing_traffic} \tab Specify the Amazon EC2 instance (destination) IP addresses and ports for a VPC subnet endpoint that can receive traffic for a custom routing accelerator\cr
#'  \link[paws.networking:globalaccelerator_create_accelerator]{create_accelerator} \tab Create an accelerator\cr
#'  \link[paws.networking:globalaccelerator_create_custom_routing_accelerator]{create_custom_routing_accelerator} \tab Create a custom routing accelerator\cr
#'  \link[paws.networking:globalaccelerator_create_custom_routing_endpoint_group]{create_custom_routing_endpoint_group} \tab Create an endpoint group for the specified listener for a custom routing accelerator\cr
#'  \link[paws.networking:globalaccelerator_create_custom_routing_listener]{create_custom_routing_listener} \tab Create a listener to process inbound connections from clients to a custom routing accelerator\cr
#'  \link[paws.networking:globalaccelerator_create_endpoint_group]{create_endpoint_group} \tab Create an endpoint group for the specified listener\cr
#'  \link[paws.networking:globalaccelerator_create_listener]{create_listener} \tab Create a listener to process inbound connections from clients to an accelerator\cr
#'  \link[paws.networking:globalaccelerator_delete_accelerator]{delete_accelerator} \tab Delete an accelerator\cr
#'  \link[paws.networking:globalaccelerator_delete_custom_routing_accelerator]{delete_custom_routing_accelerator} \tab Delete a custom routing accelerator\cr
#'  \link[paws.networking:globalaccelerator_delete_custom_routing_endpoint_group]{delete_custom_routing_endpoint_group} \tab Delete an endpoint group from a listener for a custom routing accelerator\cr
#'  \link[paws.networking:globalaccelerator_delete_custom_routing_listener]{delete_custom_routing_listener} \tab Delete a listener for a custom routing accelerator\cr
#'  \link[paws.networking:globalaccelerator_delete_endpoint_group]{delete_endpoint_group} \tab Delete an endpoint group from a listener\cr
#'  \link[paws.networking:globalaccelerator_delete_listener]{delete_listener} \tab Delete a listener from an accelerator\cr
#'  \link[paws.networking:globalaccelerator_deny_custom_routing_traffic]{deny_custom_routing_traffic} \tab Specify the Amazon EC2 instance (destination) IP addresses and ports for a VPC subnet endpoint that cannot receive traffic for a custom routing accelerator\cr
#'  \link[paws.networking:globalaccelerator_deprovision_byoip_cidr]{deprovision_byoip_cidr} \tab Releases the specified address range that you provisioned to use with your Amazon Web Services resources through bring your own IP addresses (BYOIP) and deletes the corresponding address pool\cr
#'  \link[paws.networking:globalaccelerator_describe_accelerator]{describe_accelerator} \tab Describe an accelerator\cr
#'  \link[paws.networking:globalaccelerator_describe_accelerator_attributes]{describe_accelerator_attributes} \tab Describe the attributes of an accelerator\cr
#'  \link[paws.networking:globalaccelerator_describe_custom_routing_accelerator]{describe_custom_routing_accelerator} \tab Describe a custom routing accelerator\cr
#'  \link[paws.networking:globalaccelerator_describe_custom_routing_accelerator_attributes]{describe_custom_routing_accelerator_attributes} \tab Describe the attributes of a custom routing accelerator\cr
#'  \link[paws.networking:globalaccelerator_describe_custom_routing_endpoint_group]{describe_custom_routing_endpoint_group} \tab Describe an endpoint group for a custom routing accelerator\cr
#'  \link[paws.networking:globalaccelerator_describe_custom_routing_listener]{describe_custom_routing_listener} \tab The description of a listener for a custom routing accelerator\cr
#'  \link[paws.networking:globalaccelerator_describe_endpoint_group]{describe_endpoint_group} \tab Describe an endpoint group\cr
#'  \link[paws.networking:globalaccelerator_describe_listener]{describe_listener} \tab Describe a listener\cr
#'  \link[paws.networking:globalaccelerator_list_accelerators]{list_accelerators} \tab List the accelerators for an Amazon Web Services account\cr
#'  \link[paws.networking:globalaccelerator_list_byoip_cidrs]{list_byoip_cidrs} \tab Lists the IP address ranges that were specified in calls to ProvisionByoipCidr, including the current state and a history of state changes\cr
#'  \link[paws.networking:globalaccelerator_list_custom_routing_accelerators]{list_custom_routing_accelerators} \tab List the custom routing accelerators for an Amazon Web Services account\cr
#'  \link[paws.networking:globalaccelerator_list_custom_routing_endpoint_groups]{list_custom_routing_endpoint_groups} \tab List the endpoint groups that are associated with a listener for a custom routing accelerator\cr
#'  \link[paws.networking:globalaccelerator_list_custom_routing_listeners]{list_custom_routing_listeners} \tab List the listeners for a custom routing accelerator\cr
#'  \link[paws.networking:globalaccelerator_list_custom_routing_port_mappings]{list_custom_routing_port_mappings} \tab Provides a complete mapping from the public accelerator IP address and port to destination EC2 instance IP addresses and ports in the virtual public cloud (VPC) subnet endpoint for a custom routing accelerator\cr
#'  \link[paws.networking:globalaccelerator_list_custom_routing_port_mappings_by_destination]{list_custom_routing_port_mappings_by_destination} \tab List the port mappings for a specific EC2 instance (destination) in a VPC subnet endpoint\cr
#'  \link[paws.networking:globalaccelerator_list_endpoint_groups]{list_endpoint_groups} \tab List the endpoint groups that are associated with a listener\cr
#'  \link[paws.networking:globalaccelerator_list_listeners]{list_listeners} \tab List the listeners for an accelerator\cr
#'  \link[paws.networking:globalaccelerator_list_tags_for_resource]{list_tags_for_resource} \tab List all tags for an accelerator\cr
#'  \link[paws.networking:globalaccelerator_provision_byoip_cidr]{provision_byoip_cidr} \tab Provisions an IP address range to use with your Amazon Web Services resources through bring your own IP addresses (BYOIP) and creates a corresponding address pool\cr
#'  \link[paws.networking:globalaccelerator_remove_custom_routing_endpoints]{remove_custom_routing_endpoints} \tab Remove endpoints from a custom routing accelerator\cr
#'  \link[paws.networking:globalaccelerator_remove_endpoints]{remove_endpoints} \tab Remove endpoints from an endpoint group\cr
#'  \link[paws.networking:globalaccelerator_tag_resource]{tag_resource} \tab Add tags to an accelerator resource\cr
#'  \link[paws.networking:globalaccelerator_untag_resource]{untag_resource} \tab Remove tags from a Global Accelerator resource\cr
#'  \link[paws.networking:globalaccelerator_update_accelerator]{update_accelerator} \tab Update an accelerator to make changes, such as the following:\cr
#'  \link[paws.networking:globalaccelerator_update_accelerator_attributes]{update_accelerator_attributes} \tab Update the attributes for an accelerator\cr
#'  \link[paws.networking:globalaccelerator_update_custom_routing_accelerator]{update_custom_routing_accelerator} \tab Update a custom routing accelerator\cr
#'  \link[paws.networking:globalaccelerator_update_custom_routing_accelerator_attributes]{update_custom_routing_accelerator_attributes} \tab Update the attributes for a custom routing accelerator\cr
#'  \link[paws.networking:globalaccelerator_update_custom_routing_listener]{update_custom_routing_listener} \tab Update a listener for a custom routing accelerator\cr
#'  \link[paws.networking:globalaccelerator_update_endpoint_group]{update_endpoint_group} \tab Update an endpoint group\cr
#'  \link[paws.networking:globalaccelerator_update_listener]{update_listener} \tab Update a listener\cr
#'  \link[paws.networking:globalaccelerator_withdraw_byoip_cidr]{withdraw_byoip_cidr} \tab Stops advertising an address range that is provisioned as an address pool
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname globalaccelerator
#' @export
globalaccelerator <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.networking::globalaccelerator(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' AWS Network Firewall
#'
#' @description
#' This is the API Reference for Network Firewall. This guide is for
#' developers who need detailed information about the Network Firewall API
#' actions, data types, and errors.
#'
#' -   The REST API requires you to handle connection details, such as
#'     calculating signatures, handling request retries, and error
#'     handling. For general information about using the Amazon Web
#'     Services REST APIs, see [Amazon Web Services
#'     APIs](https://docs.aws.amazon.com/general/latest/gr/).
#'
#'     To access Network Firewall using the REST API endpoint:
#'     `https://network-firewall.<region>.amazonaws.com `
#'
#' -   Alternatively, you can use one of the Amazon Web Services SDKs to
#'     access an API that's tailored to the programming language or
#'     platform that you're using. For more information, see [Amazon Web
#'     Services SDKs](https://aws.amazon.com/developer/tools/#SDKs).
#'
#' -   For descriptions of Network Firewall features, including and
#'     step-by-step instructions on how to use them through the Network
#'     Firewall console, see the [Network Firewall Developer
#'     Guide](https://docs.aws.amazon.com/network-firewall/latest/developerguide/).
#'
#' Network Firewall is a stateful, managed, network firewall and intrusion
#' detection and prevention service for Amazon Virtual Private Cloud
#' (Amazon VPC). With Network Firewall, you can filter traffic at the
#' perimeter of your VPC. This includes filtering traffic going to and
#' coming from an internet gateway, NAT gateway, or over VPN or Direct
#' Connect. Network Firewall uses rules that are compatible with Suricata,
#' a free, open source network analysis and threat detection engine.
#' Network Firewall supports Suricata version 6.0.9. For information about
#' Suricata, see the [Suricata website](https://suricata.io/).
#'
#' You can use Network Firewall to monitor and protect your VPC traffic in
#' a number of ways. The following are just a few examples:
#'
#' -   Allow domains or IP addresses for known Amazon Web Services service
#'     endpoints, such as Amazon S3, and block all other forms of traffic.
#'
#' -   Use custom lists of known bad domains to limit the types of domain
#'     names that your applications can access.
#'
#' -   Perform deep packet inspection on traffic entering or leaving your
#'     VPC.
#'
#' -   Use stateful protocol detection to filter protocols like HTTPS,
#'     regardless of the port used.
#'
#' To enable Network Firewall for your VPCs, you perform steps in both
#' Amazon VPC and in Network Firewall. For information about using Amazon
#' VPC, see [Amazon VPC User
#' Guide](https://docs.aws.amazon.com/vpc/latest/userguide/).
#'
#' To start using Network Firewall, do the following:
#'
#' 1.  (Optional) If you don't already have a VPC that you want to protect,
#'     create it in Amazon VPC.
#'
#' 2.  In Amazon VPC, in each Availability Zone where you want to have a
#'     firewall endpoint, create a subnet for the sole use of Network
#'     Firewall.
#'
#' 3.  In Network Firewall, create stateless and stateful rule groups, to
#'     define the components of the network traffic filtering behavior that
#'     you want your firewall to have.
#'
#' 4.  In Network Firewall, create a firewall policy that uses your rule
#'     groups and specifies additional default traffic filtering behavior.
#'
#' 5.  In Network Firewall, create a firewall and specify your new firewall
#'     policy and VPC subnets. Network Firewall creates a firewall endpoint
#'     in each subnet that you specify, with the behavior that's defined in
#'     the firewall policy.
#'
#' 6.  In Amazon VPC, use ingress routing enhancements to route traffic
#'     through the new firewall endpoints.
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- networkfirewall(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- networkfirewall()
#' svc$associate_firewall_policy(
#'   Foo = 123
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.networking:networkfirewall_associate_firewall_policy]{associate_firewall_policy} \tab Associates a FirewallPolicy to a Firewall\cr
#'  \link[paws.networking:networkfirewall_associate_subnets]{associate_subnets} \tab Associates the specified subnets in the Amazon VPC to the firewall\cr
#'  \link[paws.networking:networkfirewall_create_firewall]{create_firewall} \tab Creates an Network Firewall Firewall and accompanying FirewallStatus for a VPC\cr
#'  \link[paws.networking:networkfirewall_create_firewall_policy]{create_firewall_policy} \tab Creates the firewall policy for the firewall according to the specifications\cr
#'  \link[paws.networking:networkfirewall_create_rule_group]{create_rule_group} \tab Creates the specified stateless or stateful rule group, which includes the rules for network traffic inspection, a capacity setting, and tags\cr
#'  \link[paws.networking:networkfirewall_create_tls_inspection_configuration]{create_tls_inspection_configuration} \tab Creates an Network Firewall TLS inspection configuration\cr
#'  \link[paws.networking:networkfirewall_delete_firewall]{delete_firewall} \tab Deletes the specified Firewall and its FirewallStatus\cr
#'  \link[paws.networking:networkfirewall_delete_firewall_policy]{delete_firewall_policy} \tab Deletes the specified FirewallPolicy\cr
#'  \link[paws.networking:networkfirewall_delete_resource_policy]{delete_resource_policy} \tab Deletes a resource policy that you created in a PutResourcePolicy request\cr
#'  \link[paws.networking:networkfirewall_delete_rule_group]{delete_rule_group} \tab Deletes the specified RuleGroup\cr
#'  \link[paws.networking:networkfirewall_delete_tls_inspection_configuration]{delete_tls_inspection_configuration} \tab Deletes the specified TLSInspectionConfiguration\cr
#'  \link[paws.networking:networkfirewall_describe_firewall]{describe_firewall} \tab Returns the data objects for the specified firewall\cr
#'  \link[paws.networking:networkfirewall_describe_firewall_policy]{describe_firewall_policy} \tab Returns the data objects for the specified firewall policy\cr
#'  \link[paws.networking:networkfirewall_describe_logging_configuration]{describe_logging_configuration} \tab Returns the logging configuration for the specified firewall\cr
#'  \link[paws.networking:networkfirewall_describe_resource_policy]{describe_resource_policy} \tab Retrieves a resource policy that you created in a PutResourcePolicy request\cr
#'  \link[paws.networking:networkfirewall_describe_rule_group]{describe_rule_group} \tab Returns the data objects for the specified rule group\cr
#'  \link[paws.networking:networkfirewall_describe_rule_group_metadata]{describe_rule_group_metadata} \tab High-level information about a rule group, returned by operations like create and describe\cr
#'  \link[paws.networking:networkfirewall_describe_tls_inspection_configuration]{describe_tls_inspection_configuration} \tab Returns the data objects for the specified TLS inspection configuration\cr
#'  \link[paws.networking:networkfirewall_disassociate_subnets]{disassociate_subnets} \tab Removes the specified subnet associations from the firewall\cr
#'  \link[paws.networking:networkfirewall_list_firewall_policies]{list_firewall_policies} \tab Retrieves the metadata for the firewall policies that you have defined\cr
#'  \link[paws.networking:networkfirewall_list_firewalls]{list_firewalls} \tab Retrieves the metadata for the firewalls that you have defined\cr
#'  \link[paws.networking:networkfirewall_list_rule_groups]{list_rule_groups} \tab Retrieves the metadata for the rule groups that you have defined\cr
#'  \link[paws.networking:networkfirewall_list_tags_for_resource]{list_tags_for_resource} \tab Retrieves the tags associated with the specified resource\cr
#'  \link[paws.networking:networkfirewall_list_tls_inspection_configurations]{list_tls_inspection_configurations} \tab Retrieves the metadata for the TLS inspection configurations that you have defined\cr
#'  \link[paws.networking:networkfirewall_put_resource_policy]{put_resource_policy} \tab Creates or updates an IAM policy for your rule group or firewall policy\cr
#'  \link[paws.networking:networkfirewall_tag_resource]{tag_resource} \tab Adds the specified tags to the specified resource\cr
#'  \link[paws.networking:networkfirewall_untag_resource]{untag_resource} \tab Removes the tags with the specified keys from the specified resource\cr
#'  \link[paws.networking:networkfirewall_update_firewall_delete_protection]{update_firewall_delete_protection} \tab Modifies the flag, DeleteProtection, which indicates whether it is possible to delete the firewall\cr
#'  \link[paws.networking:networkfirewall_update_firewall_description]{update_firewall_description} \tab Modifies the description for the specified firewall\cr
#'  \link[paws.networking:networkfirewall_update_firewall_encryption_configuration]{update_firewall_encryption_configuration} \tab A complex type that contains settings for encryption of your firewall resources\cr
#'  \link[paws.networking:networkfirewall_update_firewall_policy]{update_firewall_policy} \tab Updates the properties of the specified firewall policy\cr
#'  \link[paws.networking:networkfirewall_update_firewall_policy_change_protection]{update_firewall_policy_change_protection} \tab Modifies the flag, ChangeProtection, which indicates whether it is possible to change the firewall\cr
#'  \link[paws.networking:networkfirewall_update_logging_configuration]{update_logging_configuration} \tab Sets the logging configuration for the specified firewall\cr
#'  \link[paws.networking:networkfirewall_update_rule_group]{update_rule_group} \tab Updates the rule settings for the specified rule group\cr
#'  \link[paws.networking:networkfirewall_update_subnet_change_protection]{update_subnet_change_protection} \tab Update subnet change protection\cr
#'  \link[paws.networking:networkfirewall_update_tls_inspection_configuration]{update_tls_inspection_configuration} \tab Updates the TLS inspection configuration settings for the specified TLS inspection configuration
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname networkfirewall
#' @export
networkfirewall <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.networking::networkfirewall(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' AWS Network Manager
#'
#' @description
#' Amazon Web Services enables you to centrally manage your Amazon Web
#' Services Cloud WAN core network and your Transit Gateway network across
#' Amazon Web Services accounts, Regions, and on-premises locations.
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- networkmanager(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- networkmanager()
#' svc$accept_attachment(
#'   Foo = 123
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.networking:networkmanager_accept_attachment]{accept_attachment} \tab Accepts a core network attachment request\cr
#'  \link[paws.networking:networkmanager_associate_connect_peer]{associate_connect_peer} \tab Associates a core network Connect peer with a device and optionally, with a link\cr
#'  \link[paws.networking:networkmanager_associate_customer_gateway]{associate_customer_gateway} \tab Associates a customer gateway with a device and optionally, with a link\cr
#'  \link[paws.networking:networkmanager_associate_link]{associate_link} \tab Associates a link to a device\cr
#'  \link[paws.networking:networkmanager_associate_transit_gateway_connect_peer]{associate_transit_gateway_connect_peer} \tab Associates a transit gateway Connect peer with a device, and optionally, with a link\cr
#'  \link[paws.networking:networkmanager_create_connect_attachment]{create_connect_attachment} \tab Creates a core network Connect attachment from a specified core network attachment\cr
#'  \link[paws.networking:networkmanager_create_connection]{create_connection} \tab Creates a connection between two devices\cr
#'  \link[paws.networking:networkmanager_create_connect_peer]{create_connect_peer} \tab Creates a core network Connect peer for a specified core network connect attachment between a core network and an appliance\cr
#'  \link[paws.networking:networkmanager_create_core_network]{create_core_network} \tab Creates a core network as part of your global network, and optionally, with a core network policy\cr
#'  \link[paws.networking:networkmanager_create_device]{create_device} \tab Creates a new device in a global network\cr
#'  \link[paws.networking:networkmanager_create_global_network]{create_global_network} \tab Creates a new, empty global network\cr
#'  \link[paws.networking:networkmanager_create_link]{create_link} \tab Creates a new link for a specified site\cr
#'  \link[paws.networking:networkmanager_create_site]{create_site} \tab Creates a new site in a global network\cr
#'  \link[paws.networking:networkmanager_create_site_to_site_vpn_attachment]{create_site_to_site_vpn_attachment} \tab Creates an Amazon Web Services site-to-site VPN attachment on an edge location of a core network\cr
#'  \link[paws.networking:networkmanager_create_transit_gateway_peering]{create_transit_gateway_peering} \tab Creates a transit gateway peering connection\cr
#'  \link[paws.networking:networkmanager_create_transit_gateway_route_table_attachment]{create_transit_gateway_route_table_attachment} \tab Creates a transit gateway route table attachment\cr
#'  \link[paws.networking:networkmanager_create_vpc_attachment]{create_vpc_attachment} \tab Creates a VPC attachment on an edge location of a core network\cr
#'  \link[paws.networking:networkmanager_delete_attachment]{delete_attachment} \tab Deletes an attachment\cr
#'  \link[paws.networking:networkmanager_delete_connection]{delete_connection} \tab Deletes the specified connection in your global network\cr
#'  \link[paws.networking:networkmanager_delete_connect_peer]{delete_connect_peer} \tab Deletes a Connect peer\cr
#'  \link[paws.networking:networkmanager_delete_core_network]{delete_core_network} \tab Deletes a core network along with all core network policies\cr
#'  \link[paws.networking:networkmanager_delete_core_network_policy_version]{delete_core_network_policy_version} \tab Deletes a policy version from a core network\cr
#'  \link[paws.networking:networkmanager_delete_device]{delete_device} \tab Deletes an existing device\cr
#'  \link[paws.networking:networkmanager_delete_global_network]{delete_global_network} \tab Deletes an existing global network\cr
#'  \link[paws.networking:networkmanager_delete_link]{delete_link} \tab Deletes an existing link\cr
#'  \link[paws.networking:networkmanager_delete_peering]{delete_peering} \tab Deletes an existing peering connection\cr
#'  \link[paws.networking:networkmanager_delete_resource_policy]{delete_resource_policy} \tab Deletes a resource policy for the specified resource\cr
#'  \link[paws.networking:networkmanager_delete_site]{delete_site} \tab Deletes an existing site\cr
#'  \link[paws.networking:networkmanager_deregister_transit_gateway]{deregister_transit_gateway} \tab Deregisters a transit gateway from your global network\cr
#'  \link[paws.networking:networkmanager_describe_global_networks]{describe_global_networks} \tab Describes one or more global networks\cr
#'  \link[paws.networking:networkmanager_disassociate_connect_peer]{disassociate_connect_peer} \tab Disassociates a core network Connect peer from a device and a link\cr
#'  \link[paws.networking:networkmanager_disassociate_customer_gateway]{disassociate_customer_gateway} \tab Disassociates a customer gateway from a device and a link\cr
#'  \link[paws.networking:networkmanager_disassociate_link]{disassociate_link} \tab Disassociates an existing device from a link\cr
#'  \link[paws.networking:networkmanager_disassociate_transit_gateway_connect_peer]{disassociate_transit_gateway_connect_peer} \tab Disassociates a transit gateway Connect peer from a device and link\cr
#'  \link[paws.networking:networkmanager_execute_core_network_change_set]{execute_core_network_change_set} \tab Executes a change set on your core network\cr
#'  \link[paws.networking:networkmanager_get_connect_attachment]{get_connect_attachment} \tab Returns information about a core network Connect attachment\cr
#'  \link[paws.networking:networkmanager_get_connections]{get_connections} \tab Gets information about one or more of your connections in a global network\cr
#'  \link[paws.networking:networkmanager_get_connect_peer]{get_connect_peer} \tab Returns information about a core network Connect peer\cr
#'  \link[paws.networking:networkmanager_get_connect_peer_associations]{get_connect_peer_associations} \tab Returns information about a core network Connect peer associations\cr
#'  \link[paws.networking:networkmanager_get_core_network]{get_core_network} \tab Returns information about the LIVE policy for a core network\cr
#'  \link[paws.networking:networkmanager_get_core_network_change_events]{get_core_network_change_events} \tab Returns information about a core network change event\cr
#'  \link[paws.networking:networkmanager_get_core_network_change_set]{get_core_network_change_set} \tab Returns a change set between the LIVE core network policy and a submitted policy\cr
#'  \link[paws.networking:networkmanager_get_core_network_policy]{get_core_network_policy} \tab Returns details about a core network policy\cr
#'  \link[paws.networking:networkmanager_get_customer_gateway_associations]{get_customer_gateway_associations} \tab Gets the association information for customer gateways that are associated with devices and links in your global network\cr
#'  \link[paws.networking:networkmanager_get_devices]{get_devices} \tab Gets information about one or more of your devices in a global network\cr
#'  \link[paws.networking:networkmanager_get_link_associations]{get_link_associations} \tab Gets the link associations for a device or a link\cr
#'  \link[paws.networking:networkmanager_get_links]{get_links} \tab Gets information about one or more links in a specified global network\cr
#'  \link[paws.networking:networkmanager_get_network_resource_counts]{get_network_resource_counts} \tab Gets the count of network resources, by resource type, for the specified global network\cr
#'  \link[paws.networking:networkmanager_get_network_resource_relationships]{get_network_resource_relationships} \tab Gets the network resource relationships for the specified global network\cr
#'  \link[paws.networking:networkmanager_get_network_resources]{get_network_resources} \tab Describes the network resources for the specified global network\cr
#'  \link[paws.networking:networkmanager_get_network_routes]{get_network_routes} \tab Gets the network routes of the specified global network\cr
#'  \link[paws.networking:networkmanager_get_network_telemetry]{get_network_telemetry} \tab Gets the network telemetry of the specified global network\cr
#'  \link[paws.networking:networkmanager_get_resource_policy]{get_resource_policy} \tab Returns information about a resource policy\cr
#'  \link[paws.networking:networkmanager_get_route_analysis]{get_route_analysis} \tab Gets information about the specified route analysis\cr
#'  \link[paws.networking:networkmanager_get_sites]{get_sites} \tab Gets information about one or more of your sites in a global network\cr
#'  \link[paws.networking:networkmanager_get_site_to_site_vpn_attachment]{get_site_to_site_vpn_attachment} \tab Returns information about a site-to-site VPN attachment\cr
#'  \link[paws.networking:networkmanager_get_transit_gateway_connect_peer_associations]{get_transit_gateway_connect_peer_associations} \tab Gets information about one or more of your transit gateway Connect peer associations in a global network\cr
#'  \link[paws.networking:networkmanager_get_transit_gateway_peering]{get_transit_gateway_peering} \tab Returns information about a transit gateway peer\cr
#'  \link[paws.networking:networkmanager_get_transit_gateway_registrations]{get_transit_gateway_registrations} \tab Gets information about the transit gateway registrations in a specified global network\cr
#'  \link[paws.networking:networkmanager_get_transit_gateway_route_table_attachment]{get_transit_gateway_route_table_attachment} \tab Returns information about a transit gateway route table attachment\cr
#'  \link[paws.networking:networkmanager_get_vpc_attachment]{get_vpc_attachment} \tab Returns information about a VPC attachment\cr
#'  \link[paws.networking:networkmanager_list_attachments]{list_attachments} \tab Returns a list of core network attachments\cr
#'  \link[paws.networking:networkmanager_list_connect_peers]{list_connect_peers} \tab Returns a list of core network Connect peers\cr
#'  \link[paws.networking:networkmanager_list_core_network_policy_versions]{list_core_network_policy_versions} \tab Returns a list of core network policy versions\cr
#'  \link[paws.networking:networkmanager_list_core_networks]{list_core_networks} \tab Returns a list of owned and shared core networks\cr
#'  \link[paws.networking:networkmanager_list_organization_service_access_status]{list_organization_service_access_status} \tab Gets the status of the Service Linked Role (SLR) deployment for the accounts in a given Amazon Web Services Organization\cr
#'  \link[paws.networking:networkmanager_list_peerings]{list_peerings} \tab Lists the peerings for a core network\cr
#'  \link[paws.networking:networkmanager_list_tags_for_resource]{list_tags_for_resource} \tab Lists the tags for a specified resource\cr
#'  \link[paws.networking:networkmanager_put_core_network_policy]{put_core_network_policy} \tab Creates a new, immutable version of a core network policy\cr
#'  \link[paws.networking:networkmanager_put_resource_policy]{put_resource_policy} \tab Creates or updates a resource policy\cr
#'  \link[paws.networking:networkmanager_register_transit_gateway]{register_transit_gateway} \tab Registers a transit gateway in your global network\cr
#'  \link[paws.networking:networkmanager_reject_attachment]{reject_attachment} \tab Rejects a core network attachment request\cr
#'  \link[paws.networking:networkmanager_restore_core_network_policy_version]{restore_core_network_policy_version} \tab Restores a previous policy version as a new, immutable version of a core network policy\cr
#'  \link[paws.networking:networkmanager_start_organization_service_access_update]{start_organization_service_access_update} \tab Enables the Network Manager service for an Amazon Web Services Organization\cr
#'  \link[paws.networking:networkmanager_start_route_analysis]{start_route_analysis} \tab Starts analyzing the routing path between the specified source and destination\cr
#'  \link[paws.networking:networkmanager_tag_resource]{tag_resource} \tab Tags a specified resource\cr
#'  \link[paws.networking:networkmanager_untag_resource]{untag_resource} \tab Removes tags from a specified resource\cr
#'  \link[paws.networking:networkmanager_update_connection]{update_connection} \tab Updates the information for an existing connection\cr
#'  \link[paws.networking:networkmanager_update_core_network]{update_core_network} \tab Updates the description of a core network\cr
#'  \link[paws.networking:networkmanager_update_device]{update_device} \tab Updates the details for an existing device\cr
#'  \link[paws.networking:networkmanager_update_global_network]{update_global_network} \tab Updates an existing global network\cr
#'  \link[paws.networking:networkmanager_update_link]{update_link} \tab Updates the details for an existing link\cr
#'  \link[paws.networking:networkmanager_update_network_resource_metadata]{update_network_resource_metadata} \tab Updates the resource metadata for the specified global network\cr
#'  \link[paws.networking:networkmanager_update_site]{update_site} \tab Updates the information for an existing site\cr
#'  \link[paws.networking:networkmanager_update_vpc_attachment]{update_vpc_attachment} \tab Updates a VPC attachment
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname networkmanager
#' @export
networkmanager <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.networking::networkmanager(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' Amazon Route 53
#'
#' @description
#' Amazon Route 53 is a highly available and scalable Domain Name System
#' (DNS) web service.
#'
#' You can use Route 53 to:
#'
#' -   Register domain names.
#'
#'     For more information, see [How domain registration
#'     works](https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/welcome-domain-registration.html).
#'
#' -   Route internet traffic to the resources for your domain
#'
#'     For more information, see [How internet traffic is routed to your
#'     website or web
#'     application](https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/welcome-dns-service.html).
#'
#' -   Check the health of your resources.
#'
#'     For more information, see [How Route 53 checks the health of your
#'     resources](https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/welcome-health-checks.html).
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- route53(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- route53()
#' # The following example associates the VPC with ID vpc-1a2b3c4d with the
#' # hosted zone with ID Z3M3LMPEXAMPLE.
#' svc$associate_vpc_with_hosted_zone(
#'   Comment = "",
#'   HostedZoneId = "Z3M3LMPEXAMPLE",
#'   VPC = list(
#'     VPCId = "vpc-1a2b3c4d",
#'     VPCRegion = "us-east-2"
#'   )
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.networking:route53_activate_key_signing_key]{activate_key_signing_key} \tab Activates a key-signing key (KSK) so that it can be used for signing by DNSSEC\cr
#'  \link[paws.networking:route53_associate_vpc_with_hosted_zone]{associate_vpc_with_hosted_zone} \tab Associates an Amazon VPC with a private hosted zone\cr
#'  \link[paws.networking:route53_change_cidr_collection]{change_cidr_collection} \tab Creates, changes, or deletes CIDR blocks within a collection\cr
#'  \link[paws.networking:route53_change_resource_record_sets]{change_resource_record_sets} \tab Creates, changes, or deletes a resource record set, which contains authoritative DNS information for a specified domain name or subdomain name\cr
#'  \link[paws.networking:route53_change_tags_for_resource]{change_tags_for_resource} \tab Adds, edits, or deletes tags for a health check or a hosted zone\cr
#'  \link[paws.networking:route53_create_cidr_collection]{create_cidr_collection} \tab Creates a CIDR collection in the current Amazon Web Services account\cr
#'  \link[paws.networking:route53_create_health_check]{create_health_check} \tab Creates a new health check\cr
#'  \link[paws.networking:route53_create_hosted_zone]{create_hosted_zone} \tab Creates a new public or private hosted zone\cr
#'  \link[paws.networking:route53_create_key_signing_key]{create_key_signing_key} \tab Creates a new key-signing key (KSK) associated with a hosted zone\cr
#'  \link[paws.networking:route53_create_query_logging_config]{create_query_logging_config} \tab Creates a configuration for DNS query logging\cr
#'  \link[paws.networking:route53_create_reusable_delegation_set]{create_reusable_delegation_set} \tab Creates a delegation set (a group of four name servers) that can be reused by multiple hosted zones that were created by the same Amazon Web Services account\cr
#'  \link[paws.networking:route53_create_traffic_policy]{create_traffic_policy} \tab Creates a traffic policy, which you use to create multiple DNS resource record sets for one domain name (such as example\cr
#'  \link[paws.networking:route53_create_traffic_policy_instance]{create_traffic_policy_instance} \tab Creates resource record sets in a specified hosted zone based on the settings in a specified traffic policy version\cr
#'  \link[paws.networking:route53_create_traffic_policy_version]{create_traffic_policy_version} \tab Creates a new version of an existing traffic policy\cr
#'  \link[paws.networking:route53_create_vpc_association_authorization]{create_vpc_association_authorization} \tab Authorizes the Amazon Web Services account that created a specified VPC to submit an AssociateVPCWithHostedZone request to associate the VPC with a specified hosted zone that was created by a different account\cr
#'  \link[paws.networking:route53_deactivate_key_signing_key]{deactivate_key_signing_key} \tab Deactivates a key-signing key (KSK) so that it will not be used for signing by DNSSEC\cr
#'  \link[paws.networking:route53_delete_cidr_collection]{delete_cidr_collection} \tab Deletes a CIDR collection in the current Amazon Web Services account\cr
#'  \link[paws.networking:route53_delete_health_check]{delete_health_check} \tab Deletes a health check\cr
#'  \link[paws.networking:route53_delete_hosted_zone]{delete_hosted_zone} \tab Deletes a hosted zone\cr
#'  \link[paws.networking:route53_delete_key_signing_key]{delete_key_signing_key} \tab Deletes a key-signing key (KSK)\cr
#'  \link[paws.networking:route53_delete_query_logging_config]{delete_query_logging_config} \tab Deletes a configuration for DNS query logging\cr
#'  \link[paws.networking:route53_delete_reusable_delegation_set]{delete_reusable_delegation_set} \tab Deletes a reusable delegation set\cr
#'  \link[paws.networking:route53_delete_traffic_policy]{delete_traffic_policy} \tab Deletes a traffic policy\cr
#'  \link[paws.networking:route53_delete_traffic_policy_instance]{delete_traffic_policy_instance} \tab Deletes a traffic policy instance and all of the resource record sets that Amazon Route 53 created when you created the instance\cr
#'  \link[paws.networking:route53_delete_vpc_association_authorization]{delete_vpc_association_authorization} \tab Removes authorization to submit an AssociateVPCWithHostedZone request to associate a specified VPC with a hosted zone that was created by a different account\cr
#'  \link[paws.networking:route53_disable_hosted_zone_dnssec]{disable_hosted_zone_dnssec} \tab Disables DNSSEC signing in a specific hosted zone\cr
#'  \link[paws.networking:route53_disassociate_vpc_from_hosted_zone]{disassociate_vpc_from_hosted_zone} \tab Disassociates an Amazon Virtual Private Cloud (Amazon VPC) from an Amazon Route 53 private hosted zone\cr
#'  \link[paws.networking:route53_enable_hosted_zone_dnssec]{enable_hosted_zone_dnssec} \tab Enables DNSSEC signing in a specific hosted zone\cr
#'  \link[paws.networking:route53_get_account_limit]{get_account_limit} \tab Gets the specified limit for the current account, for example, the maximum number of health checks that you can create using the account\cr
#'  \link[paws.networking:route53_get_change]{get_change} \tab Returns the current status of a change batch request\cr
#'  \link[paws.networking:route53_get_checker_ip_ranges]{get_checker_ip_ranges} \tab Route 53 does not perform authorization for this API because it retrieves information that is already available to the public\cr
#'  \link[paws.networking:route53_get_dnssec]{get_dnssec} \tab Returns information about DNSSEC for a specific hosted zone, including the key-signing keys (KSKs) in the hosted zone\cr
#'  \link[paws.networking:route53_get_geo_location]{get_geo_location} \tab Gets information about whether a specified geographic location is supported for Amazon Route 53 geolocation resource record sets\cr
#'  \link[paws.networking:route53_get_health_check]{get_health_check} \tab Gets information about a specified health check\cr
#'  \link[paws.networking:route53_get_health_check_count]{get_health_check_count} \tab Retrieves the number of health checks that are associated with the current Amazon Web Services account\cr
#'  \link[paws.networking:route53_get_health_check_last_failure_reason]{get_health_check_last_failure_reason} \tab Gets the reason that a specified health check failed most recently\cr
#'  \link[paws.networking:route53_get_health_check_status]{get_health_check_status} \tab Gets status of a specified health check\cr
#'  \link[paws.networking:route53_get_hosted_zone]{get_hosted_zone} \tab Gets information about a specified hosted zone including the four name servers assigned to the hosted zone\cr
#'  \link[paws.networking:route53_get_hosted_zone_count]{get_hosted_zone_count} \tab Retrieves the number of hosted zones that are associated with the current Amazon Web Services account\cr
#'  \link[paws.networking:route53_get_hosted_zone_limit]{get_hosted_zone_limit} \tab Gets the specified limit for a specified hosted zone, for example, the maximum number of records that you can create in the hosted zone\cr
#'  \link[paws.networking:route53_get_query_logging_config]{get_query_logging_config} \tab Gets information about a specified configuration for DNS query logging\cr
#'  \link[paws.networking:route53_get_reusable_delegation_set]{get_reusable_delegation_set} \tab Retrieves information about a specified reusable delegation set, including the four name servers that are assigned to the delegation set\cr
#'  \link[paws.networking:route53_get_reusable_delegation_set_limit]{get_reusable_delegation_set_limit} \tab Gets the maximum number of hosted zones that you can associate with the specified reusable delegation set\cr
#'  \link[paws.networking:route53_get_traffic_policy]{get_traffic_policy} \tab Gets information about a specific traffic policy version\cr
#'  \link[paws.networking:route53_get_traffic_policy_instance]{get_traffic_policy_instance} \tab Gets information about a specified traffic policy instance\cr
#'  \link[paws.networking:route53_get_traffic_policy_instance_count]{get_traffic_policy_instance_count} \tab Gets the number of traffic policy instances that are associated with the current Amazon Web Services account\cr
#'  \link[paws.networking:route53_list_cidr_blocks]{list_cidr_blocks} \tab Returns a paginated list of location objects and their CIDR blocks\cr
#'  \link[paws.networking:route53_list_cidr_collections]{list_cidr_collections} \tab Returns a paginated list of CIDR collections in the Amazon Web Services account (metadata only)\cr
#'  \link[paws.networking:route53_list_cidr_locations]{list_cidr_locations} \tab Returns a paginated list of CIDR locations for the given collection (metadata only, does not include CIDR blocks)\cr
#'  \link[paws.networking:route53_list_geo_locations]{list_geo_locations} \tab Retrieves a list of supported geographic locations\cr
#'  \link[paws.networking:route53_list_health_checks]{list_health_checks} \tab Retrieve a list of the health checks that are associated with the current Amazon Web Services account\cr
#'  \link[paws.networking:route53_list_hosted_zones]{list_hosted_zones} \tab Retrieves a list of the public and private hosted zones that are associated with the current Amazon Web Services account\cr
#'  \link[paws.networking:route53_list_hosted_zones_by_name]{list_hosted_zones_by_name} \tab Retrieves a list of your hosted zones in lexicographic order\cr
#'  \link[paws.networking:route53_list_hosted_zones_by_vpc]{list_hosted_zones_by_vpc} \tab Lists all the private hosted zones that a specified VPC is associated with, regardless of which Amazon Web Services account or Amazon Web Services service owns the hosted zones\cr
#'  \link[paws.networking:route53_list_query_logging_configs]{list_query_logging_configs} \tab Lists the configurations for DNS query logging that are associated with the current Amazon Web Services account or the configuration that is associated with a specified hosted zone\cr
#'  \link[paws.networking:route53_list_resource_record_sets]{list_resource_record_sets} \tab Lists the resource record sets in a specified hosted zone\cr
#'  \link[paws.networking:route53_list_reusable_delegation_sets]{list_reusable_delegation_sets} \tab Retrieves a list of the reusable delegation sets that are associated with the current Amazon Web Services account\cr
#'  \link[paws.networking:route53_list_tags_for_resource]{list_tags_for_resource} \tab Lists tags for one health check or hosted zone\cr
#'  \link[paws.networking:route53_list_tags_for_resources]{list_tags_for_resources} \tab Lists tags for up to 10 health checks or hosted zones\cr
#'  \link[paws.networking:route53_list_traffic_policies]{list_traffic_policies} \tab Gets information about the latest version for every traffic policy that is associated with the current Amazon Web Services account\cr
#'  \link[paws.networking:route53_list_traffic_policy_instances]{list_traffic_policy_instances} \tab Gets information about the traffic policy instances that you created by using the current Amazon Web Services account\cr
#'  \link[paws.networking:route53_list_traffic_policy_instances_by_hosted_zone]{list_traffic_policy_instances_by_hosted_zone} \tab Gets information about the traffic policy instances that you created in a specified hosted zone\cr
#'  \link[paws.networking:route53_list_traffic_policy_instances_by_policy]{list_traffic_policy_instances_by_policy} \tab Gets information about the traffic policy instances that you created by using a specify traffic policy version\cr
#'  \link[paws.networking:route53_list_traffic_policy_versions]{list_traffic_policy_versions} \tab Gets information about all of the versions for a specified traffic policy\cr
#'  \link[paws.networking:route53_list_vpc_association_authorizations]{list_vpc_association_authorizations} \tab Gets a list of the VPCs that were created by other accounts and that can be associated with a specified hosted zone because you've submitted one or more CreateVPCAssociationAuthorization requests\cr
#'  \link[paws.networking:route53_test_dns_answer]{test_dns_answer} \tab Gets the value that Amazon Route 53 returns in response to a DNS request for a specified record name and type\cr
#'  \link[paws.networking:route53_update_health_check]{update_health_check} \tab Updates an existing health check\cr
#'  \link[paws.networking:route53_update_hosted_zone_comment]{update_hosted_zone_comment} \tab Updates the comment for a specified hosted zone\cr
#'  \link[paws.networking:route53_update_traffic_policy_comment]{update_traffic_policy_comment} \tab Updates the comment for a specified traffic policy version\cr
#'  \link[paws.networking:route53_update_traffic_policy_instance]{update_traffic_policy_instance} \tab Updates the resource record sets in a specified hosted zone that were created based on the settings in a specified traffic policy version
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname route53
#' @export
route53 <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.networking::route53(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' Amazon Route 53 Domains
#'
#' @description
#' Amazon Route 53 API actions let you register domain names and perform
#' related operations.
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- route53domains(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- route53domains()
#' svc$accept_domain_transfer_from_another_aws_account(
#'   Foo = 123
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.networking:route53domains_accept_domain_transfer_from_another_aws_account]{accept_domain_transfer_from_another_aws_account} \tab Accepts the transfer of a domain from another Amazon Web Services account to the currentAmazon Web Services account\cr
#'  \link[paws.networking:route53domains_associate_delegation_signer_to_domain]{associate_delegation_signer_to_domain} \tab Creates a delegation signer (DS) record in the registry zone for this domain name\cr
#'  \link[paws.networking:route53domains_cancel_domain_transfer_to_another_aws_account]{cancel_domain_transfer_to_another_aws_account} \tab Cancels the transfer of a domain from the current Amazon Web Services account to another Amazon Web Services account\cr
#'  \link[paws.networking:route53domains_check_domain_availability]{check_domain_availability} \tab This operation checks the availability of one domain name\cr
#'  \link[paws.networking:route53domains_check_domain_transferability]{check_domain_transferability} \tab Checks whether a domain name can be transferred to Amazon Route 53\cr
#'  \link[paws.networking:route53domains_delete_domain]{delete_domain} \tab This operation deletes the specified domain\cr
#'  \link[paws.networking:route53domains_delete_tags_for_domain]{delete_tags_for_domain} \tab This operation deletes the specified tags for a domain\cr
#'  \link[paws.networking:route53domains_disable_domain_auto_renew]{disable_domain_auto_renew} \tab This operation disables automatic renewal of domain registration for the specified domain\cr
#'  \link[paws.networking:route53domains_disable_domain_transfer_lock]{disable_domain_transfer_lock} \tab This operation removes the transfer lock on the domain (specifically the clientTransferProhibited status) to allow domain transfers\cr
#'  \link[paws.networking:route53domains_disassociate_delegation_signer_from_domain]{disassociate_delegation_signer_from_domain} \tab Deletes a delegation signer (DS) record in the registry zone for this domain name\cr
#'  \link[paws.networking:route53domains_enable_domain_auto_renew]{enable_domain_auto_renew} \tab This operation configures Amazon Route 53 to automatically renew the specified domain before the domain registration expires\cr
#'  \link[paws.networking:route53domains_enable_domain_transfer_lock]{enable_domain_transfer_lock} \tab This operation sets the transfer lock on the domain (specifically the clientTransferProhibited status) to prevent domain transfers\cr
#'  \link[paws.networking:route53domains_get_contact_reachability_status]{get_contact_reachability_status} \tab For operations that require confirmation that the email address for the registrant contact is valid, such as registering a new domain, this operation returns information about whether the registrant contact has responded\cr
#'  \link[paws.networking:route53domains_get_domain_detail]{get_domain_detail} \tab This operation returns detailed information about a specified domain that is associated with the current Amazon Web Services account\cr
#'  \link[paws.networking:route53domains_get_domain_suggestions]{get_domain_suggestions} \tab The GetDomainSuggestions operation returns a list of suggested domain names\cr
#'  \link[paws.networking:route53domains_get_operation_detail]{get_operation_detail} \tab This operation returns the current status of an operation that is not completed\cr
#'  \link[paws.networking:route53domains_list_domains]{list_domains} \tab This operation returns all the domain names registered with Amazon Route 53 for the current Amazon Web Services account if no filtering conditions are used\cr
#'  \link[paws.networking:route53domains_list_operations]{list_operations} \tab Returns information about all of the operations that return an operation ID and that have ever been performed on domains that were registered by the current account\cr
#'  \link[paws.networking:route53domains_list_prices]{list_prices} \tab Lists the following prices for either all the TLDs supported by Route 53, or the specified TLD:\cr
#'  \link[paws.networking:route53domains_list_tags_for_domain]{list_tags_for_domain} \tab This operation returns all of the tags that are associated with the specified domain\cr
#'  \link[paws.networking:route53domains_push_domain]{push_domain} \tab Moves a domain from Amazon Web Services to another registrar\cr
#'  \link[paws.networking:route53domains_register_domain]{register_domain} \tab This operation registers a domain\cr
#'  \link[paws.networking:route53domains_reject_domain_transfer_from_another_aws_account]{reject_domain_transfer_from_another_aws_account} \tab Rejects the transfer of a domain from another Amazon Web Services account to the current Amazon Web Services account\cr
#'  \link[paws.networking:route53domains_renew_domain]{renew_domain} \tab This operation renews a domain for the specified number of years\cr
#'  \link[paws.networking:route53domains_resend_contact_reachability_email]{resend_contact_reachability_email} \tab For operations that require confirmation that the email address for the registrant contact is valid, such as registering a new domain, this operation resends the confirmation email to the current email address for the registrant contact\cr
#'  \link[paws.networking:route53domains_resend_operation_authorization]{resend_operation_authorization} \tab Resend the form of authorization email for this operation\cr
#'  \link[paws.networking:route53domains_retrieve_domain_auth_code]{retrieve_domain_auth_code} \tab This operation returns the authorization code for the domain\cr
#'  \link[paws.networking:route53domains_transfer_domain]{transfer_domain} \tab Transfers a domain from another registrar to Amazon Route 53\cr
#'  \link[paws.networking:route53domains_transfer_domain_to_another_aws_account]{transfer_domain_to_another_aws_account} \tab Transfers a domain from the current Amazon Web Services account to another Amazon Web Services account\cr
#'  \link[paws.networking:route53domains_update_domain_contact]{update_domain_contact} \tab This operation updates the contact information for a particular domain\cr
#'  \link[paws.networking:route53domains_update_domain_contact_privacy]{update_domain_contact_privacy} \tab This operation updates the specified domain contact's privacy setting\cr
#'  \link[paws.networking:route53domains_update_domain_nameservers]{update_domain_nameservers} \tab This operation replaces the current set of name servers for the domain with the specified set of name servers\cr
#'  \link[paws.networking:route53domains_update_tags_for_domain]{update_tags_for_domain} \tab This operation adds or updates tags for a specified domain\cr
#'  \link[paws.networking:route53domains_view_billing]{view_billing} \tab Returns all the domain-related billing records for the current Amazon Web Services account for a specified period
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname route53domains
#' @export
route53domains <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.networking::route53domains(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' Route53 Recovery Cluster
#'
#' @description
#' Welcome to the Routing Control (Recovery Cluster) API Reference Guide
#' for Amazon Route 53 Application Recovery Controller.
#'
#' With Route 53 ARC, you can use routing control with extreme reliability
#' to recover applications by rerouting traffic across Availability Zones
#' or Amazon Web Services Regions. Routing controls are simple on/off
#' switches hosted on a highly available cluster in Route 53 ARC. A cluster
#' provides a set of five redundant Regional endpoints against which you
#' can run API calls to get or update the state of routing controls. To
#' implement failover, you set one routing control On and another one Off,
#' to reroute traffic from one Availability Zone or Amazon Web Services
#' Region to another.
#'
#' *Be aware that you must specify a Regional endpoint for a cluster when
#' you work with API cluster operations to get or update routing control
#' states in Route 53 ARC.* In addition, you must specify the US West
#' (Oregon) Region for Route 53 ARC API calls. For example, use the
#' parameter `--region us-west-2` with AWS CLI commands. For more
#' information, see [Get and update routing control states using the
#' API](https://docs.aws.amazon.com/r53recovery/latest/dg/routing-control.update.api.html)
#' in the Amazon Route 53 Application Recovery Controller Developer Guide.
#'
#' This API guide includes information about the API operations for how to
#' get and update routing control states in Route 53 ARC. To work with
#' routing control in Route 53 ARC, you must first create the required
#' components (clusters, control panels, and routing controls) using the
#' recovery cluster configuration API.
#'
#' For more information about working with routing control in Route 53 ARC,
#' see the following:
#'
#' -   Create clusters, control panels, and routing controls by using API
#'     operations. For more information, see the [Recovery Control
#'     Configuration API Reference Guide for Amazon Route 53 Application
#'     Recovery
#'     Controller](https://docs.aws.amazon.com/recovery-cluster/latest/api/).
#'
#' -   Learn about the components in recovery control, including clusters,
#'     routing controls, and control panels, and how to work with Route 53
#'     ARC in the Amazon Web Services console. For more information, see
#'     [Recovery control
#'     components](https://docs.aws.amazon.com/r53recovery/latest/dg/introduction-components.html#introduction-components-routing)
#'     in the Amazon Route 53 Application Recovery Controller Developer
#'     Guide.
#'
#' -   Route 53 ARC also provides readiness checks that continually audit
#'     resources to help make sure that your applications are scaled and
#'     ready to handle failover traffic. For more information about the
#'     related API operations, see the [Recovery Readiness API Reference
#'     Guide for Amazon Route 53 Application Recovery
#'     Controller](https://docs.aws.amazon.com/recovery-readiness/latest/api/).
#'
#' -   For more information about creating resilient applications and
#'     preparing for recovery readiness with Route 53 ARC, see the [Amazon
#'     Route 53 Application Recovery Controller Developer
#'     Guide](https://docs.aws.amazon.com/r53recovery/latest/dg/).
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- route53recoverycluster(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- route53recoverycluster()
#' svc$get_routing_control_state(
#'   Foo = 123
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.networking:route53recoverycluster_get_routing_control_state]{get_routing_control_state} \tab Get the state for a routing control\cr
#'  \link[paws.networking:route53recoverycluster_list_routing_controls]{list_routing_controls} \tab List routing control names and Amazon Resource Names (ARNs), as well as the routing control state for each routing control, along with the control panel name and control panel ARN for the routing controls\cr
#'  \link[paws.networking:route53recoverycluster_update_routing_control_state]{update_routing_control_state} \tab Set the state of the routing control to reroute traffic\cr
#'  \link[paws.networking:route53recoverycluster_update_routing_control_states]{update_routing_control_states} \tab Set multiple routing control states
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname route53recoverycluster
#' @export
route53recoverycluster <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.networking::route53recoverycluster(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' AWS Route53 Recovery Control Config
#'
#' @description
#' Recovery Control Configuration API Reference for Amazon Route 53
#' Application Recovery Controller
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- route53recoverycontrolconfig(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- route53recoverycontrolconfig()
#' svc$create_cluster(
#'   Foo = 123
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.networking:route53recoverycontrolconfig_create_cluster]{create_cluster} \tab Create a new cluster\cr
#'  \link[paws.networking:route53recoverycontrolconfig_create_control_panel]{create_control_panel} \tab Creates a new control panel\cr
#'  \link[paws.networking:route53recoverycontrolconfig_create_routing_control]{create_routing_control} \tab Creates a new routing control\cr
#'  \link[paws.networking:route53recoverycontrolconfig_create_safety_rule]{create_safety_rule} \tab Creates a safety rule in a control panel\cr
#'  \link[paws.networking:route53recoverycontrolconfig_delete_cluster]{delete_cluster} \tab Delete a cluster\cr
#'  \link[paws.networking:route53recoverycontrolconfig_delete_control_panel]{delete_control_panel} \tab Deletes a control panel\cr
#'  \link[paws.networking:route53recoverycontrolconfig_delete_routing_control]{delete_routing_control} \tab Deletes a routing control\cr
#'  \link[paws.networking:route53recoverycontrolconfig_delete_safety_rule]{delete_safety_rule} \tab Deletes a safety rule\cr
#'  \link[paws.networking:route53recoverycontrolconfig_describe_cluster]{describe_cluster} \tab Display the details about a cluster\cr
#'  \link[paws.networking:route53recoverycontrolconfig_describe_control_panel]{describe_control_panel} \tab Displays details about a control panel\cr
#'  \link[paws.networking:route53recoverycontrolconfig_describe_routing_control]{describe_routing_control} \tab Displays details about a routing control\cr
#'  \link[paws.networking:route53recoverycontrolconfig_describe_safety_rule]{describe_safety_rule} \tab Returns information about a safety rule\cr
#'  \link[paws.networking:route53recoverycontrolconfig_list_associated_route_53_health_checks]{list_associated_route_53_health_checks} \tab Returns an array of all Amazon Route 53 health checks associated with a specific routing control\cr
#'  \link[paws.networking:route53recoverycontrolconfig_list_clusters]{list_clusters} \tab Returns an array of all the clusters in an account\cr
#'  \link[paws.networking:route53recoverycontrolconfig_list_control_panels]{list_control_panels} \tab Returns an array of control panels in an account or in a cluster\cr
#'  \link[paws.networking:route53recoverycontrolconfig_list_routing_controls]{list_routing_controls} \tab Returns an array of routing controls for a control panel\cr
#'  \link[paws.networking:route53recoverycontrolconfig_list_safety_rules]{list_safety_rules} \tab List the safety rules (the assertion rules and gating rules) that you've defined for the routing controls in a control panel\cr
#'  \link[paws.networking:route53recoverycontrolconfig_list_tags_for_resource]{list_tags_for_resource} \tab Lists the tags for a resource\cr
#'  \link[paws.networking:route53recoverycontrolconfig_tag_resource]{tag_resource} \tab Adds a tag to a resource\cr
#'  \link[paws.networking:route53recoverycontrolconfig_untag_resource]{untag_resource} \tab Removes a tag from a resource\cr
#'  \link[paws.networking:route53recoverycontrolconfig_update_control_panel]{update_control_panel} \tab Updates a control panel\cr
#'  \link[paws.networking:route53recoverycontrolconfig_update_routing_control]{update_routing_control} \tab Updates a routing control\cr
#'  \link[paws.networking:route53recoverycontrolconfig_update_safety_rule]{update_safety_rule} \tab Update a safety rule (an assertion rule or gating rule)
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname route53recoverycontrolconfig
#' @export
route53recoverycontrolconfig <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.networking::route53recoverycontrolconfig(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' AWS Route53 Recovery Readiness
#'
#' @description
#' Recovery readiness
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- route53recoveryreadiness(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- route53recoveryreadiness()
#' svc$create_cell(
#'   Foo = 123
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.networking:route53recoveryreadiness_create_cell]{create_cell} \tab Creates a cell in an account\cr
#'  \link[paws.networking:route53recoveryreadiness_create_cross_account_authorization]{create_cross_account_authorization} \tab Creates a cross-account readiness authorization\cr
#'  \link[paws.networking:route53recoveryreadiness_create_readiness_check]{create_readiness_check} \tab Creates a readiness check in an account\cr
#'  \link[paws.networking:route53recoveryreadiness_create_recovery_group]{create_recovery_group} \tab Creates a recovery group in an account\cr
#'  \link[paws.networking:route53recoveryreadiness_create_resource_set]{create_resource_set} \tab Creates a resource set\cr
#'  \link[paws.networking:route53recoveryreadiness_delete_cell]{delete_cell} \tab Delete a cell\cr
#'  \link[paws.networking:route53recoveryreadiness_delete_cross_account_authorization]{delete_cross_account_authorization} \tab Deletes cross account readiness authorization\cr
#'  \link[paws.networking:route53recoveryreadiness_delete_readiness_check]{delete_readiness_check} \tab Deletes a readiness check\cr
#'  \link[paws.networking:route53recoveryreadiness_delete_recovery_group]{delete_recovery_group} \tab Deletes a recovery group\cr
#'  \link[paws.networking:route53recoveryreadiness_delete_resource_set]{delete_resource_set} \tab Deletes a resource set\cr
#'  \link[paws.networking:route53recoveryreadiness_get_architecture_recommendations]{get_architecture_recommendations} \tab Gets recommendations about architecture designs for improving resiliency for an application, based on a recovery group\cr
#'  \link[paws.networking:route53recoveryreadiness_get_cell]{get_cell} \tab Gets information about a cell including cell name, cell Amazon Resource Name (ARN), ARNs of nested cells for this cell, and a list of those cell ARNs with their associated recovery group ARNs\cr
#'  \link[paws.networking:route53recoveryreadiness_get_cell_readiness_summary]{get_cell_readiness_summary} \tab Gets readiness for a cell\cr
#'  \link[paws.networking:route53recoveryreadiness_get_readiness_check]{get_readiness_check} \tab Gets details about a readiness check\cr
#'  \link[paws.networking:route53recoveryreadiness_get_readiness_check_resource_status]{get_readiness_check_resource_status} \tab Gets individual readiness status for a readiness check\cr
#'  \link[paws.networking:route53recoveryreadiness_get_readiness_check_status]{get_readiness_check_status} \tab Gets the readiness status for an individual readiness check\cr
#'  \link[paws.networking:route53recoveryreadiness_get_recovery_group]{get_recovery_group} \tab Gets details about a recovery group, including a list of the cells that are included in it\cr
#'  \link[paws.networking:route53recoveryreadiness_get_recovery_group_readiness_summary]{get_recovery_group_readiness_summary} \tab Displays a summary of information about a recovery group's readiness status\cr
#'  \link[paws.networking:route53recoveryreadiness_get_resource_set]{get_resource_set} \tab Displays the details about a resource set, including a list of the resources in the set\cr
#'  \link[paws.networking:route53recoveryreadiness_list_cells]{list_cells} \tab Lists the cells for an account\cr
#'  \link[paws.networking:route53recoveryreadiness_list_cross_account_authorizations]{list_cross_account_authorizations} \tab Lists the cross-account readiness authorizations that are in place for an account\cr
#'  \link[paws.networking:route53recoveryreadiness_list_readiness_checks]{list_readiness_checks} \tab Lists the readiness checks for an account\cr
#'  \link[paws.networking:route53recoveryreadiness_list_recovery_groups]{list_recovery_groups} \tab Lists the recovery groups in an account\cr
#'  \link[paws.networking:route53recoveryreadiness_list_resource_sets]{list_resource_sets} \tab Lists the resource sets in an account\cr
#'  \link[paws.networking:route53recoveryreadiness_list_rules]{list_rules} \tab Lists all readiness rules, or lists the readiness rules for a specific resource type\cr
#'  \link[paws.networking:route53recoveryreadiness_list_tags_for_resources]{list_tags_for_resources} \tab Lists the tags for a resource\cr
#'  \link[paws.networking:route53recoveryreadiness_tag_resource]{tag_resource} \tab Adds a tag to a resource\cr
#'  \link[paws.networking:route53recoveryreadiness_untag_resource]{untag_resource} \tab Removes a tag from a resource\cr
#'  \link[paws.networking:route53recoveryreadiness_update_cell]{update_cell} \tab Updates a cell to replace the list of nested cells with a new list of nested cells\cr
#'  \link[paws.networking:route53recoveryreadiness_update_readiness_check]{update_readiness_check} \tab Updates a readiness check\cr
#'  \link[paws.networking:route53recoveryreadiness_update_recovery_group]{update_recovery_group} \tab Updates a recovery group\cr
#'  \link[paws.networking:route53recoveryreadiness_update_resource_set]{update_resource_set} \tab Updates a resource set
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname route53recoveryreadiness
#' @export
route53recoveryreadiness <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.networking::route53recoveryreadiness(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' Amazon Route 53 Resolver
#'
#' @description
#' When you create a VPC using Amazon VPC, you automatically get DNS
#' resolution within the VPC from Route 53 Resolver. By default, Resolver
#' answers DNS queries for VPC domain names such as domain names for EC2
#' instances or Elastic Load Balancing load balancers. Resolver performs
#' recursive lookups against public name servers for all other domain
#' names.
#'
#' You can also configure DNS resolution between your VPC and your network
#' over a Direct Connect or VPN connection:
#'
#' **Forward DNS queries from resolvers on your network to Route 53
#' Resolver**
#'
#' DNS resolvers on your network can forward DNS queries to Resolver in a
#' specified VPC. This allows your DNS resolvers to easily resolve domain
#' names for Amazon Web Services resources such as EC2 instances or records
#' in a Route 53 private hosted zone. For more information, see [How DNS
#' Resolvers on Your Network Forward DNS Queries to Route 53
#' Resolver](https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/resolver.html#resolver-overview-forward-network-to-vpc)
#' in the *Amazon Route 53 Developer Guide*.
#'
#' **Conditionally forward queries from a VPC to resolvers on your
#' network**
#'
#' You can configure Resolver to forward queries that it receives from EC2
#' instances in your VPCs to DNS resolvers on your network. To forward
#' selected queries, you create Resolver rules that specify the domain
#' names for the DNS queries that you want to forward (such as
#' example.com), and the IP addresses of the DNS resolvers on your network
#' that you want to forward the queries to. If a query matches multiple
#' rules (example.com, acme.example.com), Resolver chooses the rule with
#' the most specific match (acme.example.com) and forwards the query to the
#' IP addresses that you specified in that rule. For more information, see
#' [How Route 53 Resolver Forwards DNS Queries from Your VPCs to Your
#' Network](https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/resolver.html#resolver-overview-forward-vpc-to-network)
#' in the *Amazon Route 53 Developer Guide*.
#'
#' Like Amazon VPC, Resolver is Regional. In each Region where you have
#' VPCs, you can choose whether to forward queries from your VPCs to your
#' network (outbound queries), from your network to your VPCs (inbound
#' queries), or both.
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- route53resolver(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- route53resolver()
#' svc$associate_firewall_rule_group(
#'   Foo = 123
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.networking:route53resolver_associate_firewall_rule_group]{associate_firewall_rule_group} \tab Associates a FirewallRuleGroup with a VPC, to provide DNS filtering for the VPC\cr
#'  \link[paws.networking:route53resolver_associate_resolver_endpoint_ip_address]{associate_resolver_endpoint_ip_address} \tab Adds IP addresses to an inbound or an outbound Resolver endpoint\cr
#'  \link[paws.networking:route53resolver_associate_resolver_query_log_config]{associate_resolver_query_log_config} \tab Associates an Amazon VPC with a specified query logging configuration\cr
#'  \link[paws.networking:route53resolver_associate_resolver_rule]{associate_resolver_rule} \tab Associates a Resolver rule with a VPC\cr
#'  \link[paws.networking:route53resolver_create_firewall_domain_list]{create_firewall_domain_list} \tab Creates an empty firewall domain list for use in DNS Firewall rules\cr
#'  \link[paws.networking:route53resolver_create_firewall_rule]{create_firewall_rule} \tab Creates a single DNS Firewall rule in the specified rule group, using the specified domain list\cr
#'  \link[paws.networking:route53resolver_create_firewall_rule_group]{create_firewall_rule_group} \tab Creates an empty DNS Firewall rule group for filtering DNS network traffic in a VPC\cr
#'  \link[paws.networking:route53resolver_create_outpost_resolver]{create_outpost_resolver} \tab Creates an Route 53 Resolver on an Outpost\cr
#'  \link[paws.networking:route53resolver_create_resolver_endpoint]{create_resolver_endpoint} \tab Creates a Resolver endpoint\cr
#'  \link[paws.networking:route53resolver_create_resolver_query_log_config]{create_resolver_query_log_config} \tab Creates a Resolver query logging configuration, which defines where you want Resolver to save DNS query logs that originate in your VPCs\cr
#'  \link[paws.networking:route53resolver_create_resolver_rule]{create_resolver_rule} \tab For DNS queries that originate in your VPCs, specifies which Resolver endpoint the queries pass through, one domain name that you want to forward to your network, and the IP addresses of the DNS resolvers in your network\cr
#'  \link[paws.networking:route53resolver_delete_firewall_domain_list]{delete_firewall_domain_list} \tab Deletes the specified domain list\cr
#'  \link[paws.networking:route53resolver_delete_firewall_rule]{delete_firewall_rule} \tab Deletes the specified firewall rule\cr
#'  \link[paws.networking:route53resolver_delete_firewall_rule_group]{delete_firewall_rule_group} \tab Deletes the specified firewall rule group\cr
#'  \link[paws.networking:route53resolver_delete_outpost_resolver]{delete_outpost_resolver} \tab Deletes a Resolver on the Outpost\cr
#'  \link[paws.networking:route53resolver_delete_resolver_endpoint]{delete_resolver_endpoint} \tab Deletes a Resolver endpoint\cr
#'  \link[paws.networking:route53resolver_delete_resolver_query_log_config]{delete_resolver_query_log_config} \tab Deletes a query logging configuration\cr
#'  \link[paws.networking:route53resolver_delete_resolver_rule]{delete_resolver_rule} \tab Deletes a Resolver rule\cr
#'  \link[paws.networking:route53resolver_disassociate_firewall_rule_group]{disassociate_firewall_rule_group} \tab Disassociates a FirewallRuleGroup from a VPC, to remove DNS filtering from the VPC\cr
#'  \link[paws.networking:route53resolver_disassociate_resolver_endpoint_ip_address]{disassociate_resolver_endpoint_ip_address} \tab Removes IP addresses from an inbound or an outbound Resolver endpoint\cr
#'  \link[paws.networking:route53resolver_disassociate_resolver_query_log_config]{disassociate_resolver_query_log_config} \tab Disassociates a VPC from a query logging configuration\cr
#'  \link[paws.networking:route53resolver_disassociate_resolver_rule]{disassociate_resolver_rule} \tab Removes the association between a specified Resolver rule and a specified VPC\cr
#'  \link[paws.networking:route53resolver_get_firewall_config]{get_firewall_config} \tab Retrieves the configuration of the firewall behavior provided by DNS Firewall for a single VPC from Amazon Virtual Private Cloud (Amazon VPC)\cr
#'  \link[paws.networking:route53resolver_get_firewall_domain_list]{get_firewall_domain_list} \tab Retrieves the specified firewall domain list\cr
#'  \link[paws.networking:route53resolver_get_firewall_rule_group]{get_firewall_rule_group} \tab Retrieves the specified firewall rule group\cr
#'  \link[paws.networking:route53resolver_get_firewall_rule_group_association]{get_firewall_rule_group_association} \tab Retrieves a firewall rule group association, which enables DNS filtering for a VPC with one rule group\cr
#'  \link[paws.networking:route53resolver_get_firewall_rule_group_policy]{get_firewall_rule_group_policy} \tab Returns the Identity and Access Management (Amazon Web Services IAM) policy for sharing the specified rule group\cr
#'  \link[paws.networking:route53resolver_get_outpost_resolver]{get_outpost_resolver} \tab Gets information about a specified Resolver on the Outpost, such as its instance count and type, name, and the current status of the Resolver\cr
#'  \link[paws.networking:route53resolver_get_resolver_config]{get_resolver_config} \tab Retrieves the behavior configuration of Route 53 Resolver behavior for a single VPC from Amazon Virtual Private Cloud\cr
#'  \link[paws.networking:route53resolver_get_resolver_dnssec_config]{get_resolver_dnssec_config} \tab Gets DNSSEC validation information for a specified resource\cr
#'  \link[paws.networking:route53resolver_get_resolver_endpoint]{get_resolver_endpoint} \tab Gets information about a specified Resolver endpoint, such as whether it's an inbound or an outbound Resolver endpoint, and the current status of the endpoint\cr
#'  \link[paws.networking:route53resolver_get_resolver_query_log_config]{get_resolver_query_log_config} \tab Gets information about a specified Resolver query logging configuration, such as the number of VPCs that the configuration is logging queries for and the location that logs are sent to\cr
#'  \link[paws.networking:route53resolver_get_resolver_query_log_config_association]{get_resolver_query_log_config_association} \tab Gets information about a specified association between a Resolver query logging configuration and an Amazon VPC\cr
#'  \link[paws.networking:route53resolver_get_resolver_query_log_config_policy]{get_resolver_query_log_config_policy} \tab Gets information about a query logging policy\cr
#'  \link[paws.networking:route53resolver_get_resolver_rule]{get_resolver_rule} \tab Gets information about a specified Resolver rule, such as the domain name that the rule forwards DNS queries for and the ID of the outbound Resolver endpoint that the rule is associated with\cr
#'  \link[paws.networking:route53resolver_get_resolver_rule_association]{get_resolver_rule_association} \tab Gets information about an association between a specified Resolver rule and a VPC\cr
#'  \link[paws.networking:route53resolver_get_resolver_rule_policy]{get_resolver_rule_policy} \tab Gets information about the Resolver rule policy for a specified rule\cr
#'  \link[paws.networking:route53resolver_import_firewall_domains]{import_firewall_domains} \tab Imports domain names from a file into a domain list, for use in a DNS firewall rule group\cr
#'  \link[paws.networking:route53resolver_list_firewall_configs]{list_firewall_configs} \tab Retrieves the firewall configurations that you have defined\cr
#'  \link[paws.networking:route53resolver_list_firewall_domain_lists]{list_firewall_domain_lists} \tab Retrieves the firewall domain lists that you have defined\cr
#'  \link[paws.networking:route53resolver_list_firewall_domains]{list_firewall_domains} \tab Retrieves the domains that you have defined for the specified firewall domain list\cr
#'  \link[paws.networking:route53resolver_list_firewall_rule_group_associations]{list_firewall_rule_group_associations} \tab Retrieves the firewall rule group associations that you have defined\cr
#'  \link[paws.networking:route53resolver_list_firewall_rule_groups]{list_firewall_rule_groups} \tab Retrieves the minimal high-level information for the rule groups that you have defined\cr
#'  \link[paws.networking:route53resolver_list_firewall_rules]{list_firewall_rules} \tab Retrieves the firewall rules that you have defined for the specified firewall rule group\cr
#'  \link[paws.networking:route53resolver_list_outpost_resolvers]{list_outpost_resolvers} \tab Lists all the Resolvers on Outposts that were created using the current Amazon Web Services account\cr
#'  \link[paws.networking:route53resolver_list_resolver_configs]{list_resolver_configs} \tab Retrieves the Resolver configurations that you have defined\cr
#'  \link[paws.networking:route53resolver_list_resolver_dnssec_configs]{list_resolver_dnssec_configs} \tab Lists the configurations for DNSSEC validation that are associated with the current Amazon Web Services account\cr
#'  \link[paws.networking:route53resolver_list_resolver_endpoint_ip_addresses]{list_resolver_endpoint_ip_addresses} \tab Gets the IP addresses for a specified Resolver endpoint\cr
#'  \link[paws.networking:route53resolver_list_resolver_endpoints]{list_resolver_endpoints} \tab Lists all the Resolver endpoints that were created using the current Amazon Web Services account\cr
#'  \link[paws.networking:route53resolver_list_resolver_query_log_config_associations]{list_resolver_query_log_config_associations} \tab Lists information about associations between Amazon VPCs and query logging configurations\cr
#'  \link[paws.networking:route53resolver_list_resolver_query_log_configs]{list_resolver_query_log_configs} \tab Lists information about the specified query logging configurations\cr
#'  \link[paws.networking:route53resolver_list_resolver_rule_associations]{list_resolver_rule_associations} \tab Lists the associations that were created between Resolver rules and VPCs using the current Amazon Web Services account\cr
#'  \link[paws.networking:route53resolver_list_resolver_rules]{list_resolver_rules} \tab Lists the Resolver rules that were created using the current Amazon Web Services account\cr
#'  \link[paws.networking:route53resolver_list_tags_for_resource]{list_tags_for_resource} \tab Lists the tags that you associated with the specified resource\cr
#'  \link[paws.networking:route53resolver_put_firewall_rule_group_policy]{put_firewall_rule_group_policy} \tab Attaches an Identity and Access Management (Amazon Web Services IAM) policy for sharing the rule group\cr
#'  \link[paws.networking:route53resolver_put_resolver_query_log_config_policy]{put_resolver_query_log_config_policy} \tab Specifies an Amazon Web Services account that you want to share a query logging configuration with, the query logging configuration that you want to share, and the operations that you want the account to be able to perform on the configuration\cr
#'  \link[paws.networking:route53resolver_put_resolver_rule_policy]{put_resolver_rule_policy} \tab Specifies an Amazon Web Services rule that you want to share with another account, the account that you want to share the rule with, and the operations that you want the account to be able to perform on the rule\cr
#'  \link[paws.networking:route53resolver_tag_resource]{tag_resource} \tab Adds one or more tags to a specified resource\cr
#'  \link[paws.networking:route53resolver_untag_resource]{untag_resource} \tab Removes one or more tags from a specified resource\cr
#'  \link[paws.networking:route53resolver_update_firewall_config]{update_firewall_config} \tab Updates the configuration of the firewall behavior provided by DNS Firewall for a single VPC from Amazon Virtual Private Cloud (Amazon VPC)\cr
#'  \link[paws.networking:route53resolver_update_firewall_domains]{update_firewall_domains} \tab Updates the firewall domain list from an array of domain specifications\cr
#'  \link[paws.networking:route53resolver_update_firewall_rule]{update_firewall_rule} \tab Updates the specified firewall rule\cr
#'  \link[paws.networking:route53resolver_update_firewall_rule_group_association]{update_firewall_rule_group_association} \tab Changes the association of a FirewallRuleGroup with a VPC\cr
#'  \link[paws.networking:route53resolver_update_outpost_resolver]{update_outpost_resolver} \tab You can use UpdateOutpostResolver to update the instance count, type, or name of a Resolver on an Outpost\cr
#'  \link[paws.networking:route53resolver_update_resolver_config]{update_resolver_config} \tab Updates the behavior configuration of Route 53 Resolver behavior for a single VPC from Amazon Virtual Private Cloud\cr
#'  \link[paws.networking:route53resolver_update_resolver_dnssec_config]{update_resolver_dnssec_config} \tab Updates an existing DNSSEC validation configuration\cr
#'  \link[paws.networking:route53resolver_update_resolver_endpoint]{update_resolver_endpoint} \tab Updates the name, or enpoint type for an inbound or an outbound Resolver endpoint\cr
#'  \link[paws.networking:route53resolver_update_resolver_rule]{update_resolver_rule} \tab Updates settings for a specified Resolver rule
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname route53resolver
#' @export
route53resolver <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.networking::route53resolver(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' AWS Cloud Map
#'
#' @description
#' Cloud Map
#'
#' With Cloud Map, you can configure public DNS, private DNS, or HTTP
#' namespaces that your microservice applications run in. When an instance
#' becomes available, you can call the Cloud Map API to register the
#' instance with Cloud Map. For public or private DNS namespaces, Cloud Map
#' automatically creates DNS records and an optional health check. Clients
#' that submit public or private DNS queries, or HTTP requests, for the
#' service receive an answer that contains up to eight healthy records.
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- servicediscovery(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- servicediscovery()
#' # This example creates an HTTP namespace.
#' svc$create_http_namespace(
#'   CreatorRequestId = "example-creator-request-id-0001",
#'   Description = "Example.com AWS Cloud Map HTTP Namespace",
#'   Name = "example-http.com"
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.networking:servicediscovery_create_http_namespace]{create_http_namespace} \tab Creates an HTTP namespace\cr
#'  \link[paws.networking:servicediscovery_create_private_dns_namespace]{create_private_dns_namespace} \tab Creates a private namespace based on DNS, which is visible only inside a specified Amazon VPC\cr
#'  \link[paws.networking:servicediscovery_create_public_dns_namespace]{create_public_dns_namespace} \tab Creates a public namespace based on DNS, which is visible on the internet\cr
#'  \link[paws.networking:servicediscovery_create_service]{create_service} \tab Creates a service\cr
#'  \link[paws.networking:servicediscovery_delete_namespace]{delete_namespace} \tab Deletes a namespace from the current account\cr
#'  \link[paws.networking:servicediscovery_delete_service]{delete_service} \tab Deletes a specified service\cr
#'  \link[paws.networking:servicediscovery_deregister_instance]{deregister_instance} \tab Deletes the Amazon Route 53 DNS records and health check, if any, that Cloud Map created for the specified instance\cr
#'  \link[paws.networking:servicediscovery_discover_instances]{discover_instances} \tab Discovers registered instances for a specified namespace and service\cr
#'  \link[paws.networking:servicediscovery_get_instance]{get_instance} \tab Gets information about a specified instance\cr
#'  \link[paws.networking:servicediscovery_get_instances_health_status]{get_instances_health_status} \tab Gets the current health status (Healthy, Unhealthy, or Unknown) of one or more instances that are associated with a specified service\cr
#'  \link[paws.networking:servicediscovery_get_namespace]{get_namespace} \tab Gets information about a namespace\cr
#'  \link[paws.networking:servicediscovery_get_operation]{get_operation} \tab Gets information about any operation that returns an operation ID in the response, such as a CreateService request\cr
#'  \link[paws.networking:servicediscovery_get_service]{get_service} \tab Gets the settings for a specified service\cr
#'  \link[paws.networking:servicediscovery_list_instances]{list_instances} \tab Lists summary information about the instances that you registered by using a specified service\cr
#'  \link[paws.networking:servicediscovery_list_namespaces]{list_namespaces} \tab Lists summary information about the namespaces that were created by the current Amazon Web Services account\cr
#'  \link[paws.networking:servicediscovery_list_operations]{list_operations} \tab Lists operations that match the criteria that you specify\cr
#'  \link[paws.networking:servicediscovery_list_services]{list_services} \tab Lists summary information for all the services that are associated with one or more specified namespaces\cr
#'  \link[paws.networking:servicediscovery_list_tags_for_resource]{list_tags_for_resource} \tab Lists tags for the specified resource\cr
#'  \link[paws.networking:servicediscovery_register_instance]{register_instance} \tab Creates or updates one or more records and, optionally, creates a health check based on the settings in a specified service\cr
#'  \link[paws.networking:servicediscovery_tag_resource]{tag_resource} \tab Adds one or more tags to the specified resource\cr
#'  \link[paws.networking:servicediscovery_untag_resource]{untag_resource} \tab Removes one or more tags from the specified resource\cr
#'  \link[paws.networking:servicediscovery_update_http_namespace]{update_http_namespace} \tab Updates an HTTP namespace\cr
#'  \link[paws.networking:servicediscovery_update_instance_custom_health_status]{update_instance_custom_health_status} \tab Submits a request to change the health status of a custom health check to healthy or unhealthy\cr
#'  \link[paws.networking:servicediscovery_update_private_dns_namespace]{update_private_dns_namespace} \tab Updates a private DNS namespace\cr
#'  \link[paws.networking:servicediscovery_update_public_dns_namespace]{update_public_dns_namespace} \tab Updates a public DNS namespace\cr
#'  \link[paws.networking:servicediscovery_update_service]{update_service} \tab Submits a request to perform the following operations:
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname servicediscovery
#' @export
servicediscovery <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.networking::servicediscovery(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' AWS Cloud9
#'
#' @description
#' Cloud9
#'
#' Cloud9 is a collection of tools that you can use to code, build, run,
#' test, debug, and release software in the cloud.
#'
#' For more information about Cloud9, see the [Cloud9 User
#' Guide](https://docs.aws.amazon.com/cloud9/latest/user-guide/).
#'
#' Cloud9 supports these operations:
#'
#' -   [`create_environment_ec2`][cloud9_create_environment_ec2]: Creates
#'     an Cloud9 development environment, launches an Amazon EC2 instance,
#'     and then connects from the instance to the environment.
#'
#' -   [`create_environment_membership`][cloud9_create_environment_membership]:
#'     Adds an environment member to an environment.
#'
#' -   [`delete_environment`][cloud9_delete_environment]: Deletes an
#'     environment. If an Amazon EC2 instance is connected to the
#'     environment, also terminates the instance.
#'
#' -   [`delete_environment_membership`][cloud9_delete_environment_membership]:
#'     Deletes an environment member from an environment.
#'
#' -   [`describe_environment_memberships`][cloud9_describe_environment_memberships]:
#'     Gets information about environment members for an environment.
#'
#' -   [`describe_environments`][cloud9_describe_environments]: Gets
#'     information about environments.
#'
#' -   [`describe_environment_status`][cloud9_describe_environment_status]:
#'     Gets status information for an environment.
#'
#' -   [`list_environments`][cloud9_list_environments]: Gets a list of
#'     environment identifiers.
#'
#' -   [`list_tags_for_resource`][cloud9_list_tags_for_resource]: Gets the
#'     tags for an environment.
#'
#' -   [`tag_resource`][cloud9_tag_resource]: Adds tags to an environment.
#'
#' -   [`untag_resource`][cloud9_untag_resource]: Removes tags from an
#'     environment.
#'
#' -   [`update_environment`][cloud9_update_environment]: Changes the
#'     settings of an existing environment.
#'
#' -   [`update_environment_membership`][cloud9_update_environment_membership]:
#'     Changes the settings of an existing environment member for an
#'     environment.
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- cloud9(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- cloud9()
#' #
#' svc$create_environment_ec2(
#'   name = "my-demo-environment",
#'   automaticStopTimeMinutes = 60L,
#'   description = "This is my demonstration environment.",
#'   instanceType = "t2.micro",
#'   ownerArn = "arn:aws:iam::123456789012:user/MyDemoUser",
#'   subnetId = "subnet-6300cd1b"
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.developer.tools:cloud9_create_environment_ec2]{create_environment_ec2} \tab Creates an Cloud9 development environment, launches an Amazon Elastic Compute Cloud (Amazon EC2) instance, and then connects from the instance to the environment\cr
#'  \link[paws.developer.tools:cloud9_create_environment_membership]{create_environment_membership} \tab Adds an environment member to an Cloud9 development environment\cr
#'  \link[paws.developer.tools:cloud9_delete_environment]{delete_environment} \tab Deletes an Cloud9 development environment\cr
#'  \link[paws.developer.tools:cloud9_delete_environment_membership]{delete_environment_membership} \tab Deletes an environment member from a development environment\cr
#'  \link[paws.developer.tools:cloud9_describe_environment_memberships]{describe_environment_memberships} \tab Gets information about environment members for an Cloud9 development environment\cr
#'  \link[paws.developer.tools:cloud9_describe_environments]{describe_environments} \tab Gets information about Cloud9 development environments\cr
#'  \link[paws.developer.tools:cloud9_describe_environment_status]{describe_environment_status} \tab Gets status information for an Cloud9 development environment\cr
#'  \link[paws.developer.tools:cloud9_list_environments]{list_environments} \tab Gets a list of Cloud9 development environment identifiers\cr
#'  \link[paws.developer.tools:cloud9_list_tags_for_resource]{list_tags_for_resource} \tab Gets a list of the tags associated with an Cloud9 development environment\cr
#'  \link[paws.developer.tools:cloud9_tag_resource]{tag_resource} \tab Adds tags to an Cloud9 development environment\cr
#'  \link[paws.developer.tools:cloud9_untag_resource]{untag_resource} \tab Removes tags from an Cloud9 development environment\cr
#'  \link[paws.developer.tools:cloud9_update_environment]{update_environment} \tab Changes the settings of an existing Cloud9 development environment\cr
#'  \link[paws.developer.tools:cloud9_update_environment_membership]{update_environment_membership} \tab Changes the settings of an existing environment member for an Cloud9 development environment
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname cloud9
#' @export
cloud9 <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.developer.tools::cloud9(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' AWS Cloud Control API
#'
#' @description
#' For more information about Amazon Web Services Cloud Control API, see
#' the [Amazon Web Services Cloud Control API User
#' Guide](https://docs.aws.amazon.com/cloudcontrolapi/latest/userguide/what-is-cloudcontrolapi.html).
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- cloudcontrolapi(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- cloudcontrolapi()
#' svc$cancel_resource_request(
#'   Foo = 123
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.developer.tools:cloudcontrolapi_cancel_resource_request]{cancel_resource_request} \tab Cancels the specified resource operation request\cr
#'  \link[paws.developer.tools:cloudcontrolapi_create_resource]{create_resource} \tab Creates the specified resource\cr
#'  \link[paws.developer.tools:cloudcontrolapi_delete_resource]{delete_resource} \tab Deletes the specified resource\cr
#'  \link[paws.developer.tools:cloudcontrolapi_get_resource]{get_resource} \tab Returns information about the current state of the specified resource\cr
#'  \link[paws.developer.tools:cloudcontrolapi_get_resource_request_status]{get_resource_request_status} \tab Returns the current status of a resource operation request\cr
#'  \link[paws.developer.tools:cloudcontrolapi_list_resource_requests]{list_resource_requests} \tab Returns existing resource operation requests\cr
#'  \link[paws.developer.tools:cloudcontrolapi_list_resources]{list_resources} \tab Returns information about the specified resources\cr
#'  \link[paws.developer.tools:cloudcontrolapi_update_resource]{update_resource} \tab Updates the specified property values in the resource
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname cloudcontrolapi
#' @export
cloudcontrolapi <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.developer.tools::cloudcontrolapi(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' AWS CodeBuild
#'
#' @description
#' CodeBuild
#'
#' CodeBuild is a fully managed build service in the cloud. CodeBuild
#' compiles your source code, runs unit tests, and produces artifacts that
#' are ready to deploy. CodeBuild eliminates the need to provision, manage,
#' and scale your own build servers. It provides prepackaged build
#' environments for the most popular programming languages and build tools,
#' such as Apache Maven, Gradle, and more. You can also fully customize
#' build environments in CodeBuild to use your own build tools. CodeBuild
#' scales automatically to meet peak build requests. You pay only for the
#' build time you consume. For more information about CodeBuild, see the
#' *\href{https://docs.aws.amazon.com/codebuild/latest/userguide/welcome.html}{CodeBuild User Guide}.*
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- codebuild(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- codebuild()
#' svc$batch_delete_builds(
#'   Foo = 123
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.developer.tools:codebuild_batch_delete_builds]{batch_delete_builds} \tab Deletes one or more builds\cr
#'  \link[paws.developer.tools:codebuild_batch_get_build_batches]{batch_get_build_batches} \tab Retrieves information about one or more batch builds\cr
#'  \link[paws.developer.tools:codebuild_batch_get_builds]{batch_get_builds} \tab Gets information about one or more builds\cr
#'  \link[paws.developer.tools:codebuild_batch_get_projects]{batch_get_projects} \tab Gets information about one or more build projects\cr
#'  \link[paws.developer.tools:codebuild_batch_get_report_groups]{batch_get_report_groups} \tab Returns an array of report groups\cr
#'  \link[paws.developer.tools:codebuild_batch_get_reports]{batch_get_reports} \tab Returns an array of reports\cr
#'  \link[paws.developer.tools:codebuild_create_project]{create_project} \tab Creates a build project\cr
#'  \link[paws.developer.tools:codebuild_create_report_group]{create_report_group} \tab Creates a report group\cr
#'  \link[paws.developer.tools:codebuild_create_webhook]{create_webhook} \tab For an existing CodeBuild build project that has its source code stored in a GitHub or Bitbucket repository, enables CodeBuild to start rebuilding the source code every time a code change is pushed to the repository\cr
#'  \link[paws.developer.tools:codebuild_delete_build_batch]{delete_build_batch} \tab Deletes a batch build\cr
#'  \link[paws.developer.tools:codebuild_delete_project]{delete_project} \tab Deletes a build project\cr
#'  \link[paws.developer.tools:codebuild_delete_report]{delete_report} \tab Deletes a report\cr
#'  \link[paws.developer.tools:codebuild_delete_report_group]{delete_report_group} \tab Deletes a report group\cr
#'  \link[paws.developer.tools:codebuild_delete_resource_policy]{delete_resource_policy} \tab Deletes a resource policy that is identified by its resource ARN\cr
#'  \link[paws.developer.tools:codebuild_delete_source_credentials]{delete_source_credentials} \tab Deletes a set of GitHub, GitHub Enterprise, or Bitbucket source credentials\cr
#'  \link[paws.developer.tools:codebuild_delete_webhook]{delete_webhook} \tab For an existing CodeBuild build project that has its source code stored in a GitHub or Bitbucket repository, stops CodeBuild from rebuilding the source code every time a code change is pushed to the repository\cr
#'  \link[paws.developer.tools:codebuild_describe_code_coverages]{describe_code_coverages} \tab Retrieves one or more code coverage reports\cr
#'  \link[paws.developer.tools:codebuild_describe_test_cases]{describe_test_cases} \tab Returns a list of details about test cases for a report\cr
#'  \link[paws.developer.tools:codebuild_get_report_group_trend]{get_report_group_trend} \tab Analyzes and accumulates test report values for the specified test reports\cr
#'  \link[paws.developer.tools:codebuild_get_resource_policy]{get_resource_policy} \tab Gets a resource policy that is identified by its resource ARN\cr
#'  \link[paws.developer.tools:codebuild_import_source_credentials]{import_source_credentials} \tab Imports the source repository credentials for an CodeBuild project that has its source code stored in a GitHub, GitHub Enterprise, or Bitbucket repository\cr
#'  \link[paws.developer.tools:codebuild_invalidate_project_cache]{invalidate_project_cache} \tab Resets the cache for a project\cr
#'  \link[paws.developer.tools:codebuild_list_build_batches]{list_build_batches} \tab Retrieves the identifiers of your build batches in the current region\cr
#'  \link[paws.developer.tools:codebuild_list_build_batches_for_project]{list_build_batches_for_project} \tab Retrieves the identifiers of the build batches for a specific project\cr
#'  \link[paws.developer.tools:codebuild_list_builds]{list_builds} \tab Gets a list of build IDs, with each build ID representing a single build\cr
#'  \link[paws.developer.tools:codebuild_list_builds_for_project]{list_builds_for_project} \tab Gets a list of build identifiers for the specified build project, with each build identifier representing a single build\cr
#'  \link[paws.developer.tools:codebuild_list_curated_environment_images]{list_curated_environment_images} \tab Gets information about Docker images that are managed by CodeBuild\cr
#'  \link[paws.developer.tools:codebuild_list_projects]{list_projects} \tab Gets a list of build project names, with each build project name representing a single build project\cr
#'  \link[paws.developer.tools:codebuild_list_report_groups]{list_report_groups} \tab Gets a list ARNs for the report groups in the current Amazon Web Services account\cr
#'  \link[paws.developer.tools:codebuild_list_reports]{list_reports} \tab Returns a list of ARNs for the reports in the current Amazon Web Services account\cr
#'  \link[paws.developer.tools:codebuild_list_reports_for_report_group]{list_reports_for_report_group} \tab Returns a list of ARNs for the reports that belong to a ReportGroup\cr
#'  \link[paws.developer.tools:codebuild_list_shared_projects]{list_shared_projects} \tab Gets a list of projects that are shared with other Amazon Web Services accounts or users\cr
#'  \link[paws.developer.tools:codebuild_list_shared_report_groups]{list_shared_report_groups} \tab Gets a list of report groups that are shared with other Amazon Web Services accounts or users\cr
#'  \link[paws.developer.tools:codebuild_list_source_credentials]{list_source_credentials} \tab Returns a list of SourceCredentialsInfo objects\cr
#'  \link[paws.developer.tools:codebuild_put_resource_policy]{put_resource_policy} \tab Stores a resource policy for the ARN of a Project or ReportGroup object\cr
#'  \link[paws.developer.tools:codebuild_retry_build]{retry_build} \tab Restarts a build\cr
#'  \link[paws.developer.tools:codebuild_retry_build_batch]{retry_build_batch} \tab Restarts a failed batch build\cr
#'  \link[paws.developer.tools:codebuild_start_build]{start_build} \tab Starts running a build\cr
#'  \link[paws.developer.tools:codebuild_start_build_batch]{start_build_batch} \tab Starts a batch build for a project\cr
#'  \link[paws.developer.tools:codebuild_stop_build]{stop_build} \tab Attempts to stop running a build\cr
#'  \link[paws.developer.tools:codebuild_stop_build_batch]{stop_build_batch} \tab Stops a running batch build\cr
#'  \link[paws.developer.tools:codebuild_update_project]{update_project} \tab Changes the settings of a build project\cr
#'  \link[paws.developer.tools:codebuild_update_project_visibility]{update_project_visibility} \tab Changes the public visibility for a project\cr
#'  \link[paws.developer.tools:codebuild_update_report_group]{update_report_group} \tab Updates a report group\cr
#'  \link[paws.developer.tools:codebuild_update_webhook]{update_webhook} \tab Updates the webhook associated with an CodeBuild build project
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname codebuild
#' @export
codebuild <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.developer.tools::codebuild(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' CodeArtifact
#'
#' @description
#' CodeArtifact is a fully managed artifact repository compatible with
#' language-native package managers and build tools such as npm, Apache
#' Maven, pip, and dotnet. You can use CodeArtifact to share packages with
#' development teams and pull packages. Packages can be pulled from both
#' public and CodeArtifact repositories. You can also create an upstream
#' relationship between a CodeArtifact repository and another repository,
#' which effectively merges their contents from the point of view of a
#' package manager client.
#'
#' **CodeArtifact Components**
#'
#' Use the information in this guide to help you work with the following
#' CodeArtifact components:
#'
#' -   **Repository**: A CodeArtifact repository contains a set of [package
#'     versions](https://docs.aws.amazon.com/codeartifact/latest/ug/welcome.html#welcome-concepts-package-version),
#'     each of which maps to a set of assets, or files. Repositories are
#'     polyglot, so a single repository can contain packages of any
#'     supported type. Each repository exposes endpoints for fetching and
#'     publishing packages using tools like the **`npm`** CLI, the Maven
#'     CLI ( **`mvn`** ), Python CLIs ( **`pip`** and `twine`), and NuGet
#'     CLIs (`nuget` and `dotnet`).
#'
#' -   **Domain**: Repositories are aggregated into a higher-level entity
#'     known as a *domain*. All package assets and metadata are stored in
#'     the domain, but are consumed through repositories. A given package
#'     asset, such as a Maven JAR file, is stored once per domain, no
#'     matter how many repositories it's present in. All of the assets and
#'     metadata in a domain are encrypted with the same customer master key
#'     (CMK) stored in Key Management Service (KMS).
#'
#'     Each repository is a member of a single domain and can't be moved to
#'     a different domain.
#'
#'     The domain allows organizational policy to be applied across
#'     multiple repositories, such as which accounts can access
#'     repositories in the domain, and which public repositories can be
#'     used as sources of packages.
#'
#'     Although an organization can have multiple domains, we recommend a
#'     single production domain that contains all published artifacts so
#'     that teams can find and share packages across their organization.
#'
#' -   **Package**: A *package* is a bundle of software and the metadata
#'     required to resolve dependencies and install the software.
#'     CodeArtifact supports
#'     [npm](https://docs.aws.amazon.com/codeartifact/latest/ug/using-npm.html),
#'     [PyPI](https://docs.aws.amazon.com/codeartifact/latest/ug/using-python.html),
#'     [Maven](https://docs.aws.amazon.com/codeartifact/latest/ug/using-maven.html),
#'     and
#'     [NuGet](https://docs.aws.amazon.com/codeartifact/latest/ug/using-nuget.html)
#'     package formats.
#'
#'     In CodeArtifact, a package consists of:
#'
#'     -   A *name* (for example, `webpack` is the name of a popular npm
#'         package)
#'
#'     -   An optional namespace (for example, `@@types` in `@@types/node`)
#'
#'     -   A set of versions (for example, `1.0.0`, `1.0.1`, `1.0.2`, etc.)
#'
#'     -   Package-level metadata (for example, npm tags)
#'
#' -   **Package version**: A version of a package, such as
#'     `@@types/node 12.6.9`. The version number format and semantics vary
#'     for different package formats. For example, npm package versions
#'     must conform to the [Semantic Versioning
#'     specification](https://semver.org/). In CodeArtifact, a package
#'     version consists of the version identifier, metadata at the package
#'     version level, and a set of assets.
#'
#' -   **Upstream repository**: One repository is *upstream* of another
#'     when the package versions in it can be accessed from the repository
#'     endpoint of the downstream repository, effectively merging the
#'     contents of the two repositories from the point of view of a client.
#'     CodeArtifact allows creating an upstream relationship between two
#'     repositories.
#'
#' -   **Asset**: An individual file stored in CodeArtifact associated with
#'     a package version, such as an npm `.tgz` file or Maven POM and JAR
#'     files.
#'
#' CodeArtifact supports these operations:
#'
#' -   `associate_external_connection`:
#'     Adds an existing external connection to a repository.
#'
#' -   `copy_package_versions`:
#'     Copies package versions from one repository to another repository in
#'     the same domain.
#'
#' -   `create_domain`: Creates a domain
#'
#' -   `create_repository`: Creates a
#'     CodeArtifact repository in a domain.
#'
#' -   `delete_domain`: Deletes a domain. You
#'     cannot delete a domain that contains repositories.
#'
#' -   `delete_domain_permissions_policy`:
#'     Deletes the resource policy that is set on a domain.
#'
#' -   `delete_package`: Deletes a package
#'     and all associated package versions.
#'
#' -   `delete_package_versions`:
#'     Deletes versions of a package. After a package has been deleted, it
#'     can be republished, but its assets and metadata cannot be restored
#'     because they have been permanently removed from storage.
#'
#' -   `delete_repository`: Deletes a
#'     repository.
#'
#' -   `delete_repository_permissions_policy`:
#'     Deletes the resource policy that is set on a repository.
#'
#' -   `describe_domain`: Returns a
#'     `DomainDescription` object that contains information about the
#'     requested domain.
#'
#' -   `describe_package`: Returns a
#'     [PackageDescription](https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_PackageDescription.html)
#'     object that contains details about a package.
#'
#' -   `describe_package_version`:
#'     Returns a
#'     [PackageVersionDescription](https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_PackageVersionDescription.html)
#'     object that contains details about a package version.
#'
#' -   `describe_repository`: Returns a
#'     `RepositoryDescription` object that contains detailed information
#'     about the requested repository.
#'
#' -   `dispose_package_versions`:
#'     Disposes versions of a package. A package version with the status
#'     `Disposed` cannot be restored because they have been permanently
#'     removed from storage.
#'
#' -   `disassociate_external_connection`:
#'     Removes an existing external connection from a repository.
#'
#' -   `get_authorization_token`:
#'     Generates a temporary authorization token for accessing repositories
#'     in the domain. The token expires the authorization period has
#'     passed. The default authorization period is 12 hours and can be
#'     customized to any length with a maximum of 12 hours.
#'
#' -   `get_domain_permissions_policy`:
#'     Returns the policy of a resource that is attached to the specified
#'     domain.
#'
#' -   `get_package_version_asset`:
#'     Returns the contents of an asset that is in a package version.
#'
#' -   `get_package_version_readme`:
#'     Gets the readme file or descriptive text for a package version.
#'
#' -   `get_repository_endpoint`:
#'     Returns the endpoint of a repository for a specific package format.
#'     A repository has one endpoint for each package format:
#'
#'     -   `maven`
#'
#'     -   `npm`
#'
#'     -   `nuget`
#'
#'     -   `pypi`
#'
#' -   `get_repository_permissions_policy`:
#'     Returns the resource policy that is set on a repository.
#'
#' -   `list_domains`: Returns a list of
#'     `DomainSummary` objects. Each returned `DomainSummary` object
#'     contains information about a domain.
#'
#' -   `list_packages`: Lists the packages in
#'     a repository.
#'
#' -   `list_package_version_assets`:
#'     Lists the assets for a given package version.
#'
#' -   `list_package_version_dependencies`:
#'     Returns a list of the direct dependencies for a package version.
#'
#' -   `list_package_versions`:
#'     Returns a list of package versions for a specified package in a
#'     repository.
#'
#' -   `list_repositories`: Returns a
#'     list of repositories owned by the Amazon Web Services account that
#'     called this method.
#'
#' -   `list_repositories_in_domain`:
#'     Returns a list of the repositories in a domain.
#'
#' -   `publish_package_version`:
#'     Creates a new package version containing one or more assets.
#'
#' -   `put_domain_permissions_policy`:
#'     Attaches a resource policy to a domain.
#'
#' -   `put_package_origin_configuration`:
#'     Sets the package origin configuration for a package, which determine
#'     how new versions of the package can be added to a specific
#'     repository.
#'
#' -   `put_repository_permissions_policy`:
#'     Sets the resource policy on a repository that specifies permissions
#'     to access it.
#'
#' -   `update_package_versions_status`:
#'     Updates the status of one or more versions of a package.
#'
#' -   `update_repository`: Updates the
#'     properties of a repository.
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- codeartifact(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- codeartifact()
#' svc$associate_external_connection(
#'   Foo = 123
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.developer.tools:codeartifact_associate_external_connection]{associate_external_connection} \tab Adds an existing external connection to a repository\cr
#'  \link[paws.developer.tools:codeartifact_copy_package_versions]{copy_package_versions} \tab Copies package versions from one repository to another repository in the same domain\cr
#'  \link[paws.developer.tools:codeartifact_create_domain]{create_domain} \tab Creates a domain\cr
#'  \link[paws.developer.tools:codeartifact_create_repository]{create_repository} \tab Creates a repository\cr
#'  \link[paws.developer.tools:codeartifact_delete_domain]{delete_domain} \tab Deletes a domain\cr
#'  \link[paws.developer.tools:codeartifact_delete_domain_permissions_policy]{delete_domain_permissions_policy} \tab Deletes the resource policy set on a domain\cr
#'  \link[paws.developer.tools:codeartifact_delete_package]{delete_package} \tab Deletes a package and all associated package versions\cr
#'  \link[paws.developer.tools:codeartifact_delete_package_versions]{delete_package_versions} \tab Deletes one or more versions of a package\cr
#'  \link[paws.developer.tools:codeartifact_delete_repository]{delete_repository} \tab Deletes a repository\cr
#'  \link[paws.developer.tools:codeartifact_delete_repository_permissions_policy]{delete_repository_permissions_policy} \tab Deletes the resource policy that is set on a repository\cr
#'  \link[paws.developer.tools:codeartifact_describe_domain]{describe_domain} \tab Returns a DomainDescription object that contains information about the requested domain\cr
#'  \link[paws.developer.tools:codeartifact_describe_package]{describe_package} \tab Returns a PackageDescription object that contains information about the requested package\cr
#'  \link[paws.developer.tools:codeartifact_describe_package_version]{describe_package_version} \tab Returns a PackageVersionDescription object that contains information about the requested package version\cr
#'  \link[paws.developer.tools:codeartifact_describe_repository]{describe_repository} \tab Returns a RepositoryDescription object that contains detailed information about the requested repository\cr
#'  \link[paws.developer.tools:codeartifact_disassociate_external_connection]{disassociate_external_connection} \tab Removes an existing external connection from a repository\cr
#'  \link[paws.developer.tools:codeartifact_dispose_package_versions]{dispose_package_versions} \tab Deletes the assets in package versions and sets the package versions' status to Disposed\cr
#'  \link[paws.developer.tools:codeartifact_get_authorization_token]{get_authorization_token} \tab Generates a temporary authorization token for accessing repositories in the domain\cr
#'  \link[paws.developer.tools:codeartifact_get_domain_permissions_policy]{get_domain_permissions_policy} \tab Returns the resource policy attached to the specified domain\cr
#'  \link[paws.developer.tools:codeartifact_get_package_version_asset]{get_package_version_asset} \tab Returns an asset (or file) that is in a package\cr
#'  \link[paws.developer.tools:codeartifact_get_package_version_readme]{get_package_version_readme} \tab Gets the readme file or descriptive text for a package version\cr
#'  \link[paws.developer.tools:codeartifact_get_repository_endpoint]{get_repository_endpoint} \tab Returns the endpoint of a repository for a specific package format\cr
#'  \link[paws.developer.tools:codeartifact_get_repository_permissions_policy]{get_repository_permissions_policy} \tab Returns the resource policy that is set on a repository\cr
#'  \link[paws.developer.tools:codeartifact_list_domains]{list_domains} \tab Returns a list of DomainSummary objects for all domains owned by the Amazon Web Services account that makes this call\cr
#'  \link[paws.developer.tools:codeartifact_list_packages]{list_packages} \tab Returns a list of PackageSummary objects for packages in a repository that match the request parameters\cr
#'  \link[paws.developer.tools:codeartifact_list_package_version_assets]{list_package_version_assets} \tab Returns a list of AssetSummary objects for assets in a package version\cr
#'  \link[paws.developer.tools:codeartifact_list_package_version_dependencies]{list_package_version_dependencies} \tab Returns the direct dependencies for a package version\cr
#'  \link[paws.developer.tools:codeartifact_list_package_versions]{list_package_versions} \tab Returns a list of PackageVersionSummary objects for package versions in a repository that match the request parameters\cr
#'  \link[paws.developer.tools:codeartifact_list_repositories]{list_repositories} \tab Returns a list of RepositorySummary objects\cr
#'  \link[paws.developer.tools:codeartifact_list_repositories_in_domain]{list_repositories_in_domain} \tab Returns a list of RepositorySummary objects\cr
#'  \link[paws.developer.tools:codeartifact_list_tags_for_resource]{list_tags_for_resource} \tab Gets information about Amazon Web Services tags for a specified Amazon Resource Name (ARN) in CodeArtifact\cr
#'  \link[paws.developer.tools:codeartifact_publish_package_version]{publish_package_version} \tab Creates a new package version containing one or more assets (or files)\cr
#'  \link[paws.developer.tools:codeartifact_put_domain_permissions_policy]{put_domain_permissions_policy} \tab Sets a resource policy on a domain that specifies permissions to access it\cr
#'  \link[paws.developer.tools:codeartifact_put_package_origin_configuration]{put_package_origin_configuration} \tab Sets the package origin configuration for a package\cr
#'  \link[paws.developer.tools:codeartifact_put_repository_permissions_policy]{put_repository_permissions_policy} \tab Sets the resource policy on a repository that specifies permissions to access it\cr
#'  \link[paws.developer.tools:codeartifact_tag_resource]{tag_resource} \tab Adds or updates tags for a resource in CodeArtifact\cr
#'  \link[paws.developer.tools:codeartifact_untag_resource]{untag_resource} \tab Removes tags from a resource in CodeArtifact\cr
#'  \link[paws.developer.tools:codeartifact_update_package_versions_status]{update_package_versions_status} \tab Updates the status of one or more versions of a package\cr
#'  \link[paws.developer.tools:codeartifact_update_repository]{update_repository} \tab Update the properties of a repository
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname codeartifact
#' @export
codeartifact <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.developer.tools::codeartifact(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' AWS CodeCommit
#'
#' @description
#' CodeCommit
#'
#' This is the *CodeCommit API Reference*. This reference provides
#' descriptions of the operations and data types for CodeCommit API along
#' with usage examples.
#'
#' You can use the CodeCommit API to work with the following objects:
#'
#' Repositories, by calling the following:
#'
#' -   `batch_get_repositories`, which
#'     returns information about one or more repositories associated with
#'     your Amazon Web Services account.
#'
#' -   `create_repository`, which creates
#'     an CodeCommit repository.
#'
#' -   `delete_repository`, which deletes
#'     an CodeCommit repository.
#'
#' -   `get_repository`, which returns
#'     information about a specified repository.
#'
#' -   `list_repositories`, which lists all
#'     CodeCommit repositories associated with your Amazon Web Services
#'     account.
#'
#' -   `update_repository_description`,
#'     which sets or updates the description of the repository.
#'
#' -   `update_repository_name`, which
#'     changes the name of the repository. If you change the name of a
#'     repository, no other users of that repository can access it until
#'     you send them the new HTTPS or SSH URL to use.
#'
#' Branches, by calling the following:
#'
#' -   `create_branch`, which creates a branch
#'     in a specified repository.
#'
#' -   `delete_branch`, which deletes the
#'     specified branch in a repository unless it is the default branch.
#'
#' -   `get_branch`, which returns information
#'     about a specified branch.
#'
#' -   `list_branches`, which lists all
#'     branches for a specified repository.
#'
#' -   `update_default_branch`, which
#'     changes the default branch for a repository.
#'
#' Files, by calling the following:
#'
#' -   `delete_file`, which deletes the content
#'     of a specified file from a specified branch.
#'
#' -   `get_blob`, which returns the base-64 encoded
#'     content of an individual Git blob object in a repository.
#'
#' -   `get_file`, which returns the base-64 encoded
#'     content of a specified file.
#'
#' -   `get_folder`, which returns the contents of
#'     a specified folder or directory.
#'
#' -   `list_file_commit_history`,
#'     which retrieves a list of commits and changes to a specified file.
#'
#' -   `put_file`, which adds or modifies a single
#'     file in a specified repository and branch.
#'
#' Commits, by calling the following:
#'
#' -   `batch_get_commits`, which returns
#'     information about one or more commits in a repository.
#'
#' -   `create_commit`, which creates a commit
#'     for changes to a repository.
#'
#' -   `get_commit`, which returns information
#'     about a commit, including commit messages and author and committer
#'     information.
#'
#' -   `get_differences`, which returns
#'     information about the differences in a valid commit specifier (such
#'     as a branch, tag, HEAD, commit ID, or other fully qualified
#'     reference).
#'
#' Merges, by calling the following:
#'
#' -   `batch_describe_merge_conflicts`,
#'     which returns information about conflicts in a merge between commits
#'     in a repository.
#'
#' -   `create_unreferenced_merge_commit`,
#'     which creates an unreferenced commit between two branches or commits
#'     for the purpose of comparing them and identifying any potential
#'     conflicts.
#'
#' -   `describe_merge_conflicts`,
#'     which returns information about merge conflicts between the base,
#'     source, and destination versions of a file in a potential merge.
#'
#' -   `get_merge_commit`, which returns
#'     information about the merge between a source and destination commit.
#'
#' -   `get_merge_conflicts`, which
#'     returns information about merge conflicts between the source and
#'     destination branch in a pull request.
#'
#' -   `get_merge_options`, which returns
#'     information about the available merge options between two branches
#'     or commit specifiers.
#'
#' -   `merge_branches_by_fast_forward`,
#'     which merges two branches using the fast-forward merge option.
#'
#' -   `merge_branches_by_squash`,
#'     which merges two branches using the squash merge option.
#'
#' -   `merge_branches_by_three_way`,
#'     which merges two branches using the three-way merge option.
#'
#' Pull requests, by calling the following:
#'
#' -   `create_pull_request`, which
#'     creates a pull request in a specified repository.
#'
#' -   `create_pull_request_approval_rule`,
#'     which creates an approval rule for a specified pull request.
#'
#' -   `delete_pull_request_approval_rule`,
#'     which deletes an approval rule for a specified pull request.
#'
#' -   `describe_pull_request_events`,
#'     which returns information about one or more pull request events.
#'
#' -   `evaluate_pull_request_approval_rules`,
#'     which evaluates whether a pull request has met all the conditions
#'     specified in its associated approval rules.
#'
#' -   `get_comments_for_pull_request`,
#'     which returns information about comments on a specified pull
#'     request.
#'
#' -   `get_pull_request`, which returns
#'     information about a specified pull request.
#'
#' -   `get_pull_request_approval_states`,
#'     which returns information about the approval states for a specified
#'     pull request.
#'
#' -   `get_pull_request_override_state`,
#'     which returns information about whether approval rules have been set
#'     aside (overriden) for a pull request, and if so, the Amazon Resource
#'     Name (ARN) of the user or identity that overrode the rules and their
#'     requirements for the pull request.
#'
#' -   `list_pull_requests`, which lists
#'     all pull requests for a repository.
#'
#' -   `merge_pull_request_by_fast_forward`,
#'     which merges the source destination branch of a pull request into
#'     the specified destination branch for that pull request using the
#'     fast-forward merge option.
#'
#' -   `merge_pull_request_by_squash`,
#'     which merges the source destination branch of a pull request into
#'     the specified destination branch for that pull request using the
#'     squash merge option.
#'
#' -   `merge_pull_request_by_three_way`,
#'     which merges the source destination branch of a pull request into
#'     the specified destination branch for that pull request using the
#'     three-way merge option.
#'
#' -   `override_pull_request_approval_rules`,
#'     which sets aside all approval rule requirements for a pull request.
#'
#' -   `post_comment_for_pull_request`,
#'     which posts a comment to a pull request at the specified line, file,
#'     or request.
#'
#' -   `update_pull_request_approval_rule_content`,
#'     which updates the structure of an approval rule for a pull request.
#'
#' -   `update_pull_request_approval_state`,
#'     which updates the state of an approval on a pull request.
#'
#' -   `update_pull_request_description`,
#'     which updates the description of a pull request.
#'
#' -   `update_pull_request_status`,
#'     which updates the status of a pull request.
#'
#' -   `update_pull_request_title`,
#'     which updates the title of a pull request.
#'
#' Approval rule templates, by calling the following:
#'
#' -   `associate_approval_rule_template_with_repository`,
#'     which associates a template with a specified repository. After the
#'     template is associated with a repository, CodeCommit creates
#'     approval rules that match the template conditions on every pull
#'     request created in the specified repository.
#'
#' -   `batch_associate_approval_rule_template_with_repositories`,
#'     which associates a template with one or more specified repositories.
#'     After the template is associated with a repository, CodeCommit
#'     creates approval rules that match the template conditions on every
#'     pull request created in the specified repositories.
#'
#' -   `batch_disassociate_approval_rule_template_from_repositories`,
#'     which removes the association between a template and specified
#'     repositories so that approval rules based on the template are not
#'     automatically created when pull requests are created in those
#'     repositories.
#'
#' -   `create_approval_rule_template`,
#'     which creates a template for approval rules that can then be
#'     associated with one or more repositories in your Amazon Web Services
#'     account.
#'
#' -   `delete_approval_rule_template`,
#'     which deletes the specified template. It does not remove approval
#'     rules on pull requests already created with the template.
#'
#' -   `disassociate_approval_rule_template_from_repository`,
#'     which removes the association between a template and a repository so
#'     that approval rules based on the template are not automatically
#'     created when pull requests are created in the specified repository.
#'
#' -   `get_approval_rule_template`,
#'     which returns information about an approval rule template.
#'
#' -   `list_approval_rule_templates`,
#'     which lists all approval rule templates in the Amazon Web Services
#'     Region in your Amazon Web Services account.
#'
#' -   `list_associated_approval_rule_templates_for_repository`,
#'     which lists all approval rule templates that are associated with a
#'     specified repository.
#'
#' -   `list_repositories_for_approval_rule_template`,
#'     which lists all repositories associated with the specified approval
#'     rule template.
#'
#' -   `update_approval_rule_template_description`,
#'     which updates the description of an approval rule template.
#'
#' -   `update_approval_rule_template_name`,
#'     which updates the name of an approval rule template.
#'
#' -   `update_approval_rule_template_content`,
#'     which updates the content of an approval rule template.
#'
#' Comments in a repository, by calling the following:
#'
#' -   `delete_comment_content`, which
#'     deletes the content of a comment on a commit in a repository.
#'
#' -   `get_comment`, which returns information
#'     about a comment on a commit.
#'
#' -   `get_comment_reactions`, which
#'     returns information about emoji reactions to comments.
#'
#' -   `get_comments_for_compared_commit`,
#'     which returns information about comments on the comparison between
#'     two commit specifiers in a repository.
#'
#' -   `post_comment_for_compared_commit`,
#'     which creates a comment on the comparison between two commit
#'     specifiers in a repository.
#'
#' -   `post_comment_reply`, which creates
#'     a reply to a comment.
#'
#' -   `put_comment_reaction`, which
#'     creates or updates an emoji reaction to a comment.
#'
#' -   `update_comment`, which updates the
#'     content of a comment on a commit in a repository.
#'
#' Tags used to tag resources in CodeCommit (not Git tags), by calling the
#' following:
#'
#' -   `list_tags_for_resource`, which
#'     gets information about Amazon Web Servicestags for a specified
#'     Amazon Resource Name (ARN) in CodeCommit.
#'
#' -   `tag_resource`, which adds or updates
#'     tags for a resource in CodeCommit.
#'
#' -   `untag_resource`, which removes tags
#'     for a resource in CodeCommit.
#'
#' Triggers, by calling the following:
#'
#' -   `get_repository_triggers`,
#'     which returns information about triggers configured for a
#'     repository.
#'
#' -   `put_repository_triggers`,
#'     which replaces all triggers for a repository and can be used to
#'     create or delete triggers.
#'
#' -   `test_repository_triggers`,
#'     which tests the functionality of a repository trigger by sending
#'     data to the trigger target.
#'
#' For information about how to use CodeCommit, see the [CodeCommit User
#' Guide](https://docs.aws.amazon.com/codecommit/latest/userguide/welcome.html).
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- codecommit(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- codecommit()
#' svc$associate_approval_rule_template_with_repository(
#'   Foo = 123
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.developer.tools:codecommit_associate_approval_rule_template_with_repository]{associate_approval_rule_template_with_repository} \tab Creates an association between an approval rule template and a specified repository\cr
#'  \link[paws.developer.tools:codecommit_batch_associate_approval_rule_template_with_repositories]{batch_associate_approval_rule_template_with_repositories} \tab Creates an association between an approval rule template and one or more specified repositories\cr
#'  \link[paws.developer.tools:codecommit_batch_describe_merge_conflicts]{batch_describe_merge_conflicts} \tab Returns information about one or more merge conflicts in the attempted merge of two commit specifiers using the squash or three-way merge strategy\cr
#'  \link[paws.developer.tools:codecommit_batch_disassociate_approval_rule_template_from_repositories]{batch_disassociate_approval_rule_template_from_repositories} \tab Removes the association between an approval rule template and one or more specified repositories\cr
#'  \link[paws.developer.tools:codecommit_batch_get_commits]{batch_get_commits} \tab Returns information about the contents of one or more commits in a repository\cr
#'  \link[paws.developer.tools:codecommit_batch_get_repositories]{batch_get_repositories} \tab Returns information about one or more repositories\cr
#'  \link[paws.developer.tools:codecommit_create_approval_rule_template]{create_approval_rule_template} \tab Creates a template for approval rules that can then be associated with one or more repositories in your Amazon Web Services account\cr
#'  \link[paws.developer.tools:codecommit_create_branch]{create_branch} \tab Creates a branch in a repository and points the branch to a commit\cr
#'  \link[paws.developer.tools:codecommit_create_commit]{create_commit} \tab Creates a commit for a repository on the tip of a specified branch\cr
#'  \link[paws.developer.tools:codecommit_create_pull_request]{create_pull_request} \tab Creates a pull request in the specified repository\cr
#'  \link[paws.developer.tools:codecommit_create_pull_request_approval_rule]{create_pull_request_approval_rule} \tab Creates an approval rule for a pull request\cr
#'  \link[paws.developer.tools:codecommit_create_repository]{create_repository} \tab Creates a new, empty repository\cr
#'  \link[paws.developer.tools:codecommit_create_unreferenced_merge_commit]{create_unreferenced_merge_commit} \tab Creates an unreferenced commit that represents the result of merging two branches using a specified merge strategy\cr
#'  \link[paws.developer.tools:codecommit_delete_approval_rule_template]{delete_approval_rule_template} \tab Deletes a specified approval rule template\cr
#'  \link[paws.developer.tools:codecommit_delete_branch]{delete_branch} \tab Deletes a branch from a repository, unless that branch is the default branch for the repository\cr
#'  \link[paws.developer.tools:codecommit_delete_comment_content]{delete_comment_content} \tab Deletes the content of a comment made on a change, file, or commit in a repository\cr
#'  \link[paws.developer.tools:codecommit_delete_file]{delete_file} \tab Deletes a specified file from a specified branch\cr
#'  \link[paws.developer.tools:codecommit_delete_pull_request_approval_rule]{delete_pull_request_approval_rule} \tab Deletes an approval rule from a specified pull request\cr
#'  \link[paws.developer.tools:codecommit_delete_repository]{delete_repository} \tab Deletes a repository\cr
#'  \link[paws.developer.tools:codecommit_describe_merge_conflicts]{describe_merge_conflicts} \tab Returns information about one or more merge conflicts in the attempted merge of two commit specifiers using the squash or three-way merge strategy\cr
#'  \link[paws.developer.tools:codecommit_describe_pull_request_events]{describe_pull_request_events} \tab Returns information about one or more pull request events\cr
#'  \link[paws.developer.tools:codecommit_disassociate_approval_rule_template_from_repository]{disassociate_approval_rule_template_from_repository} \tab Removes the association between a template and a repository so that approval rules based on the template are not automatically created when pull requests are created in the specified repository\cr
#'  \link[paws.developer.tools:codecommit_evaluate_pull_request_approval_rules]{evaluate_pull_request_approval_rules} \tab Evaluates whether a pull request has met all the conditions specified in its associated approval rules\cr
#'  \link[paws.developer.tools:codecommit_get_approval_rule_template]{get_approval_rule_template} \tab Returns information about a specified approval rule template\cr
#'  \link[paws.developer.tools:codecommit_get_blob]{get_blob} \tab Returns the base-64 encoded content of an individual blob in a repository\cr
#'  \link[paws.developer.tools:codecommit_get_branch]{get_branch} \tab Returns information about a repository branch, including its name and the last commit ID\cr
#'  \link[paws.developer.tools:codecommit_get_comment]{get_comment} \tab Returns the content of a comment made on a change, file, or commit in a repository\cr
#'  \link[paws.developer.tools:codecommit_get_comment_reactions]{get_comment_reactions} \tab Returns information about reactions to a specified comment ID\cr
#'  \link[paws.developer.tools:codecommit_get_comments_for_compared_commit]{get_comments_for_compared_commit} \tab Returns information about comments made on the comparison between two commits\cr
#'  \link[paws.developer.tools:codecommit_get_comments_for_pull_request]{get_comments_for_pull_request} \tab Returns comments made on a pull request\cr
#'  \link[paws.developer.tools:codecommit_get_commit]{get_commit} \tab Returns information about a commit, including commit message and committer information\cr
#'  \link[paws.developer.tools:codecommit_get_differences]{get_differences} \tab Returns information about the differences in a valid commit specifier (such as a branch, tag, HEAD, commit ID, or other fully qualified reference)\cr
#'  \link[paws.developer.tools:codecommit_get_file]{get_file} \tab Returns the base-64 encoded contents of a specified file and its metadata\cr
#'  \link[paws.developer.tools:codecommit_get_folder]{get_folder} \tab Returns the contents of a specified folder in a repository\cr
#'  \link[paws.developer.tools:codecommit_get_merge_commit]{get_merge_commit} \tab Returns information about a specified merge commit\cr
#'  \link[paws.developer.tools:codecommit_get_merge_conflicts]{get_merge_conflicts} \tab Returns information about merge conflicts between the before and after commit IDs for a pull request in a repository\cr
#'  \link[paws.developer.tools:codecommit_get_merge_options]{get_merge_options} \tab Returns information about the merge options available for merging two specified branches\cr
#'  \link[paws.developer.tools:codecommit_get_pull_request]{get_pull_request} \tab Gets information about a pull request in a specified repository\cr
#'  \link[paws.developer.tools:codecommit_get_pull_request_approval_states]{get_pull_request_approval_states} \tab Gets information about the approval states for a specified pull request\cr
#'  \link[paws.developer.tools:codecommit_get_pull_request_override_state]{get_pull_request_override_state} \tab Returns information about whether approval rules have been set aside (overridden) for a pull request, and if so, the Amazon Resource Name (ARN) of the user or identity that overrode the rules and their requirements for the pull request\cr
#'  \link[paws.developer.tools:codecommit_get_repository]{get_repository} \tab Returns information about a repository\cr
#'  \link[paws.developer.tools:codecommit_get_repository_triggers]{get_repository_triggers} \tab Gets information about triggers configured for a repository\cr
#'  \link[paws.developer.tools:codecommit_list_approval_rule_templates]{list_approval_rule_templates} \tab Lists all approval rule templates in the specified Amazon Web Services Region in your Amazon Web Services account\cr
#'  \link[paws.developer.tools:codecommit_list_associated_approval_rule_templates_for_repository]{list_associated_approval_rule_templates_for_repository} \tab Lists all approval rule templates that are associated with a specified repository\cr
#'  \link[paws.developer.tools:codecommit_list_branches]{list_branches} \tab Gets information about one or more branches in a repository\cr
#'  \link[paws.developer.tools:codecommit_list_file_commit_history]{list_file_commit_history} \tab Retrieves a list of commits and changes to a specified file\cr
#'  \link[paws.developer.tools:codecommit_list_pull_requests]{list_pull_requests} \tab Returns a list of pull requests for a specified repository\cr
#'  \link[paws.developer.tools:codecommit_list_repositories]{list_repositories} \tab Gets information about one or more repositories\cr
#'  \link[paws.developer.tools:codecommit_list_repositories_for_approval_rule_template]{list_repositories_for_approval_rule_template} \tab Lists all repositories associated with the specified approval rule template\cr
#'  \link[paws.developer.tools:codecommit_list_tags_for_resource]{list_tags_for_resource} \tab Gets information about Amazon Web Servicestags for a specified Amazon Resource Name (ARN) in CodeCommit\cr
#'  \link[paws.developer.tools:codecommit_merge_branches_by_fast_forward]{merge_branches_by_fast_forward} \tab Merges two branches using the fast-forward merge strategy\cr
#'  \link[paws.developer.tools:codecommit_merge_branches_by_squash]{merge_branches_by_squash} \tab Merges two branches using the squash merge strategy\cr
#'  \link[paws.developer.tools:codecommit_merge_branches_by_three_way]{merge_branches_by_three_way} \tab Merges two specified branches using the three-way merge strategy\cr
#'  \link[paws.developer.tools:codecommit_merge_pull_request_by_fast_forward]{merge_pull_request_by_fast_forward} \tab Attempts to merge the source commit of a pull request into the specified destination branch for that pull request at the specified commit using the fast-forward merge strategy\cr
#'  \link[paws.developer.tools:codecommit_merge_pull_request_by_squash]{merge_pull_request_by_squash} \tab Attempts to merge the source commit of a pull request into the specified destination branch for that pull request at the specified commit using the squash merge strategy\cr
#'  \link[paws.developer.tools:codecommit_merge_pull_request_by_three_way]{merge_pull_request_by_three_way} \tab Attempts to merge the source commit of a pull request into the specified destination branch for that pull request at the specified commit using the three-way merge strategy\cr
#'  \link[paws.developer.tools:codecommit_override_pull_request_approval_rules]{override_pull_request_approval_rules} \tab Sets aside (overrides) all approval rule requirements for a specified pull request\cr
#'  \link[paws.developer.tools:codecommit_post_comment_for_compared_commit]{post_comment_for_compared_commit} \tab Posts a comment on the comparison between two commits\cr
#'  \link[paws.developer.tools:codecommit_post_comment_for_pull_request]{post_comment_for_pull_request} \tab Posts a comment on a pull request\cr
#'  \link[paws.developer.tools:codecommit_post_comment_reply]{post_comment_reply} \tab Posts a comment in reply to an existing comment on a comparison between commits or a pull request\cr
#'  \link[paws.developer.tools:codecommit_put_comment_reaction]{put_comment_reaction} \tab Adds or updates a reaction to a specified comment for the user whose identity is used to make the request\cr
#'  \link[paws.developer.tools:codecommit_put_file]{put_file} \tab Adds or updates a file in a branch in an CodeCommit repository, and generates a commit for the addition in the specified branch\cr
#'  \link[paws.developer.tools:codecommit_put_repository_triggers]{put_repository_triggers} \tab Replaces all triggers for a repository\cr
#'  \link[paws.developer.tools:codecommit_tag_resource]{tag_resource} \tab Adds or updates tags for a resource in CodeCommit\cr
#'  \link[paws.developer.tools:codecommit_test_repository_triggers]{test_repository_triggers} \tab Tests the functionality of repository triggers by sending information to the trigger target\cr
#'  \link[paws.developer.tools:codecommit_untag_resource]{untag_resource} \tab Removes tags for a resource in CodeCommit\cr
#'  \link[paws.developer.tools:codecommit_update_approval_rule_template_content]{update_approval_rule_template_content} \tab Updates the content of an approval rule template\cr
#'  \link[paws.developer.tools:codecommit_update_approval_rule_template_description]{update_approval_rule_template_description} \tab Updates the description for a specified approval rule template\cr
#'  \link[paws.developer.tools:codecommit_update_approval_rule_template_name]{update_approval_rule_template_name} \tab Updates the name of a specified approval rule template\cr
#'  \link[paws.developer.tools:codecommit_update_comment]{update_comment} \tab Replaces the contents of a comment\cr
#'  \link[paws.developer.tools:codecommit_update_default_branch]{update_default_branch} \tab Sets or changes the default branch name for the specified repository\cr
#'  \link[paws.developer.tools:codecommit_update_pull_request_approval_rule_content]{update_pull_request_approval_rule_content} \tab Updates the structure of an approval rule created specifically for a pull request\cr
#'  \link[paws.developer.tools:codecommit_update_pull_request_approval_state]{update_pull_request_approval_state} \tab Updates the state of a user's approval on a pull request\cr
#'  \link[paws.developer.tools:codecommit_update_pull_request_description]{update_pull_request_description} \tab Replaces the contents of the description of a pull request\cr
#'  \link[paws.developer.tools:codecommit_update_pull_request_status]{update_pull_request_status} \tab Updates the status of a pull request\cr
#'  \link[paws.developer.tools:codecommit_update_pull_request_title]{update_pull_request_title} \tab Replaces the title of a pull request\cr
#'  \link[paws.developer.tools:codecommit_update_repository_description]{update_repository_description} \tab Sets or changes the comment or description for a repository\cr
#'  \link[paws.developer.tools:codecommit_update_repository_name]{update_repository_name} \tab Renames a repository
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname codecommit
#' @export
codecommit <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.developer.tools::codecommit(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' AWS CodeDeploy
#'
#' @description
#' CodeDeploy is a deployment service that automates application
#' deployments to Amazon EC2 instances, on-premises instances running in
#' your own facility, serverless Lambda functions, or applications in an
#' Amazon ECS service.
#'
#' You can deploy a nearly unlimited variety of application content, such
#' as an updated Lambda function, updated applications in an Amazon ECS
#' service, code, web and configuration files, executables, packages,
#' scripts, multimedia files, and so on. CodeDeploy can deploy application
#' content stored in Amazon S3 buckets, GitHub repositories, or Bitbucket
#' repositories. You do not need to make changes to your existing code
#' before you can use CodeDeploy.
#'
#' CodeDeploy makes it easier for you to rapidly release new features,
#' helps you avoid downtime during application deployment, and handles the
#' complexity of updating your applications, without many of the risks
#' associated with error-prone manual deployments.
#'
#' **CodeDeploy Components**
#'
#' Use the information in this guide to help you work with the following
#' CodeDeploy components:
#'
#' -   **Application**: A name that uniquely identifies the application you
#'     want to deploy. CodeDeploy uses this name, which functions as a
#'     container, to ensure the correct combination of revision, deployment
#'     configuration, and deployment group are referenced during a
#'     deployment.
#'
#' -   **Deployment group**: A set of individual instances, CodeDeploy
#'     Lambda deployment configuration settings, or an Amazon ECS service
#'     and network details. A Lambda deployment group specifies how to
#'     route traffic to a new version of a Lambda function. An Amazon ECS
#'     deployment group specifies the service created in Amazon ECS to
#'     deploy, a load balancer, and a listener to reroute production
#'     traffic to an updated containerized application. An Amazon
#'     EC2/On-premises deployment group contains individually tagged
#'     instances, Amazon EC2 instances in Amazon EC2 Auto Scaling groups,
#'     or both. All deployment groups can specify optional trigger, alarm,
#'     and rollback settings.
#'
#' -   **Deployment configuration**: A set of deployment rules and
#'     deployment success and failure conditions used by CodeDeploy during
#'     a deployment.
#'
#' -   **Deployment**: The process and the components used when updating a
#'     Lambda function, a containerized application in an Amazon ECS
#'     service, or of installing content on one or more instances.
#'
#' -   **Application revisions**: For an Lambda deployment, this is an
#'     AppSpec file that specifies the Lambda function to be updated and
#'     one or more functions to validate deployment lifecycle events. For
#'     an Amazon ECS deployment, this is an AppSpec file that specifies the
#'     Amazon ECS task definition, container, and port where production
#'     traffic is rerouted. For an EC2/On-premises deployment, this is an
#'     archive file that contains source content—source code, webpages,
#'     executable files, and deployment scripts—along with an AppSpec file.
#'     Revisions are stored in Amazon S3 buckets or GitHub repositories.
#'     For Amazon S3, a revision is uniquely identified by its Amazon S3
#'     object key and its ETag, version, or both. For GitHub, a revision is
#'     uniquely identified by its commit ID.
#'
#' This guide also contains information to help you get details about the
#' instances in your deployments, to make on-premises instances available
#' for CodeDeploy deployments, to get details about a Lambda function
#' deployment, and to get details about Amazon ECS service deployments.
#'
#' **CodeDeploy Information Resources**
#'
#' -   [CodeDeploy User
#'     Guide](https://docs.aws.amazon.com/codedeploy/latest/userguide/)
#'
#' -   [CodeDeploy API Reference
#'     Guide](https://docs.aws.amazon.com/codedeploy/latest/APIReference/)
#'
#' -   [CLI Reference for
#'     CodeDeploy](https://docs.aws.amazon.com/cli/latest/reference/deploy/)
#'
#' -   CodeDeploy Developer Forum
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- codedeploy(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- codedeploy()
#' svc$add_tags_to_on_premises_instances(
#'   Foo = 123
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.developer.tools:codedeploy_add_tags_to_on_premises_instances]{add_tags_to_on_premises_instances} \tab Adds tags to on-premises instances\cr
#'  \link[paws.developer.tools:codedeploy_batch_get_application_revisions]{batch_get_application_revisions} \tab Gets information about one or more application revisions\cr
#'  \link[paws.developer.tools:codedeploy_batch_get_applications]{batch_get_applications} \tab Gets information about one or more applications\cr
#'  \link[paws.developer.tools:codedeploy_batch_get_deployment_groups]{batch_get_deployment_groups} \tab Gets information about one or more deployment groups\cr
#'  \link[paws.developer.tools:codedeploy_batch_get_deployment_instances]{batch_get_deployment_instances} \tab This method works, but is deprecated\cr
#'  \link[paws.developer.tools:codedeploy_batch_get_deployments]{batch_get_deployments} \tab Gets information about one or more deployments\cr
#'  \link[paws.developer.tools:codedeploy_batch_get_deployment_targets]{batch_get_deployment_targets} \tab Returns an array of one or more targets associated with a deployment\cr
#'  \link[paws.developer.tools:codedeploy_batch_get_on_premises_instances]{batch_get_on_premises_instances} \tab Gets information about one or more on-premises instances\cr
#'  \link[paws.developer.tools:codedeploy_continue_deployment]{continue_deployment} \tab For a blue/green deployment, starts the process of rerouting traffic from instances in the original environment to instances in the replacement environment without waiting for a specified wait time to elapse\cr
#'  \link[paws.developer.tools:codedeploy_create_application]{create_application} \tab Creates an application\cr
#'  \link[paws.developer.tools:codedeploy_create_deployment]{create_deployment} \tab Deploys an application revision through the specified deployment group\cr
#'  \link[paws.developer.tools:codedeploy_create_deployment_config]{create_deployment_config} \tab Creates a deployment configuration\cr
#'  \link[paws.developer.tools:codedeploy_create_deployment_group]{create_deployment_group} \tab Creates a deployment group to which application revisions are deployed\cr
#'  \link[paws.developer.tools:codedeploy_delete_application]{delete_application} \tab Deletes an application\cr
#'  \link[paws.developer.tools:codedeploy_delete_deployment_config]{delete_deployment_config} \tab Deletes a deployment configuration\cr
#'  \link[paws.developer.tools:codedeploy_delete_deployment_group]{delete_deployment_group} \tab Deletes a deployment group\cr
#'  \link[paws.developer.tools:codedeploy_delete_git_hub_account_token]{delete_git_hub_account_token} \tab Deletes a GitHub account connection\cr
#'  \link[paws.developer.tools:codedeploy_delete_resources_by_external_id]{delete_resources_by_external_id} \tab Deletes resources linked to an external ID\cr
#'  \link[paws.developer.tools:codedeploy_deregister_on_premises_instance]{deregister_on_premises_instance} \tab Deregisters an on-premises instance\cr
#'  \link[paws.developer.tools:codedeploy_get_application]{get_application} \tab Gets information about an application\cr
#'  \link[paws.developer.tools:codedeploy_get_application_revision]{get_application_revision} \tab Gets information about an application revision\cr
#'  \link[paws.developer.tools:codedeploy_get_deployment]{get_deployment} \tab Gets information about a deployment\cr
#'  \link[paws.developer.tools:codedeploy_get_deployment_config]{get_deployment_config} \tab Gets information about a deployment configuration\cr
#'  \link[paws.developer.tools:codedeploy_get_deployment_group]{get_deployment_group} \tab Gets information about a deployment group\cr
#'  \link[paws.developer.tools:codedeploy_get_deployment_instance]{get_deployment_instance} \tab Gets information about an instance as part of a deployment\cr
#'  \link[paws.developer.tools:codedeploy_get_deployment_target]{get_deployment_target} \tab Returns information about a deployment target\cr
#'  \link[paws.developer.tools:codedeploy_get_on_premises_instance]{get_on_premises_instance} \tab Gets information about an on-premises instance\cr
#'  \link[paws.developer.tools:codedeploy_list_application_revisions]{list_application_revisions} \tab Lists information about revisions for an application\cr
#'  \link[paws.developer.tools:codedeploy_list_applications]{list_applications} \tab Lists the applications registered with the IAM user or Amazon Web Services account\cr
#'  \link[paws.developer.tools:codedeploy_list_deployment_configs]{list_deployment_configs} \tab Lists the deployment configurations with the IAM user or Amazon Web Services account\cr
#'  \link[paws.developer.tools:codedeploy_list_deployment_groups]{list_deployment_groups} \tab Lists the deployment groups for an application registered with the IAM user or Amazon Web Services account\cr
#'  \link[paws.developer.tools:codedeploy_list_deployment_instances]{list_deployment_instances} \tab The newer BatchGetDeploymentTargets should be used instead because it works with all compute types\cr
#'  \link[paws.developer.tools:codedeploy_list_deployments]{list_deployments} \tab Lists the deployments in a deployment group for an application registered with the IAM user or Amazon Web Services account\cr
#'  \link[paws.developer.tools:codedeploy_list_deployment_targets]{list_deployment_targets} \tab Returns an array of target IDs that are associated a deployment\cr
#'  \link[paws.developer.tools:codedeploy_list_git_hub_account_token_names]{list_git_hub_account_token_names} \tab Lists the names of stored connections to GitHub accounts\cr
#'  \link[paws.developer.tools:codedeploy_list_on_premises_instances]{list_on_premises_instances} \tab Gets a list of names for one or more on-premises instances\cr
#'  \link[paws.developer.tools:codedeploy_list_tags_for_resource]{list_tags_for_resource} \tab Returns a list of tags for the resource identified by a specified Amazon Resource Name (ARN)\cr
#'  \link[paws.developer.tools:codedeploy_put_lifecycle_event_hook_execution_status]{put_lifecycle_event_hook_execution_status} \tab Sets the result of a Lambda validation function\cr
#'  \link[paws.developer.tools:codedeploy_register_application_revision]{register_application_revision} \tab Registers with CodeDeploy a revision for the specified application\cr
#'  \link[paws.developer.tools:codedeploy_register_on_premises_instance]{register_on_premises_instance} \tab Registers an on-premises instance\cr
#'  \link[paws.developer.tools:codedeploy_remove_tags_from_on_premises_instances]{remove_tags_from_on_premises_instances} \tab Removes one or more tags from one or more on-premises instances\cr
#'  \link[paws.developer.tools:codedeploy_skip_wait_time_for_instance_termination]{skip_wait_time_for_instance_termination} \tab In a blue/green deployment, overrides any specified wait time and starts terminating instances immediately after the traffic routing is complete\cr
#'  \link[paws.developer.tools:codedeploy_stop_deployment]{stop_deployment} \tab Attempts to stop an ongoing deployment\cr
#'  \link[paws.developer.tools:codedeploy_tag_resource]{tag_resource} \tab Associates the list of tags in the input Tags parameter with the resource identified by the ResourceArn input parameter\cr
#'  \link[paws.developer.tools:codedeploy_untag_resource]{untag_resource} \tab Disassociates a resource from a list of tags\cr
#'  \link[paws.developer.tools:codedeploy_update_application]{update_application} \tab Changes the name of an application\cr
#'  \link[paws.developer.tools:codedeploy_update_deployment_group]{update_deployment_group} \tab Changes information about a deployment group
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname codedeploy
#' @export
codedeploy <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.developer.tools::codedeploy(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' Amazon CodeGuru Profiler
#'
#' @description
#' This section provides documentation for the Amazon CodeGuru Profiler API
#' operations.
#'
#' Amazon CodeGuru Profiler collects runtime performance data from your
#' live applications, and provides recommendations that can help you
#' fine-tune your application performance. Using machine learning
#' algorithms, CodeGuru Profiler can help you find your most expensive
#' lines of code and suggest ways you can improve efficiency and remove CPU
#' bottlenecks.
#'
#' Amazon CodeGuru Profiler provides different visualizations of profiling
#' data to help you identify what code is running on the CPU, see how much
#' time is consumed, and suggest ways to reduce CPU utilization.
#'
#' Amazon CodeGuru Profiler currently supports applications written in all
#' Java virtual machine (JVM) languages and Python. While CodeGuru Profiler
#' supports both visualizations and recommendations for applications
#' written in Java, it can also generate visualizations and a subset of
#' recommendations for applications written in other JVM languages and
#' Python.
#'
#' For more information, see [What is Amazon CodeGuru
#' Profiler](https://docs.aws.amazon.com/codeguru/latest/profiler-ug/what-is-codeguru-profiler.html)
#' in the *Amazon CodeGuru Profiler User Guide*.
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- codeguruprofiler(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- codeguruprofiler()
#' svc$add_notification_channels(
#'   Foo = 123
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.developer.tools:codeguruprofiler_add_notification_channels]{add_notification_channels} \tab Add up to 2 anomaly notifications channels for a profiling group\cr
#'  \link[paws.developer.tools:codeguruprofiler_batch_get_frame_metric_data]{batch_get_frame_metric_data} \tab Returns the time series of values for a requested list of frame metrics from a time period\cr
#'  \link[paws.developer.tools:codeguruprofiler_configure_agent]{configure_agent} \tab Used by profiler agents to report their current state and to receive remote configuration updates\cr
#'  \link[paws.developer.tools:codeguruprofiler_create_profiling_group]{create_profiling_group} \tab Creates a profiling group\cr
#'  \link[paws.developer.tools:codeguruprofiler_delete_profiling_group]{delete_profiling_group} \tab Deletes a profiling group\cr
#'  \link[paws.developer.tools:codeguruprofiler_describe_profiling_group]{describe_profiling_group} \tab Returns a ProfilingGroupDescription object that contains information about the requested profiling group\cr
#'  \link[paws.developer.tools:codeguruprofiler_get_findings_report_account_summary]{get_findings_report_account_summary} \tab Returns a list of FindingsReportSummary objects that contain analysis results for all profiling groups in your AWS account\cr
#'  \link[paws.developer.tools:codeguruprofiler_get_notification_configuration]{get_notification_configuration} \tab Get the current configuration for anomaly notifications for a profiling group\cr
#'  \link[paws.developer.tools:codeguruprofiler_get_policy]{get_policy} \tab Returns the JSON-formatted resource-based policy on a profiling group\cr
#'  \link[paws.developer.tools:codeguruprofiler_get_profile]{get_profile} \tab Gets the aggregated profile of a profiling group for a specified time range\cr
#'  \link[paws.developer.tools:codeguruprofiler_get_recommendations]{get_recommendations} \tab Returns a list of Recommendation objects that contain recommendations for a profiling group for a given time period\cr
#'  \link[paws.developer.tools:codeguruprofiler_list_findings_reports]{list_findings_reports} \tab List the available reports for a given profiling group and time range\cr
#'  \link[paws.developer.tools:codeguruprofiler_list_profile_times]{list_profile_times} \tab Lists the start times of the available aggregated profiles of a profiling group for an aggregation period within the specified time range\cr
#'  \link[paws.developer.tools:codeguruprofiler_list_profiling_groups]{list_profiling_groups} \tab Returns a list of profiling groups\cr
#'  \link[paws.developer.tools:codeguruprofiler_list_tags_for_resource]{list_tags_for_resource} \tab Returns a list of the tags that are assigned to a specified resource\cr
#'  \link[paws.developer.tools:codeguruprofiler_post_agent_profile]{post_agent_profile} \tab Submits profiling data to an aggregated profile of a profiling group\cr
#'  \link[paws.developer.tools:codeguruprofiler_put_permission]{put_permission} \tab Adds permissions to a profiling group's resource-based policy that are provided using an action group\cr
#'  \link[paws.developer.tools:codeguruprofiler_remove_notification_channel]{remove_notification_channel} \tab Remove one anomaly notifications channel for a profiling group\cr
#'  \link[paws.developer.tools:codeguruprofiler_remove_permission]{remove_permission} \tab Removes permissions from a profiling group's resource-based policy that are provided using an action group\cr
#'  \link[paws.developer.tools:codeguruprofiler_submit_feedback]{submit_feedback} \tab Sends feedback to CodeGuru Profiler about whether the anomaly detected by the analysis is useful or not\cr
#'  \link[paws.developer.tools:codeguruprofiler_tag_resource]{tag_resource} \tab Use to assign one or more tags to a resource\cr
#'  \link[paws.developer.tools:codeguruprofiler_untag_resource]{untag_resource} \tab Use to remove one or more tags from a resource\cr
#'  \link[paws.developer.tools:codeguruprofiler_update_profiling_group]{update_profiling_group} \tab Updates a profiling group
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname codeguruprofiler
#' @export
codeguruprofiler <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.developer.tools::codeguruprofiler(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' Amazon CodeGuru Reviewer
#'
#' @description
#' This section provides documentation for the Amazon CodeGuru Reviewer API
#' operations. CodeGuru Reviewer is a service that uses program analysis
#' and machine learning to detect potential defects that are difficult for
#' developers to find and recommends fixes in your Java and Python code.
#'
#' By proactively detecting and providing recommendations for addressing
#' code defects and implementing best practices, CodeGuru Reviewer improves
#' the overall quality and maintainability of your code base during the
#' code review stage. For more information about CodeGuru Reviewer, see the
#' *\href{https://docs.aws.amazon.com/codeguru/latest/reviewer-ug/welcome.html}{Amazon CodeGuru Reviewer User Guide}.*
#'
#' To improve the security of your CodeGuru Reviewer API calls, you can
#' establish a private connection between your VPC and CodeGuru Reviewer by
#' creating an *interface VPC endpoint*. For more information, see
#' [CodeGuru Reviewer and interface VPC endpoints (Amazon Web Services
#' PrivateLink)](https://docs.aws.amazon.com/codeguru/latest/reviewer-ug/vpc-interface-endpoints.html)
#' in the *Amazon CodeGuru Reviewer User Guide*.
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- codegurureviewer(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- codegurureviewer()
#' svc$associate_repository(
#'   Foo = 123
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.developer.tools:codegurureviewer_associate_repository]{associate_repository} \tab Use to associate an Amazon Web Services CodeCommit repository or a repository managed by Amazon Web Services CodeStar Connections with Amazon CodeGuru Reviewer\cr
#'  \link[paws.developer.tools:codegurureviewer_create_code_review]{create_code_review} \tab Use to create a code review with a CodeReviewType of RepositoryAnalysis\cr
#'  \link[paws.developer.tools:codegurureviewer_describe_code_review]{describe_code_review} \tab Returns the metadata associated with the code review along with its status\cr
#'  \link[paws.developer.tools:codegurureviewer_describe_recommendation_feedback]{describe_recommendation_feedback} \tab Describes the customer feedback for a CodeGuru Reviewer recommendation\cr
#'  \link[paws.developer.tools:codegurureviewer_describe_repository_association]{describe_repository_association} \tab Returns a RepositoryAssociation object that contains information about the requested repository association\cr
#'  \link[paws.developer.tools:codegurureviewer_disassociate_repository]{disassociate_repository} \tab Removes the association between Amazon CodeGuru Reviewer and a repository\cr
#'  \link[paws.developer.tools:codegurureviewer_list_code_reviews]{list_code_reviews} \tab Lists all the code reviews that the customer has created in the past 90 days\cr
#'  \link[paws.developer.tools:codegurureviewer_list_recommendation_feedback]{list_recommendation_feedback} \tab Returns a list of RecommendationFeedbackSummary objects that contain customer recommendation feedback for all CodeGuru Reviewer users\cr
#'  \link[paws.developer.tools:codegurureviewer_list_recommendations]{list_recommendations} \tab Returns the list of all recommendations for a completed code review\cr
#'  \link[paws.developer.tools:codegurureviewer_list_repository_associations]{list_repository_associations} \tab Returns a list of RepositoryAssociationSummary objects that contain summary information about a repository association\cr
#'  \link[paws.developer.tools:codegurureviewer_list_tags_for_resource]{list_tags_for_resource} \tab Returns the list of tags associated with an associated repository resource\cr
#'  \link[paws.developer.tools:codegurureviewer_put_recommendation_feedback]{put_recommendation_feedback} \tab Stores customer feedback for a CodeGuru Reviewer recommendation\cr
#'  \link[paws.developer.tools:codegurureviewer_tag_resource]{tag_resource} \tab Adds one or more tags to an associated repository\cr
#'  \link[paws.developer.tools:codegurureviewer_untag_resource]{untag_resource} \tab Removes a tag from an associated repository
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname codegurureviewer
#' @export
codegurureviewer <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.developer.tools::codegurureviewer(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' AWS CodePipeline
#'
#' @description
#' CodePipeline
#'
#' **Overview**
#'
#' This is the CodePipeline API Reference. This guide provides descriptions
#' of the actions and data types for CodePipeline. Some functionality for
#' your pipeline can only be configured through the API. For more
#' information, see the [CodePipeline User
#' Guide](https://docs.aws.amazon.com/codepipeline/latest/userguide/welcome.html).
#'
#' You can use the CodePipeline API to work with pipelines, stages,
#' actions, and transitions.
#'
#' *Pipelines* are models of automated release processes. Each pipeline is
#' uniquely named, and consists of stages, actions, and transitions.
#'
#' You can work with pipelines by calling:
#'
#' -   `create_pipeline`, which creates a
#'     uniquely named pipeline.
#'
#' -   `delete_pipeline`, which deletes the
#'     specified pipeline.
#'
#' -   `get_pipeline`, which returns
#'     information about the pipeline structure and pipeline metadata,
#'     including the pipeline Amazon Resource Name (ARN).
#'
#' -   `get_pipeline_execution`,
#'     which returns information about a specific execution of a pipeline.
#'
#' -   `get_pipeline_state`, which
#'     returns information about the current state of the stages and
#'     actions of a pipeline.
#'
#' -   `list_action_executions`,
#'     which returns action-level details for past executions. The details
#'     include full stage and action-level details, including individual
#'     action duration, status, any errors that occurred during the
#'     execution, and input and output artifact location details.
#'
#' -   `list_pipelines`, which gets a
#'     summary of all of the pipelines associated with your account.
#'
#' -   `list_pipeline_executions`,
#'     which gets a summary of the most recent executions for a pipeline.
#'
#' -   `start_pipeline_execution`,
#'     which runs the most recent revision of an artifact through the
#'     pipeline.
#'
#' -   `stop_pipeline_execution`,
#'     which stops the specified pipeline execution from continuing through
#'     the pipeline.
#'
#' -   `update_pipeline`, which updates a
#'     pipeline with edits or changes to the structure of the pipeline.
#'
#' Pipelines include *stages*. Each stage contains one or more actions that
#' must complete before the next stage begins. A stage results in success
#' or failure. If a stage fails, the pipeline stops at that stage and
#' remains stopped until either a new version of an artifact appears in the
#' source location, or a user takes action to rerun the most recent
#' artifact through the pipeline. You can call
#' `get_pipeline_state`, which displays
#' the status of a pipeline, including the status of stages in the
#' pipeline, or `get_pipeline`, which returns
#' the entire structure of the pipeline, including the stages of that
#' pipeline. For more information about the structure of stages and
#' actions, see [CodePipeline Pipeline Structure
#' Reference](https://docs.aws.amazon.com/codepipeline/latest/userguide/reference-pipeline-structure.html).
#'
#' Pipeline stages include *actions* that are categorized into categories
#' such as source or build actions performed in a stage of a pipeline. For
#' example, you can use a source action to import artifacts into a pipeline
#' from a source such as Amazon S3. Like stages, you do not work with
#' actions directly in most cases, but you do define and interact with
#' actions when working with pipeline operations such as
#' `create_pipeline` and
#' `get_pipeline_state`. Valid action
#' categories are:
#'
#' -   Source
#'
#' -   Build
#'
#' -   Test
#'
#' -   Deploy
#'
#' -   Approval
#'
#' -   Invoke
#'
#' Pipelines also include *transitions*, which allow the transition of
#' artifacts from one stage to the next in a pipeline after the actions in
#' one stage complete.
#'
#' You can work with transitions by calling:
#'
#' -   `disable_stage_transition`,
#'     which prevents artifacts from transitioning to the next stage in a
#'     pipeline.
#'
#' -   `enable_stage_transition`,
#'     which enables transition of artifacts between stages in a pipeline.
#'
#' **Using the API to integrate with CodePipeline**
#'
#' For third-party integrators or developers who want to create their own
#' integrations with CodePipeline, the expected sequence varies from the
#' standard API user. To integrate with CodePipeline, developers need to
#' work with the following items:
#'
#' **Jobs**, which are instances of an action. For example, a job for a
#' source action might import a revision of an artifact from a source.
#'
#' You can work with jobs by calling:
#'
#' -   `acknowledge_job`, which confirms
#'     whether a job worker has received the specified job.
#'
#' -   `get_job_details`, which returns the
#'     details of a job.
#'
#' -   `poll_for_jobs`, which determines
#'     whether there are any jobs to act on.
#'
#' -   `put_job_failure_result`,
#'     which provides details of a job failure.
#'
#' -   `put_job_success_result`,
#'     which provides details of a job success.
#'
#' **Third party jobs**, which are instances of an action created by a
#' partner action and integrated into CodePipeline. Partner actions are
#' created by members of the Amazon Web Services Partner Network.
#'
#' You can work with third party jobs by calling:
#'
#' -   `acknowledge_third_party_job`,
#'     which confirms whether a job worker has received the specified job.
#'
#' -   `get_third_party_job_details`,
#'     which requests the details of a job for a partner action.
#'
#' -   `poll_for_third_party_jobs`,
#'     which determines whether there are any jobs to act on.
#'
#' -   `put_third_party_job_failure_result`,
#'     which provides details of a job failure.
#'
#' -   `put_third_party_job_success_result`,
#'     which provides details of a job success.
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- codepipeline(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- codepipeline()
#' svc$acknowledge_job(
#'   Foo = 123
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.developer.tools:codepipeline_acknowledge_job]{acknowledge_job} \tab Returns information about a specified job and whether that job has been received by the job worker\cr
#'  \link[paws.developer.tools:codepipeline_acknowledge_third_party_job]{acknowledge_third_party_job} \tab Confirms a job worker has received the specified job\cr
#'  \link[paws.developer.tools:codepipeline_create_custom_action_type]{create_custom_action_type} \tab Creates a new custom action that can be used in all pipelines associated with the Amazon Web Services account\cr
#'  \link[paws.developer.tools:codepipeline_create_pipeline]{create_pipeline} \tab Creates a pipeline\cr
#'  \link[paws.developer.tools:codepipeline_delete_custom_action_type]{delete_custom_action_type} \tab Marks a custom action as deleted\cr
#'  \link[paws.developer.tools:codepipeline_delete_pipeline]{delete_pipeline} \tab Deletes the specified pipeline\cr
#'  \link[paws.developer.tools:codepipeline_delete_webhook]{delete_webhook} \tab Deletes a previously created webhook by name\cr
#'  \link[paws.developer.tools:codepipeline_deregister_webhook_with_third_party]{deregister_webhook_with_third_party} \tab Removes the connection between the webhook that was created by CodePipeline and the external tool with events to be detected\cr
#'  \link[paws.developer.tools:codepipeline_disable_stage_transition]{disable_stage_transition} \tab Prevents artifacts in a pipeline from transitioning to the next stage in the pipeline\cr
#'  \link[paws.developer.tools:codepipeline_enable_stage_transition]{enable_stage_transition} \tab Enables artifacts in a pipeline to transition to a stage in a pipeline\cr
#'  \link[paws.developer.tools:codepipeline_get_action_type]{get_action_type} \tab Returns information about an action type created for an external provider, where the action is to be used by customers of the external provider\cr
#'  \link[paws.developer.tools:codepipeline_get_job_details]{get_job_details} \tab Returns information about a job\cr
#'  \link[paws.developer.tools:codepipeline_get_pipeline]{get_pipeline} \tab Returns the metadata, structure, stages, and actions of a pipeline\cr
#'  \link[paws.developer.tools:codepipeline_get_pipeline_execution]{get_pipeline_execution} \tab Returns information about an execution of a pipeline, including details about artifacts, the pipeline execution ID, and the name, version, and status of the pipeline\cr
#'  \link[paws.developer.tools:codepipeline_get_pipeline_state]{get_pipeline_state} \tab Returns information about the state of a pipeline, including the stages and actions\cr
#'  \link[paws.developer.tools:codepipeline_get_third_party_job_details]{get_third_party_job_details} \tab Requests the details of a job for a third party action\cr
#'  \link[paws.developer.tools:codepipeline_list_action_executions]{list_action_executions} \tab Lists the action executions that have occurred in a pipeline\cr
#'  \link[paws.developer.tools:codepipeline_list_action_types]{list_action_types} \tab Gets a summary of all CodePipeline action types associated with your account\cr
#'  \link[paws.developer.tools:codepipeline_list_pipeline_executions]{list_pipeline_executions} \tab Gets a summary of the most recent executions for a pipeline\cr
#'  \link[paws.developer.tools:codepipeline_list_pipelines]{list_pipelines} \tab Gets a summary of all of the pipelines associated with your account\cr
#'  \link[paws.developer.tools:codepipeline_list_tags_for_resource]{list_tags_for_resource} \tab Gets the set of key-value pairs (metadata) that are used to manage the resource\cr
#'  \link[paws.developer.tools:codepipeline_list_webhooks]{list_webhooks} \tab Gets a listing of all the webhooks in this Amazon Web Services Region for this account\cr
#'  \link[paws.developer.tools:codepipeline_poll_for_jobs]{poll_for_jobs} \tab Returns information about any jobs for CodePipeline to act on\cr
#'  \link[paws.developer.tools:codepipeline_poll_for_third_party_jobs]{poll_for_third_party_jobs} \tab Determines whether there are any third party jobs for a job worker to act on\cr
#'  \link[paws.developer.tools:codepipeline_put_action_revision]{put_action_revision} \tab Provides information to CodePipeline about new revisions to a source\cr
#'  \link[paws.developer.tools:codepipeline_put_approval_result]{put_approval_result} \tab Provides the response to a manual approval request to CodePipeline\cr
#'  \link[paws.developer.tools:codepipeline_put_job_failure_result]{put_job_failure_result} \tab Represents the failure of a job as returned to the pipeline by a job worker\cr
#'  \link[paws.developer.tools:codepipeline_put_job_success_result]{put_job_success_result} \tab Represents the success of a job as returned to the pipeline by a job worker\cr
#'  \link[paws.developer.tools:codepipeline_put_third_party_job_failure_result]{put_third_party_job_failure_result} \tab Represents the failure of a third party job as returned to the pipeline by a job worker\cr
#'  \link[paws.developer.tools:codepipeline_put_third_party_job_success_result]{put_third_party_job_success_result} \tab Represents the success of a third party job as returned to the pipeline by a job worker\cr
#'  \link[paws.developer.tools:codepipeline_put_webhook]{put_webhook} \tab Defines a webhook and returns a unique webhook URL generated by CodePipeline\cr
#'  \link[paws.developer.tools:codepipeline_register_webhook_with_third_party]{register_webhook_with_third_party} \tab Configures a connection between the webhook that was created and the external tool with events to be detected\cr
#'  \link[paws.developer.tools:codepipeline_retry_stage_execution]{retry_stage_execution} \tab Resumes the pipeline execution by retrying the last failed actions in a stage\cr
#'  \link[paws.developer.tools:codepipeline_start_pipeline_execution]{start_pipeline_execution} \tab Starts the specified pipeline\cr
#'  \link[paws.developer.tools:codepipeline_stop_pipeline_execution]{stop_pipeline_execution} \tab Stops the specified pipeline execution\cr
#'  \link[paws.developer.tools:codepipeline_tag_resource]{tag_resource} \tab Adds to or modifies the tags of the given resource\cr
#'  \link[paws.developer.tools:codepipeline_untag_resource]{untag_resource} \tab Removes tags from an Amazon Web Services resource\cr
#'  \link[paws.developer.tools:codepipeline_update_action_type]{update_action_type} \tab Updates an action type that was created with any supported integration model, where the action type is to be used by customers of the action type provider\cr
#'  \link[paws.developer.tools:codepipeline_update_pipeline]{update_pipeline} \tab Updates a specified pipeline with edits or changes to its structure
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname codepipeline
#' @export
codepipeline <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.developer.tools::codepipeline(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' AWS CodeStar
#'
#' @description
#' This is the API reference for AWS CodeStar. This reference provides
#' descriptions of the operations and data types for the AWS CodeStar API
#' along with usage examples.
#'
#' You can use the AWS CodeStar API to work with:
#'
#' Projects and their resources, by calling the following:
#'
#' -   `delete_project`, which deletes a
#'     project.
#'
#' -   `describe_project`, which lists the
#'     attributes of a project.
#'
#' -   `list_projects`, which lists all projects
#'     associated with your AWS account.
#'
#' -   `list_resources`, which lists the
#'     resources associated with a project.
#'
#' -   `list_tags_for_project`, which
#'     lists the tags associated with a project.
#'
#' -   `tag_project`, which adds tags to a project.
#'
#' -   `untag_project`, which removes tags from a
#'     project.
#'
#' -   `update_project`, which updates the
#'     attributes of a project.
#'
#' Teams and team members, by calling the following:
#'
#' -   `associate_team_member`, which
#'     adds an IAM user to the team for a project.
#'
#' -   `disassociate_team_member`,
#'     which removes an IAM user from the team for a project.
#'
#' -   `list_team_members`, which lists all
#'     the IAM users in the team for a project, including their roles and
#'     attributes.
#'
#' -   `update_team_member`, which updates a
#'     team member's attributes in a project.
#'
#' Users, by calling the following:
#'
#' -   `create_user_profile`, which creates
#'     a user profile that contains data associated with the user across
#'     all projects.
#'
#' -   `delete_user_profile`, which deletes
#'     all user profile information across all projects.
#'
#' -   `describe_user_profile`, which
#'     describes the profile of a user.
#'
#' -   `list_user_profiles`, which lists all
#'     user profiles.
#'
#' -   `update_user_profile`, which updates
#'     the profile for a user.
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The default is 60 seconds.}
#' \item{\strong{s3_force_path_style}:} {Set this to `true` to force the request to use path-style addressing, i.e. `http://s3.amazonaws.com/BUCKET/KEY`.}
#' \item{\strong{sts_regional_endpoint}:} {Set sts regional endpoint resolver to regional or legacy \url{https://docs.aws.amazon.com/sdkref/latest/guide/feature-sts-regionalized-endpoints.html}}
#' }
#' @param
#' credentials
#' Optional credentials shorthand for the config parameter
#' \itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' }
#' @param
#' endpoint
#' Optional shorthand for complete URL to use for the constructed client.
#' @param
#' region
#' Optional shorthand for AWS Region used in instantiating the client.
#'
#' @section Service syntax:
#' ```
#' svc <- codestar(
#'   config = list(
#'     credentials = list(
#'       creds = list(
#'         access_key_id = "string",
#'         secret_access_key = "string",
#'         session_token = "string"
#'       ),
#'       profile = "string",
#'       anonymous = "logical"
#'     ),
#'     endpoint = "string",
#'     region = "string",
#'     close_connection = "logical",
#'     timeout = "numeric",
#'     s3_force_path_style = "logical",
#'     sts_regional_endpoint = "string"
#'   ),
#'   credentials = list(
#'     creds = list(
#'       access_key_id = "string",
#'       secret_access_key = "string",
#'       session_token = "string"
#'     ),
#'     profile = "string",
#'     anonymous = "logical"
#'   ),
#'   endpoint = "string",
#'   region = "string"
#' )
#' ```
#'
#' @examples
#' \dontrun{
#' svc <- codestar()
#' svc$associate_team_member(
#'   Foo = 123
#' )
#' }
#'
#' @section Operations:
#' \tabular{ll}{
#'  \link[paws.developer.tools:codestar_associate_team_member]{associate_team_member} \tab Adds an IAM user to the team for an AWS CodeStar project\cr
#'  \link[paws.developer.tools:codestar_create_project]{create_project} \tab Creates a project, including project resources\cr
#'  \link[paws.developer.tools:codestar_create_user_profile]{create_user_profile} \tab Creates a profile for a user that includes user preferences, such as the display name and email address assocciated with the user, in AWS CodeStar\cr
#'  \link[paws.developer.tools:codestar_delete_project]{delete_project} \tab Deletes a project, including project resources\cr
#'  \link[paws.developer.tools:codestar_delete_user_profile]{delete_user_profile} \tab Deletes a user profile in AWS CodeStar, including all personal preference data associated with that profile, such as display name and email address\cr
#'  \link[paws.developer.tools:codestar_describe_project]{describe_project} \tab Describes a project and its resources\cr
#'  \link[paws.developer.tools:codestar_describe_user_profile]{describe_user_profile} \tab Describes a user in AWS CodeStar and the user attributes across all projects\cr
#'  \link[paws.developer.tools:codestar_disassociate_team_member]{disassociate_team_member} \tab Removes a user from a project\cr
#'  \link[paws.developer.tools:codestar_list_projects]{list_projects} \tab Lists all projects in AWS CodeStar associated with your AWS account\cr
#'  \link[paws.developer.tools:codestar_list_resources]{list_resources} \tab Lists resources associated with a project in AWS CodeStar\cr
#'  \link[paws.developer.tools:codestar_list_tags_for_project]{list_tags_for_project} \tab Gets the tags for a project\cr
#'  \link[paws.developer.tools:codestar_list_team_members]{list_team_members} \tab Lists all team members associated with a project\cr
#'  \link[paws.developer.tools:codestar_list_user_profiles]{list_user_profiles} \tab Lists all the user profiles configured for your AWS account in AWS CodeStar\cr
#'  \link[paws.developer.tools:codestar_tag_project]{tag_project} \tab Adds tags to a project\cr
#'  \link[paws.developer.tools:codestar_untag_project]{untag_project} \tab Removes tags from a project\cr
#'  \link[paws.developer.tools:codestar_update_project]{update_project} \tab Updates a project in AWS CodeStar\cr
#'  \link[paws.developer.tools:codestar_update_team_member]{update_team_member} \tab Updates a team member's attributes in an AWS CodeStar project\cr
#'  \link[paws.developer.tools:codestar_update_user_profile]{update_user_profile} \tab Updates a user's profile in AWS CodeStar
#' }
#'
#' @return
#' A client for the service. You can call the service's operations using
#' syntax like `svc$operation(...)`, where `svc` is the name you've assigned
#' to the client. The available operations are listed in the
#' Operations section.
#'
#' @rdname codestar
#' @export
codestar <- function(config = list(), credentials = list(), endpoint = NULL, region = NULL) {
  paws.developer.tools::codestar(
    config = config,
    credentials = credentials,
    endpoint = endpoint,
    region = region
  )
}

#' AWS CodeStar connections
#'
#' @description
#' AWS CodeStar Connections
#'
#' This Amazon Web Services CodeStar Connections API Reference provides
#' descriptions and usage examples of the operations and data types for the
#' Amazon Web Services CodeStar Connections API. You can use the
#' connections API to work with connections and installations.
#'
#' *Connections* are configurations that you use to connect Amazon Web
#' Services resources to external code repositories. Each connection is a
#' resource that can be given to services such as CodePipeline to connect
#' to a third-party repository such as Bitbucket. For example, you can add
#' the connection in CodePipeline so that it triggers your pipeline when a
#' code change is made to your third-party code repository. Each connection
#' is named and associated with a unique ARN that is used to reference the
#' connection.
#'
#' When you create a connection, the console initiates a third-party
#' connection handshake. *Installations* are the apps that are used to
#' conduct this handshake. For example, the installation for the Bitbucket
#' provider type is the Bitbucket app. When you create a connection, you
#' can choose an existing installation or create one.
#'
#' When you want to create a connection to an installed provider type such
#' as GitHub Enterprise Server, you create a *host* for your connections.
#'
#' You can work with connections by calling:
#'
#' -   `create_connection`, which
#'     creates a uniquely named connection that can be referenced by
#'     services such as CodePipeline.
#'
#' -   `delete_connection`, which
#'     deletes the specified connection.
#'
#' -   `get_connection`, which
#'     returns information about the connection, including the connection
#'     status.
#'
#' -   `list_connections`, which
#'     lists the connections associated with your account.
#'
#' You can work with hosts by calling:
#'
#' -   `create_host`, which creates a
#'     host that represents the infrastructure where your provider is
#'     installed.
#'
#' -   `delete_host`, which deletes the
#'     specified host.
#'
#' -   `get_host`, which returns
#'     information about the host, including the setup status.
#'
#' -   `list_hosts`, which lists the
#'     hosts associated with your account.
#'
#' You can work with tags in Amazon Web Services CodeStar Connections by
#' calling the following:
#'
#' -   `list_tags_for_resource`,
#'     which gets information about Amazon Web Services tags for a
#'     specified Amazon Resource Name (ARN) in Amazon Web Services CodeStar
#'     Connections.
#'
#' -   `tag_resource`, which adds or
#'     updates tags for a resource in Amazon Web Services CodeStar
#'     Connections.
#'
#' -   `untag_resource`, which
#'     removes tags for a resource in Amazon Web Services CodeStar
#'     Connections.
#'
#' For information about how to use Amazon Web Services CodeStar
#' Connections, see the [Developer Tools User
#' Guide](https://docs.aws.amazon.com/dtconsole/latest/userguide/welcome-connections.html).
#'
#' @param
#' config
#' Optional configuration of credentials, endpoint, and/or region.
#' \itemize{
#' \item{\strong{credentials}:} {\itemize{
#' \item{\strong{creds}:} {\itemize{
#' \item{\strong{access_key_id}:} {AWS access key ID}
#' \item{\strong{secret_access_key}:} {AWS secret access key}
#' \item{\strong{session_token}:} {AWS temporary session token}
#' }}
#' \item{\strong{profile}:} {The name of a profile to use. If not given, then the default profile is used.}
#' \item{\strong{anonymous}:} {Set anonymous credentials.}
#' \item{\strong{endpoint}:} {The complete URL to use for the constructed client.}
#' \item{\strong{region}:} {The AWS Region used in instantiating the client.}
#' }}
#' \item{\strong{close_connection}:} {Immediately close all HTTP connections.}
#' \item{\strong{timeout}:} {The time in seconds till a timeout exception is thrown when attempting to make a connection. The de