% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/generate.R
\name{generateFactorItems}
\alias{generateFactorItems}
\title{Generate pairwise comparison data with a common factor that
accounts for some proportion of the variance}
\usage{
generateFactorItems(df, prop, th = 0.5, scale = 1, name)
}
\arguments{
\item{df}{a data frame with pairs of vertices given in columns \code{pa1} and \code{pa2}, and item response data in other columns}

\item{prop}{the number of items or a vector of proportions of variance}

\item{th}{a vector of thresholds}

\item{scale}{the scaling constant}

\item{name}{a vector of item names}
}
\description{
Imagine that there are people that play in tournaments of more than
one board game. For example, the computer player AlphaZero (Silver
et al. 2018) has trained to play chess, shogi, and Go. We can take
the tournament match outcome data and find rankings among the
players for each of these games. We may also suspect that there is
a latent board game skill that accounts for some proportion of the
variance in the per-board game rankings.
}
\details{
The pairwise comparison item response model has thresholds and a
scale parameter similar to the partial credit model (Masters,
1982). The model is cumbersome to describe in traditional
mathematical notation, but the R code is fairly straightforward,

\preformatted{
softmax <- function(y) exp(y) / sum(exp(y))

cmp_probs <- function(scale, pa1, pa2, thRaw) {
  th <- cumsum(thRaw)
  diff <- scale * (pa2 - pa1)
  unsummed <- c(0, c(diff + rev(th)), c(diff - th), use.names = FALSE)
  softmax(cumsum(unsummed))
}
}

The function \code{cmp_probs} takes a \code{scale} constant, the
latent scores for two objects \code{pa1} and \code{pa2}, and a
vector of thresholds \code{thRaw}. The thresholds are parameterized
as the difference from the previous threshold. For example,
thresholds c(0.5, 0.5) are not at the same location but are at
locations c(0.5, 1.0). Thresholds are symmetric. If there is one
thresholds then the model admits three possible response outcomes
(e.g. win, tie, and lose). Responses are always stored centered
with zero representing a tie. Therefore, it is necessary to add one
plus the number of thresholds to response data to index into the
vector returned by \code{cmp_probs}. For example, if our response
data (-1, 0, 1) has one threshold then we would add 2 (1 + 1
threshold) to obtain the indices (1, 2, 3).

Use \code{\link{itemModelExplorer}} to explore the item model. In
this \pkg{shiny} app, the \emph{discrimination} parameter does what is
customary in item response models. However, it is not difficult to
show that discrimination is a function of thresholds and
scale. That is, discrimination is not an independent parameter and
cannot be estimated. In pairwise comparison models, discrimination
and measurement error are confounded.
}
\examples{
df <- twoLevelGraph(letters[1:10], 100)
df <- generateFactorItems(df, 3)
}
\references{
Masters, G. N. (1982). A Rasch model for partial credit scoring.
Psychometrika, 47, 149–174. doi: 10.1007/BF02296272

Silver, D., Hubert, T., Schrittwieser, J., Antonoglou, I., Lai, M.,
Guez, A., ... & Lillicrap, T. (2018). A general reinforcement
learning algorithm that masters chess, shogi, and Go through
self-play. Science, 362(6419), 1140-1144.
}
\seealso{
Other item generators: \code{\link{generateCovItems}},
  \code{\link{generateItem}}
}
\concept{item generators}
