.coxfit <- function(response) {

  n <- nrow(response)
  type <- attr(response, "type")
  if (!type %in% c("right", "counting"))
    stop("Cox model doesn't support \"", type, "\" survival data")
 
  if (ncol(response) == 2) {
    time <- response[,1]
    status <- response[,2]
    Riskset <- outer(time, time, "<=")
  } else {
    time <- response[,2]
    start <- response[,1]
    status <- response[,3]
    Riskset <- outer(time, time, "<=") & outer(time, start, ">")
  }    

  # Finds local gradient and subject weights
  fit <- function(lp, leftout) {

    if (!missing(leftout)) {
      status <- status[!leftout]
      time <- time[!leftout]
      Riskset <- Riskset[!leftout, !leftout]
    }
    ws <- drop(exp(lp))
    if (any(ws == Inf | ws == 0)) { 
      ws <- 1e-10 + 1e10 * status
      exploded <- TRUE
    } else {
      exploded <- FALSE
    }

    breslows <- drop(status / Riskset %*% ws)
    breslow <- drop(breslows[status==1] %*% Riskset[status==1,,drop=FALSE])
    
    # The martingale residuals
    residuals <- status - breslow * ws
    
    # The loglikelihood
    if (!exploded)
      loglik <- -sum(ws * breslow) + sum(log(breslows[status==1])) + sum(lp[status==1])
    else
      loglik <- NA

    # The weights matrix
    Pij <- outer(ws, breslows) * t(Riskset)
    W <- list(P = Pij[,status==1], diagW = breslow * ws)        # construct: W = diag(diagW) - P %*% t(P)
    
    # The fitted baseline
    dtimes <- time[status==1]
    sdtimes <- sort(dtimes)
    basecumhaz <- cumsum(breslows[status==1][sort.list(dtimes)])
    uniquetimes <- sapply(seq_along(sdtimes)[-length(sdtimes)], function(i) sdtimes[i] != sdtimes[i+1])
    basesurv <- exp(-basecumhaz[uniquetimes])
    if (max(dtimes) < max(time)) {
      basetimes <- c(0, sdtimes[uniquetimes], max(time))
      basesurv <- c(1, basesurv, basesurv[length(basesurv)])
    } else {
      basetimes <- c(0, sdtimes[uniquetimes])
      basesurv <- c(1, basesurv)
    }
    
    baseline <- new("breslow")
    baseline@time <- basetimes
    baseline@curves <- matrix(basesurv,1,byrow=TRUE)

    return(list(residuals = residuals, loglik = loglik, W = W, lp = lp, fitted = exp(lp), nuisance = list(baseline = baseline)))
  }
  
  #cross-validated likelihood
  cvl <- function(lp, leftout)
  { 
    ws <- exp(lp)
    somw <- apply(Riskset, 1, function(rr) sum(ws[rr]))
    cvls <- numeric(length(leftout))
    for (k in which(leftout)) {
      pij <- ws[k] / somw
      cvls[k] <- sum(log(1 - pij[(status ==1) & !Riskset[k,]])) + status[k] * log(pij[k])
    }
    return(sum(cvls[leftout]))
  }
 
  # mapping from the linear predictor lp to an actual prediction
  prediction <- function(lp, nuisance) {
    out <- nuisance$baseline
    out@curves <- nuisance$baseline@curves ^ exp(lp)
    out
  }
 
  return(list(fit = fit, cvl = cvl, prediction = prediction))
}



.coxgamma <- function(response, unpenalized, data) {

  if (is.matrix(unpenalized)) {
    if (ncol(unpenalized) > 0) 
      startgamma <- coefficients(coxph(response ~ ., data = as.data.frame(unpenalized), method = "breslow"))
    else
      startgamma <- numeric(0)
  } else {
    .response <- response
    form <- as.formula(paste(".response~", paste(c("1", attr(terms(unpenalized),"term.labels")), collapse="+"))) 
    startgamma <- coefficients(coxph(form, data = data, method = "breslow"))
  }
}

                                        
# merges predicted survival curves with different time points
# input: a list of breslow objects
.coxmerge <- function(predictions) {
                      
  times <- sort(unique(unlist(lapply(predictions, time))))
  curves <- sapply(predictions, function(pred) {
    res <- rep(NA, length(times))
    res[times %in% time(pred)] <- pred@curves[1,]
    # We interpolate all NAs except in the tail
    endnas <- rev(cumsum(is.na(rev(res)))==1:length(res)) 
    ready <- !any(is.na(res[!endnas]))
    while (!ready) {
      nas <- which(is.na(res[!endnas]))
      ready <- !any(is.na(res[nas-1]))
      res[nas] <- res[nas-1]
    }
    res
  })
  out <- new("breslow")
  out@time <- times
  out@curves <-  t(curves)
  out
}





