% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/performance_pencox.R
\name{performance_pencox}
\alias{performance_pencox}
\title{Predictive performance of the penalized Cox model
with time-independent covariates}
\usage{
performance_pencox(fitted_pencox, metric = c("tdauc", "c", "brier"),
  times = c(2, 3), n.cores = 1, verbose = TRUE)
}
\arguments{
\item{fitted_pencox}{the output of \code{\link{pencox}}}

\item{metric}{the desired performance measure(s). Options include: 'tdauc',
'c' and 'brier'}

\item{times}{numeric vector with the time points at which
to estimate the time-dependent AUC and time-dependent Brier score}

\item{n.cores}{number of cores to use to parallelize part of
the computations. If \code{ncores = 1} (default), 
no parallelization is done. Pro tip: you can use 
\code{parallel::detectCores()} to check how many 
cores are available on your computer}

\item{verbose}{if \code{TRUE} (default and recommended value), information
on the ongoing computations is printed in the console}
}
\value{
A list containing the following objects:
\itemize{
\item \code{call}: the function call;
\item \code{concordance}: a data frame with the naive and
optimism-corrected estimates of the concordance (C) index;
\item \code{tdAUC}: a data frame with the naive and
optimism-corrected estimates of the time-dependent AUC
at the desired time points.
}
}
\description{
This function computes the naive and optimism-corrected
measures of performance (C index, time-dependent AUC and time-dependent 
Brier score) for a penalized Cox model with time-independent covariates. 
The optimism correction is computed based on a cluster bootstrap
optimism correction procedure (CBOCP, Signorelli et al., 2021)
}
\examples{
# generate example data
set.seed(1234)
p = 4 # number of longitudinal predictors
simdata = simulate_prclmm_data(n = 100, p = p, p.relev = 2, 
             seed = 123, t.values = c(0, 0.5, 1, 1.5, 2))
# create dataframe with baseline measurements only
baseline.visits = simdata$long.data[which(!duplicated(simdata$long.data$id)),]
df = merge(simdata$surv.data, baseline.visits, by = 'id')
df = df[ , -c(5:6)]

do.bootstrap = FALSE
# IMPORTANT: set do.bootstrap = TRUE to compute the optimism correction!
n.boots = ifelse(do.bootstrap, 100, 0)
more.cores = FALSE
# IMPORTANT: set more.cores = TRUE to speed computations up!
if (!more.cores) n.cores = 2
if (more.cores) {
   # identify number of available cores on your machine
   n.cores = parallel::detectCores()
   if (is.na(n.cores)) n.cores = 2
}

form = as.formula(~ baseline.age + marker1 + marker2
                     + marker3 + marker4)
base.pcox = pencox(data = df, 
              formula = form, 
              n.boots = n.boots, n.cores = n.cores) 
ls(base.pcox)
                   
# compute the performance measures
perf = performance_pencox(fitted_pencox = base.pcox, 
          metric = 'tdauc', times = 3:5, n.cores = n.cores)
 # use metric = 'brier' for the Brier score and metric = 'c' for the
 # concordance index

# time-dependent AUC estimates:
ls(perf)
perf$tdAUC
}
\references{
Signorelli, M. (2023). pencal: an R Package for the Dynamic 
Prediction of Survival with Many Longitudinal Predictors.
arXiv preprint: arXiv:2309.15600

Signorelli, M., Spitali, P., Al-Khalili Szigyarto, C, 
The MARK-MD Consortium, Tsonaka, R. (2021). 
Penalized regression calibration: a method for the prediction 
of survival outcomes using complex longitudinal and 
high-dimensional data. Statistics in Medicine, 40 (27), 6178-6196.
DOI: 10.1002/sim.9178
}
\seealso{
\code{\link{pencox}}
}
\author{
Mirko Signorelli
}
