% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fit_aq_response.R
\name{fit_aq_response}
\alias{fit_aq_response}
\title{Fitting light responses of net CO2 assimilation}
\usage{
fit_aq_response(
  data,
  varnames = list(A_net = "A_net", PPFD = "PPFD"),
  usealpha_Q = FALSE,
  alpha_Q = 0.84,
  title = NULL
)
}
\arguments{
\item{data}{Dataframe containing CO2 assimilation light response}

\item{varnames}{Variable names where varnames = list(A_net = "A_net",
PPFD = "PPFD"). A_net is net CO2 assimilation in umol m-2 s-1, PPFD is
incident irradiance. PPFD can be corrected for light absorbance by using
useapha_Q and setting alpha_Q.}

\item{usealpha_Q}{Correct light intensity for absorbance? Default is FALSE.}

\item{alpha_Q}{Absorbance of incident light. Default value is 0.84.}

\item{title}{Title for graph}
}
\value{
fit_aq_response fits the light response of net CO2 assimilation.
Output is a dataframe containing light saturated net CO2 assimilation,
quantum yield of CO2 assimilation (phi_J), curvature of the light response
(theta_J), respiration (Rd), light compensation point (LCP), and residual
sum of squares (resid_SS). Note that Rd fitted in this way is essentially
the same as the Kok method, and represents a respiration value in the
light that may not be accurate. Rd output should thus be interpreted more
as a residual parameter to ensure an accurate fit of the light response
parameters. Model originally from Marshall et al. 1980.
}
\description{
Fitting light responses of net CO2 assimilation
}
\examples{
\donttest{
# Read in your data
# Note that this data is coming from data supplied by the package
# hence the complicated argument in read.csv()
# This dataset is a CO2 by light response curve for a single sunflower
data <- read.csv(system.file("extdata", "A_Ci_Q_data_1.csv",
  package = "photosynthesis"
))

# Fit many AQ curves
# Set your grouping variable
# Here we are grouping by CO2_s and individual
data$C_s <- (round(data$CO2_s, digits = 0))

# For this example we need to round sequentially due to CO2_s setpoints
data$C_s <- as.factor(round(data$C_s, digits = -1))

# To fit one AQ curve
fit <- fit_aq_response(data[data$C_s == 600, ],
  varnames = list(
    A_net = "A",
    PPFD = "Qin"
  )
)

# Print model summary
summary(fit[[1]])

# Print fitted parameters
fit[[2]]

# Print graph
fit[[3]]

# Fit many curves
fits <- fit_many(
  data = data,
  varnames = list(
    A_net = "A",
    PPFD = "Qin",
    group = "C_s"
  ),
  funct = fit_aq_response,
  group = "C_s"
)

# Look at model summary for a given fit
# First set of double parentheses selects an individual group value
# Second set selects an element of the sublist
summary(fits[[3]][[1]])

# Print the parameters
fits[[3]][[2]]

# Print the graph
fits[[3]][[3]]

# Compile graphs into a list for plotting
fits_graphs <- compile_data(fits,
  list_element = 3
)

# Compile parameters into dataframe for analysis
fits_pars <- compile_data(fits,
  output_type = "dataframe",
  list_element = 2
)
}
}
\references{
Marshall B, Biscoe P. 1980. A model for C3 leaves describing the
dependence of net photosynthesis on irradiance. J Ex Bot 31:29-39
}
