#include <stdio.h>
#include <math.h>
#include "stats.h"
#include "nrutil.h"


void meanvar(float X[], unsigned int n, float *mean, float *var)
{
  // Sokal & Rohlf Box 4.2
  // fast, single-loop method, not requiring precalculation of mean

  unsigned long i;
  long double sumx    = 0.0; // sum of X - long, just in case
  long double sumxsq  = 0.0; // sum 0f X^2
  
  for (i = 0; i < n; i++)
    { 
      sumx   += X[i];
      sumxsq += X[i] * X[i];
    }

  *mean = sumx / (double) n;
  *var = ( sumxsq - ( (sumx * sumx) / (double) n ) ) / (double) (n-1);
}

void ttest(float X[], unsigned int nx, float Y[], unsigned int ny, \
           float *t, float *p)
{
  // Simple t-test, assuming equal variance
  // Sokal & Rohlf, Box 9.6
  // For unequal variance, need to code up Box 13.3

  double sumv;
  float mnx, mny, varx, vary;
  unsigned long int df;

  meanvar(X, nx, &mnx, &varx);
  meanvar(Y, ny, &mny, &vary);

  df = nx + ny - 2;

  sumv = ( ( (nx-1) * varx ) + ( ( ny-1) * vary) ) / df;

  *t = (mnx-mny) / sqrt( sumv * ( (float) (nx+ny) / (float) (nx*ny) ) );

  *p = ttable(*t, df, 2);

}

void correl(float X[], float Y[], unsigned int n, \
            float *r, float *p)
{
  // Product-moment correlation coefficient; S & R Box 15.1, and 15.3
  
  unsigned long int i;
  long double sumx, sumxsq, sumy, sumysq, sumxy;
  float t;
  
  sumx = sumxsq = sumy = sumysq = sumxy = 0.0;
  for (i = 0; i < n; i++)
    {
      sumx   += X[i];
      sumxsq += X[i] * X[i];
      sumy   += Y[i];
      sumysq += Y[i] * Y[i];
      sumxy  += X[i] * Y[i];
    }

  // reuse sumxsq, sumysq for Sum of Squares
  sumxsq = sumxsq - ( ( sumx * sumx ) / (float) n );
  sumysq = sumysq - ( ( sumy * sumy ) / (float) n );
  sumxy  = sumxy -  ( ( sumx * sumy ) / (float) n );

  *r = sumxy / sqrt( sumxsq * sumysq );
  
  t = *r * sqrt( (float) (n-2) / (1.0 - ((*r) * (*r))) );  

  // t-test on n-2 df, two tailed
  *p = ttable(t, (n-2), 2);
}


void contabg(int **X, unsigned int nx, \
             unsigned int ny, float *g, float *p)
{
  // Contingency table anal for independence of classification, using 
  // G stat - S&R Box 17.6
  
  unsigned int i, j;
  long double cellfreq = 0.0;
  long double rctot = 0.0;
  long double alltot = 0.0;
  long double rtot;
  long double ctot;

  // for each row
  for (i = 0; i < nx; i++)
    {
      rtot = 0.0;
      // cross each column
      for (j = 0; j < ny; j++)
        {
          if (X[i][j] != 0) cellfreq += (float) X[i][j] * log((float)X[i][j]);
          rtot += (float) X[i][j];
        }
      // could watch for empty rows (trick from NR)?
      if (rtot != 0.0) rctot += rtot * log(rtot);
      alltot += rtot;
    }

  // again for col totals; for each col
  for (j = 0; j < ny; j++)
    {
      ctot = 0.0;
      // descend each row
      for (i = 0; i < nx; i++) ctot += (float) X[i][j];
      if (ctot != 0.0) rctot += ctot * log(ctot);
    }

  alltot = alltot * log(alltot);

  *g = (float) ( 2.0 * (cellfreq - rctot + alltot) );
  *p = chitable(*g, 1);

}



// Table functions
// Obviously inferior to claculation from scratch.

float ttable(float score, int dfin, int tailed)
{
  int i, dfx;
  float pv[12] = {0.25, 0.20, 0.15, 0.10, 0.05, 0.025, 0.02, 0.01, 0.005, 0.0025, 0.001, 0.0005};

  int df[37] = {1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 40, 50, 60, 80, 100, 1000, 10000 }; 
  float val[37][12] = { {1.000, 1.376, 1.963, 3.078, 6.314, 12.71, 15.89, 31.82, 63.66, 127.3, 318.3, 636.6}, \
                        {0.816, 1.061, 1.386, 1.886, 2.920, 4.303, 4.849, 6.965, 9.925, 14.09, 22.33, 31.60}, \
                        {0.765, 0.978, 1.250, 1.638, 2.353, 3.182, 3.482, 4.541, 5.841, 7.453, 10.21, 12.92}, \
                        {0.741, 0.941, 1.190, 1.533, 2.132, 2.776, 2.999, 3.747, 4.604, 5.598, 7.173, 8.610}, \
                        {0.727, 0.920, 1.156, 1.476, 2.015, 2.571, 2.757, 3.365, 4.032, 4.773, 5.893, 6.869}, \
                        {0.718, 0.906, 1.134, 1.440, 1.943, 2.447, 2.612, 3.143, 3.707, 4.317, 5.208, 5.959}, \
                        {0.711, 0.896, 1.119, 1.415, 1.895, 2.365, 2.517, 2.998, 3.499, 4.029, 4.785, 5.408}, \
                        {0.706, 0.889, 1.108, 1.397, 1.860, 2.306, 2.449, 2.896, 3.355, 3.833, 4.501, 5.041}, \
                        {0.703, 0.883, 1.100, 1.383, 1.833, 2.262, 2.398, 2.821, 3.250, 3.690, 4.297, 4.781}, \
                        {0.700, 0.879, 1.093, 1.372, 1.812, 2.228, 2.359, 2.764, 3.169, 3.581, 4.144, 4.587}, \
                        {0.697, 0.876, 1.088, 1.363, 1.796, 2.201, 2.328, 2.718, 3.106, 3.497, 4.025, 4.437}, \
                        {0.695, 0.873, 1.083, 1.356, 1.782, 2.179, 2.303, 2.681, 3.055, 3.428, 3.930, 4.318}, \
                        {0.694, 0.870, 1.079, 1.350, 1.771, 2.160, 2.282, 2.650, 3.012, 3.372, 3.852, 4.221}, \
                        {0.692, 0.868, 1.076, 1.345, 1.761, 2.145, 2.264, 2.624, 2.977, 3.326, 3.787, 4.140}, \
                        {0.691, 0.866, 1.074, 1.341, 1.753, 2.131, 2.249, 2.602, 2.947, 3.286, 3.733, 4.073}, \
                        {0.690, 0.865, 1.071, 1.337, 1.746, 2.120, 2.235, 2.583, 2.921, 3.252, 3.686, 4.015}, \
                        {0.689, 0.863, 1.069, 1.333, 1.740, 2.110, 2.224, 2.567, 2.898, 3.222, 3.646, 3.965}, \
                        {0.688, 0.862, 1.067, 1.330, 1.734, 2.101, 2.214, 2.552, 2.878, 3.197, 3.611, 3.922}, \
                        {0.688, 0.861, 1.066, 1.328, 1.729, 2.093, 2.205, 2.539, 2.861, 3.174, 3.579, 3.883}, \
                        {0.687, 0.860, 1.064, 1.325, 1.725, 2.086, 2.197, 2.528, 2.845, 3.153, 3.552, 3.850}, \
                        {0.686, 0.859, 1.063, 1.323, 1.721, 2.080, 2.189, 2.518, 2.831, 3.135, 3.527, 3.819}, \
                        {0.686, 0.858, 1.061, 1.321, 1.717, 2.074, 2.183, 2.508, 2.819, 3.119, 3.505, 3.792}, \
                        {0.685, 0.858, 1.060, 1.319, 1.714, 2.069, 2.177, 2.500, 2.807, 3.104, 3.485, 3.768}, \
                        {0.685, 0.857, 1.059, 1.318, 1.711, 2.064, 2.172, 2.492, 2.797, 3.091, 3.467, 3.745}, \
                        {0.684, 0.856, 1.058, 1.316, 1.708, 2.060, 2.167, 2.485, 2.787, 3.078, 3.450, 3.725}, \
                        {0.684, 0.856, 1.058, 1.315, 1.706, 2.056, 2.162, 2.479, 2.779, 3.067, 3.435, 3.707}, \
                        {0.684, 0.855, 1.057, 1.314, 1.703, 2.052, 2.15, 2.473, 2.771, 3.057, 3.421, 3.690}, \
                        {0.683, 0.855, 1.056, 1.313, 1.701, 2.048, 2.154, 2.467, 2.763, 3.047, 3.408, 3.674}, \
                        {0.683, 0.854, 1.055, 1.311, 1.699, 2.045, 2.150, 2.462, 2.756, 3.038, 3.396, 3.659}, \
                        {0.683, 0.854, 1.055, 1.310, 1.697, 2.042, 2.147, 2.457, 2.750, 3.030, 3.385, 3.646}, \
                        {0.681, 0.851, 1.050, 1.303, 1.684, 2.021, 2.123, 2.423, 2.704, 2.971, 3.307, 3.551}, \
                        {0.679, 0.849, 1.047, 1.295, 1.676, 2.009, 2.109, 2.403, 2.678, 2.937, 3.261, 3.496}, \
                        {0.679, 0.848, 1.045, 1.296, 1.671, 2.000, 2.099, 2.390, 2.660, 2.915, 3.232, 3.460}, \
                        {0.678, 0.846, 1.043, 1.292, 1.664, 1.990, 2.088, 2.374, 2.639, 2.887, 3.195, 3.416}, \
                        {0.677, 0.845, 1.042, 1.290, 1.660, 1.984, 2.081, 2.364, 2.626, 2.871, 3.174, 3.390}, \
                        {0.675, 0.842, 1.037, 1.282, 1.646, 1.962, 2.056, 2.330, 2.581, 2.813, 3.098, 3.300}, \
                        {0.674, 0.841, 1.036, 1.282, 1.64, 1.960, 2.054, 2.326, 2.576, 2.807, 3.091, 3.291} };

  i = 0;
  while ((dfin >= df[i+1]) && (i < 36)) i++; 

  dfx = i;
  i = 0;
  while ((score >= val[dfx][i+1]) && (i < 11)) i++;

  return pv[i] * (float) tailed;
}


float chitable(float score, int dfin)
{
  int i, dfx;
  float pv[13] = {0.995, 0.990, 0.975, 0.950, 0.900, 0.750, 0.500, 0.250, 0.100, 0.050, 0.025, 0.010, 0.005};

  int df[30] = {1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30}; 

  float val[30][13] = { { 0.00004, 0.00016, 0.00098, 0.00393, 0.01579, 0.10153, 0.45494, 1.32330, 2.70554, 3.84146, 5.02389, 6.63490, 7.87944},
                        { 0.01003, 0.02010, 0.05064, 0.10259, 0.21072, 0.57536, 1.38629, 2.77259, 4.60517, 5.99146, 7.37776, 9.21034, 10.59663},
                        { 0.07172, 0.11483, 0.21580, 0.35185, 0.58437, 1.21253, 2.36597, 4.10834, 6.25139, 7.81473, 9.34840, 11.34487, 12.83816},
                        { 0.20699, 0.29711, 0.48442, 0.71072, 1.06362, 1.92256, 3.35669, 5.38527, 7.77944, 9.48773, 11.14329, 13.27670, 14.86026},
                        { 0.41174, 0.55430, 0.83121, 1.14548, 1.61031, 2.67460, 4.35146, 6.62568, 9.23636, 11.07050, 12.83250, 15.08627, 16.74960},
                        { 0.67573, 0.87209, 1.23734, 1.63538, 2.20413, 3.45460, 5.34812, 7.84080, 10.64464, 12.59159, 14.44938, 16.81189, 18.54758},
                        { 0.98926, 1.23904, 1.68987, 2.16735, 2.83311, 4.25485, 6.34581, 9.03715, 12.01704, 14.06714, 16.01276, 18.47531, 20.27774},
                        { 1.34441, 1.64650, 2.17973, 2.73264, 3.48954, 5.07064, 7.34412, 10.21885, 13.36157, 15.50731, 17.53455, 20.09024, 21.95495},
                        { 1.73493, 2.08790, 2.70039, 3.32511, 4.16816, 5.89883, 8.34283, 11.38875, 14.68366, 16.91898, 19.02277, 21.66599, 23.58935},
                        { 2.15586, 2.55821, 3.24697, 3.94030, 4.86518, 6.73720, 9.34182, 12.54886, 15.98718, 18.30704, 20.48318, 23.20925, 25.18818},
                        { 2.60322, 3.05348, 3.81575, 4.57481, 5.57778, 7.58414, 10.34100, 13.70069, 17.27501, 19.67514, 21.92005, 24.72497, 26.75685},
                        { 3.07382, 3.57057, 4.40379, 5.22603, 6.30380, 8.43842, 11.34032, 14.84540, 18.54935, 21.02607, 23.33666, 26.21697, 28.29952},
                        { 3.56503, 4.10692, 5.00875, 5.89186, 7.04150, 9.29907, 12.33976, 15.98391, 19.81193, 22.36203, 24.73560, 27.68825, 29.81947},
                        { 4.07467, 4.66043, 5.62873, 6.57063, 7.78953, 10.16531, 13.33927, 17.11693, 21.06414, 23.68479, 26.11895, 29.14124, 31.31935},
                        { 4.60092, 5.22935, 6.26214, 7.26094, 8.54676, 11.03654, 14.33886, 18.24509, 22.30713, 24.99579, 27.48839, 30.57791, 32.80132},
                        { 5.14221, 5.81221, 6.90766, 7.96165, 9.31224, 11.91222, 15.33850, 19.36886, 23.54183, 26.29623, 28.84535, 31.99993, 34.26719},
                        { 5.69722, 6.40776, 7.56419, 8.67176, 10.08519, 12.79193, 16.33818, 20.48868, 24.76904, 27.58711, 30.19101, 33.40866, 35.71847},
                        { 6.26480, 7.01491, 8.23075, 9.39046, 10.86494, 13.67529, 17.33790, 21.60489, 25.98942, 28.86930, 31.52638, 34.80531, 37.15645},
                        { 6.84397, 7.63273, 8.90652, 10.11701, 11.65091, 14.56200, 18.33765, 22.71781, 27.20357, 30.14353, 32.85233, 36.19087, 38.58226},
                        { 7.43384, 8.26040, 9.59078, 10.85081, 12.44261, 15.45177, 19.33743, 23.82769, 28.41198, 31.41043, 34.16961, 37.56623, 39.99685},
                        { 8.03365, 8.89720, 10.28290, 11.59131, 13.23960, 16.34438, 20.33723, 24.93478, 29.61509, 32.67057, 35.47888, 38.93217, 41.40106},
                        { 8.64272, 9.54249, 10.98232, 12.33801, 14.04149, 17.23962, 21.33704, 26.03927, 30.81328, 33.92444, 36.78071, 40.28936, 42.79565},
                        { 9.26042, 10.19572, 11.68855, 13.09051, 14.84796, 18.13730, 22.33688, 27.14134, 32.00690, 35.17246, 38.07563, 41.63840, 44.18128},
                        { 9.88623, 10.85636, 12.40115, 13.84843, 15.65868, 19.03725, 23.33673, 28.24115, 33.19624, 36.41503, 39.36408, 42.97982, 45.55851},
                        { 10.51965, 11.52398, 13.11972, 14.61141, 16.47341, 19.93934, 24.33659, 29.33885, 34.38159, 37.65248, 40.64647, 44.31410, 46.92789},
                        { 11.16024, 12.19815, 13.84390, 15.37916, 17.29188, 20.84343, 25.33646, 30.43457, 35.56317, 38.88514, 41.92317, 45.64168, 48.28988},
                        { 11.80759, 12.87850, 14.57338, 16.15140, 18.11390, 21.74940, 26.33634, 31.52841, 36.74122, 40.11327, 43.19451, 46.96294, 49.64492},
                        { 12.46134, 13.56471, 15.30786, 16.92788, 18.93924, 22.65716, 27.33623, 32.62049, 37.91592, 41.33714, 44.46079, 48.27824, 50.99338},
                        { 13.12115, 14.25645, 16.04707, 17.70837, 19.76774, 23.56659, 28.33613, 33.71091, 39.08747, 42.55697, 45.72229, 49.58788, 52.33562},
                        { 13.78672, 14.95346, 16.79077, 18.49266, 20.59923, 24.47761, 29.33603, 34.79974, 40.25602, 43.77297, 46.97924, 50.89218, 53.67196} };


  i = 0;
  while ((dfin >= df[i+1]) && (i < 29)) i++; 

  dfx = i;

  i = 0;
  while ((score >= val[dfx][i+1]) && (i < 12)) i++;

  return pv[i];
}

// test cases from S&R
// compile on its own with: gcc stats.c nrutil.c
/* int main() */
/*  { */
/*    float A1[7] = {7.2, 7.1, 9.1, 7.2, 7.3, 7.2, 7.5}; */
/*    float A2[7] = {8.8, 7.5, 7.7, 7.6, 7.4, 6.7, 7.2}; */
/*    float t, p, r, g; */
/*    float meanx, varx, meany, vary; */
/*    float Gill[12] = {159, 179,100, 45, 384, 230, 100, 320, 80, 220, 320, 210}; */
/*    float Body[12] = {14.4,15.2,11.3,2.5,22.7,14.9,1.41,15.81,4.19,15.39,17.25,9.52}; */
/*    int **trees; */
/*    trees = imatrix(0,1,0,1); */
/*    trees[0][0] = 12; trees[0][1] =22; trees[1][0] =16; trees[1][1] = 50; */

/*    // Mean and var and t-test */
/*    meanvar(A1, 6, &meanx, &varx); */
/*    meanvar(A2, 5, &meany, &vary); */
/*    ttest(A1, 7, A2, 7, &t, &p); */
/*    printf("t = %f, p = %f mnx=%f varx=%f mny=%f vary=%f \n", t, p, meanx, varx, meany, vary); */
 
/*    // correlation */
/*    correl(Gill, Body, 12, &r, &p); */
/*    printf("r = %f, p = %f\n", r, p); */

/*    // conting tab */
/*    contabg(trees, 2, 2, &g, &p); */
/*    printf("g = %f, p = %f\n", g, p); */

/*    return 1; */
/* } */
