% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sample_belief_space.R
\name{sample_belief_space}
\alias{sample_belief_space}
\title{Sample from the Belief Space}
\usage{
sample_belief_space(model, projection = NULL, n = 1000, method = "random", ...)
}
\arguments{
\item{model}{a unsolved or solved \link{POMDP}.}

\item{projection}{Sample in a projected belief space. See \code{\link[=projection]{projection()}} for details.}

\item{n}{size of the sample. For trajectories, it is the number of trajectories.}

\item{method}{character string specifying the sampling strategy. Available
are \code{"random"}, \code{"regular"}, and \code{"trajectories"}.}

\item{...}{for the trajectory method, further arguments are passed on to \code{\link[=simulate_POMDP]{simulate_POMDP()}}. Further arguments are ignored for the other methods.}
}
\value{
Returns a matrix. Each row is a sample from the belief space.
}
\description{
Sample points from belief space using a several sampling strategies.
}
\details{
The purpose of sampling from the belief space is to provide good coverage or to sample belief points
that are more likely to be encountered (see trajectory method).
The following sampling methods are available:
\itemize{
\item \code{'random'} samples uniformly sample from the projected belief space using
the method described by Luc Devroye (1986). Sampling is be done in parallel
after a foreach backend is registered.
\item \code{'regular'} samples points using a
regularly spaced grid. This method is only available for projections on 2 or
3 states.
\item \code{"trajectories"} returns the belief states encountered in \code{n} trajectories of length \code{horizon} starting at the
model's initial belief. Thus it returns \code{n} x \code{horizon} belief states and will contain duplicates.
Projection is not supported for trajectories. Additional
arguments can include the simulation \code{horizon} and the start \code{belief} which are passed on to \code{\link[=simulate_POMDP]{simulate_POMDP()}}.
}
}
\examples{
data("Tiger")

# random sampling can be done in parallel after registering a backend.
# doparallel::registerDoParallel()

sample_belief_space(Tiger, n = 5)
sample_belief_space(Tiger, n = 5, method = "regular")
sample_belief_space(Tiger, n = 1, horizon = 5, method = "trajectories")

# sample, determine the optimal action and calculate the expected reward for a solved POMDP
# Note: check.names = FALSE is used to preserve the `-` for the state names in the dataframe.
sol <- solve_POMDP(Tiger)
samp <- sample_belief_space(sol, n = 5, method = "regular")
data.frame(samp, action = optimal_action(sol,  belief = samp), 
  reward = reward(sol, belief = samp), check.names = FALSE)
  
# sample from a 3 state problem
data(Three_doors)
Three_doors

sample_belief_space(Three_doors, n = 5)
sample_belief_space(Three_doors, n = 5, projection = c(`tiger-left` = .1))

if ("Ternary" \%in\% installed.packages()) {
sample_belief_space(Three_doors, n = 9, method = "regular")
sample_belief_space(Three_doors, n = 9, method = "regular", projection = c(`tiger-left` = .1))
}

sample_belief_space(Three_doors, n = 1, horizon = 5, method = "trajectories")
}
\references{
Luc Devroye, Non-Uniform Random Variate Generation, Springer
Verlag, 1986.
}
\seealso{
Other POMDP: 
\code{\link{MDP2POMDP}},
\code{\link{POMDP}()},
\code{\link{accessors}},
\code{\link{actions}()},
\code{\link{add_policy}()},
\code{\link{plot_belief_space}()},
\code{\link{projection}()},
\code{\link{reachable_and_absorbing}},
\code{\link{regret}()},
\code{\link{simulate_POMDP}()},
\code{\link{solve_POMDP}()},
\code{\link{solve_SARSOP}()},
\code{\link{transition_graph}()},
\code{\link{update_belief}()},
\code{\link{value_function}()},
\code{\link{write_POMDP}()}
}
\author{
Michael Hahsler
}
\concept{POMDP}
