% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/scan1.R
\name{scan1}
\alias{scan1}
\title{Genome scan with a single-QTL model}
\usage{
scan1(
  genoprobs,
  pheno,
  kinship = NULL,
  addcovar = NULL,
  Xcovar = NULL,
  intcovar = NULL,
  weights = NULL,
  reml = TRUE,
  model = c("normal", "binary"),
  hsq = NULL,
  cores = 1,
  ...
)
}
\arguments{
\item{genoprobs}{Genotype probabilities as calculated by
\code{\link[=calc_genoprob]{calc_genoprob()}}.}

\item{pheno}{A numeric matrix of phenotypes, individuals x phenotypes.}

\item{kinship}{Optional kinship matrix, or a list of kinship matrices (one
per chromosome), in order to use the LOCO (leave one chromosome
out) method.}

\item{addcovar}{An optional numeric matrix of additive covariates.}

\item{Xcovar}{An optional numeric matrix with additional additive covariates used for
null hypothesis when scanning the X chromosome.}

\item{intcovar}{An numeric optional matrix of interactive covariates.}

\item{weights}{An optional numeric vector of positive weights for the
individuals. As with the other inputs, it must have \code{names}
for individual identifiers.}

\item{reml}{If \code{kinship} provided: if \code{reml=TRUE}, use
REML; otherwise maximum likelihood.}

\item{model}{Indicates whether to use a normal model (least
squares) or binary model (logistic regression) for the phenotype.
If \code{model="binary"}, the phenotypes must have values in \eqn{[0, 1]}.}

\item{hsq}{Considered only if \code{kinship} is provided, in which case
this is taken as the assumed value for the residual
heritability. It should be a vector with length corresponding
to the number of columns in \code{pheno}, or (if \code{kinship}
corresponds to a list of LOCO kinship matrices) a matrix with dimension
\verb{length(kinship) x ncol(pheno)}.}

\item{cores}{Number of CPU cores to use, for parallel calculations.
(If \code{0}, use \code{\link[parallel:detectCores]{parallel::detectCores()}}.)
Alternatively, this can be links to a set of cluster sockets, as
produced by \code{\link[parallel:makeCluster]{parallel::makeCluster()}}.}

\item{...}{Additional control parameters; see Details.}
}
\value{
An object of class \code{"scan1"}: a matrix of LOD scores, positions x phenotypes.
Also contains one or more of the following attributes:
\itemize{
\item \code{sample_size} - Vector of sample sizes used for each
phenotype
\item \code{hsq} - Included if \code{kinship} provided: A matrix of
estimated heritabilities under the null hypothesis of no
QTL. Columns are the phenotypes. If the \code{"loco"} method was
used with \code{\link[=calc_kinship]{calc_kinship()}} to calculate a list
of kinship matrices, one per chromosome, the rows of \code{hsq}
will be the heritabilities for the different chromosomes (well,
leaving out each one). If \code{Xcovar} was not NULL, there will at
least be an autosome and X chromosome row.
}
}
\description{
Genome scan with a single-QTL model by Haley-Knott regression or a
linear mixed model, with possible allowance for covariates.
}
\details{
We first fit the model \eqn{y = X \beta + \epsilon}{y = Xb + e}
where \eqn{X} is a matrix of covariates (or just an intercept) and
\eqn{\epsilon}{e} is multivariate normal with mean 0 and covariance
matrix \eqn{\sigma^2 [h^2 (2 K) + I]}{sigmasq*[hsq*2*K+I]} where
\eqn{K} is the kinship matrix and \eqn{I} is the identity matrix.

We then take \eqn{h^2}{hsq} as fixed and then scan the genome, at
each genomic position fitting the model \eqn{y = P \alpha + X \beta
+ \epsilon}{y = Xb + e} where \eqn{P} is a matrix of genotype
probabilities for the current position and again \eqn{X} is a
matrix of covariates \eqn{\epsilon}{e} is multivariate normal with
mean 0 and covariance matrix \eqn{\sigma^2 [h^2 (2 K) +
I]}{sigmasq*[hsq*2*K+I]}, taking \eqn{h^2}{hsq} to be known.

For each of the inputs, the row names are used as
individual identifiers, to align individuals. The \code{genoprobs}
object should have a component \code{"is_x_chr"} that indicates
which of the chromosomes is the X chromosome, if any.

The \code{...} argument can contain several additional control
parameters; suspended for simplicity (or confusion, depending on
your point of view). \code{tol} is used as a tolerance value for linear
regression by QR decomposition (in determining whether columns are
linearly dependent on others and should be omitted); default
\code{1e-12}. \code{intcovar_method} indicates whether to use a high-memory
(but potentially faster) method or a low-memory (and possibly
slower) method, with values \code{"highmem"} or \code{"lowmem"}; default
\code{"lowmem"}. \code{max_batch} indicates the maximum number of phenotypes
to run together; default is unlimited. \code{maxit} is the maximum
number of iterations for converence of the iterative algorithm
used when \code{model=binary}. \code{bintol} is used as a tolerance for
converence for the iterative algorithm used when \code{model=binary}.
\code{eta_max} is the maximum value for the "linear predictor" in the
case \code{model="binary"} (a bit of a technicality to avoid fitted
values exactly at 0 or 1).

If \code{kinship} is absent, Haley-Knott regression is performed.
If \code{kinship} is provided, a linear mixed model is used, with a
polygenic effect estimated under the null hypothesis of no (major)
QTL, and then taken as fixed as known in the genome scan.

If \code{kinship} is a single matrix, then the \code{hsq}
in the results is a vector of heritabilities (one value for each phenotype). If
\code{kinship} is a list (one matrix per chromosome), then
\code{hsq} is a matrix, chromosomes x phenotypes.
}
\examples{
# read data
iron <- read_cross2(system.file("extdata", "iron.zip", package="qtl2"))
\dontshow{iron <- iron[,c("19", "X")] # subset to chr 19 and X}

# insert pseudomarkers into map
map <- insert_pseudomarkers(iron$gmap, step=1)

# calculate genotype probabilities
probs <- calc_genoprob(iron, map, error_prob=0.002)

# grab phenotypes and covariates; ensure that covariates have names attribute
pheno <- iron$pheno
covar <- match(iron$covar$sex, c("f", "m")) # make numeric
names(covar) <- rownames(iron$covar)
Xcovar <- get_x_covar(iron)

# perform genome scan
out <- scan1(probs, pheno, addcovar=covar, Xcovar=Xcovar)

# leave-one-chromosome-out kinship matrices
kinship <- calc_kinship(probs, "loco")

# genome scan with a linear mixed model
out_lmm <- scan1(probs, pheno, kinship, covar, Xcovar)

}
\references{
Haley CS, Knott SA (1992) A simple
regression method for mapping quantitative trait loci in line
crosses using flanking markers.  Heredity 69:315--324.

Kang HM, Zaitlen NA, Wade CM, Kirby A, Heckerman D, Daly MJ, Eskin
E (2008) Efficient control of population structure in model
organism association mapping. Genetics 178:1709--1723.
}
\seealso{
\code{\link[=scan1perm]{scan1perm()}}, \code{\link[=scan1coef]{scan1coef()}}, \code{\link[=cbind.scan1]{cbind.scan1()}}, \code{\link[=rbind.scan1]{rbind.scan1()}}, \code{\link[=scan1max]{scan1max()}}
}
