\name{createS}
\alias{createS}
\title{
  Simulate sample covariances or datasets
}
\description{
Simulate data from a p-dimensional (zero-mean) gaussian graphical model (GGM) with a specified (or random) topology and return the sample covariance matrix or matrices.
Can also return the original simulated data or underlying precision matrix.
}
\usage{
createS(n, p,
        topology = "identity",  # See details for other choices
        dataset = FALSE, precision = FALSE,
        nonzero = 0.25, m = 1L, banded.n = 2L,
        invwishart = FALSE, nu = p + 1, Plist)
}
\arguments{
  \item{n}{A \code{numeric} vector giving number of samples. If the length is larger than 1, the covariance matrices are returned as a list.}
  \item{p}{A \code{numeric} of length 1 giving the dimension of the samples/covariance.}
  \item{topology}{character. The topology to use for the simulations. See the details.}
  \item{dataset}{A \code{logical} value specifying whether the sample covariance or the simulated data itself should be returned.}
  \item{precision}{A \code{logical} value. If \code{TRUE} the constructed precision matrix
  is returned.}
  \item{nonzero}{A \code{numeric} of length 1 giving the value of the nonzero entries used in some topologies.}
  \item{m}{A \code{integer} giving the number of blocks (i.e. conditionally independent components) to create. If \code{m} is greater than 1, then the given \code{topology} is used on \code{m} blocks of approximately equal size.}
  \item{banded.n}{A \code{integer} of length one giving the number of bands. Only used if \code{topology} is one of \code{"banded"}, \code{"small-world"}, or \code{"Watts-Strogatz"}.}
  \item{invwishart}{\code{logical}. If \code{TRUE} the constructed precision matrix is used as the scale matrix of an inverse Wishart distribution and class covariance matrices are drawn from this distribution.}
  \item{nu}{\code{numeric} greater than \code{p + 1} giving the degrees of freedom in the inverse Wishart distribution.
  A large \code{nu} implies high class homogeneity.
  A small \code{nu} near \code{p + 1} implies high class heterogeneity.}
  \item{Plist}{An optional \code{list} of \code{numeric} matrices giving the
  precision matrices to simulate from. Useful when random matrices have already
  been generated by setting \code{precision = TRUE}.}
}

\details{
The data is simulated from a zero-mean \code{p}-dimensional multivariate gaussian distribution with some precision matrix determined by the argument \code{topology} which defines the GGM.
If \code{precision} is \code{TRUE} the population precision matrix is returned.
This is useful to see what the actual would-be-used precision matrices are.
The available values of \code{topology} are described below.
Unless otherwise stated the diagonal entries are always one.
If \code{m} is 2 or greater block diagonal precision matrices are constructed and used.
\itemize{
  \item \code{"identity"}: uses the identity matrix (\code{diag(p)}) as precision matrix.
  Corresponds to no conditional dependencies.
  \item \code{"star"}: simulate from a star topology. Within each block the first
  node is selected as the "hub". The off-diagonal entries \eqn{(1,j)} and \eqn{(j,1)}
  values taper off with the value \eqn{1/(j + 1)}.
  \item \code{"clique"}: simulate from clique topology where each block is a complete
  graph with off-diagonal elements equal to \code{nonzero}.
  \item \code{"complete"}: alias for (and identical to) \code{"clique"}.
  \item \code{"chain"}: simulate from a chain topology where the precision matrix
  is a tridiagonal matrix with off-diagonal elements (in each block) given
  by argument \code{nonzero}.
  \item \code{"banded"}: precision elements \code{(i,j)} are given by
  \eqn{1/(|i-j|+1)} if \eqn{|i-j|} is less than or equal to \code{banded.n}
  and zero otherwise.
  \item  \code{"scale-free"}: The non-zero pattern of each block is generated by a
  Barabassi random graph. Non-zero off-diagonal values are given by \code{nonzero}.
  Gives are very "hubby" network.
  \item \code{"Barabassi"}: alias for \code{"scale-free"}.
  \item \code{"small-world"}: The non-zero pattern of each block is generated by a
  1-dimensional Watts-Strogatz random graph with \code{banded.n}
  starting neighbors and \eqn{5\%} probability of rewiring.
  Non-zero off-diagonal values are given by
  \code{nonzero}. Gives are very "bandy" network.
  \item \code{"Watts-Strogatz"}: alias for \code{"small-world"}
  \item \code{"random-graph"}: The non-zero pattern of each block is generated by a
  Erdos-Renyi random graph where each edge is present with probability \eqn{1/p}.
  Non-zero off-diagonal values are given by \code{nonzero}.
  \item \code{"Erdos-Renyi"}: alias for \code{"random-graph"}
}
When \code{n} has length greater than 1, the datasets are generated i.i.d. given the topology and number of blocks.

Arguments \code{invwishart} and \code{nu} allows for introducing class homogeneity.
Large values of \code{nu} imply high class homogeneity.
\code{nu} must be greater than \code{p + 1}.
More precisely, if \code{invwishart == TRUE} then the constructed precision matrix is used as the scale parameter in an inverse Wishart distribution with \code{nu} degrees of freedom.
Each class covariance is distributed according to this inverse Wishart and independent.
}
\value{
The returned type is dependent on \code{n} and \code{covariance}.
The function generally returns a \code{list} of \code{numeric} matrices with the same length as \code{n}.
If \code{covariance} is \code{FALSE} the simulated datasets with size \code{n[i]} by \code{p} are given in the \code{i} entry of the output.
If \code{covariance} is \code{TRUE} the \code{p} by \code{p} sample covariances of the datasets are given.
When \code{n} has length 1 the \code{list} structure is dropped and the matrix is returned.
}
\author{
  Anders E. Bilgrau,
  Carel F.W. Peeters <cf.peeters@vumc.nl>,
  Wessel N. van Wieringen
}
\examples{
## Generate some simple sample covariance matrices
createS(n = 10, p = 3)
createS(n = c(3, 4, 5), p = 3)
createS(n = c(32, 55), p = 7)

## Generate some datasets and not sample covariance matrices
createS(c(3, 4), p = 6, dataset = TRUE)

## Generate sample covariance matrices from other topologies:
A <- createS(2000, p = 4, topology = "star")
round(solve(A), 3)
B <- createS(2000, p = 4, topology = "banded", banded.n = 2)
round(solve(B), 3)
C <- createS(2000, p = 4, topology = "clique")  # The complete graph (as m = 1)
round(solve(C), 3)
D <- createS(2000, p = 4, topology = "chain")
round(solve(D), 3)

## Generate smaple covariance matrices from block topologies:
C3 <- createS(2000, p = 10, topology = "clique", m = 3)
round(solve(C3), 1)
C5 <- createS(2000, p = 10, topology = "clique", m = 5)
round(solve(C5), 1)

## Can also return the precision matrix to see what happens
## m = 2 blocks, each "banded" with 4 off-diagonal bands
round(createS(1, 12, "banded", m = 2, banded.n = 4, precision = TRUE), 2)

## Simulation using graph-games
round(createS(1, 10, "small-world", precision = TRUE), 2)
round(createS(1, 5, "scale-free", precision = TRUE), 2)
round(createS(1, 5, "random-graph", precision = TRUE), 2)

## Simulation using inverse Wishart distributed class covariance
## Low class homogeneity
createS(n = c(10,10), p = 5, "banded", invwishart = TRUE, nu = 10)
## Extremely high class homogeneity
createS(n = c(10,10), p = 5, "banded", invwishart = TRUE, nu = 1e10)

# The precision argument can again be used to see the actual realised class
# precision matrices used when invwishart = TRUE.

# The Plist argument is used to reuse old precision matrices or
# user-generated ones
P <- createS(n = 1, p = 5, "banded", precision = TRUE)
lapply(createS(n = c(1e5, 1e5), p = 5, Plist = list(P, P+1)), solve)
}

