################################################################################
#' @title rAtte: Estimation and Inference in Random Attention Models
#'
#' @description Given a random sample with menu information and choices, \code{rAtte}
#'   returns test statistics and critical values against a specific preference.
#'   Five methods for choosing critical values are available: (i) \code{PI}: critical values
#'   based on plug-in (estimated) moment inequalities (this is not uniformly valid);
#'   (ii) \code{LF}: least favorable critical values; (iii) \code{MS}: moment selection;
#'   (iv) \code{UB}: refined moment selection (plug-in upper bound of moment inequalities);
#'   and (v) \code{S}: plug-in (estimated) moment inequalities with shrinkage (this is the
#'   recommended procedure).
#'
#' \code{\link{sumData}} is a low-level function that generates summary statistics, and
#'   \code{\link{genMat}} can be used to construct the constraint matrices. The simulated dataset
#'   \code{\link{ramdata}} is also provided for illustration.
#'
#' @param menu Numeric matrix of 0s and 1s, the collection of menus.
#' @param choice Numeric matrix of 0s and 1s, the collection of choices.
#' @param pref_list Numeric matrix, each row corresponds to a preference. For example \code{c(2, 3, 1)} means
#'   2 is preferred to 3 and to 1. When set to \code{NULL}, the default \code{c(1, 2, 3, ...)}
#'   will be used.
#' @param nCritSimu Integer, number of simulations for critical value. Default is \code{2000}.
#' @param BARatioSELECT Numeric, beta-to-alpha ratio for moment selection method. Default is \code{0.1}.
#' @param BARatioREFINE Numeric, beta-to-alpha ratio for refined moment selection method. Default is \code{0.1}.
#' @param MNRatioSHRINK Numeric, shrinkage parameter. Default is \code{sqrt(1/log(N))}, where N is the sample size.
#' @param limDataCorr Boolean, whether assumes limited data (default is \code{TRUE}). When set to
#'   \code{FALSE}, will assume all menus/choice problems are observed, hence no correction.
#'
#' @return
#' \item{sumStats}{Summary statistics, generated by \code{\link{sumData}}.}
#' \item{constraints}{Matrices of constraints, generated by \code{\link{genMat}}.}
#' \item{Tstat}{Test statistic.}
#' \item{critVal}{Critical values.}
#'
#' @references
#' M. D. Cattaneo, X. Ma, Y. Masatlioglu and E. Suleymanov (2017). \href{http://www-personal.umich.edu/~cattaneo/papers/Cattaneo-Ma-Masatlioglu-Suleymanov_2017_RAM.pdf}{A Random Attention Model}. Working Paper, University of Michigan.
#'
#' @author
#' Matias D. Cattaneo, University of Michigan. \email{cattaneo@umich.edu}.
#'
#' Xinwei Ma (maintainer), University of Michigan. \email{xinweima@umich.edu}
#'
#' Yusufcan Masatlioglu, University of Maryland. \email{yusufcan@umd.edu}
#'
#' Elchin Suleymanov, University of Michigan. \email{elchin@umich.edu}
#'
#' @examples
#' # load data
#' data(ramdata)
#'
#' # seed, to replicate simulated critical values
#' set.seed(42)
#'
#' result <- rAtte(menu = menu, choice = choice,
#'   pref_list=matrix(c(1, 2, 3, 4, 5, 6,
#'                      2, 3, 4, 5, 6, 1,
#'                      3, 4, 5, 6, 2, 1,
#'                      4, 5, 6, 3, 2, 1,
#'                      6, 5, 4, 3, 2, 1), ncol=6, byrow=TRUE))
#' summary(result)
#'
#' @export
rAtte <- function(menu, choice, pref_list=NULL, nCritSimu=2000,
                  BARatioSELECT=0.1, BARatioREFINE=0.1, MNRatioSHRINK=NULL,
                  limDataCorr=TRUE) {

  ################################################################################
  # Error Check
  ################################################################################

  # menu
  if (!is.matrix(menu)) {
    stop("Input 'menu' has to be a matrix.\n")
  } else if (min(dim(menu)) == 0) {
    stop("Input 'menu' has at least one dimension 0.\n")
  } else if (!all((menu == 0) | (menu == 1))) {
    stop("Input 'menu' can only contain 0 (FALSE) and 1 (TRUE).\n")
  } else {
    # nothing
  }

  # choice
  if (!is.matrix(choice)) {
    stop("Input 'choice' has to be a matrix.\n")
  } else if (min(dim(choice)) == 0) {
    stop("Input 'choice' has at least one dimension 0.\n")
  } else if (!all((choice == 0) | (choice == 1))) {
    stop("Input 'choice' can only contain 0 (FALSE) and 1 (TRUE).\n")
  } else if (!all(rowSums(choice) == 1)) {
    stop("Input 'choice' should contain one and only one 1 (TRUE) in each row.\n")
  } else {
    # nothing
  }

  # menu and choice
  if (!all(dim(menu) == dim(choice))) {
    stop("Input 'menu' and 'choice' have to have the same dimensions.\n")
  }
  else if (!all(as.integer(rowSums((menu == 1) & (choice == 1))))) {
    stop("Input 'menu' and 'choice' should contain one and only one common 1 (TRUE) in each row.\n")
  } else {
    # nothing
  }

  # preference
  if (length(as.vector(pref_list)) == 0) {
    pref_list <- matrix(1:ncol(menu), nrow=1)
  } else if (!is.matrix(pref_list)) {
    stop("Input 'pref_list' has to be a matrix.\n")
  } else if (min(dim(pref_list)) == 0) {
    stop("Input 'pref_list' has at least one dimension 0.\n")
  } else if (ncol(pref_list) != ncol(menu)) {
    stop("Input 'pref_list' has to have the same number of columns as 'menu'.\n")
  } else if (!all(apply(pref_list, MARGIN=1, FUN=function(x) all(sort(x)==1:ncol(menu))))) {
    stop("Input 'pref_list' incorrectly specified.\n")
  } else {
    # nothing
  }

  if (length(MNRatioSHRINK) == 0) {
    MNRatioSHRINK <- 1 / log(nrow(menu))
  } else {
    # nothing
  }

  level <- c(0.9, 0.95, 0.99)

  ################################################################################
  # Initialization
  ################################################################################
  # maximum statistic
  Tstat <- rep(NA, nrow(pref_list))

  # PI: plugin critical value
  CritValPLUG <- matrix(NA, nrow=nrow(pref_list), ncol=length(level))
  colnames(CritValPLUG) <- 1 - level
  rownames(CritValPLUG) <- 1:nrow(pref_list)

  # LF: critical value, least favorable
  CritValLF <- CritValPLUG

  # MS: critical value, moment selection
  CritValSELECT <- CritValPLUG

  # UB: critical value, refined moment selection
  CritValREFINE <- CritValPLUG

  # S: critical value, shrinkage
  CritValSHRINK <- CritValPLUG

  # first obtain point estimates

  # summary statistics
  sumStats <- sumData(menu, choice)
  n <- sum(sumStats$sumN) # sample size

  # generate matrices
  constraints <- genMat(sumStats$sumMenu, sumStats$sumMsize, pref_list, limDataCorr)

  # simulate normal distributions, scale by sample size since Sigma is asymptotically stable.
  normSimu <- t(mvrnorm(n=nCritSimu, mu=rep(0, nrow(sumStats$Sigma)), Sigma=sumStats$Sigma) / sqrt(n))
  normSimu[diag(sumStats$Sigma) == 0, ] <- 0;

  # find test statistics and critical values
  temp <- matrix(0, nrow=5, ncol=nCritSimu)
  j <- 0
  for (i in 1:nrow(pref_list)) {
    if (constraints$ConstN[i] == 0) {# if there are no constraints
      Tstat[i]           <- 0
      CritValPLUG[i, ]   <- 0
      CritValLF[i, ]     <- 0
      CritValSELECT[i, ] <- 0
      CritValREFINE[i, ] <- 0
      CritValSHRINK[i, ] <- 0
    } else {# if there are constraints
      R_temp <- constraints$R[(j+1):(j+constraints$ConstN[i]), ]
      RSigma <- R_temp %*% sumStats$Sigma %*% t(R_temp)

      tempTstat <- (R_temp %*% sumStats$sumProbVec) / sqrt(diag(RSigma))
      tempTstat[is.na(tempTstat)] <- 0
      tempTstat <- sqrt(n) * tempTstat
      Tstat[i] = max(max(tempTstat), 0) # maximum statistic

      # plug-in center
      centerPlug <- R_temp %*% sumStats$sumProbVec
      centerPlug[centerPlug > 0] <- 0

      # plug-in centering with shrinkage
      centerShrink <- R_temp %*% sumStats$sumProbVec * sqrt(MNRatioSHRINK)
      centerShrink[centerShrink > 0] <- 0

      # to store simulated moment inequalities, for moment selection
      tempTstatSimu <- matrix(0, nrow=constraints$ConstN[i], ncol=nCritSimu)
      for (k in 1:nCritSimu) {
        # least favorable
        temptemp <- (R_temp %*% normSimu[, k]) / sqrt(diag(RSigma))
        temptemp[is.na(temptemp)] <- 0
        temp[1, k] <- sqrt(n) * max(max(temptemp), 0)
        tempTstatSimu[, k] = sqrt(n) * temptemp

        # shrinkage recentering
        temptemp <- (R_temp %*% normSimu[, k] + centerShrink) / sqrt(diag(RSigma))
        temptemp[is.na(temptemp)] <- 0
        temp[3, k] <- sqrt(n) * max(max(temptemp), 0)

        # plug-in center
        temptemp <- (R_temp %*% normSimu[, k] + centerPlug) / sqrt(diag(RSigma))
        temptemp[is.na(temptemp)] <- 0
        temp[4, k] = sqrt(n) * max(max(temptemp), 0)
      }

      # plug-in center critical value
      CritValPLUG[i, ]   <- quantile(temp[4, ], level)

      # least favorable critical value
      CritValLF[i, ]     <- quantile(temp[1, ], level)

      # shrinkage recentering
      CritValSHRINK[i, ] <- quantile(temp[3, ], level)

      for (i_level in 1:length(level)) {# enumerate all levels
        # refined moment selection / moment inequality upper-bounding
        centerUpper <- R_temp %*% sumStats$sumProbVec + sqrt(diag(RSigma)) / sqrt(n) *
          quantile(temp[1, ], 1 - ((1-level[i_level]) * BARatioREFINE))
        centerUpper[centerUpper > 0] <- 0

        for (k in 1:nCritSimu) {
          temptemp <- (R_temp %*% normSimu[, k] + centerUpper) / sqrt(diag(RSigma))
          temptemp[is.na(temptemp)] <- 0
          temp[5, k] <- sqrt(n) * max(max(temptemp), 0)
        }

        CritValREFINE[i, i_level] <- quantile(temp[5, ],
           level[i_level] + (1-level[i_level]) * BARatioREFINE)

        # critical value with moment selection
        tempCritBeta <- quantile(temp[1, ],
                                 1 - ((1-level[i_level]) * BARatioSELECT))
        eff_moment <- (tempTstat > -2 * tempCritBeta)

        if (sum(eff_moment) == 0) {# no effective moment
          CritValSELECT[i, i_level] <- 0
        } else{
          CritValSELECT[i, i_level] <-
            quantile(
              apply(tempTstatSimu[eff_moment, ], MARGIN=2, FUN=function(x) max(max(x), 0)),
              level[i_level] + (1 - level[i_level]) * BARatioSELECT * 2)
        }
      }
    }

    j = j + constraints$ConstN[i];
  }

  Result <- list(sumStats=sumStats, constraints=constraints, Tstat=Tstat,
                 critVal=list(S=CritValSHRINK,
                              PI=CritValPLUG,
                              LF=CritValLF,
                              MS=CritValSELECT,
                              UB=CritValREFINE),
                 pref=pref_list)

  class(Result) <- "CMMRandomAttention"
  return(Result)
}

################################################################################
#' Internal function.
#'
#' @param object Class \code{CMMRandomAttention} objects.
#'
#' @keywords internal
#' @export
summary.CMMRandomAttention <- function(object, ...) {
  x <- object
  cat("\n Inference in Random Attention Models.\n")
  cat("\n")

  cat(paste(format("Number of obs =", width=25), toString(sum(x$sumStats$sumN)), sep="")); cat("\n")
  cat(paste(format("Number of elements =", width=25), toString(ncol(x$sumStats$sumMenu)), sep="")); cat("\n")
  cat(paste(format("Number of menus =", width=25), toString(nrow(x$sumStats$sumMenu)), sep="")); cat("\n")
  cat("\n")

  cat(paste(format("Min elements in menu =", width=25), toString(min(x$sumStats$sumMsize)), sep="")); cat("\n")
  cat(paste(format("Max elements in menu =", width=25), toString(max(x$sumStats$sumMsize)), sep="")); cat("\n")
  cat("\n")

  cat(paste(format("Min eff. observations =", width=25), toString(min(x$sumStats$sumN)), sep="")); cat("\n")
  cat(paste(format("Max eff. observations =", width=25), toString(max(x$sumStats$sumN)), sep="")); cat("\n")
  cat("\n")

  for (i in 1:nrow(x$pref)) {
    cat(paste(rep("=", 60), collapse="")); cat("\n")
    cat(paste("Preference[", toString(i), "]",": ", sep=""))
    cat(x$pref[i, ]); cat("\n")

    if (sum(x$Tstat[i] > x$critVal$S[i, ]) == 3) {
      stars <- " ***"
    } else if (sum(x$Tstat[i] > x$critVal$S[i, ]) == 2) {
      stars <- " **"
    } else if (sum(x$Tstat[i] > x$critVal$S[i, ]) == 1) {
      stars <- " *"
    } else {
      stars <- " "
    }
    cat(paste("Statistic: ", toString(round(x$Tstat[i], 4)), stars, sep="")); cat("\n")

    cat(paste(rep("-", 60), collapse="")); cat("\n")
    cat(paste(format("Crit Val", width=15), format("0.10", width=15), format("0.05", width=15), format("0.01", width=15), sep="")); cat("\n")
    cat(paste(rep("-", 60), collapse="")); cat("\n")
    cat(paste(format("S"       , width=15),
              format(toString(round(x$critVal$S[i, 1], 4)), width=15),
              format(toString(round(x$critVal$S[i, 2], 4)), width=15),
              format(toString(round(x$critVal$S[i, 3], 4)), width=15), sep="")); cat("\n")
    cat(paste(format("PI"       , width=15),
              format(toString(round(x$critVal$PI[i, 1], 4)), width=15),
              format(toString(round(x$critVal$PI[i, 2], 4)), width=15),
              format(toString(round(x$critVal$PI[i, 3], 4)), width=15), sep="")); cat("\n")
    cat(paste(format("LF"       , width=15),
              format(toString(round(x$critVal$LF[i, 1], 4)), width=15),
              format(toString(round(x$critVal$LF[i, 2], 4)), width=15),
              format(toString(round(x$critVal$LF[i, 3], 4)), width=15), sep="")); cat("\n")
    cat(paste(format("MS"       , width=15),
              format(toString(round(x$critVal$MS[i, 1], 4)), width=15),
              format(toString(round(x$critVal$MS[i, 2], 4)), width=15),
              format(toString(round(x$critVal$MS[i, 3], 4)), width=15), sep="")); cat("\n")
    cat(paste(format("UB"       , width=15),
              format(toString(round(x$critVal$UB[i, 1], 4)), width=15),
              format(toString(round(x$critVal$UB[i, 2], 4)), width=15),
              format(toString(round(x$critVal$UB[i, 3], 4)), width=15), sep="")); cat("\n")


  }
  cat(paste(rep("=", 60), collapse="")); cat("\n")
  cat("\n")
}

################################################################################
#' Internal function.
#'
#' @param object Class \code{CMMRandomAttention} objects.
#'
#' @keywords internal
#' @export
print.CMMRandomAttention <- function(x, ...) {
  cat("\n")
  cat("Inference in Random Attention Models.\n")
  cat("\n")

  cat(paste(format("Number of obs =", width=25), toString(sum(x$sumStats$sumN)), sep="")); cat("\n")
  cat(paste(format("Number of elements =", width=25), toString(ncol(x$sumStats$sumMenu)), sep="")); cat("\n")
  cat(paste(format("Number of menus =", width=25), toString(nrow(x$sumStats$sumMenu)), sep="")); cat("\n")
  cat("\n")

  cat(paste(format("Min elements in menu =", width=25), toString(min(x$sumStats$sumMsize)), sep="")); cat("\n")
  cat(paste(format("Max elements in menu =", width=25), toString(max(x$sumStats$sumMsize)), sep="")); cat("\n")
  cat("\n")

  cat(paste(format("Min eff. observations =", width=25), toString(min(x$sumStats$sumN)), sep="")); cat("\n")
  cat(paste(format("Max eff. observations =", width=25), toString(max(x$sumStats$sumN)), sep="")); cat("\n")
  cat("\n")

  for (i in 1:nrow(x$pref)) {
    cat(paste(rep("=", 60), collapse="")); cat("\n")
    cat(paste("Preference[", toString(i), "]",": ", sep=""))
    cat(x$pref[i, ]); cat("\n")

    if (sum(x$Tstat[i] > x$critVal$S[i, ]) == 3) {
      stars <- " ***"
    } else if (sum(x$Tstat[i] > x$critVal$S[i, ]) == 2) {
      stars <- " **"
    } else if (sum(x$Tstat[i] > x$critVal$S[i, ]) == 1) {
      stars <- " *"
    } else {
      stars <- " "
    }
    cat(paste("Statistic: ", toString(round(x$Tstat[i], 4)), stars, sep="")); cat("\n")

    cat(paste(rep("-", 60), collapse="")); cat("\n")
    cat(paste(format("Crit Val", width=15), format("0.10", width=15), format("0.05", width=15), format("0.01", width=15), sep="")); cat("\n")
    cat(paste(rep("-", 60), collapse="")); cat("\n")
    cat(paste(format("S"       , width=15),
              format(toString(round(x$critVal$S[i, 1], 4)), width=15),
              format(toString(round(x$critVal$S[i, 2], 4)), width=15),
              format(toString(round(x$critVal$S[i, 3], 4)), width=15), sep="")); cat("\n")
    cat(paste(format("PI"       , width=15),
              format(toString(round(x$critVal$PI[i, 1], 4)), width=15),
              format(toString(round(x$critVal$PI[i, 2], 4)), width=15),
              format(toString(round(x$critVal$PI[i, 3], 4)), width=15), sep="")); cat("\n")
    cat(paste(format("LF"       , width=15),
              format(toString(round(x$critVal$LF[i, 1], 4)), width=15),
              format(toString(round(x$critVal$LF[i, 2], 4)), width=15),
              format(toString(round(x$critVal$LF[i, 3], 4)), width=15), sep="")); cat("\n")
    cat(paste(format("MS"       , width=15),
              format(toString(round(x$critVal$MS[i, 1], 4)), width=15),
              format(toString(round(x$critVal$MS[i, 2], 4)), width=15),
              format(toString(round(x$critVal$MS[i, 3], 4)), width=15), sep="")); cat("\n")
    cat(paste(format("UB"       , width=15),
              format(toString(round(x$critVal$UB[i, 1], 4)), width=15),
              format(toString(round(x$critVal$UB[i, 2], 4)), width=15),
              format(toString(round(x$critVal$UB[i, 3], 4)), width=15), sep="")); cat("\n")


  }
  cat(paste(rep("=", 60), collapse="")); cat("\n")
  cat("\n")
}
