% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/reReg.R
\name{reReg}
\alias{reReg}
\title{Fits Semiparametric Regression Models for Recurrent Events and Failure Time}
\usage{
reReg(formula, data, B = 200, method = c("cox.LWYY", "cox.HW", "am.GL",
  "am.XCHWY", "sc.XCYH"), se = c("NULL", "bootstrap", "resampling"),
  plot.ci = FALSE, contrasts = NULL, control = list())
}
\arguments{
\item{formula}{a formula object, with the response on the left of a "~" operator, and the terms on the right.
The response must be a recurrent event survival object as returned by function \code{reSurv}.}

\item{data}{an optional data frame in which to interpret the variables occurring in the "formula".}

\item{B}{a numeric value specifies the number of resampling for variance estimation.
When B = 0, only the point estimates will be displayed.}

\item{method}{a character string specifying the underlying model. See \code{Details}.}

\item{se}{a character string specifying the method for standard error estimation. See \code{Details}.}

\item{plot.ci}{a logical value indicating whether the 95% confidence interval for the estimated cumulative rate function
and/or the estimated cumulative hazard function should be computed. Default is "FALSE".}

\item{contrasts}{an optional list.}

\item{control}{a list of control parameters.}
}
\description{
Fits a survival model for the recurrent event data.
The rate function of the underlying process for the recurrent event process
can be specified as a Cox-type model, an accelerated mean model, an accelerated rate model, or a generalized scale-change model.
When a joint model is fitted (e.g., \code{method = "cox.HW"} or \code{method = "am.XCHWY"}),
the hazard function of the terminal event is either in a Cox model or an accelerated failure time model.
}
\details{
The underlying models and assumptions are different for different methods.
The available methods are:
\describe{
  \item{\code{method == "cox.LWYY"}}{models the rate function  for the recurrent event through a Cox-type model.
This function returns results equivalent to that of \code{coxph}. See reference Lin et al. (2000).}
  \item{\code{method == "cox.HW"}}{jointly model the recurrent events and failure time.
This method assumes a Cox-type model for both the intensity function of the recurrent event process and the hazard function of the failure time.
Informative censoring is accounted for via a shared frailty variable.
See the references See reference Wang, Qin and Chiang(2001) and Huang and Wang (2004).}
  \item{\code{method == "am.GL"}}{jointly model the recurrent events and failure time.
This method assumes an accelerated mean model for both the recurrent event process and the failure time.
This method uses artificial censoring to allow for an unspecified association between the two types of outcomes.
Informative censoring is not allowed. See the reference Ghosh and Lin (2003).}
  \item{\code{method == "am.XCHWY"}}{jointly model the recurrent events and failure time.
This method assumes an accelerated mean model for both the recurrent event process and the failure time.
Informative censoring is accounted for via a shared frailty variable.
See the reference Xu et al. (2017).}
  \item{\code{method == "sc.XCYH"}}{models the rate function for the recurrent events via a generalized scale-change model that includes
Cox-type models, accelerated mean models, and accelerated rate models as special cases.
The methods also provide a hypothesis test of these submodels.
Informative censoring is accounted for via a shared frailty variable.}
}

For available method for standard errors estimation are:

\describe{
  \item{NULL}{means do not calculate standard errors.}
  \item{\code{se == "resampling"}}{performs the efficient resampling-based sandwich estimator that works with \code{method == "am.XCHWY"} and \code{method == "sc.XCYH"}.}
  \item{\code{"bootstrap"}}{works with all fitting methods.}
}
}
\examples{
## readmission data
data(readmission)
set.seed(123)
## Acceralted Mean Model
(fit <- reReg(reSurv(t.stop, event, death, id) ~ sex + chemo,
              data = subset(readmission, id < 50),
              method = "am.XCHWY", se = "resampling", B = 20))
summary(fit)

## Generalized Scale-Change Model
set.seed(123)
(fit <- reReg(reSurv(t.stop, event, death, id) ~ sex + chemo,
              data = subset(readmission, id < 50),
              method = "sc.XCYH", se = "resampling", B = 20))
summary(fit)

\dontrun{
## simulation data
simDat <- function(n, a, b, latent = FALSE) {
    ## setting rate function
    Lam.f <- function(t, z, x, w) .5 * z * exp(-x + w) * log(1 + t * exp(x))
    Lam.f0 <- function(t) .5 * log(1 + t)
    invLam.f  <- function(t, z, x, w) (exp((2 / z) * exp(x - w) * t )- 1) / exp(x)
    ## setting hazard funciton
    ## Haz.f0 <- function(t) .5 * log(1 + t) # assume constant hazard for now
    dat <- NULL
    for (id in 1:n) {
        z <- ifelse(latent, rgamma(1, 4, 4), 1)
        x1 <- rnorm(1)
        x2 <- rnorm(1)
        x <- c(x1, x2)
        cen <- rexp(1, z * exp(x \%*\% b) / 60) ## this gives constant hazard of 1/60
        y <- min(cen, 60)
        D <- 1 * (cen == y)
        tmpt <- NULL
        while(sum(tmpt) < Lam.f(y, z, c(x \%*\% a), c(x \%*\% b))) {
            tmpt <- c(tmpt, rexp(1))
        }
        m <- length(tmpt) - 1
        if (m > 0) {
            tt <- invLam.f(cumsum(tmpt[1:m]), z, c(x \%*\% a), c(x \%*\% b))
            dat <- rbind(dat, cbind(ID = id, Time = c(tt[order(tt)], y),
                                    event = c(rep(1, m), 0), status = c(rep(0, m), D),
                                    Z = z, M = m, X1 = x1, X2 = x2))
        } else {
            dat <- rbind(dat, cbind(ID = id, Time = y, event = 0, status = D,
                                    Z = z, M = m, X1 = x1, X2 = x2))
        }
    }
    return(data.frame(dat))
}
set.seed(2017)
dat <- simDat(200, c(0, 0), c(0, 0), TRUE) ## generate data under informative censoring
fm <- reSurv(Time, event, status, ID) ~ X1 + X2
fit.HW <- reReg(fm, data = dat, method = "cox.HW", B = 5)
}
}
\references{
Xu, G., Chiou, S.H., Huang, C.-Y., Wang, M.-C. and Yan, J. (2017). Joint Scale-change Models for Recurrent Events and Failure Time.
\emph{Journal of the American Statistical Association} 112.518: 796-805.

Lin, D., Wei, L., Yang, I. and Ying, Z. (2000). Semiparametric Regression for the Mean and Rate Functions of Recurrent Events.
\emph{Journal of the Royal Statistical Society: Series B (Methodological)}, \bold{62}: 711 -- 730.

Wang, M.C., Qin, J., and Chiang, C.T. (2001). Analyzing Recurrent Event Data with Informative Censoring.
\emph{Journal of the American Statistical Association} \bold{96}{455}: 1057--1065.

Ghosh, D. and D.Y. Lin (2003). Semiparametric Analysis of Recurrent Events Data in the Presence of Dependent Censoring.
\emph{Biometrics}, \bold{59}: 877 -- 885.

Huang, C.Y. and Wang, M.C. (2004). Joint Modeling and Estimation for Recurrent Event Processes and Failure Time Data.
\emph{Journal of the American Statistical Association} \bold{99}(468), 1153--1165.
}
\seealso{
\code{\link{reSurv}}
}
